! ========================================================================
! Copyright (C) by GemTalk Systems 1991-2020.  All Rights Reserved
! ========================================================================

! ------------------- Class definition for CPreprocessor
expectvalue /Class
doit
Object subclass: 'CPreprocessor'
  instVarNames: #( definitions header line
                   path readStream searchPaths tokens tmpFnames cppArchMType 
                   systemSearchPaths )
  classVars: #( )
  classInstVars: #()
  poolDictionaries: #()
  inDictionary: Globals
  options: #()

%
expectvalue /Class
doit
CPreprocessor comment: 
'This class is internal to the FFI implementation and 
implements a preprocessor for a C header file.'
%
expectvalue /Class
doit
CPreprocessor category: 'FFI'
%
! ------------------- Remove existing behavior from CPreprocessor
expectvalue /Metaclass3       
doit
CPreprocessor removeAllMethods.
CPreprocessor class removeAllMethods.
%
set compile_env: 0
set class CPreprocessor
! ------------------- Class methods for CPreprocessor
category: 'other'
classmethod:
evaluateHeader: aCHeader

	^self basicNew evaluateHeader: aCHeader.
%
category: 'other'
classmethod:
new

	^self basicNew
		initialize;
		yourself.
%
category: 'other'
classmethod:
parseFileAt: aString

	^self new
		includePath: aString;
		yourself.
%
category: 'other'
classmethod:
parseString: aString
	^ self new
	  	parseString: aString ignoreWarnings: false ;
	  	yourself.
%
classmethod:
parseString: aString ignoreWarnings: aBoolean 
	^ self new
	  	parseString: aString ignoreWarnings: aBoolean ;
	  	yourself.
%
category: 'other'
classmethod:
references

^'
	http://www.acm.uiuc.edu/webmonkeys/book/c_guide/index.html
	http://gcc.gnu.org/onlinedocs/gcc-3.0.1/cpp_1.html#SEC1
'.
%
! ------------------- Instance methods for CPreprocessor
category: 'directives'
method:
addRestOfLineTo: aCollection

	[
		readStream atEnd or: [readStream peek == Character lf].
	] whileFalse: [
		aCollection add: self nextToken.
	].
%
category: 'other'
method:
applyConcatenationToken: aToken

	| left right list |
	left := tokens last.
	right := self nextToken.
	list := { left . aToken . right }.
	self applyToken: right.
	left 
		collectOriginalFrom: list;
		concatenate: tokens last.
	tokens size: tokens size - 1.
%
category: 'directives'
method:
applyDirectiveToken: aToken

	| directive |
	directive := aToken value.
	directive = 'define' 			ifTrue: [^self define: 	aToken].
	directive = 'undef' 			ifTrue: [^self undef: 	aToken].
	directive = 'ifdef' 				ifTrue: [^self ifdef: 		aToken defined: true].
	directive = 'ifndef' 			ifTrue: [^self ifdef: 		aToken defined: false].
	directive = 'if'					ifTrue: [^self if:			aToken].
	directive = 'include'			ifTrue: [^self include: 	aToken afterMe: false].
	directive = 'include_next'	ifTrue: [^self include: 	aToken afterMe: true].
	directive = 'pragma'			ifTrue: [^self ignoreRestOfLine].
	directive = 'warning'			ifTrue: [^self ignoreRestOfLine].
	self error: 'Unrecognized directive token: ' , aToken printString.
%
category: 'other'
method:
applyFunctionMacroToken: aToken

	| char name arguments1 arguments2 definition oldLine oldStream string oldSize |
	"http://gcc.gnu.org/onlinedocs/cpp/Function_002dlike-Macros.html specifies to ignore if no parenthesis"
	(char := readStream peek) == $(
		ifFalse: 
			[tokens add: aToken.
			^self].
	name := aToken value.
	arguments1 := self macroFunctionArgumentsFor: aToken.
	arguments2 := self expandMacroFunctionArguments: arguments1.
	definition := definitions at: name.
	string := definition expandArguments: arguments2.
	string isEmpty ifTrue: [^self].

	oldLine := line.
	oldSize := tokens size.
	oldStream := readStream.
	readStream := CPreprocessorStream on: string.
	self readTillEnd.
	readStream := oldStream.
	oldSize == tokens size ifTrue: [tokens add: self cPreprocessorTokenSpecies empty].
	(tokens at: oldSize + 1) replaceFirst: aToken.
	oldSize + 2 to: tokens size do: [:i | (tokens at: i) replaceOther: aToken].
	line := oldLine	"Multi-line definitions would confuse the subsequent line numbers"
%
category: 'other'
method:
applyIdentifierToken: aToken

	| name definition oldTokens list |
	(1 <= tokens size and: [
		tokens last isDefinedIdentifierToken]) ifTrue: [tokens add: aToken. ^self].
	(2 <= tokens size and: [
		tokens last isOpenParenthesisToken and: [
		(tokens at: tokens size - 1) isDefinedIdentifierToken]]) ifTrue: [tokens add: aToken. ^self].
	name := aToken value.
	definition := definitions 
		at: name 
		ifAbsent: [tokens add: aToken. ^self].
	oldTokens := tokens.
	tokens := list := Array new.
	definition isFunctionLikeMacroDefinition ifTrue: [
		self applyFunctionMacroToken: aToken.
	] ifFalse: [
		self applySimpleMacroToken: aToken.
	].
	tokens := oldTokens.
	definitions removeKey: name.
	list do: [:each | self applyToken: each].
	self definitionsAt: name put: definition.
%
method:
definitionsAt: name put: definition
  "in a seperate method for easier set breakpoint."
  ^ definitions at: name put: definition
%
category: 'other'
method:
applySimpleMacroToken: aToken

	| definition oldLine oldSize oldStream |
	definition := definitions removeKey: aToken value.
	oldLine := line.
	oldSize := tokens size.
	oldStream := readStream.
	readStream := CPreprocessorStream on: definition value.
	self readTillEnd.
	readStream := oldStream.
	oldSize == tokens size ifTrue: [tokens add: self cPreprocessorTokenSpecies empty].
	(tokens at: oldSize + 1) replaceFirst: aToken.
	oldSize + 2 to: tokens size do: [:i | (tokens at: i) replaceOther: aToken].
	line := oldLine.		"Multi-line definitions would confuse the subsequent line numbers"
	self definitionsAt: aToken value put: definition.
%
category: 'other'
method:
applyToken: aToken
  "GsFile gciLogServer:'applyToken  ' , aToken printString ." "uncomment to debug"
	aToken isDirectiveToken ifTrue: [^self applyDirectiveToken: aToken].
	aToken isConcatenationToken ifTrue: [^self applyConcatenationToken: aToken].
	aToken isIdentifierToken ifTrue: [^self applyIdentifierToken: aToken].
	tokens add: aToken.
%
category: 'other'
method:
atEnd

	^tokens isEmpty.
%
category: 'other'
method:
clearTokens
  tokens := nil.
  tmpFnames ifNotNil:[:fNames |
    fNames do:[:f | GsFile removeServerFile: f ].
    tmpFnames := nil .
  ].
%
category: 'default definitions'
method: CPreprocessor
defaultAix61Definitions
  self nativeCppCmd ifNotNil:[ 
    ^ self defaultNativeCppDefinitions 
  ].
  ^ self _defaultAix61DefinitionsOld
%
category: 'default definitions'
method:
_defaultAix61DefinitionsOld
	"The usual ... echo '' | cpp -dM ... doesn't work on AIX.
	 The AIX equivalent is:
		echo 'int main() { return 0; }' > x.C; /usr/vacpp/bin/xlC_r -qshowmacros -q64 -E x.C
"
	^'
#define _POWER 1
#define _IBMR2 1
#define _AIX61 1
#define _AIX53 1
#define _AIX52 1
#define _AIX51 1
#define _AIX50 1
#define _AIX43 1
#define _AIX41 1
#define _AIX32 1
#define _AIX 1
#define __VACPP_MULTI__ 1
#define _THREAD_SAFE 1
#define __IBM_INCLUDE_NEXT 1
#define __IBM_MACRO_WITH_VA_ARGS 1
#define __C99_MACRO_WITH_VA_ARGS 1
#define __C99__FUNC__ 1
#define __EXCEPTIONS 1
#define _CPPUNWIND 1
#define __DIGRAPHS__ 1
#define __LONGDOUBLE64 1
#define __IBM__ALIGN 1
#define __ALIGN 1
#define __C99_HEX_FLOAT_CONST 1
#define __IBM_CHAR32_T__ 1
#define __IBM_CHAR16_T__ 1
#define __IBM_ATTRIBUTES 1
#define __IBM_EXTENSION_KEYWORD 1
#define __IBM__TYPEOF__ 1
#define __IBM__ALIGNOF__ 1
#define __C99_PRAGMA_OPERATOR 1
#define __IBM__RESTRICT__ 1
#define __C99_RESTRICT 1
#define __BOOL__ 1
#define __OBJECT_MODEL_CLASSIC__ 1
#define __OBJECT_MODEL_COMPAT__ 1
#define __NO_RTTI__ 1
', "#define defined defined" 	"This seems to break the parser, allowing it to think 'defined' is a type declaration"
'#define __CHAR_UNSIGNED__ 1
#define __IBMCPP_EXTERN_TEMPLATE 1
#define __XPLINK_CALLBACK__ 1
#define __IBM_UTF_LITERAL 1
#define __IBM_LOCAL_LABEL 1
#define __IBM_LABEL_VALUE 1
#define __IBM_COMPUTED_GOTO 1
#define __C99_VARIABLE_LENGTH_ARRAY 1
#define __C99_COMPOUND_LITERAL 1
#define _CHAR_UNSIGNED 1
#define _LONG_LONG 1
#define _ARCH_PPC64 1
#define _ARCH_PPC 1
#define _ARCH_COM 1
#define __IBM_GCC_ASM 1
#define _EXT 1
#define _MI_BUILTIN 1
#define __BASE_FILE__ "x.C"
#define __V6ALIGN__ 1
#define __XLC13__ 1
#define __XLC121__ 1
#define __MATH__ 1
#define __STR__ 1
#define _WCHAR_T 1
#define __TOS_AIX__ 1
#define __THW_RS6000__ 1
#define _LP64 1
#define __64BIT__ 1
#define __ppc__ 1
#define __PPC__ 1
#define __PPC 1
#define __powerpc__ 1
#define __powerpc 1
#define __BIG_ENDIAN__ 1
#define _BIG_ENDIAN 1
#define __THW_PPC__ 1
#define __THW_BIG_ENDIAN__ 1
#define __unix__ 1
#define __unix 1
#define __FUNCTION__ __FUNCTION__
#define __STDC__ 0
#define __xlC_ver__ 0x0000000c
#define __xlC__ 0x0b01
#define __HOS_AIX__ 1
#define __HHW_RS6000__ 1
#define __HHW_BIG_ENDIAN__ 1
#define __IBMCPP__ 1110
#define __cplusplus 199711L
' , 
	"additional definitions (some of this looks like rubbish!)"
'
#define FLG_AIX_VERSION 61

#define ptrdiff_t long
#define wchar_t int
#define throw()
#define restrict
#define __restrict
#define __const const
#define __inline inline
#define __inline__ inline
#define __signed signed
#define __volatile
'.
%
category: 'default definitions'
method: CPreprocessor
defaultAix71Definitions
  self nativeCppCmd ifNotNil:[ ^ self defaultNativeCppDefinitions ].
  ^ self _defaultAix71DefinitionsOld
%
category: 'default definitions'
method:
_defaultAix71DefinitionsOld
	"The usual ... echo '' | cpp -dM ... doesn't work on AIX.
	 The AIX equivalent is:
		echo 'int main() { return 0; }' > x.C; /usr/vacpp/bin/xlC_r -qshowmacros -q64 -E x.C
"
	^'
#define _POWER 1
#define _IBMR2 1
#define _AIX71 1
#define _AIX61 1
#define _AIX53 1
#define _AIX52 1
#define _AIX51 1
#define _AIX50 1
#define _AIX43 1
#define _AIX41 1
#define _AIX32 1
#define _AIX 1
#define __VACPP_MULTI__ 1
#define _THREAD_SAFE 1
#define __IBM_INCLUDE_NEXT 1
#define __IBM_MACRO_WITH_VA_ARGS 1
#define __C99_MACRO_WITH_VA_ARGS 1
#define __C99__FUNC__ 1
#define __EXCEPTIONS 1
#define _CPPUNWIND 1
#define __DIGRAPHS__ 1
#define __LONGDOUBLE64 1
#define __IBM__ALIGN 1
#define __ALIGN 1
#define __C99_HEX_FLOAT_CONST 1
#define __IBM_CHAR32_T__ 1
#define __IBM_CHAR16_T__ 1
#define __IBM_ATTRIBUTES 1
#define __IBM_EXTENSION_KEYWORD 1
#define __IBM__TYPEOF__ 1
#define __IBM__ALIGNOF__ 1
#define __C99_PRAGMA_OPERATOR 1
#define __IBM__RESTRICT__ 1
#define __C99_RESTRICT 1
#define __BOOL__ 1
#define __OBJECT_MODEL_CLASSIC__ 1
#define __OBJECT_MODEL_COMPAT__ 1
#define __NO_RTTI__ 1
', "#define defined defined" 	"This seems to break the parser, allowing it to think 'defined' is a type declaration"
'#define __CHAR_UNSIGNED__ 1
#define __IBMCPP_EXTERN_TEMPLATE 1
#define __XPLINK_CALLBACK__ 1
#define __IBM_UTF_LITERAL 1
#define __IBM_LOCAL_LABEL 1
#define __IBM_LABEL_VALUE 1
#define __IBM_COMPUTED_GOTO 1
#define __C99_VARIABLE_LENGTH_ARRAY 1
#define __C99_COMPOUND_LITERAL 1
#define _CHAR_UNSIGNED 1
#define _LONG_LONG 1
#define _ARCH_PPC64 1
#define _ARCH_PPC 1
#define _ARCH_COM 1
#define __IBM_GCC_ASM 1
#define _EXT 1
#define _MI_BUILTIN 1
#define __BASE_FILE__ "x.C"
#define __V6ALIGN__ 1
#define __XLC13__ 1
#define __XLC121__ 1
#define __MATH__ 1
#define __STR__ 1
#define _WCHAR_T 1
#define __TOS_AIX__ 1
#define __THW_RS6000__ 1
#define _LP64 1
#define __64BIT__ 1
#define __ppc__ 1
#define __PPC__ 1
#define __PPC 1
#define __powerpc__ 1
#define __powerpc 1
#define __BIG_ENDIAN__ 1
#define _BIG_ENDIAN 1
#define __THW_PPC__ 1
#define __THW_BIG_ENDIAN__ 1
#define __unix__ 1
#define __unix 1
#define __FUNCTION__ __FUNCTION__
#define __STDC__ 0
#define __xlC_ver__ 0x0000000a
#define __xlC__ 0x0b01
#define __HOS_AIX__ 1
#define __HHW_RS6000__ 1
#define __HHW_BIG_ENDIAN__ 1
#define __IBMCPP__ 1110
#define __cplusplus 199711L
' , 
	"additional definitions (some of this looks like rubbish!)"
'
#define FLG_AIX_VERSION 71

#define ptrdiff_t long
#define wchar_t int
#define throw()
#define restrict
#define __restrict
#define __const const
#define __inline inline
#define __inline__ inline
#define __signed signed
#define __volatile
'.
%

category: 'default definitions'
method: CPreprocessor
defaultAix72Definitions
  self nativeCppCmd ifNotNil:[ ^ self defaultNativeCppDefinitions ].
  ^ self _defaultAix72DefinitionsOld
%

category: 'default definitions'
method: CPreprocessor
_defaultAix72DefinitionsOld
	"The usual ... echo '' | cpp -dM ... doesn't work on AIX.
	 The AIX equivalent is:
		echo 'int main() { return 0; }' > x.C; /usr/vacpp/bin/xlC_r -qshowmacros -q64 -E x.C
"
	^'
#define _POWER 1
#define _IBMR2 1
#define _AIX72 1
#define _AIX71 1
#define _AIX61 1
#define _AIX53 1
#define _AIX52 1
#define _AIX51 1
#define _AIX50 1
#define _AIX43 1
#define _AIX41 1
#define _AIX32 1
#define _AIX 1
#define __VACPP_MULTI__ 1
#define _THREAD_SAFE 1
#define __IBM_INCLUDE_NEXT 1
#define __IBM_MACRO_WITH_VA_ARGS 1
#define __C99_MACRO_WITH_VA_ARGS 1
#define __C99__FUNC__ 1
#define __EXCEPTIONS 1
#define _CPPUNWIND 1
#define __DIGRAPHS__ 1
#define __LONGDOUBLE64 1
#define __IBM__ALIGN 1
#define __ALIGN 1
#define __C99_HEX_FLOAT_CONST 1
#define __IBM_CHAR32_T__ 1
#define __IBM_CHAR16_T__ 1
#define __IBM_ATTRIBUTES 1
#define __IBM_EXTENSION_KEYWORD 1
#define __IBM__TYPEOF__ 1
#define __IBM__ALIGNOF__ 1
#define __C99_PRAGMA_OPERATOR 1
#define __IBM__RESTRICT__ 1
#define __C99_RESTRICT 1
#define __BOOL__ 1
#define __OBJECT_MODEL_CLASSIC__ 1
#define __OBJECT_MODEL_COMPAT__ 1
#define __NO_RTTI__ 1
', "#define defined defined"    "This seems to break the parser, allowing it to think 'defined' is a type declaration"
'#define __CHAR_UNSIGNED__ 1
#define __IBMCPP_EXTERN_TEMPLATE 1
#define __XPLINK_CALLBACK__ 1
#define __IBM_UTF_LITERAL 1
#define __IBM_LOCAL_LABEL 1
#define __IBM_LABEL_VALUE 1
#define __IBM_COMPUTED_GOTO 1
#define __C99_VARIABLE_LENGTH_ARRAY 1
#define __C99_COMPOUND_LITERAL 1
#define _CHAR_UNSIGNED 1
#define _LONG_LONG 1
#define _ARCH_PPC64 1
#define _ARCH_PPC 1
#define _ARCH_COM 1
#define __IBM_GCC_ASM 1
#define _EXT 1
#define _MI_BUILTIN 1
#define __BASE_FILE__ "x.C"
#define __V6ALIGN__ 1
#define __XLC13__ 1
#define __XLC121__ 1
#define __MATH__ 1
#define __STR__ 1
#define _WCHAR_T 1
#define __TOS_AIX__ 1
#define __THW_RS6000__ 1
#define _LP64 1
#define __64BIT__ 1
#define __ppc__ 1
#define __PPC__ 1
#define __PPC 1
#define __powerpc__ 1
#define __powerpc 1
#define __BIG_ENDIAN__ 1
#define _BIG_ENDIAN 1
#define __THW_PPC__ 1
#define __THW_BIG_ENDIAN__ 1
#define __unix__ 1
#define __unix 1
#define __FUNCTION__ __FUNCTION__
#define __STDC__ 0
#define __xlC_ver__ 0x00000017
#define __xlC__ 0x0b01
#define __HOS_AIX__ 1
#define __HHW_RS6000__ 1
#define __HHW_BIG_ENDIAN__ 1
#define __IBMCPP__ 1110
#define __cplusplus 199711L
' , 
	"additional definitions (some of this looks like rubbish!)"
'
#define FLG_AIX_VERSION 72

#define ptrdiff_t long
#define wchar_t int
#define throw()
#define restrict
#define __restrict
#define __const const
#define __inline inline
#define __inline__ inline
#define __signed signed
#define __volatile
'.
%
category: 'default definitions'
method:
defaultDarwinDefinitions
self nativeCppCmd ifNotNil:[
  "We are running the OS's preprocessor first"
  ^ self defaultNativeCppDefinitions , 
'#define __compar
#define _close
#define _read
#define _seek
#define _write
#define availability(x)
#define _Nonnull 
'
].
^ self _defaultDarwinDefinitionsOld
%

category: 'default definitions'
method:
_defaultDarwinDefinitionsOld
	"echo '' | cpp -dM
"
	^'
#define OBJC_NEW_PROPERTIES 1
#define _LP64 1
#define __APPLE_CC__ 6000
#define __APPLE__ 1
#define __ATOMIC_ACQUIRE 2
#define __ATOMIC_ACQ_REL 4
#define __ATOMIC_CONSUME 1
#define __ATOMIC_RELAXED 0
#define __ATOMIC_RELEASE 3
#define __ATOMIC_SEQ_CST 5
#define __BLOCKS__ 1
#define __BYTE_ORDER__ __ORDER_LITTLE_ENDIAN__
#define __CHAR16_TYPE__ unsigned short
#define __CHAR32_TYPE__ unsigned int
#define __CHAR_BIT__ 8
#define __CONSTANT_CFSTRINGS__ 1
#define __DBL_DENORM_MIN__ 4.9406564584124654e-324
#define __DBL_DIG__ 15
#define __DBL_EPSILON__ 2.2204460492503131e-16
#define __DBL_HAS_DENORM__ 1
#define __DBL_HAS_INFINITY__ 1
#define __DBL_HAS_QUIET_NAN__ 1
#define __DBL_MANT_DIG__ 53
#define __DBL_MAX_10_EXP__ 308
#define __DBL_MAX_EXP__ 1024
#define __DBL_MAX__ 1.7976931348623157e+308
#define __DBL_MIN_10_EXP__ (-307)
#define __DBL_MIN_EXP__ (-1021)
#define __DBL_MIN__ 2.2250738585072014e-308
#define __DECIMAL_DIG__ 21
#define __DYNAMIC__ 1
#define __ENVIRONMENT_MAC_OS_X_VERSION_MIN_REQUIRED__ 1090
#define __FINITE_MATH_ONLY__ 0
#define __FLT_DENORM_MIN__ 1.40129846e-45F
#define __FLT_DIG__ 6
#define __FLT_EPSILON__ 1.19209290e-7F
#define __FLT_EVAL_METHOD__ 0
#define __FLT_HAS_DENORM__ 1
#define __FLT_HAS_INFINITY__ 1
#define __FLT_HAS_QUIET_NAN__ 1
#define __FLT_MANT_DIG__ 24
#define __FLT_MAX_10_EXP__ 38
#define __FLT_MAX_EXP__ 128
#define __FLT_MAX__ 3.40282347e+38F
#define __FLT_MIN_10_EXP__ (-37)
#define __FLT_MIN_EXP__ (-125)
#define __FLT_MIN__ 1.17549435e-38F
#define __FLT_RADIX__ 2
#define __GCC_ATOMIC_BOOL_LOCK_FREE 2
#define __GCC_ATOMIC_CHAR16_T_LOCK_FREE 2
#define __GCC_ATOMIC_CHAR32_T_LOCK_FREE 2
#define __GCC_ATOMIC_CHAR_LOCK_FREE 2
#define __GCC_ATOMIC_INT_LOCK_FREE 2
#define __GCC_ATOMIC_LLONG_LOCK_FREE 2
#define __GCC_ATOMIC_LONG_LOCK_FREE 2
#define __GCC_ATOMIC_POINTER_LOCK_FREE 2
#define __GCC_ATOMIC_SHORT_LOCK_FREE 2
#define __GCC_ATOMIC_TEST_AND_SET_TRUEVAL 1
#define __GCC_ATOMIC_WCHAR_T_LOCK_FREE 2
#define __GCC_HAVE_SYNC_COMPARE_AND_SWAP_1 1
#define __GCC_HAVE_SYNC_COMPARE_AND_SWAP_16 1
#define __GCC_HAVE_SYNC_COMPARE_AND_SWAP_2 1
#define __GCC_HAVE_SYNC_COMPARE_AND_SWAP_4 1
#define __GCC_HAVE_SYNC_COMPARE_AND_SWAP_8 1
#define __GNUC_MINOR__ 2
#define __GNUC_PATCHLEVEL__ 1
#define __GNUC_STDC_INLINE__ 1
#define __GNUC__ 4
#define __GXX_ABI_VERSION 1002
#define __GXX_RTTI 1
#define __INT16_TYPE__ short
#define __INT32_TYPE__ int
#define __INT64_C_SUFFIX__ LL
#define __INT64_TYPE__ long long int
#define __INT8_TYPE__ char
#define __INTMAX_MAX__ 9223372036854775807L
#define __INTMAX_TYPE__ long int
#define __INTMAX_WIDTH__ 64
#define __INTPTR_TYPE__ long int
#define __INTPTR_WIDTH__ 64
#define __INT_MAX__ 2147483647
#define __LDBL_DENORM_MIN__ 3.64519953188247460253e-4951L
#define __LDBL_DIG__ 18
#define __LDBL_EPSILON__ 1.08420217248550443401e-19L
#define __LDBL_HAS_DENORM__ 1
#define __LDBL_HAS_INFINITY__ 1
#define __LDBL_HAS_QUIET_NAN__ 1
#define __LDBL_MANT_DIG__ 64
#define __LDBL_MAX_10_EXP__ 4932
#define __LDBL_MAX_EXP__ 16384
#define __LDBL_MAX__ 1.18973149535723176502e+4932L
#define __LDBL_MIN_10_EXP__ (-4931)
#define __LDBL_MIN_EXP__ (-16381)
#define __LDBL_MIN__ 3.36210314311209350626e-4932L
#define __LITTLE_ENDIAN__ 1
#define __LONG_LONG_MAX__ 9223372036854775807LL
#define __LONG_MAX__ 9223372036854775807L
#define __LP64__ 1
#define __MACH__ 1
#define __MMX__ 1
#define __NO_INLINE__ 1
#define __NO_MATH_INLINES 1
#define __ORDER_BIG_ENDIAN__ 4321
#define __ORDER_LITTLE_ENDIAN__ 1234
#define __ORDER_PDP_ENDIAN__ 3412
#define __PIC__ 2
#define __POINTER_WIDTH__ 64
#define __PRAGMA_REDEFINE_EXTNAME 1
#define __PTRDIFF_TYPE__ long int
#define __PTRDIFF_WIDTH__ 64
#define __REGISTER_PREFIX__ 
#define __SCHAR_MAX__ 127
#define __SHRT_MAX__ 32767
#define __SIG_ATOMIC_WIDTH__ 32
#define __SIZEOF_DOUBLE__ 8
#define __SIZEOF_FLOAT__ 4
#define __SIZEOF_INT128__ 16
#define __SIZEOF_INT__ 4
#define __SIZEOF_LONG_DOUBLE__ 16
#define __SIZEOF_LONG_LONG__ 8
#define __SIZEOF_LONG__ 8
#define __SIZEOF_POINTER__ 8
#define __SIZEOF_PTRDIFF_T__ 8
#define __SIZEOF_SHORT__ 2
#define __SIZEOF_SIZE_T__ 8
#define __SIZEOF_WCHAR_T__ 4
#define __SIZEOF_WINT_T__ 4
#define __SIZE_MAX__ 18446744073709551615UL
#define __SIZE_TYPE__ long unsigned int
#define __SIZE_WIDTH__ 64
#define __SSE2_MATH__ 1
#define __SSE2__ 1
#define __SSE3__ 1
#define __SSE_MATH__ 1
#define __SSE__ 1
#define __SSP__ 1
#define __SSSE3__ 1
#define __STDC_HOSTED__ 1
#define __STDC_UTF_16__ 1
#define __STDC_UTF_32__ 1
#define __STDC_VERSION__ 199901L
#define __UINTMAX_TYPE__ long unsigned int
#define __USER_LABEL_PREFIX__ _
#define __VERSION__ "4.2.1 Compatible Apple LLVM 6.0 (clang-600.0.56)"
#define __WCHAR_MAX__ 2147483647
#define __WCHAR_TYPE__ int
#define __WCHAR_WIDTH__ 32
#define __WINT_TYPE__ int
#define __WINT_WIDTH__ 32
#define __amd64 1
#define __amd64__ 1
#define __apple_build_version__ 6000056
#define __block __attribute__((__blocks__(byref)))
#define __clang__ 1
#define __clang_major__ 6
#define __clang_minor__ 0
#define __clang_patchlevel__ 0
#define __clang_version__ "6.0 (clang-600.0.56)"
#define __core2 1
#define __core2__ 1
#define __llvm__ 1
#define __pic__ 2
#define __strong 
#define __tune_core2__ 1
#define __unsafe_unretained 
#define __weak __attribute__((objc_gc(weak)))
#define __x86_64 1
#define __x86_64__ 1
' , 
	"additional definitions"
'
#define NO_ANSI_KEYWORDS
#define restrict
#define throw()
#define __const const
#define __inline inline
#define __inline__ inline
#define __restrict
#define __signed signed
#define __STDC__
#define __volatile
'
%
method:
defaultNativeCppDefinitions
^ 
'#define __inline inline
#define __inline__ inline
#define __restrict_arr
#define __restrict
#define restrict
#define __const const
#define __signed signed
#define __extension__ 
#define __volatile 
#define throw()
'
%
category: 'default definitions'
method:
defaultLinuxDefinitions

self nativeCppCmd ifNotNil:[
  "We are running the OS's preprocessor first"
  ^ self defaultNativeCppDefinitions
].
^ System performOnServer: 'cpp -dM /dev/null'
%
method
_defaultLinuxDefinitionsOld
"from 3.5 trunk r45820 "
	^'
#define __DBL_MIN_EXP__ (-1021)
#define __FLT_MIN__ 1.17549435e-38F
#define __CHAR_BIT__ 8
#define __WCHAR_MAX__ 2147483647
#define __GCC_HAVE_SYNC_COMPARE_AND_SWAP_1 1
#define __GCC_HAVE_SYNC_COMPARE_AND_SWAP_2 1
#define __GCC_HAVE_SYNC_COMPARE_AND_SWAP_4 1
#define __DBL_DENORM_MIN__ 4.9406564584124654e-324
#define __GCC_HAVE_SYNC_COMPARE_AND_SWAP_8 1
#define __FLT_EVAL_METHOD__ 0
#define __unix__ 1
#define __x86_64 1
#define __DBL_MIN_10_EXP__ (-307)
#define __FINITE_MATH_ONLY__ 0
#define __GNUC_PATCHLEVEL__ 3
#define __DEC64_MAX_EXP__ 385
#define __SHRT_MAX__ 32767
#define __LDBL_MAX__ 1.18973149535723176502e+4932L
#define __UINTMAX_TYPE__ long unsigned int
#define __linux 1
#define __DEC32_EPSILON__ 1E-6DF
#define __unix 1
#define __LDBL_MAX_EXP__ 16384
#define __linux__ 1
#define __SCHAR_MAX__ 127
#define __DBL_DIG__ 15
#define _FORTIFY_SOURCE 2
#define __SIZEOF_INT__ 4
#define __SIZEOF_POINTER__ 8
#define __USER_LABEL_PREFIX__ 
#define __STDC_HOSTED__ 1
#define __LDBL_HAS_INFINITY__ 1
#define __FLT_EPSILON__ 1.19209290e-7F
#define __LDBL_MIN__ 3.36210314311209350626e-4932L
#define __DEC32_MAX__ 9.999999E96DF
#define __SIZEOF_LONG__ 8
#define __DECIMAL_DIG__ 21
#define __gnu_linux__ 1
#define __LDBL_HAS_QUIET_NAN__ 1
#define __GNUC__ 4
#define __MMX__ 1
#define __FLT_HAS_DENORM__ 1
#define __SIZEOF_LONG_DOUBLE__ 16
#define __BIGGEST_ALIGNMENT__ 16
#define __DBL_MAX__ 1.7976931348623157e+308
#define __DBL_HAS_INFINITY__ 1
#define __DEC32_MIN_EXP__ (-94)
#define __LDBL_HAS_DENORM__ 1
#define __DEC128_MAX__ 9.999999999999999999999999999999999E6144DL
#define __DEC32_MIN__ 1E-95DF
#define __DBL_MAX_EXP__ 1024
#define __DEC128_EPSILON__ 1E-33DL
#define __SSE2_MATH__ 1
#define __amd64 1
#define __LONG_LONG_MAX__ 9223372036854775807LL
#define __SIZEOF_SIZE_T__ 8
#define __SIZEOF_WINT_T__ 4
#define __GCC_HAVE_DWARF2_CFI_ASM 1
#define __GXX_ABI_VERSION 1002
#define __FLT_MIN_EXP__ (-125)
#define __DBL_MIN__ 2.2250738585072014e-308
#define __LP64__ 1
#define __DECIMAL_BID_FORMAT__ 1
#define __DEC128_MIN__ 1E-6143DL
#define __REGISTER_PREFIX__ 
#define __DBL_HAS_DENORM__ 1
#define __NO_INLINE__ 1
#define __FLT_MANT_DIG__ 24
#define __VERSION__ "4.4.3"
#define __DEC64_EPSILON__ 1E-15DD
#define __DEC128_MIN_EXP__ (-6142)
#define unix 1
#define __SIZE_TYPE__ long unsigned int
#define __ELF__ 1
#define __FLT_RADIX__ 2
#define __LDBL_EPSILON__ 1.08420217248550443401e-19L
#define __SSE_MATH__ 1
#define __k8 1
#define __SIZEOF_PTRDIFF_T__ 8
#define __x86_64__ 1
#define __DEC32_SUBNORMAL_MIN__ 0.000001E-95DF
#define __FLT_HAS_QUIET_NAN__ 1
#define __FLT_MAX_10_EXP__ 38
#define __LONG_MAX__ 9223372036854775807L
#define __DEC128_SUBNORMAL_MIN__ 0.000000000000000000000000000000001E-6143DL
#define __FLT_HAS_INFINITY__ 1
#define __DEC64_MAX__ 9.999999999999999E384DD
#define __CHAR16_TYPE__ short unsigned int
#define __DEC64_MANT_DIG__ 16
#define __DEC32_MAX_EXP__ 97
#define linux 1
#define __SSE2__ 1
#define __LDBL_MANT_DIG__ 64
#define __DBL_HAS_QUIET_NAN__ 1
#define __k8__ 1
#define __WCHAR_TYPE__ int
#define __SIZEOF_FLOAT__ 4
#define __DEC64_MIN_EXP__ (-382)
#define __FLT_DIG__ 6
#define __INT_MAX__ 2147483647
#define __amd64__ 1
#define __FLT_MAX_EXP__ 128
#define __DBL_MANT_DIG__ 53
#define __DEC64_MIN__ 1E-383DD
#define __WINT_TYPE__ unsigned int
#define __SIZEOF_SHORT__ 2
#define __SSE__ 1
#define __LDBL_MIN_EXP__ (-16381)
#define __SSP__ 1
#define __LDBL_MAX_10_EXP__ 4932
#define __DBL_EPSILON__ 2.2204460492503131e-16
#define _LP64 1
#define __SIZEOF_WCHAR_T__ 4
#define __DEC_EVAL_METHOD__ 2
#define __INTMAX_MAX__ 9223372036854775807L
#define __FLT_DENORM_MIN__ 1.40129846e-45F
#define __CHAR32_TYPE__ unsigned int
#define __FLT_MAX__ 3.40282347e+38F
#define __SIZEOF_DOUBLE__ 8
#define __FLT_MIN_10_EXP__ (-37)
#define __INTMAX_TYPE__ long int
#define __DEC128_MAX_EXP__ 6145
#define __GNUC_MINOR__ 4
#define __DEC32_MANT_DIG__ 7
#define __DBL_MAX_10_EXP__ 308
#define __LDBL_DENORM_MIN__ 3.64519953188247460253e-4951L
#define __STDC__ 1
#define __PTRDIFF_TYPE__ long int
#define __DEC64_SUBNORMAL_MIN__ 0.000000000000001E-383DD
#define __DEC128_MANT_DIG__ 34
#define __LDBL_MIN_10_EXP__ (-4931)
#define __SIZEOF_LONG_LONG__ 8
#define __LDBL_DIG__ 18
#define __GNUC_GNU_INLINE__ 1
' , 
	"additional definitions"
'
#define ptrdiff_t long
#define restrict
#define size_t unsigned long
#define wchar_t int
#define throw()
#define _XOPEN_SOURCE 1
#define __const const
#define __cplusplus
#define __inline inline
#define __inline__ inline
#define __restrict
#define __signed signed
#define __volatile
#define _BSD_SOURCE 1
#define _SIZE_T
#define __GNUG__
'
%
category: 'Private'
method:
_initializePaths
	| list |
  self nativeCppCmd .
	list := { '' }.
	(path ~~ nil and: [path first == $$]) ifTrue: [list add: self directory].
  cppArchMType ifNil:[  "not using native cpp"
	  list add: '/usr/include/'; add: '/usr/local/include/' .
    systemSearchPaths := { } .
  ] ifNotNil:[
    systemSearchPaths := { '/usr/include/' . '/usr/local/include/' }.
    cppArchMType == 50 ifTrue:[ systemSearchPaths add: '/usr/include/linux' ].
  ].
  searchPaths := list .
%

category: 'default definitions'
method:
defaultSparcSolarisDefinitions
	"echo '' | cpp -dM
"
	^'
#define __GCC_NEW_VARARGS__ 1 
#define __sparc 1 
#define __svr4__ 1 
#define __sun 1 
#define sparc 1 
#define __sun__ 1 
#define __unix 1 
#define __unix__ 1 
#define __SVR4 1 
#define sun 1 
#define __sparc__ 1 
#define unix 1 
' , 
	"additional definitions"
'
#define ptrdiff_t long
#define restrict
#define throw()
#define __const const
#define __inline inline
#define __inline__ inline
#define __restrict
#define __signed signed
#define __STDC__
#define __volatile
'.
%
category: 'default definitions'
method:
defaultX86SolarisDefinitions
  "echo '' | cpp -dM"

^ '
#define __DBL_MIN_EXP__ (-1021)
#define __FLT_MIN__ 1.17549435e-38F
#define __CHAR_BIT__ 8
#define __WCHAR_MAX__ 2147483647
#define __DBL_DENORM_MIN__ 4.9406564584124654e-324
#define __FLT_EVAL_METHOD__ 2
#define __DBL_MIN_10_EXP__ (-307)
#define __FINITE_MATH_ONLY__ 0
#define __GNUC_PATCHLEVEL__ 2
#define __SHRT_MAX__ 32767
#define __LDBL_MAX__ 1.18973149535723176502e+4932L
#define __unix 1
#define __LDBL_MAX_EXP__ 16384
#define __SCHAR_MAX__ 127
#define __USER_LABEL_PREFIX__ 
#define __STDC_HOSTED__ 1
#define __LDBL_HAS_INFINITY__ 1
#define __DBL_DIG__ 15
#define __FLT_EPSILON__ 1.19209290e-7F
#define __LDBL_MIN__ 3.36210314311209350626e-4932L
#define __unix__ 1
#define __DECIMAL_DIG__ 21
#define __LDBL_HAS_QUIET_NAN__ 1
#define __GNUC__ 3
#define __DBL_MAX__ 1.7976931348623157e+308
#define __DBL_HAS_INFINITY__ 1
#define __SVR4 1
#define __DBL_MAX_EXP__ 1024
#define __LONG_LONG_MAX__ 9223372036854775807LL
#define __GXX_ABI_VERSION 1002
#define __FLT_MIN_EXP__ (-125)
#define __DBL_MIN__ 2.2250738585072014e-308
#define __DBL_HAS_QUIET_NAN__ 1
#define __tune_i386__ 1
#define __sun 1
#define __REGISTER_PREFIX__ 
#define __NO_INLINE__ 1
#define __i386 1
#define __FLT_MANT_DIG__ 24
#define __VERSION__ "3.4.2"
#define i386 1
#define sun 1
#define unix 1
#define __i386__ 1
#define __SIZE_TYPE__ unsigned int
#define __ELF__ 1
#define __FLT_RADIX__ 2
#define __LDBL_EPSILON__ 1.08420217248550443401e-19L
#define __FLT_HAS_QUIET_NAN__ 1
#define __FLT_MAX_10_EXP__ 38
#define __LONG_MAX__ 2147483647L
#define __FLT_HAS_INFINITY__ 1
#define __PRAGMA_REDEFINE_EXTNAME 1
#define __LDBL_MANT_DIG__ 64
#define __WCHAR_TYPE__ long int
#define __FLT_DIG__ 6
#define __INT_MAX__ 2147483647
#define __FLT_MAX_EXP__ 128
#define __DBL_MANT_DIG__ 53
#define __WINT_TYPE__ long int
#define __LDBL_MIN_EXP__ (-16381)
#define __LDBL_MAX_10_EXP__ 4932
#define __DBL_EPSILON__ 2.2204460492503131e-16
#define __sun__ 1
#define __svr4__ 1
#define __FLT_DENORM_MIN__ 1.40129846e-45F
#define __FLT_MAX__ 3.40282347e+38F
#define __FLT_MIN_10_EXP__ (-37)
#define __GNUC_MINOR__ 4
#define __DBL_MAX_10_EXP__ 308
#define __LDBL_DENORM_MIN__ 3.64519953188247460253e-4951L
#define __PTRDIFF_TYPE__ int
#define __LDBL_MIN_10_EXP__ (-4931)
#define __LDBL_DIG__ 18
' , 
	"additional definitions"
'
#define ptrdiff_t long
#define restrict
#define throw()
#define __const const
#define __inline inline
#define __inline__ inline
#define __restrict
#define __signed signed
#define __STDC__
#define __volatile
'.
%
category: 'directives'
method:
define: defineToken

	| expansionTokens nameToken name parameters list string token |
	readStream peek == Character lf ifTrue: [ defineToken error: 'unexpected end of line' ].
	(nameToken := self nextToken) isIdentifierToken ifFalse: [ nameToken error: 'expected an identifier'].
	name := nameToken value.
	self definitionsAt: name put: defineToken.
	list := { defineToken . nameToken }.	"collect the tokens so we can extract the source"
	nameToken peek == $( ifTrue: [			"capture parameters"
		list add: self nextToken.
		parameters := Array new.
		token := list add: self nextToken.
		[
			token isCloseParenthesisToken.
		] whileFalse: [
			token isIdentifierToken ifTrue: [
				parameters add: token value.
				(token := list add: self nextToken) isCommaToken ifTrue: [
					token := list add: self nextToken.
				].
			] ifFalse: [
				3 timesRepeat: [
					token isDotToken ifFalse: [self error: 'Expected identifier or dots'].
					token := list add: self nextToken.
				].
			].
		].
	].
	string := String new.
	expansionTokens := {}.
	[
		(token := self nextTokenOnLine) ~~ nil .
	] whileTrue: [
		expansionTokens add: token.
		string add: token source.
	].
	defineToken collectOriginalFrom: list.
	defineToken source add: string.
	defineToken
		defineParameters: parameters
		expansionTokens: expansionTokens.
%
category: 'accessors'
method:
definitions

	^definitions.
%
method:
path
	^ path
%
! fix 48054
method:
searchPaths
  "Return the list of directories used to search for a file that is the
   argument to includePath: .
   If using the native cpp , does not include the system search paths
   which are also used to search for file specified by #include directives.
   See also  CPreprocessor >> allSearchPaths "

  ^ searchPaths
%
method:
allSearchPaths
  "Return the list of directories used to search for a file that
   is the argument of a #include directive."
  | res |
  (res := searchPaths copy) addAll: systemSearchPaths .
  ^ res
%

category: 'other'
method:
directory

	path ifNil: [^''].
	path size to: 1 by: -1 do: [:i | 
		(path at: i) == $/ ifTrue: [
			^path copyFrom: 1 to: i.
		].
	].
	^''.
%
category: 'directives'
method:
doIf: aBoolean

	| inTrueBlock anyWereTrue token ifDepth |
	inTrueBlock := aBoolean.
	anyWereTrue := false.
	[true] whileTrue: [
		inTrueBlock ifTrue: [
			anyWereTrue := true.
			[
				(token := self nextToken) isConditionalEndDirective.
			] whileFalse: [
				self applyToken: token.
			].
			(readStream atEnd or: [readStream peek == Character lf or: [token value = 'elif']]) ifFalse: [
         token error:'expected ''elif'' or end of line '
      ].
			token value = 'endif' ifTrue: [
				self ignoreRestOfLine.
				^self.
			].
			token value = 'else' ifTrue: [
				self ignoreRestOfLine.
				inTrueBlock := false.
			] ifFalse: [
				self ignoreRestOfLine.
				(anyWereTrue and: [token value = 'elif']) ifTrue: [
					inTrueBlock := false.
				] ifFalse: [
					self halt.
				].
			].
		] ifFalse: [		"not inTrueBlock"
			ifDepth := 0.
			token := self skipWhile: [:each | 
				each isConditionalBeginDirective ifTrue: [ifDepth := ifDepth + 1].
				0 < ifDepth ifTrue: [
					each isEndifDirective ifTrue: [ifDepth := ifDepth - 1].
					true.
				] ifFalse: [
					each isConditionalEndDirective not.
				].
			].
			(readStream atEnd or: [readStream peek == Character lf or: [token value = 'if' or: [token value = 'elif']]]) ifFalse: [ token error:'expected ''if'' or ''elif'' or end of line' ].
			token value = 'endif' ifTrue: [^self].
			token value = 'else' ifTrue: [
				inTrueBlock := anyWereTrue not.
			] ifFalse: [
				token value = 'if' ifTrue: [
					self skipWhile: [:each | each isEndifDirective not].
					readStream peek == Character lf ifFalse: [token error: 'missing end of line' ].
				] ifFalse: [
					token value = 'elif' ifTrue: [
						inTrueBlock := self evaluateIfOrElif: token.
					] ifFalse: [
						self halt.
					].
				].
			].
		].
	].
%
category: 'expression evaluation'
method:
evaluate: aStream
	"Returns an Integer from a stream of tokens"

	| expr value |
	expr := self parse: aStream.
	value := self valueOf: expr.
	^value.
%
category: 'expression evaluation'
method:
evaluateHeader: aCHeader

	header := aCHeader.
	definitions := aCHeader types.
	^self evaluate: header.
%
category: 'directives'
method:
evaluateIfOrElif: aToken

	| oldTokens stream expression value lf |
	oldTokens := tokens.
	tokens := { aToken } .
        lf := Character lf .
	[
		readStream atEnd or: [readStream peek == lf ].
	] whileFalse: [
		self applyToken: self nextToken.
	].
	stream := CPreprocessorStream on: tokens.
	tokens := oldTokens.
	stream next.
	expression := self parse: stream.
	value := self valueOf: expression.
	^value ~= 0.
%
category: 'other'
method:
expandMacroFunctionArguments: aList

	| list |
	list := aList collect: [:eachArray | 
		| string |
		string := String new.
		eachArray do: [:each | string add: each source].
		string.
	].
	^list select: [:each | each notEmpty].
%
category: 'directives'
method:
if: aToken

	self doIf: (self evaluateIfOrElif: aToken).
%
category: 'directives'
method:
ifdef: aToken defined: aBoolean

	| nameToken name |
	readStream peek == Character lf ifTrue: [ aToken error:'unexpected end of line' ].
	(nameToken := self nextToken) isIdentifierToken ifFalse: [ nameToken error:'expected an identifier'].
	name := nameToken value.
	readStream peek == Character lf ifFalse: [ nameToken error:'missing end of line' ].
	self doIf: (definitions includesKey: name) == aBoolean.
%
category: 'directives'
method:
ignoreRestOfLine

	self addRestOfLineTo: { } 
%
category: 'directives'
method:
include: aToken afterMe: aBoolean
	| token string existingPath fullPath |
	aBoolean ifTrue: [| index |
			index := path indexOfLastByte: $/ codePoint startingAt: path size.
			existingPath := path copyFrom: 1 to: index].
	token := self nextToken.
	aToken source add: token source.
	token isStringToken ifTrue: [
     "If including a name in double quotes, first try getting the file from the same 
      directory as the file containing the include."
			string := self directory , token value.
			(GsFile existsOnServer: string) ifTrue: [^self includePath: string].
			string := token value
  ] ifFalse: [
    token isOpenAngleBracketToken ifFalse: [token error: 'expected ''<'' '].
		string := String new.
		[(token := self nextToken) isCloseAngleBracketToken] whileFalse:[
      "we want the source rather than the valueString since 'gnu/stubs-64.h' 
          ends up with an integer token of '64.h'!"
			string add: token source.
			aToken source add: token source]
  ].
	fullPath := self searchForInclude: string excluding: existingPath.
	fullPath ifNotNil: [^self includePath: fullPath]
		ifNil: [
      string = 'stdarg.h' ifTrue: [^self].	"This appears to be missing from some systems and can be ignored without problem!"
		  UserDefinedError signal: 'Include file ' , string , ' not found!']
%

method:
nativeCppCmd
  | arch cmd archInt verRpt |
  "Returns a String, the native C preprocessor command , 
   or nil if the native C preprocessor is not used on the platform running the session."
  
  "false ifTrue:[ ^ nil ]." "uncomment to not use native cpp ."

  archInt := 0 .
  arch := (verRpt := System gemVersionReport) at: 'gsBuildArchitecture' .
  arch = 'x86-64 (Linux)' ifTrue:[
    cmd := '/usr/bin/cpp' .
    (GsFile existsOnServer: cmd ) ifFalse:[
       Error signal: cmd , ' not found,  package cpp must be installed on host running the gem or topaz -l process'.
    ].
    cmd := cmd , ' ' .
    archInt := 50.
  ] ifFalse:[
  arch = 'Darwin (macOS)' ifTrue:[
    cmd := '/usr/bin/clang'.
    (GsFile existsOnServer: cmd ) ifFalse:[
       Error signal: cmd , ' not found,  clang must be installed on host running the gem or topaz -l process'.
    ].
    "Need to define _ANSI_SOURCE so we don't get enumerator_attributes feature that causes
     trouble for us in /usr/include/time.h on __CLOCK_AVAILABILITY on Darwin 17."
    cmd := cmd , ' -D_ANSI_SOURCE=1 -E -x c ' .
    archInt := 42 .
  ] ifFalse:[
  (arch at: 1 equals: 'RISC 6000 (AIX' ) ifTrue:[
    "NOTE:  AIX cpp does not allow C++ style comments beginning with //  .
      /usr/bin/cpp is not the preprocessor built into xlc . "
    cmd := '/usr/bin/cpp' .
    (GsFile existsOnServer: cmd ) ifFalse:[
       Error signal: cmd , ' not found,  /usr/bin/cpp must be installed on host running the gem or topaz -l process'.
    ].
    archInt := 9 .
    cmd := cmd , ' -DAIX=1 '. "amazingly cpp does not predefine 'AIX' " 
    (verRpt at: 'osVersion') = '6' ifTrue:[ cmd addAll: ' -DFLG_AIX_VERSION=61 -D__STDC__=1 '] 
                  ifFalse:[ cmd addAll: ' -DFLG_AIX_VERSION=71 ' ].
  ]]].
  cmd ifNotNil:[
    tmpFnames := { } .
    cppArchMType := archInt .
    searchPaths ifNotNil:[:paths | paths do:[:str | 
       cmd addAll: ' -I'; 
           addAll: (str size == 0 ifTrue:[ '.' ] ifFalse:[str]);
            add: $  ""]].
  ].
  ^ cmd
%

category: 'directives'
method:
includePath: aString
  "Start a CPreprocessor using file specified by aString
    or recurse to process argument of an #include  "
	| oldPath oldLine oldStream fName f |
	oldPath := path.
	oldLine := line.
	oldStream := readStream.
	path := aString.
	line := 1.
  fName := GsFile _expandFilename: aString isClient: false .
    (f := GsFile openReadOnServer: fName) ifNil: [ 
 	    Error signal: 'cannot open file ', fName , '; ', GsFile serverErrorString.
  ].
  [ self nativeCppCmd ifNotNil:[:cmd|  | cppOut |
      f close .
      cppOut := GsHostProcess execute: ( cmd  , fName) encodeAsUTF8 .
		  (cppOut at: 1 equals:'WARNING') ifTrue:[ 
         Error signal: 'non-empty stderr from ', cmd, ', ' , cppOut
      ].
      readStream := self _streamOnCppOutput: cppOut .
      path := aString "file name to associate with error numbers"
    ] ifNil:[
      readStream := CPreprocessorStream on: f contents .
      f close .
      readStream file: path ; line: 1 .
    ] .
		self readTillEnd.
	] ensure: [
		readStream close.
		readStream := oldStream.
		line := oldLine.
    oldPath ifNotNil:[ path := oldPath ].
	].
%
method:
parseString: aString ignoreWarnings: aBoolean 
  | cmd |
  (cmd := self nativeCppCmd) ifNotNil:[ 
    | cppOut cppRes cppErr cppInStr |
    cppInStr := aString .
    (cmd includesString: 'clang') ifTrue:[ 
      "Darwin clang will not read from stdin"
      | tmpIn |
      tmpIn := '/tmp/cppIn', Random new integer asString , '.s' .
      (GsFile openWriteOnServer: tmpIn ) nextPutAll: aString ; flush ; close. 
      cmd := cmd , ' ' , tmpIn .
      cppInStr := nil .
      tmpFnames add: tmpIn  . "for later deletion"
    ].
    cppRes := GsHostProcess _execute: cmd  input: aString .
    cppOut := cppRes at: 1 .
    cppErr := cppRes at: 2 .
    readStream := self _streamOnCppOutput: cppOut .
    aBoolean ifFalse:[ cppErr size > 0 ifTrue:[
      Error signal: 'non-empty stderr from ',cmd, ', ', cppErr 
    ]].
  ] ifNil: [
	  readStream := CPreprocessorStream on: aString.
  ].
	line := 1.
	self readTillEnd.
%
category: 'Private'
method:
_streamOnCppOutput: aString
  | strm tmpOut |
  "write stdout of cpp to a file for use in debugging the preprocessor."
  tmpOut := '/tmp/cppOut', Random new integer asString , '.s' .
  (GsFile openWriteOnServer: tmpOut ) nextPutAll: aString ; flush ; close. 
  tmpFnames add: tmpOut  . "for later deletion"
  
  (strm := CPreprocessorStream on: aString )
    cppArchMType:  cppArchMType; file: path ; line: 1 .
  ^ strm .
%

category: 'other'
method:
parseString: aString
 ^ self parseString: aString ignoreWarnings: false
%

category: 'Private'
method:
_parseStringNoCpp: aString
  readStream := CPreprocessorStream on: aString.
  line := 1.
  self readTillEnd.
%

category: 'other'
method:
initialize
	tokens := { } .
	self _initializePaths .
	self initializeDefinitions .
%

method:
_init
| str |
str := self defaultLinuxDefinitions .
definitions := KeyValueDictionary new .
tokens := { } .
self _parseStringNoCpp: str .
^ self
%

category: 'default definitions'
method:
initializeDefinitions
	| defs tmpsKey |
  "Do not cache definitions in the repository ; sessions could be executing on
   hosts from a mix of architectures."
  defs := SessionTemps current at: (tmpsKey := #GemStone_CPreprocessorInitialDefinitions) 
                               otherwise: nil.
  defs ifNil:[ | arch string |
	  arch := System gemVersionReport at: 'gsBuildArchitecture'.
    string := 
      arch = 'x86-64 (Linux)' ifTrue: [self defaultLinuxDefinitions] ifFalse: [
		  arch = 'Darwin (macOS)' ifTrue: [self defaultDarwinDefinitions] ifFalse: [
		  arch = 'SPARC (Solaris)' ifTrue: [self defaultSparcSolarisDefinitions] ifFalse: [ 
		  arch = 'x86-64 (Solaris i386)' ifTrue: [self defaultX86SolarisDefinitions] ifFalse: [
		  arch = 'RISC 6000 (AIX 7.2)' ifTrue: [self defaultAix72Definitions] ifFalse: [
		  arch = 'RISC 6000 (AIX 7.1)' ifTrue: [self defaultAix71Definitions] ifFalse: [
		  arch = 'RISC 6000 (AIX 6.1)' ifTrue: [self defaultAix61Definitions] ifFalse: [
		    Error signal: arch asString , ' build type is not (yet?) supported']]]]]]
      ].
    definitions := KeyValueDictionary new . 
    self _parseStringNoCpp: string. "fill in definitions"
    defs := definitions .
    SessionTemps current at: tmpsKey put: defs .
  ].
  definitions := defs copy.
%

! fix 48056
category: 'other'
method:
insertSearchPath: aString
  "Install the given path at the front of the list of search paths."

  searchPaths insertObject: aString at: 1
%
category: 'expression evaluation'
method:
integerFromBoolean: aBoolean

	^aBoolean 
		ifTrue: [1] 
		ifFalse: [0].
%
category: 'other'
method:
macroFunctionArgumentsFor: aToken

	| arguments index parenthesisCount list token |
	(token := self nextToken) isOpenParenthesisToken ifFalse: [ token error: 'expected ''('' ' ].
	list := { aToken . token }.
	arguments := { } .
	parenthesisCount := 1.
	index := 3.
	[
		token := self nextToken.
		(parenthesisCount == 1 and: [token isCommaToken]) ifTrue: [
			arguments add: (list copyFrom: index to: list size). 
			index := list size + 2.
		] ifFalse: [
			token isOpenParenthesisToken ifTrue: [
				parenthesisCount := parenthesisCount + 1.
			] ifFalse: [
				token isCloseParenthesisToken ifTrue: [parenthesisCount := parenthesisCount - 1].
			].
		].
		list add: token.
		0 < parenthesisCount.
	] whileTrue: [].
	arguments add: (list copyFrom: index to: list size - 1).
	aToken collectOriginalFrom: list.
	^arguments.
%
category: 'expression evaluation'
method:
mergeLastTwoElementsIn: stack

	| x y token |
	y := stack removeLast.
	x := stack last.
	token := (y at: 2) ->  {  (x at: 3) . (y at: 3)} .
	x at: 3 put: token.
%
category: 'other'
method:
next

	tokens isEmpty ifTrue: [^nil].
	^tokens removeFirst.
%
category: 'other'
method:
nextToken
		"This can return nil if there are no more tokens.
		The stream might not be atEnd because of trailing comments."

	| token |
	token := self cPreprocessorTokenSpecies
		nextFrom: readStream 
		filename: path 
		line: line.
	token ifNotNil: [line := token line].
	^token.
%
category: 'other'
method:
nextTokenOnLine
		"This can return nil if there are no more tokens.
		The stream might not be atEnd because of trailing comments."

	| token |
	token := self cPreprocessorTokenSpecies
		nextOnCurrentLineFrom: readStream 
		filename: path 
		line: line.
	token ifNotNil: [line := token line].
	^token.
%
category: 'expression evaluation'
method:
numberFrom: aString
	"Returns an number literal token"

	| string value isLong isUnsigned |
	isLong := false.
	isUnsigned := false.
	(2 < aString size and: [(aString at: 1) == $0 and: [(aString at: 2) = $x]]) ifTrue: [
		string := '16r' , (aString copyFrom: 3 to: aString size).
	] ifFalse: [
		aString first == $0 ifTrue: [
			string := '8r' , aString.
		] ifFalse: [
			string := aString.
		].
	].
	('lL' includes: string last) ifTrue: [
		isLong := true.
		string := string copyFrom: 1 to: string size - 1.
	].
	('uU' includes: string last) ifTrue: [
		isUnsigned := true.
		string := string copyFrom: 1 to: string size - 1.
		('lL' includes: string last) ifTrue: [
			isLong := true.
			string := string copyFrom: 1 to: string size - 1.
		].
	].
	value := Integer fromString: string.
	(value class ~~ SmallInteger and: [16rFFFFFFFFFFFFFFFF < value]) ifTrue: [self halt].
	(isLong or: [16rFFFFFFFF < value]) ifTrue: [
		(isUnsigned or: [16r7FFFFFFFFFFFFFFF < value]) 
			ifTrue: [^self cPreprocessorTokenSpecies uint64: value]
			ifFalse: [^self cPreprocessorTokenSpecies int64: value].
	].
	16rFFFF < value ifTrue: [
		(isUnsigned or: [16r7FFFFFFF < value]) 
			ifTrue: [^self cPreprocessorTokenSpecies uint32: value]
			ifFalse: [^self cPreprocessorTokenSpecies int32: value].
	].
	^(isUnsigned or: [16r7FFF < value]) 
		ifTrue: [self cPreprocessorTokenSpecies uint16: value]
		ifFalse: [self cPreprocessorTokenSpecies int16: value].
%
category: 'expression evaluation'
method:
numberLiteralKeys

	^#(
		#'int8' #'uint8' #'int16' #'uint16' #'int32' #'uint32' #'int64' #'uint64'
	).
%
category: 'expression evaluation'
method:
operandFrom: aStream
	"Returns a token"

	| token |
	token := aStream next.
	token key = #'integer' ifTrue: [^token].
	token key = #'number' ifTrue: [^self numberFrom: token value].
	token key = #'defined' ifTrue: [^#'integer' -> (self integerFromBoolean: (definitions includesKey: token value))].
	token key = #'punctuator' ifTrue: [
		token value = '!' ifTrue: [
			token := self operandFrom: aStream.
			token value: (token value == 1 ifTrue: [0] ifFalse: [1]).
			^token.
		].
		token value = '(' ifTrue: [
			| value |
			value := self evaluate: aStream.
			token := aStream next.
			(token key = #'punctuator' and: [token value = ')']) ifTrue: [^#'integer' -> value].
self halt.
		].
self halt.
	].
	token key = #'identifier' ifTrue: [
		| definition |
		definition := definitions
			at: token value
			ifAbsent: [^#'integer' -> 0].
		definition key ifNil: [
			^#'integer' -> (self evaluate: (CPreprocessorStream on: definition value)).
		] ifNotNil: [
			self halt.
		].
	].
self halt.
%
category: 'expression evaluation'
method:
parse: aStream
	"Returns a token from a stream of tokens"

	| list stack |
	list := self parseA: aStream.		"Each element is a three-element array: { precedence. operator. operand }"
	stack := { list removeFirst } .
	1 to: list size do: [:i | 
		| each |
		each := list at: i.
		[
			stack last first >= each first.
		] whileTrue: [
			self mergeLastTwoElementsIn: stack.
		].
		stack add: each.
	].
	[
		1 < stack size.
	] whileTrue: [
		self mergeLastTwoElementsIn: stack.
	].
	^stack first last.
%
category: 'expression evaluation'
method:
parseA: aStream
	"Returns a list of the following: { precedence. operator. operand }"

	| list |
	list := { { -999 . '' .  (self parseForOperandFrom: aStream) } }.
	[true] whileTrue: [
		| operator precedence |
		(operator := aStream peek) ifNil: [^list].
		operator key == #'punctuator' ifFalse: [^list].
		(precedence := self precedenceOf: operator value) ifNil: [^list].
		aStream next.
		list add: { precedence .  operator value asSymbol .  (self parseForOperandFrom: aStream) }.
	].
%
category: 'expression evaluation'
method:
parseForDefinedFrom: aStream

	| token isFunction key found |
	(isFunction := (token := aStream next) isOpenParenthesisToken) ifTrue: [
		token := aStream next.
	].
	token isIdentifierToken ifFalse: [token error:'expected an identifier'].
	key := token value.
	isFunction ifTrue: [(token := aStream next) isCloseParenthesisToken ifFalse: [ token error:'expected '')'' ']].
	found := definitions includesKey: key.
	^found 
		ifTrue: [#'int16' -> 1]
		ifFalse: [#'int16' -> 0].
%
category: 'expression evaluation'
method:
parseForFunction: aString from: aStream

	| token |
	(token := aStream next) isOpenParenthesisToken ifFalse: [ token error:'expected ''('' '].
	aString = 'sizeof' ifTrue: [^self parseForSizeofFrom: aStream].
	self halt.
%
category: 'expression evaluation'
method:
parseForOperandFrom: aStream
	"Returns an expression token (which is different from a preprocessor token?)"

	| token |
	token := aStream next.
	(self numberLiteralKeys includes: token key) ifTrue: [^token].
	token key == #'integer' 		ifTrue: [^token].
	token key == #'identifier' 		ifTrue: [
		aStream atEnd ifTrue: [^token].
		token isDefinedIdentifierToken ifTrue: [^self parseForDefinedFrom: aStream].
		aStream peek isOpenParenthesisToken ifTrue: [
			^self parseForFunction: token value from: aStream.
		].
		^token.
	].
	token key == #'number' 		ifTrue: [^self numberFrom: token value].
	token key == #'defined' 		ifTrue: [^token].
	token isEmptyToken 			ifTrue: [^token].
	token key == #'character'		ifTrue: [^token].
	token key == #'punctuator' 	ifTrue: [
		token value = '!' 				ifTrue: [
			token := self parseForOperandFrom: aStream.
			^#'not' -> token.
		].
		token value = '-' 				ifTrue: [
			token := self parseForOperandFrom: aStream.
			^#'negate' -> token.
		].
		token value = '(' 				ifTrue: [
			| typeCast endToken |
			(typeCast := self readTypeSpecifierFrom: aStream) ifNil: [
				token := self parse: aStream.
			].
			(endToken := aStream next) isCloseParenthesisToken ifFalse: [endToken error: 'Expected '')'' '].
			typeCast ifNotNil: [
				token := self parseForOperandFrom: aStream.
				token key == typeCast ifTrue: [^token].
				"self halt."
			].
			^token.
		].
	].
	token error: 'unhandled expression token!'.
%
category: 'expression evaluation'
method:
parseForSizeofFrom: aStream

	| class declaration token |
	class := self cDeclarationSpecies .
	declaration := class header: header.
	(token := aStream next) isCloseParenthesisToken ifFalse: [ token error:'expected '')'' '].
	^#'int32' -> declaration byteSize.
%
category: 'expression evaluation'
method:
parseForSizeofStructFrom: aList

	| token fields size |
	aList size = 2 ifFalse: [ Error signal: 'invalid arg size in parseForSizeofStructFrom'].
	(token := aList at: 2) key == #'identifier' ifFalse: [ token error: 'Expected an identifier'].
	fields := header fieldsForStruct: token value.
	size := fields
		inject: 0
		into: [:sum :each | sum + each byteSize].
	^self cPreprocessorTokenSpecies int32: size.
%
category: 'other'
method:
peek

	tokens isEmpty ifTrue: [^nil].
	^tokens first.
%
category: 'expression evaluation'
method:
precedenceOf: aString
	"Answer a negative number so that * (-3) is GREATER than + (-4) "

	(#('*' '/' '%') includes: aString) 			ifTrue: [^-3].
	(#('+' '-') includes: aString) 				ifTrue: [^-4].
	(#('<<' '>>') includes: aString)			ifTrue: [^-5].
	(#('<' '<=' '>' '>=') includes: aString) 	ifTrue: [^-6].
	(#('==' '!=') includes: aString) 			ifTrue: [^-7].
	'&' = aString									ifTrue: [^-8].
	'^' = aString									ifTrue: [^-9].
	'|' = aString									ifTrue: [^-10].
	'&&' = aString									ifTrue: [^-11].
	'||' = aString									ifTrue: [^-12].
	(#('?' ':') includes: aString) 				ifTrue: [^-13].
	^nil.
%
category: 'other'
method:
readTillEnd

	| token |
	[
		(token := self nextToken) ~~ nil .
	] whileTrue: [
		self applyToken: token.
	].
%
category: 'expression evaluation'
method:
readTypeSpecifierFrom: aStream

	| token value signed size |
	(token := aStream peek) key == #'identifier' ifFalse: [^nil].
	value := token value.
	(definitions includesKey: value) ifTrue: [aStream next. ^(definitions at: value) type].
	value = 'struct'		ifTrue: [^nil].
	value = 'void' 			ifTrue: [aStream next. ^aStream peek isStarToken ifTrue: [aStream next. #'ptr'] ifFalse: [#'void']].
	value = 'float' 			ifTrue: [aStream next. ^#'float'		].
	value = '__float128'	ifTrue: [aStream next. ^#'float128'	].
	value = 'double' 		ifTrue: [aStream next. ^#'double'	].
	value = '_Complex' 	ifTrue: [aStream next. ^#'complex'	].		"this essentially means that there are two of something else (https://gcc.gnu.org/onlinedocs/gcc-4.8.0/gcc/Complex.html)"
	(#('signed' 'unsigned') includes: value) ifTrue: [
		signed := value = 'signed'.
		(token := aStream next; peek) key == #'identifier' ifFalse: [token error: 'Expected an identifier'].
		value := token value.
	].
	value = 'char' ifTrue: [
		aStream next.
		^signed == true
		ifTrue: [#'int8']
		ifFalse: [#'uint8'].
	].
	(#('long' 'short') includes: value) ifTrue: [
		size := value.
		(token := aStream next; peek) key == #'identifier' ifFalse: [token error: 'Expected an identifier'].
		value := token value.
	].
	value = 'double' ifTrue: [
		aStream next.
		size = 'long' ifFalse: [token error: 'Unexpected type'].
		^#'longDouble'.
	].
	value = 'int' ifTrue: [
		aStream next.
		^self typeForIntSigned: signed size: size.
	].
	size ifNotNil: [
		^self typeForIntSigned: signed size: size.
	].
	signed ifNotNil: [token error: 'Unexpected type (expected signed == nil) '].
	(signed == nil and: [size == nil ]) ifTrue: [^nil].
	^self
		typeForIntSigned: signed
		size: size.
%
category: 'directives'
method:
removeCommentsFrom: aString for: aToken

	| string i j |
	string := aString.
	[
		0 < (i := string indexOfSubCollection: '/*' startingAt: 1).
	] whileTrue: [
		j := string indexOfSubCollection: '*/' startingAt: i.
		0 == j ifTrue: [ aToken error: 'Expected end of comment'].
		string := (string copyFrom: 1 to: i - 1) , (string copyFrom: j + 2 to: string size).
	].
	^string.
%
category: 'other'
method:
searchForInclude: includePath excluding: skipPath
	"Search our searchPaths for file with the given includePath.
   If includePath begins with $/ , our searchPaths is ignored .
	If skipPath is not nil and matches an entry in our searchPaths,
	only search entries after that one.
	If found, answer the full path, if not answer nil."
  | iArg | 
  iArg := includePath .
  (iArg includesIdentical: $$ ) ifTrue:[
    iArg := GsFile _expandFilename: iArg isClient: false .
    iArg ifNil:[ Error signal:'invalid environment variable in ''', includePath asString, ''''].
  ].
  (iArg size > 0 and:[ (iArg at: 1) == $/]) ifTrue:[
		(GsFile existsOnServer: iArg) ifTrue:[ ^ iArg ].
  ] ifFalse:[
	  | skipIndex paths blk |
	  skipIndex := skipPath ifNil:[ 0 ] ifNotNil:[ searchPaths indexOf: skipPath ].
    paths := { } .
	  skipIndex + 1 to: searchPaths size do: [:idx | paths add: (searchPaths at: idx)].
    blk := [:str | | aPath fullPath |
      aPath := str . 
      aPath size == 0 ifTrue:[ aPath := GsFile serverCurrentDirectory ].
      aPath last == $/ ifFalse:[ aPath := aPath , $/ ].
			fullPath := aPath , iArg.
      (fullPath includesIdentical: $$ ) ifTrue:[
        fullPath := GsFile _expandFilename: fullPath isClient: false .
      ].
			(GsFile existsOnServer: fullPath) ifTrue:[ ^fullPath ] .
    ].
    paths do: blk .
    systemSearchPaths do: blk .
  ].
	^nil
%
category: 'directives'
method:
skipWhile: aBlock

	| token |
	[
		token := self nextToken.
		aBlock value: token.
	] whileTrue: [].
	^token.
%
category: 'accessors'
method:
tokens

	^tokens.
%
category: 'expression evaluation'
method:
typeForIntSigned: aBooleanOrNil size: aStringOrNil
"
// From Linux, we have the following: 
#define __SIZEOF_SHORT__ 2
#define __SIZEOF_INT__ 4
#define __SIZEOF_LONG__ 8
#define __SIZEOF_LONG_LONG__ 8
"
	aStringOrNil = 'long' ifTrue: [
		^aBooleanOrNil == false 
			ifTrue: [#'uint64']
			ifFalse: [#'int64']. 
	].
	aStringOrNil = 'short' ifTrue: [
		^aBooleanOrNil == false 
			ifTrue: [#'uint16']
			ifFalse: [#'int16']. 
	].
	^aBooleanOrNil == false 
		ifTrue: [#'uint32']
		ifFalse: [#'int32'].
%
category: 'directives'
method:
undef: defineToken

	| nameToken name |
	readStream peek == Character lf ifTrue: [ defineToken error: 'unexpected end of line'].
	(nameToken := self nextToken) isIdentifierToken ifFalse: [ nameToken error: 'expected an identifier'].
	readStream peek == Character lf ifFalse: [ nameToken error:'missing end of line' ].
	name := nameToken value.
	definitions 
		removeKey: name
		ifAbsent: [].
%
category: 'expression evaluation'
method:
valueOf: aToken
	"Returns an Integer from a token."

	| x y key |
	key := aToken key.
	(self numberLiteralKeys includes: key)		ifTrue: [^aToken value].
	key == #'character'	ifTrue: [^aToken value codePoint].
	key == #'empty'		ifTrue: [^0].
	key == #'identifier' 	ifTrue: [
		| assoc |
		assoc := definitions 
			at: aToken value 
			ifAbsent: [nil].
		assoc ifNil: [
			header ifNil: [^0].
			^header enums 
				at: aToken value 
				ifAbsent: [0].
		].
		^self evaluate: (CPreprocessorStream on: assoc value)].
	key == #'defined' ifTrue: [^self integerFromBoolean: (definitions includesKey: aToken value)].
	key == #'not'		ifTrue: [^self integerFromBoolean: (self valueOf: aToken value) = 0].
	key == #'negate'	ifTrue: [^(self valueOf: aToken value) negated].
	key == #'&&' 		ifTrue: [
		(self valueOf: (aToken value at: 1)) = 0 ifTrue: [^0].
		^self valueOf: (aToken value at: 2). 
	].
	key == #'||' 			ifTrue: [
		(self valueOf: (aToken value at: 1)) = 1 ifTrue: [^1].
		^self valueOf: (aToken value at: 2). 
	].
	x := self valueOf: (aToken value at: 1).
	y := self valueOf: (aToken value at: 2).
	key == #'=='	ifTrue: [^self integerFromBoolean: x = y].
	key == #'!='	ifTrue: [^self integerFromBoolean: x ~= y].
	key == #'>='	ifTrue: [^self integerFromBoolean: x >= y].
	key == #'>'		ifTrue: [^self integerFromBoolean: x > y].
	key == #'<'		ifTrue: [^self integerFromBoolean: x < y].
	key == #'<='	ifTrue: [^self integerFromBoolean: x <= y].
	key == #'*'		ifTrue: [^x * y].
	key == #'+'		ifTrue: [^x + y].
	key == #'/'		ifTrue: [^x // y].
	key == #'-'		ifTrue: [^x - y].
	key == #'%'	ifTrue: [^x \\ y].
	key == #'<<'	ifTrue: [^x bitShift: y].
	key == #'>>'	ifTrue: [^x bitShift: y negated].
	key == #'?'		ifTrue: [^x == 1 ifTrue: [y] ifFalse: [nil]].
	key == #':'		ifTrue: [^x ifNotNil: [x] ifNil: [y]].
	key == #'|'		ifTrue: [^x bitOr: y].
	key == #'&'		ifTrue: [^x bitAnd: y].

	self halt.
%

! bug 47422
category: 'Class Membership'
method:
cPreprocessorTokenSpecies
"Answer the class to be used for CPreprocessorToken objects. Subclasses may
 overload this method as needed."
^ CPreprocessorToken
%

category: 'Class Membership'
method:
cDeclarationSpecies
"Answer the class to be used for CDeclaration objects. Subclasses may
 overload this method as needed."
^ CDeclaration
%
