! ========================================================================
! Copyright (C) by GemTalk Systems 1991-2020.  All Rights Reserved
! ========================================================================

! ------------------- Class definition for CPreprocessorToken
expectvalue /Class
doit
Object subclass: 'CPreprocessorToken'
  instVarNames: #( file line source
                    type value peek cppMType )
  classVars: #()
  classInstVars: #()
  poolDictionaries: #()
  inDictionary: Globals
  options: #()

%
expectvalue /Class
doit
CPreprocessorToken comment: 
'This class is internal to the FFI implementation and 
represents a preprocessor token from a C header file.'
%
expectvalue /Class
doit
CPreprocessorToken category: 'FFI'
%
! ------------------- Remove existing behavior from CPreprocessorToken
expectvalue /Metaclass3       
doit
CPreprocessorToken removeAllMethods.
CPreprocessorToken class removeAllMethods.
%
set compile_env: 0
set class CPreprocessorToken
! ------------------- Class methods for CPreprocessorToken
category: 'other'
classmethod:
args: anArray
	| res |
	(res := self basicNew) initializeArgs: anArray .
	^ res
%
category: 'other'
classmethod:
concatenation
	| res |
	(res := self basicNew) initializeConcatenation .
	^ res
%
category: 'other'
classmethod:
define: aString
	| res |
	(res := self basicNew) initializeDefine: aString .
    ^ res
%
category: 'other'
classmethod:
directive: aString
	| res |
	(res := self basicNew) initializeDirective: aString .
    ^ res
%
category: 'other'
classmethod:
empty
	| res |
	(res := self basicNew) initializeEmpty .
	^ res
%
category: 'other'
classmethod:
identifier: aString
	| res |
	(res := self basicNew) initializeIdentifier: aString .
	^ res
%
category: 'other'
classmethod:
int16: anInteger
	| res |
	(res := self basicNew ) initializeInt16: anInteger .
	^ res
%
category: 'other'
classmethod:
int32: anInteger
	| res |
	(res := self basicNew) initializeInt32: anInteger .
	^ res
%
category: 'other'
classmethod:
int64: anInteger
	| res |
	(res := self basicNew) initializeInt64: anInteger .
	^ res
%
category: 'other'
classmethod:
nextFrom: aStream filename: aString line: anInteger
	"Note that this might return nil if no more tokens!"

	^self
		nextFrom: aStream
		filename: aString
		line: anInteger
		onOneLogicalLine: false
%
category: 'other'
classmethod:
nextFrom: aStream filename: aString line: anInteger onOneLogicalLine: aBoolean
	"Note that this might return nil if no more tokens!"

	^self basicNew
		initializeStream: aStream
		filename: aString
		line: anInteger
		singleLogicalLine: aBoolean
%
category: 'other'
classmethod:
nextOnCurrentLineFrom: aStream filename: aString line: anInteger
	"Note that this might return nil if no more tokens!"

	^self
		nextFrom: aStream
		filename: aString
		line: anInteger
		onOneLogicalLine: true
%
category: 'other'
classmethod:
number: aNumber
	| res |
	(res := self basicNew) initializeNumber: aNumber .
	^ res
%
category: 'other'
classmethod:
punctuator: aString
	| res |
	(res := self basicNew) initializePunctuator: aString .
	^ res
%
category: 'other'
classmethod:
string: aString
	| res |
	(res := self basicNew) initializeString: aString .
	^ res
%
category: 'other'
classmethod:
uint16: anInteger
	| res |
	(res := self basicNew) initializeUInt16: anInteger .
	^ res
%
category: 'other'
classmethod:
uint32: anInteger
	| res |
	(res := self basicNew) initializeUInt32: anInteger .
	^ res
%
category: 'other'
classmethod:
uint64: anInteger
	| res |
	(res := self basicNew) initializeUInt64: anInteger.
	^ res
%
! ------------------- Instance methods for CPreprocessorToken
category: 'other'
method:
asAssociation
	"To facilitate automated tests"
	^type -> value.
%
category: 'other'
method:
collectOriginalFrom: aList

	| string |
	string := String new.
	aList do: [:each | string add: each source].
	source := string.
%
category: 'other'
method:
concatenate: aToken

	value add: aToken source trimSeparators.
%
category: 'Initializing'
method:
defineParameters: parametersArray expansionTokens: expansionTokens

	| expansionArray i j string |
	type := #'define'.
	parametersArray ifNil: [
		value := expansionTokens
			inject: String new
			into: [:priorString :token | priorString add: token source; yourself].
		^self.
	].
	expansionArray := {}.
	string := String new.
	expansionTokens do: [:each | 
		(each isIdentifierToken and: [0 < (i := parametersArray indexOf: each value)]) ifTrue: [
			j := each source indexOfSubCollection: each value startingAt: 1.
			1 < j ifTrue: [string add: (each source copyFrom: 1 to: j - 1)].
			expansionArray add: string; add: i.
			string := String new.
			j + each value size - 1 < each source size ifTrue: [string add: (each source copyFrom: j + each value size to: each source size)].
		] ifFalse: [
			string add: each source.
		].
	].
	expansionArray add: string.
	value := expansionArray.
%
category: 'Errors'
method:
error: aString

  "aString is typically of the form 'Expected semicolon ...' "

  Error signal: (aString , ', found ''' , self printString , '''', self positionString)
%
category: 'other'
method:
expandArguments: arguments

	| string |
	string := String new.
	value do: [:each1 | 
		(each1 isKindOf: Integer) ifTrue: [	"substitution"
			| flag |
			flag := string isEmpty or: [
				(string notEmpty and: [string last ~~ $#]) or: [
				2 < string size and: [(string copyFrom: string size - 1 to: string size) = '##']]].
			flag  ifTrue: [	"simple case"
				string add: (arguments at: each1).
			] ifFalse: [		"# macro preprocessing operator creates a string"
				string at: string size put: $".		"replace the # token with the start of a string"
				(arguments at: each1) trimSeparators do: [:each2 | 
					('\"' includes: each2) ifTrue: [string add: $\].
					string add: each2.
				].
				string add: $".
			].
		] ifFalse: [
			string add: each1.
		].
	].
	^string
%
category: 'Accessing'
method:
file

	^file.
%
category: 'Reading'
method:
hexCharacterFrom: aStream
	"Enter with pointing to 'x' of '\0xhh'"

	| string |
	string := String withAll: '16r0'.
	[true] whileTrue: [
		| char |
		char := self peekFrom: aStream.
		(string size < 6 and: [char isAlphaNumeric]) ifFalse: [
			value add: (Character codePoint: (Integer fromString: string)).
			^self.
		].
		string add: char.
		source add: aStream next.
	].
%
category: 'Initializing'
method:
initializeArgs: anArray

	type := #'args'.
	value := anArray.
%
category: 'Initializing'
method:
initializeConcatenation

	type := #'concatenation'.
%
category: 'Initializing'
method:
initializeDefine: aString

	type := #'define'.
	value := aString.
%
category: 'Initializing'
method:
initializeDirective: aString

	type := #'directive'.
	value := aString.
%
category: 'Initializing'
method:
initializeEmpty

	type := #'empty'.
%
category: 'Initializing'
method:
initializeIdentifier: aString

	type := #'identifier'.
	value := aString.
%
category: 'Initializing'
method:
initializeInt16: anInteger

	type := #'int16'.
	value := anInteger.
%
category: 'Initializing'
method:
initializeInt32: anInteger

	type := #'int32'.
	value := anInteger.
%
category: 'Initializing'
method:
initializeInt64: anInteger

	type := #'int64'.
	value := anInteger.
%
category: 'Initializing'
method:
initializeNumber: aNumber

	type := #'number'.
	value := aNumber.
%
category: 'Initializing'
method:
initializePunctuator: aString

	type := #'punctuator'.
	value := aString.
%

method:
initializeStream: aStream filename: aString line: anInteger singleLogicalLine: aBoolean 
		"The return value from the class-side method will be what this method returns, not always a new object"
	| isBeginningOfLine char1 char2 |
	file := aStream file ifNil:[ aString ].
	line := anInteger. 
  cppMType := aStream cppArchMType ifNil:[ 0 ] .
	source := String new.
  cppMType ~~ 0 ifTrue:[ (self skipCppDebugInfo: aStream ) ifNil:[ ^ nil ]].
	isBeginningOfLine := self isBeginningOfLineFor: aStream.
	aBoolean
		ifTrue: [self skipSingleLogicalLineCommentsInStream: aStream ifNothingFound: [^nil]]
		ifFalse:[ self skipMultipleBlankLinesAndCommentsInStream: aStream].
  (char1 := self peekFrom: aStream) == nil  ifTrue: [^nil].
	char2 := aStream peek2.

	(char1 == $L and: [char2 == $']) ifTrue: [self readWideCharacterFrom: aStream] ifFalse: [
	(char1 == $L and: [char2 == $"]) ifTrue: [self readWideStringFrom: aStream] ifFalse: [
	char1 isLetter	ifTrue: [	self readIdentifierFrom: aStream] ifFalse: [
	char1 == $_	ifTrue: [	self readIdentifierFrom: aStream] ifFalse: [
	char1 == $" 	ifTrue: [	self readStringFrom: aStream] ifFalse: [
	char1 == $' 	ifTrue: [	self readCharacterFrom: aStream] ifFalse: [
  char1 == $#   ifTrue: [ self readHashFrom: aStream isBeginningOfLine: isBeginningOfLine] 
               ifFalse: [
  char1 isDigit   ifTrue: [ self readNumberFrom: aStream] ifFalse: [
                  self readPunctuatorFrom: aStream]]]]]]]].
	"Include trailing comments starting on this line with the source for this token"
	self skipCommentsIn: aStream.
%
category: 'Initializing'
method:
initializeString: aString

	type := #'string'.
	value := aString.
%
category: 'Initializing'
method:
initializeUInt16: anInteger

	type := #'uint16'.
	value := anInteger.
%
category: 'Initializing'
method:
initializeUInt32: anInteger

	type := #'uint32'.
	value := anInteger.
%
category: 'Initializing'
method:
initializeUInt64: anInteger

	type := #'uint64'.
	value := anInteger.
%
category: 'Testing'
method:
isAttributeIdentifierToken

	^type == #'identifier' and: [value = '__attribute__'].
%
category: 'other'
method:
isBeginningOfLineFor: aStream

	aStream atBeginning ifTrue: [^true].
	^aStream peek == Character lf.
%
category: 'Testing'
method:
isCloseAngleBracketToken

	^type == #'punctuator' and: [value = '>'].
%
category: 'Testing'
method:
isCloseCurlyBracketToken

	^type == #'punctuator' and: [value = '}'].
%
category: 'Testing'
method:
isCloseParenthesisToken

	^type == #'punctuator' and: [value = ')'].
%
category: 'Testing'
method:
isCloseSquareBracketToken

	^type == #'punctuator' and: [value = ']'].
%
category: 'Testing'
method:
isColonToken

	^type == #'punctuator' and: [value = ':'].
%
category: 'Testing'
method:
isCommaToken

	^type == #'punctuator' and: [value = ','].
%
category: 'Testing'
method:
isConcatenationToken

	^type == #'concatenation'.
%
category: 'Testing'
method:
isConditionalBeginDirective

	| list |
	list := #('if' 'ifdef' 'ifndef').
	^type == #'directive' and: [list includes: value].
%
category: 'Testing'
method:
isConditionalDirective

	| list |
	list := #('if' 'else' 'endif' 'ifdef' 'ifndef' 'elif').
	^type == #'directive' and: [list includes: value].
%
category: 'Testing'
method:
isConditionalEndDirective

	| list |
	list := #('else' 'endif' 'elif').
	^type == #'directive' and: [list includes: value].
%
category: 'Testing'
method:
isDefinedIdentifierToken

	^type == #'identifier' and: [value = 'defined'].
%
category: 'Testing'
method:
isDirectiveToken

	^type == #'directive'.
%
category: 'Testing'
method:
isDotToken

	^type == #'punctuator' and: [value = '.'].
%
category: 'Testing'
method:
isEmptyToken

	^type == #'empty'.
%
category: 'Testing'
method:
isEndifDirective

	^type == #'directive' and: [value = 'endif'].
%
category: 'Testing'
method:
isEqualsToken

	^type == #'punctuator' and: [value = '='].
%
category: 'Testing'
method:
isExtensionIdentifierToken

	^type == #'identifier' and: [value = '__extension__'].
%
category: 'Testing'
method:
isFunctionLikeMacroDefinition

	^type == #'define' and: [value isKindOf: Array].
%
category: 'Testing'
method:
isHashToken

	^type == #'punctuator' and: [value = '#'].
%
category: 'Testing'
method:
isIdentifierToken

	^type == #'identifier'.
%
category: 'Testing'
method:
isIntegerToken

	^#(#int16 #int32) includes: type
%
category: 'Testing'
method:
isOpenAngleBracketToken

	^type == #'punctuator' and: [value = '<'].
%
category: 'Testing'
method:
isOpenCurlyBracketToken

	^type == #'punctuator' and: [value = '{'].
%
category: 'Testing'
method:
isOpenParenthesisToken

	^type == #'punctuator' and: [value = '('].
%
category: 'Testing'
method:
isOpenSquareBracketToken

	^type == #'punctuator' and: [value = '['].
%
category: 'Testing'
method:
isPunctuatorToken

	^type == #'punctuator'.
%
category: 'Testing'
method:
isReservedWord

	^#(
		'__float128'
		'_Complex'
		'auto'
		'break'
		'case'
		'char'
		'const'
		'continue'
		'default'
		'do'
		'double'
		'else'
		'entry'
		'extern'
		'float'
		'for'
		'goto'
		'if'
		'int'
		'enum'
		'long'
		'register'
		'return'
		'short'
		'signed'
		"'sizeof'"		"this should be followed by a parenthesis"
		'static'
		'struct'
		'switch'
		'typedef'
		'union'
		'unsigned'
		'void'
		'volatile'
		'while'
	) includes: value.
%
category: 'Testing'
method:
isSemicolonToken

	^type == #'punctuator' and: [value = ';'].
%
category: 'Testing'
method:
isStarToken

	^type == #'punctuator' and: [value = '*'].
%
category: 'Testing'
method:
isStringToken

	^type == #'string'.
%
category: 'Testing'
method:
isStructIdentifierToken

	^type == #'identifier' and: [value = 'struct'].
%
category: 'Testing'
method:
isTildeToken

	^type == #'punctuator' and: [value = '~'].
%
category: 'Testing'
method:
isUpArrowToken

	^type == #'punctuator' and: [value = '^'].
%
category: 'other'
method:
key

	^type.
%
category: 'Accessing'
method:
line

	^ line
%
category: 'Reading'
method:
octalCharacterFrom: aStream
	"Enter with pointing to '0' of '\0ooo'"

	| string |
	string := String withAll: '8r0'.
	[true] whileTrue: [
		| char |
		char := self peekFrom: aStream.
		(string size < 6 and: [char isDigit]) ifFalse: [
			value add: (Character codePoint: (Integer fromString: string)).
			^self.
		].
		string add: char.
		source add: aStream next.
	].
%
category: 'Accessing'
method:
peek
	"In order to distinguish a function-like macro definition from 
	an object-like macro definition that begins with an open parenthesis,
	we need to know if there was any whitespace after the identifier.

	# define foo(bar)		/* a function-like macro */
	# define foo (bar)	/* an object-like macro */
"
	^peek
%
category: 'other'
method:
peekFrom: aStream
	"Give back the next non-mergeLine token; does not skip whitespace, does not skip comments"

	| char1 char2 char3 lf cr |
	(char1 := aStream peek) == nil  ifTrue:[ ^ nil ].
	char1 == $\ ifFalse: [^char1].
	(char2 := aStream peek2) == nil  ifTrue:[ ^ $\  ].
	lf := Character lf .
	cr := Character cr .
	(char2 == cr or: [char2 == lf]) ifFalse:[ ^ $\ ].
	"$\ was followed by CR or LF"
	source 
		add: aStream next;
		add: aStream next.
	(char3 := aStream peek) == nil  ifTrue: [^nil].
	(char2 ~~ char3 and: [char3 == cr or: [char3 == lf]]) ifFalse: [^char3].
	^self peekFrom: aStream.
%
category: 'Printing'
method:
positionString
  | str |
  (str := ' at line ' copy) 
    add: line asString ;
    add: ' in file ' ; add: file asString .
  ^ str
%
category: 'other'
method:
printOn: aStream

	(source == nil  or: [source isEmpty]) ifTrue: [
		type printOn: aStream.
		aStream nextPutAll: ' -> '.
		value printOn: aStream.
	] ifFalse: [
		aStream nextPutAll: source.
	].
%
category: 'Reading'
method:
readCharacterFrom: aStream

	type := #'uint8'. 
	self 
		readLiteralUpTo: $'
		from: aStream.
	255 < (value := value first codePoint) ifTrue: [self error: 'Should be wide character!'].
%
category: 'Reading'
method:
readHashFrom: aStream isBeginningOfLine: aBoolean
	| char |
	source add: aStream next.	"$#"
	(char := self peekFrom: aStream) == $# ifTrue: [	"## means to merge adjacent tokens"
		source add: aStream next.
		type := #'concatenation'.
		^self.
	].
	aBoolean ifFalse: [
		type := #'punctuator'.
		value := '#'.
		^self.
	].
	self 
		skipCommentsIn: aStream;
		readIdentifierFrom: aStream .
	type := #'directive'.
	value isEmpty ifTrue: [self error: 'Missing preprocessor directive'].
%
category: 'Reading'
method:
readIdentifierFrom: aStream

	|  char |
	type := #'identifier'.
	value := String new.
	[
		char := self peekFrom: aStream.
		char ~~ nil  and: [char isAlphaNumeric or: [char == $_ or: [char == $: and: [aStream peek2 == $: or: [value last == $:]]]]].
	] whileTrue: [
		value add: aStream next.
	].
	peek := aStream peek.		"Curious to know if it is an open parenthesis"
	source add: value.
%
category: 'Reading'
method:
readLiteralUpTo: aCharacter from: aStream

	| char lf |
	source add: aStream next.	"The open quote character"
	value := String new.
  lf := Character lf .
	[
		(char := aStream next) == lf ifTrue: [self error: 'End of literal not found on current line!'].
		source add: char.
		char = aCharacter.
	] whileFalse: [
		char == $\ ifTrue: [
			source add: (char := aStream next).
			char == $a 	ifTrue: [value add: (Character codePoint: 7	)	] ifFalse: [		"\a 	= alert, <Ctrl>+<G>"
			char == $b 	ifTrue: [value add: (Character codePoint: 8	)	] ifFalse: [		"\b 	= backspace, <Ctrl>+<H>"
			char == $f	ifTrue: [value add: (Character codePoint: 12	)	] ifFalse: [		"\f 	= form feed, <Ctrl>+<L>"
			char == $n	ifTrue: [value add: (Character codePoint: 13	)	] ifFalse: [		"\n 	= new line, <Ctrl>+<M>"
			char == $r	ifTrue: [value add: (Character codePoint: 10	)	] ifFalse: [		"\r 	= carriage return, <Ctrl>+<J>"
			char == $t	ifTrue: [value add: (Character codePoint: 9	)	] ifFalse: [		"\t 	= horizontal tab, <Ctrl>+<I>"
			char == $v	ifTrue: [value add: (Character codePoint: 11	)	] ifFalse: [		"\v 	= vertical tab, <Ctrl>+<K>"
			char == $x	ifTrue: [self hexCharacterFrom: aStream			] ifFalse: [		"\xdd = hex number"
			char == $0	ifTrue: [self octalCharacterFrom: aStream		] ifFalse: [		"\0ddd	= octal number"
			char == $'	ifTrue: [value add: char									] ifFalse: [		"\'		= single quote"
			char == $"	ifTrue: [value add: char									] ifFalse: [		"""		= double quote"
			char == $?	ifTrue: [value add: char									] ifFalse: [		"?		= question mark"
			char == $\	ifTrue: [value add: char									] ifFalse: [		"\\		= single backslash"
			]]]]]]]]]]]]].
		] ifFalse: [
			value add: char.
		].
	].
%
category: 'Reading'
method:
readNumberFrom: aStream
"A preprocessing number has a rather bizarre definition. The category includes all the 
normal integer and floating point constants one expects of C, but also a number of other 
things one might not initially recognize as a number. Formally, preprocessing numbers begin 
with an optional period, a required decimal digit, and then continue with any sequence of 
letters, digits, underscores, periods, and exponents. 
Exponents are the two-character sequences `e+', `e-', `E+', `E-', `p+', `p-', `P+', and `P-'. 
(The exponents that begin with `p' or `P' are new to C99. 
They are used for hexadecimal floating-point constants.)"

	| char string isLong isUnsigned |
	type := #'number'.
	value := String with: (source add: aStream next).
	[
		aStream atEnd not and: [
			char := self peekFrom: aStream.
			char isAlphaNumeric or: [char == $. or: [char == $_ or: [(char == $+ or: [char == $-]) and: ['eEpP' includes: value last]]]].
		].
	] whileTrue: [
		value add: (source add: aStream next).
	].
	isLong := false.
	isUnsigned := false.
	(2 < value size and: [(value at: 1) == $0 and: [(value at: 2) == $x ]]) ifTrue: [
		string := '16r' , (value copyFrom: 3 to: value size).
	] ifFalse: [
		value first == $0 ifTrue: [
			string := '8r' , value.
		] ifFalse: [
			string := value.
		].
	].
	('lL' includes: string last) ifTrue: [
		isLong := true.
		string := string copyFrom: 1 to: string size - 1.
	].
	('uU' includes: string last) ifTrue: [
		isUnsigned := true.
		string := string copyFrom: 1 to: string size - 1.
		('lL' includes: string last) ifTrue: [
			isLong := true.
			string := string copyFrom: 1 to: string size - 1.
		].
	].
	value := Integer fromString: string.
	(value class ~~ SmallInteger and: [16rFFFFFFFFFFFFFFFF < value]) ifTrue: [self halt].
	type := (isLong or: [16rFFFFFFFF < value]) ifTrue: [
		(isUnsigned or: [16r7FFFFFFFFFFFFFFF < value]) 
			ifTrue: [#'uint64']
			ifFalse: [#'int64'].
	] ifFalse: [
		16rFFFF < value ifTrue: [
			(isUnsigned or: [16r7FFFFFFF < value]) 
				ifTrue: [#'uint32']
				ifFalse: [#'int32'].
		] ifFalse: [
			(isUnsigned or: [16r7FFF < value]) 
				ifTrue: [#'uint16']
				ifFalse: [#'int16'].
		].
	].
%
category: 'Reading'
method:
readPunctuatorFrom: aStream

	| char1 char2 char3 twoChars threeChars |
	type := #'punctuator'.
	source add: (char1 := aStream next).
	value := String with: char1.
	(char2 := aStream peek) == nil  ifTrue: [^self].
	(char3 := aStream peek2) == nil  ifTrue: [
		(twoChars := value copy) add: char2 .
	] ifFalse: [
		(twoChars := value copy) add: char2 .
		(threeChars := twoChars copy) add: char3 .
	].
	(#('<<=' '>>=' '->*') includes: threeChars) ifTrue: [
		value := threeChars.
		source add: char2; add: char3.
		aStream next; next.
		^self.
	].
	(#('++' '--' '==' '!=' '<=' '>=' '&&' '||' '<<' '>>' '+=' '-=' '*=' '/=' '%=' '&=' '|=' '^=' '->' '.*' '::') includes: twoChars) ifTrue: [
		value := twoChars.
		source add: char2.
		aStream next.
	].
%
category: 'Reading'
method:
readStringFrom: aStream

	type := #'string'. 
	self 
		readLiteralUpTo: $" 
		from: aStream.
%
category: 'Reading'
method:
readWideCharacterFrom: aStream
		"#define __WCHAR_MAX__ 2147483647"

	(source add: aStream next) == $L ifFalse: [self error: 'How did we get here!?'].
	type := #'int32'.
	self 
		readLiteralUpTo: $'
		from: aStream.
	16r7FFFFFFF < (value := value first codePoint) ifTrue: [self error: 'Should be wide character!'].
%
category: 'Reading'
method:
readWideStringFrom: aStream

	(source add: aStream next) == $L ifFalse: [self error: 'How did we get here!?'].
	type := #'string'.
	self 
		readLiteralUpTo: $' 
		from: aStream.
	self halt.
%
category: 'other'
method:
replaceFirst: aToken
	file := aToken file.
	line := aToken line.
	source := aToken source.
%
category: 'other'
method:
replaceOther: aToken
	file := aToken file.
	line := aToken line.
	source := ''.
%
category: 'other'
method:
setValueToArrayWithEmptyToken

	value := { self class empty }.
%
category: 'other'
method:
skipBlockCommentsIn: aStream
	| char lf |
	source add: aStream next.	"This should be the $* following the $/"
  lf := Character lf .
	[
		source add: (char := aStream next).
		char == lf ifTrue:[ line := line + 1  ]. 
		char == $* and: [(self peekFrom: aStream) == $/ ].
	] whileFalse: [].
	source add: aStream next.	"This should be the $/ following the $*"
	^self skipCommentsIn: aStream.
%
category: 'other'
method:
skipCommentsIn: aStream
		"answer true if came to end-of-line, otherwise false"
	| char position lf |
  lf := Character lf .
	[
		(char := aStream peek) ~~ nil  and: [char isSeparator].
	] whileTrue: [
		char == lf ifTrue:[
      ^true
    ].
		source add: aStream next.
	].
	char == $/ ifFalse: [^false].
	position := aStream position.
	source add: aStream next.
	char := self peekFrom: aStream.
	char == $* ifTrue: [^self skipBlockCommentsIn: aStream].
	char == $/ ifTrue: [^self skipLineCommentsIn: aStream].
	source size: source size - 1.
	aStream position: position.
	^false.
%
category: 'other'
method:
skipLineCommentsIn: aStream
  | lf |
	source add: aStream next.	"This should be the $/ following the $/"
  lf := Character lf .
	[
		aStream peek == lf
	] whileFalse: [
		source add: aStream next.
	].
	^true.
%
category: 'other'
method:
skipMultipleBlankLinesAndCommentsInStream: aStream
cppMType ~~ 0 ifTrue:[
  [ | skipped |
    skipped := false .
	  [
		  [
			  self skipCommentsIn: aStream.
		  ] whileTrue: [  
			  source add: aStream next.
        skipped := true .
		  ].
		  aStream peek == $\ .
	  ] whileTrue: [
		  source add: aStream next.  "line concatenation"
      skipped := true .
	  ].
    skipped ifTrue:[ skipped := self skipCppDebugInfo: aStream ].
    skipped == true
  ] whileTrue.
] ifFalse:[
  [
    [
      self skipCommentsIn: aStream.
    ] whileTrue: [
      line := line + 1.
      source add: aStream next.
    ].
    aStream peek == $\.
  ] whileTrue: [
    source add: aStream next.
  ].
]
%
category: 'other'
method:
skipCppDebugInfo: aStream 
  "Skip consecutive # <integer> <fileName> debuginfo lines from /usr/bin/cpp output .
   (AIX skip #line integer fileName)
   Skip #pragma lines from clang output .
   Returns nil if EOF, true if one or more lines skipped, false otherwise ."
  | char1 char2 px cx lf skipped |
  skipped := false .
  lf := Character lf .
 	[ | isDebug |
    [ char1 := self peekFrom: aStream .
      char1 == lf 
    ] whileTrue:[
      aStream next.  "skip empty line"
      skipped := true .
      line := line + 1 . 
    ].
    char1 ifNil:[ ^ nil ].
	  char2 := aStream peek2.
    (char1 == lf and:[ char2 == nil]) ifTrue:[ ^ nil ].
    isDebug := false .
    char1 == $# ifTrue:[
	    (char2 == $   ""
      and:[ (px := aStream position + 3) < (cx := aStream collection) size
      and:[  (cx at: px) isDigit ]]) ifTrue:[
         isDebug := true  " a # <int> <filename> line " .
         aStream next ; next. "consume char1, char2"
      ].
      (cppMType == 9 and:[ (aStream peekN: 6) = '#line ']) ifTrue:[ "AIX only"
         6 timesRepeat:[ aStream next ]. "consume '#line ' "
         isDebug := true  " a # <int> <filename> line " .
      ].
      isDebug ifTrue:[
        line := self readDebugLineNumberFrom: aStream .
        file := self readDebugFileNameFrom: aStream . 
        cppMType == 9 ifTrue:[ | key |  
          key := #GemStone_CPreprocessor_lastFile .
          file size == 0 ifTrue:[ "#line <int>  with no file name  , use last file name seen"
            file := (SessionTemps current at: key otherwise: '') copy 
          ] ifFalse: [ 
            SessionTemps current at: key put: file copy .
          ]. 
        ].
      ] ifFalse:[
        "clang preprocessor produces #pragma lines on Darwin"
        char2 == $p ifTrue:[ | pStr |
          pStr := '#pragma' .
          (aStream peekN: pStr size ) = pStr ifTrue:[
            isDebug := true .
            [ 
              (self peekFrom: aStream ) == lf
            ] whileFalse:[
              aStream next.  "skip #pragma line"
              line := line + 1 .
            ].
          ].
        ].
      ].
    ].
    isDebug
  ] whileTrue:[
    skipped := true .  "skip to end of a debug info line"
    [aStream next == lf ] whileFalse .
    "line := line + 1 " "don't count debuginfo line"
  ].
  ^ skipped
%
method:
readDebugLineNumberFrom: aStream
  | ch n zeroCp |
  n := 0 .
  zeroCp := $0 codePoint .
  [ ch := aStream peek .
    (ch >= $0 and:[ ch <= $9 ]) 
  ] whileTrue:[ 
    n := (n * 10) + (ch codePoint - zeroCp) .
    aStream next
  ].
  aStream line: n .
  ^ n
%

method:
readDebugFileNameFrom: aStream
  | ch fn lf |
  fn := String new .
  lf := Character lf .
  [ ch := aStream peek .
    ch == $" or:[ch == lf]
  ] whileFalse:[ 
    aStream next 
  ].
  ch == lf ifTrue:[ ^ fn ].
  aStream next . "consume double quote char"
  [ | endOfFn | 
    ch := aStream next .
    (endOfFn := ch == $" or:[ ch == lf]) ifFalse:[ fn add: ch ].
    endOfFn
  ] whileFalse .
  aStream file: fn .
  ^ fn
%

category: 'other'
method:
skipSingleLogicalLineCommentsInStream: aStream ifNothingFound: aBlock
	"Skip white space and comments on the current line, extending the
	 definition of current for lines ending with a backslash+newline."
  | lf |
	[ (self skipCommentsIn: aStream) == true ifTrue:[ ^ aBlock value ].
	  (self peekFrom: aStream) == $\ 
       and:[ aStream peek2 == (lf ifNil:[lf :=Character lf]) ]
  ] whileTrue:[
    cppMType ~~ 0 ifFalse:[ line := line + 1 ].
		source add: aStream next.
		source add: aStream next
  ]
%
category: 'Accessing'
method:
source

	source == nil  ifTrue: [^''].
	^source.
%
category: 'Accessing'
method:
type

	^type.
%
category: 'Accessing'
method:
value

	^value.
%
category: 'other'
method:
value: anObject

	value := anObject.
%
category: 'other'
method:
valueString

	type == #'empty' ifTrue: [^''].
	type == #'identifier' ifTrue: [^value].
	type == #'int16' ifTrue: [^value printString].
	type == #'int32' ifTrue: [^value printString].
	type == #'int64' ifTrue: [^value printString].
	type == #'punctuator' ifTrue: [^value].
	type == #'string' ifTrue: [^'"' , value , '"'].
	type == #'uint16' ifTrue: [^value printString].
	type == #'uint32' ifTrue: [^value printString].
	type == #'uint64' ifTrue: [^value printString].
	self error: 'unrecognized type: ' , type printString.
%
