!=========================================================================
! Copyright (C) GemTalk Systems 2008-2020.  All Rights Reserved.
!
! File Capi.gs    
!
!  classes CLibrary, CFunction, CCallout, CCallin, CPointer, CByteArray 
!    are created in bom.c or preconversion

!=========================================================================

run
CLibrary category: 'FFI'.
CFunction category: 'FFI'.
CCallout category: 'FFI'.
CCallin category: 'FFI'.
CPointer category: 'FFI'.
CByteArray category: 'FFI'.
true
%

set class CLibrary
removeallmethods
removeallclassmethods

category: 'Documentation'
classmethod:
comment
^ 'An intances of CLibrary represents a shared library.
   
   instVars
      name  , a String, the name of the library .
      expandedName, a String,  name after expansion of environment variables
         and symbolic links , and adding .so or other suffix, 
         as was passed to dlopen(). nil in a committed instance.
         The instVar is a cache to be accessed by the primitives only.
      
      lastError , a String, value from dlerror() after a failure of dlopen(),
               always nil in a committed instance .
      loadAttempted,  nil in a committed instance, true if load attempted
      onLoadBlock , nil, or an ExecBlock to be run each time the
          shared library is successfully loaded; was used by Ruby.

   cData (hidden instVar) , in an in-memory instance is a handle 
       returned from a successful dlopen() , or nil if library is not loaded.
   If available, the RTLD_FIRST(Solaris, Mac) flag 
   is specified when calling dlopen().

   Loading of shared libraries via CLibrary is disallowed if the session''s
   UserProfile has the NoUserAction inverse privilege set .

Constraints:
	name: String
	expandedName: String
	lastError: String
	loadAttempted: Boolean
	onLoadBlock: ExecBlock
'
%


category: 'Instance creation'
classmethod:
named: libraryName

  "if libraryName does not include a  '.'  character, then the OS dependent
   suffix such as  '.so'   will be appended prior to attempting to load
   the library"

  ^ self _basicNew _initialize: libraryName
%

category: 'Private'
classmethod:
_basicNew

  "creates an instance registered with VM for finalization of cData"
  <primitive: 674> 
  ^self _primitiveFailed: #_basicNew
% 
classmethod:
basicNew

 "disallowed"
  self shouldNotImplement: #basicNew
%
classmethod:
new
 "disallowed"
  self shouldNotImplement: #new
%
method:
_initialize: aName
  (aName isKindOfClass: String) ifFalse:[
    aName _error: #rtErrBadArgKind args: { String }.
  ].
  aName size > 2047 ifTrue:[  "must agree with loadCLibrary in VM"
    aName _error: #errArgTooLarge args:{ 2047 } .
  ].   
  name := aName 
%

method:
_oneArgPrim: opcode arg: anArg

" opcode==0  isLoaded,  anArg not used .
  opcode==1  load  , anArg is Boolean rtldNow
  opcode==2  expandedName  , anArg not used .
  opcode==3  dlsym(). anArg is a String.  If receiver not loaded, 
               does  (self _oneArgPrim:1 arg:false) before calling dlsym()
"

<primitive: 709>
opcode == 3 ifTrue:[ anArg _validateClass: String ].
opcode == 1 ifTrue:[ anArg _validateClass: Boolean ].
self _primitiveFailed: #_oneArgPrim:arg: args: { opcode . anArg }
%

classmethod:
_oneArgPrim: opcode arg: anArg
  "opcode==4  process dlsym()"

<primitive: 709>
anArg _validateClass: String .
self _primitiveFailed: #_oneArgPrim:arg: args: { opcode . anArg }
%


category: 'Accessing'
method:
isLoaded

"Return true if shared library described by receiver has been loaded
 since the receiver was created or faulted into memory,
 false otherwise."

^ self _oneArgPrim: 0 arg: nil
%

method:
hasCSymbol: aString

"Return true if dlsym() finds specified name in the shared library
 represented by receiver. dladdr() used if available to qualify
 the result of dlsym() to within receiver's shared library.

 dladdr() not implemented on AIX, so this method will return
 true if the specified name exists in any library loaded thus
 far into the process."

| dladdrLibNam expName |
dladdrLibNam := self _libNameForSymbol: aString .
self isLoaded ifFalse:[ ^ false ].
dladdrLibNam ifNil:[ ^ false ].
dladdrLibNam == true ifTrue:[  
  "found by dlsym() and dladdr() not supported by OS"
  ^ true 
].
expName := self expandedName .
"ignore version numbers on .so  which is target of a symlink"
(dladdrLibNam at: 1 equals: expName ) ifTrue:[ ^ true ].
(expName at: ((expName size - 2) max: 1) equals:'.so') ifTrue:[ 
  "solaris or linux"
  ^ (dladdrLibNam findString: expName startingAt: 1) >= 1 .
].
(expName at: ((expName size - 5) max: 1) equals:'.dylib') ifTrue:[ | k |
   "mac"
  k := expName indexOf: $. startingAt: 1 .
  k >= 1 ifTrue:[ | prefix |
    prefix := expName copyFrom: 1 to: k .
    ^ (dladdrLibNam findString: prefix startingAt: 1) >= 1
  ].
].
^ false
%

method: 
_libNameForSymbol: aString
  "Call dlsym() for aString use receiver's handle from dlopen().
   If dlsym() returns non-NULL  
     if dladdr() supported by the OS
       return a String containing dladdr(dlsym(aString))->dli_fname 
     else
       return true .
   Returns nil if shared library represented by receiver cannot
   be loaded, dlsym() fails to find aString, or dladdr() fails.
  "
  ^ self _oneArgPrim: 3 arg: aString
%

classmethod:
hasCSymbol: aString

"return true if dlsym() finds specified name in any loaded library
 in the current process."

^ self _oneArgPrim: 4 arg: aString
%

method:
lastError
 "Return the String returned by dlerror() from the last failed dlopen()
  since this object was created or faulted into memory ."

  self isLoaded . "ensure current C state"
  ^ lastError
%

category: 'Loading'
method:
load
"  Load the shared library described by receiver if not already loaded.
   Before calling dlopen(), any environment variables in the name 
   (beginning with $$ and limited by $/ ) are expanded using the 
   current environment of the session's gem or topaz -l process, 
   and symbolic links are followed.  The name must not exceed 2046
   characters and the expanded form must not exceed 8190 characters.
   The flags to dlopen include RTLD_LAZY.  
   On AIX, lazy bind is not implemented (RTLD_LAZY has RTLD_NOW semantics).
   See CLibrary(C)>>comment for details of flags passed to dlopen().

   Returns true if successful, or a String describing the error from dlopen().
"

 ^ self _oneArgPrim: 1 arg: false
%

method:
loadResolveAll

  "Like  Clibrary>>load, except flags to dlopen include RTLD_NOW."

 ^ self _oneArgPrim: 1 arg: true
%


category: 'Accessing'
method:
name

^ name
%

method:
expandedName

 "If dlopen() has been successfully called for the in-memory copy 
  of the receiver, then return the expanded name as it was passed to dlopen,
  otherwise return the name after expanding environment variables and
  symbolic links as described in CLibary>>load. "

 name == nil ifTrue:[ self error:'nil name'].
 ^ self _oneArgPrim: 2 arg: nil
%

! =========================================================================
set class CFunction
removeallmethods
removeallclassmethods

category: 'Documentation'
classmethod:
comment
^'CFunction is an abstract class representing the type signature of a C function.
Concrete subclasses are CCallout, CCallin .

  instVars
     fName  , a String, the name of a callout function to be passed to dlsym() ,
          or the name of a callin function type.

     argTypes, an Array of zero or more Symbols describing the arguments .
        When native code generation is enabled by the gem config file, 
          Functions using varArgs are limited to a maximum of
          20 variable arguments after the fixed arguments.
        When native code generation is disabled by the gem config file,
          Arguments and results of C type   float    are not supported.
          Functions which have one or more args of C type   double
          or whose result type is   double, 
          are limited to a maximum of 4 fixed arguments.
          Functions using varArgs are limited to 4 fixed args and 10 total args.
          Otherwise functions are limited to 15 total arguments.
        See method library:name:result:args:  for more details .

     argTypesDict , if variable arguments supported by this function,
        references the class variable ArgTypesDict , otherwise nil.

     resultType, a Symbols describing the result of the C function.
       See method library:name:result:args:  for more details .
  
     cTypes , an Array for internal use, 
         translated representation of argTypes and resultType .

     argCounts , a SmallInteger generated from argTypes.

Constraints:
	fName: String
	argTypes: Array
	resultType: Symbol
	cTypes: ByteArray
	argCounts: SmallInteger
	argTypesDict: GsMethodDictionary
'
%

classmethod:
_basicNew

  "creates an instance registered with VM for finalization of cData"
  <primitive: 674> 
  ^self _primitiveFailed: #_basicNew
% 

classmethod:
basicNew
 "disallowed"
  self shouldNotImplement: #basicNew
%
classmethod:
new
 "disallowed"
  self shouldNotImplement: #new
%

category: 'Private'
method:
_name: aName result: resType args: argumentTypes varArgsAfter: varArgsAfter

 "If C varargs allowed, varArgsAfter must be >=0   and   
    <  argumentTypes size   , and must also be <=4  ,
  else varArgsAfter==-1 to denote no varargs ."

 | rType numAtypes ctyps |
 (aName isKindOfClass: String) ifFalse:[
   ArgumentTypeError new name: 'aName' expectedClass: String actualArg: aName ; signal .
   ^ nil
 ].
 aName size > 1023 ifTrue:[  "must agree with code in VM prim 710"
   ArgumentError new name: 'fName' ; signal: 'CFunction name exceeds 1023 chars' .
   ^ nil
 ].
 fName := aName .

 (rType := ResTypesDict at: resType otherwise: nil ) ifNil:[
   ArgumentError new name: 'resType' ; signal: 'not a valid CFunction result type'  .
   ^ nil
 ].
 resultType := resType .
 numAtypes := argumentTypes size .
 ctyps := ByteArray new: numAtypes + 1 . 
 ctyps at: 1 put: rType .  "for C type of result"
 argumentTypes ifNotNil:[
   numAtypes > CFUNC_max_params ifTrue:[  
     ArgumentError new name: 'argumentTypes' ;
       signal: 'CFunction number of args must be <= ', CFUNC_max_params asString  .
     ^ nil
   ].
   1 to: numAtypes do:[:j | | aSym typ |
     aSym :=  argumentTypes at: j .
     (typ := ArgTypesDict at: aSym otherwise: nil ) ifNil:[
       ((aSym isKindOfClass: CCallin) and:[ self isKindOfClass: CCallout]) ifTrue:[
         argumentTypes at: j put:( aSym _copyForCFunction ) .
         typ := ArgTypesDict at: #ffiCallback .  
       ] ifFalse:[
         ArgumentError signal: aSym printString , ' is not a valid CFunction arg type' .
         ^ nil
       ].
     ].
     ctyps at: j + 1 put: typ .  "type of an arg"
   ]. 
 ]. 
 argTypes := argumentTypes .
 varArgsAfter >= 0 ifTrue:[
   varArgsAfter <= numAtypes ifTrue:[ 
      argCounts := varArgsAfter bitOr:  ArgCounts_varArgsMask .
      argTypesDict := ArgTypesDict .
   ] ifFalse:[
     ArgumentError new name:'varArgsAfter' ; 
	signal:'varArgsAfter must be -1, or <= number of types'.
   ]
 ] ifFalse:[
   argCounts := numAtypes .
   varArgsAfter == -1 ifFalse:[
     ArgumentError new name:'varArgsAfter'; 
	signal:'varArgsAfter must be -1, or <= number of types'.
   ].
 ].
 ctyps immediateInvariant.
 cTypes := ctyps .
%

category: 'Instance Creation'
classmethod:
name: aName result: resType args: argumentTypes
  "aName is the C function name of the new Callout .

  resType, argumentTypes, varArgsAfter are per the comments in 
    CCallout>>name:result:args: . "

  | res |
  (res := self _basicNew) 
     _name: aName result: resType args: argumentTypes varArgsAfter: -1 .
  res immediateInvariant .
  ^ res
%

classmethod:
name: aName result: resType args: argumentTypes
varArgsAfter: varArgsAfter
  "aName is the C function name of the new Callin .

  resType, argumentTypes, varArgsAfter are per the comments in 
    CCallout>>library:name:result:args:varArgsAfter: . "

  | res |
  (res := self _basicNew) 
      _name: aName result: resType
                        args: argumentTypes varArgsAfter: varArgsAfter .
  res immediateInvariant .
  ^ res
%

category: 'Formatting'
method:
asString
  | res fn |
  res := super asString .
  (fn := fName) ifNotNil:[  res add: $  ; add: fn ].
  ^ res
%

method:
signatureString
  | res fn types nTypes nFixed varAllowed |
  (res := resultType asString) .
  (fn := fName) ifNotNil:[  
     types := argTypes .
     nTypes := types size .
     nFixed := self numFixedArgs .
     varAllowed := self varArgsAllowed .
     res add: $  ; add: fn ; add: $(  .
     1 to: nFixed do:[:n | | aType | 
       n > 1 ifTrue:[ res add: ', ' ].
       aType := types at: n .
       aType _isSymbol ifTrue:[ res add: aType ]
                      ifFalse:[ res add:( aType callinName ); add: '()' ].
     ].
     varAllowed ifTrue:[
       res add:', (VarArgs:) '.
       nFixed + 1 to: nTypes do:[:m | | aType |
         m > (nFixed + 1) ifTrue:[ res add: ', ' ].
         aType := types at: m .
         aType _isSymbol ifTrue:[ res add: aType ]
                      ifFalse:[ res add:( aType callinName ); add: '()' ].
       ].
     ].
     res add: $)  .
  ].
  ^ res
%
category: 'Accessing'
method:
numFixedArgs
  ^ argCounts bitAnd: ArgCounts_fixedMask
%
method:
numParameterTypes
  ^ argTypes size 
%
method:
varArgsAllowed
  ^ (argCounts bitAnd: ArgCounts_varArgsMask) ~~ 0
%
method:
name
 ^ fName
%
method:
coerceNameToSymbol
  fName := fName asSymbol
%

! ==============================================================
set class CCallout
removeallmethods
removeallclassmethods

category: 'Documentation'
classmethod:
comment
^'An instance of CCallout represents the information needed to 
call a C function using the FFI .

   instVars
     library , a CLibrary, nil, or 1 .  If nil, the function will be
         searched for in the current process using dlsym(RTLD_NEXT) ,
         searching any libraries which were loaded AFTER libgcilnk.so .
         If 1 , the function represents a call into a Ruby C extension,
         and cannot be re-resolved if this CCallout is faulted in.

         If a CLibrary, the specified shared library will be loaded on-demand 
         using dlopen() and that library searched using dlsym().
         On some platforms where RTLD_FIRST not available, the whole process
         will be searched if function not found in specified library.

     lastError, normally nil, in an in-memory instance holds the last
        error from the callWith: primitive .

     untaggedEnumsDict , a GsMethodDictionary which is a copy
        of the Ruby Enums.__Transient_kv_map and Enums::Persistent_kv_map
        created at the time of a Ruby attach_function

   cData (hidden instVar) , in an in-memory instance contains the result from dlsym().
      and pointers to one or more calloutStub''s emitted by the native code generator.

    To avoid repeated calls to dlsym() a committed instance of CFunction
    must be kept in memory (such as by a reference from session state)
    while it is being used.  At the time a dlsym() call is made,
    if library is non-nil that instVar will become a not-stubbable memory
    reference to the CLibrary.

   See CCallout class >>library:name:result:args:  for details of 
   arguments and results semantics.


Constraints:
	fName: String
	argTypes: Array
	resultType: Symbol
	cTypes: ByteArray
	argCounts: SmallInteger
	argTypesDict: GsMethodDictionary
	library: CLibrary
	lastError: String
	untaggedEnumsDict: GsMethodDictionary
'
%

category: 'Instance creation'
classmethod:
library: aCLibrary name: aName result: resType args: argumentTypes

" aCLibrary may be a CLibrary,  or nil if current process should
  be searched with dlsym(), 
  or an Array of CLibraries to be searched.  
  See CFunction(C)>>comment for more details.

  aName must be a String .  It will be used as arg to dlsym() at the first
  time the function is called.

  resType is one of 
    #int64    -->  C function returns an int64, or any unsigned C integer 
                smaller than 64bits ; call will return an Integer
    #uint64   -->  C function returns a uint64 ; call will return an Integer
    #int32    -->  C function returns a signed C integer 32bits or smaller;  
		   call will return a SmallInteger;
		   #int64 can be used instead of #int32 on Sparc only .
    #uint32   -->  C function returns a C uint ; call returns a SmallInteger.
    #int16    -->  C function returns a C short ; call returns a SmallInteger
    #uint16   -->  C function returns a C ushort ; call returns a SmallInteger
    #int8     -->  C function returns a C char ; call returns a SmallInteger
    #uint8    -->  C function returns a C uchar ; call returns a SmallInteger

    #bool     --  C function returns zero or non-zero C integer of any size
                     call returns TRUE or FALSE
    #double    -->  C function returns an C double, 
                call will return a SmallDouble or Float .
    #float     -->  C function returns an C float,
                call will return a SmallDouble or Float .
    #ptr      -->  call will return nil or a CPointer 
    #'char*'  -->  call will return nil or a String 
    #void     -->  call always returns nil
  
  argumentTypes is an Array of zero or more elements. Each element is one of
    #int64     --> arg must be an Integer representable as a C int64
    #uint64    --> arg must be an Integer representable as a C uint64
    #int32     --> arg must be an Integer representable as a C 32bit int
    #uint32    --> arg must be an Integer representable as a C 32bit uint
    #int16     --> arg must be an Integer representable as a C 16bit short
    #uint16    --> arg must be an Integer representable as a C 16bit ushort
    #int8     --> arg must be an Integer representable as a C 8 bit signed char
    #uint8    --> arg must be an Integer representable as a C 8 bit uchar

    #bool     --> arg must ba a Boolean 

    #double    --> arg must be a SmallDouble or Float ,
    #float    --> arg must be a SmallDouble or Float
    #ptr       --> arg must be nil, a CByteArray or a CPointer  ,
                if nil , C  NULL is passed .
		if CByteArray, address of body is passed .
                if CPointer, the encapsulated  pointer is passed .
    #'&ptr'    -->  arg must be a CPointer whose value
                  will be passed and updated on return.
    #'char*'   --> arg must be a String in which case
		     body is copied to C memory before call and copied
                    from C memory (and possible grown/shrunk) after call.
                   C memory will not be valid after the call finishes.
                   The C memory copy of the String has a zero byte appended.
    #'const char*'  --> arg must be nil( to pass NULL) 
                        or a String (body is copied to C memory before call) 
                       C memory will not be valid after the call finishes.
                       The C memory copy of the String has a zero byte appended.

    #'const UChar*'  --> arg must be nil (to pass NULL)
 	 or a Unicode7, Uncode16, or Unicode32.
         The body of the string is copied to C memory in UTF16 form before
         the call.  
         The C memory copy has a codepoint zero appended.
         C memory will not be valid after the call finishes.
          
    #'UChar*' --> arg must be a Unicode7, Uncode16, or Unicode32.
         The body of the string is copied to C memory in UTF16 form before 
         the call, and copied back from C memory after the call.
         The C memory copy has a codepoint zero appended.
         The copy back may grow or shrink the arg, and may become the arg 
         to a Uncode16 or a Unicode32.


    Note C strings which should persist in C memory after a function call
    should be passed using CByteArray created from an instance of String.

    When native code is disabled in the gem config file, use of double or 
    float args imply a maximum of 4 arguments , excluding var args .
"
  | res |
  (res := self _basicNew) _setLibrary: aCLibrary ;
     _name: aName result: resType args: argumentTypes varArgsAfter: -1 .
  ^ res
         
%

category: 'Instance creation'
classmethod:
library: aCLibrary name: aName result: resType args: argumentTypes
varArgsAfter: varArgsAfter

"aCLibrary, aName, resType are per comments in  library:name:result:args: .
 
 argumentTypes must specify all of the arguments to be passed ,
 per comments in  library:name:result:args:  .
 
 If C varargs allowed, varArgsAfter must be >=0   and   
    <  argumentTypes size   , and must also be <=4  ,
 else varArgsAfter==-1 to denote no varargs .

 For example, if varArgsAfter==3 and  (argumentTypes size == 5)
 and none of argumentTypes or resType specify #double, 
 a C function call of the form   
    f(a,b,c,d,e);
 will be made using a prototype of the form 
    intptr_t f(intptr_t a, intptr_t b, intptr_t c, ...);
" 
  | res |
  (res := self _basicNew) _setLibrary: aCLibrary ;
      _name: aName result: resType 
			args: argumentTypes varArgsAfter: varArgsAfter .
  ^ res
%

category: 'Private'
method:
_setLibrary: aCLibrary

 aCLibrary ifNotNil:[ aCLibrary _validateClass: CLibrary ] .
 library := aCLibrary .
%

method:
_library
  ^ library
%

category: 'Calling'
method:
callWith: argsArray 

"invoke the function described by the receiver.  
 First send of this method during a lifetime of the receiver in memory
 triggers dlopen() (if needed) , dlsym() , and generation of native code for 
 callout stub(if native code enabled by configuration file) . 
 If function expects zero args,  argsArray must be an empty array or nil. 
 If function supports varArgs, then the variable args come after the
 fixed arguments in argsArray, and each variable arg is represented
 as 2 elements of argsArray,    
    a typeSymbol, and an argValue .
 The typeSymbol must occur as a key in the class variable ArgTypesDict. 
 If native code configured, uses the callout stub's native
 code to marshal arguments into C register/stack, else uses a C switch
 statement which only supports a limited range of function signatures
 and does not support C float arguments . 

 See  CCallout class >>library:name:result:args:  for further details.

 The VM's session state contains a field    int  ffiErrno which
 is initialized to zero at login .
 Before calling the function, this primitive executes   errno = ffiErrno .
 After the function returns the primitive excecutes  ffiErrno = errno .
 ffiErrno is accessable with the methods  CCallout class >> errno
 and CCallout class >> errno:  . "

<primitive: 710>
| nPassed nRequired |
lastError ifNotNil:[   "lastError was set by the primitive"
   ^ ArgumentError signal: lastError .
].
argsArray class == Array ifFalse:[
  ArgumentTypeError new name:'argsArray' expectedClass: Array actualArg: argsArray ; 
       signal .
].
(nPassed := argsArray size) == (nRequired := argTypes size) ifFalse:[
  ArgumentError new signal:  
    nPassed asString , ' args given , ' , nRequired asString, ' args required for:  ' ,
      self signatureString . 
].
^ self _primitiveFailed: #callWith: args: { argsArray }
%

classmethod:
errno
 
"Returns the SmallInteger value of the session's ffiErrno.
 ffiErrno is used and updated by CCallout >> callWith: ."

^ self _errno: -1
%

classmethod:
errno: aSmallInteger
 "Stores aSmallInteger into the session's ffiErrno ,
  and returns the previous value of ffiErrno as a SmallInteger."

aSmallInteger _isSmallInteger ifFalse:[ 
  aSmallInteger _validateClass: SmallInteger .
].
(aSmallInteger >= 0 and:[ aSmallInteger <= 16r7fffffff]) ifFalse:[ 
  OutOfRange new  
    name: 'errno' min: 0 max: 16r7fffffff actual: aSmallInteger ;
    signal
].
^ self _errno: aSmallInteger
%

! fixed 46138
category: 'Private'
classmethod: 
_errno: aSmallInteger

"If aSmallInteger == -1, Returns the SmallInteger value of the  
 sessions's  ffiErrno .  
 If aSmallInteger >= 0, stores aSmallInteger into the session's
 ffiErrno state.  ffiErrno is used and updated by CCallout >> callWith: . "
 
<primitive: 474>
aSmallInteger _validateClass: SmallInteger .
^ self _primitiveFailed: #_errno: args: { aSmallInteger }
% 

method:
bind
  "Attempt to resolve the C function described by the receiver,
   using dlopen() if needed, and dlsym() ."

  <primitive: 529>
  lastError ifNotNil:[   "lastError was set by the primitive"
     ^ ArgumentError signal: lastError .
  ].
  self _primitiveFailed: #bind
%

method:
_bindFunction: aCpointer
  "Install  aCpointer memoryAddress  as the address of the C function"
  <primitive: 1036>
  aCpointer _validateClass: CPointer .
  self _primitiveFailed: #_bindFunction args: { aCpointer }
%

! ==============================================================
set class CCallin
removeallmethods
removeallclassmethods

category: 'Documentation'
classmethod:
comment
^'An instance of CCallin represents the type signature of a C function to be called
by C code .  Use of CCallin is not supported when native code is disabled.

  instVars
    envId -  message send environment in which to invoke a block .

  cData (hidden instVar) references native code in fixed code memory area which
    is the stable(not relocated by code_gen GC) C address to be called by C code .

  The block to be executed is referenced by cData.blockId 

Constraints:
	fName: String
	argTypes: Array
	resultType: Symbol
	cTypes: ByteArray
	argCounts: SmallInteger
	argTypesDict: GsMethodDictionary
	envId: SmallInteger
'
%

category: 'Instance creation'
classmethod:
name: aName result: resType args: argumentTypes  envId: envId

"aName, resType are per comments in  CCallout(C)>>library:name:result:args: .
 
 argumentTypes must specify all of the arguments to be passed ,
 per comments in  library:name:result:args:  argumentTypes may not contain
 a CCallin .

 resType may not be a CCallin
 
 C varargs are not allowed
" 
  | res |
  (res := self _basicNew) 
      _name: aName result: resType 
			args: argumentTypes varArgsAfter: -1 ;
     envId: envId  .
  res immediateInvariant . "uses must make a copy"
  ^ res
%

classmethod:
name: aName result: resType args: argumentTypes  
  ^ self name: aName result: resType args: argumentTypes envId: 0"default smalltalk" 
%

category: 'Initialization'
method:
envId: anEnvId

  envId := anEnvId
%

category: 'Private'
method:
_copyForCFunction

 | res | 
  "must use _basicNew to create cData slot"
 (res := self class _basicNew) 
   _fname: fName argTypes: argTypes resultType: resultType
        cTypes: cTypes argCounts: argCounts
        argTypesDict: argTypesDict envId: envId  .
 ^ res 
%
category: 'Private'
method:
_fname: afName argTypes: aargTypes resultType: aresultType
  cTypes: acTypes argCounts: aargCounts
  argTypesDict: aargTypesDict envId: aenvId 

  fName := afName  .
  argTypes := aargTypes  .
  resultType := aresultType .
  cTypes := acTypes  .
  argCounts := aargCounts .
  argTypesDict := aargTypesDict  .
  envId := aenvId  .
%
category: 'Accessing'
method:
callinName
  ^ fName
%

category: 'Binding'
method:
copyBindTo: aBlock

"Return a copy of the receiver which is bound to
 the specified block.  The result would typically be
 used as second arg to CByteArray>>ccallinAt:put:  when
 building an Array of function pointers."

^ self _copyForCFunction _bindTo: aBlock
%


category: 'Private'
method:
_bindTo: aBlock

"Install aBlock as the code to be executed when
 the receiver is called from C ."

<primitive: 324>
aBlock _validateClass: ExecBlock .
self _primitiveFailed: #_bindTo:
      args: { aBlock }
%

! ==============================================================
set class CPointer
removeallmethods
removeallclassmethods

category: 'Documentation'
classmethod:
comment
^'A CPointer encapsulates a C pointer  which does not have auto-free
semantics.  New instances are created by CFunction calls with result
type #ptr,  and are also used for certain arguments of CFunctions.

   instVars
      derivedFrom,    nil, or a CByteArray or a CPointer.  If non-nil,
         a CByteArray or CPointer from which this object was derived;
         this instVar keeps the referenced object alive while this
         object is alive.
'
%

category: 'Private'
classmethod:
basicNew
 "disallowed"
  self shouldNotImplement: #basicNew
%
classmethod:
new
 "disallowed"
  self shouldNotImplement: #new
%
method:
_initFrom: aCByteArray offset: zeroBasedOffset

 <primitive: 718>
 aCByteArray _validateClass: CByteArray .
 aCByteArray memoryAddress == 0 ifTrue:[
   "CByteArray might be a committed instance faulted."
   ArgumentError signal:'CByteArray has NULL C memory address'. 
 ].
 zeroBasedOffset _validateClass: SmallInteger .
 ArgumentError signal:'zeroBasedOffset out of bounds of aCByteArray'.
 self _primitiveFailed: #_initFrom:offset:
      args: { aCByteArray . zeroBasedOffset }
%

category: 'Instance creation'
classmethod:
newNull
  "create a new instance with value NULL.
   The result can be used as an argument a CFunction which has
   type #'ptr' or #'&ptr' .

   Non-NULL instances are also returned from CFunction calls when result
   type is  #ptr ; pass such instances to CByteArray classmethods to
   get an instance of CByteArray with which to access the data.
 "

<primitive: 674>  "instance is registered with VM for CData finalization"
^ self _primitiveFailed: #newNull
%

classmethod:
newFrom: aCByteArray

"Return a new instance of CPointer which is a reference to body
 of a CByteArray.
 The result can be used as an argument a CFunction which has
 type #'ptr' or #'&ptr' . "

^ self newNull _initFrom: aCByteArray offset: 0
%

method:
memoryAddress
  "Returns the starting address of the C memory, as a signed 64bit Integer."

<primitive: 846>
^ self _primitiveFailed: #memoryAddress
%

category: 'Formatting'
method:
asString
  | res |
  (res := super asString ) add: self _inspect .
  ^ res
%
method:
_inspect
  | res |
  (res := String new)
    add: ' address=0x' ; add: self memoryAddress asHexString .
  ^ res
%

! ==============================================================
set class CByteArray
removeallmethods
removeallclassmethods

category: 'Documentation'
classmethod:
comment
^'An instance of CByteArray encapsulates allocation of C memory or a reference to C memory.

  instVars
    info , a SmallInteger containing 
        gcFree   , 3 bits, mask 16r7 
           0 means no memory to free,  
           bit 0x1 means call free() when in-memory instance is garbage collected.
           bit 0x2 means the memory has fence words
           bit 0x4 means VM owns the C memory  .
        dead     ,  1 bit, mask 16r8 
           application has requested memory be freed .
        sizeBytes,  upper 57  bits

    derivedFrom -  nil or a CByteArray or a CPointer.  If non-nil,
         a CByteArray or CPointer from which this object was derived.
         This object represents a pointer into the body of derivedFrom,
         used to keep the referenced object alive.
         In Ruby, derivedFrom may be an Integer address in which case
         the instance has gcFree==0 and sizeBytes==0.

    referencesTo -  nil or an Array of CByteArrays .
       this object contains C pointers to the bodies of the
       elements of referencesTo.  used to keep referenced
       CByteArrays alive.

    typeSize - a SmallInteger , default value is 1 , 
        not used by Smalltalk code , used by Pointer class  in Ruby

  The methods in category ''Accessing elements'' use ZERO BASED offsets
  and support  unaligned accesses are supported.  They will check for 
  and generate out-of-bounds errors based in the  sizeBytes of the receiver.
 
  The 8bit, 16bit, and 32bit accessing methods, when receiver is of size 8 ,
  and zeroBasedOffset == -1 , will  return the specified number of least signifigant
  bits of the receiver , as would have been stored by a function arg
  of type  char* , short*, int*   .

Constraints:
	info: SmallInteger
	derivedFrom: Object
	referencesTo: Array
	typeSize: SmallInteger
'
%

category: 'Private'
classmethod:
basicNew
 "disallowed"
  self shouldNotImplement: #basicNew
%
classmethod:
new
 "disallowed"
  self shouldNotImplement: #new
%

category: 'Instance creation'
classmethod:
_newFrom: aCPointer offset: anOffset numBytes: aSize gcFree: gcKind

"aCPointer may also be aCByteArray, or anInteger address.
 If aCPointer is a CByteArray with memory size 0 , it is assumed to
 be something like a  pointer field of a Struct which references another
 struct, and no checks are done for aSize <= anOffset + aCPointer.size "
<primitive: 711>
anOffset _validateClass: SmallInteger .
anOffset < 0 ifTrue:[ anOffset _error: #errArgTooSmall args:{ 0 } ].
aSize _validateClass: SmallInteger .
aSize < -1 ifTrue:[ aSize _error: #rtErrArgOutOfRange args:{ -1 } ].
gcKind _validateClass: SmallInteger .
aCPointer ifNotNil:[ | argCls |
  ((argCls:=aCPointer class) == CByteArray or:[ argCls == CPointer
        or:[ aCPointer _isInteger ]]) ifFalse:[
     ArgumentTypeError new name: 'aCPointer' expectedClass: { CPointer. CByteArray. Integer }
		actualArg: aCPointer ; signal
  ].
  (gcKind == 0 or:[ gcKind == 1]) ifFalse:[ 
     gcKind _error: #rtErrArgOutOfRange args:{ 0 . 1 } ].
  argCls == CByteArray ifTrue:[ | csz |
    (anOffset + aSize) >= (csz := aCPointer size) ifTrue:[
      (anOffset + aSize) _error: #errArgTooLarge args:{ csz } .
    ].
  ].
  aCPointer _isInteger ifTrue:[
    (aSize ~~ 0 or:[ gcKind ~~ 0]) ifTrue:[
      ArgumentError signal: 'invalid args for fromAddress:' .
    ].
  ].
] ifNil:[
  (gcKind == 0 or:[ gcKind == 2]) ifFalse:[ gcKind _error: #rtErrArgOutOfRange].
]. 
"if we get here malloc probably failed with no more C heap"
^ self _primitiveFailed: #_newFrom:offset:numBytes:gcFree: 
       args: { aCPointer . anOffset . aSize . gcKind }
%

classmethod:
fromCPointer: aCPointer numBytes: anInt
 "no fence, not auto freed. aCPointer may also be aCByteArray"
 
^ self _newFrom: aCPointer offset: 0 numBytes: anInt gcFree: 0
%

method:
newFrom: anOffset numBytes: anInt
 "Returns a new instance of CByteArray representing a portion
  of the receiver.  no fence, not auto freed, 
  anInt == -1 means  numBytes == strlen(self.memory) - anOffset .
  anInt == -2 means  numBytes == self.memory_size - anOffset .
  anOffset is zero based"

^ CByteArray _newFrom: self offset: anOffset numBytes: anInt gcFree: 0
%

classmethod:
fromCharStar: aCPointer

 "no fence, not auto freed, primitive does strlen() to determine size"

^ self _newFrom: aCPointer offset: 0 numBytes: -1 gcFree: 0 
%


classmethod:
fromCPointerGcFree: aCPointer numBytes: anInt  
 "no fence, will auto-free with free() "

^ self _newFrom: aCPointer offset: 0 numBytes: anInt gcFree: AutoFree 
%

classmethod:
withAll: anObject
 
"anObject may be a String, ByteArray or CByteArray.
 Returns a new instance of the receiver containing a copy of
 the bytes of the argument.   If anObject is a String, the
 resulting ByteArray contains the bytes of the String plus
 a terminator byte with value zero.

 Warning, ByteArray usually stores integers in big-endian order, 
 not CPU native byte order."

| res argSiz allocSiz |
anObject stringCharSize == 1 ifFalse:[
  (anObject isKindOfClass: CByteArray) ifFalse:[
     anObject _validateKindOfClasses: { String . ByteArray . CByteArray }.
  ]
].
argSiz := anObject size .
allocSiz := argSiz .
anObject _isOneByteString ifTrue:[ allocSiz := allocSiz + 1 ].
res := self gcMalloc: allocSiz .
argSiz ~~ 0 ifTrue:[
  res copyBytesFrom: anObject from: 1 to: argSiz into: 0 .
].
^ res
%


classmethod:
fromCharStarGcFree: aCPointer 
 "no fence, will auto-free with free(), primitive does strlen() to determine size"

^ self _newFrom: aCPointer offset: 0 numBytes: -1 gcFree: AutoFree
%

classmethod:
gcMalloc: numBytes  
 "(with fence words, freed by in-memory GC, memory is zeroed"

^ self _newFrom: nil offset: 0 numBytes: numBytes gcFree: AutoFreeFenced
%

classmethod:
newNull
  "create an instance representing a NULL pointer."

  ^ self malloc: -1
%

classmethod:
malloc: numBytes    
  "no fence words, never freed , or freed by a C function to be called, 
   memory is zeroed.
   numBytes == -1 means encapsulate a NULL pointer. "

^ self _newFrom: nil offset: 0 numBytes: numBytes gcFree: 0 
%

classmethod:
fromRegionOf: aCByteArray offset: zeroBasedOffset numBytes: aSize
  ^ self _newFrom: aCByteArray offset: zeroBasedOffset numBytes: aSize gcFree: 0
% 

category: 'Accessing'
method:
size
  "The result may be zero if the receiver encapsulates memory for
   which the size is not known, such as instances created by class methods
     fromCharStar:
     fromCharStarGcFree:
   Instances created by the class methods
     fromCPointer:numBytes:
     fromCPointerGcFree:numBytes:
     fromRegionOf:offset:numBytes:
   will haave the size known ."

  ^ info bitShift: -4
%
method:
isDead
  ^ (info bitAnd: AppDeadMask) ~~ 0
%
method:
gcFreeKind
  ^ info bitAnd: GcFreeMask 
%
! fix 48558
method:
autoRelease
  ^ (info bitAnd: AutoFree) ~~ 0
%

method
setDead
  "sets the isDead bit, without marking object dirty"
  <primitive: 847>
  self _primitiveFailed: #setDead
%

method:
derivedFrom: aCByteArray
  "used to ensure that aCByteArray is kept alive by GC as long
   as receiver is alive. "
  derivedFrom := aCByteArray
%

category: 'Accessing elements'
method:
_signed: byteSize at: zeroBasedOffset with: thirdArg

"byteSize must be 1, 2, 4, 8, 9 or 10
 Values are accessed in CPU native byte order.

 If receiver is of size 8, zeroBasedOffset == -1 , and byteSize 1 2 or 4,
 accesses the value  (self int64At:0) in native byte order and 
 returns the specified number of least signifigant
 bytes of that value .

 If zeroBasedOffset == -2 , and byteSize == 8,
 returns the starting address of the C memory, as a signed 64bit Integer.
 There is no required alignment for zeroBasedOffset.

 if byteSize == 9 , treats the 8 bytes at zeroBasedOffset 
   as a  char*  pointer and returns nil if the pointer is NULL,
   or a new String based on strlen() of that pointer

 if byteSize == 10,  thirdArg is a subclass of CByteArray or CPointer, 
   create an instance of thirdArg which encapsulates a C pointer fetched from
   the specified zeroBasedOffset in receiver's C memory .
"
<primitive: 715>
self _primitiveFailed: #_signed:at:with:
     args: { byteSize . zeroBasedOffset . thirdArg }
%

method:
stringFromCharStarAt: zeroBasedOffset
  "Result is a String"
  ^ self _signed: 9 at: zeroBasedOffset with: nil
%

method:
pointerAt: zeroBasedOffset resultClass: aClass
  "Result is an instance of aClass encapsulating the C pointer
   fetched from 8 bytes of the receiver starting at zeroBasedOffset.
   zeroBasedOffset may need to be aligned on 8 bytes on some CPUs.
   aClass must be CByteArray, CPointer, a subclass of CByteArray 
   or a subclass of CPointer."

  ^ self _signed: 10 at: zeroBasedOffset with: aClass
%

method:
memoryAddress
  "Returns the starting address of the C memory, as a signed 64bit Integer."
  ^ self _signed: 8 at: -2 with: nil
%

method:
int8At: zeroBasedOffset

 "return specified signed 8 bit integer"
  ^ self _signed: 1 at: zeroBasedOffset  with: nil
%

method:
int16At: zeroBasedOffset
 "return specified signed 16 bit integer"
  ^ self _signed: 2 at: zeroBasedOffset  with: nil
%
method:
int32At: zeroBasedOffset
 "return specified signed 32 bit integer"
  ^ self _signed: 4 at: zeroBasedOffset  with: nil
%
method:
int64At: zeroBasedOffset

 "return specified signed 64 bit integer"
  ^ self _signed: 8 at: zeroBasedOffset  with: nil
%

method:
_unsigned: byteSize at: zeroBasedOffset

"For byteSize one of 0(double) 1, 2, 4, 8, 9(float)
   For zeroBasedOffset >= 0 , returns the integer 
   of specified byte size , or the 8 byte double, 
   starting at specified offset.

 Values are accessed in CPU native byte order.

 If receiver is of size 8, zeroBasedOffset == -1 , and byteSize 1 2 or 4,
 accesses the value  (self uint64At:0) in native byte order and 
 returns the specified number of least signifigant
 bytes of that value .

 If byteSize == -1, then receiver is searched for a byte value of 0
 starting at zeroBasedOffset, and the zero-based offset of the
 first such byte is returned, or -1 returned of no such byte found.

 There is no required alignment for zeroBasedOffset.
"

<primitive: 716>
self _primitiveFailed: #_unsigned:at: 
     args: { byteSize . zeroBasedOffset }
% 

method:
uint8At: zeroBasedOffset
 "return specified unsigned 8 bit integer"
 ^ self _unsigned: 1 at: zeroBasedOffset
%
method:
uint16At: zeroBasedOffset
 "return specified unsigned 16 bit integer"
 ^ self _unsigned: 2 at: zeroBasedOffset
%
method:
uint32At: zeroBasedOffset
 "return specified unsigned 32 bit integer"
 ^ self _unsigned: 4 at: zeroBasedOffset
%
method:
uint64At: zeroBasedOffset
 "return specified unsigned 64 bit integer"
 ^ self _unsigned: 8 at: zeroBasedOffset
%
method:
doubleAt: zeroBasedOffset 
 "fetch 8 byte float and return a Float or SmallDouble"
 ^ self _unsigned: 0 at: zeroBasedOffset
%
method:
floatAt: zeroBasedOffset 
 "fetch 4 byte float and return a Float or SmallDouble"
 ^ self _unsigned: 9 at: zeroBasedOffset
%
method: 
ccalloutAt: zeroBasedStart name: aString result: resType args: argumentTypes
  "Return a CCallout which encapsulates the pointer at zeroBasedStart in 
   the receiver as a C function that can be be called.
   resType and argumentTypes must conform to documentation in
     CCallout class >> library:name:result:args:
   " 
  | fAddr cco |
  fAddr := self pointerAt: zeroBasedStart resultClass: CPointer .
  cco := CCallout library: nil name: aString result: resType args: argumentTypes  .
  cco _bindFunction: fAddr .
  ^ cco 
%

category: 'Updating'

method:
autoRelease: aBoolean

"change the auto-free behavior of the receiver's C to match aBoolean.
 Generates an error if the receiver has VmOwned or Fenced bits set.
" 
  
<primitive: 866>
self _primitiveFailed: #autoRelease: args: { aBoolean }
%

method:
_signed: byteSize at: zeroBasedOffset truncatePut: aValue

"byteSize describes bytes of the receiver into which 
 aValue will be stored:
    -1 - 4 byte C float
    0 -  8 byte C double
    1, 2, 4, or 8 - integer of specified size
    9  -  aValue is a CByteArray( or nil)
          address of Cdata of CByteArray (or NULL)
         is to be stored for pointerAt:put:  ,
          or aValue is a CPointer, the encapsulated C pointer
         is to be stored for pointerAt:put:  ,
   10 - value is a CCallin or nil ,
        The native code address to be used for a call back , or NULL
         is to be stored for ccallinAt:put: .
   11 - value is a CCallout or nil ,
        The native code address to be used for a call out , or NULL
         is to be stored for ccalloutAt:put: .
 stores into the specified bytes of the receiver in CPU native byte order
 the specified least sigificant bytes of aValue . 
 There is no required alignment for zeroBasedOffset.
 If aValue is a Float and truncation to 4 byte C float
 would loose exponent bits and produce an Infinity
 an error is signaled, otherwise aValue is silently 
 truncated as needed to fit in the destination bytes. 
"

<primitive: 717>
byteSize _validateClass: SmallInteger .
zeroBasedOffset _validateClass: SmallInteger .
aValue ifNotNil:[ aValue _validateClass: CByteArray ].
(byteSize < -1 or:[ byteSize > 11]) ifTrue:[
   ArgumentError new name:'byteSize'; signal:'invalid byteSize arg'
].
byteSize <= 0 ifTrue:[ aValue _validateClass: Float ].
byteSize == 9 ifTrue:[
 aValue ifNotNil:[ aValue _validateClasses:{ CByteArray . CPointer }]
].
byteSize == 10 ifTrue:[
 aValue ifNotNil:[ aValue _validateClass: CCallin ].
].
byteSize == 11 ifTrue:[
 aValue ifNotNil:[ aValue _validateClass: CCallout ].
].
aValue _validateClass: Integer.
self _primitiveFailed: #_signed:at:truncatePut: 
     args: { byteSize . zeroBasedOffset . aValue }
%

method:
int8At: zeroBasedOffset truncatePut: anInteger
  "truncate anInteger to 8 bits and store"
  self _signed: 1 at: zeroBasedOffset truncatePut: anInteger
%
method:
int16At: zeroBasedOffset truncatePut: anInteger
  "truncate anInteger to 16 bits and store"
  self _signed: 2 at: zeroBasedOffset truncatePut: anInteger
%
method:
int32At: zeroBasedOffset truncatePut: anInteger
  "truncate anInteger to 32 bits and store"
  self _signed: 4 at: zeroBasedOffset truncatePut: anInteger
%
method:
int64At: zeroBasedOffset truncatePut: anInteger
  "truncate anInteger to 64 bits and store"
  self _signed: 8 at: zeroBasedOffset truncatePut: anInteger
%
method:
doubleAt: zeroBasedOffset truncatePut: aFloat
  "store a Float or SmallDouble into 64bits"
  self _signed: 0 at: zeroBasedOffset truncatePut: aFloat
%
method:
floatAt: zeroBasedOffset truncatePut: aFloat
  "store a Float or SmallDouble into 32bits"
  self _signed: -1 at: zeroBasedOffset truncatePut: aFloat
%

method:
pointerAt: zeroBasedOffset put: aCByteArray
  self _signed: 9 at: zeroBasedOffset truncatePut: aCByteArray .
  self _updateReferences: zeroBasedOffset value: aCByteArray .
%

method:
_updateReferences: zeroBasedOffset value: anObject
  | childs idx |
  anObject ifNotNil:[
    (childs := referencesTo) ifNil:[
      childs  := Array new .
      referencesTo := childs .
    ].
    childs size < (idx := zeroBasedOffset+1) ifTrue:[ childs size: idx ].
    childs at: idx put: anObject .
  ] ifNil:[
    (childs := referencesTo) ifNotNil:[
      idx := zeroBasedOffset + 1.
      childs size >= idx ifTrue:[ childs at: idx put: nil ].
    ]
  ]
%

! fixed 45644
method:
ccallinAt: zeroBasedOffset put: aCCallin
  "Store the C address for a bound CCallin. Returns receiver."
  self _signed: 10 at: zeroBasedOffset truncatePut: aCCallin .
  self _updateReferences: zeroBasedOffset value: aCCallin .
%

method:
ccalloutAt: zeroBasedOffset put: aCCallout
  "Store the C address for a CCallout. Returns receiver."

  (aCCallout isKindOf: CCallout) ifFalse:[ aCCallout _validateClass: CCallout].
  aCCallout bind .
  self _signed: 11 at: zeroBasedOffset truncatePut: aCCallout
%

method:
_signed: byteSize at: zeroBasedOffset put: aValue

"byteSize describes bytes of the receiver into which
 aValue is to be stored:
    1, 2, 4, or 8 - unsigned integer of specified size
   -1, -2, -4, -8 - signed integer of specified size
    9 -  8 byte C double 
    10 - 4 byte C float
 stores into the specified bytes of the receiver in CPU native byte order.
 the specified least sigificant bytes of aValue . 
 There is no required alignment for zeroBasedOffset.
 If aValue is an Integer and would loose bits by truncating 
 to the specified size, an error is signaled.
 If aValue is a Float, and truncation would loose exponent bits 
 or convert a non-zero to a zero, an error is signalled.
"

<primitive: 246>
byteSize _validateClass: SmallInteger .
zeroBasedOffset _validateClass: SmallInteger .
aValue ifNotNil:[ aValue _validateClass: CByteArray ].
(byteSize == 9 or:[ byteSize == 10]) ifTrue:[ 
  aValue _validateClass: Float 
] ifFalse:[
  (aValue isKindOf: Integer) ifTrue:[
    ArgumentError signal:'argument out of range for destination'
  ].
  aValue _validateClass: Integer.
].
"at this point byteSize is invalid"
self _primitiveFailed: #_signed:at:truncatePut:
     args: { byteSize . zeroBasedOffset . aValue }
%

method:
int8At: zeroBasedOffset put: anInteger
  "check that anInteger is representable in 8 bits and store"
  self _signed: -1 at: zeroBasedOffset put: anInteger
%
method:
int16At: zeroBasedOffset put: anInteger
  "check that anInteger is representable in 16 bits and store"
  self _signed: -2 at: zeroBasedOffset put: anInteger
%
method:
int32At: zeroBasedOffset put: anInteger
  "check that anInteger is representable in 32 bits and store"
  self _signed: -4 at: zeroBasedOffset put: anInteger
%
method:
int64At: zeroBasedOffset put: anInteger
  "check that anInteger is representable in 64 bits and store"
  self _signed: -8 at: zeroBasedOffset put: anInteger
%
method:
uint8At: zeroBasedOffset put: anInteger
  "check that anInteger is representable in 8 bits and store"
  self _signed: 1 at: zeroBasedOffset put: anInteger
%
method:
uint16At: zeroBasedOffset put: anInteger
  "check that anInteger is representable in 16 bits and store"
  self _signed: 2 at: zeroBasedOffset put: anInteger
%
method:
uint32At: zeroBasedOffset put: anInteger
  "check that anInteger is representable in 32 bits and store"
  self _signed: 4 at: zeroBasedOffset put: anInteger
%
method:
uint64At: zeroBasedOffset put: anInteger
  "check that anInteger is representable in 64 bits and store"
  self _signed: 8 at: zeroBasedOffset put: anInteger
%
method:
doubleAt: zeroBasedOffset put: aFloat
  "store a Float or SmallDouble into 64bits"
  self _signed: 9 at: zeroBasedOffset put: aFloat
%
method:
floatAt: zeroBasedOffset put: aFloat
  "store a Float or SmallDouble into 32bits.
  if aFloat would loose exponent bits 
  or convert a non-zero to a zero when converted to
  a 4 byte C float, signals an error."
  self _signed: 10 at: zeroBasedOffset put: aFloat
%

method:
memset: anInt from: zeroBasedStart to: zeroBasedEnd

 "set specified bytes of the receiver to the value anInt .
  end==-1 means to the end of the receiver.
  anInt must be >=0  and <= 255 . "

 <primitive: 712>
 | sz |
 anInt _validateClass: SmallInteger .
 (anInt < 0 or:[ anInt > 255]) ifTrue:[ 
   anInt _error: #rtErrArgOutOfRange args:{ 0 . 255 }
 ].
 zeroBasedStart _validateClass: SmallInteger .
 sz := self size .
 (zeroBasedStart < 0 or:[ zeroBasedStart >= sz ]) ifTrue:[
   zeroBasedStart _error: #rtErrArgOutOfRange args:{ 0 . sz - 1 }
 ].
 zeroBasedEnd _validateClass: SmallInteger .
 (zeroBasedEnd < -1 or:[ zeroBasedEnd >= sz ]) ifTrue:[
   zeroBasedEnd _error: #rtErrArgOutOfRange: args:{ -1 . sz - 1 }
 ]. 
 "if we get here the receiver is probably a committed instance that
  has been faulted in and has no Cdata."
 self _primitiveFailed: #memset:from:to:
      args: { anInt . zeroBasedStart . zeroBasedEnd }
%

! rename this to  replaceFrom: start to: stop with: anObject startingAt: repIdx
method:
copyBytesFrom: anObject from: oneBasedStart to: oneBasedEnd into: zeroBasedDestOffset

"anObject may be any byte format object or a CByteArray.
 copies specified bytes of anObject into receiver.

 An error is signalled of any of the following are true
    oneBasedStart < 1
    zeroBasedDestOffset < 0
    self size > 0 and:[ zeroBasedDestOffset >= self size ]

 Returns number of bytes copied (possibly zero) .

 Warning, ByteArray usually stores integers in big-endian order, 
 not CPU native byte order. Most other byte format classes
 use CPU native byte order."

<primitive: 713>
zeroBasedDestOffset _validateClass: SmallInteger . 
oneBasedStart _validateClass: SmallInteger .
oneBasedEnd _validateClass: SmallInteger .

"if we get here, either receiver or a CByteArray passed as anObject 
 is probably a committed instance that has been faulted in and has no Cdata."
self _primitiveFailed: #copyBytesFrom:from:to:into:
     args: { anObject . oneBasedStart .  oneBasedEnd . zeroBasedDestOffset }
%

! fixed 46935
method:
copyFrom: anObj from: oneBasedStart to: oneBasedEnd into: zeroBasedDestOffset

"anObject may be a String, ByteArray or CByteArray .
 copies specified bytes of anObject into receiver . 

 An error is signalled of any of the following are true
    oneBasedStart < 1
    zeroBasedDestOffset < 0
    self size > 0 and:[ zeroBasedDestOffset >= self size ]

 The number of bytes copied may be zero.  
 The method copyBytesFrom:from:to:into:   returns the number of bytes copied.

 Warning, ByteArray usually stores integers in big-endian order, 
 not CPU native byte order."

 self copyBytesFrom: anObj from: oneBasedStart to: oneBasedEnd into: zeroBasedDestOffset .
 ^ self
%



category: 'Private'
method: 
instVarAt: offset put: value

 "disallowed"
 self shouldNotImplement: #instVarAt:put:
%

method:
_copyFrom: zeroBasedStart to: zeroBasedEnd resKind: aClass
  "zeroBasedStart, zeroBasedEnd must be SmallInteger in range
    of the receiver's C memory.
   aClass must be a subclass of CByteArray, String or ByteArray ."
<primitive: 714>
zeroBasedStart _validateClass: SmallInteger .
zeroBasedEnd _validateClass: SmallInteger .
"at this point byteSize or zeroBasedOffset are probably out of range, 
  or aClass is invalid"
^self _primitiveFailed: #_copyFrom:to:resKind:
      args: { zeroBasedStart . zeroBasedEnd . aClass }
%

category: 'Copying'
method:
stringFrom: zeroBasedStart to: zeroBasedEnd

 "return a new String containing the specified bytes of the receiver."

^ self _copyFrom: zeroBasedStart to: zeroBasedEnd resKind: String 
%

method:
byteArrayFrom: zeroBasedStart to: zeroBasedEnd

 "return a new ByteArray containing the specified bytes of the receiver."

^ self _copyFrom: zeroBasedStart to: zeroBasedEnd resKind: ByteArray 
%

method:
CByteArrayFrom: zeroBasedStart to: zeroBasedEnd
 "return a new CByteArray , auto freed, containing specified bytes of receiver."

^ self _copyFrom: zeroBasedStart to: zeroBasedEnd resKind: CByteArray 
%

category: 'Formatting'

method:
asString
  | res |
  (res := super asString ) add: self _inspect .
  ^ res
%
method:
_inspect
  | res |
  (res := String new)
    add: ' size='; add: self size asString ;
    add: ' gcFree=0x'; add: self gcFreeKind asHexString ;
    add: ' dead='; add: self isDead asString ;
    add: ' address=0x' ; add: self memoryAddress asHexString .
  ^ res
%

category: 'Copying'
method:
copy
 "disallowed"
  self shouldNotImplement: #copy
%  
