!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
! Superclass Hierarchy:
!   GsFindSingleRefPathForObject, Object.
!
!=========================================================================


expectvalue /Class
run
GsSingleRefPathFinderForObject comment: 
'GsSingleRefPathFinderForObject holds the state of a reference path search for one
search object.  It has the following instance variables:

completed - A Boolean indicating if the search for searchOop has finished.
True if the reference path was found or if the object was
determined to be dead.

isDead - A Boolean indicating if the object is unreachable and eligible for
garbage collection.

refPathFinder - Reference to the instance of GsSingleRefPathFinder that created
this object.

limitOopsFound - a GsBitmap containing the objects in the limit set which can
be traversed to reach searchOop.

searchOopsUnion - a GsBitmap containing all objects already searched for in
this reference path scan.

searchOop - the object for which the reference path is to be found.

resultObject - an instance of GsSingleRefPathResult which contains the
reference path to searchOop.

parentBitmaps - an Array of GsBitmaps which contain ancestors of searchOop.
'
%

! ------------------- Remove existing behavior from GsSingleRefPathFinderForObject
removeallmethods GsSingleRefPathFinderForObject
removeallclassmethods GsSingleRefPathFinderForObject
! ------------------- Class methods for GsSingleRefPathFinderForObject
set compile_env: 0
category: 'Instance Creation'
classmethod: GsSingleRefPathFinderForObject
newForSearchObject: anObject refPathFinder: aFinder

|result|
result := self new.
result initializeForSearchObject: anObject refPathFinder: aFinder.
^ result
%
! ------------------- Instance methods for GsSingleRefPathFinderForObject

category: 'Accessing'
method: GsSingleRefPathFinderForObject
completed
^completed
%
category: 'Accessing'
method: GsSingleRefPathFinderForObject
isDead
^isDead
%
category: 'Accessing'
method: GsSingleRefPathFinderForObject
limitObjects

^ refPathFinder limitObjects last.
%
category: 'Accessing'
method: GsSingleRefPathFinderForObject
limitOopsFound
^limitOopsFound
%
category: 'Accessing'
method: GsSingleRefPathFinderForObject
parentBitmaps
^parentBitmaps
%
category: 'Accessing'
method: GsSingleRefPathFinderForObject
refPathFinder
^refPathFinder
%
category: 'Accessing'
method: GsSingleRefPathFinderForObject
resultObject
^resultObject
%
category: 'Accessing'
method: GsSingleRefPathFinderForObject
searchOop
^searchOop
%
category: 'Accessing'
method: GsSingleRefPathFinderForObject
searchOopsUnion
^searchOopsUnion
%

category: 'Initialization'
method: GsSingleRefPathFinderForObject
initializeForSearchObject: anObject refPathFinder: finderObj

completed := false.
isDead := false.
refPathFinder := finderObj.
limitOopsFound := nil.
searchOopsUnion := GsBitmap new.
searchOop := anObject.
parentBitmaps := Array with: (GsBitmap with: anObject).
^ self
%

category: 'Logging'
method: GsSingleRefPathFinderForObject
buildSummaryHeader

| msg |
msg := String withAll: '   Summary for search oop '.
msg
addAll: searchOop asOop asString;
add: Character lf.
^msg
%
category: 'Logging'
method: GsSingleRefPathFinderForObject
logReferencePath

refPathFinder printToLog
   ifTrue:[ self buildResultObject logReferencePath ]

%
category: 'Logging'
method: GsSingleRefPathFinderForObject
printScanCompleted

refPathFinder printToLog ifTrue:[| msg |
  msg := self buildSummaryHeader.
  msg addAll: '      scan completed'.
  isDead ifTrue: [msg addAll: ' (object is dead)'].
  msg add: Character lf.
  GsFile gciLogServer: msg]
%
category: 'Logging'
method: GsSingleRefPathFinderForObject
printScanSummary: parentOops

refPathFinder printToLog ifTrue:[
  | msg numSeen numLimit scanCompleted oopIsDead parentsCopy |
  msg := self buildSummaryHeader.
  numLimit := parentOops intersectSize: self limitObjects.
  numSeen := parentOops intersectSize: searchOopsUnion.
  msg
    addAll: '      ';
    addAll: self childrenToFind size asString;
    addAll: ' child oops are referenced by ';
    addAll: parentOops size asString;
    addAll: ' parent oops (';
    addAll: numSeen asString;
    addAll: ' were already seen and ';
    addAll: numLimit asString;
    addAll: ' are limit objects)';
    add: Character lf.
  parentsCopy := parentOops copy.
  parentsCopy removeAll: self limitObjects.
  parentsCopy removeAll: searchOopsUnion.
  scanCompleted := parentsCopy isEmpty or:[numLimit > 0].
  parentsCopy removeAll .
  scanCompleted
    ifTrue:[
      oopIsDead := numLimit == 0.
      msg addAll: '      *** Scan completed ***'.
      oopIsDead ifTrue: [msg addAll: ' (object is dead)'].
      msg addAll: ' Total time is ';
      addAll: self totalTime asString;
      addAll: ' seconds' ;
      add: Character lf].
  GsFile gciLogServer: msg]

%
category: 'Logging'
method: GsSingleRefPathFinderForObject
totalTime

^System timeGmt2005 - refPathFinder scanStartTime
%

category: 'Results'
method: GsSingleRefPathFinderForObject
buildPathToDefaultLimitSet

^ self buildPathToDefaultLimitSetStartingAt: refPathFinder limitObjects size
%
category: 'Results'
method: GsSingleRefPathFinderForObject
buildPathToDefaultLimitSetStartingAt: index

"Build the reference path going from an object referenced by a descendant of
 the default limit set up to an object in the default limit set."

| result childObj limitObjsArray |
result := Array new.
childObj := limitOopsFound peek.
"Caller will add the first object in limitOopsFound to the path.  Do not do it here."
limitObjsArray := refPathFinder limitObjects.
(index - 1) downTo: 1 do:[:n| | parent bm |
  bm := limitObjsArray at: n.
  parent := bm primFirstObjectThatReferences: childObj.
  parent ifNil:[
    "This should never happen"
    self halt:
     'Logic error: no references to child object found in set of limit object descendants'].
  result add: parent.
  childObj := parent.
].
^ result reverse
%
category: 'Results'
method: GsSingleRefPathFinderForObject
buildReferencePath

| result nextObject index done|

(isDead or:[ completed not]) ifTrue:[ ^ Array new ].

result := self buildPathToDefaultLimitSet .
index := parentBitmaps size.
nextObject := limitOopsFound peek.
done := nextObject == nil.
[done] whileFalse:[ |nextObjectChildren childrenInPath intersection|
  result add: nextObject.
  nextObjectChildren := (GsBitmap with: nextObject) primReferencedObjects.
  childrenInPath := parentBitmaps at: index.
  intersection := nextObjectChildren * childrenInPath.
  intersection isEmpty ifTrue:[ self halt: 'unexpected empty intersection'].
  done := intersection includes: searchOop. 
  nextObject := intersection peek.
  index := index - 1.
].
result add: searchOop.
^ result
%
category: 'Results'
method: GsSingleRefPathFinderForObject
buildResultObject

resultObject == nil
  ifTrue:[
    resultObject := GsSingleRefPathResult 
         newForSearchObject: searchOop
         isDead: isDead
         path: self buildReferencePath].
^ resultObject
%

category: 'Scanning'
method: GsSingleRefPathFinderForObject
childrenToFind

"Answer a GsBitmap of the children to search for in the next scan, or nil if 
 the scan is finished"
 
^ completed ifTrue:[nil] ifFalse:[ parentBitmaps last].
%
category: 'Scanning'
method: GsSingleRefPathFinderForObject
handleLimitSetDescendantsWithSearchObject

| limitObjectsArray |
limitObjectsArray := refPathFinder limitObjects.
2 to: limitObjectsArray size do:[:n |
  ((limitObjectsArray at: n) includes: searchOop) ifTrue:[ | path |
    limitOopsFound := GsBitmap with: searchOop.
    path := self buildPathToDefaultLimitSetStartingAt: n.
    path add: searchOop.
    resultObject := GsSingleRefPathResult
       newForSearchObject: searchOop
       isDead: false
       path: path.
    self setReferencePathFound.
    ^self]
].
^self
%
category: 'Scanning'
method: GsSingleRefPathFinderForObject
processResultsOfScan: parentOopsBm

|limitOops|
self printScanSummary: parentOopsBm .
parentOopsBm removeAll: searchOopsUnion. "remove ones we've already seen"
parentOopsBm isEmpty
  ifTrue:[ ^ self setObjectIsDead ].

limitOops := parentOopsBm * self limitObjects .
limitOops isEmpty
ifTrue:[ parentBitmaps add: parentOopsBm]
ifFalse:[
  limitOopsFound := limitOops.
  self setReferencePathFound .
].
^ self
%
category: 'Scanning'
method: GsSingleRefPathFinderForObject
updateSearchOopsUnion

"Add parent objects found in the previous scan to searchOopsUnion"
completed ifFalse:[ searchOopsUnion addAll: parentBitmaps last].
%

category: 'Updating'
method: GsSingleRefPathFinderForObject
completed: newValue
completed := newValue
%
category: 'Updating'
method: GsSingleRefPathFinderForObject
isDead: newValue
isDead := newValue
%
category: 'Updating'
method: GsSingleRefPathFinderForObject
limitOopsFound: newValue
limitOopsFound := newValue
%
category: 'Updating'
method: GsSingleRefPathFinderForObject
parentBitmaps: newValue
parentBitmaps := newValue
%
category: 'Updating'
method: GsSingleRefPathFinderForObject
refPathFinder: newValue
refPathFinder := newValue
%
category: 'Updating'
method: GsSingleRefPathFinderForObject
resultObject: newValue
resultObject := newValue
%
category: 'Updating'
method: GsSingleRefPathFinderForObject
searchOop: newValue
searchOop := newValue
%
category: 'Updating'
method: GsSingleRefPathFinderForObject
searchOopsUnion: newValue
searchOopsUnion := newValue
%
category: 'Updating'
method: GsSingleRefPathFinderForObject
setObjectIsDead

completed := true. "no unique parents found.  object is dead"
isDead := true.
refPathFinder completedOneSearch.
^ self
%
category: 'Updating'
method: GsSingleRefPathFinderForObject
setReferencePathFound

completed := true.
refPathFinder completedOneSearch.
self logReferencePath.
^ self
%

