! ========================================================================
! Copyright (C) by GemTalk Systems 1991-2020.  All Rights Reserved
!
! $Id$
!
! ========================================================================

expectvalue %String
run
^ Object _newKernelSubclass: #GsTlsCredential
  instVarNames: #()
  classVars: #()
  classInstVars: #()
  poolDictionaries: { }
  inDictionary: Globals
  options: #(#instancesNonPersistent)
  reservedOop: 1953
%

doit
GsTlsCredential category: 'X509'.
true
%

! Remove existing behavior from GsTlsCredential
removeallmethods GsTlsCredential
removeallclassmethods GsTlsCredential
set class GsTlsCredential

! ------------------- Class methods for GsTlsCredential
classmethod:
comment
^'GsTlsCredential is a class that encapsulates TLS private keys, public keys,
  and X509 certificates. Instances contain a hidden reference to C pointer
  to the OpenSSL representation of the TLS object.'
%

input $upgradeDir/tlsclassprim.gs

category: 'Private'
classmethod:
newFromPemString: aString
  self subclassResponsibility: #newFromPemString:
%  

category: 'Private'
classmethod:
tls0ArgClassPrim: opCode

<primitive: 1057>
^ self _primitiveFailed: #tls0ArgClassPrim: args: { opCode }
%

category: 'Private'
method:
tls0ArgInstPrim: opCode

<primitive: 1057>
^ self _primitiveFailed: #tls0ArgInstPrim: args: { opCode }
%


category: 'Private'
method:
tls1ArgInstPrim: opCode with: anObj

<primitive: 1058>
^ self _primitiveFailed: #tls1ArgInstPrim: args: { opCode . anObj }
%
category: 'Converting'
method:
asPemString

^ self tls0ArgInstPrim: 0
%
category: 'Hashing'
method:
hash

^ self tls0ArgInstPrim: 1
%
category: 'Testing'
method:
= anObj

^ self tls1ArgInstPrim: 0 with: anObj
%

category: 'Testing'
method:
isPrivateKey
  self subclassResponsibility: #isPrivateKey
%
category: 'Testing'
method:
isPublicKey
  self subclassResponsibility: #isPublicKey
%
category: 'Testing'
method:
isX509Certificate
  self subclassResponsibility: #isX509Certificate
%
category: 'Accessing'
method:
algorithm

"Answers a Symbol indicating the type of high-level PKI algorithm the 
receiver uses. The high-level PKI algorithms currently supported are:

   #RSA - Rivest-Shamir-Adleman
   #DSA - Data Signature Algorithm
   #EC  - Elliptic Curve Cryptography

 All high-level algorithm have various sub-types. Use the sslAlgorithm 
 method to obtain information about the specific PKI algorithm of the
 receiver.

 If the receiver is an instance of GsX509Certificate, the result 
 indicates the algorithm of the public key contained therein.

 Returns #UNSUPPORTED if the algorithm could not be determined."
 
^ self tls0ArgInstPrim: 2
%

category: 'Accessing'
method:
sslAlgorithm

"Answers a Symbol indicating the SSL type of PKI algorithm the 
 receiver uses. The values returned by OpenSSL as of version
 1.1.1 are:

    #EVP_PKEY_NONE
    #EVP_PKEY_RSA
    #EVP_PKEY_RSA2
    #EVP_PKEY_RSA_PSS
    #EVP_PKEY_DSA
    #EVP_PKEY_DSA1
    #EVP_PKEY_DSA2
    #EVP_PKEY_DSA3
    #EVP_PKEY_DSA4
    #EVP_PKEY_DH
    #EVP_PKEY_DHX
    #EVP_PKEY_EC
    #EVP_PKEY_SM2
    #EVP_PKEY_HMAC
    #EVP_PKEY_CMAC
    #EVP_PKEY_SCRYPT
    #EVP_PKEY_TLS1_PRF
    #EVP_PKEY_HKDF
    #EVP_PKEY_POLY1305
    #EVP_PKEY_SIPHASH
    #EVP_PKEY_X25519
    #EVP_PKEY_ED25519
    #EVP_PKEY_X448
    #EVP_PKEY_ED448

 If the recevier is an instance of GsX509Certificate, the result 
 indicates the algorithm of the public key contained therein."
 
^ self tls0ArgInstPrim: 3
%

category: 'Printing'
method:
description
"Answers a String obtained from SSL describing the receiver. The contents
 and format of the string vary depending on the receiver's class."
 
^ self tls0ArgInstPrim: 4
%

category: 'Accessing'
method:
securityBits
"Answers a SmallInteger representing the number of security bits of the 
 receiver.  Bits of security is defined in NIST SP800-57.  See 
   https://nvlpubs.nist.gov/nistpubs/SpecialPublications/NIST.SP.800-57pt1r4.pdf
 for further information.

 For instances of GsX509Certificate, answers the security bits of the public
 key contained therein."

^ self tls0ArgInstPrim: 5
%

category: 'Testing'
method:
supportsDigitalSignatures
"Answers a Boolean indicating if the receiver supports digital signatures.
 The type of support offered, either signing or verifying, depends upon the
 class of the receiver."

^ self tls0ArgInstPrim: 6
%
category: 'Testing'
method:
canVerifyDigitalSignatures
  self subclassResponsibility: #canVerifyDigitalSignatures
%
category: 'Testing'
method:
canCreateDigitalSignatures
  self subclassResponsibility: #canCreateDigitalSignatures
%

category: 'Private'
method:
_validateAlgorithm: expectedSymbol

^ self algorithm == expectedSymbol
    ifTrue:[ true ]
   ifFalse:[ CryptoError signal:
  ('Incorrect key algorithm. Expected: ', expectedSymbol asString, ' Actual: ', self algorithm asString)]
%

category: 'Private'
method:
_validateIsPrivateKey
^ self isPrivateKey
    ifTrue:[ true ]
   ifFalse:[ CryptoError signal: 'Incorrect key type: expected a private key']
%

category: 'Private'
method:
_validateIsPublicKey
^ self isPublicKey
    ifTrue:[ true ]
   ifFalse:[ CryptoError signal: 'Incorrect key type: expected a public key']
%

category: 'Storing and Loading'
classmethod:
loadFrom: passiveObj

"Creates and returns an active instance of the receiver from the passive form
 of the object"

| inst |
inst := self newFromPemString: passiveObj readObject .
passiveObj hasRead: inst.
^inst.
%

category: 'Storing and Loading'
method:
writeTo: passiveObj

"Writes the passive form of the receiver into passiveObj, expressed as
 a PEM string."

passiveObj writeClass: self class.
self asPemString writeTo: passiveObj .
passiveObj space
%

category: 'Private'
method:
_validateIsRsa

^ self algorithm == #RSA
     ifTrue:[ true ]
    ifFalse:[  CryptoError signal: 'An RSA key or certificate is required']
%

category: 'Comparing'
method:
matches: anotherKey

"Determines if the receiver and anotherKey match each other as a valid
 public-private key pair. If the receiver is a public key, anotherKey
 is expected to be a private key. If the receiver is a private key, 
 another key is expected to be a public key.

 If the receiver or anotherKey is an instance of GsX509Certificate, the
 public key is extracted from the certificate and the comparison is
 performed as described above.

 RSA and DSA key pairs match if both keys use the same modulus.
 Elliptic curve key pairs match if both keys use the same curve and the 
 same point on that curve.

 Returns true if the keys match, false if they do not match.
 Raises an error if the receiver and anotherKey are both the same type
 (public or private keys)."
 
^ self tls1ArgInstPrim: 2 with: anotherKey
%

category: 'Private'
method:
_validateCreatesDigitalSignatures
^ self canCreateDigitalSignatures
    ifTrue:[ true ]
   ifFalse:[ CryptoError signal: 'This key is not valid for creating digital signatures']
%

category: 'Private'
method:
_validateValidatesDigitalSignatures
^ self canVerifyDigitalSignatures
    ifTrue:[ true ]
   ifFalse:[ CryptoError signal: 'This key is not valid for validating digital signatures']
%
