! ========================================================================
! Copyright (C) by GemTalk Systems 1991-2020.  All Rights Reserved
!
! $Id$
!
! Superclass Hierarchy
!	AbstractDictionary, Collection, Object.
!
! ========================================================================

removeallmethods AbstractDictionary 
removeallclassmethods AbstractDictionary 
set class AbstractDictionary

category: 'For Documentation Installation only'
classmethod:
installDocumentation

self comment: 
'AbstractDictionary is an abstract class that provides the protocol for
 collections whose elements can be accessed by an associated lookup key.
 Concrete classes of AbstractDictionary store the key-value pairs either
 directly or as Associations.  See the documentation for the Dictionary and
 KeyValueDictionary classes for more information.'.
%

! ------------------- Class methods for AbstractDictionary

category: 'Instance Creation'
classmethod:
new

"(Subclass responsibility.)  Returns a new instance of AbstractDictionary."

^ self subclassResponsibility: #new

%
category: 'Instance Creation'
classmethod:
new: count

"(Subclass responsibility.) Returns a new instance of AbstractDictionary. The 
 argument count provides a hint of the number of elements the instance should 
 be designed to hold."

^ self subclassResponsibility: #new:
%

! ------------------- Instance methods for AbstractDictionary

! edited to fix 37666
category: 'Private'
method:
_asCollectionForSorting

"Creates a new collection that can be easily indexed for sorting."

| result |

result := { } .
self do: [ :aValue | result add: aValue].
^ result.
%

category: 'Private'
method:
_errorKeyNotFound: aKey

"No Association or key-value pair with given key, aKey, was found."

^ self _error: #rtErrKeyNotFound args: { aKey }
%

category: 'Adding'
method:
add: anAssociation

"Adds the Association or the key-value pair contained in anAssociation to the
 receiver.  If the receiver already includes an Association/key-value pair 
 whose key is equal to that of anAssociation, then this method redefines the 
 value portion of that Association/key-value pair. Returns anAssociation."

self at: anAssociation key put: anAssociation value.
^ anAssociation.
%

! changed with fix 36675
category: 'Adding'
method:
addAll: aCollection

"Adds to the receiver all the Associations or key-value pairs contained in
 aCollection.  aCollection must be a collection of Associations or a dictionary.
 Returns the argument, aCollection."

(aCollection isKindOf: AbstractDictionary) ifTrue: [
  aCollection accompaniedBy: self 
    keysAndValuesDo:[ :rcvr :aKey :aVal| rcvr at: aKey put: aVal ]. 
] ifFalse: [
  aCollection accompaniedBy: self do: [ :me :anAssociation | me add: anAssociation ]
].
^ aCollection.
%

category: 'Enumerating'
method:
associationsDo: aBlock

"Iteratively evaluates the one argument block, aBlock, using each Association
 in the receiver as the argument to the block. If the receiver stores the 
 key-value pairs directly, new Associations are created for the key-value pairs
 and used as the arguments to aBlock.  Returns the receiver."

^ self subclassResponsibility: #associationsDo:
%

category: 'Searching'
method:
includesAssociation: anAssociation

"Returns true if anAssociation is equal to one of the Associations of the 
 receiver.  Returns false otherwise."

self associationsDo: [ :assoc | (anAssociation = assoc) ifTrue: [ ^true ]].
^false
%

category: 'Searching'
method:
includesIdenticalAssociation: anAssociation

"Returns true if anAssociation is identical to one of the Associations of the
 receiver.  Returns false otherwise."

self associationsDo: [ :assoc | (anAssociation == assoc) ifTrue: [ ^true ]].
^false
%

category: 'Enumerating'
method:
associationsDetect: aBlock

"Evaluates aBlock repeatedly, with the Associations of the receiver as the
 argument.  Returns the first Association for which the block evaluates to true
 when the Association is used as the argument to the block.  If none of the
 receiver's Associations evaluates to true, generates an error.  The argument
 aBlock must be a one-argument block."

^ self associationsDetect: aBlock
       ifNone: [^ self _error: #assocErrNoElementsDetected args: { aBlock }]
%

category: 'Enumerating'
method:
associationsDetect: aBlock ifNone: exceptionBlock

"Evaluates aBlock repeatedly, with the Associations of the receiver as the 
 argument.  Returns the first Association for which aBlock evaluates to true 
 when the Association is used as the argument to the block.  If none of the
 receiver's Associations evaluates to true, this method evaluates the argument
 exceptionBlock and returns its value.  The argument aBlock must be a
 one-argument block, and exceptionBlock must be a zero-argument block."

self associationsDo: [ :anAssoc| 
  (aBlock value: anAssoc) ifTrue: [ ^anAssoc ].
  ].

^ exceptionBlock value.
%

category: 'Accessing'
method:
at: aKey

"Returns the value of the Association with key aKey.  Generates an error if no 
 such key exists."

    | sentinel result |
    sentinel := AbsDictSentinel"created in bom.c"  .
    (result := self at: aKey otherwise: sentinel) == sentinel ifTrue:
        [^self _errorKeyNotFound: aKey].
    ^result
%

category: 'Accessing'
method:
at: aKey ifAbsent: aBlock

"Returns the value associated with key aKey.  If no such key/value pair or
 Association exists, returns the result of evaluating the zero-argument block
 aBlock."

^ self subclassResponsibility: #at:ifAbsent.
%

category: 'Accessing'
method:
at: aKey ifAbsentPut: aBlock

"Returns the value associated with key aKey. If no such key/value pair or
 Association exists, returns the result of evaluating the zero-argument block
 aBlock. In the latter case, the result of  evaluating the block aBlock is 
 also stored in the receiver using the key aKey."

| known existing value |

known := 'a'.
existing := self at: aKey otherwise: known.
existing == known 
  ifTrue: [
    value := aBlock value.
    self at: aKey put: value.
    ^value 
    ].
^existing
%

category: 'Accessing'
method:
at: aKey otherwise: value

"Returns the value that corresponds to aKey.  If no such key/value pair or
 Association exists, returns the given alternate value."

    ^self at: aKey ifAbsent: [value]
%

category: 'Updating'
method:
at: aKey put: aValue

"Creates a new Association with the given key and value and adds it to the
 receiver or adds the key-value pair to the receiver depending on the class of
 the receiver.  If the receiver already contains an Association or key-value
 pair with the given key, this method makes aValue the value of that 
 Association or key-value pair.  Returns aValue."

^ self subclassResponsibility: #at:put:
%

category: 'Enumerating'
method:
collect: aBlock

"Evaluates aBlock with each of the receiver's values as the argument and
 collects the resulting values into the appropriate Dictionary at the
 corresponding key values."

| result |

result := self speciesForCollect new: (self size).
self keysAndValuesDo: [ :aKey :aValue |
  result at: aKey put: (aBlock value: aValue).
  ].

^ result. 
%

category: 'Enumerating'
method:
collectAssociations: aBlock

"Evaluates aBlock with each of the receiver's Associations (or Associations 
 created using the key-value pairs) as the argument and collects the resulting 
 values into an Array. Returns the newly created Array."

| anArray index |

anArray := Array new: (self size).
index := 0.
self associationsDo: [ :anAssoc |
  index := index + 1.     
  anArray at: index put: (aBlock value: anAssoc).
  ].

^ anArray. 
%

category: 'Deprecated'
method:
collectValues: aBlock

self deprecated: 'AbstractDictionary>>collectValues: deprecated long before v3.0. Use collect: instead.'.
^ self collect: aBlock.
%

category: 'Enumerating'
method:
collectValuesAsArray: aBlock

"Evaluates aBlock with each of the receiver's values as the argument and
 collects the resulting values into an Array. Returns the new Array."

| result |
result := { } .
self valuesDo: [ :aValue |
  result add: (aBlock value: aValue).
  ].
^ result. 
%

category: 'Deprecated'
method:
detectValues: aBlock 

self deprecated: 'AbstractDictionary>>detectValues: deprecated long before v3.0.  Use keysAndValuesDo: instead.'.
^ self detectValues: aBlock ifNone: [ ^ self errorNoElementDetected: aBlock ].
%

category: 'Deprecated'
method:
detectValues: aBlock ifNone: exceptionBlock

self deprecated: 'AbstractDictionary>>detectValues:ifNone: deprecated long before v3.0. Use keysAndValuesDo: instead.'.
self keysAndValuesDo: [ :aKey :aValue |
  (aBlock value: aValue) ifTrue: [ ^ aKey ].
  ].
^ exceptionBlock value.
%

category: 'Deprecated'
method:
detectAssociations: aBlock 

self deprecated: 'AbstractDictionary>>detectAssociations: deprecated long before v3.0. Use associationsDetect: instead.'.
^ self associationsDetect: aBlock 
%

category: 'Deprecated'
method:
detectAssociations: aBlock ifNone: exceptionBlock

self deprecated: 'AbstractDictionary>>detectAssociations:ifNone: deprecated long before v3.0. Use associationsDetect:ifNone: instead.'.
^ self associationsDetect: aBlock ifNone: exceptionBlock
%

category: 'Deprecated'
method:
doAssociations: aBlock

self deprecated: 'AbstractDictionary>>doAssociations: deprecated long before v3.0. Use associationsDo: instead.'.
^ self associationsDo: aBlock.
%

category: 'Deprecated'
method:
doKeys: aBlock

self deprecated: 'AbstractDictionary>>doKeys: deprecated long before v3.0. Use keysDo: instead.'.
^ self keysDo: aBlock.
%

category: 'Deprecated'
method:
doKeysAndValues: aBlock

self deprecated: 'AbstractDictionary>>doKeysAndValues: deprecated long before v3.0. Use keysAndValuesDo: instead.'.
^ self keysAndValuesDo: aBlock.
%

category: 'Deprecated'
method:
doValues: aBlock

self deprecated: 'AbstractDictionary>>doValues: deprecated long before v3.0. Use valuesDo: instead.'.
^ self valuesDo: aBlock
%

category: 'Private'
method:
errorKeyNotFound: aKey

"No Association or key/value pair with the given key, aKey was found."

^ self _error: #rtErrKeyNotFound args: { aKey }
%

category: 'Private'
method:
errorNilKey

"A nil key was provided as an argument."

^ self _error: #rtErrNilKey.
%

category: 'Private'
method:
errorNoElementDetected: aBlock

"Private."

^ self _error: #assocErrNoElementsDetected args: { aBlock }
%

category: 'Private'
method:
tableSize

"Returns the size of hash table used for storing the entries."

^ self subclassResponsibility: #tableSize.
%

category: 'Accessing'
method:
size

"Returns the number of elements (Associations/key-value pairs) contained in the
 receiver."

^ self subclassResponsibility: #size.
%

category: 'Updating'
method:
size: newSize

"Disallowed.  You should not change the size of a dictionary explicitly."

self shouldNotImplement: #size: .
%

category: 'Hashing'
method:
hashFunction: aKey

"The hash function should perform some operation on the value of the
 key (aKey) which returns a value in the range 1..tableSize."

^(aKey hash \\  self tableSize) + 1
%

category: 'Searching'
method:
includes: aValue

"Returns true if the receiver contains a value that is equal to aValue.
 Returns false otherwise."

self valuesDo: [ :element | (aValue = element) ifTrue: [ ^true ]].
^ false.
%

category: 'Searching'
method:
includesIdentical: aValue

"Returns true if the receiver contains a value that is identical to aValue.
 Returns false otherwise."

self valuesDo: [ :element | (aValue == element) ifTrue: [ ^true ]].
^ false.
%

! NOTE: gs64 v3.0
! edited  includesKey: do not do a return to home from within
!  the if-absent block so usage from protected methods in UserProfile 
!  will not fail due to protected mode being cleared by  return to home .
category: 'Searching'
method:
includesKey: aKey

"Returns true if the receiver contains an Association or a key-value pair whose
 key is equal to aKey.  Returns false otherwise."

| res |
res := true .
self at: aKey ifAbsent:[ res := false ] .
^ res .
%

category: 'Deprecated'
method:
includesValue: aValue

self deprecated: 'AbstractDictionary>>includesValue: deprecated long before v3.0. Use includes: instead.'.
^ self includes: aValue.
%

category: 'Accessing'
method:
keyAtValue: anObject

"Returns the key of the first value equal to anObject. If no match is found, 
 runtime error objErrNotInColl is signaled."

"Note: In some implementations of Smalltalk, nil is returned if a match is 
 not found."

^self keyAtValue: anObject
      ifAbsent: [^ self _error: #objErrNotInColl args: { anObject }]
%

category: 'Accessing'
method:
keyAtValue: anObject ifAbsent: aBlock

"Returns the key of the first value equal to the given object, anObject.  
 If no match is found, evaluates and returns the result of the block aBlock."

^ self subclassResponsibility: #keyAtValue:ifAbsent: 
%

category: 'Accessing'
method:
keys

"Returns a Set containing the receiver's keys."

| aSet |

aSet := Set new.
self keysDo: [ :key | aSet add: key ].
^ aSet
%

category: 'Enumerating'
method:
do: aBlock

"Iteratively evaluates the one argument block, aBlock, using the value part of
 each Association or key-value pair as the argument of the block.  Returns the 
 receiver."

^ self subclassResponsibility: #do.
%

category: 'Enumerating'
method:
keysAndValuesDo: aBlock

"Iteratively evaluates the two argument block, aBlock, using each key and value
 of the receiver as the argument to the block.  Returns the receiver."

^ self subclassResponsibility: #keysAndValuesDo:.
%

category: 'Enumerating'
method:
accompaniedBy: anObj keysAndValuesDo: aBlock

"Iteratively evaluates the threee argument block, aBlock, 
 using anObj, each key and each value
 of the receiver as the arguments to the block.  Returns the receiver."

^ self subclassResponsibility: #accompaniedBy:keysAndValuesDo: .
%


category: 'Enumerating'
method:
keysDo: aBlock

"Iteratively evaluates the one argument block, aBlock, using each key of
 the receiver as the argument to the block. Returns the receiver."

^ self subclassResponsibility: #keysDo:.
%

category: 'Enumerating'
method:
valuesDo: aBlock

"Iteratively evaluates the one argument block, aBlock, using each value of
 the receiver as the argument to the block. Returns the receiver."

"Note: This method has the same behavior as #do:."

^ self subclassResponsibility: #valuesDo:.
%

category: 'Searching'
method:
occurrencesOf: aValue

"Returns the number of Associations or key-value pairs in the receiver with 
 value equal to 'aValue'."

| numOccurrences |

numOccurrences := 0.
self valuesDo: [ :aVal |  
  aValue = aVal ifTrue: [ numOccurrences := numOccurrences + 1 ]
  ].

^ numOccurrences
%

category: 'Searching'
method:
occurrencesOfIdentical: aValue

"Returns the number of Associations or key-value pairs in the receiver with 
 a value that is identical to aValue."

| numOccurrences |

numOccurrences := 0.
self valuesDo: [ :aVal |  
  aValue == aVal ifTrue: [ numOccurrences := numOccurrences + 1 ]
  ].

^ numOccurrences
%

category: 'Deprecated'
method:
occurrencesOfValue: aValue

self deprecated: 'AbstractDictionary>>occurrencesOfValue: deprecated long before v3.0. Use occurrencesOf: instead.'.
^ self occurrencesOf: aValue.
%

category: 'Enumerating'
method:
reject: aBlock

"Evaluates aBlock with each of the receiver's values as the argument. Stores
 the key-value pairs for which aBlock is false into a dictionary of the same 
 class as the receiver, and returns the new dictionary. The argument aBlock 
 must be a one-argument block."

| result |

result := self species new: (self size).
self keysAndValuesDo: [ :aKey :aValue |
  (aBlock value: aValue) ifFalse: [result at: aKey put: aValue]
  ].

^ result.
%

category: 'Deprecated'
method:
rejectValues: aBlock

self deprecated: 'AbstractDictionary>>rejectValues: deprecated long before v3.0. Use reject: instead.'.
^ self reject: aBlock.
%

category: 'Enumerating'
method:
rejectValuesAsArray: aBlock

"Evaluates aBlock with each of the receiver's values as the argument and
 returns an Array containing the values for which aBlock evaluates false."

| result |
result := { } .
self valuesDo: [ :aValue |
  (aBlock value: aValue) ifFalse: [result add: aValue].
  ].

^ result. 
%

category: 'Removing'
method:
remove: anObject

"Disallowed.  Use #removeKey: instead."

self shouldNotImplement: #remove:
%

category: 'Removing'
method:
removeIdentical: anObject

"Disallowed."

self shouldNotImplement: #removeIdentical:
%

category: 'Removing'
method:
remove: anObject ifAbsent: anExceptionBlock

"Disallowed.  Use #removeKey:ifAbsent: instead."

self shouldNotImplement: #remove:ifAbsent.
%

category: 'Removing'
method:
removeIdentical: anObject ifAbsent: anExceptionBlock

"Disallowed."

self shouldNotImplement: #removeIdentical:ifAbsent.
%

category: 'Removing'
method:
removeAll: aCollection

"Disallowed.  Use #removeAllKeys: instead."

self shouldNotImplement: #removeAll:
%

category: 'Removing'
method:
removeAllIdentical: aCollection

"Disallowed."

self shouldNotImplement: #removeAllIdentical:
%

category: 'Removing'
method:
removeAllKeys: keys

"Removes all the keys equal to the given keys from the receiver. An error is not
 generated if keys equal to any of the specified keys are not present. Returns
 the collection keys."

^ keys accompaniedBy: self do: [ :me :aKey | me removeKey: aKey ifAbsent: [nil] ].
%

category: 'Removing'
method:
removeAllKeys: keys ifAbsent: aBlock

"Removes all the keys equal to the given keys from the receiver and returns the
 collection keys. For any key which is not a valid key of the receiver, aBlock 
 is evaluated with the key as the argument."

^ keys accompaniedBy: self do: [:me :aKey | me removeKey: aKey ifAbsent: [aBlock value: aKey] ].
%

category: 'Removing'
method:
removeKey: aKey

"Removes the Association or key-value pair with key equal to aKey from the 
 receiver and returns the value portion of that Association or key-value pair
 respectively. If no Association is present with key equal to aKey, reports an 
 error."

^ self removeKey: aKey ifAbsent: [self _errorKeyNotFound: aKey ].
%

category: 'Removing'
method:
removeKey: aKey ifAbsent: aBlock

"Removes the Association or key-value pair with key equal to aKey from the 
 receiver and returns the value associated with the Association or the key-value
 pair.  If no Association or key-value pair is present with key equal to
 aKey, evaluates the zero-argument block aBlock and returns the result of that 
 evaluation."

^ self subclassResponsibility: #removeKey:ifAbsent.
%

! added for 36675
category: 'Removing'
method:
removeKey: aKey otherwise: notFoundValue

"Removes the Association or key-value pair with key equal to aKey from the
 receiver and returns the value associated with the Association or the key-value
 pair.  If no Association or key-value pair is present with key equal to
 aKey, returns notFoundValue .

 KeyValueDictionary has an optimized implementation.
"

^ self removeKey: aKey ifAbsent:[ notFoundValue ]
%

category: 'Deprecated'
method:
removeKeys: keys

self deprecated: 'AbstractDictionary>>removeKeys: deprecated long before v3.0. Use removeAllKeys: instead.'.
^ self removeAllKeys: keys
%

category: 'Private'
method:
_reportKeyNotFound: aKey with: aBlock

"Returns the value of aBlock if aBlock is not nil.  Otherwise, raises an error."

aBlock == nil ifTrue:[^ self _errorKeyNotFound: aKey ] .
^aBlock value
%

category: 'Enumerating'
method:
select: aBlock

"Evaluates aBlock with each of the receiver's values as the argument.  Stores
 the values for which aBlock is true into the dictionary of the same class as
 the receiver, and returns the new dictionary.  The argument aBlock must be a
 one-argument block."

|result|

result := self species new.
self keysAndValuesDo: [ :aKey :aValue |
  (aBlock value: aValue) ifTrue: [result at: aKey put: aValue]
  ].

^ result.
%

category: 'Deprecated'
method:
selectValues: aBlock

self deprecated: 'AbstractDictionary>>selectValues: deprecated long before v3.0. Use select: instead.'.
^ self select: aBlock.
%

category: 'Deprecated'
method:
selectValuesAsArray: aBlock


"Evaluates aBlock with each of the receiver's values as the argument and
 returns an Array containing the values for which aBlock evaluates true."

| result index |
self deprecated: 'AbstractDictionary>>selectValuesAsArray: deprecated long before v3.0.'.
result := Array new: (self size).
index := 1.
self valuesDo: [ :aValue |
  (aBlock value: aValue) 
    ifTrue: [result at: index put: aValue].
  index := index + 1.
  ].

^ result. 
%

category: 'Formatting'
set compile_env: 0
method: AbstractDictionary
printNonRecursiveRepresentationOn: aStream recursionSet: anIdentitySet
	"Put a displayable representation of the receiver on the given stream
	 while avoiding recursion from object reference loops."

	| count sz myCls |

	myCls := self class .
	aStream nextPutAll: myCls name describeClassName .
	(myCls whichClassIncludesSelector: #associationsDo:) == AbstractDictionary
		ifTrue:[ ^ self "can't safely execute associationsDo: " ].

	aStream nextPutAll: '( ' .
	count := 1 .
	sz := self size .
	self associationsDo: [:anAssoc | 
		aStream position > 700 ifTrue:[
			"prevent infinite recursion when printing cyclic structures, and 
			limit the size of result when printing large collections."
			aStream nextPutAll: '...)' .
			^ self 
			] .
		anAssoc printOn: aStream recursionSet: anIdentitySet.
		count < sz ifTrue:[ aStream nextPutAll: ', ' ].
		count := count + 1.
		].
	aStream nextPut: $).
%

category: 'Formatting'
set compile_env: 0
method: AbstractDictionary
printOn: aStream
	"Put a displayable representation of the receiver on the given stream."

	self printNonRecursiveOn: aStream
%

category: 'Accessing'
method:
values

"Returns an OrderedCollection containing the receiver's values."

|result|
result:= OrderedCollection new. 
self valuesDo: [ :value | result add: value ].
^ result.
%

! deleted #sortAscending:
! deleted #sortDescending:
! deleted #sortWith:

! delete duplicates

category: 'Enumerating'
method:
rejectAssociations: aBlock

"Evaluates aBlock with each of the receiver's elements as the argument.  Stores
 the values for which aBlock is false into a collection of the same class as
 the receiver, and returns the new collection.  The argument aBlock must be a
 one-argument block.  Uses associative access when the argument is a
 SelectionBlock."

| newCollection |

newCollection := self species new.
self associationsDo: [ :element | 
  (aBlock value: element) ifFalse: [ newCollection add: element ].
  ].

^ newCollection.
%

category: 'Enumerating'
method:
selectAssociations: aBlock

"Evaluates aBlock with each of the receiver's elements as the argument.  Stores
 the values for which aBlock is true into a collection of the same class as the
 receiver, and returns the new collection.  The argument aBlock must be a
 one-argument block.  Uses associative access when the argument is a
 SelectionBlock."

| newCollection |

newCollection := self species new.
self associationsDo: [ :element | 
  (aBlock value: element) ifTrue: [ newCollection add: element ].
  ].

^ newCollection.
%

! gemstone64, I added selectAssociationsAsArray: and associationsAsArray
!  because I could not find a convienent way to get these results.  these methods
!  make it easy to get a listing of the contents of a Dictionary by doing
!  a level 2 display of the result in topaz .   		Allen

category: 'Enumerating'
method:
selectAssociationsAsArray: aBlock

"Evaluates aBlock with each of the receiver's associations as the argument.  Stores
 the values for which aBlock is true into a new Array and returns the Array.
 The argument aBlock must be a one-argument block.  "

| newCollection |

newCollection := { }  .
self associationsDo: [ :element | 
  (aBlock value: element) ifTrue: [ newCollection add: element ].
  ].

^ newCollection.
%

category: 'Enumerating'
method:
associationsAsArray

"Returns an Array containing all of the receiver's Associations"

^ self selectAssociationsAsArray: [ :assoc| true ]
%

category: 'Storing and Loading'
method:
loadFrom: passiveObj size: varyingSize

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into the receiver."

"This method is similar to loadFrom:, but is used for objects whose size
 is not known when they are first instantiated (such as an IdentitySet)."

^self basicLoadFrom: passiveObj size: varyingSize
%

category: 'Storing and Loading'
classmethod:
loadFrom: passiveObj

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| size inst |
size := passiveObj readSize.
inst := self new.
inst loadFrom: passiveObj size: size.
^inst
%

category: 'Private'
method:
_deferredGciUpdateWith: valueArray

^ self subclassResponsibility: #_deferredGciUpdateWith:
%

category: 'Private'
method:
_deepCopyWith: copiedObjDict

| copy myClass |

copy := copiedObjDict at: self otherwise: nil.
copy ~~ nil ifTrue: [ ^ copy ].

myClass := self class .
copy := myClass new: (self size).
copiedObjDict at: self put: copy.

self _deepCopyNamedIvsWith: copiedObjDict to: copy .

self keysAndValuesDo: [ :aKey :aValue |
  copy at: (aKey _deepCopyWith: copiedObjDict)
       put: (aValue _deepCopyWith: copiedObjDict).
  ].
  
^ copy.
%

! fix 42281, avoid infinite recursion on self-referencing dictionaries
category: 'Comparing'
method: AbstractDictionary
= anAbstractDictionary
"Returns true if all of the following conditions are true:

 1.  The receiver and anAbstractDictionary are of the same class.
 2.  The two dictionaries are of the same size.
 3.  The corresponding keys and values of the receiver and anAbstractDictionary
     are equal."

| notFound |
(self == anAbstractDictionary) ifTrue:[ ^ true ].

(self class == anAbstractDictionary class) ifFalse:[ ^ false ].

(self size == anAbstractDictionary size) ifFalse:[ ^ false ].
notFound := Object new .
self keysAndValuesDo: [ :aKey :aValue | | otherVal|
  otherVal := anAbstractDictionary at: aKey otherwise: notFound . 
  (aValue == self or:[ aValue == anAbstractDictionary]) ifTrue:[
    (otherVal == self or:[ otherVal == anAbstractDictionary]) ifFalse:[ ^ false].
  ] ifFalse:[
    aValue = otherVal ifFalse:[ ^ false ]
  ].
].
^ true.
%

category: 'Comparing'
method:
hash

"Returns a numeric hash key for the receiver."

| hashValue |

hashValue := 97633 bitXor: (self size).
"For large dictionaries, the hash value is just a function of its size"
(self size > 64) ifTrue: [ ^ hashValue abs ].
self keysDo: [ :aKey |
   "Skip if the key is a dictionary."
   (aKey isKindOf: AbstractDictionary)
     ifFalse: [
       hashValue := hashValue bitXor: aKey hash 
       ]
     ].
^ hashValue abs
%

category: 'Printing'
method:
asReportString

"Returns a String that lists the key-value pairs, one on each line."

| result lf tab |
result := String new.
tab := Character tab .
lf := Character lf .
self keysAndValuesDo: [ :key :val |
  result add: key printString; add: tab ; add: val printString; add: lf 
].
^ result
%

! deleted _canonicalizeSymbolAt: offset oldSymbol: oldSym newSymbol: newSym

! deleted _resetParentRef, moved it to Object

category: 'Instance Migration'
method:
migrateIndexable: anotherObject myClass: cls otherClass: othercls

"For dictionaries, we need to reconstruct the indexable component rather
 than just copy it over (in case the new class uses a different structure)."

anotherObject keysAndValuesDo: [ :key :value |
    self at: key put: value ].

%
category: 'Updating'
method: AbstractDictionary
objectSecurityPolicy: anObjectSecurityPolicy

  "Assigns the receiver and all its components to the given security policy.
   Returns the receiver."

  super objectSecurityPolicy: anObjectSecurityPolicy.
  self _nodesObjectSecurityPolicy: anObjectSecurityPolicy.
%

category: 'Private'
method: AbstractDictionary
_nodesObjectSecurityPolicy: anObjectSecurityPolicy
  "Assigns receiver's components to the given security policy. "

  self subclassResponsibility: #_nodesObjectSecurityPolicy:
%

category: 'Accessing'
method:
_basicAt: anIndex

"Signals an error if the receiver is not indexable, 
 or if anIndex is not a SmallInteger or is out of bounds.

 This method is for internal use in implementation of the kernel classes.

 The methods _at:, _basicAt:, and _basicSize operate on the logically
 indexable contents of the receiver.

 This method must not be overridden."

<primitive: 699>
(anIndex _isInteger)
  ifTrue: [^ self _errorIndexOutOfRange: anIndex]
  ifFalse: [^ self _errorNonIntegerIndex: anIndex].
self _primitiveFailed: #at: args: { anIndex }.
self _uncontinuableError
%

category: 'Accessing'
method:
squeakBasicAt: anIndex
  ^ self _basicAt: anIndex
%

category: 'Accessing'
method:
_at: anIndex

"Signals an error if the receiver is not indexable, 
 or if anIndex is not a SmallInteger or is out of bounds.

 This method is for internal use in implementation of the kernel classes.

 The methods _at:, _basicAt:, and _basicSize operate on the logically
 indexable contents of the receiver.
 
 This method must not be overridden."
 
<primitive: 699>
(anIndex _isInteger)
  ifTrue: [^ self _errorIndexOutOfRange: anIndex]
  ifFalse: [^ self _errorNonIntegerIndex: anIndex].
self _primitiveFailed: #at: args: { anIndex }.
self _uncontinuableError
%

category: 'Updating'
method:
squeakBasicAt: anIndex put: aValue

  ^ self _basicAt: anIndex put: aValue
%

category: 'Updating'
method:
_basicAt: anIndex put: aValue

"Stores the argument aValue in the indexed variable of the
 receiver indicated by anIndex.  The argument anIndex must not be
 larger than 1 + the size of the receiver, and must not be less than 1.

 Returns aValue.

 Generates an error if anIndex is not a SmallInteger or is out of
 bounds."

<primitive: 745>

(anIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
((anIndex < 1) | (anIndex > (self size + 1))) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].

self _primitiveFailed: #at:put: args: { anIndex . aValue }
%
category: 'Updating'
method:
_at: anIndex put: aValue

"Stores the argument aValue in the indexed variable of the
 receiver indicated by anIndex.  The argument anIndex must not be
 larger than 1 + the size of the receiver, and must not be less than 1.

 Returns aValue.

 Generates an error if anIndex is not a SmallInteger or is out of
 bounds."

<primitive: 745>

(anIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
((anIndex < 1) | (anIndex > (self size + 1))) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].

self _primitiveFailed: #at:put: args: { anIndex . aValue }.
self _uncontinuableError
%

category: 'Private'
method:
_atZ: zOffset
  "return varying instVar specified by zero-based offset zOffset"
<primitive: 858>
| sz |
zOffset _validateClass: SmallInteger .
sz := self _basicSize .
(zOffset < 0 or:[ zOffset >= sz ]) ifTrue:[
  OffsetError new offset: zOffset maximum: sz - 1 ; signal
].
self _primitiveFailed: #_atZ: args: { zOffset }
%

method:
_atZ: zOffset put: aValue
  "store aValue into varying instVar specified by zero-based offset zOffset.
   will auto-grow up to 6 instVars past current end"
<primitive: 860>
| max |
zOffset _validateClass: SmallInteger .
max := self _basicSize + 6 .
(zOffset < 0 or:[ zOffset >= max ]) ifTrue:[
  OffsetError new offset: zOffset maximum: max ; signal
].
self _primitiveFailed: #_atZ:put: args: { zOffset . aValue }
%

method:
_atZ: zOffset putKey: aKey value: aValue
  "store aKey into varying instVar specified by zero-based offset zOffset.
   store aValue into varying instVar specified by zero-based offset zOffset + 1.
   will auto-grow up to 7 instVars past current end"
<primitive: 892>
| max |
zOffset _validateClass: SmallInteger .
max := self _basicSize + 6 .
(zOffset < 0 or:[ zOffset >= max ]) ifTrue:[
  OffsetError new offset: zOffset maximum: max ; signal
].
self _primitiveFailed: #_atZ:putKey:value: 
     args: { zOffset . aKey . aValue }
%
category: 'Json'
method: AbstractDictionary
printJsonOn: aStream

	| delimiter |
	delimiter := ''.
	aStream nextPut: ${.
	self keysAndValuesDo: [:key :value |
		aStream nextPutAll: delimiter.
		key asString printJsonOn: aStream.
		aStream nextPut: $:.
		value printJsonOn: aStream.
		delimiter := ','.
	].
	aStream nextPut: $}.
%
category: 'Repository Conversion'
method: AbstractDictionary
fixRefsAfterConversion

"Default Dictionary method for fixing references to
 ObsLargePositiveInteger and ObsLargeNegativeInteger
 instances that can now be represented as a
 SmallInteger and Floats and SmallFloats which can now be represented as
 a SmallDouble."

| array convBm |

convBm := (GsBitmap newForHiddenSet: #Conversion).
(convBm includes: self)
        ifTrue:[^false]. "already fixed this one"

"Fix inst var refs first"
self fixInstVarRefsAfterConversion.

"now handle keys and values"
array := { } .
self keysAndValuesDo:[:k :v|
        (k needsFixingAfterConversion or:[v needsFixingAfterConversion])
                ifTrue:[array add: k; add: v.].
        ].
1 to: array size by: 2 do:[:n| |k v newKey newValue|
        k := array at: n.
        v := array at: (n + 1).
        self removeKey: k.
        k needsFixingAfterConversion
                ifTrue:[newKey := k + 0]
                ifFalse:[newKey := k].
        v needsFixingAfterConversion
                ifTrue:[newValue := v + 0]
                ifFalse:[newValue := v].
        self at: newKey put: newValue.
].
convBm add: self.
^true
%

