!=========================================================================
! Copyright (C) by GemTalk Systems 1991-2020.  All Rights Reserved
!
! $Id$
!
!=========================================================================

removeallmethods CharacterCollection
removeallclassmethods CharacterCollection
set class CharacterCollection

! ------------------- Class methods for CharacterCollection

category: 'For Documentation Installation only'
classmethod:
installDocumentation

self comment:
'CharacterCollection is an abstract superclass for behavior that is common
 to all indexed collections of Characters.

 Subclasses must reimplement the following selectors:

 size
 size:
 at:
 at:put:
 removeFrom:to:
 insertAll:at:

 However these selectors do not generate the subclass-responsibility error
 (error 2008) because to do so would break the Object | printString method.

--- Category:  Comparing
Some of these methods determine whether one String collates before another.
 In collation, the values of the receiver and aCharCollection are compared
 Character-by-Character, from left to right, in case-sensitive fashion.  If two
 CharacterCollections are of different length, and all Characters in the
 shorter collection are equal to their counterparts in the longer one, the
 shorter collection collates before the longer.

 Unlike the comparison methods for the superclass SequenceableCollection, these
 methods merely require that both the receiver and argument be kinds of
 CharacterCollection (rather than requiring both to be of the same class).
'
%

category: 'Deprecated'
classmethod:
fromServerTextFile: aFileSpec

self deprecated: 'CharacterCollection class >> fromServerTextFile: deprecated long before v3.0. Use an instance of GsFile 
 to access the file system.'.
 "For multi-byte characters, assumes the byte order of the file
 matches the current session's cpu's in-memory byte order.  "

^ self new _fromServerTextFile: aFileSpec
%

category 'Instance Creation'
classmethod:
_newString

"Returns a new instance of the receiver, or instance of String as appropriate. 
 Reimplemented in subclasses as needed to handle canonical symbols."

^ self new
%

category 'Private'
classmethod:
_newString: aSize

"Returns a new instance of the receiver, or instance of String as appropriate. 
 Reimplemented in subclasses as needed to handle canonical symbols."

^ self new: aSize
%

category: 'Instance Creation'
classmethod:
fromStream: aStream width: anInteger

"Returns a new instance of the receiver's class that contains the next
 anInteger Characters of aStream."

| result |

self _checkReadStream: aStream forClass: CharacterCollection.
result:= self new: anInteger.
1 to: anInteger do: [ :i | result at: i put: aStream next ].
^ result
%

category: 'Instance Creation'
classmethod:
withAll: aSequenceableCollection

"Returns a new instance of the receiver that contains the elements in the
 argument aSequenceableCollection."

| result |
result:= self new.
aSequenceableCollection accompaniedBy: result do: [:res :each | res add: each].
^result
%
classmethod:
new
"(Subclass responsibility.)"

^ self subclassResponsibility: #new
%
classmethod:
new: anInteger
"(Subclass responsibility.)"

^ self subclassResponsibility: #new:
%

! deleted CharacterCollection>>withBytes:

category 'Formatting'
classmethod:
charSize

"(Subclass responsibility.)
 Returns number of bytes that make up a character for instances of this class."
 
^ self subclassResponsibility: #charSize
%




! ------------------- Instance methods for CharacterCollection
category: 'Concatenating'
method:
, aCharOrCharCollection

"Returns a new instance of the receiver's class that contains the elements of
 the receiver followed by the elements of aCharOrCharCollection.

 Warning: Creating a new instance and copying the receiver take time.  If you
 can safely modify the receiver, it can be much faster to use the addAll:
 method.  See the documentation of the Concatenating category of class
 SequenceableCollection for more details."

| newCollection |

newCollection := self copy.
newCollection addAll: aCharOrCharCollection.
^ newCollection
%

category: 'Comparing'
method:
< aCharCollection

"Returns true if the receiver collates before the argument.  Returns false
 otherwise.
 (Subclass responsibility.)"

^ self subclassResponsibility: #<
%

category: 'Comparing'
method:
<= aCharCollection

"Returns true if the receiver collates before the argument or if all of the
 corresponding Characters in the receiver and argument are equal.
 Returns false otherwise.
 (Subclass responsibility.)"

^ self subclassResponsibility: #<=
%

category: 'Comparing'
method:
= aCharCollection

"Returns true if all of the corresponding Characters in the receiver and
 argument are equal.  Returns false otherwise.
 (Subclass responsibility.)"

^ self subclassResponsibility: #=
%

category: 'Comparing'
method:
> aCharCollection

"Returns true if the receiver collates after the argument.  Returns false
 otherwise.
 (Subclass responsibility.)"

^ self subclassResponsibility: #>
%

category: 'Comparing'
method:
>= aCharCollection

"Returns true if the receiver collates after the argument or if all of the
 corresponding Characters in the receiver and argument are equal.  Returns
 false otherwise.
 (Subclass responsibility.)"

^ self subclassResponsibility: #>= 
%

set compile_env: 0
category: 'Comparing'
method: CharacterCollection
between: min and: max
	"Answer whether the receiver is less than or equal to the argument max and
	 greater than or equal to the argument min."

	^self >= min and: [self <= max].
%

set compile_env: 0
category: 'Comparing'
method: CharacterCollection
max: another

"If the receiver is greater than the argument, return the receiver.  
 Otherwise return the argument."

	^self > another
		ifTrue: [self]
		ifFalse: [another].
%

set compile_env: 0
category: 'Comparing'
method: CharacterCollection
min: another

"If the receiver is less than the argument, return the receiver.  
 Otherwise return the argument."

	^self < another
		ifTrue: [self]
		ifFalse: [another].
%

category: 'Accessing'
method:
_at: anIndex

"Private.  Reimplemented to return a kind of Character."

^ self at: anIndex
%
category: 'Private'
method:
_charCollCompare: aCharCollection

"Returns -1 if self < aCharCollection, returns 0 if self = aCharCollection,
 and returns 1 if self > aCharCollection."

| selfSize argSize selfElement argElement |

aCharCollection _validateClass: CharacterCollection.

selfSize:= self size.
argSize:= aCharCollection size.

1 to: (selfSize min: argSize) do: [:i |

  selfElement := self at: i.
  argElement  := aCharCollection at: i.

  (selfElement < argElement) ifTrue: [ ^ -1 ].
  (selfElement > argElement) ifTrue: [ ^ 1 ]
  ].

"All elements of self and argument are equal."

(selfSize < argSize) ifTrue: [ ^ -1 ].
(selfSize > argSize) ifTrue: [ ^ 1 ].

^ 0 "equal"
%

category: 'Instance Creation'
method:
_fromServerTextFile: aFileSpec

"Import the contents of aFileSpec into the receiver.
 For multi-byte characters, assumes the byte order of the file
 matches the current session's cpu's in-memory byte order.
"
<primitive: 302>

aFileSpec _validateByteClass: CharacterCollection .
aFileSpec _error: #hostErrFileImport args: #()
%

! deleted _idxCompareForCompareEqualTo: v2.0
! deleted _idxCompareLessThanOrEqual: v2.0
! deleted _idxCompareGreaterThanOrEqual: v2.0
! deleted _idxCompareEqualTo: v2.0
! deleted _idxCompareForSortEqualTo: v2.0


category: 'Adding'
method:
add: aCharOrCharColl

"Appends all of the elements of aCharOrCharColl to the receiver and returns
 aCharOrCharColl.  Returns aCharOrCharColl"

| index |

(aCharOrCharColl isKindOf: Collection) ifTrue:[
    index := self size.
    aCharOrCharColl accompaniedBy: self do: [:me :aChar |
      index := index + 1.
      me at: index put: aChar.
    ].
    ^ aCharOrCharColl.
].

^ self at: (self size + 1) put: aCharOrCharColl.
%

! fixed 40865
category: 'Adding'
method:
addAll: aCharOrCharCollection

"Equivalent to add: aCharOrCharCollection."

^ self add: aCharOrCharCollection
%

category: 'Adding'
method:
addLast: aCharOrCharCollection

"Equivalent to add: aCharOrCharCollection."

^ self add: aCharOrCharCollection
%

category: 'Converting'
method:
asArrayOfKeywords

"Returns an Array of keyword substrings held by the receiver.  The receiver
 is assumed to be a colon-separated list of substrings.  These substrings
 are extracted and collected in an Array.  If the receiver contains no
 colons, the Array will hold a copy of the receiver."

| c nextName result |

result := { }.
nextName := self speciesForConversion new.
1 to: self size do: [ :i |
  c := self at: i.
  nextName add: c.
  c == $: ifTrue: [
    result add: nextName.
    nextName := self speciesForConversion new.
  ].
].
nextName size ~~ 0 ifTrue: [
  result add: nextName
]
ifFalse: [
  result size == 0 ifTrue: [result add: nextName]
].
^result
%

category: 'Converting'
method:
asArrayOfSubstrings

"Returns an Array of substrings held by the receiver. The receiver
 is assumed to be a separator-separated list of substrings.  These substrings
 are extracted and collected in an Array.  If the receiver contains no
 separators, the Array will hold a copy of the receiver.  Separators not meant
 to separate substrings may be escaped with a $\ Character."

| nextName result esc sz |

result := { } .
(sz := self size) == 0 ifTrue: [
  ^result
].
nextName := self speciesForConversion new.
esc := false.
1 to: sz do: [ :i | | c |
  c := self at: i.
  esc ifTrue: [
    nextName add: c.
    esc := false.
  ] ifFalse: [
    c == $\ ifTrue: [ esc := true ]
    ifFalse: [
	c isSeparator ifTrue: [
          nextName size ~~ 0 ifTrue: [result add: nextName].
          nextName := self speciesForConversion new.
        ] ifFalse: [ 
          nextName add: c 
        ].
    ].
  ].
].

esc ifTrue:[ nextName add: $\ ].

(nextName size ~~ 0 or: [result size == 0]) ifTrue:[ result add: nextName ].

^result
%

category: 'Converting'
method:
asDecimalFloat

"Returns a DecimalFloat whose value is represented by the receiver."

^ DecimalFloat fromString: self
%

category: 'Converting'
method:
asDoubleByteString

"Returns a DoubleByteString representation of the receiver."

^ DoubleByteString withAll: self asString .       "fix 39372"
%

category: 'Converting'
method:
asQuadByteString

"Returns a QuadByteString representation of the receiver."

^ QuadByteString withAll: self.
%
category: 'Converting'
method:
asMultiByteString: example

"Returns a Double/QuadByteString representation of the receiver,
 depending on the class of the example string."

^ example class withAll: self
%

category: 'Converting'
method:
asFloat

"Returns a SmallDouble or Float whose value is represented by the receiver."

^ Float fromString: self
%

category: 'Converting'
method:
asInteger

"Returns an Integer whose value is represented by the receiver."

^ Integer fromString: self
%

category 'Formatting'
method:
charSize

"Returns number of bytes that make up a character for this string class.
 (Subclass responsibility.)"

^ self subclassResponsibility: #charSize
%

category: 'Formatting'
method:
trimWhiteSpace

"Returns a copy of the receiver with leading and trailing white space removed."

| first limit selfSize |

((selfSize := self size) == 0) ifTrue: [
   ^ self class _newString 
].

limit := selfSize + 1. 

first := 1 .
(self at: 1) isSeparator ifTrue: [ | j |
  first := nil .
  j := 2.
  [ j == limit ] whileFalse: [ 
      (self at: j) isSeparator ifTrue: [
         j := j + 1.
      ] ifFalse:[
         first := j.
         j := limit . 
       ].
  ].
  first ifNil: [ ^ self class _newString ].
].

(self at: selfSize) isSeparator ifTrue: [ | k |
  k := selfSize - 1.
  [ k == 0 ] whileFalse: [
     (self at: k) isSeparator ifFalse:[
       ^ self copyFrom: first to: k 
     ].
     k := k - 1.
  ].
  ^ self class _newString
].

first == 1 ifTrue:[  ^ self copy ].
^ self copyFrom: first to: selfSize .
%

category: 'Converting'
method:
asNumber

"Returns the receiver converted to a kind of number.  If the receiver contains
 all digits (with optional radix notation), returns a kind of Integer.  If the
 receiver has a slash, returns a Fraction.  Otherwise conversion to a Float is
 attempted.  An error may result if the receiver does not contain the proper
 format for a kind of Number."

| v zero nine str idx ch |
str := self trimWhiteSpace.
str size == 0 ifTrue: [ ^0 ].
zero := $0 codePoint.
nine := $9 codePoint.
idx := 1.
(ch := str at: 1) == $- ifTrue: [ idx := 2 ] 
              ifFalse: [ ch == $+ ifTrue: [ idx := 2 ] ].

idx to: str size do: [:i | 
  v := (ch := str at: i) codePoint.
  (zero <= v and: [v <= nine]) ifFalse: [
    (ch == $#  or: [ ch == $r ]) ifTrue: [
      ^ Integer fromString: str .
    ] ifFalse:[
      ch == $/ ifTrue: [
        ^Fraction fromStream: (ReadStream on: str)
      ].
      ^ Float fromString: str
    ]
  ]
].
^Integer fromString: str
%

category: 'Converting'
method:
asUppercase

"Returns a new instance of the receiver's class, with all lower-case
 characters in the receiver changed to upper-case.

 (Subclass responsibility.)"

^ self subclassResponsibility: #asUppercase
%

category: 'Converting'
method:
asLowercase

"Returns a new instance of the receiver's class, with all upper-case
 characters in the receiver changed to lower-case.

 (Subclass responsibility.)"

^ self subclassResponsibility: #asLowercase
%

category: 'Deprecated'
method:
asSmallFloat

"SmallFloat is deprecated. Return a SmallDouble or Float whose value 
 is represented by the receiver."

self deprecated: 'CharacterCollection>>asSmallFloat deprecated v3.0. Use asFloat or Float class>>#fromString: '.
^ (Float fromString: self) asSmallFloat
%

category: 'Converting'
method:
asString

"Returns a String representation of the receiver."

^ String withAll: self
%

category: 'Converting'
method:
asHexString

"Returns a String containing a hexadecimal printed representation of the
 contents of the receiver.  For example, the message 'abc' asHexString
 returns the String '616263'.

 The receiver must be a byte format object."

<primitive: 467>
self _validateByteClass: CharacterCollection .
self _primitiveFailed: #asHexString .
%

category: 'Converting'
method:
asSymbolKind

"Returns a canonical symbol containing the same Characters as the receiver."

CharacterCollection subclassResponsibility: #asSymbolKind
%

category: 'Comparing'
method:
at: anIndex equals: aCharCollection

"Returns true if aCharCollection is contained in the receiver starting at
 anIndex.  Returns false otherwise.

 Note that this method returns true only if aCharCollection begins exactly at
 the position designated by anIndex.  To locate a pattern beginning on or after
 anIndex, see the method findPattern:startingAt: in category 'Searching'.

 (Subclass responsibility.)"

^ self subclassResponsibility: #at:equals: 
%

! deleted CharacterCollection>>byteAt: ,  use codePointAt: instead

category: 'Accessing'
method:
codePointAt: anIndex

CharacterCollection subclassResponsibility: #'codePointAt:'
%

category: 'Updating'
method:
codePointAt: anIndex put: aValue

CharacterCollection subclassResponsibility: #'codePointAt:put:'
%

category: 'Formatting'
method:
describeClassName

"Returns a copy of the receiver with the Character $a prepended to the
 receiver's contents.  This method is used for formatting class names in object
 descriptions, where the receiver is a string containing the name of a class.
 For example, the String 'UserClass', when sent the message describeClassName,
 returns 'aUserClass'."

| result |
result := self speciesForPrint new .
self size ~~ 0 ifTrue:[   "inline simplified isVowel using Unicode asUppercase"
  ( #( $A $E $I $O $U $Y ) includesIdentical: (self at: 1) asUppercase ) ifTrue:[
    result add: 'an' ; addAll: self .
    ^ result 
  ]
].
result add: $a ; add: self.
^ result
%

category: 'Deprecated'
method:
insert: aCharOrCharCollection at: anIndex

self deprecated: 'CharacterCollection>>insert:at: deprecated long before v3.0. Use insertAll:at: instead.'.
^ self insertAll: aCharOrCharCollection at: anIndex.
%

category: 'Adding'
method:
insertAll: aCharOrCharCollection at: anIndex

"Inserts aCharOrCharCollection into the receiver at the specified index and
 returns aCharOrCharCollection."

(aCharOrCharCollection isKindOf: CharacterCollection)
   ifTrue: [ ^ super insertAll: aCharOrCharCollection at: anIndex ].

(aCharOrCharCollection isKindOf: AbstractCharacter)
  ifTrue: [
    self replaceFrom: anIndex + 1 to: 1 + self size with: self startingAt: anIndex .
    self at: anIndex put: aCharOrCharCollection.
    ^aCharOrCharCollection.
    ].

^ aCharOrCharCollection _error: #rtErrInvalidArgClass 
                        args: { AbstractCharacter . CharacterCollection }.
%

category: 'Adding'
method:
insertObject: anObject at: anIndex

"Inserts anObject into the receiver at index anIndex and returns
 aCollection."

"reimplemented to use at:put: instead of _basicAt:put:"
| selfSize |

anIndex <= 0 ifTrue:[ ^ self _errorIndexOutOfRange: anIndex ].
selfSize := self size.
anIndex > selfSize ifTrue:[
  anIndex > (selfSize + 1) ifTrue:[ ^ self _errorIndexOutOfRange: anIndex].
  ^ self at: anIndex put: anObject.
  ].

"Not adding to the end of the receiver. Create a gap for anObject to
 be copied into."
self replaceFrom: anIndex + 1 to: 1 + selfSize with: self startingAt: anIndex .

^ self at: anIndex put: anObject.
%

category: 'Comparing'
method: CharacterCollection
sameAs: aCharCollection

"Returns true if the receiver is equivalent to aCharCollection.  The receiver
 is equivalent to aCharCollection if the receiver contains the same Characters
 as aCharCollection regardless of case or internal representation.  For
 example, if $a is in aCharCollection, it is equivalent to any representation
 of an 'a' in the receiver's character set.
 From ANSI.  Used in Seaside. "

(aCharCollection isKindOf: CharacterCollection) ifFalse: [ ^false ].

self size ~~ aCharCollection size ifTrue: [ ^false ].

^ self isEquivalent: aCharCollection 
%


category: 'Testing'
method:
equalsNoCase: aCharCollection
    "Returns true if the receiver is equivalent to aCharCollection.
    The receiver is equivalent to aCharCollection if the receiver
    contains the same Characters as aCharCollection regardless of case
    or internal representation.  For example, if $a is in
    aCharCollection, it is equivalent to any representation of an 'a'
    in the receiver's character set.
 
    (Subclass responsibility.)"

^ self subclassResponsibility: #equalsNoCase: 
%

category: 'Testing'
method:
isEquivalent: aCharCollection

"Returns true if the receiver is equivalent to aCharCollection.  The receiver
 is equivalent to aCharCollection if the receiver contains the same Characters
 as aCharCollection regardless of case or internal representation.  For
 example, if $a is in aCharCollection, it is equivalent to any representation
 of an 'a' in the receiver's character set.

 (Subclass responsibility.)"

^ self subclassResponsibility: #isEquivalent: 
%
! fixed 13411

category: 'Updating'
method:
lf

"Appends a line-feed to the receiver and returns the receiver."

self addCodePoint: 10
%

! merge fix from 6.x,   use "self class with:"
category: 'Updating'
method:
addLineDelimiters

"Returns a copy of the receiver that contains each occurrence of the backslash 
 Character replaced by the line-feed Character."

^ self copyReplaceAll: '\' with: (self class with: Character lf).
%

! deleted CharacterCollection>>byteAt:put:  

category: 'Comparing'
method:
matchPattern: aPattern

"Returns true if the receiver matches aPattern, false if it doesn't.  An exact
 match is required.  For partial matching, use the 'Searching' method
 findPattern:startingAt: instead.

 The argument aPattern is a kind of Array containing zero or more
 CharacterCollections, plus zero or more occurrences of the special Characters
 $* or $?.  If either $* or $? occurs in aPattern, it acts as a wild card.
 The Character $? matches any single Character in the receiver, and $* matches
 any sequence of zero or more Characters in the receiver.  For example,

 'weimaraner' matchPattern: #('w' $* 'r')

 returns true, because the Character $* is interpreted as a wild card.

 If either of these special Characters occurs in the receiver, it is
 interpreted literally.  For example,

 'w*r' matchPattern: #('weimaraner')

 returns false - because the Character $* occurs in the receiver, it is
 interpreted as a literal asterisk (not as a wild card)."

| match         "indicates if the current pattern matched"
  pattern       "the Array of pattern elements"
  selfSize      "the size of the receiver"
  patternSize   "the number of elements in the pattern Array"
  startIndexArr "an Array of indexes into the receiver; this Array is
                 parallel to the pattern Array (each element in this Array
                 corresponds to the starting index for each pattern element)"
  index         "index into the pattern Array and the startIndexArr Array"
  thisPattern   "the current element of the pattern to match"
  selfIndex     "an index into the receiver"
|

 "The pattern Array must be processed so that there are no *? pairs in it.
 They must all be converted to ?* pairs for the algorithm to work correctly."
 pattern:= Array withAll: aPattern.
 patternSize:= pattern size.
 index := 1.
 [ index < patternSize ]
 whileTrue:
 [ (((pattern at: index) isEquivalent: $* ) 
    and:[ (pattern at: index+1) isEquivalent: $? ]) 
   ifTrue:[ 
     pattern at: index put: $?.
     pattern at: index+1 put: $*.
     index := 1 max: index-1.
   ]
   ifFalse:
   [ index := index + 1 ].
 ].

 "initialize"
 selfSize := self size.
 startIndexArr:= Array new: (patternSize + 1).
                 "last element is set, but not used"
 index := 1.
 startIndexArr at: 1 put: 1.

 "if no patterns to match, exit early"
 (patternSize == 0)
 ifTrue:
    [^ selfSize == 0 ].

 [index <= patternSize]
 whileTrue:
 [ thisPattern := pattern at: index.
   selfIndex := startIndexArr at: index.
   match := true.

   (thisPattern isKindOf: CharacterCollection) "pattern to match is a string"
   ifTrue:
   [ (selfIndex + thisPattern size - 1) > selfSize
     ifTrue: "this pattern too big to match rest of receiver"
        [^ false ].

     index = patternSize "processing the final pattern"
     ifTrue:
     [ ((index > 1) and: [ (pattern at: index - 1) isEquivalent: $* ])
       ifTrue: "is not the first pattern _and_ previous pattern was a $*"
       [ ((selfSize == 0) and: [thisPattern size == 0])
         ifTrue: [^ true].
         ^(self findString: thisPattern
                startingAt: selfSize - thisPattern size + 1) ~~ 0
         "find the pattern far enough back in the string so that only
         the final chars match"
       ]
       ifFalse: "processing first pattern _or_ previous pattern was not $*"
       [ (match:= (selfIndex + thisPattern size - 1 = selfSize))
         ifTrue: "exactly enough chars in self to match thisPattern"
         [ (selfSize == 0)
           ifTrue: [match:= (thisPattern size == 0)]
           ifFalse: [match:= self at: selfIndex equals: thisPattern ].
         ]
       ].
     ]
     ifFalse: "not processing the final pattern"
     [ ((index > 1) and: [ (pattern at: index - 1) isEquivalent: $* ])
       ifTrue: "not first pattern _and_ previous pattern was $*"
       [ (((selfSize == 0) and: [thisPattern size == 0]) or:
         [(selfIndex:= self findString: thisPattern
                           startingAt: selfIndex) ~~ 0])
         ifTrue: "thisPattern was found"
         [ startIndexArr at: index put: selfIndex.
           startIndexArr at: index + 1 put: selfIndex + thisPattern size.
         ]
         ifFalse: "thisPattern was not found"
            [^ false ]
       ]
       ifFalse: "first pattern _or_ previous pattern was not $*"
       [ (((selfSize == 0) and: [thisPattern size == 0]) or:
         [(self at: selfIndex equals: thisPattern)])
         ifTrue:
            [startIndexArr at: index + 1 put: selfIndex+ thisPattern size]
         ifFalse:
            [match := false ].
       ].
     ]
   ]
   ifFalse: "thisPattern is not a string"
   [ (thisPattern isEquivalent: $*)
     ifTrue:
        [startIndexArr at: (index + 1) put: selfIndex]
     ifFalse:
     [ (thisPattern isEquivalent: $?)
       ifTrue:
       [ selfIndex > selfSize
         ifTrue: "no char to match; already at end of self"
            [^ false ].
         startIndexArr at: (index + 1) put: (selfIndex + 1).
         index = patternSize "processing the last pattern"
         ifTrue:
            [match := selfIndex = selfSize.].
       ]
       ifFalse: "next pattern is neither a $* or $?"
       [ ^ aPattern _error: #rtErrBadPattern].
     ].
   ].  "end ifTrue:ifFalse"

   match
   ifTrue:
      [index := index + 1 ] "advance to the next term in the pattern"
   ifFalse:
   [ "If there is a preceding $* term in the pattern, backup to the
      term following it, and advance position in the string by 1."
      [ index := index - 1.
        index < 2 ifTrue:
           [^ false ].
        (pattern at: index - 1) isEquivalent: $*
      ] untilTrue.
      startIndexArr at: index put: ((startIndexArr at: index) + 1).
   ].
].  "end whileTrue:"

^ true
%

category: 'Comparing'
method:
matchesAnyOf: aCollectionOfCharacterColls

"Returns true if the receiver returns true to the message match: with any of
 the objects in the given collection; returns false otherwise.  Examples:

   'xyz' matchesAnyOf: #('xyz' 'abc*')
     true
   'xyz' matchesAnyOf: #('ayz' 'abc')
     false
   'x#z' matchesAnyOf: #('x@z' '*')
     false

 The deprecated class JISString does not support this method."

aCollectionOfCharacterColls do: [:coll |
  (self match: coll) ifTrue: [ ^true ]
].
^false
%

category: 'Comparing'
method:
match: prefix

"Returns true if the argument prefix is a prefix of the receiver, and
 false if not.  The comparison is case-sensitive."

self size == 0 ifTrue: [ ^ prefix size == 0 ].
^ self at: 1 equals: prefix
%

category: 'Formatting'
method:
linesIndentedBy: anInt

"Returns a copy of the receiver in which all lines have been indented
 by anInt spaces."

| newStr indentStr lfIdx lf targEndInd selfCurrInd |

indentStr := self class new: anInt.
indentStr atAllPut: Character space.
lf := Character lf.
lfIdx := self indexOf: lf startingAt: 1 .
lfIdx == 0 ifTrue:[
  indentStr addAll: self .
  ^ indentStr
].

newStr := self class new.
selfCurrInd := 1.
[ true ] whileTrue:[  | destIdx |
  targEndInd := newStr size + indentStr size.
  newStr addAll: indentStr.
 "self copyFrom: selfCurrInd to: lfIdx into: newStr startingAt: targEndInd + 1 . "
  destIdx := targEndInd + 1 .
  newStr replaceFrom: destIdx to: destIdx + lfIdx - selfCurrInd 
		with: self startingAt: selfCurrInd .
  lfIdx := lfIdx + 1 .
  selfCurrInd := lfIdx .
  (lfIdx := self indexOf: lf startingAt: lfIdx ) == 0 ifTrue:[ | sz |
     selfCurrInd < (sz := self size) ifTrue:[
       newStr addAll: indentStr.
       "self copyFrom: selfCurrInd to:  sz into: newStr startingAt: newStr size + 1. "
       destIdx := newStr size + 1 .
       newStr replaceFrom: destIdx to: destIdx + sz - selfCurrInd 
		with: self startingAt: selfCurrInd .
     ].
     ^ newStr
  ].
].
%

category: 'Formatting'
set compile_env: 0
method: CharacterCollection
printOn: aStream recursionSet: anIdentitySet
	"Put a displayable representation of the receiver on the given stream
	 since CharacterCollections cannot have recursive references."

	self printOn: aStream
%

category: 'Formatting'
method:
printString

"Returns a CharacterCollection whose contents are a displayable representation of the
 receiver."

^ self printStringWithMaxSize: 100000
%

category: 'Formatting'
method:
printStringWithMaxSize: n

"Returns a CharacterCollection whose contents are a displayable representation of the
 receiver, limited to a specified number of characters <n>.

 If the number of characters in the displayable representation  exceeds <n>,
 display only the first <n>-1 characters and then display '. . .'. "

| ws |

ws := PrintStream printingOn: self speciesForPrint new maxSize: n.
self printOn: ws.
^ws contents
%

category: 'Formatting'
method:
printOn: aStream

"Puts a displayable representation of the receiver on the given stream."

"Should be reimplemented for more efficiency in subclasses."

aStream nextPutAll: self quoted .
%

category: 'Formatting'
method:
quoted

"Returns a copy of the receiver enclosed in single-quote marks, with contained
 single-quote Characters doubled.  The copy is of the same class as the
 receiver."

| sz result targetIdx lastIdx idx |
sz := self size.
result := self class _newString: sz + 2.
result at: 1 put: $'.
targetIdx := 2.
lastIdx := 1.
[ (idx := self indexOf: $' startingAt: lastIdx) == 0 ] whileFalse: [
 "self copyFrom: lastIdx to: idx into: result startingAt: targetIdx . "
  result replaceFrom: targetIdx to: targetIdx + idx - lastIdx 
	 with: self startingAt: lastIdx .

  targetIdx := targetIdx + (idx - lastIdx) + 2.
  result at: targetIdx - 1 put: $'.
  lastIdx := idx + 1 
].
lastIdx <= sz ifTrue: [
 "self copyFrom: lastIdx to: sz into: result startingAt: targetIdx "
  result replaceFrom: targetIdx to: targetIdx + sz - lastIdx 
         with: self startingAt: lastIdx .
].
result at: targetIdx + (sz - lastIdx) + 1 put: $'.
^result
%

category: 'Updating'
method:
space

"Appends a space to the receiver and returns the receiver."

self add: $ .
%

category: 'Private'
method:
speciesForConversion

"Return the class of the receiver.  Subclasses should reimplement this method."

^ self class.
%

category: 'Updating'
method:
tab

"Appends a tab to the receiver and returns the receiver."
"Recompiled in charcoll2.gs"

self add:  Character tab 
%

category: 'Deprecated'
method:
toServerTextFile: aFileSpec

self deprecated: 'CharacterCollection >> toServerTextFile: deprecated long before v3.0. Use an instance of GsFile 
 to access the file system.'.
self _toServerTextFile: aFileSpec
%

category: 'Deprecated'
method:
_toServerTextFile: aFileSpec

 "Writes the receiver to the specified file
 using the current session's cpu's in-memory byte order of any multi-byte characters.

 The argument aFileSpec must be convertable to a Utf8 , otherwise an error is signaled
 by the primitive."

<primitive: 301>
self deprecated: 'CharacterCollection>>_toServerTextFile: deprecated long before v3.0.  Use an instance of GsFile 
 to access the file system.'.
 
aFileSpec _validateByteClass: CharacterCollection .
self _error: #hostErrFileExport args: { aFileSpec }
%

category: 'Formatting'
method:
width: anInteger

"Pads the receiver with spaces to create an object of size anInteger.
 If anInteger is positive, the spaces are added to the right of the receiver.
 If anInteger is negative, the spaces are added to the left of the receiver.
 If the size of the receiver is already greater than anInteger, the receiver
 is left unchanged."

| onRight s change |

change := anInteger abs - self size.
change > 0
ifTrue:
  [ onRight := anInteger > 0.
    s := (self speciesForConversion) new: change.
    s atAllPut: $ .
    onRight
    ifTrue:
      [ self addAll: s ]
    ifFalse:
      [ self insertAll: s at: 1 ]
  ]
%

category: 'Formatting'
method:
wrapTo: col 

"Word-wrap the receiver to column col, treating tab Characters as modulo-8."

^ self _wrapTo: col indentingWith: ''
%

category: 'Formatting'
method:
_wrapTo: col indentingWith: indentStr

"Returns a new instance of the class of the receiver.

 Word-wrap the receiver to column col, treating tab Characters as modulo-8.
 Whenever a line-feed is inserted, prepend indentStr to the subsequent line."

| ch curcol linestart wordstart lf tab str sz |

lf := Character lf.
tab := Character tab.
curcol := 1.
wordstart := 0.
linestart := 1.
str := self class _newString.

1 to: (sz := self size) do: [ :i |
  ch := self at: i.
  ch == lf ifTrue: [
    str add: (self copyFrom: linestart to: i).
    linestart := i + 1.
    wordstart := 0.
    curcol := 1.
  ]
  ifFalse: [
    ch isSeparator ifTrue: [
      wordstart := 0
    ]
    ifFalse: [
      wordstart == 0 ifTrue: [
        wordstart := i
      ].
    ].

    ch == tab ifTrue: [
      curcol := (curcol + 8) \\ 8
    ]
    ifFalse: [
      curcol := curcol + 1
    ].

    curcol > col ifTrue: [
      (wordstart == 0 or: [linestart == wordstart]) ifTrue: [
	str add: (self copyFrom: linestart to: i).
	linestart := i + 1.
	curcol := 1.
      ]
      ifFalse: [
	str add: (self copyFrom: linestart to: wordstart - 1).
	linestart := wordstart.
	curcol := i - wordstart + 1.
      ].
      str add: lf.
      str add: indentStr .
      curcol := curcol + indentStr size . 
    ].
  ].
].

linestart <= sz ifTrue: [
  str add: (self copyFrom: linestart to: sz )
].

^str.
%

! fix bug 11542
! fixed bug 35799
category: 'Copying'
method:
copyFrom: startIndex to: stopIndex

"Returns a new SequenceableCollection containing the elements of the receiver
 between startIndex and stopIndex, inclusive.  The result is of the same class
 as the receiver, unless the receiver is a Symbol or DoubleByteSymbol,
 in which case the result class is respectively String or DoubleByteString.

 Both startIndex and stopIndex must be positive integers not larger than the
 size of the receiver, with startIndex <= stopIndex.
 If startIndex > stopIndex and both are positive, an empty collection is returned.
 "

| result |
(startIndex > stopIndex) ifTrue: [ 
  stopIndex < 0 ifTrue:[ self _error: #rtErrBadCopyFromTo args: { stopIndex } ].
  ^ self class new 
].
(startIndex < 1)
   ifTrue: [ ^ self _errorIndexOutOfRange: startIndex].

((stopIndex > self size) or: [(stopIndex < 0)])
   ifTrue: [ ^ self _errorIndexOutOfRange: stopIndex].

result := (self class _newString: (stopIndex - startIndex + 1)).
result replaceFrom: 1 to: 1 + stopIndex - startIndex 
	with: self startingAt: startIndex  .
^ result
%

! deleted _copyFrom:to:into:startingAt:

category: 'Copying'
method:
copyWithout: anObject

"Returns a copy of the receiver that does not contain the given object.
 Comparisons are by equality."

| copy element sz |

copy := self class _newString .

sz := 0.
1 to: self size do: [:i |
  element := self at: i.
  (element = anObject)
    ifFalse: [
      sz := sz + 1.
      copy at: sz put: element.
      ]
  ].

^copy
%

category: 'Case-Sensitive Searching'
method:
findString: subString startingAt: startIndex

"If a receiver contains subString beginning at some point at or after
 startIndex, this returns the index at which subString begins.  If the
 receiver does not contain subString, this returns 0.

 The search is case-sensitive."

^ self _findString: subString startingAt: startIndex ignoreCase: false
%

method:
findLastSubString: subString startingAt: startIndex

"startIndex should be >= 1 and <= self size . Search is backwards
 through the receiver. Returns 0 if no match found. "
^  self _findLastString: subString startingAt: startIndex ignoreCase: false
%

category: 'Case-Insensitive Searching'
method:
includesString: aString

"Returns true if aString is contained as a subString within the receiver,
 using a case-insensitive search.  Returns false otherwise."

^ (self _findString: aString startingAt: 1 ignoreCase: true) ~~ 0
%

category: 'Case-Insensitive Searching'
method:
findStringNoCase: subString startingAt: startIndex

"If a receiver contains subString beginning at some point at or after
 startIndex, this returns the index at which subString begins.  If the
 receiver does not contain subString, this returns 0.

 The search is case-insensitive."

^ self _findString: subString startingAt: startIndex ignoreCase: true
%

category: 'Private'
method:
_findString: subString startingAt: startIndex ignoreCase: aBoolean

CharacterCollection
  subclassResponsibility: #_findString:startingAt:ignoreCase: .

self _uncontinuableError
%

category: 'Private'
method:
_findLastString: subString startingAt: startIndex ignoreCase: aBoolean

"If a receiver contains subString beginning at some point at or before
 startIndex, this returns the index at which subString begins.  If the
 receiver does not contain subString, this returns 0. 
 Search is backwards. "

 self  subclassResponsibility: #_findLastString:startingAt:ignoreCase: .

 self _uncontinuableError
%

category: 'Searching'
method:
findPattern: aPattern startingAt: anIndex 

"This method searches the receiver, beginning at anIndex, for a substring that
 matches aPattern.  If a matching substring is found, this method returns the
 index of the first Character of the substring.  Otherwise, this returns 0.

 The argument aPattern is an Array containing zero or more CharacterCollections
 plus zero or more occurrences of the special Characters asterisk or
 question-mark.  See the description of the matchPattern: method for more
 information about this argument.

 Performs a case-sensitive search."

^ self _findPattern: aPattern startingAt: anIndex ignoreCase: false
%

category: 'Searching'
method:
findPatternNoCase: aPattern startingAt: anIndex 

"This method searches the receiver, beginning at anIndex, for a substring that
 matches aPattern.  If a matching substring is found, this method returns the
 index of the first Character of the substring.  Otherwise, this returns 0.

 The argument aPattern is an Array containing zero or more CharacterCollections
 plus zero or more occurrences of the special Characters asterisk or
 question-mark.  See the description of the matchPattern: method for more
 information about this argument.

 Performs a case-insensitive search."

^ self _findPattern: aPattern startingAt: anIndex ignoreCase: true
%

category: 'Searching'
method:
_findPattern: aPattern startingAt: anIndex ignoreCase: caseInsens

"This method searches the receiver, beginning at anIndex, for a substring that
 matches aPattern.  If a matching substring is found, this method returns the
 index of the first Character of the substring.  Otherwise, this returns 0.

 The argument aPattern is an Array containing zero or more CharacterCollections
 plus zero or more occurrences of the special Characters asterisk or
 question-mark.  See the description of the matchPattern: method for more
 information about this argument.

 If caseInsens is true, a case-insensitive search is performed.  Otherwise,
 caseInsens should be false and a case-sensitive search is performed."

| i             "loop counter"
  pattern       "the argument aPattern, converted to an Array"
  selfSize      "the size of the receiver"
  patternSize   "the number of elements in the pattern Array"
  startIndex    "an Array that corresponds to the pattern Array, with each
                 element containing the starting index into the receiver of
                 the corresponding pattern"
  index         "the index into both the pattern and the startIndex Arrays"
  next          "the current pattern (element of the pattern Array) to
                 be matched"
  cursor        "the index into the receiver"
|

aPattern _validateClass: Array.
(anIndex <= 0) ifTrue: [ ^ self _errorIndexOutOfRange: anIndex ].

pattern := Array withAll: aPattern.
patternSize := pattern size.

" First, the pattern Array must be processed so that there are no *? pairs
  it in.  They must all be converted to ?* pairs for the algorithm to work
  correctly."
i := 1.
[ i < patternSize ]
whileTrue:
  [ ( ((pattern at: i) isEquivalent: $*) and:[(pattern at: i+1) isEquivalent: $?])
    ifTrue:
      [ pattern at: i put: $?.
        pattern at: i+1 put: $*.
        i := 1 max: i-1.
      ]
    ifFalse:
      [ i := i + 1 ].
  ].

"initialize"
selfSize := self size.
startIndex := Array new: (patternSize + 1)."last element is set, but not used"
index := 1.
anIndex > selfSize
   ifTrue: [startIndex at: 1 put: (selfSize + 1)] "Fix for bug 15038"
   ifFalse: [startIndex at: 1 put: anIndex].

(patternSize == 0) "no pattern to match"
ifTrue:
  [ startIndex <= selfSize
    ifTrue:
      [ ^ startIndex ]
    ifFalse:
      [ ^ 0 ].
  ].

[index <= patternSize]
whileTrue:
  [ next := pattern at: index.
    cursor := startIndex at: index.

    (next isKindOf: CharacterCollection) "pattern element is a string"
    ifTrue:
      [ ((cursor > selfSize) or:
        [(cursor + next size - 1) > selfSize]) "pattern element too big to "
        ifTrue: "match beginning at cursor location"
          [ ^ 0 ].
        ((index == 1) or: [ (pattern at: index - 1) isEquivalent: $* ])
        ifTrue: "is first pattern or end of *; can skip chars to find match"
          [ cursor := self _findString: next startingAt: cursor 
				ignoreCase: caseInsens .
            (cursor == 0)
            ifTrue:
              [ ^ 0 ]
            ifFalse:
              [ startIndex at: index put: cursor.
                startIndex at: index + 1 put: cursor + next size.
              ]
          ]
        ifFalse: "can't skip chars to find match"
          [ (self _at: cursor equals: next ignoreCase: caseInsens )
            ifTrue:
              [ startIndex at: index + 1 put: cursor + next size ]
            ifFalse:
              [
                [ (index := index - 1) < 1
                  ifTrue:
                    [ ^ 0 ].
                  ((pattern at: index) isKindOf: CharacterCollection) or:
                  [ (index == 1) or: [ (pattern at: index - 1) isEquivalent: $* ] ]
                ]
                untilTrue.
                startIndex at: index put: ((startIndex at: index) + 1).
                index := index - 1.
              ].
          ].
      ]
    ifFalse: "pattern element not a string"
      [ (next isEquivalent: $*) "pattern element is *"
        ifTrue:
          [ startIndex at: (index + 1) put: cursor
          ]
        ifFalse:
          [ (next isEquivalent: $?)  "pattern element is ?"
            ifTrue:
              [ cursor > selfSize
                ifTrue:
                  [ ^ 0 ].
                startIndex at: (index + 1) put: (cursor + 1)
              ]
            ifFalse: "found a pattern element other than ?, * or string"
              [ ^ aPattern _error: #rtErrBadPattern
              ].
          ].
      ].

    index := index + 1
  ].

^ startIndex at: 1
%

category: 'Searching'
method:
indexOf: pattern matchCase: flag startingAt: startIndex

"Searches the receiver, beginning at anIndex, for a substring that
 matches aPattern.  If a matching substring is found, returns the
 index of the first Character of the substring.  Otherwise, returns 0.

 The argument pattern is an Array containing zero or more CharacterCollections
 plus zero or more occurrences of the special Characters asterisk or
 question-mark.  See the description of the matchPattern: method for more
 information about this argument.

 If the flag argument is true, a case-sensitive search is performed.  Otherwise,
 a case-insensitive search is performed."

^ self _findPattern: pattern startingAt: startIndex ignoreCase: (flag not).
%

category: 'Searching'
method:
maxConsecutiveSubstring

"Returns the largest substring within the receiver that contains Characters with
 consecutive ASCII values.  For example, the message

   'abxabcdxabc' maxConsecutiveSubstring

 yields the result 'abcd'.

 If there are no such substrings larger than 2 Characters, returns a String that
 contains the first Character in the receiver."

| mySize bigSize bigStart aStart aSize lastVal thisVal |

mySize := self size .
mySize < 2 ifTrue:[ ^ self ].
bigSize := 1 .
bigStart := 1 .
aStart := 1 .
aSize := 1 .
lastVal := (self at: 1) codePoint  .
2 to: mySize do:[:j |
  lastVal := lastVal + 1 .
  thisVal := (self at: j) codePoint .
  thisVal == lastVal 
    ifTrue:[ aSize := aSize + 1 ]
    ifFalse:[
      aSize > bigSize ifTrue:[
	bigStart := aStart .
	bigSize := aSize .
	].
      aSize := 1 .
      lastVal := thisVal .
      aStart := j .
      ].
   ].
aSize > bigSize ifTrue:[
  bigStart := aStart.
  bigSize := aSize.
  ].
^ self copyFrom: bigStart to: (bigStart + bigSize - 1) .
%

category: 'Searching'
method:
maxRepeatingSubstring

"Returns the largest substring within the receiver that contains repetitions of
 a Character, using case-sensitive comparison.  For example, the message

   'aaxbbbBxccc' maxRepeatingSubstring

 yields the result 'bbb'.

 If there are no such substrings larger than 1 Character, returns a String that
 contains the first Character in the receiver."

| mySize bigSize bigStart aStart aSize lastChar |

mySize := self size .
mySize < 2 ifTrue:[ ^ self ].
bigSize := 1 .
bigStart := 1 .
aStart := 1 .
aSize := 1 .
lastChar := self at: 1 .
2 to: mySize do:[:j | | thisChar |
  thisChar := self at: j .
  thisChar == lastChar 
    ifTrue:[ aSize := aSize + 1 ]
    ifFalse:[
      aSize > bigSize ifTrue:[
	bigStart := aStart .
	bigSize := aSize .
	].
      aSize := 1 .
      lastChar := thisChar .
      aStart := j .
      ].
   ].
aSize > bigSize ifTrue:[
  bigStart := aStart.
  bigSize := aSize.
  ].
^ self copyFrom: bigStart to: (bigStart + bigSize - 1) .
%

category: 'Private'
method:
_deepCopyWith: copiedObjDict

| copy |

copy := copiedObjDict at: self otherwise: nil.
copy ifNotNil:[ ^ copy ].

^ self copy.
%

category: 'Searching'
method:
maxSameTypeSubstring

"Returns the largest substring within the receiver that contains either all
 digits, all alphabetic characters, or all special characters.  For example, the
 message

   'axv2435,.-' maxSameTypeSubstring

 yields the result '2435'.

 If there are no such substrings larger than 1 Character, returns a String that
 contains the first Character in the receiver.

 This method may generate an error if the receiver is a JapaneseString."

| mySize bigSize bigStart aStart aSize lastType |

mySize := self size .
mySize < 2 ifTrue:[ ^ self ].
bigSize := 1 .
bigStart := 1 .
aStart := 1 .
aSize := 1 .
lastType := (self at: 1) _type  .
2 to: mySize do:[:j | | thisType |
  thisType := (self at: j) _type .
  thisType == lastType 
    ifTrue:[ aSize := aSize + 1 ]
    ifFalse:[
      aSize > bigSize ifTrue:[
	bigStart := aStart .
	bigSize := aSize .
	].
      aSize := 1 .
      lastType := thisType .
      aStart := j .
      ].
   ].
aSize > bigSize ifTrue:[
  bigStart := aStart.
  bigSize := aSize.
  ].
^ self copyFrom: bigStart to: (bigStart + bigSize - 1) .
%

category: 'Converting'
method:
trimLeadingSeparators

"Returns a CharacterCollection containing the same Characters as the receiver,
 but with leading separators removed."

| idx sz |
(sz := self size) == 0 ifTrue:[ ^ self ].
(self at: 1) isSeparator ifFalse: [ ^ self ].
idx := 2.
[ true ] whileTrue:[
  idx <= sz ifFalse:[ ^ '' ].
  (self at: idx) isSeparator ifFalse:[ ^ self copyFrom: idx to: sz].
  idx := idx + 1
]
%

category: 'Converting'
method:
trimLeadingBlanks

"Returns a CharacterCollection containing the same Characters as the receiver,
 but with leading blanks removed."

| idx blank sz |
(sz := self size) == 0 ifTrue: [ ^ self ].
blank := Character space.
((self at: 1) == blank) ifFalse: [ ^ self ].
idx := 2.
[ true ] whileTrue:[
  idx <= sz ifFalse:[ ^ '' ].
  (self at: idx) == blank ifFalse:[ ^ self copyFrom: idx to: sz].
  idx := idx + 1
]
%

category: 'Converting'
method:
trimTrailingSeparators

"Returns a CharacterCollection containing the same Characters as the receiver,
 but with trailing separators removed."

| idx sz |

(sz := self size) == 0 ifTrue: [ ^ self ].
(self at: sz ) isSeparator ifFalse: [ ^ self ].
idx := sz  - 1.
[ true ] whileTrue:[
  idx < 1 ifTrue:[ ^ '' ] .
  (self at: idx) isSeparator ifFalse:[ ^ self copyFrom: 1 to: idx ].
  idx := idx - 1
]
%

category: 'Converting'
method:
trimTrailingBlanks

"Returns a CharacterCollection containing the same Characters as the receiver,
 but with trailing blanks removed."

| idx blank sz |

(sz := self size) == 0 ifTrue: [ ^ self ].
blank := Character space.
(self at: sz ) == blank ifFalse: [ ^ self ].
idx := sz  - 1.
[ true ] whileTrue:[
  idx < 1 ifTrue:[ ^ '' ] .
  (self at: idx) == blank ifFalse:[ ^ self copyFrom: 1 to: idx ].
  idx := idx - 1
]
%

category: 'Converting'
method:
trimSeparators

"Returns a CharacterCollection containing the same Characters as the receiver,
 but with leading and trailing separators removed."

self size == 0 ifTrue: [ ^ self ].
^ (self trimLeadingSeparators) trimTrailingSeparators.
%

category: 'Converting'
method:
trimBlanks

"Returns a CharacterCollection containing the same Characters as the receiver,
 but with leading and trailing blanks removed."

self size == 0 ifTrue: [ ^ self ].
^ (self trimLeadingBlanks) trimTrailingBlanks.
%

category: 'Converting'
method:
subStrings

"Returns an Array of CharacterCollections where element represents a word in 
 the receiver.  A word is a group of Characters separated by one or more 
 separators."

^ self asArrayOfSubstrings.
%
    
! merge fix 18152  from 6,x     to fix 34079
category: 'Converting'
method:
subStrings: separators
	"Returns an Array of CharacterCollections in which each element represents a
	 substring separated by any of the Characters in separators.  For compatibility with
	 existing code, separators may be a single Character or a collection of Characters.
	 The result will include empty substrings when two adjacent separators exist, as well as if
	 a separator is the first or last element."

	^separators size = 0
		ifTrue: [self subStringsDelimitedBy: separators]
		ifFalse: [self subStringsDelimitedByAny: separators].
%


category: 'Converting'
method:
subStringsDelimitedBy: aCharacter
	"Returns an Array of CharacterCollections in which each element represents a
	 substring separated by aCharacter.  The result will include empty substrings when
	 two adjacent separators exist, as well as if a separator is the first or last element."

	| result startIndex endIndex sz |
	result := {}.
	startIndex := 1.
	
	[endIndex := self indexOf: aCharacter startingAt: startIndex.
	endIndex == 0]
			whileFalse: 
				[endIndex == startIndex ifTrue: [result add: self class _newString].
				endIndex > startIndex
					ifTrue: [result add: (self copyFrom: startIndex to: endIndex - 1)].
				startIndex := endIndex + 1].
	startIndex > (sz := self size)
		ifTrue: [result add: self class _newString]
		ifFalse: [result add: (self copyFrom: startIndex to: sz)].
	^result
%


category: 'Converting'
method:
subStringsDelimitedByAny: separators
	"Returns an Array of CharacterCollections in which each element represents a
	 substring separated by any of the Characters in separators.
	 The result will include empty substrings when two adjacent separators exist,
	 as well as if a separator is the first or last element."

	| nextName result |
	result := {}.
	nextName := self speciesForConversion new.
	self do: 
			[:c |
			(separators includes: c)
				ifTrue: 
					[result add: nextName.
					nextName := self speciesForConversion new]
				ifFalse: [nextName add: c]].
	result add: nextName.
	^result
%


category: 'Converting'
method:
asArrayOfPathTerms
  "Returns an Array of path substrings held by the receiver.  The receiver
 is assumed to be a period-separated list of substrings.  These substrings
 are extracted and collected in an Array.  If the receiver contains no
 periods, the Array will hold a copy of the receiver.  The $\ Character
 is no longer recognized as an escape character.

 Raises an error if an element is not a valid path term."

  | nextName result |
  result := {}.
  nextName := self speciesForConversion new.
  self
    do: [ :c | 
      c == $.
        ifTrue: [ 
          nextName _isValidPathTermName
            ifFalse: [ ^ self _error: #'rtErrInvalidIndexPathExpression' args: {nextName} ].
          result add: nextName asSymbol.
          nextName := self speciesForConversion new ]
        ifFalse: [ nextName add: c ] ].
  nextName size ~~ 0
    ifTrue: [ 
      nextName _isValidPathTermName
        ifFalse: [ ^ self _error: #'rtErrInvalidIndexPathExpression' args: {nextName} ].
      result add: nextName asSymbol ]
    ifFalse: [ 
      result size == 0
        ifTrue: [ result add: nextName asSymbol ] ].
  (result at: 1) = #'*'
    ifTrue: [ 
      "* not allowed as first term"
      ^ self _error: #'rtErrInvalidIndexPathExpression' args: {(result at: 1)} ].
  ^ result
%

method: 
asArrayOf32PathTerms
  "Returns an Array of path substrings held by the receiver.  The receiver
 is assumed to be a period-separated list of substrings.  These substrings
 are extracted and collected in an Array.  If the receiver contains no
 periods, the Array will hold a copy of the receiver.  The $\ Character
 is no longer recognized as an escape character.

 Raises an error if an element is not a valid path term."

  | nextName result |
  result := {}.
  nextName := self speciesForConversion new.
  self
    do: [ :c | 
      c == $.
        ifTrue: [ 
          nextName _isValid32PathTermName
            ifFalse: [ ^ self _error: #'rtErrInvalidIndexPathExpression' args: { nextName } ].
          result add: nextName asSymbol.
          nextName := self speciesForConversion new ]
        ifFalse: [ nextName add: c ] ].
  nextName size ~~ 0
    ifTrue: [ 
      nextName _isValid32PathTermName
        ifFalse: [ ^ self _error: #'rtErrInvalidIndexPathExpression' args: { nextName } ].
      result add: nextName asSymbol ]
    ifFalse: [ 
      result size == 0
        ifTrue: [ result add: nextName asSymbol ] ].
  ^ result
%

method: 
_isValid32PathTermName
  "Returns true if the receiver is a valid term in a path expression."

  | first sz maxSize |
  self = '*'
    ifTrue: [ ^ true ].
  maxSize := 64.
  ((sz := self size) > maxSize or: [ sz == 0 ])
    ifTrue: [ ^ false ].
  first := self at: 1.
  (first == $# or: [ first == $_ or: [ first isLetter ] ])
    ifFalse: [ ^ false ].
  2 to: sz do: [ :i |
    | c |
    c := self at: i.
    (c == $_ or: [ c == $| or: [ c isAlphaNumeric ] ])
      ifFalse: [ ^ false ] ].
  ^ true
%

category: 'Testing'
method:
isValidIdentifier

"Returns true if the receiver is a valid GemStone Smalltalk variable name, 
 and false otherwise."

^ self _validIdentifier: 64
%

method:
_validIdentifier: maxSize 
| first sz |

((sz := self size) > maxSize or:[sz == 0]) ifTrue:[ ^false ].

first := self at: 1.
(first == $_ or:[first isLetter]) ifFalse:[ ^false ].

2 to: sz do: [:i | | c |
  c := self at: i.
  (c == $_  or:[ c isAlphaNumeric]) ifFalse:[ ^false ].
].
^ true
%

category: 'Indexing Support'
method:
_isValidPathTermName
  "Returns true if the receiver is a valid term in a path expression."

  ^ self _isValid32PathTermName
%

category: 'Accessing'
method:
numArgs

"Returns the number of arguments the receiver would take, were the receiver
 a message selector."

| idx count sz |
(sz := self size) == 0 ifTrue:[ ^ 0 ].
(self at: sz) == $: ifTrue:[
  count := 1 .
  idx := 0 .
  [ idx := self indexOf: $: startingAt: idx + 1 .
    idx ~~ sz 
  ] whileTrue:[
    count := count + 1
  ].  
  ^ count
] ifFalse:[
  self isInfix ifTrue: [ ^ 1 ].
  ^ 0
]
%

! fix 46696
category: 'Testing'
method:
isKeyword

"Returns true if the receiver is a keyword; that is, a legal keyword method
 selector. Returns false otherwise."

| mySize limit ch idx |

mySize := self size.
mySize > 1  ifFalse: [ ^ false ].
idx := 1 .
[ idx <= mySize ] whileTrue:[
  ch := self at: idx .  "first char of next keyword within selector"
  ( ch == $_ or:[ ch isLetter ] ) ifFalse: [ ^ false ].
  idx := idx + 1 .
  limit := mySize .
  [ idx <= limit ] whileTrue:[
    ch := self at: idx .
    ch == $:  ifTrue:[ 
      idx == mySize ifTrue:[ ^ true "hit ending $: " ].
      limit := 0 . "end of a keyword, exit inner loop" 
    ] ifFalse:[
      (ch == $_  or:[ ch isAlphaNumeric]) ifFalse:[ ^ false ].
    ] .
    idx := idx + 1 
  ]
].
^ false
%

category: 'New Indexing Comparison'
method:
_classSortOrdinal

^ 10
%
category: 'New Indexing Comparison'
method:
_idxForCompareEqualTo: arg
  ""

  ^ arg _idxForCompareEqualToCharacterCollection: self
%
category: 'New Indexing Comparison'
method:
_idxForCompareGreaterThan: arg

""

^arg _idxForCompareCharacterCollectionGreaterThanSelf: self
%
category: 'New Indexing Comparison'
method:
_idxForCompareGreaterThanOrEqualTo: arg

""

^arg _idxForCompareCharacterCollectionGreaterThanOrEqualToSelf: self
%
category: 'New Indexing Comparison'
method:
_idxForCompareLessThan: arg

""

^arg _idxForCompareCharacterCollectionLessThanSelf: self
%
category: 'New Indexing Comparison'
method:
_idxForCompareLessThanOrEqualTo: arg

""

^arg _idxForCompareCharacterCollectionLessThanOrEqualToSelf: self
%
category: 'New Indexing Comparison'
method:
_idxForSortEqualTo: arg

""

^arg _idxForSortEqualToCharacterCollection: self
%
category: 'New Indexing Comparison'
method:
_idxForSortGreaterThan: arg

""

^arg _idxForSortCharacterCollectionGreaterThanSelf: self
%
category: 'New Indexing Comparison'
method:
_idxForSortGreaterThanOrEqualTo: arg

""

^arg _idxForSortCharacterCollectionGreaterThanOrEqualToSelf: self
%
category: 'New Indexing Comparison'
method:
_idxForSortLessThan: arg

""

^arg _idxForSortCharacterCollectionLessThanSelf: self
%
category: 'New Indexing Comparison'
method:
_idxForSortLessThanOrEqualTo: arg

""

^arg _idxForSortCharacterCollectionLessThanOrEqualToSelf: self
%

category: 'New Indexing Comparison'
method:
_idxForSortNotEqualTo: arg

""

^(arg _idxForSortEqualTo: self) not
%

category: 'New Indexing Comparison - for Sort'
method:
_idxForSortCharacterCollectionGreaterThanOrEqualToSelf: aCharacterCollection

"second half of a double dispatch call from CharacterCollection>>_idxForSortGreaterThanOrEqualTo:. Note that aCharacterCollection should be the receiver in any >= comparison"

^(aCharacterCollection _idxPrimCompareLessThan: self) not
%
category: 'New Indexing Comparison - for Sort'
method:
_idxForSortCharacterCollectionGreaterThanSelf: aCharacterCollection

"second half of a double dispatch call from CharacterCollection>>_idxForSortGreaterThan:. Note that aCharacterCollection should be the receiver in any > comparison"

^aCharacterCollection _idxPrimCompareGreaterThan: self
%
category: 'New Indexing Comparison - for Sort'
method:
_idxForSortCharacterCollectionLessThanOrEqualToSelf: aCharacterCollection

"second half of a double dispatch call from CharacterCollection>>_idxForSortLessThanOrEqualTo:. Note that aCharacterCollection should be the receiver in any <= comparison"

^(aCharacterCollection _idxPrimCompareGreaterThan: self) not
%
category: 'New Indexing Comparison - for Sort'
method:
_idxForSortCharacterCollectionLessThanSelf: aCharacterCollection

"second half of a double dispatch call from CharacterCollection>>_idxForSortLessThan:. Note that aCharacterCollection should be the receiver in any < comparison"

^aCharacterCollection _idxPrimCompareLessThan: self
%
category: 'New Indexing Comparison - for Sort'
method:
_idxForSortEqualToCharacterCollection: aCharacterCollection
  "second half of a double dispatch call from CharacterCollection>>_idxForSortEqualTo:."

  ^ self _idxPrimCompareEqualTo: aCharacterCollection
%

category: 'New Indexing Comparison - for Sort'
method:
 _idxForSortNotEqualToCharacterCollection: aCharacterCollection

"second half of a double dispatch call from CharacterCollection>>_idxForSortNotEqualTo:."

^ (self _idxPrimCompareEqualTo: aCharacterCollection) not
%

category: 'New Indexing Comparison - for Compare'
method:
_idxForCompareCharacterCollectionGreaterThanOrEqualToSelf: aCharacterCollection

"second half of a double dispatch call from CharacterCollection>>_idxForCompareGreaterThanOrEqualTo:. Note that aCharacterCollection should be the receiver in any >= comparison"

^(aCharacterCollection _idxPrimCompareLessThan: self) not
%
category: 'New Indexing Comparison - for Compare'
method:
_idxForCompareCharacterCollectionGreaterThanSelf: aCharacterCollection

"second half of a double dispatch call from CharacterCollection>>_idxForCompareGreaterThan:. Note that aCharacterCollection should be the receiver in any > comparison"

^aCharacterCollection _idxPrimCompareGreaterThan: self
%
category: 'New Indexing Comparison - for Compare'
method:
_idxForCompareCharacterCollectionLessThanOrEqualToSelf: aCharacterCollection

"second half of a double dispatch call from CharacterCollection>>_idxForCompareLessThanOrEqualTo:. Note that aCharacterCollection should be the receiver in any <= comparison"

^(aCharacterCollection _idxPrimCompareGreaterThan: self) not
%
category: 'New Indexing Comparison - for Compare'
method:
_idxForCompareCharacterCollectionLessThanSelf: aCharacterCollection

"second half of a double dispatch call from CharacterCollection>>_idxForCompareLessThan:. Note that aCharacterCollection should be the receiver in any < comparison"

^aCharacterCollection _idxPrimCompareLessThan: self
%

category: 'New Indexing Comparison'
method:
_idxForCompareNotEqualTo: arg

""

^ (self _idxForCompareEqualTo: arg) not
%

category: 'New Indexing Comparison - prims'
method:
_idxPrimCompareEqualTo: arg
  "This comparison operation is used for the indexing subsystem to
 determine an ordering for insertion into indexing objects.

 This method collates letters AaBb..Zz."

  "The comparison should be compatible with the case-insensitive semantics
 of the String method with selector #= .
 Same primitive as String>>lessThan: "

  self subclassResponsibility: #_idxPrimCompareEqualTo:
%

category: 'New Indexing Comparison - prims'
method:
_idxPrimCompareLessThan: arg
  "This comparison operation is used for the indexing subsystem to
 determine an ordering for insertion into indexing objects.

 This method collates letters AaBb..Zz."

  "The comparison should be compatible with the case-insensitive semantics
 of the String method with selector #< .
 Same primitive as String>>lessThan: "

  self subclassResponsibility: #_idxPrimCompareLessThan:
%
category: 'New Indexing Comparison - prims'
method:
_idxPrimCompareGreaterThan: arg
  "This comparison operation is used for the indexing subsystem to
 determine an ordering for insertion into indexing objects.

 This method collates letters AaBb..Zz."

  "The comparison should be compatible with the case-insensitive semantics
 of the String method with selector #< .
 Same primitive as String>>lessThan: "

  self subclassResponsibility: #_idxPrimCompareGreaterThan:
%

category: 'Converting'
method:
copyWrappedTo: rightMargin

"Returns a String with the receiver's contents word-wrapped to the given
 right margin."

| res col ch lf |

self size < rightMargin ifTrue: [^self].
lf := Character lf.
res := self species new.
col := 0.
1 to: self size do: [ :i |
  ch := self at: i.
  res add: ch.
  ch == lf ifTrue: [ col := 0 ]
  ifFalse: [
    col := col + 1.
    (col > rightMargin and: [ch isSeparator]) ifTrue: [
      res add: lf.
      col := 1.
    ].
  ].
].
^ res
%

category: 'Formatting'
method:
linesIndentedBy: anInt

"Returns a copy of the receiver in which all lines have been indented
 by anInt spaces."

| c newStr indentStr lfInd lf targEndInd selfCurrInd sz destIdx |

indentStr := self class new: anInt.
indentStr atAllPut: Character space.
lf := Character lf.
lfInd := self indexOf: lf.
lfInd == 0 ifTrue:[ | res |
  (res := indentStr) addAll: self .
  ^ res 
].

newStr := self class new.
selfCurrInd := 1.
c := self copy.

[(lfInd := c indexOf: lf) ~~ 0 ] whileTrue: [ 
  targEndInd := newStr size + indentStr size.
  newStr addAll: indentStr.
 "self copyFrom: selfCurrInd to: lfInd into: newStr startingAt: targEndInd + 1. "
  destIdx := targEndInd + 1 .
  newStr replaceFrom: destIdx to: destIdx + lfInd - selfCurrInd 
	  with: self startingAt: selfCurrInd .
  selfCurrInd := lfInd + 1.
  c at: lfInd put: Character space.
].
selfCurrInd < (sz := self size) ifTrue: [
  newStr addAll: indentStr.
 "self copyFrom: selfCurrInd to: sz into: newStr startingAt: newStr size + 1."
  destIdx := newStr size + 1 .
  newStr replaceFrom: destIdx to: destIdx + sz - selfCurrInd 
         with: self startingAt: selfCurrInd .
].
^newStr
%

category: 'Formatting'
method:
printString

"Returns a CharacterCollection whose contents are a displayable representation of the
 receiver."

^ self printStringWithMaxSize: 100000
%


category: 'Copying'
method:
copyReplaceAll: subString with: newSubString

"Returns a copy of the receiver with all occurrences of the given substring
 replaced with newSubString."

| copy csize matches ssize idx |

copy := self copy.
matches := { } .
ssize := subString size.
ssize == 0 ifTrue: [^copy].
idx := 1 - ssize.
csize := copy size.
[ idx := idx + subString size.
  idx <= csize and: [
    idx := copy findString: subString startingAt: idx.
    idx ~~ 0
  ]
] whileTrue: [
  matches add: idx
].
matches reverseDo: [:p |
  copy removeFrom: p to: p+ssize-1.
  copy insertAll: newSubString at: p
].
^copy
%

category: 'Copying'
method:
copyReplaceChar: aCharacter with: secondCharacter

"Returns a copy of the receiver with all occurrences of the given
 Character replaced with secondCharacter."

| copy |
secondCharacter _validateClass: Character .
copy := self speciesForCollect withAll: self .
1 to: copy size do:[:n |
 (copy at: n) == aCharacter ifTrue:[ copy at: n put: secondCharacter].
].
^ copy
%

category: 'Copying'
method: 
withCRs

"Supplied for Smalltalk-80 compatibility.  This is equivalent to withLFs."

^self withLFs
%

category: 'Testing'
method:
isDigits

"Returns true if the receiver contains only digits.  Returns false if the
 receiver contains non-digit Characters."

1 to: self size do: [:i |
    (self at: i) isDigit ifFalse: [^false]
].
^true
%

category: 'Testing'
method:
isInfix

"Returns true if the receiver is an infix (binary) selector.  Returns false
 otherwise."

| binaryChars mySize fch |

binaryChars := '+-\*~<>=|/&@%,?!'.   "fixed bug 14109"

mySize := self size.
mySize == 0 ifTrue: [ ^ false ].

fch := self at: 1 .
(fch == $_  or:[ (binaryChars indexOf: fch startingAt: 1) ~~ 0] ) ifFalse:[
  ^ false 
].
2 to: mySize do: [ :i |
  ((binaryChars indexOf: (self at: 1) startingAt: 1) == 0) ifTrue:[
     ^ false
  ].
].
^ true.
%

category: 'Testing'
method:
isMinusAndDigits

"Returns true if the receiver contains a minus sign followed only by digits.
 Returns false if the receiver has any other Characters."

((self at: 1) == $- ) ifFalse:[ ^ false ].
2 to: self size do: [:i |
    (self at: i) isDigit ifFalse:[ ^ false]
].
^true
%
category: 'Hashing'
method: 
hash

"Returns a positive Integer based on a case-sensitive hash of the contents 
 of the receiver.  The algorithm implemented is described in:

 [Pearson 90]
 Pearson, Peter K., Fast Hashing of Variable-Length Text Strings,
 Communications of the ACM 33, 6, (June 1990), 677-680.

 This implementation inherited by JapaneseString.  String and 
 MultiByteString each reimplement hash."

<primitive: 31>
self _primitiveFailed: #hash .
self _uncontinuableError
%
category: 'Searching'
method:
indexOfSubCollection: aSubColl startingAt: anIndex ifAbsent: anExceptionBlock

"Returns the index of the first element of the receiver where that element and
 the subsequent ones are equal to those in aSubColl. The search is begun in the
 receiver at starting at anIndex. Returns the value of evaluating 
 anExceptionBlock if no match is found."

| idx |
(aSubColl size == 0) ifTrue: [ ^ anExceptionBlock value ].
idx := self findString: aSubColl startingAt: anIndex .
idx == 0 ifTrue:[ ^ anExceptionBlock value ].
^ idx
%

category: 'Searching'
method:
indexOfSubCollection: aSubColl startingAt: anIndex 

"Returns the index of the first element of the receiver where that element and
 the subsequent ones are equal to those in aSubColl. The search is begun in the
 receiver at starting at anIndex. Returns zero if no match is found."

(aSubColl size == 0) ifTrue: [ ^ 0 ].
^ self findString: aSubColl startingAt: anIndex .
%

category: 'Formatting'
method: CharacterCollection
addLineWith: aCharacterCollection centeredToWidth: anInt

"Add a centered line of width anInt to the receiver.
 Returns the receiver."

| numSpaces aSpace|
aSpace := Character space.
numSpaces := anInt - aCharacterCollection size.
numSpaces <= 0 "No room, just do the add"
  ifTrue:[self addAll: aCharacterCollection.]
  ifFalse:[ | leftSpaces |
    leftSpaces := numSpaces // 2.
    leftSpaces timesRepeat:[self add: aSpace].
    self addAll: aCharacterCollection.
    (numSpaces - leftSpaces) timesRepeat:[self add: aSpace].
].
^self
%

category: 'LDAP Support'
method: CharacterCollection
hasLdapWildcardPattern

"Answer true if the receiver contains one and only one '%' character, 
 and the character after the '%' is 's'."

(self select:[:e| e == $%]) size == 1
  ifFalse:[^ false ].

^(self findPattern: #('%' 's') startingAt: 1) ~~ 0 
%
category: 'Repository Conversion'
method: CharacterCollection
convert
"If receiver is a committed object and receiver's class on disk was 
 a subclass of one of
   ObsDoubleByteString, ObsDoubleByteSymbol, ObsQuadByteString
 the receiver is added to the writeSet of current transaction,
 otherwise has no effect.
 Returns true if receiver was added to the writeSet, false otherwise."

<primitive: 744>
self _primitiveFailed:#convert
%

category: 'Searching'
method: CharacterCollection
containsDigit

"Answers true if the receiver contains at least one digit, otherwise
 answers false."

1 to: self size do:[:n| (self at: n) isDigit ifTrue:[ ^ true ] ] .
^ false
%

category: 'Searching'
method: CharacterCollection
containsUppercase

"Answers true if the receiver contains at least one upper case character,
 otherwise answers false."

1 to: self size do:[:n| (self at: n) isUppercase ifTrue:[ ^ true ] ] .
^ false
%

category: 'Searching'
method: CharacterCollection
containsLowercase

"Answers true if the receiver contains at least one lower case character,
 otherwise answers false.  
 Differs from legacy method containsLowercaseOld for 
 code points 170, 186, and about 1700 code points >= 256"

1 to: self size do:[:n| (self at: n) isLowercase ifTrue:[ ^ true ] ] .
^ false
%
category: 'Legacy'
method: CharacterCollection
containsLowercaseOld

"Answers true if the receiver contains at least one lower case character,
 otherwise answers false."

1 to: self size do:[:n| (self at: n) isLowercaseOld ifTrue:[ ^ true ] ] .
^ false
%

category: 'Private'
method: CharacterCollection
containsPasswordSymbol

"Answers true if the receiver contains at least one symbol character, with
 symbol in this case meaning any Character that is neither a letter nor a 
 digit; otherwise answers false."

1 to: self size do:[:n| (self at: n) _isPasswordSymbol ifTrue:[ ^ true ] ] .
^ false
%
category: 'Json'
method: CharacterCollection
printJsonOn: aStream

	aStream nextPut: $".
	1 to: self size do: [:j |  
		| codePoint char |
		codePoint := (char := self at: j) codePoint.
    (32 <= codePoint and: [codePoint <= 127]) ifTrue: [
      char == $" "34" ifTrue: [aStream nextPutAll: '\"' ] ifFalse: [
      char == $\ "92" ifTrue: [aStream nextPutAll: '\\' ] ifFalse: [
        aStream nextPut: char
      ]]
    ] ifFalse:[
      (codePoint <= 13 and:[ codePoint >= 8]) ifTrue:[
		    codePoint == 8 	ifTrue: [aStream nextPutAll: '\b'	. codePoint := nil. ] ifFalse: [
		    codePoint == 9 	ifTrue: [aStream nextPutAll: '\t'	. codePoint := nil. ] ifFalse: [
		    codePoint == 10 	ifTrue: [aStream nextPutAll: '\n'. codePoint := nil. 	] ifFalse: [
		    codePoint == 12 	ifTrue: [aStream nextPutAll: '\f'. codePoint := nil. 	] ifFalse: [
		    codePoint == 13 	ifTrue: [aStream nextPutAll: '\r'. codePoint := nil. 	]
      ]]]]].
      codePoint ifNotNil:[ | hex |
        hex := '0123456789ABCDEF'.
			  aStream nextPutAll: '\u' ;
             nextPut: (hex at: 1 + ((codePoint bitShift: -12) bitAnd: 16rF)) ;
             nextPut: (hex at: 1 + ((codePoint bitShift: -8) bitAnd: 16rF)) ;
             nextPut: (hex at: 1 + ((codePoint bitShift: -4) bitAnd: 16rF)) ;
             nextPut: (hex at: 1 + (codePoint bitAnd: 16rF)) .
      ]
	  ].
  ].
	aStream nextPut: $".
%
category: 'Copying'
method:
copy
  "optimization, no need for postCopy"
<primitive: 885>
self _primitiveFailed: #copy .
self _uncontinuableError
%

category: 'Private'
method:
_asUtf8WithEol
  "Return a Utf8 copy of the receiver, with a linefeed
   appended if receiver does not end with a linefeed.  Used by topaz"
  | lastCp sz str |
  sz := self size .
  sz ~~ 0 ifTrue:[ lastCp := self codePointAt: sz ].
  str := self .
  lastCp == 10 ifFalse:[
    (str := self copy) codePointAt: sz + 1 put: 10
  ].
  ^ str encodeAsUTF8
%

! File in methods common to both ByteArray and CharacterCollection.
! It's a shame they don't share a common byte-format superclass
! SequenceableCollection does not qualify!
!
input $upgradeDir/message_digests.gs

input $upgradeDir/encrypt_decrypt.gs

input $upgradeDir/sign_verify_asym.gs

category: 'New Indexing Comparison'
set compile_env: 0
method: CharacterCollection
_idxForCompareEqualTo: aCharacterCollection collator: anIcuCollator
  "treat receiver and aCharacterCollection as Unicode, since anIcuCollator is supplied"

  ^ aCharacterCollection
    _idxForCompareEqualToUnicode: self
    collator: anIcuCollator
%
category: 'New Indexing Comparison'
set compile_env: 0
method: CharacterCollection
_idxForCompareGreaterThan: aCharacterCollection collator: anIcuCollator
  "treat receiver and aCharacterCollection as Unicode, since anIcuCollator is supplied"

  ^ aCharacterCollection
    _idxForCompareGreaterThanUnicode: self
    collator: anIcuCollator
%
category: 'New Indexing Comparison'
set compile_env: 0
method: CharacterCollection
_idxForCompareGreaterThanOrEqualTo: aCharacterCollection collator: anIcuCollator

  ^ aCharacterCollection
    _idxForCompareGreaterThanOrEqualToUnicode: self
    collator: anIcuCollator
%
category: 'New Indexing Comparison'
set compile_env: 0
method: CharacterCollection
_idxForCompareLessThan: aCharacterCollection collator: anIcuCollator
 
  ^ aCharacterCollection
    _idxForCompareLessThanUnicode: self
    collator: anIcuCollator
%
category: 'New Indexing Comparison'
set compile_env: 0
method: CharacterCollection
_idxForCompareLessThanOrEqualTo: aCharacterCollection collator: anIcuCollator

  ^ aCharacterCollection
    _idxForCompareLessThanOrEqualToUnicode: self
    collator: anIcuCollator
%
category: 'New Indexing Comparison'
set compile_env: 0
method: CharacterCollection
_idxForCompareNotEqualTo: aCharacterCollection collator: anIcuCollator
  "REL_SIG_STRING_SIZE = 900"

  ^ (self _idxForCompareEqualTo: aCharacterCollection collator: anIcuCollator)
    not
%
category: 'New Indexing Comparison'
set compile_env: 0
method: CharacterCollection
_idxForSortEqualTo: aCharacterCollection collator: anIcuCollator

  ^ aCharacterCollection
    _idxForSortEqualToCharacterCollection: self
    collator: anIcuCollator
%
category: 'New Indexing Comparison'
set compile_env: 0
method: CharacterCollection
_idxForSortGreaterThan: aCharacterCollection collator: anIcuCollator

  ^ aCharacterCollection
    _idxForSortGreaterThanCharacterCollection: self
    collator: anIcuCollator
%
category: 'New Indexing Comparison'
method: CharacterCollection
_idxForSortGreaterThanOrEqualTo: aCharacterCollection collator: anIcuCollator

  ^ aCharacterCollection
    _idxForSortGreaterThanOrEqualToCharacterCollection: self
    collator: anIcuCollator
%
category: 'New Indexing Comparison'
method: CharacterCollection
_idxForSortLessThan: aCharacterCollection collator: anIcuCollator

  ^ aCharacterCollection
    _idxForSortLessThanCharacterCollection: self
    collator: anIcuCollator
%
category: 'New Indexing Comparison'
method: CharacterCollection
_idxForSortLessThanOrEqualTo: aCharacterCollection collator: anIcuCollator

  ^ aCharacterCollection
    _idxForSortLessThanOrEqualToCharacterCollection: self
    collator: anIcuCollator
%
category: 'New Indexing Comparison'
set compile_env: 0
method: CharacterCollection
_idxForSortNotEqualTo: aCharacterCollection collator: anIcuCollator

  ^ (self _idxForSortEqualTo: aCharacterCollection collator: anIcuCollator) not
%
! fixed 46981
category: 'Testing'
set compile_env: 0
method: CharacterCollection
_literalEqual: anotherLiteral
  "Petit Parser definition:
     For two literals to be _literalEqual, their class must be identical and  
     otherwise equal.
     For CharacterCollections equality is sufficent
   In GemStone we allow Unicode7 to compare to String, etc."

  ^ self _unicodeEqual: anotherLiteral
%
category: 'Indexing Support'
set compile_env: 0
classmethod: CharacterCollection
_idxBasicCanCompareWithCharacterCollectionInstance: aCharacterCollection
  "Returns true if <aCharacterCollection> may be inserted into a basic BtreeNode 
   whose #lastElementClass is the receiver (see RangeEqualityIndex 
   class>>isBasicClass:)."

  "If using Unicode compares, then *String and Unicode* instances may be compared.
   If not using Unicode compares then *String and Unicode* instances may not be compared."

  ^ true
%
category: 'Indexing Support'
set compile_env: 0
method: CharacterCollection
_idxBasicCanCompareWithClass: aClass
  "Returns true if the receiver may be inserted into a basic BtreeNode whose 
   #lastElementClass is <aClass> (see RangeEqualityIndex class>>isBasicClass:)."

  "If using Unicode compares, then *String and Unicode* instances may be compared.
   If not using Unicode compares then *String and Unicode* instances may not be compared."

  ^ aClass _idxBasicCanCompareWithCharacterCollectionInstance: self
%
category: 'Indexing Support'
set compile_env: 0
classmethod: CharacterCollection
_idxBasicCanCompareWithUnicodeInstance: aUnicodeString
  "Returns true if <aUnicodeString> may be inserted into a basic BtreeNode whose 
   #lastElementClass is the receiver (see RangeEqualityIndex class>>isBasicClass:)."

  "If using Unicode compares, then *String and Unicode* instances may be compared.
   If not using Unicode compares then *String and Unicode* instances may not be compared."

  ^ Unicode16 usingUnicodeCompares
%

category: 'Comparing'
method: CharacterCollection
compareTo: aString collator: anIcuCollator

"Returns -1, 0 or 1,  when receiver is less than,
 equal to, or greater than argString .
 argString must be a String, MultiByteString, or a Utf8 .
 anIcuCollator == nil is interpreted as   IcuCollator default ."

^ self compareTo: aString collator: anIcuCollator useMinSize: false
%

category: 'Comparing'
method: CharacterCollection
compareTo: argString collator: anIcuCollator useMinSize: aMinSize
  "Returns -1, 0 or 1,  when receiver is less than,
 equal to, or greater than argString .
 argString must be a String, MultiByteString, or a Utf8.
 anIcuCollator == nil is interpreted as   IcuCollator default .

 If aMinSize==true, compare stops at (self size min: argString size),
 which is Squeak semantics for comparison .

 If aMinSize is a SmallInteger >= 1, compare stops at
 aMinSize min: (self size min: argString size) ."

  ^ self subclassResponsibility: #'compareTo:collator:useMinSize:'
%

! added for 44088
category: 'Session Control'
classmethod: CharacterCollection
enableUnicodeComparisonMode

  "Causes subsequent sessions, after login, to use Unicode methods for 
   #< #> #= and various methods in the indexing
   when a receiver is a kind of String or MultiByteString.

   Signals an Error if the current user is not SystemUser .

   Returns a Boolean , the previous state of 
     (Globals at: #StringConfiguration) == Unicode16 ."

   | prev |
   System myUserProfile userId = 'SystemUser' ifFalse:[
     Error signal:'Only SystemUser should execute this method'.
   ].
   prev := (Globals at: #StringConfiguration otherwise: nil) == Unicode16 .
   Globals at: #StringConfiguration put: Unicode16 .
   System commit .
   ^ prev
%
category: 'Session Control'
classmethod: CharacterCollection
disableUnicodeComparisonMode

  "Causes subsequent sessions, after login, to use Legacy methods for 
   #< #> #= and various methods in the indexing
   when a receiver is a kind of String or MultiByteString.

   Signals an Error if the current user is not SystemUser .

   Returns a Boolean , the previous state of
     (Globals at: #StringConfiguration) == Unicode16 ."

   | prev |
   System myUserProfile userId = 'SystemUser' ifFalse:[
     Error signal:'Only SystemUser should execute this method'.
   ].
   prev := (Globals at: #StringConfiguration otherwise: nil) == Unicode16 .
   Globals at: #StringConfiguration put: String .
   System commit .
   ^ prev
%
category: 'Session Control'
classmethod: CharacterCollection
isInUnicodeComparisonMode

  "Returns a Boolean, true if Unicode versions of methods 
   #< #> #= and various methods in the indexing
   are used when a receiver is a kind of String or MultiByteString"

  ^ Unicode16 usingUnicodeCompares
%

category: 'Compatibility'
method:
_encodeAsUTF8intoString

^ self encodeAsUTF8IntoString
%
category: 'Adding'
method:
addCodePoint: aSmallInteger

 ^ self add: (Character codePoint: aSmallInteger)
%

category: 'Comparing'
method:
_coerceToUnicode
  ^ self asUnicodeString
%

