!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   GsNMethod, Object.
!
!   GsNMethod is an indexable subclass of Object
!
!=========================================================================

! remove existing behavior 

set class GsNMethod
removeallmethods GsNMethod
removeallclassmethods GsNMethod

input $upgradeDir/gsnmethod0.gs

category: 'For Documentation Installation only'
classmethod:
installDocumentation

self comment:
'A GsNMethod is a compiled form of a GemStone Smalltalk method.
 Subclasses of GsNMethod are disallowed.

 For Gs64 v3.x, all "step points" are expressed in terms of
 a source offset within the method (or home method) source.  
 Source offsets are 1-based in Smalltalk 
 and zero-based in primitives C code.

Constraints:
	iFields1: SmallInteger
	selector: SmallInteger
	inClass: Object
	debugInfo: Array

--- instVar debugInfo
An Array that contains debugging info and reference to sourceString,
   see file opalcls.ht for details .

--- instVar iFields1
A SmallInteger encoding the various fields per opalcls.ht .

--- instVar inClass
The Behavior (a Class or Metaclass or Module) for which the method was compiled,
 or nil in an anonymous method ,
 or the home method if this method contains code for a block .

--- instVar selector
A SmallInteger that encodes the oopNumber of a Symbol plus a
 16 bit unsigned environment identifer.
'.
%

category: 'Constants'
classmethod:
maxArgs
  "Returns the maximum number of arguments to a method."

  ^ GEN_MAX_ARGS
%


category: 'Accessing'
method:
_primitiveAt: anIndex

"Returns the value of an instance variable in the receiver.
 The argument anIndex must be a SmallInteger   >= 1 and 
 < (named size plus varying size) of receiver.

 Attempt to fetch from the send-site caches area of a method that
 has been loaded for execution will return nil.
 Use GsNMethod>>literals to get the complete literals pool.
"

<primitive: 806>
anIndex _validateClass: SmallInteger .
self _errorIndexOutOfRange: anIndex .
self _primitiveFailed: #_primitiveAt:  args: { anIndex }
%

method:
at: anIndex

"Returns value of specified varying instVar of receiver.

 Attempt to fetch from the send-site caches area of a method that
 has been loaded for execution will return nil.
 Use GsNMethod>>literals to get the complete literals pool.
"

anIndex < 1 ifTrue:[ self _errorIndexOutOfRange: anIndex ].
^ self _primitiveAt: (anIndex + GsNMethod_InstSize)
%

method:
_at: anIndex

"Returns value of specified varying instVar of receiver.
 Attempt to fetch from the send-site caches area of a method that
 has been loaded for execution will return nil.
 Use GsNMethod>>literals to get the complete literals pool.
"

anIndex < 1 ifTrue:[ self _errorIndexOutOfRange: anIndex ].
^ self _primitiveAt: (anIndex + GsNMethod_InstSize)
%
method:
_basicAt: anIndex

"Returns value of specified varying instVar of receiver.
 Attempt to fetch from the send-site caches area of a method that
 has been loaded for execution will return nil.
 Use GsNMethod>>literals to get the complete literals pool.
"

anIndex < 1 ifTrue:[ self _errorIndexOutOfRange: anIndex ].
^ self _primitiveAt: (anIndex + GsNMethod_InstSize)
%

method:
squeakBasicAt: anIndex
 ^ self _basicAt: anIndex
%

category: 'Accessing'
method:
_primitiveSize
"Returns total the number of instance variables in the receiver.
 For a GsNMethod in code_gen memory, the size includes the
 words in the send_cache area, and will be larger than the
 on-disk size by  (2*self numSends) ."

<primitive: 805>
self _primitiveFailed: #_primitiveSize .
self _uncontinuableError
%

method:
basicSize

"Returns varying size of receiver."

^ self _primitiveSize - GsNMethod_InstSize
%

method:
squeakBasicSize
^ self basicSize
%

method:
size
"Returns varying size of receiver."

^ self _primitiveSize - GsNMethod_InstSize
%

category: 'Accessing'
method:
at: anOffset put: aValue

"For use only on methods not yet invariant, such as during compilation
 of a CCallout for Ruby.  Most instances of GsNMethod are produced
 as invariant by primitives 679 or 228 .  Stores aValue into specified
 varying instVar of the receiver."

<primitive: 572>
anOffset _validateClass: SmallInteger .
self _errorIndexOutOfRange: anOffset .
self _primitiveFailed: #at:put:  args: { anOffset . aValue }
%

category: 'Execution'
method:
_executeInContext: anObject

"The receiver should be a zero argument GsNMethod obtained by invoking
 String/_compileInContext:symbolList: and the argument should be the argument
 passed to the _compileInContext: keyword for that compilation.

 Returns the result of executing the receiver with anObject as self."

<primitive: 2008>
^ self _primitiveFailed: #_executeInContext: args: { anObject }
%


category: 'Debugging Support'
classmethod: 
_sourceWithErrors: compilerError fromString: aString

 ^ self _sourceWithErrors: compilerError fromString: aString tabSize: 8
%

category: 'Debugging Support'
classmethod: 
_sourceWithErrors: compilerError fromString: aString tabSize: tabSize

"This method returns an instance of aString's class containing the text in a
 string with compiler errors marked, plus the error text for each error.

 The argument compilerError is the result Array from either the 
 Behavior | compileMethod:dictionaries:category:  method or the
 GsNMethod | _recompileWithSource: method.

 The argument aString is the source string which was an input to either the
 Behavior | compileMethod:dictionaries:category: method or the
 GsNMethod | _recompileWithSource: method."

| lineFeed result aStringSize offsets errNumbers thisErr pos
  markerArray errDict errMsgs auxMsgs errsz |

"initialize"
lineFeed := Character lf.
offsets := Array new: (errsz:= compilerError size) .
errNumbers := Array new: errsz  .
errMsgs := Array new: errsz  .
auxMsgs := Array new: errsz  .
aString == nil ifTrue:[ 
  result := String new  .
  result addAll:' "method source not available" '; add: lineFeed .
].

"get an Array of source offsets with errors, and an Array of error numbers"
1 to: errsz   do: [:i | | errNum |
   thisErr := compilerError at: i.
   offsets at: i put: (thisErr at: 2 "source offset").
   errNumbers at: i put: (errNum := thisErr at: 1 "error number") .
   result ~~ nil ifTrue:[
     result addAll:'error '; add: errNum asString; addAll: ', ' 
   ].
   thisErr size >= 3 ifTrue:[ | eMsg |
     errMsgs at: i put: (eMsg := thisErr at: 3"error message String") . 
     (result ~~ nil and:[ eMsg ~~ nil]) ifTrue:[
        result addAll: eMsg asString ; add: $,
     ].
     thisErr size >= 4 ifTrue:[ 
       auxMsgs at: i put: (eMsg := thisErr at: 4 "additional message text").
       (result ~~ nil and:[ eMsg ~~ nil]) ifTrue:[
          result addAll: eMsg asString ; add: $,
       ].
     ].
   ].
   result ~~ nil ifTrue:[ result add: lineFeed ].
].
result ~~ nil ifTrue:[ ^ result ].

aStringSize:= aString size.

"build an Array parallel to the source that contains nil if no error at
 that source position, and an index into offsets if there is an error at
 that source position"
markerArray:= self _buildMarkersFrom: offsets ofSize: aStringSize.

result:= self _buildMarkedSourceFrom: aString
                          sourceSize: aStringSize
                             markers: markerArray tabSize: tabSize .

"add error strings"
errDict := GemStoneError at: System myUserProfile nativeLanguage.
1 to: errNumbers size do: [:i | | msg |
  result add: lineFeed.
  result addAll: i asString.
  result addAll: ': ['.
  pos := errNumbers at: i.
  result addAll: pos asString.
  result addAll: '] '.
  msg := errMsgs at: i .
  msg == nil ifTrue:[
    pos > errDict size
      ifTrue: [ msg := '(unknown error number)']
      ifFalse: [ msg := (errDict at: pos) asString].
    ].
  result addAll: msg .
  (auxMsgs at: i) ~~ nil ifTrue:[ result addAll: (auxMsgs at: i) ].
  ].
result add: lineFeed.

^result
%

category: 'Debugging Support'
classmethod:
_buildMarkedSourceFrom: sourceStrArg sourceSize: aSize markers: markerArray 

^ self _buildMarkedSourceFrom: sourceStrArg sourceSize: aSize markers: markerArray tabSize: 8
%

category: 'Debugging Support'
classmethod:
_buildMarkedSourceFrom: sourceStrArg sourceSize: aSize markers: markerArray tabSize: tabSize

"Given a source string, its size (passed in for efficiency), and a marker
 Array, returns an instance of sourceStr's class containing the marked source."

| lineFeed tab space
  placesMarked     "an Array of markers marked on the current line"
  markerLineIndex "index into the current marker line"
  result          "the result of this method"
  markerLine      "the current marker line"
  aChar           "one Character of the source"
  displayWidth    "the number of positions it takes to display aChar"
  lineSz          
  sourceStr
|

 "initialize"
 lineFeed := Character lf.
 tab:= Character tab.
 space:= Character space.

 placesMarked:= { } .
 markerLineIndex:= 1.
 sourceStr := sourceStrArg .
 sourceStr == nil ifTrue:[ sourceStr := String withAll:' "source not available" '].
 result:= sourceStr class new .
 result addAll: '   ' .
 lineSz := 0 .
 markerLine:= String new .
 markerLine add: $  .
 1 to: aSize do: [:i |
   aChar:= sourceStr at: i.  "fetch a char"
   displayWidth := 1 .
   "Add the char to the result"
   (aChar == tab) ifTrue: [
      displayWidth:= tabSize - (lineSz \\ tabSize).
      displayWidth timesRepeat: [result add: space].
      lineSz := lineSz + displayWidth .
   ] ifFalse: [
      result add: aChar.
      lineSz := lineSz + displayWidth .
      ((i == aSize) and: [aChar ~~ lineFeed]) ifTrue: [
        result add: lineFeed .
      ].
   ].

   ((markerArray at: i) == nil) ifTrue: [ "no marker at this position"
      displayWidth timesRepeat:[ markerLine add: space].
   ] ifFalse: [ "found an error at this position"
      placesMarked add: { markerLineIndex + 1 . markerArray at: i }.
      markerLine add: $^ .
      displayWidth - 1 timesRepeat: [markerLine add: space].
   ].
   markerLineIndex:= markerLineIndex + displayWidth.

   ((aChar == lineFeed) or: [i == aSize]) ifTrue: [ "we are at end of line"
      "add error identifiers to marker line "
      (placesMarked size ~~ 0) ifTrue: [
         self _addMarkerIds: { placesMarked . markerLine . markerLineIndex }.
         result add: $  ; add: $* .
         result add: markerLine.
       ] .
      (i == aSize) ifFalse: [
         result addAll: '   ' .
         lineSz := 0 .
      ].
      markerLine size: 1.
      markerLineIndex:= 1.
      placesMarked size: 0.
   ]
 ].
 ^result
%

category: 'Debugging Support'
classmethod:
_buildMarkersFrom: sourceOffsets ofSize: sizeArg

"Given an Array of source offsets, build an Array of size sizeArg containing
 the index into anArray at the position corresponding to anArray's element.
 The remainder of the Array contains nil.  Negative offsets denote disabled
 breakpoints."

| markerArray anOffset posOffset aSize |
aSize := 1 max: sizeArg .                          "fix bug 14976"
markerArray:= Array new: aSize.
1 to: sourceOffsets size do: [:i |
  anOffset := sourceOffsets at: i .
  anOffset == nil ifFalse:[
    posOffset:= (anOffset abs max: 1) min: aSize.  "limit within range"
    (markerArray at: posOffset) ifNotNil:[ "one retry for fix 45431"
      posOffset := posOffset + 1 min: aSize .
    ].
    (markerArray at: posOffset) ifNil:[
       anOffset < 0 ifTrue:[ markerArray at: posOffset put: i negated ]
		    ifFalse:[ markerArray at: posOffset put: i ]
       ]
    ]
  ].
^markerArray
%

category: 'Debugging Support'
classmethod:
_addMarkerIds: anArray

""

| placesMarked markerLine markerLineSize space addToEnd markPosition
  aStr neededSize subStr |

placesMarked:= anArray at: 1.
markerLine:= anArray at: 2.
space:= Character space.

"have the source marked at each error with ^; now add marker identifier"
addToEnd:= false.
1 to: (placesMarked size) do: [:i |
   markPosition:= (placesMarked at: i) at: 1.
   aStr:= ((placesMarked at: i) at: 2) asString.
   neededSize:= markPosition + aStr size.
   markerLineSize := markerLine size .
   (markerLineSize < neededSize) ifTrue: [
       markerLine size: neededSize.
       markerLineSize + 1 to: neededSize do: [:k |
          markerLine at: k put: space].
       markerLineSize:= neededSize.
   ].

   (addToEnd) ifFalse: [
      subStr:= markerLine copyFrom: markPosition + 1
                                to: (markPosition + aStr size).
      subStr do: [:each | (each == $ ) ifFalse: [addToEnd := true]].
   ].
   (addToEnd) ifTrue: [ 
       markerLine add: aStr.
       (i == placesMarked size) ifFalse: [ markerLine add: ',']
   ] ifFalse: [  | destIdx |
      destIdx := markPosition + 1 .
      markerLine replaceFrom: destIdx to: destIdx + aStr size - 1 with: aStr startingAt: 1 .
   ]
].
(68 - markerLine size) timesRepeat:[ markerLine add: $ ].
(75 - markerLine size) timesRepeat:[ markerLine add: $* ] .
markerLine add: Character lf.
^ true
%

category: 'Instance Creation'
classmethod:
new: anInteger

"Disallowed.  You cannot create new instances of GsNMethod."

self shouldNotImplement: #new:
%

category: 'Instance Creation'
classmethod:
new

"Disallowed.  You cannot create new instances of GsNMethod."

self shouldNotImplement: #new
%

! size:, at:put: , etc   not disallowed ; methods protected by invariance

category: 'Debugging Support'
method:
_buildMarkerArray: allSteps ofSize: aSize

"This method returns a marker Array for the receiver's source code string,
 each element of the result is the source offset of a step point. 

 allSteps == true , show all steps
          == false, show steps where a breakpoint currently exists
          a SmallInteger, show just that step point 

 The result Array is the same size as the source string and 
 contains step numbers at offsets corresponding to the source string."

| srcOffsets |

srcOffsets := self _blockSourceOffsets  .
(allSteps _isSmallInteger ) ifTrue:[  | numSteps stepToDisplay |
  stepToDisplay := allSteps .
  numSteps := srcOffsets size .
  stepToDisplay < 1 ifTrue:[ stepToDisplay := 1 ].
  stepToDisplay > numSteps ifTrue:[ stepToDisplay:= numSteps ].
  1 to: numSteps do:[ :j |
     j == stepToDisplay ifFalse:[ srcOffsets at: j put: nil ].
  ].
] ifFalse:[
  allSteps ifFalse:[ self _setBreakpointsInSourceOffsets: srcOffsets ].
].

^ GsNMethod _buildMarkersFrom: srcOffsets ofSize: aSize
%

category: 'Debugging Support'
method:
_buildIpMarkerArray

"This method builds a marker Array for the receiver's source code string.
 containing IPs of all step points, not including step points in inner blocks.

 The result Array is the same size as the source string and 
 contains IP numbers at offsets corresponding to the source string."

| srcOffsets ipsArr srcSize mrkSize markerArray |

srcOffsets := self _debugInfoAccess: 1 at: -1 .
ipsArr := self _debugInfoAccess:2 at: -1 .
srcSize := self sourceString size .

mrkSize := 1 max: srcSize .                          "fix bug 14976"
markerArray:= Array new: mrkSize .
1 to: srcOffsets size do: [:i | | anOffset anIp posOffset |
  anOffset := srcOffsets at: i .
  anIp := (ipsArr at: i ) .
  posOffset := (anOffset abs max: 1) min: mrkSize.  "limit within range"
  (markerArray at: posOffset) ifNotNil:[  "one retry for fix 45431"
     posOffset := posOffset + 1 min: mrkSize
  ].
  (markerArray at: posOffset) ifNil:[
     markerArray at: posOffset put: anIp 
  ]
].
^ markerArray
%

category: 'Debugging Support'
method:
_sourceAtIp: anIp

"Return the source string with marker for step point closest to
 the specified IP . 
 Assumes that the IP is from a frame that is Not at top of Stack."

| aStep |
aStep := self _previousStepPointForIp: anIp .
^ self _sourceWithSteps: aStep
%
category: 'Debugging Support'
method:
_sourceAtTosIp: anIp

"Return the source string with marker for step point closest to
 the specified IP . 
 Assumes that the IP is from a frame that IS at top of Stack."

| aStep |
aStep := self _nextStepPointForIp: anIp .
^ self _sourceWithSteps: aStep
%

category: 'Debugging Support'
method:
_lineNumberForIp: targetIp

"Returns the line number in the receiver's source string for the
 specifed IP value within the receiver.
 Assumes that the IP is from a frame that is Not at top of Stack."

| stepPoint |
stepPoint := self _previousStepPointForIp: targetIp .
^ self _lineNumberForStep: stepPoint 
%

category: 'Debugging Support'
method:
_lineNumberForTosIp: targetIp

"Returns the line number in the receiver's source string for the
 specifed IP value within the receiver.
 Assumes that the IP is from a frame that IS at top of Stack ."

| stepPoint |
stepPoint := self _nextStepPointForIp: targetIp .
^ self _lineNumberForStep: stepPoint 
%

category: 'Debugging Support'
method:
_sourceWithSteps: allSteps 

  ^ self _sourceWithSteps: allSteps tabSize: 8
%

category: 'Debugging Support'
method:
_sourceWithSteps: allSteps tabSize: tabSize 
 
^ self _sourceWithSteps: allSteps tabSize: tabSize window: 0
%

category: 'Debugging Support'
method:
_sourceWithSteps: allSteps tabSize: tabSize window: winSize
"This method returns the source string with intermixed control information
 indicating where step points are.

 allSteps == true , show all steps
          == false, show steps where a breakpoint currently exists
          a SmallInteger, show just that step point

 winSize == 0 means list entire method ,
         non zero and  allSteps is a SmallInteger causes specified
         number of lines to be listed, centered on specified step point,
         plus the fileName/lineNumber comment line if present.
 "
| markerArray "A parallel Array to the source string.  It is filled with nils
               except for locations pointed to by the sourceOffsets Array."
  srcSize       "the size of the source"
  srcStr fullResult res |

srcStr := winSize == 0 ifTrue:[ self _sourceStringWithFileName ] 
                       ifFalse:[ self sourceString ].
srcSize:= srcStr size.
markerArray := self _buildMarkerArray: allSteps ofSize: srcSize.
fullResult := GsNMethod _buildMarkedSourceFrom: srcStr
                             sourceSize: srcSize markers: markerArray tabSize: tabSize .
res := fullResult .
(winSize > 0 and:[ allSteps _isSmallInteger]) ifTrue:[
   | targLine lineCnt lines lf halfWin |
  targLine := 0 .   halfWin := winSize // 2 .
  lf := Character lf .
  lineCnt := 0 .
  1 to: srcSize do:[ :n |
    ((markerArray at: n) ~~ nil and:[ targLine == 0]) ifTrue:[ targLine := lineCnt + 1 ].
    (srcStr at: n)  == lf ifTrue:[  lineCnt := lineCnt + 1 ].
  ].
  lines := fullResult subStrings: lf .
  lineCnt := lines size .
  (lineCnt > winSize and:[ lineCnt > 10]) ifTrue:[
    | start end |
    res := fullResult class new .
    start := (targLine - halfWin) max: 1 .
    end := (targLine + halfWin) min: lineCnt .
    start > 2 ifTrue:[
      1 to: 2 do:[:k | res addAll: (lines at: k); add: lf  ].
      res addAll:'...'; add: lf .
    ].
    start to: end do:[:k | res addAll: (lines at: k) ; add: lf ].
    end < (lineCnt - 2) ifTrue:[
      res addAll:'...'; add: lf .
    ].
  ]  
].
winSize ~~ 0 ifTrue:[ 
  res add: self _sourceFileLineComment .
]. 
^ res
%

category: 'Debugging Support'
method:
_sourceWithStepIps

^ self _sourceWithStepIps: 8
%
category: 'Debugging Support'
method:
_lastBlockLiteralOffset
  "Returns offset of last block literal.
   If there are no block literals, result will be offset of the last literal.
   If there are no literals , result will be offset of last instruction word" 
  ^ self _debugInfoAccess: 9 at: -1
%

category: 'Debugging Support'
method:
_sourceWithStepIps: tabSize

"This method returns the source string with intermixed control information
 indicating IP values of each step point.  

 Result shows the step points for the receiver only, not any inner blocks.
 "

| markerArray "A parallel Array to the source string.  It is filled with nils
               except for locations pointed to by the sourceOffsets Array."
  aSize        "the size of the source"
  srcStr result |

srcStr := self sourceString .
aSize:= srcStr size.
markerArray := self _buildIpMarkerArray .
result := GsNMethod _buildMarkedSourceFrom: srcStr
                             sourceSize: aSize
                                markers: markerArray tabSize: tabSize .
^ result
%

category: 'Debugging Support'
method:
_debugInfoAccess: kind at: ofs

"Return debugging information per kind and ofs for the  receiver.
 Does not search any methods for blockLiterals of the receiver.

 Signals a Notification and returns nil if the requested item is 
 not available because the receiver needs recompilation.

  kind = 1  sourceOffsets (ofs > 0 , source Offset for specified step point
                           ofs == -1, all sourceOffsets
                  for ofs > 0 , if ofs beyond last step point, returns
                  SmallInteger maximumValue  )

  kind = 2  all ipSteps  (ofs == -1)
  kind = 3  offset into ipSteps for instruction PRECEEDING the
                absolute instruction offset specified by ofs
  kind = 4  offset into ipSteps for instruction FOLLOWING the
                absolute instruction offset specified by ofs
  kind = 5  _sourceOffsetOfFirstSendOf: , ofs is aSymbol

  kind = 6  _ipForStepPoint:   ofs is a stepPoint, result is an instr offset,
		or nil if stepPoint does not exist in the receiver.

  kind = 7  _selectorPool   result is a IdentitySet of all selectors sent
                         (ofs == -1)
  kind = 8  argAndTmpOffsets , zero based word offsets  (ofs == -1)
            for the args,  offsets are positive with respect to FP
            for the temps,
              negative offsets denote offsets with respect to FP
              positive offsets encode the fields
                 16rFFF00 zero-based offset wrt varying instVar 0 of VC, 
                 16r000FF number of VC.parent refs to follow to the defining VC

  kind = 9  _lastBlockLiteralOffset   (ofs == -1)
  
  kind = 10 _sourceOffsetsOfSends  (ofs not used)
            returns Array of pairs { anOffset , aSymbol, ... anOffset, aSymbol}

  kind = 11  returns Array of pragma info or nil.  (ofs == -1) 
             returns the canonical empty Array if receiver is method for a block
"

<primitive: 672>
self methodCompilerVersion < 4 ifTrue:[ 
  Notification signal: (self _classAndSelectorNameWidth: 0) , 
		      ' method from Gs64 v3.2.x needs recompile' .
  ^ nil .
].
self _primitiveFailed: #_debugInfoAccess:at: args: { kind . ofs }
%


category: 'Debugging Support'
method:
_allDebugInfo: kind

| res blks |
res := self _debugInfoAccess: kind  at: -1 .
blks := self blockLiterals .
blks do:[ :aBlk | | blkRes |
    blkRes := aBlk method _debugInfoAccess: kind  at: -1 .
    blkRes size ~~ 0 ifTrue:[ res addAll: blkRes ].
].
^ res
%

category: 'Debugging Support'
method:
_allDebugInfoWithMeths: kind

| res arr blks |
res := { }  .
arr := self _debugInfoAccess: kind  at: -1 .
1 to: arr size do:[:j | res addLast: { self . (arr at: j) } ].
blks := self blockLiterals .
blks do:[ :aBlk | | blkRes aMeth |
    aMeth := aBlk method .
    blkRes := aMeth _debugInfoAccess: kind  at: -1 .
    1 to: blkRes size do:[:j | res addLast: { aMeth . (blkRes at: j) }].
].
^ res
%

category: 'Debugging Support'
method:
_sourceOffsets

"Returns an Array of SmallIntegers describing all the step points of
 the receiver and any blocks.
 Each element of the result is the source offset of a step point.
 The result is not sorted by source offset."

| res |
res := self _allDebugInfo: 1 .
res size == 0 ifTrue:[ ^ #( 1) "a primitive method" ].
^ res
%

category: 'Debugging Support'
method:
_blockSourceOffsets
  "If receiver is a home method, return an array containing source offsets
   of all step point in receiver and its blocks,
   otherwise return an Array of size equal all source offsets in the home method,
   and containing source offsets of step points within the receiver,
   and nils for all other step points of the home method and its blocks."
| homeMth res blks |
homeMth := self homeMethod .
homeMth == self ifTrue:[ ^ self _sourceOffsets ].
res := homeMth _debugInfoAccess: 1  at: -1 .
res := Array new:(res size) . "convert to nils"
blks := homeMth blockLiterals .
blks do:[ :aBlk | | blkMth blkRes |
    blkMth := aBlk method .
    blkRes := blkMth _debugInfoAccess: 1  at: -1 .
    blkMth == self ifTrue:[ res addAll: blkRes "just these step points"]
        ifFalse:[ res size: (res size + blkRes size) "append nils"].
].
^ res
%

category: 'Debugging Support'
method:
_sourceOffsetsOfSends

"Returns an Array,  { anOffset, aSelectorSymbol, ... anOffset, aSelectorSymbol}
 containing source offsets and selectors of all of the sends in the receiver.
 The results are not sorted by source offset."

^ self _allDebugInfo: 10
%


category: 'Debugging Support'
method:
_sourceOffsetsAt: aStepPoint

"Returns the source offset for the step point with number aStepPoint.
 Returns beginning or ending source offset for a step point out of range.

 Step points are expressed as source offsets (Gs64 v3.0 and above). "
 | ofs max stp |
 aStepPoint < 1 ifTrue:[ ^ 1 ].
 stp := aStepPoint .
 self isMethodForBlock ifTrue:[ stp := stp - self _stepPointOffset ].
 ofs := self _debugInfoAccess: 1 at: stp .
 ofs ifNil:[ ^ 1 ].
 max := self sourceString size .
 ofs > max ifTrue:[  ofs := max ].
 ^ ofs 
%

! fix 41280
category: 'Debugging Support'
method:
_lineNumberForStep: aStepPoint

"Returns the line number in the receiver's source string for the
 step point with number aStepPoint.
 Returns 1 if aStepPoint is out of range."

  | sourceOffset |
  sourceOffset := self _sourceOffsetsAt: aStepPoint .
  sourceOffset == nil ifTrue:[ ^ 1 ].

  ^ self _lineNumberForOffset: sourceOffset inString: self sourceString
%

category: 'Private'
method:
_lineNumberForOffset: sourceOffset inString: srcStr
  " find the first end-of-line which is at or after the 
    sourceOffset of the step point."

  | offset lf cr lineNumber |
  lf := Character withValue: 10 .
  cr := Character withValue: 13 .
  offset := 1 .
  lineNumber := 0 .

  (srcStr indexOf: cr startingAt: offset) == 0 ifTrue:[
    "no CR's in source"
    [ offset <= sourceOffset ] whileTrue:[
      lineNumber := lineNumber + 1 .
      offset := srcStr indexOf: lf startingAt: offset .
      offset == 0 ifTrue:[ ^ lineNumber ].
      offset := offset + 1 .
    ].
  ] ifFalse:[
    "at least one CR present in source"
    [ offset <= sourceOffset ] whileTrue:[ | lfOfs crOfs |
      lineNumber := lineNumber + 1 .
      lfOfs := srcStr indexOf: lf startingAt: offset .
      crOfs := srcStr indexOf: cr startingAt: offset .
      crOfs ~~ 0 ifTrue:[
        offset := lfOfs ~~ 0 ifTrue:[ crOfs min: lfOfs ] ifFalse:[ crOfs ]
      ] ifFalse:[
        offset := lfOfs ~~ 0 ifTrue:[ lfOfs ] ifFalse:[ 0 ].
      ].
      offset == 0 ifTrue:[ ^ lineNumber ].
      (crOfs ~~ 0 and:[ lfOfs == (crOfs + 1) ]) ifTrue:[
        offset := offset + 2 .  "skip CR LF pair"
      ] ifFalse:[
        offset := offset + 1 .  "skip CR or LF "
      ]
    ].
  ].
  ^ lineNumber .
%

category: 'Accessing'
method:
sourceString

"Returns a CharacterCollection that contains the source code of the receiver."

^ debugInfo at: DebugInfo_source_offset
%

method:
_debugInfo
 "Returns an Array with varying elements.
  See $GEMSTONE/upgrade/opalcls.ht for details . "
     
  ^ debugInfo
%

method:
_fileAndLine
  "Return an Array, { fileName . lineNumber } , 
   or nil if method  does not have that information.
   Smalltalk methods return nil."
 
  ^ nil
%
   

method:
_sourceStringWithFileName
 "Return source string with file name and starting line number appended.
  Ruby methods usually have a file and line; Smalltalk methods do not.
  Used by topaz."

  | sStr |
  sStr := self sourceString .
  ^ [ | res flStr |
      res := sStr . 
      (flStr := self _sourceFileLineComment) size > 0 ifTrue:[ | lf |
	lf := Character lf .
	res := res copy .
	(res at: res size) == lf ifFalse:[ res add: lf ].
	res add: flStr . 
      ]. 
      res
    ] onException: Error do:[:ex | sStr ]
%

method: 
_sourceStringWithLineNumbers
 "Return source string with lines prefixed with line numbers.
  Used by topaz."
 | src res n nStr idx lf prev |
 src := self sourceString .
 res := src class new .
 lf := Character lf .
 n := 1 .
 [ nStr := n asString .
   (3 - nStr size) timesRepeat:[ res add: $  ].
   res add: nStr .
   2 timesRepeat:[ res add: $  ].
   n := n + 1 .
   idx ifNil:[ prev := 1 ] ifNotNil:[ prev := idx + 1 ].
   idx := src indexOf: lf startingAt: prev .
   idx == 0 ifTrue:[ res add: (src copyFrom: prev to: src size) ]
         ifFalse:[ res add: (src copyFrom: prev to: idx) ].
   idx == src size ifTrue:[ idx := 0 ].
   idx == 0
 ] untilTrue .
 ^ res
%

method:
_sourceFileLineComment
  "Smalltalk methods are not expected to not have file/line info"
  ^ ''
%

category: 'Accessing'
method:
isMethodForBlock

"Returns true if the receiver is the method for an ExecBloc."

^ ((iFields1 bitShift: 0 - IsMethodForBlock_shift) bitAnd: 1) == 1 
%

! category: 'Accessing'
! method:
! rubyPrivate
! 
! "Returns true if the receiver is a ruby method with private attribute"
! 
! ^ ((iFields1 bitShift: 0 - RubyPrivate_shift) bitAnd: 1) == 1
! %

! method:
! rubyProtected
! 
! "Returns true if the receiver is a ruby method with protected attribute"
! 
! ^ ((iFields1 bitShift: 0 - RubyProtected_shift) bitAnd: 1) == 1
! %

! method:
! rubyMethodProtection
! 
! "return 0==no protection, 1==protected, 2==private"
! 
! ^ (iFields1 bitShift: 0 - RubyProtected_shift) bitAnd: 3
! %

! method:
! isRubyBridgeMethod
!   ^ ((iFields1 bitShift: 0 - IsRubyBridgeMeth_shift) bitAnd: 1) == 1
! %

! method:
! rubyOptArgsBits
!   "optArgsBits is either zero , or the first SmallInteger
!    in the literals area, describing which of the first 61
!    args have explicitly coded default value initializers.
!    First arg is described by  bitAt:0 in the result.  "
!   | numWords |
!   numWords := (iFields1 bitShift: 0 - NumArgDescrLits_shift) bitAnd: NumArgDescrLits_mask .
!   numWords == 0 ifTrue:[ ^ 0 ]
!             ifFalse:[ ^ self at: self literalsOffset ]
! %

category: 'Accessing'
method:
_sourceStringForBlock

"Return the portion of the home method's source for this block"

| src word firstOfs lastOfs isLineNum |

self isMethodForBlock ifFalse:[ ^ nil ].
src := debugInfo at: DebugInfo_source_offset .
word := debugInfo at: DebugInfo_FirstLastSrcOffset_offset .
isLineNum := false .
word < 0 ifTrue:[ isLineNum := true.  word := 0 - word ].
firstOfs := word bitAnd: 16r3fffffff .
lastOfs := word bitShift: -30 .
isLineNum ifTrue:[ | lines res lf sz | "ruby line numbers"
  lines := src subStrings: (lf := Character lf) .
  res := String new .
  firstOfs to: (lastOfs min: lines size) do:[:n | res addAll: (lines at: n); add: lf ].
  res addAll:'...'; add: lf .
  lines := self homeMethod sourceString subStrings: lf .
  sz := lines size .
  (lines at: sz) size == 0 ifTrue:[ sz := sz - 1 ].
  res add: (lines at: sz) ; add: lf . "append comment 'starts at line N of file...' "
  ^ res
] ifFalse:[
  ^ src copyFrom:firstOfs to: lastOfs .
]
%

method:
_sourceStartForBlock

"Return the 1-based offset in the home methods source of
 the start of the block."
| word |
self isMethodForBlock ifFalse:[ ^ 1].
word := debugInfo at: DebugInfo_FirstLastSrcOffset_offset .
word < 0 ifTrue:[ ^ 1 "ruby line numbers no longer expected"].

^ word bitAnd: 16r3fffffff "firstOfs"
%

method:
_lineDeltaForBlock

"Return the line number of the start of the block relative
 to the start of the home method source."

| ofs src |
ofs := self _sourceStartForBlock .
ofs <= 1 ifTrue:[ ^ 0 ].
src := debugInfo at: DebugInfo_source_offset .

^ (self _lineNumberForOffset: ofs inString: src) - 1
%


category: 'Accessing'
method:
argsAndTemps

"Returns an Array of Symbols which are the names of arguments and
 temporaries for this method,  not including inner blocks"

| offset numArgsTmps |
numArgsTmps := self _numArgsTempsCblkargs .
numArgsTmps < 1 ifTrue:[ ^  #()  ].

offset := 
self isMethodForBlock ifTrue:[ DEBUGINFO_BLK_HDRSIZE ]
   ifFalse:[ self _hasPragmaInfo ifTrue:[ DEBUGINFO_BLK_HDRSIZE]
       ifFalse:[ self _debugInfoHasFileAndLine ~~ 0 ifTrue:[ DEBUGINFO_RubyMTH_HDRSIZE ]
          ifFalse:[ DEBUGINFO_MTH_HDRSIZE ]]].
offset := offset + 1 .
^ debugInfo copyFrom: offset to: (offset + numArgsTmps - 1)
%

category: 'Accessing'
method:
_argsAndTempsOffsets

"Returns an Array of SmallIntegers .
 Each SmallInteger has bits
     16rFF , lexical level , number of VC.parent refs to follow to the defining VC
     high order bits are a signed   offset*256 .
 The Array represents
   numArgs positive zero-based offsets with respect to FP ( level == 0)
   numTemps offsets ,
      negative offsets are zero-based with respect to FP (level == 0)
      positive offsets are zero based wrt.  instVar 0 of a VC
        the bits 16r000FF are number of VC.parent refs to follow to VC
   Does not include any info for  inner blocks"

 ^ self _debugInfoAccess: 8 at: -1
%


category: 'Accessing'
method:
_ipSteps

"Returns an Array containing the step points for the portable code for the
 method."

| res |
res := self _allDebugInfo: 2  .
res size == 0 ifTrue:[ ^ #( 1) "a primitive method" ].
^ res
%

category: 'Debugging Support'
method:
_numArgsTempsCblkargs
"return (number of args + num temps) + copyingBlockIdxSize "
| info1 info2 |
info1 := debugInfo at: DebugInfo_fields1_offset  .
info2 := debugInfo at: DebugInfo_fields2_offset  .
^ ((info1 bitShift: 0 - NumArgsPlusTemps_shift) bitAnd: NumArgsPlusTemps_mask) +
  ((info2 bitShift: 0 - CopyingBlockIdxSize_shift) bitAnd: CopyingBlockIdxSize_mask) 
%

category: 'Debugging Support'
method:
_numCopyingBlockArgs

^ 0
" maglev only:
^ ((debugInfo at: DebugInfo_fields2_offset) bitShift: 0 - CopyingBlockIdxSize_shift) 
     bitAnd: CopyingBlockIdxSize_mask
"
%

method:
_debugInfoHasFileAndLine
  "result is a SmallInteger 0,1, or 2 per opalcls.ht"
| info1 |
info1 := debugInfo at: DebugInfo_fields1_offset .
^ ((info1 bitShift: 0 - DbgI1fileInfo_shift ) bitAnd: DbgI1fileInfo_mask) 
%
method:
_hasPragmaInfo
  "Returns true or false.
   Result always false for methods compiled before Gs64 v3.5"

^ (((debugInfo at: DebugInfo_fields2_offset) 
      bitShift: 0 - debugI2_hasPragmas_shift) bitAnd: 1) ~~ 0
%

category: 'Debugging Support'
method:
_numArgs

"Returns the number of arguments expected by the method."

^ iFields1 bitAnd: NArgs_mask
%

category: 'Debugging Support'
method:
_numIpSteps

^ ((debugInfo at: DebugInfo_fields1_offset) bitShift: 0 - NumIpSteps_shift) 
 	bitAnd: NumIpSteps_mask 
%

category: 'Debugging Support'
method:
_numBlockLiterals

^ (debugInfo at: DebugInfo_Fields2_offset) bitAnd: NumBlockLiterals_mask 
%

category: 'Debugging Support'
method: 
_lineNumberBias

| v |
v :=  ((debugInfo at: DebugInfo_Fields2_offset) bitShift: 0 - LineNumberBias_shift)
     bitAnd: LineNumberBias_mask .
(v bitAnd:( 1 bitShift: LineNumberBias_bits - 1)) ~~ 0 ifTrue:[
  v := 0 - (v bitInvert + 1)  "sign extend"
].
^ v
%

category: 'Debugging Support'
method:
_numSourceOffsets

^ self _numIpSteps
%


category: 'Debugging Support'
method:
_breakPointKind: anIp

"This method infers the kind of action associated with a given bytecode."

self error:'GsNMethod>>_breakPointKind:  not implemented'.
%

category: 'Debugging Support'
method:
_setBreakpointsInSourceOffsets: sourceOffsets

"Given all the source offsets for the receiver, replace with nil those
 that do not correspond to a breakpoint.  Negate those which correspond
 to a disabled breakpoint."

| allBreaks |

allBreaks := self _allBreakpointsSourceOffsets .
allBreaks == nil ifTrue:[ ^ sourceOffsets size: 0 ].

1 to: sourceOffsets size do:[:j | | aStepPoint  |
   aStepPoint := sourceOffsets at: j .  "aStepPoint is a source offset"
   (allBreaks indexOf: aStepPoint) = 0 ifTrue:[ | negStepPoint |
     negStepPoint := aStepPoint negated .
     (allBreaks indexOf: negStepPoint) = 0 ifTrue:[
       sourceOffsets at: j put: nil
     ] ifFalse:[
       sourceOffsets at: j put: negStepPoint 
     ].
   ]
].
^ self
%

category: 'Debugging Support'
method:
_allBreakpointsSourceOffsets

| brksArr result |
brksArr := self _allBreakpoints .         "includes methods of inner blocks"
brksArr == nil ifTrue:[ ^ nil ].
result := { }  .
1 to: brksArr size by: 3 do:[:k | | meth allIps stepPoint anIp |
  meth := brksArr at: k + 1 .
  anIp := brksArr at: k + 2 .
  allIps := meth _debugInfoAccess: 2 at: -1 .    
  stepPoint := allIps indexOf: anIp . 
  stepPoint ~~ 0 ifTrue:[  "convert to source offset"
    result add:( meth _debugInfoAccess: 1 at: stepPoint ) 
  ] ifFalse:[
    result add: 1 . "unexpected path"
  ].
].
^ result
%

category: 'Debugging Support'
method:
_firstSourceOffset

 | arr |
 arr := self _debugInfoAccess: 1 at: -1.
 arr size > 0 ifTrue:[ ^ arr at: 1 ].
 ^ 1
%

! fix 46940
category: 'Debugging Support'
method:
_stepPointOffset
  "If receiver is method for a block, return the value to add to a steppoint
   for the block to obtain the steppoint as viewed from the home method,
   otherwise return zero."

  self isMethodForBlock ifTrue:[ | offsets |
    offsets := self _blockSourceOffsets .
    1 to: offsets size do:[:n | (offsets at: n) ifNotNil:[ ^ n - 1]].
    "Error signal:'step point not found in _stepPointOffset'." "for debugging"
  ].
  ^ 0
%

category: 'Debugging Support'
method:
_nextStepPointForIp: anIp 

"Returns the step point which represents
 the step point at or just after the given instruction pointer offset.  If
 anIp is after the last step point, returns an integer one greater than
 the last legal step point.

 The anIp argument is an absolute instruction offset ."
 ^ ( self _debugInfoAccess: 4  at: anIp ) + self _stepPointOffset .
%

category: 'Debugging Support'
method:
_previousStepPointForIp: anIp 

"Returns the step point which represents
 the step point preceding the given instruction pointer offset.

 The anIp argument is an absolute instruction offset .
 The argument isQuick is ignored."

^ ( self _debugInfoAccess: 3  at: anIp ) + self _stepPointOffset .
%

category: 'Debugging Support'
method:
_stepPointForIp: ipOffset level: aLevel useNext: nextBool

"Result is step point for specified instruction offset and level in a stack
 ipOffset is an absolute instruction offset ."

aLevel = 1 ifTrue:[  "top of stack"
   ^ self _nextStepPointForIp: ipOffset 
 ] ifFalse:[
   nextBool ifTrue:[
     ^ self _nextStepPointForIp: ipOffset 
   ] ifFalse:[
     ^ self _previousStepPointForIp: ipOffset 
   ].
 ]
%

! _sourceOffsetOfSendAt deleted

category: 'Reporting'
method:
_sourceOffsetOfFirstSendOf: aSymbol

"Returns the source offset of the step point for the first send in the receiver
 that sends aSymbol.  If the receiver is not a sender of aSymbol, or if aSymbol
 is not a Symbol, returns nil."
 
| ofs blks |
ofs := self _debugInfoAccess: 5  at: aSymbol .         
ofs ~~ nil ifTrue:[ ^ ofs ].
blks := self blockLiterals .
blks do:[ :aBlk | | blkOfs | 
    blkOfs := aBlk method _debugInfoAccess: 5  at: aSymbol .
    blkOfs ~~ nil ifTrue:[ ^ blkOfs ].
].
^ nil .
%

category: 'Debugging Support'
method:
_meth_ip_ForStepPoint: aStepPoint

"Returns an Array , { aMethod . anInstructionOffset }
 for the specified step point . 
 Instruction offsets are absolute byte offsets relative to start of the method.
 Returns nil if aStepPoint is not a legal step point for the receiver"

| allIps |
allIps := self _allDebugInfoWithMeths: 2 .
(aStepPoint >= 1 and:[ aStepPoint <= allIps size]) ifTrue:[
  ^ allIps at: aStepPoint 
].
^ nil
%
category: 'Debugging Support'
method:
_stepPointForMeth: aMethod ip: ipOffset
"Given a aMethod which may be self or method for an inner block of self,
 and anIp , an ipOffset within aMethod (possibly negated),
 Return the stepPointer number within self, or nil ." 
| absIp allIps  |
absIp := ipOffset abs .
allIps := self _allDebugInfoWithMeths: 2 .
1 to: allIps size do:[:j | | anIpArr|
  anIpArr :=  allIps at: j .
  ((anIpArr at: 1) == aMethod and:[ (anIpArr at: 2) = absIp]) ifTrue:[
     ipOffset > 0 ifTrue:[ ^ j ] ifFalse:[ ^ j negated ].
  ].
].
^ nil
%

category: 'Debugging Support'
method:
setBreakAtStepPoint: aStepPoint

"Set method breakpoint at specified step point.
 Returns nil if aStepPoint is not legal otherwise returns receiver."

^ self _breakOperation: 0 forStepPoint: aStepPoint 
%

category: 'Debugging Support'
method:
_breakOperation: anInt forStepPoint: aStepPoint

"Returns nil if aStepPoint is not legal otherwise returns receiver.
 anInt is an opcode as defined for second arg to 
 _setBreakAtIp:operation:frame:process:
"

| info |
info := self _meth_ip_ForStepPoint: aStepPoint abs .
info == nil ifTrue:[ ^ nil ].
(info at:1) _setBreakAtIp: (info at: 2)  operation: anInt frame: nil process: nil .
%

category: 'Private'
classmethod:
_breakpoint: aNumber perform: aSelector

"For the breakpoint listed with number aNumber 
 in the result of GsNMethod (C) | _breakReport:,  
 perform  aSelector .  
 Return true if the breakpoint found, false otherwise."

| reportArray |
reportArray := (self _breakReport: true ) at: 2 .

1 to: reportArray size do:[:j| | aBrk |
  aBrk := reportArray at: j .
  (aBrk at:1) = aNumber ifTrue:[  | aMethod stepPoint |
    aMethod := aBrk at: 5 .
    stepPoint := aBrk at: 4 .
    aMethod perform: aSelector with: stepPoint .
    ^ true
  ].
].
^ false
%

category: 'Private'
classmethod:
_allBreakpointsPerform: aSelector

"For all breakpoint in the result of GsNMethod (C) | _breakReport:,  
 perform  aSelector .  "

| reportArray |
reportArray := (self _breakReport: true ) at: 2 .

1 to: reportArray size do:[:j| | aBrk aMethod stepPoint |
  aBrk := reportArray at: j .
  aMethod := aBrk at: 5 .
  stepPoint := aBrk at: 4 .
  aMethod perform: aSelector with: stepPoint .
].
%

category: 'Debugging Support'
classmethod:
_deleteBreakNumber: aNumber 

"Delete the breakpoint listed with number aNumber in the result of
 GsNMethod (C) | _breakReport:.
 Return true if breakpoint deleted, false if not found.
 Used by Topaz"

^ self _breakpoint: aNumber perform: #clearBreakAtStepPoint:
%

category: 'Debugging Support'
classmethod:
_disableBreakNumber: aNumber 

"disable the breakpoint listed with number aNumber in the result of
 GsNMethod (C) | _breakReport: . 
 Return true if breakpoint disabled, false if not found.
 Used by Topaz"

^ self _breakpoint: aNumber perform: #disableBreakAtStepPoint:
%

category: 'Debugging Support'
classmethod:
_enableBreakNumber: aNumber 

"Enable the breakpoint listed with number aNumber in the result of
 GsNMethod (C) | _breakReport: . 
 Return true if breakpoint enabled, false if not found.
 Used by Topaz"

^ self _breakpoint: aNumber perform: #setBreakAtStepPoint:
%

category: 'Debugging Support'
classmethod:
_enableAllBreaks

"Enable all breakpoints listed in the result of GsNMethod (C) | _breakReport:."

self _allBreakpointsPerform: #setBreakAtStepPoint: 
%

category: 'Debugging Support'
classmethod:
_disableAllBreaks

"Disable all breakpoints listed in the result of GsNMethod (C) | _breakReport:."

self _allBreakpointsPerform: #disableBreakAtStepPoint:
%

category: 'Debugging Support'
classmethod:
_deleteAllBreaks

"Delete all breakpoints listed in the result of GsNMethod (C) | _breakReport:."

self _allBreakpointsPerform: #clearBreakAtStepPoint:
%

category: 'Debugging Support'
classmethod:
clearBreakInClass: aClass selector: aSelector stepPoint: aStepPoint

"Clear the breakpoint at aStepPoint in method aSelector of class aClass."

(aClass compiledMethodAt: aSelector) clearBreakAtStepPoint: aStepPoint
%

category: 'Debugging Support'
classmethod:
enableBreakInClass: aClass selector: aSelector stepPoint: aStepPoint

"Set or reenable the breakpoint previously set at aStepPoint in method
 aSelector of class aClass."

(aClass compiledMethodAt: aSelector) setBreakAtStepPoint: aStepPoint
%

category: 'Debugging Support'
classmethod:
disableBreakInClass: aClass selector: aSelector stepPoint: aStepPoint

"Disable the breakpoint previously set at aStepPoint in method aSelector of
 class aClass."

(aClass compiledMethodAt: aSelector) disableBreakAtStepPoint: aStepPoint 
%

category: 'Debugging Support'
method:
clearBreakAtStepPoint: aStepPoint

"Clear method breakpoint at specified step point. 
 Returns nil if aStepPoint is not legal otherwise returns receiver."

self _breakOperation: 2 forStepPoint: aStepPoint 
%

category: 'Debugging Support'
method:
disableBreakAtStepPoint: aStepPoint

"Disable method breakpoint at specified step point. 
 Returns nil if aStepPoint is not legal otherwise returns receiver."

self _breakOperation: 4 forStepPoint: aStepPoint 
%

category: 'Debugging Support'
method: 
_allBreaksOp: anInt frame: fpOffset process: aGsProcess

| blks |
self _setBreakAtIp: -1 operation: anInt frame: fpOffset process: aGsProcess .
blks := self blockLiterals .
"Don't use do:  here,  we might be setting single step breakpoints
  within a do: implementation..."
1 to: blks size do:[ :j | 
    (blks at: j) method _setBreakAtIp: -1 
	operation: anInt frame: fpOffset process: aGsProcess
]. 
%

category: 'Debugging Support'
method:
clearAllBreaks

"Clear all method breakpoints in the receiver."

self _allBreaksOp: 2 frame: nil process: nil 
%

category: 'Debugging Support'
classmethod:
clearAllBreaks

"Clear all method breakpoints that have been set in any methods."

self _setBreakAtIp: -1 operation: 2 frame: nil process: nil 
%

classmethod:
_clearAllStepBreaks

"Legacy code, no longer used.
 Clear all single step breakpoints from all loaded methods."
self _setBreakAtIp: -1 operation: 3 frame: nil process: nil
%

category: 'Debugging Support'
method:
disableAllBreaks

"Disable all method breakpoints in the receiver."


self _allBreaksOp: 4 frame: nil process: nil 
%

! _sendCount:   not implemented yet

! size: now allowed.

category: 'Accessing'
method:
_nArgs

"Returns the number of arguments expected by the method."

^ iFields1 bitAnd: NArgs_mask
%

! _selector deleted

category: 'Accessing'
method:
_sourceString

"Returns sourceString."

^  self sourceString
%

category: 'Clustering'
method:
_clusterDepthFirst

"This method clusters the receiver, its bytecodes, its selector pool, and its
 selector in depth-first order.  Returns true if the receiver has already been
 clustered during the current transaction; returns false otherwise."

| savedBucket systm nonBlkLits |

self cluster ifTrue: [ ^ true ] .
systm := System .
savedBucket := systm clusterBucket.
systm clusterBucket: 5.  "kernel classes description bucket"
debugInfo cluster.   "sourceString part of debugInfo now"
self sourceString cluster .
systm clusterBucket: savedBucket.
nonBlkLits := self _literals .
1 to: nonBlkLits size do:[:k| | aLit |
  aLit := nonBlkLits at: k .
  aLit _isSymbol ifFalse:[
    (aLit class ~~ SymbolAssociation and:[ systm canWrite: aLit ]) ifTrue:[ 
      aLit clusterDepthFirst 
    ].
  ].
].
^ false
%

category: 'Clustering'
method:
clusterDepthFirst

"This method clusters the receiver, its bytecodes, its selector pool, and its
 selector in depth-first order.  Returns true if the receiver has already been
 clustered during the current transaction; returns false otherwise."

self _clusterDepthFirst ifFalse:[ | blks |
  blks := self blockLiterals .
  blks do:[ :aBlk | 
      aBlk clusterDepthFirst
  ]. 
  ^ false
].
^ true
%

category: 'Copying'
method:
copy

"Disallowed.  You may not create new instances of GsNMethod."

self shouldNotImplement: #copy
%


category: 'Private for Class Modification'
method:
_copyForRecompilation

self _validatePrivilege ifTrue:[
  ^ super copy .
].
^ nil
%

category: 'Private for Class Modification'
method:
_debugInfo: anArray

self _validatePrivilege ifTrue:[
  debugInfo := anArray 
].
%

category: 'Private for Class Modification'
method:
_inClass: aClass

"for use only by _copyToForceRecompilation"

self _validatePrivilege ifTrue:[
  inClass := aClass 
]
%

! added send of immediateInvariant with fix 32811 
category: 'Private for Class Modification'
method:
_copyToForceRecompilation

"Returns a copy of the receiver, with bytecodes that will force
 recompilation of the receiver prior to subsequent execution.

 Cannot be used to recompile the method for an individual block,
 you must recompile the home method for the block.
"

| newMethod dbgInfo |
self _validatePrivilege ifFalse:[ ^ nil ].
self isMethodForBlock ifTrue:[
  ^ self error:'cannot recompile method for a block, must recompile the home method.'
].
newMethod := (GsNMethod compiledMethodAt: #_methodNeedingRecompilation)
                _copyForRecompilation .
dbgInfo := { 0 . self sourceString . 0 } .
dbgInfo immediateInvariant .
newMethod _debugInfo: dbgInfo .
newMethod _inClass: inClass .
newMethod immediateInvariant . "required to pass assertions in object manager"
^ newMethod
%

category: 'Error Handling'
method:
_methodNeedingRecompilation

"Generate an error for a method that needs to be recompiled.

 During schema modification, this method is copied, 
 the sourceString in the copy replaced with the sourceString
 of a method needing recompilation, and the copy then installed
 in the modified class' method dictionary as the value for the
 selector needing recompilation.  See also _copyToForceRecompilation . "

| srcPrefix prefixSize thisMeth srcStr |
thisMeth := GsProcess _methodInFrameContents:(GsProcess _frameContentsAt:1).
srcStr := thisMeth sourceString .
prefixSize := srcStr size min: 40 .
srcPrefix := srcStr copyFrom:1 to: prefixSize .
srcPrefix addAll: '...' .
self _error: #rtErrUncompiledMethod args:{ srcPrefix . thisMeth inClass }. 
self _uncontinuableError
%

category: 'Accessing'
method:
_stackBase

"Private."

self error:'GsNMethod>>_stackBase  not implemented'.
%

category: 'Accessing'
method:
_literals

"Returns an Array containing the literal pool of the receiver ,  
 excluding inner blocks, and including the selector pool."

| litOfs res last |
res := { }  .
(litOfs := self literalsOffset) ~~ 0 ifTrue:[
  last := self _lastBlockLiteralOffset - self _numBlockLiterals .
  litOfs to: last do: [:k || aLit |
    aLit := self at: k .
    res add: aLit .
  ].
].
res addAll: self _selectorPool .
^ res 
%

category: 'Accessing'
method:
literals

"Returns an Array containing the literal pool of the receiver.,
 including literal pools of all inner blocks. "

| res blks |
res := self _literals .
blks := self blockLiterals .
1 to: blks size do:[:k | | aBlk| 
  aBlk := blks at: k .
  res addAll: aBlk method _literals
].
^ res
%

category: 'Stripping Sources'
method:
removeAllSourceButFirstComment

"Installs a new source string for the receiver so that only the method signature
 and the first comment (if it exists) are left.  For use in stripping a method
 in place in GemStone.  Bypasses the invariance of the receiver."

self _validatePrivilege ifTrue:[
  "v3.0, debuginfo ok to modify if committed."
  self _debugInfo _unsafeAt: DebugInfo_source_offset put: self _sourceToFirstComment
].
%

category: 'Stripping Sources'
method:
emptySource

"Returns nil in place of the source string.  The #emptySource selector may be
 used as an argument to the stripWith: keyword of the method
 GsNMethod>>decompileForCategory:classRef:stripWith:classMethod:, where it causes
 the string 'source not available...' to be used as the source string when
 reloading the decompiled method."

^ nil 
%

category: 'Stripping Sources'
method:
fullSource

"Returns the complete source string.  The #fullSource selector may be used as an
 argument to the stripWith: keyword of the method
 GsNMethod>>decompileForCategory:classRef:stripWith:classMethod:."

^ self sourceString
%

category: 'Stripping Sources'
method:
sourceToFirstComment

"Returns a new source string for the receiver that contains only the method
 signature and the first comment (if it exists).  Does not modify the
 receiver.  The #sourceToFirstComment selector may be used as an argument to
 the stripWith: keyword of the method
 GsNMethod>>decompileForCategory:classRef:stripWith:classMethod:."

| i tmpString sz srcStr nArgs selectorSym |

i := 0.
srcStr := self sourceString .
sz := srcStr size.
selectorSym := self selector .
(selectorSym occurrencesOf: $:) timesRepeat: [
    i := srcStr indexOf: $: startingAt: i + 1.
].
" check if it's a binary selector (it has an argument but no colons) "
nArgs := self numArgs .
(i == 0 and: [ nArgs = 1 ])
  ifTrue: [ i := selectorSym size ].
" i is the offset of the last colon in the signature, or the last character
of a binary selector, or zero "
i := i + 1.

" scan past any white space "
[ (srcStr at: i) isSeparator ] whileTrue: [ i := i + 1 ].

i > sz
  ifTrue: [ ^ srcStr copy ].

" scan past any non-white space to get to end of argument to last keyword"
[ i <= sz and: [ (srcStr at: i) isSeparator not ]] whileTrue: [ i := i + 1 ].
" i is now the offset of the first white space past the signature "

i > sz
  ifTrue: [ ^ srcStr copy ].

" scan past any white space "
[ i <= sz and: [ (srcStr at: i) isSeparator ] ] whileTrue: [ i := i + 1 ].
" i is now the offset of the initial comment or first line of code "

i > sz
  ifTrue: [ ^ srcStr copy ].

" if i is the offset of the initial comment, jump to the end of the comment "
(srcStr at: i) == $"
  ifTrue: [ i := srcStr indexOf: $" startingAt: i + 1 ]
  ifFalse: [ i := i - 1 ].

" create the string to replace the original source with "
tmpString := srcStr copyFrom: 1 to: i .
tmpString addAll: '

< source code not available >'.

^ tmpString
%

category: 'Stripping Sources'
method:
isSourceStripped

    "Answer true if the source code has been stripped for the
    receiver. Otherwise answer false. Determine this by asking the
    class if the method's selector is one of the known stripped
    selectors."

 "Stripping not implemented yet for GsNMethod"
 ^ false
%

! deleted _decompileForCategory:... , decompileForCategory:

category: 'Deprecated'
method:
_removeAllSourceButFirstComment

self deprecated: '_removeAllSourceButFirstComment obsolete, Use #removeAllSourceButFirstComment'.
^ self removeAllSourceButFirstComment
%

category: 'Deprecated'
method:
_emptySource

self deprecated: '_emptySource obsolete, Use #emptySource'.
^ self emptySource
%

category: 'Deprecated'
method:
_fullSource

self deprecated: '_fullSource obsolete, Use #fullSource'.
^ self fullSource
%

category: 'Deprecated'
method:
_sourceToFirstComment

self deprecated: '_sourceToFirstComment obsolete, Use #sourceToFirstComment'.
^ self sourceToFirstComment
%


! delete _new

category: 'Storing and Loading'
method:
writeTo: aPassiveObject

"Instances of GsNMethod cannot be converted to passive form.  This method writes
 nil to aPassiveObject and stops GemStone Smalltalk execution with a notifier."

aPassiveObject writeObject: nil.
self _error: #rtErrAttemptToPassivateInvalidObject.
%

category: 'Debugging Support'
method:
__setBreakAtIp: ipOffset operation: opcFlags frame: fpOffset process: aGsProcess

"Set breakpoint at specified ipOffset.  ipOffset is an instruction offset,
 which is a absolute byte offset within the receiver.

 opcFlags is a SmallInteger , fields have these masks
    opcode 		   16rFF
    includeMethodStartBcs 16r100 , a Boolean , only affects opcode==1
        when ipOffset == -1 .

 opcode  action
   0	 set or reenable method breakpoint
   1     set single step breakpoint (takes precedence over method break)
   2     delete method breakpoint or disabled method breakpoint
   3     delete single step breakpoint
   4     disable method breakpoint, no action if breakpoint not set

  If ipOffset == -1, then apply the action to all step points within the
  method.  

  For opcode 1 only, to restrict breakpoint to a specific frame 
  and/or GsProcess , fpOffset and/or aGsProcess must be non-nil. "

<primitive: 190>
self _primitiveFailed: #__setBreakAtIp:operation:frame:process: 
	args: { ipOffset . opcFlags . fpOffset . aGsProcess }
%

category: 'Debugging Support'
classmethod:
_setBreakAtIp: ipOffset operation: opcode frame: fpOffset process: aGsProcess

"Apply the following action to all methods that contain breakpoints.

 opcode  action
   2     delete all method breakpoints and disabled method breakpoints
   3     delete single step breakpoints
   (note function of prim 190 varies depending on receiver)

 Other opcodes that are defined for the instance method version of this
 primitive are illegal when used as a class method.

 ipOffset must be == -1 "

<primitive: 190>
self _primitiveFailed: #_setBreakAtIp:operation:frame:process:
  args: { ipOffset . opcode . fpOffset . aGsProcess }
%

category: 'Debugging Support'
method:
_setAllStepBreaks: includeMethodStartBcs frame: fpOffset process: aGsProcess

"includeMethodStartBcs is a Boolean, true means include setting of breakpoints
 on method-entry bytecodes which check the interruptWord ."

| opcFlags |
opcFlags := 1 .
includeMethodStartBcs ifTrue:[ opcFlags := 16r101 ].
self _allBreaksOp: opcFlags frame: fpOffset process: aGsProcess .
%

method:
_setStepOverBreaks: includeMethodStartBcs frame: fpOffset process: aGsProcess 

^ self _setStepOverBreaks: includeMethodStartBcs frame: fpOffset process: aGsProcess 
       stepThrough: false
%

method:
_setStepOverBreaks: includeMethodStartBcs frame: fpOffset process: aGsProcess 
   stepThrough: thruBool
| opcFlags |
opcFlags := 1 .
includeMethodStartBcs ifTrue:[ opcFlags := 16r101 ].
self _setBreakAtIp: -1 operation: opcFlags frame: fpOffset process: aGsProcess .
thruBool ifTrue:[ | blks |
  blks := self homeMethod blockLiterals .
  blks do:[ :aBlk | | meth |
      meth := aBlk method .
      meth ~~ self ifTrue:[
        meth _setBreakAtIp: -1 operation: 1 frame: nil process: aGsProcess
      ].
  ].
].
%

category: 'Debugging Support'
method:
_clearAllStepBreaks

"Legacy code, no longer used"

self _allBreaksOp: 3 frame: nil process: nil 
%

! changed result to include <GsNMethod> for each breakpoint, to fix 31953
! result of this method also used by GBS , be careful with changes

category: 'Debugging Support'
classmethod:
_breakReport: withLineNumbersBool

"Returns an Array describing all method breakpoints currently set.
 The Array contains a string report of all breakpoints and an Array
 of 6 element Arrays .  

 The Array is { <break report string as displayed in topaz, one line
		  per breakpoint> .
               { { <breakNumber>. <class> . <selector>. <stepPoint>. <GsNMethod> . <disabledBoolean>}.
                     ...
                 {<breakNumber>. <class> . <selector>. <stepPoint>. <GsNMethod> . <disabledBoolean>}
               }
               }"

| allBreaksRaw sortedBreaks report descriptors |

allBreaksRaw := GsNMethod _allMethodBreakpoints .
allBreaksRaw size == 0 ifTrue:[ ^ { 'No breaks set' , Character lf . { } } ] .

sortedBreaks := SortedCollection new .
allBreaksRaw do:[: methodBreakArray | 
  1 to: methodBreakArray size by: 3 do:[ :j | 
    | brkNum aMeth ipOfs hmMeth assoc stepPt |
    brkNum := methodBreakArray at: j .
    aMeth :=  methodBreakArray at: j + 1 .
    ipOfs :=  methodBreakArray at: j + 2 .

    hmMeth  := aMeth homeMethod .
    stepPt := hmMeth _stepPointForMeth: aMeth ip: ipOfs .
    stepPt == nil ifTrue:[ self error:'could not translate IP to a step point'].
    assoc := Association newWithKey: brkNum
	value: { brkNum . hmMeth inClass . hmMeth selector . stepPt . hmMeth } .
    sortedBreaks add: assoc . 
  ].
].

report := String new .
descriptors := { }  .
sortedBreaks do:[ :assoc | 
  | aBreakArray breakNumber aMethod theClass selector className stepPoint disabled env |
  aBreakArray := assoc value .
  breakNumber := aBreakArray at: 1 .
  aMethod := aBreakArray at: 5 .
  theClass := aBreakArray at: 2 .
  className := theClass name .
  selector := aBreakArray at: 3 .
  stepPoint := aBreakArray at: 4 .
  withLineNumbersBool ifTrue:[ 
    report addAll: breakNumber asString; addAll:': ' .
  ] .
  report addAll: className; addAll: ' >> ' ;
	 addAll: selector;  addAll: ' @ ' ;
	 addAll: stepPoint asString .
  (env := aMethod environmentId) ~~ 0 ifTrue:[
    report addAll: ' env ' , env asString .
  ].
  (disabled := stepPoint < 0) ifTrue:[ 
    stepPoint := stepPoint negated .
    report addAll: ' (disabled)' 
  ].
  report add: Character lf .
  aBreakArray at: 4 put: stepPoint .
  descriptors add: aBreakArray .
  aBreakArray at: 6 put: disabled .
].
^ { report . descriptors }
%

category: 'Accessing'
method:
numArgs

"Returns the number of arguments expected by the method."

^ iFields1 bitAnd: NArgs_mask
%

category: 'Accessing'
method:
numSends

"Returns the number of non-special sends in the method."

^ (iFields1 bitShift: 0 - NumSends_shift) bitAnd: NumSends_mask
%

category: 'Accessing'
method:
literalsOffset

"Returns the one-based offset to the start of the non-selector literals.
 Returns zero if there are no literals other than those in the selectorPool."

| ofs high |
ofs := (iFields1 bitShift: 0 - LiteralsOffset_shift) bitAnd: LiteralsOffset_mask .
ofs ~~ 0 ifTrue:[
  (ofs bitAnd: LiteralsOffset_lrgBit) ~~ 0 ifTrue:[
    ofs := ofs bitXor: LiteralsOffset_lrgBit .
    high := debugInfo at: DebugInfo_Fields2_offset .
    high := (high bitShift: 0 - HighLiteralsOffset_shift ) bitAnd: HighLiteralsOffset_mask .
    high := high bitShift: LiteralsOffset_smallBits .
    ofs := ofs bitOr: high .
  ].
  ofs := ofs + 1  "convert to one-based"
].
^ ofs
%

category: 'Accessing'
method:
inClass

"Returns the class in which the receiver was compiled.
 Returns nil for an anonymous method."

self isMethodForBlock ifTrue:[ 
  ^ inClass"the home method"   inClass
] ifFalse:[
  ^ inClass   "result is nil for anonymous method"
]
%

category: 'GBS compatibility'
method: 
_inClass
  ^ self inClass
%

category: 'Accessing'
method:
homeMethod

"Returns the home method if the receiver is method for a block,
 otherwise returns self."

^ self isMethodForBlock ifTrue:[ inClass ] ifFalse:[ self ]
%

category: 'Accessing'
method:
invocationCount

"no longer supported"
^ 0
%

! see bugmail for 35004  for more details on conversion of methods
category: 'Accessing'
method:
methodCompilerVersion

"Returns the method compiler version.
 4 indicates a method compiled by Gs64 v3.3 or above method compiler,
 3 indicates a method compiled by Gs64 v3.0 or above method compiler,
 2 indicates a method compiled by Gs64 v2.0 or above method compiler,
 1 indicates a method compiled in a previous version and processed by
   repository conversion.

 Any other value indicates a method from a previous version that
 did not get converted. " 

^ (iFields1 bitShift: MethCompilerVers_shift negated) bitAnd: MethCompilerVers_mask
%
category: 'Repository Conversion'
method:
needsRecompile

"Returns true if the receiver needs recompilation, false otherwise."

 ^ self methodCompilerVersion < 4
%

category: 'Accessing'
method:
selector

"Returns the Symbol which is the selector of this method."

| selIv selectorObjId |
" in blocks and anonymous methods, selector instVar contains envId and #'' "
self isMethodForBlock ifTrue:[ ^ nil ].  
inClass == nil ifTrue:[ ^ nil "anonymous method"].
selIv := selector .
"note bit operations are on integer value of SmallInteger, not on a OopType"
selectorObjId := ((selIv bitAnd: 16r1fffffffffe0 ) bitShift:3) bitOr: 1 .
^ Object _objectForOop: selectorObjId .
%

category: 'GBS compatibility'
method:
_selector

^ self selector
%

category: 'Accessing'
method:
environmentId

"Return a SmallInteger, 
 the 8 bit unsigned compilation environment identifier of this method."

 ^ (selector bitShift: 0 - SI_SELECTORID_ENV_shift) bitAnd: SELECTORID_ENV_mask  
%


category: 'Debugging Support'
method:
_opcodeAtIsCallPrimitive: ipOffset 

" ipOffset is obtained from result of GsProcess >> _frameContentsAt:"

^ ((self _opcodeKindAt: ipOffset abs ) bitAnd: 16r1) ~~ 0
%
method:
_opcodeAtIsSend: ipOffset 

" ipOffset is obtained from result of GsProcess >> _frameContentsAt:"

^ ((self _opcodeKindAt: ipOffset abs ) bitAnd: 16r4) ~~ 0
%

category: 'Private'
method:
_opcodeKindAt: ipOffset

"primitive will fail if ipOffset is out of range .
 argument ipOffset is zero based  and must be positive.

 result contains these bits
   16r1 if instruction at ipOffset is a call to a primitive,
   16r2 if instruction requires protected mode  
   16r4 if instruction is a send opcode."
<primitive: 189>
self _primitiveFailed: #_opcodeKindAt: args: { ipOffset }
%

category: 'Debugging Support'
method:
_isProtected

| firstOfs |
firstOfs := GSNMETHOD_FIRST_INSTR_OFFSET .
^ ((self _opcodeKindAt: firstOfs) bitAnd: 16r2) ~~ 0
%

category: 'Debugging Support'
method:
_isPrimitive
| firstOfs |
firstOfs := GSNMETHOD_FIRST_INSTR_OFFSET .
^ self _opcodeAtIsCallPrimitive: firstOfs 
%

category: 'Debugging Support'
method:
__allBreakpoints

"Returns an Array of the form

       { breakpointNumber1 . self . ipOffset1 .
          ...
          breakpointNumberN . self . ipOffsetN .
       }

 The ipOffsets are instruction offsets.
 Negative ipOffsets denote disabled breakpoints.

 Returns nil if no method breakpoints set in the receiver.
 Single step breakpoints managed by GciStep() are not reported."

<primitive: 193>
self _primitiveFailed: #__allBreakpoints
%

category: 'Debugging Support'
method:
_allBreakpoints

"Returns an Array of the form

       { breakpointNumber1 . method . ipOffset1 .
          ...
          breakpointNumberN . method . ipOffsetN .
       }

 The ipOffsets are instruction offsets.
 Negative ipOffsets denote disabled breakpoints.

 Returns nil if no method breakpoints set in the receiver.
 Single step breakpoints managed by GciStep() are not reported."

| res blks |
res := self __allBreakpoints .
blks := self blockLiterals .
blks do:[ :aBlk | | blkRes |
    blkRes := aBlk method __allBreakpoints .
    blkRes size ~~ 0 ifTrue:[ 
      res == nil ifTrue:[ res := { }  ].
      res addAll: blkRes
    ].
].
^ res
%


! _breakpointIpOffsets deleted
category: 'Debugging Support'
classmethod:
_hasBreakpoints

^ self _allMethodBreakpoints size > 0
%

category: 'Debugging Support'
classmethod:
_allMethodBreakpoints

"Returns an Array of Arrays of the form

  { { <breakpointNumber>. <aGsNMethod>. <ipOffset>.
        ...
        <breakpointNumber>. <aGsNMethod>. <ipOffset>}.
      ...
    { <breakpointNumber>. <aGsNMethod>. <ipOffset>
         ...
       {
   }

 The interior Arrays are as described for GsNMethod | _allBreakpoints.

 Breakpoints installed by single-step are not reported.
 If no breakpoints are set, returns an Array of size zero." 

^ self _oneArgPrim: 0 with: 0 
%
category: 'Debugging Support'
classmethod:
_oneArgPrim: opcode with: aSmallInteger
"opcode == 0 , _allMethodBreakpoints
 opcode == 1 , send-site caches report  

 opcode == 2 , set ComGenStateSType.dynamicSmalltalkLitVars, 
               arg is 0 or 1,
               result is previous value, aBoolean 
 opcode == 3 , _firstPortableIpoffset , arg is 0, result SmallInteger. "

<primitive: 194>
self _primitiveFailed: #_oneArgPrim args: { opcode . aSmallInteger }

%

category: 'Dynamic Literal Variables'
classmethod:
dynamicLiteralVariables: aBoolean

"Returns a Boolean, the previous state of the 
  compiler control variable. 

 aBoolean enables generation of dynamic literal variables 
 in all subsequent compilations of smalltalk methods. 
 Ruby methods (environment 1) are not affected.

 If dynamic literal variables are enabled, 
 a literal variable whose association 
 is not an instance of SymbolAssociation and which is
 a kind of SymbolAssociation, and which is not invariant,
 will have code emitted to send #_value to the association
 on fetchs, and #'_value:' on  on stores. 

 _value and _value: have default implementations in Association.
"
 
^ self _oneArgPrim: 2 with: (aBoolean ifTrue:[1] ifFalse:[0]) 
%


category: 'Reporting'
method:
isSenderOf: aSymbol

"Returns true if the receiver sends the message aSymbol.  Returns false
 otherwise."
 
| ofs |
ofs := self _sourceOffsetOfFirstSendOf: aSymbol .
^ ofs ~~ nil 
%

category: 'Reporting'
method:
_selectorPool

"Returns an IdentitySet containing the selectors sent by the receiver.
 The result includes selectors sent by inner blocks of the receiver."

^ self _allDebugInfo: 7
%

category: 'Reporting'
method:
_classAndSelectorNameWidth: anInt 

"Return a String of the form className | selector with the className substring
 padded to width anInt.
 Used by ProfMonitor"

|text sel|
self isMethodForBlock ifTrue:[ | homeMeth homeCls |
  homeMeth := self homeMethod .
  homeCls := homeMeth inClass .
  homeCls ifNotNil:[
    text := 'block in ', homeCls name .
  ] ifNil:[
    text := 'block in executed code' copy .
  ].
  sel := homeMeth selector .
] ifFalse:[  | inCls |
  (inCls := inClass) ifNil: [ 
    text := 'executed code' copy .  
  ] ifNotNil:[
    text := String withAll: inClass name .
    sel := self selector . 
  ].
].
text width: anInt .
sel ifNotNil:[ text addAll: ' >> '; addAll: sel ].
^ text
%

! fix 41195, 47154
category: 'Debugging Support'
method:
_recompileWithSource: aString

"Recompiles the receiver using the new source string aString.  If the
 compilation has no errors, installs a new method in the receiver's class's
 method dictionary, and returns true, otherwise signals a CompileError .
 May signal a CompileWarning after installing a new method.

 No check is made as to whether the new method has the 
 same selector as the receiver.

 The environmentId of the receiver is preserved.

 Raises an error and does not recompile if the receiver is an anonymous method."
| env |
env := self environmentId .
self _validatePrivilege ifFalse:[ ^ nil ].
self isMethodForBlock ifTrue:[
  self _halt: 'Attempt to recompile method for a block. You must recompile the home method.' .
  ^ nil 
].
inClass ifNil:[ 
  self _halt: 'Attempt to recompile anonymous method.' .  
  ^ nil 
].
inClass compileMethod: aString 
        dictionaries: GsCurrentSession currentSession symbolList
	      category: (inClass categoryOfSelector: self selector environmentId: env) 
        environmentId: env .
^ true 
%


category: 'Disassembly'
method:
_opcodeInfoAt: ipOffset

"ipOffset must be a valid absolute byte offset to an instruction in the receiver.
 If the specifed instruction represents a currently installed breakpoint,
 the result is for the instruction that the breakpoint is replacing.

 returns a SmallInteger with the following bit fields 
           16rFF  instruction size in bytes , including the opcode word and any
	  	  in-line literal words , an unsigned 8-bit int
       16rFFFF00  instVar offset for instVarAccess 
      16r1000000  boolean, 1 means opcode is an instVarAccess
      16r2000000  boolean, 1 means opcode is a pushBlock
   16rFFF0000000  opcode 
 "  
<primitive: 670>
self _primitiveFailed: #_opcodeInfoAt: args: { ipOffset } .
^ 1
%

category: 'Disassembly'
method:
_instVarsAccessed: kind into: result

"Adds to result  
  the names of the instVars which are read or written by the method,
  per the kind argument, not including accesses by any inner blocks . 
 or Adds to result the source offsets of accesses to the specified instVar

  kind == 0 --> read , 
  kind == -1 --> written, 
  kind == -2 --> read or written 
  kind is one based offset of an instVar"

<primitive: 671>
self _primitiveFailed: #_instVarsAccessed:into: args: { kind . result }
%

category: 'Disassembly'
method:
_allInstVarsAccessed: kind into: anIdentitySet

"Adds to result the instVarNames that the method accesses,
 including accesses by all inner blocks , as specified by kind.
 Returns anIdentitySet .
 
  kind == 0 --> read ,   result is an IdentitySet of instVar names
  kind == -1 --> written,  result is an IdentitySet of instVar names
  kind == -2 --> read or written, result is an IdentitySet of instVar names
  kind is one based offset of an instVar,  result is IdentitySet of source offsets of access to that instVar"

| blks |
self _instVarsAccessed: kind into: anIdentitySet .
blks := self blockLiterals .
blks do:[ :aBlk | 
  aBlk method _instVarsAccessed: kind into: anIdentitySet .
].
^ anIdentitySet
%

category: 'Disassembly'
method:
instVarsAccessed

"Returns an IdentitySet of instVarNames that the method accesses,
 including accesses by all inner blocks."

^ self _allInstVarsAccessed: -2 into: IdentitySet new 
%
category: 'Disassembly'
method:
instVarsRead

"Returns an IdentitySet of instVarNames that the method accesses,
 including accesses by all inner blocks "

^ self _allInstVarsAccessed: 0 into: IdentitySet new 
%
category: 'Disassembly'
method:
instVarsWritten

"Returns an IdentitySet of instVarNames that the method accesses,
 including accesses by all inner blocks "

^ self _allInstVarsAccessed: -1 into: IdentitySet new
%

! fix 43180
category: 'Disassembly'
method:
sourceOffsetsOfInstVar: aString

"Returns an Array of sourceOffsets of access to the specifed instVar.
 Returns nil if there is no instVar named aString in the receiver's class."
| ofs sym |
(sym := Symbol _existingWithAll: aString ) ifNil:[ ^ nil ].
ofs := self inClass offsetOfInstVar: sym .
ofs == 0 ifTrue:[ ^ nil ].
^ Array withAll:(self _allInstVarsAccessed: ofs into: IdentitySet new)
%

category: 'Disassembly'
method:
_blockLiterals

"Return an Array of block literals that the receiver contains,
 or nil if the receiver contains no block literals.
 In-line blocks never have any block literals.
 Literals for inner blocks are not included."
 
| arr lastBlkIdx numBlks |
numBlks := self _numBlockLiterals .
numBlks ~~ 0 ifTrue:[  | resIdx |
  arr := Array new: numBlks .
  lastBlkIdx := self _lastBlockLiteralOffset .
  resIdx := 1 .
  lastBlkIdx - numBlks + 1 to: lastBlkIdx do:[:n| 
    arr at: resIdx put:( self at: n ) .
    resIdx := resIdx + 1 .
  ].
].
^ arr
%

method:
_addBlockLiteralsTo: anArray
  "Add all block literals of receiver plus inner blocks to anArray"
  | lastBlkIdx numBlks |
  numBlks := self _numBlockLiterals .
  numBlks ~~ 0 ifTrue:[
    lastBlkIdx := self _lastBlockLiteralOffset .
    lastBlkIdx - numBlks + 1 to: lastBlkIdx do:[:n| | aBlk | "recursively process blocks"
      aBlk := self at: n .
      anArray add: aBlk .
      aBlk method _addBlockLiteralsTo: anArray
    ].
  ].
%


category: 'Disassembly'
method:
blockLiterals

"Return an Array of Block Literals that the receiver contains,
 including all inner blocks. Returns nil if the receiver 
 contains no blocks.
 In-line blocks never have any block literals."

| arr | 
self _numBlockLiterals ~~ 0 ifTrue:[
  arr := { } .
  self _addBlockLiteralsTo: arr .
] ifFalse:[
  arr := #() .
].
^ arr
%

category: 'Disassembly'
method:
hasBlockLiteralsOfCost: aClassOrCost

"Return true if the receiver contains a block as costly as, or
 more costly than aClassOrCost, return false otherwise.

 For the purposes of this analysis, we have:
   simple block  cost == 1  (old class SimpleBlock)
   complex block cost == 2  (old class ComplexBlock)
   complex block with return to home, cost == 3 (ComplexVCBlock)

 Note that SimpleBlock, ComplexBlock, ComplexVCBlock
 are supported for compatiblity only . New code should use
 the integer costs since there is now only one block class, ExecBlock . 
 In-line blocks never have an actual ExecBlock and are never included
 in this analysis.
 "

| blockLits |
blockLits := self _blockLiterals .
blockLits size ~~ 0 ifTrue:[  | argCost |
  aClassOrCost _isSmallInteger ifTrue:[
    argCost := aClassOrCost
  ] ifFalse:[
    argCost := aClassOrCost _cost .
  ].
  1 to: blockLits size do:[:k| | aBlk |
    aBlk := blockLits at: k .
    aBlk _cost >= argCost ifTrue:[ ^ true ] .
  ].
  1 to: blockLits size do:[:k| | aBlk |
    aBlk := blockLits at: k .
    "recursively check inner blocks"
    (aBlk method hasBlockLiteralsOfCost: argCost) ifTrue:[ ^ true].
  ].
].
^ false
%

! _finishConvertLargeMethod deleted

category: 'CodeModification Override'
method:
_at: anIndex put: aValue

self shouldNotImplement:#_at:put:
%

category: 'CodeModification Override'
method:
_basicAt: anIndex put: aValue

self shouldNotImplement:#_basicAt:put:
%

category: 'CodeModification Override'
method:
squeakBasicAt: anIndex put: aValue

self shouldNotImplement:#squeakBasicAt:put:
%

category: 'CodeModification Override'
method:
_primitiveAt: anIndex put: aValue

self shouldNotImplement:#_primitiveAt:put:
%

category: 'CodeModification Override'
method:
_unsafeAt: anIndex put: aValue

self shouldNotImplement:#_unsafeAt:put:
%

category: 'CodeModification Override'
method: 
_validatePrivilege

^ System myUserProfile _validateCodeModificationPrivilege
%



category: 'Debugging Support'
method:
_nativeIpOffsetToPortable: aNativeIpOffset asReturn: aBoolean

"for the given offset in native code, return the corresponding
 offset in intepreted code.
 If aBoolean == true, result is as an IP to which the callee will return.
 Returns 0 if the receiver does not have native code,
    -1 if the argument is out of range,
    value < -1 if translation not exact
 "
<primitive: 673>
self _primitiveFailed: #_nativeIpOffsetToPortableRtnAddr:asReturn:
     args: { aNativeIpOffset . aBoolean }
%

category: 'Debugging Support'
method:
_natIpToPort: aNativeIpOffset
  "For the given offset in native code, interpreted as a return address,
   return the corresponding offset in intepreted code.
  Returns 0 if the receiver does not have native code,
    -1 if the argument is out of range,
    value < -1 if translation not exact. "

  ^ self _nativeIpOffsetToPortable: aNativeIpOffset asReturn: true
%

!  _descrForStackPadTo: was reimplemented in image/ruby/GsNMethod_ruby.gs
category: 'Debugging Support'
method:
_descrForStackPadTo: minSize 

^ self _descrForStackPadTo: minSize rcvr: _remoteNil
%

! fixed 46626
category: 'Debugging Support'
method:
_descrForStackPadTo: minSize rcvr: aRcvr

| result hmMeth inCls isBlock |
result := String new.
self isMethodForBlock ifTrue:[
  isBlock := true.
  result add: '[] in '.
  hmMeth := inClass.
  inCls := hmMeth inClass
] ifFalse:[
  isBlock := false.
  hmMeth := self.
  inCls := inClass
].
inCls ifNil: [result add: 'Executed Code ']
   ifNotNil:[
     aRcvr ~~ _remoteNil ifTrue:[| rcvCls |
       (isBlock not and: [(rcvCls := Reflection classOf: aRcvr) ~~ inCls]) ifTrue:[
         result add: rcvCls name; add: ' ('; add: inCls name; add: $)
       ] ifFalse:[result add: inCls name]
     ] ifFalse: [result add: inCls name].
     result add: ' >> '; add: hmMeth selector
].
[ result size < minSize ] whileTrue:[ result add: $ ].    "pad with spaces"
inClass ifNotNil:[ | env |
   env := hmMeth environmentId.
   env ~~ 0 ifTrue:[ result add: ' (envId '; add: env asString; add: $)]
].
^result
%


category: 'Class organizer support'
method:
_referencedStringsInto: aSet 
  "Add all literal Strings of receiver to aSet. Returns receiver." 
  | litOfs lastBlkIdx numBlks |
  (litOfs := self literalsOffset) ~~ 0 ifTrue:[
    lastBlkIdx := self _lastBlockLiteralOffset .
    numBlks := self _numBlockLiterals .
    litOfs to: (lastBlkIdx - numBlks) do:[:k| | lit |
      lit := self at: k.
      lit _isSymbol ifFalse:[
        (lit isKindOf: CharacterCollection) ifTrue:[ 
          aSet add: lit
        ].
      ].
    ].
    lastBlkIdx - numBlks + 1 to: lastBlkIdx do:[:n| | aBlk | "recursively search blocks"
      aBlk := self at: n .
      aBlk method _referencedStringsInto: aSet
    ].
  ].
%

! fix 41606
method:
_literalsIncludesSymbol: aSymbol value: anObject
  "if anObject == nil , it is not searched for as a value"
  | lastBlkIdx numBlks litOfs |
  (self _selectorPool includes: aSymbol) ifTrue:[ ^ true ].
  (litOfs := self literalsOffset) ~~ 0 ifTrue:[
    lastBlkIdx := self _lastBlockLiteralOffset .
    numBlks := self _numBlockLiterals .
    anObject ifNil:[ 
      litOfs to: (lastBlkIdx - numBlks) do:[:k| | lit |
        lit := self at: k.
        (lit == aSymbol 
          or:[ (lit isKindOf: SymbolAssociation) and:[ lit key == aSymbol ]]
         ) ifTrue:[ ^ true ]
      ].
    ] ifNotNil:[
      litOfs to: (lastBlkIdx - numBlks) do:[:k| | lit |
        lit := self at: k.
        (lit == aSymbol  or:[ lit == anObject 
          or:[ (lit isKindOf: SymbolAssociation) 
               and:[ lit key == aSymbol or:[ lit _value == anObject]]]]
         ) ifTrue:[ ^ true ]
      ].
    ].
    lastBlkIdx - numBlks + 1 to: lastBlkIdx do:[:n| | aBlk | 
      "recursively search blocks"
      aBlk := self at: n .
      (aBlk method _literalsIncludesSymbol: aSymbol value: anObject) ifTrue:[ ^ true ] .
    ].
  ].
  ^ false
%

category: 'Class organizer support'
method:
_literalsIncludesValue: anObject
  | litOfs lastBlkIdx numBlks |
  (self _selectorPool includes: anObject) ifTrue:[ ^ true ].
  (litOfs := self literalsOffset) ~~ 0 ifTrue:[
    lastBlkIdx := self _lastBlockLiteralOffset .
    numBlks := self _numBlockLiterals .
    litOfs to: (lastBlkIdx - numBlks) do:[:k| | lit |
      lit := self at: k.
      (lit == anObject or:[
	   (lit isKindOf: SymbolAssociation) and:[ lit _value == anObject]
        ] ) ifTrue:[ ^ true ]
    ].
    lastBlkIdx - numBlks + 1 to: lastBlkIdx do:[:n| | aBlk | "recursively search blocks"
      aBlk := self at: n .
      (aBlk method _literalsIncludesValue: anObject) ifTrue:[ ^ true ]
    ].
  ].
  ^ false
%

category: 'Debugging Support'
method:
_descrForStack
^ self _descrForStackPadTo: 0
%

category: 'Accessing'
method: 
loadedSizeBytes

 "Return the number of bytes of code generation memory that
  the receiver will occupy when loaded for execution, not
  including the native code."

  "header + namedIvs = 3 + 4 = 7 words"

  ^ (7 + self size + (self numSends * 2)) * 8
%

category: 'Pragmas'
method: 
_pragmasArray
  "Return the raw pragmas array for a method."

  (self _debugInfoAccess: 11 at: -1) ifNotNil:[:parray | ^ parray ].
  ^ #()
%

category: 'Pragmas'
method: 
pragmas
  "Returns a possibly empty Array of Pragmas."
  (self _debugInfoAccess: 11 at: -1) ifNotNil:[:parray | 
    parray == #() ifTrue:[ ^ #() ].
    ^ self class createPragmaFrom: parray for: self  
  ].
  self environmentId ~~ 0 ifTrue:[ ^ #() ].
  self inClass ifNotNil:[:cls | ^ cls _pragmasForMethod: self ]
%

! fix 44868
category: 'Private
classmethod:
_gsReturnToC

"Return from Smalltalk execution to C code. 
 The C code may be a primitive within the VM,
 a user action or FFI function which invoked Smalltalk ,
 or a GCI client process which made a GCI call to run Smalltalk.

 To be sent only from within the virtual machine. Do not do a send or perform
 of this method in any Smalltalk or GCI code. "

<primitive: 2013>
"If primitive succeeds control returns to the GCI, useraction/FFI, or the VM. 
 If primitive fails, the scheduler needs to regain control
 for process termination." 

self _terminateNonForkedProcess .
%

classmethod:
_terminateNonForkedProcess
  2 timesRepeat:[ self _class ]. "loop to detect/handle termination interrupt"
  TerminateProcess new details:'Cannot continue execution'; signalNotTrappable
%

category: 'Private
classmethod:
_noopReturnTos

"Used in implementation of partial continuation evaluation only."
<primitive: 2032>
"Execution returns to the calling frame"
%

category: 'Repository Conversion'
method: 
recompile
  "Returns self if recompile not needed, otherwise
   returns a new instance of GsNMethod using session's default symbolList"

  self needsRecompile ifFalse:[ ^ self "recompile not needed"].
  ^ self recompileIntoMethodDict: nil intoCategories: nil .
%

category: 'Class organizer support'
classmethod:
optimizedSelectors
^ #( #~~
    #==
    #_and:
    #_downTo:by:do:
    #_downTo:do:
    #_isArray
    #_isExceptionClass
    #_isExecBlock
    #_isFloat
    #_isInteger
    #_isNumber
    #_isOneByteString
    #_isRange
    #_isRubyHash
    #_isScaledDecimal
    #_isSmallInteger
    #_isSymbol
    #_leaveProtectedMode
    #_or:
    #_stringCharSize
    #and:
    #ifFalse:
    #ifFalse:ifTrue:
    #ifNil:
    #ifNil:ifNotNil:
    #ifNotNil:
    #ifNotNil:ifNil:
    #ifTrue:
    #ifTrue:ifFalse:
    #isKindOf:
    #or:
    #repeat
    #timesRepeat:
    #to:by:do:
    #to:do:
    #untilFalse
    #untilFalse:
    #untilTrue
    #untilTrue:
    #whileFalse
    #whileFalse:
    #whileTrue
    #whileTrue: )
%
category: 'Private"
classmethod:
_firstPortableIpoffset

  "returns the VM constant OC_GSNMETHOD_FIRST_INSTR_OFFSET"
  ^ self _oneArgPrim: 3 with: 0
%

!fix 46405
category: 'Accessing'
method:
instVarAt: anIndex

"If the receiver has an instance variable at anIndex, returns
 its value.  Generates an error if anIndex is not a SmallInteger or
 is out of bounds, or if the receiver has no instance variables."

| numInstVars |
(anIndex _isSmallInteger) ifTrue: [
  numInstVars:= self class instSize .
  ((anIndex >= 1) and:[anIndex <= numInstVars]) ifFalse:[ 
        ^ self _errorIndexOutOfRange: anIndex "out of bounds"
  ].
  ^ self _primitiveAt: anIndex
] ifFalse: [ 
  ^ self _errorNonIntegerIndex: anIndex 
].
%

! fix 46473
category: 'Formatting'
method:
printOn: aStream
  | classOrNil selectorOrNil |
  aStream nextPutAll: self class name; nextPut: $  .
  self isMethodForBlock ifTrue: [aStream nextPutAll: '[] in '].
  classOrNil := self inClass.
  selectorOrNil := self homeMethod selector.
  aStream nextPutAll: (classOrNil ifNil:[ '<nil>' ] ifNotNil:[ classOrNil name]);
          nextPutAll: '>>';
          nextPutAll: (selectorOrNil ifNil:[ '<anonymous>'] ifNotNil:[ selectorOrNil]).
%
category: 'Debugging Support'
method:
_primitiveNumber

"Return the primitive number of the method or -1 if the method
 is not a primitive."
| firstOfs |
firstOfs := GSNMETHOD_FIRST_INSTR_OFFSET .
(self _opcodeAtIsCallPrimitive: firstOfs) ifTrue:[
 ^ ((self _opcodeInfoAt: firstOfs) bitShift: -40) bitAnd: 16rFFF
].
^ -1
%


