!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   GsProcess, Object.
!
!=========================================================================

set class GsProcess

removeallmethods 
removeallclassmethods 

!  Globals at: #Semaphore put: nil    now done in bom.c
!  Globals at: #ProcessorScheduler put: nil    now done in bom.c

! ------------------- Class methods for GsProcess

category: 'Documentation'
classmethod:
comment
^ '
 A GsProcess represents a suspended or active GemStone Smalltalk 
 green-thread process.  
 The in-memory state of a committed GsProcess is not changed by a transaction abort.

 instVars are:

  stackId - a SmallInteger  
     zero-based stackId specifying the stack memory area holding
     the process, or -1 means not in a stack memory area.

  signalTime - used by the scheduler .

 priority_use - a SmallInteger used by the scheduler, bit masks
          16rFFFFFFFFFF    unsigned 40 bit use count, incremented each time
               the process is run or resumed by the scheduler.
      16rFFFF0000000000    unsigned priority in the range of a C ushort.

  uaCount_depth -  a SmallInteger,  with bit masks
            16rFFFFFFFF   stack depth , for a process not in stack memory
      16rFFFFFFF00000000 user action count
     Maybe nil if receiver is running. 

  interruptFlag_stEntryCount - a SmallInteger,  with bit masks
          16rFFFFFFFF  stEntryCount
   16rFFFFFFF00000000   interruptFlags 

  modeInfo - a SmallInteger, with bit masks (see symbolic definitions ModeInfo_)
                16rFF  threadRubyEnvId
              16rFF00  nativeStackKind, 0..255
      16rFF0000  breakpointsToIgnore, 0..255  
           16r1000000  debuggerEnabled, a boolean
     16r2000000  debuggingInProgress, a boolean
           16r4000000  terminated , a boolean
           16r8000000  terminationStarted , a boolean
          16r10000000  isRubyThread, a boolean
          16r60000000  continuationKind, 1 == continuation, 2 == partial continuation
          16r80000000  forked , a boolean
         16r100000000  partialContinuationLoaded, private boolean for VM 
         16r200000000  Gs64 v3.3 version bit.
         16rC00000000  spareBits
       16rFF000000000  OM.reapEventsCount (in a saved stack)
      nativeStackKind  is one of GC_GSPROCESS_cpuKind values from gcioc.ht ,
        1 means interpreted , > 1 is os-cpu-specific native code
      debuggingInProgress == 0, debugging or single stepping is completed.
      debuggingInProgress == 1, debugging or single stepping is in progress ;
  breakpoints or single step exceptions are handled by the debugger
        and will not be seen by any user-installed Exception handlers.
      Breakpoints are not supported while executing in native code. 
      A process'' stack can be converted from native to portable as part 
      of a GciContinue or GciStep.

 topFpOffset - a SmallInteger, byte offset from StackLimit back to the topFP,
   for a process not in a stack memory area.

 arStack - a GsStackBuffer, the saved evaluation stack, for a process
   not in a stack memory area.

 remoteProcess - reserved for use by GemEnterprise , no support in base image

 block - an ExecBlock or nil, the block that the GsProcess is to execute.

 args - an Array of arguments for the block, or nil.

 blockResult - result returned by the block of a Smalltalk process 
   has value _remoteNil if process has not terminated normally

 clientData - For application use, such as for thread-specific data.

 environment - A Dictionary containing thread-specific values,
   for application usage  with environmentAt:, environmentAt:put: .
   If the process is a Smalltalk continuation and environment is nil, 
   then the environment dictionary from the calling process
   is used as the environment dictionary for the resumed continuation.
   Not related to environmentId of message sends. 
   For Ruby, the dictionary of thread-local values.

 waitingOn - used by the scheduler, a GsSocket, Semaphore, Delay, or SharedQueue
    which this process is waiting on. 

 group - a SmallInteger or RubyThreadGroup representing a process group .

 joiners - nil, or an Array of Delays or Semaphores representing other threads
     waiting in GsProcess>>join:  for this thread to finish.

 msgEnvironmentId - message send environmentId , 0 for default Smalltalk,
                      1 for Ruby .

 signalledException - used in the implementation of Exceptions .

 ioSelectResult - used by the scheduler for implementation of Ruby IO#select,
    see primitive 845  .

 stackSerialNum - private, for use by the scheduler and VM only

 dbgFpsCache - for use by  GsProcess>>_frameDescrAt: only, cleared by the VM
        whenever arStack is cleared.

 lastBreakpt - for use by implementation of breakpoints and single step.

Constraints:
  stackId: SmallInteger
  signalTime: SmallInteger
  priority_use: SmallInteger
  uaCount_depth: SmallInteger
  interruptFlag_stEntryCount: SmallInteger
  modeInfo: SmallInteger
  topFpOffset: SmallInteger
  arStack: GsStackBuffer
  remoteProcess: Object
  block: ExecBlock
  args: Array
  blockResult: Object
  clientData: Object
  waitingOn: Object
  group: SmallInteger
  environment: AbstractDictionary
  onQueue: Object
  joiners: Object
  msgEnvironmentId: SmallInteger
  signalledException: AbstractException
  ioSelectResult: Array
  stackSerialNum: SmallInteger
  dbgFpsCache: Array
  lastBreakpt: Array
'
%

category: 'Instance Creation'
classmethod:
new

"Disallowed."

self shouldNotImplement: #new
%

! edited to fix 36121
category: 'Debugging Support'
method:
_reportWithFrameContents: frameContents level: aLevel
"Return a one line String describing the argument.
 Example result:  
   SmallInteger (Object) >> doesNotUnderstand:  @9 line 10 "

| gsMethod receiver aself result ip stepPt |

gsMethod := self _methodInFrameContents: frameContents .
receiver := self _receiverInFrameContents: frameContents .

aself := self _selfInFrameContents: frameContents .

gsMethod ifNil:[ ^ self == GsProcess _current ifTrue:['GCI'] ifFalse:['User Action' ]].

result := String new .
result add: (gsMethod _descrForStackPadTo: 0 rcvr: receiver) .
ip := (frameContents at: 2) abs .
stepPt := gsMethod _stepPointForIp: ip level: aLevel 
        useNext: (self _nativeStack or:[ self _calleeIsAsync: aLevel]) .
result add: ' @' ; add: stepPt asString ;
       add: ' line ' ;
       add: (gsMethod _lineNumberForStep:  stepPt) asString .
^ result
%

! fixed 45218
category: 'Debugging Support'
method:
_gsiDetailedReportWithFrameContents: frameContents forLevel: aLevel

"Private."
"See _gsiDetailedReportWithFrameContents: for documentation."

| result receiver gsMethod argAndTempNames argAndTempValues aself |

result := Array new: 11 .

result
  at: 1 put: (gsMethod := frameContents at: 1  "virtual machine constant") ; "gsMethod"
  at: 2 put: (receiver := frameContents at: 10 "virtual machine constant" ); "receiver"
  at: 3 put: (aself := frameContents at: 8 "virtual machine constant" ); "self"
  at: 4 put: (gsMethod == nil
    ifTrue: [#'<UserAction>']
    ifFalse: [ gsMethod _selector]); "selector"
  at: 5 put: (gsMethod == nil
    ifTrue: [ 0 ]
    ifFalse: [ gsMethod
      _stepPointForIp: (frameContents at: 2 "virtual machine constant") abs
      level: aLevel useNext: (self _nativeStack or:[ self _calleeIsAsync: aLevel])
    ]) ;
  at: 6 put: (gsMethod == nil
    ifTrue: [ #() ]
    ifFalse: [ gsMethod _blockSourceOffsets ]). "source offsets"

argAndTempNames := frameContents at: 9 "virtual machine constant" .

argAndTempValues := (frameContents size >= 11 "virtual machine constant")
    ifTrue:[ frameContents copyFrom: 11 to: frameContents size]
    ifFalse:[ { }  ].

result  at: 7 put: argAndTempNames ;
        at: 8 put: argAndTempValues ;
        at: 9 put: (gsMethod == nil
          ifTrue: [ 'source not available' ] 
          ifFalse: [ gsMethod sourceString  ]);
        at: 10 put: (frameContents at: 2 "virtual machine constant") abs ;  "ipOffset"
        at: 11 put: (frameContents at: 6 "virtual machine constant"). "markerOrException"

^ result
%

category: 'Debugging Support'
classmethod:
cpuOsKind

"Returns an integer specifying the CPU and OS on which
 this VM is running. The result is one of
   1 - reserved for use within VM 
   2 - sparc Solaris
   3 - x86_64 Linux
   4 - PowerPc AIX
   5 - x86_64 Apple Unix
   6 - x86_64 Solaris
   7 - Itanium HPUX
"
<primitive: 345>
self _primitiveFailed: #cpuOsKind
%

method:
_methodInFrameContents: aFrameContentsArray

^ aFrameContentsArray at: 1 "virtual machine constant"
%
classmethod:
_methodInFrameContents: aFrameContentsArray

^ aFrameContentsArray at: 1 "virtual machine constant"
%

category: 'Debugging Support'
method:
_receiverInFrameContents: aFrameContentsArray

^ aFrameContentsArray at: 10 "virtual machine constant"
%

category: 'Debugging Support'
method:
_selfInFrameContents: aFrameContentsArray

^ aFrameContentsArray at: 8 "virtual machine constant"
%

classmethod:
_fullStackReport
  "used in ernie framework"
| curr res |
curr := GsProcess _current .
[ res := curr stackReportToLevel: 300 withArgsAndTemps: true andMethods: false ] forkAt: 39 .
Processor yield .
^ res asString .
%

category: 'Debugging Support'
classmethod:
stackReportToLevel: aLevel

"Returns a String describing the currently active stack, starting with
 to the sender of this method (which is considered level 1).  The aLevel

 The format of the result is subject to change with each release of GemStone."

| framesArr aFrame report level lf |
framesArr := { }  .
level := 1 .
[ level <= aLevel and:[ (aFrame := self _frameContentsAt: level + 1) ~~ nil] ]
whileTrue:[
  framesArr at: level put: aFrame.
  level := level + 1.
  ].
report := String new .
lf := Character lf .
1 to: framesArr size do:[:j | | gsMethod |
  report add: j asString; add:' ' .
  aFrame := framesArr at: j .
  gsMethod := aFrame at: 1 .
  gsMethod ifNil:[
    report add:'<Reenter marker>'; add: lf .
  ] ifNotNil:[ | stepPoint aRcvr |
    aRcvr := aFrame at: 10 . 
    report add:  (gsMethod _descrForStackPadTo: 0 rcvr: aRcvr ) .
    stepPoint := gsMethod _previousStepPointForIp: (aFrame at: 2) .
    report add:' @' ; add: stepPoint asString ;
      add: ' line ';
      add: (gsMethod _lineNumberForStep: stepPoint) asString ;
      add:'  [GsNMethod '; add: gsMethod asOop asString ; add:']'; 
      add: lf .
    ].
  ].
^ report
%


category: 'Debugging Support'
classmethod:
_frameContentsAt: aLevel

"Returns an Array describing the specified stack level in the currently
 executing process.  

 aLevel == 1 means top of stack with respect to the 
               sender of _frameContentsAt: .

 If aLevel is out of range, returns nil.

 The contents of the result Array are the same as for the instance method
 GsProcess | _frameContentsAt:, except for :
    element  9 (argAndTempNames) is nil
             8 (self) is nil
             4  varContext is always nil .
 "

<primitive: 195>
self _primitiveFailed: #_frameContentsAt: args: { aLevel }
% 

category: 'Debugging Support'
classmethod:
_selfAt: aLevel

"Returns self at the given level of the currently executing process' stack.

 aLevel == 1 means top of stack with respect to the 
               sender of _selfAt: .

 If aLevel is out of range, returns nil."
 
|frame|
(frame := self _frameContentsAt: aLevel + 1) ifNil:[ ^ nil ].
^ frame at: 8 "virtual machine constant"
%

category: 'Debugging Support'
classmethod:
_receiverAt: aLevel

"Returns the receiver at the given level of the currently executing 
 process' stack.

 aLevel == 1 means top of stack with respect to the
               sender of _selfAt: .

 If aLevel is out of range, returns nil."

|frame|
(frame := self _frameContentsAt: aLevel) ifNil:[ ^ nil ].
^ frame at: 10 "virtual machine constant"
%



! removed GsProcess(C) >> _current  to fix 31305 .
!  It was referencing primitive 492 which is used for a different
!  primitive in ProcessorScheduler .

category: 'Private'
method:
_init: aBlock args: aArgs env: environmentId stackSerial: serialNum
  forked: forkedBit
  "returns receiver.
   caller must _scheduleNewProcess or _suspendNewProcess "
stackId := -1 .
"scheduler will set priority_use to match active process" 

modeInfo := ModeInfo_debugEnabled bitOr: forkedBit .
block := aBlock.
blockResult := _remoteNil .
args := aArgs.
msgEnvironmentId := environmentId .
serialNum _isSmallInteger ifFalse:[ 
    InternalError new details:'GsProcess invalid stackSerialNum' ; signalNotTrappable.
].
stackSerialNum := serialNum .

environmentId == 0 ifTrue:[ | asz bNargs |
  (asz := args size) ~~ (bNargs := block argumentCount) ifTrue: [
    "Generate an error because we don't have the correct number of args."
    OutOfRange new _number: 2373; expectedNumArgs: bNargs actual: asz ;
   object: block ; details:'wrong number of arguments for invoking a block' ;
        signal.
    ^ nil 
  ].
].
  "self _setClientData . debugging only"
%

!  uncomment this method and sends of it in this class and in
!    ProcessorScheduler  if you want process serial-number tracing for debugging
! category: 'Private' 
! method:
! _setClientData
!   clientData ifNil:[ | dict val | 
!     dict := SessionTemps current .
!     val := dict at: #GsProcId ifAbsent:[ dict at: #GsProcId put: 1 . 1 ].
!     clientData := val .
!     "GsFile gciLogServer: 'GsProcess _setClientData ' , val asString ."
!     "val == 11854 ifTrue:[  self pause ]. "
!     dict at: #GsProcId put: val + 1 
!   ].
! %

category: 'Private'
classmethod:
_scheduler

"Returns   ProcessorScheduler scheduler "
<primitive: 457>

self _primitiveFailed: #_scheduler 
%
category: 'Private'
method:
_scheduler

"Returns   ProcessorScheduler scheduler "
<primitive: 457>

self _primitiveFailed: #_scheduler 
%

! fixed 34080/14756
category: 'Private'
classmethod:
_forkBlock: aBlock with: blockArgs env: envId prio: aPriority
  "Answer an ready instance of the receiver representing a potential
   process on the given block and with the active priority."
  | proc |
  (blockArgs _isArray ) ifFalse:[
    blockArgs _error: #rtErrInvalidArgClass args: { Array } .
    self _uncontinuableError .
  ].
  (proc := self _basicNew) _init: aBlock args: blockArgs env: envId 
      stackSerial: self _current _stackSerialNum 
      forked:  ModeInfo_forked .
  self _scheduler _scheduleNewProcess: proc prio: aPriority  .
  ^ proc
%
classmethod:
_newForBlock: aBlock with: blockArgs env: envId
  "Answer a suspended instance of the receiver representing a potential
   process on the given block and with the active priority."
  | proc |
  (blockArgs _isArray ) ifFalse:[
    blockArgs _error: #rtErrInvalidArgClass args: { Array } .
    self _uncontinuableError .
  ].
  (proc := self _basicNew) _init: aBlock args: blockArgs env: envId 
       stackSerial: self _current _stackSerialNum 
      forked:  ModeInfo_forked .
  self _scheduler _suspendNewProcess: proc .
  ^ proc
%
category: 'Instance Creation'
classmethod:
continuationFromLevel: numLevels

"Returns an instance of the receiver, that is a copy of the current
 process'  stack , from the stack base to the frame that is the
 specified number of levels above the frame executing this primitive.
 anInt must be >= 1 .

 When  value:   is sent to the result,
 execution resumes in the frame at the top of the instance's saved stack,
 and with top-of-stack value replaced by the argument to value: .

 The result contains copies of all method temps and args from the
 stack it was created from ;  value: sent to the result restores 
 the contents of of all method temps and args to the identical values when
 the instance was created. "

<primitive: 47>
System continuationsEnabled ifFalse:[
  "see enableContinuations, disableContinuations in System."
  self error:'continuations are disabled in this session' .
].
numLevels _validateClass: SmallInteger .
numLevels < 1 ifTrue:[ self _errorIndexOutOfRange: numLevels ].
self _primitiveFailed: #continuationFromLevel: args: { numLevels } .
self _uncontinuableError
%

category: 'Partial Continuations'
classmethod:
partialContinuationFromLevel: levelsToTrim to: targetLevel

"Returns an instance of the receiver, that is a copy of the stack 
 for the current process from frame levelsToTrim+1 to frame targetLevel-1.
 This method (partialContinuationFromLevel:to: will be at top of stack
 which is frame 1 , when the partialContinuation is created.
 levelsToTrim and targetLevel must be >= 1 .
 targetLevel must be >= levelsToTrim+2 .

 The result is expected to be used as an argument to 
 #installPartialContinuation:atLevel:value: . 
 
 The result contains copies of all method temps and args from the
 stack it was created from and when used as an argument to
 #installPartialContinuation:atLevel:value:, the contents of all
 method temps and args are restored to the identical values when 
 the instance was created."

<primitive: 900>
levelsToTrim _validateClass: SmallInteger .
levelsToTrim < 1 ifTrue:[ self _errorIndexOutOfRange: levelsToTrim ].
targetLevel _validateClass: SmallInteger .
targetLevel < 1 ifTrue:[ self _errorIndexOutOfRange: targetLevel ].
targetLevel > (levelsToTrim + 1) ifFalse:[ self _errorIndexOutOfRange: targetLevel ].
self _primitiveFailed: #partialContinuationFromLevel:to:
     args: { levelsToTrim . targetLevel } .
self _uncontinuableError
%

classmethod: 
installPartialContinuation: aGsProcess atLevel: targetLevel value: anObject

" aGsProcess must be a partial continuation, i.e., the result from
  partialContnuationFromLevel:to:, or an error will be generated.

  Replaces targetLevel frames of stack with partialContinuation
  and resumes execution with top-of-stack to be replaced by anObject."

<primitive: 2031>
self _uncontinuableError  "should never reach here"
%


! ------------------- Instance methods for GsProcess
! deleted _basicControlStackEntrySize , _controlStackEntrySize

category: 'Accessing'
method:
_cannotReturn: aValue

"Raises an error message in the event that the virtual machine cannot 
 resume after the current method or block context."

^ self _error: #rtErrCantReturn
%

category: 'Accessing'
method:
localStackDepth

"Returns the local stack depth of the receiver, or 0 if receiver is running.
 If receiver not running but has an in-memory stack, forces the receiver out
 to object memory.  If receiver not running but is locked in memory,
 this method will unlock it.
 Returns 0 if receiver was found in memory but has been terminated, or
 has never been run.
"

<primitive: 719>  "process push to OM, primitive always fails"
| uacd |
(uacd := uaCount_depth) ifNil:[
  (self _isTerminated or:[arStack == nil]) ifTrue:[ ^ 0 ].
  ^ nil
].
^ uacd bitAnd: 16rFFFFFFFF
%

! setter   localStackDepth:  inline within _localTrimStackToLevel: only

category: 'Accessing'
method:
interruptFlag

"Return the interruptFlag value"
^ interruptFlag_stEntryCount bitShift: -32
%

category: 'Updating'
method:
setInterruptFlag: anInt

" set the interruptFlag value. Receiver is expected to have been suspended."
| val |
stackId >= 0 ifTrue:[ self error:'cannot update C stack state correctly' ] .
val := interruptFlag_stEntryCount bitAnd:(16rFFFFFFFF). "keep stEntryCount"
val := val bitOr:(( anInt bitAnd:16rFFFFFFF) bitShift: 32) .
interruptFlag_stEntryCount := val 
%

category: 'Accessing'
method:
stackDepth

"Returns the stack depth of the receiver."

| result |
result := self localStackDepth .
^ result
%

category: 'Accessing'
method:
methodAt: aLevel

"Returns the GsNMethod that is active at aLevel in the receiver, where
 aLevel == 1 is the top of the stack.  Generates an error if aLevel less than
 zero or greater than stackDepth.  Returns nil if there is a reenter marker at
 the specified level."

^ self localMethodAt: aLevel
%

category: 'Accessing'
method:
localMethodAt: aLevel

"Returns the GsNMethod that is active at aLevel in the receiver, where
 aLevel == 1 is the top of the stack.  Generates an error if aLevel less than
 zero or greater than stackDepth.  Returns nil if there is a reenter marker at
 the specified level."

| fpIdx dp |

(aLevel < 1 or:[ aLevel > (dp := self localStackDepth) ]) ifTrue:[
  aLevel _error: #rtErrArgOutOfRange args:{ 1 . dp } .
  ].
fpIdx := self _frameOffsetAt: aLevel .
fpIdx ifNil:[ ^ nil].
^ self _fetchMethod:( arStack at: (fpIdx + FP_codePtr_OFS )) .
%

category: 'Accessing'
method:
_fetchMethod: aCodePtr

"translate a codePtr to a GsNMethod"
| codeCls  |
codeCls := aCodePtr class . 
codeCls == GsNMethod ifTrue:[
 ^ aCodePtr
] ifFalse:[
  self _halt: 'should not be here' .
  ^ nil
].
%

category: 'Private'
method:
_toWordOfs: aByteOffset

^ aByteOffset bitShift: -3
%

category: 'Accessing'
method:
_frameDescrAt: aLevel

"Private.  
 Returns nil if aLevel is less than zero or greater than stackDepth, 
 otherwise returns an Array  { aLevel . fpIdx . calleeFpIdx } ,
 where fpIdx is one-based frameOffset at aLevel in the receiver,
 and  where aLevel == 1 is top of stack.
 If aLevel == 1, calleeFpIdx will be zero.

 (self.arStack at: fpIdx) is the savedFP for aLevel.
 if aLevel > 1, (self.arStack at: calleeIdx) is the savedFP for aLevel - 1.
 "
| fpIdx calleeFpIdx cache dp |

(aLevel < 1 or:[ aLevel > (dp := self localStackDepth) ]) ifTrue:[ 
  ^ nil.  
].
(cache := dbgFpsCache) ifNil:[  | ar_size idx ar_stk depth |
  (ar_stk := arStack) ifNil:[ ^ nil ].
  ar_size := ar_stk size .
  fpIdx := ar_size + (self _toWordOfs:topFpOffset) + 1.
  depth := dp .
  cache := Array new: depth .
  cache at: 1 put: fpIdx .
  idx := 1 .
  [ idx < depth ] whileTrue:[ | callerFpOfs |
    idx := idx + 1 .
    callerFpOfs := ar_stk at: fpIdx"+ FP_savedFP_OFS==0" .
    calleeFpIdx := fpIdx .
    fpIdx := ar_size + (self _toWordOfs:callerFpOfs) + 1.
    cache at: idx put: fpIdx .
  ].
  dbgFpsCache := cache .
].
fpIdx := cache at: aLevel .
calleeFpIdx := aLevel > 1 ifTrue:[ cache at: aLevel - 1 ] ifFalse:[ 0 ] .
^ { aLevel . fpIdx . calleeFpIdx }
%

category: 'Accessing'
method:
_frameOffsetAt: aLevel

"Private.
 Returns nil if aLevel is less than zero or greater than stackDepth,
 otherwise returns  idx ,
 where idx is one-based frameOffset at aLevel in the receiver,
 and  where aLevel == 1 is top of stack.

 In self.arStack at idx is the savedFP for aLevel."

| arr |
arr := self _frameDescrAt: aLevel .
arr ifNil:[ ^ nil ].
^ arr at: 2
%

category: 'Accessing'
method:
_ipOffsetAt: aLevel 

"return the portable code IP offset for the specified frame"
^ self _ipOffsetFromFrameDescr: (self _frameDescrAt: aLevel)

%

category: 'Accessing'
method:
_ipOffsetFromFrameDescr: frDescr

"return the portable code IP offset for frame described by 
 frDescr.  frDescr must be non-nil result from _frameDescrAt: 

 if receiver is a stack from native code, result maybe approximate
"

| ipOff aLevel notTopFrame |
aLevel := frDescr at: 1.
(notTopFrame := aLevel >  1) ifTrue:[ | calleeFpIdx |
  calleeFpIdx := frDescr at: 3 .
  ipOff := arStack at:( calleeFpIdx + FP_rtnAddr_OFS ) .
] ifFalse:[
  ipOff := arStack at: 1 "savedIP at TOS" .
].
(ipOff == nil or:[ ipOff < 0]) ifTrue:[
  "due to biasing of return addresses for native code, can have negative
   offset for start of method, so treat as zero for stack display.
   can have nil for reenter markers. "
  ipOff := 0
].
self _nativeStack ifTrue:[ | fpIdx meth anIp |
  fpIdx := frDescr at: 2 .
  meth := self _fetchMethod:( arStack at:( fpIdx + FP_codePtr_OFS )).
  " arg to asReturn: is false here because callers (or topaz) are using one of
      GsProcess>>_StepPointAt:
      GsNMethod>>_stepPointForIp:level:useNext:
    to convert the IP to a step point
  " 
  anIp := meth _nativeIpOffsetToPortable: ipOff asReturn: false . 
  anIp <= 0 ifTrue:[
    anIp < -1 ifTrue:[
      anIp := anIp negated.  "approximate result"
    ] ifFalse:[
      self error:'ipOff=', anIp asString, ' out of range' .
    ].
  ].
  ipOff := anIp .
].
^ ipOff .
%


category: 'Copying'
method:
copy

"Disallowed."

self shouldNotImplement: #copy
%

category: 'Updating'
method:
instVarAt: anIndex put: aValue

"Disallowed."

self shouldNotImplement: #instVarAt:put:
%

! fixed 44440
category: 'Debugging Support'
method:
_reportOfSize: aSize

"Returns an Array describing the receiver up to aSize entries in length.  Each
 element in the result is a String produced by GsProcess | _reportAt:."

| result dp |
result := { }  .
dp := self == GsProcess _current ifTrue:[ System stackDepth ]
                                ifFalse:[ self stackDepth ].
1 to: (dp min: aSize) do:[:j |
   (self _reportAt: j) ifNotNil:[:r | result add: r ]
].
^ result
%

category: 'Debugging Support'
method:
_reportAt: aLevel

"Return a one line String describing the specified level in the receiver.
 aLevel == 1 means top of stack with respect

 If aLevel is out of range,  nil is returned. "

| fc isCurr |
(isCurr :=  self == GsProcess _current) ifTrue:[
   (fc := GsProcess _frameContentsAt: aLevel ) ifNil:[ ^ nil ].
   fc at: 8 put: (GsProcess _selfAt: aLevel) . "get self"
] ifFalse:[
   (fc := self _frameContentsAt: aLevel) ifNil:[ ^ nil ].
].
^ self _reportWithFrameContents: fc level: aLevel  .
%

classmethod: 
anonymousSelectors
  "will be reimplemented for ruby"

  ^ #( )
%


! fixed 41002 , 41357 . added ruby support
! fixed 42802 
! fixed 43651
category: 'Debugging Support'
method: 
_trimStackToLevel: aLevel

"Deletes stack entries from 1 to (aLevel - 1) inclusive, thus making aLevel the
 new top-of-stack(TOS).  At new TOS, looks in the implementation class of the
 method at TOS, using the selector of the method at TOS.  If that class's method
 dictionary contains a different GsNMethod, the GsNMethod currently in the method
 dictionary is installed as the method at the TOS.  The saved instruction
 pointer for TOS is set to the entry of the method at TOS.

 Limitations:
   If the receiver was executing in native code, and the stack
   is trimmed, any resumption of execution will be in interpreted mode.

   If the new top-of-stack is an anonymous method, it is not possible to
   determine whether that method has been recompiled, and the method at new top
   of stack will not be changed.  Debuggers should use the
   GsNMethod | _recompileWithSource method to implement recompilation from a
   debugger pane.  _recompileWithSource: raises an error on attempts to
   recompile an anonymous method.

   Raises an error if the new top-of-stack would represent the entry to an
   ExecBlock and the home method of the block has been recompiled.

   Raises an error if the new top-of-stack would represent a return
   across an FFI , Ruby C extension, or UserAction call.

 If aLevel = 1, reset instruction pointer to beginning method.

 Debuggers must not cache or directly manipulate VariableContexts
 when examining or altering stacks.

 Provides the ability for a debugger to restart execution after recompiling
 a method that is active on the stack of the receiver."

 | fpIdx aMeth oldHome oldmClass newTosIdx oldDepth newDepth newTosFpOfs
   prevFpIdx envId |
  self convertToPortableStack: aLevel .
  
  "check for argument aLevel out of range"
  oldDepth := self localStackDepth .
  aLevel > oldDepth  ifTrue:[ ^ self ].
  newDepth := oldDepth - (aLevel - 1) .

  1 to: aLevel do:[:n|
    prevFpIdx := fpIdx .
    fpIdx := self _frameOffsetAt: n  .
    aMeth := self _fetchMethod:( arStack at:( fpIdx + FP_codePtr_OFS)) .
    aMeth ifNil:[ Error signal:'Cannot trim stack across a reenter marker'].
  ].

  "check to see whether new TOS method has been recompiled, and if so,
   install the new method."
  oldHome := aMeth homeMethod .
  oldmClass := oldHome inClass .
  envId := (oldHome respondsTo: #environmentId) 
            ifTrue: [oldHome environmentId] ifFalse: [0].
  (oldmClass == nil or: [self class anonymousSelectors includes: oldHome selector]) ifFalse: [
    "not an anonymous method"
    | newMeth oldCptr |
    newMeth := oldmClass compiledMethodAt: oldHome selector environmentId: envId
			otherwise: aMeth .
    newMeth == aMeth ifFalse:[
      aMeth == oldHome ifFalse:[
        Error signal:'Cannot trim stack to a block in recompiled method'.
      ].
      "install recompiled method"
      oldCptr := arStack at: (fpIdx + FP_codePtr_OFS) .
      oldCptr class == GsNMethod ifTrue:[
        arStack at: (fpIdx + FP_codePtr_OFS) put: newMeth .
      ] ifFalse:[
        self _halt:'should not be here'
      ].
    ].
  ].
  newTosIdx := fpIdx + FP_codePtr_OFS .
  "push savedIP which is byte offset to first instr in the method"
  newTosIdx := newTosIdx - 1 .
  arStack at: newTosIdx put: OC_GSNMETHOD_FIRST_INSTR_OFFSET .
  aLevel > 1 ifTrue:[
    "don't need  _tosIsIpNil in arStack object"
    newTosFpOfs := arStack at: prevFpIdx .
    topFpOffset := newTosFpOfs .  "adjust offset to Top FP"
    "adjust the depth of the receiver."
     uaCount_depth := ((uaCount_depth bitShift: -32)  bitShift:32)
         bitOr: (newDepth bitAnd: 16rFFFFFFFF) ].
  newTosIdx > 1 ifTrue:[ 
    "delete the unwanted stack frames.
     Since FP linkages in arStack are relative to   arStack size
       there is no other fixup needed. "
    arStack removeFrom:1 to: newTosIdx - 1 .
  ].
  dbgFpsCache := nil . "clear frame offsets cache"
%

! fixed 32225
category: 'Debugging Support'
method:
_gsiStackReportFromLevel: startLevel toLevel: stopLevel

"Returns an Array describing the receiver.  For each stack level in the
 receiver, the result contains 2 elements,  a String describing that
 level, and an Array containing the result of the _gsiDebuggerDetailedReportAt:
 method for that level.

 Level 1 is the top of the stack.  If startLevel is out of range, or if
 stopLevel is less than startLevel, returns nil.  If stopLevel is beyond the end
 of the stack, returns information up to the end of the stack."

|result actualStop frameContents dp |

stopLevel < startLevel ifTrue:[ ^ nil ].
startLevel > (dp := self localStackDepth) ifTrue:[ ^ nil ].

actualStop := (stopLevel > dp ) 
  ifTrue: [ dp ] 
  ifFalse: [ stopLevel ].

result := { } .
startLevel to: actualStop do:[:j| 
  frameContents := self _frameContentsAt: j .
  frameContents 
    ifNil:[ result addLast: nil; addLast:nil ]
    ifNotNil:[ | frameStr |
      frameStr := String new .
      frameStr addAll: (self _reportWithFrameContents: frameContents level: j) .
      result addLast: frameStr ;
             addLast:(self _gsiDetailedReportWithFrameContents: frameContents
                            forLevel: j ) .
      ].
  ].
^ result
%

! fixed 45218
category: 'Debugging Support'
method:
_gsiDebuggerDetailedReportAt: aLevel

"If aLevel is less than 1 or greater than stackDepth of the receiver, 
 returns nil.  Otherwise, returns an Array containing:

 offset item
 ------ ----
   1    gsMethod  (a GsNMethod)
   2    receiver
   3    self
   4    selector  (a Symbol)
   5    quickStepPoint (SmallInteger index into sourceOffsets, or nil)
   6    sourceOffsets  (Array of SmallIntegers)
   7    argAndTempNames  (Array of Symbols)
   8    argAndTempValues (Array) (may be smaller than argAndTempNames if
   execution halted at entry to a method, may be larger than
         argAndTempNames if compiler has allocated additional
   unnamed stack temporaries. For aLevel == 1, Last element is TOS )
   9    sourceString
  10    ipOffset   in portable code (SmallInteger)
  11    markerOrException  (SmallInteger in a reenter marker, 
         or an Exception, or nil )

 The quickStepPoint is the step point in the method, if the method has 
 50 or less step points.  Otherwise, the quickStepPoint will be nil
 and the expression

 gsMethod _stepPointForIp: ipOffset level: aLevel 
    useNext: (self _nativeStack or:[ self _calleeIsAsync: aLevel])

 may be used to obtain search for and obtain the step point."
| frameContents |

(frameContents := self _frameContentsAt: aLevel) ifNil:[ ^ nil ].
^ self _gsiDetailedReportWithFrameContents: frameContents
             forLevel: aLevel
%

category: 'Debugging Support'
method:
_reportString

"Returns a String with each Activation object in the receiver, up to 2000
 levels, described on separate lines."

| s arr |
s := String new.
s add: self class name.
s add: '(oop=' ; add: self asOop printString ; add: ', ' .

s add: 'status=' ; add: (self _statusString) ; add: ', '.

clientData ifNotNil:[
  s add: 'clientData=' ; add: (clientData printString) ; add: ', '.
].

s add: 'priority=' ; add: (self priority printString) ; add: ', '.

group ifNotNil:[
  s add: 'group=' ; add: (group printString) ; add: ', '.
].

block ifNotNil:[
  s add: 'block=' ; add: (block printString) ; add: ', '.
  s add: 'args=' ; add: (args printString) ; add: ', '.
].
s lf .

arr := self _reportOfSize: 2000.
arr size > 0 ifTrue:[
  1 to: arr size do:[:j | s add: '   ' ; add: (arr at: j) ; lf ].
  s add: $) ; lf .
] ifFalse:[
  s add: $) 
].
^ s
%

category: 'Formatting'
method:
printString

"Returns a String whose contents are a displayable representation of the
 receiver."

^ self _reportString
%

category: 'Formatting'
method:
printOn: aStream

"Puts a displayable representation of the receiver on the given stream."

aStream nextPutAll: self _reportString
%

! edited for 44225 , 48068
method: GsProcess
_frameContentsAt: aLevel

"Private.  Returns an Array describing the specified level in the receiver. 
 aLevel == 1 is top of stack.  If aLevel is less than 1 or greater than
 stackDepth, returns nil.  

 The result Array contains:
 offset item
 -----  -----
   1    gsMethod  (a GsNMethod)
   2    ipOffset    (absolute instruction offset in portable code
                     negative means a stack breakpoint is present)
   3    frameOffset (always nil , not compatible with Gs64 v2.x)
   4    varContext
   5    saveProtectedMode (always nil in v3.0)
   6    markerOrException (always nil in v3.0)
   7    homeMethod if gsMethod is for a block, otherwise nil .
   8    self   (possibly nil in a ComplexBlock)
   9    argAndTempNames   (an Array of Symbols or Strings)
  10    receiver
  11... arguments and temp values, if any"
 
"Notes to GemStone implementors:

 If result of this method is changed, you must change tpaux.c in the 
 topaz C sources, other methods in this class, and the code for primitive 195.

 Note that if execution stops at a breakpoint at the entry to a method,
 the method temporaries may not be allocated yet and so some or all of
 the method temporaries may be missing from the result."

| result fpIdx theMeth theVc argsTmpsOfss ofsIdx frDesc vcOfs 
  codePtr theIp absIp |

(aLevel < 1 or:[ aLevel > self localStackDepth ]) ifTrue:[
  ^ nil .
].
frDesc := self _frameDescrAt: aLevel .
frDesc ifNil:[ ^ nil ].
fpIdx := frDesc at: 2 .
result := Array new: 10 "virtual machine constant" .
arStack ifNotNil:[ 
  codePtr := arStack at:( fpIdx + FP_codePtr_OFS ).
  theMeth := self _fetchMethod: codePtr  .
].
result at: 1 put:  theMeth .
theIp := (self _ipOffsetFromFrameDescr: frDesc) .
result at: 2 put: theIp .
absIp := theIp abs .
result at: 3 put: nil.  "previous semantics of frameOffset not usable"

vcOfs := fpIdx + FP_VC_OFS .
arStack ifNotNil:[ theVc := arStack at: vcOfs ].

(Reflection classOf: theVc) == VariableContext ifTrue:[ 
  result at: 4 put: theVc 
] ifFalse: [
  theVc := nil "IP at start of method, VC not allocated yet"
].
"result at: 5 , leave as nil,  saveProtectedMode not available"
theMeth ifNotNil:[
  | homeMeth nArgs receiver theSelf argAndTempNames lastTmpIdx |
  homeMeth := theMeth homeMethod .
  homeMeth ~~ theMeth ifTrue:[ result at: 7 put: homeMeth ].

  nArgs := theMeth numArgs .
  receiver := arStack at:(fpIdx + FP_lastArg_OFS + nArgs ) .
  result at: 10 put: receiver .
  (receiver _isExecBlock and:[theMeth inClass ~~ ExecBlock]) ifTrue:[
    theSelf := receiver selfValue  .
  ] ifFalse:[
    theSelf := receiver 
  ].
  result at: 8 "virtual machine constant" put: theSelf. 

  argAndTempNames := theMeth  argsAndTemps .
  argAndTempNames size == 0 ifTrue:[ argAndTempNames := { } ] .
  result at: 9 put: argAndTempNames .

  argsTmpsOfss := theMeth _argsAndTempsOffsets .
  lastTmpIdx := argsTmpsOfss size - theMeth _numCopyingBlockArgs .
  ofsIdx := 1 .
  1 to: nArgs do:[:k | | anOfs xofs |
    anOfs := argsTmpsOfss at: ofsIdx  .
    xofs := anOfs bitShift:-8 .
    result addLast:( arStack ifNotNil:[:ars | ars at:(fpIdx + xofs )] ) .
    ofsIdx := ofsIdx + 1 .
  ].
  absIp <= OC_GSNMETHOD_FIRST_INSTR_OFFSET ifTrue:[
    "temps not allocated yet"
    argAndTempNames size > nArgs ifTrue:[
      argAndTempNames size: nArgs
    ]. 
  ] ifFalse:[ | aOfs tmpVal lastStkTmpIdx calleeLastArgIdx calleeFpIdx tIdx |
    [ ofsIdx <= lastTmpIdx ] whileTrue:[
      aOfs := argsTmpsOfss at: ofsIdx  .
      tmpVal := nil .
      aOfs < 0 ifTrue:[ | idx |
  " temp on evaluation stack"
        idx := fpIdx + (aOfs bitShift:-8) .
        lastStkTmpIdx ifNil:[ lastStkTmpIdx := idx ]
           ifNotNil:[ lastStkTmpIdx := lastStkTmpIdx min: idx ].
  tmpVal := arStack ifNotNil:[:ars | ars at: idx ].
      ] ifFalse:[
  aOfs > 0 ifTrue:[  | ofs lev aVc |
          ofs := aOfs bitShift: -8 .
          lev := aOfs bitAnd: 16rFF .
          aVc := theVc .
          [ lev > 0 ] whileTrue:[
            aVc := aVc ifNotNil:[ aVc parent ] .
            lev := lev - 1 .
          ].
    " temp allocated in a VariableContext, 
            allow for theVc==nil to handle corrupted stacks during development. "
    tmpVal := aVc ifNotNil:[ aVc _primitiveAt: (ofs + 1)] ifNil:[ nil] 
  ] ifFalse:[
    self error:'illegal offset'.
  ].
      ].
      result addLast: tmpVal .
      ofsIdx := ofsIdx + 1 .
    ].
   "maglev only"
   "[ ofsIdx <= argsTmpsOfss size ] whileTrue:[ //handle copying Block args
      aOfs := argsTmpsOfss at: ofsIdx .
      tmpVal := receiver at: aOfs + 1 .
      result addLast: tmpVal .
      ofsIdx := ofsIdx + 1 .
    ]. "
    calleeFpIdx := frDesc at: 3 .
    calleeLastArgIdx := calleeFpIdx + FP_lastArg_OFS .
    tIdx := 1 .
    lastStkTmpIdx ifNil:[ lastStkTmpIdx := fpIdx + FP_codePtr_OFS ].
    (lastStkTmpIdx - 1) to: calleeLastArgIdx by: -1 do:[:idx |
      tmpVal := arStack ifNotNil:[:ars | ars at: idx ].
      result addLast: tmpVal .
      argAndTempNames add: '.t' , (tIdx asString ) .
      tIdx := tIdx + 1 .
    ].  
  ].
].
^ result
%

category: 'Private Debugging Support'
method:
_topazDescrForFrame: frameId padTo: minSize
  "Used by topaz stack display code .
   Returns nil if frameId is out of range, or an Array
    { aGsNMethod . aString } .
   aGsNMethod in the result is nil if frame is a reenter marker.
  " 
  | frame aMethod |
  frame := self _frameContentsAt: frameId .
  frame ifNil:[ ^ nil ].
  aMethod := frame at: 1 . 
  aMethod ifNil:[ ^ { nil . 'reenter marker' } ].
  ^ { aMethod .
      aMethod _descrForStackPadTo: minSize rcvr: (frame at: 10 "rcvr") 
    }
%


! fixed  45301
category: 'Debugging Support'
method:
_frameAt: aLevel offsetOfTempNamed: aString

"For the method activation at level aLevel, returns a SmallInteger which
 is the 1-based offset of the method argument or temporary with name aString.  
 Returns 0 if no temp exists at aLevel with name aString. 
 
 Generates an error if aLevel is out of range."

| frameContents aSym tempNames dp |

(aLevel < 1 or:[ aLevel > (dp := self localStackDepth) ]) ifTrue:[
  aLevel _error: #rtErrArgOutOfRange args:{ 1 . dp } .
  ].

aSym := Symbol _existingWithAll: aString .
aSym ifNil:[ ^ 0 ].
frameContents := self _frameContentsAt: aLevel .
tempNames := frameContents at: 9 "virtual machine constant" .
^ tempNames indexOf: aSym
%

! fix 45301
category: 'Debugging Support'
method: 
_frameAt: aLevel tempAt: anOffset put: aValue

"In the method activation at level aLevel, alter the method argument or
 temporary at anOffset to have value aValue.
 
 Generates an error if aLevel or anOffset is out of range."

| frDesc fpIdx codePtr theMeth nArgs argAndTempNames vcOfs
  theVc argsTmpsOfss absIp lastTmpIdx sd |
(aLevel < 1 or:[ aLevel > (sd := self localStackDepth) ]) ifTrue:[
  aLevel _error: #rtErrArgOutOfRange args:{ 1 . sd } .
].
frDesc := self _frameDescrAt: aLevel .
fpIdx := frDesc at: 2 .
codePtr := arStack at:( fpIdx + FP_codePtr_OFS ).
theMeth := self _fetchMethod: codePtr  .
  nArgs := theMeth numArgs .
argAndTempNames := theMeth  argsAndTemps .
vcOfs := fpIdx + FP_VC_OFS .
theVc := arStack at: vcOfs .
(theVc class == VariableContext) ifFalse:[ theVc := nil ].
argAndTempNames := theMeth  argsAndTemps .
argAndTempNames ifNil:[ argAndTempNames := { } ] .
argsTmpsOfss := theMeth _argsAndTempsOffsets .
anOffset <= nArgs ifTrue:[ | anOfs |
  anOfs := argsTmpsOfss at: anOffset  .
  arStack at:(fpIdx + (anOfs bitShift:-8) ) put: aValue .
  ^ self
].
absIp := (self _ipOffsetFromFrameDescr: frDesc) abs .
absIp <= OC_GSNMETHOD_FIRST_INSTR_OFFSET ifTrue:[
  ArgumentError signal:'method temps not allocated yet' . 
  ^ nil
].
lastTmpIdx := argsTmpsOfss size - theMeth _numCopyingBlockArgs .
anOffset <= lastTmpIdx ifTrue:[ | aOfs |
  aOfs := argsTmpsOfss at: anOffset  .
  aOfs < 0 ifTrue:[
    " temp on evaluation stack"
    arStack at: (fpIdx + (aOfs bitShift:-8) ) put: aValue .
    ^ self
  ] ifFalse:[
    aOfs > 0 ifTrue:[ | lev ofs aVc |
      " temp allocated in a VariableContext,
         allow for theVc==nil to handle corrupted stacks during development. "
      ofs := aOfs bitShift: -8 .
      lev := aOfs bitAnd: 16rFF . 
      aVc := theVc .
      [ lev > 0 ] whileTrue:[
	aVc := aVc ifNotNil:[ aVc parent ] .
	lev := lev - 1 .
      ].
      aVc ifNotNil:[ 
         aVc _primitiveAt: (ofs + 1) put: aValue . ^ self 
      ] ifNil:[ ArgumentError signal:'theVc not found'. ^ nil ].
    ] ifFalse:[
      ArgumentError signal:'internal error, invalid offset for VC access'.
      ^ nil
    ].
  ].
].
anOffset <= argsTmpsOfss size ifTrue:[ "handle copying Block args"
  | aOfs rcvr |
  rcvr := arStack at:(fpIdx + FP_lastArg_OFS + nArgs ) .
  aOfs := argsTmpsOfss at: anOffset .
  rcvr at: aOfs + 1 put: aValue .
  ^ self 
].
ArgumentError signal:'method temp or arg not found' .
^ nil
%

category: 'Private'
method:
_getEnsureBlocks

"For a GsProcess returns an Array of pairs,
     FP offset,  block that was arg to an ensure:  
 for the ensure blocks currently on the stack of the receiver"

<primitive: 826>  "primitive fails if receiver not in stack memory"
| ar_stk pairs fpOfs ar_size |
pairs := { }  .
(ar_stk := arStack) ifNotNil:[  "handle a suspended process not in stack memory"
  fpOfs := topFpOffset .
  ar_size := ar_stk size .
  [ fpOfs < 0 ] whileTrue:[ | fpIdx arOfs anFP |
    fpIdx := ar_size  + (anFP := self _toWordOfs: fpOfs) + 1 .
    arOfs := fpIdx + FP_markerNil_OFS .
    arOfs >= 1 ifTrue:[ | marker aBlk |
      marker := ar_stk at: arOfs .
      marker == OOP_ENSURE_Mark_NIL ifTrue:[
        aBlk := ar_stk at:(fpIdx + FP_lastArg_OFS).
        aBlk _isExecBlock ifTrue:[
          pairs add: anFP ; add: aBlk .
        ] ifFalse:[
          self _halt:'illegal class of argument' .
        ].
      ].
    ].
    fpOfs := ar_stk at: fpIdx"+ FP_savedFP_OFS==0" . "get callerFpOfs"
  ].
].
^ pairs
%
method:
_removeEnsureAtFP: anFP

 "Dereference the ensure block in specified frame,
  so it won't be executed more than once.
  Frame changed  from  ensure:[] to  ensure: nil ."

 <primitive: 958>
 "handle receiver not in stack memory "
 | ar_stk ar_size fpIdx arOfs |
 anFP _isSmallInteger ifTrue:[
   ar_stk := arStack .
   ar_size := ar_stk size .
   fpIdx := ar_size  + anFP + 1.
   arOfs := fpIdx + FP_markerNil_OFS .
   arOfs >= 1 ifTrue:[ | marker |
     marker := ar_stk atOrNil: arOfs .
     marker == OOP_ENSURE_Mark_NIL ifTrue:[
      ar_stk at:(fpIdx + FP_lastArg_OFS) put: nil .
      ^ self
     ]
   ]
 ].
 self _primitiveFailed: #_removeEnsureAtFP: args: { anFP }
%


category: 'Debugging Support'
method:
_stepPointAt: aLevel

"Used by topaz debugger,
 returns a stepPoint , or nil if no step point available "

^ (self _localStepPointAt: aLevel) at: 1
%

category: 'Debugging Support'
method:
_calleeIsAsync: aLevel
  self _current == self ifTrue:[ ^ false "an approximation"].
  aLevel > 1 ifTrue:[
    (self _frameDescrAt: aLevel -1) ifNotNil:[:calleeDesc |
      (self _fetchMethod:(arStack at:(calleeDesc at:2) + FP_codePtr_OFS)) ifNotNil:[:calleeMeth|
        | sel |
        sel := calleeMeth selector .
        (sel == #_signalAsync or:[ sel == #_signalGcFinalize or:[ sel == #_signalTimeout]]) ifTrue:[
           ^ calleeMeth inClass == AbstractException 
        ]
      ]   
    ]
  ].
  ^ false
%

category: 'Debugging Support'
method:
_localStepPointAt: aLevel

"Used by topaz debugger,
 returns an array { stepPoint . aGsNMethod }
 or nil if no step point available "

| fpIdx theMeth  ipOffset stepPt frDesc dp |
(aLevel < 1 or:[ aLevel > (dp := self localStackDepth) ]) ifTrue:[
  aLevel _error: #rtErrArgOutOfRange args:{ 1 . dp } .
  ].
(frDesc := self _frameDescrAt: aLevel) ifNil:[ ^ nil ].
fpIdx := frDesc at: 2 .
theMeth := self _fetchMethod:( arStack at:( fpIdx + FP_codePtr_OFS)) .
theMeth ifNil:[ ^  nil  ].
ipOffset := self _ipOffsetFromFrameDescr: frDesc .
stepPt := theMeth _stepPointForIp: ipOffset level: aLevel  
        useNext: (self _nativeStack or:[ self _calleeIsAsync: aLevel]) .
stepPt ifNil:[ ^ nil ].
^ { stepPt . theMeth }
%

category: 'Debugging Support'
method:
_stepPointStringAt: aLevel

"Used by topaz debugger."

| stepPtArr result stepPoint theMeth |
stepPtArr := self _localStepPointAt: aLevel .
stepPtArr ifNil:[ ^ '' ].

stepPoint := stepPtArr at: 1 .
theMeth := stepPtArr at: 2 .
result := String withAll:'@' .
result addAll: stepPoint asString ;
    addAll: ' line ' ; 
    addAll: (theMeth _lineNumberForStep: stepPoint) asString .
^ result

%

! edited for 41422 , 41220
category: 'Private Debugging Support'
method:
_setBreaksForStepInto

"Set breakpoints so that a subsequent IntContinue on the receiver will
 execute a step-into.  step-into across a fork of a new GsProcess
 is not supported.  (The step breakpoint in the forked block will
 be ignored since it not in the current GsProcess).

 For use only from within implementation of GciStep.

 Return a 3 element Array, 
     { <the GsNMethod in which single-step breaks were set> .
       <the level at which stack break was set, or zero> .
       <a SmallInteger, number of breakpoints to ignore> }

Algorithm
Step Into (is always from TOS == level 1)
  1. set all step breakpoints in TOS.codeOop
  2. if next instruction to execute is BC_CALL_PRIMITIVE,
      set all step breakpoints in level 2 and set stack breakpoint in level 3
    else
      set stack breakpoint at level 2
"

| tosFrame aMethod stackBreakLevel result breakpointsToIgnore ipOfs |

self convertToPortableStack .

result := Array new: 3 .
tosFrame := self _frameContentsAt: 1 .
aMethod := tosFrame at: 1 "virtual machine constant" .
ipOfs := tosFrame at: 2 "virtual machine constant" "ipOffset" .
(aMethod _opcodeAtIsCallPrimitive: ipOfs) ifTrue:[
  "next instruction to be executed is a call-primitive"
  aMethod := self localMethodAt: 2 .
  aMethod _setAllStepBreaks: true frame: nil process: self .
  result at: 1 put: aMethod.
  breakpointsToIgnore := 0.
  stackBreakLevel := 3  .
] ifFalse:[
  "next instruction is not a primitive call"
  aMethod _setAllStepBreaks: false frame: nil process: self .
  result at: 1 put: aMethod .
  breakpointsToIgnore := 0 .  "fix 41539"
  stackBreakLevel := 2 .
].
           " use <  in following compare to not set stack break in _gsReturnToC"
stackBreakLevel < self stackDepth      
  ifTrue:[ self _setStackBreakAt: stackBreakLevel ]
  ifFalse:[ stackBreakLevel := 0 "no stack breaks to set" ].
result at: 2 put: stackBreakLevel .
result at: 3 put: breakpointsToIgnore.
^ result 
%

! edited for 41422 , 45320, 45472
category: 'Private Debugging Support'
method: GsProcess
_setBreaksForStepOverFromLevel: flags

"Set breakpoints so that a subsequent IntContinue on the receiver will
 execute a step-over.  
 For use only from within implementation of GciStep.
 Return an Array, { <the GsNMethod in which single-step breaks were set> .
        <the level at which stack break was set, or zero> .
                    <number of breakpoints to ignore> }."
| aFrame aMethod stackBreakLevel result breakpointsToIgnore aLevel thruBool fpOffset 
  trc doTrace |
doTrace ifNotNil:[ trc := String new ].
self convertToPortableStack .
result := Array new: 3 .
aLevel :=    flags bitAnd:  16rFFFFFFFF .
thruBool := (flags bitAnd: 16r100000000) ~~ 0 .
aFrame := self _frameContentsAt: aLevel .
aMethod := aFrame at: 1 "virtual machine constant" .
fpOffset := self _toWordOfs: ( aLevel == 1 ifTrue:[ topFpOffset ]
                       ifFalse:[ ((self _frameDescrAt: aLevel) at: 3) ]).
aLevel >= 2 ifTrue:[
  stackBreakLevel := aLevel + 1.
  breakpointsToIgnore := 0 .
  aMethod _setStepOverBreaks: false frame: nil process: self stepThrough: thruBool .
  doTrace ifNotNil:[ trc add: $A ].
] ifFalse:[ "aLevel is 1" | hmm |
  "set breaks in TOS method, excluding first step point"
  aMethod _setStepOverBreaks: false frame: fpOffset process: self stepThrough: thruBool .
  result at: 1 put: aMethod .
  breakpointsToIgnore := 0 .  "no special logic after lastBreakpt instVar added"
  (thruBool and:[ (hmm := aMethod homeMethod) ~~ aMethod ]) ifTrue:[ | depth j |
     depth := self localStackDepth. 
     j := 2 .
     stackBreakLevel := 0 .
     [ j < depth ] whileTrue:[
       ((self _frameContentsAt: j) at: 1) == hmm ifTrue:[
          stackBreakLevel := j . "stack break for return to/across home method"
          j := depth "break out of whileTrue" . doTrace ifNotNil:[trc add: $B ].
       ].
       j := j + 1 .
     ].
  ] ifFalse:[
    stackBreakLevel := aLevel + 1 . doTrace ifNotNil:[trc add: $C ].
  ]
].
     " use <  in following compare to not set stack break in _gsReturnToC"
doTrace ifNotNil:[
 GsFile gciLogServer: 'dp ' , self localStackDepth asString ,
        ' stackBreakLevel ' , stackBreakLevel asString ].
(stackBreakLevel > 0 and:[ stackBreakLevel <  self localStackDepth ])
  ifTrue:[ self _setStackBreakAt: stackBreakLevel . doTrace ifNotNil:[trc add: $D] ]
  ifFalse:[ stackBreakLevel := 0 "no stack breaks to set" . doTrace ifNotNil:[trc add: $E] ].
doTrace ifNotNil:[
  GsFile gciLogServer:
  'level ' , aLevel asString , ' aMethod ' , aMethod asOop asString ,
   ' hmm ' , aMethod homeMethod asOop asString ,
  ' breakpointsToIgnore ' , breakpointsToIgnore asString , 
	' stackBreakLevel ' , stackBreakLevel asString , ' trc ' , trc ].
result at: 3 put: breakpointsToIgnore .
result at: 2 put: stackBreakLevel .
^ result 
%

category: 'Private Debugging Support'
method:
__returnAddrIdxAt: aLevel

| fpIdx theMeth frDesc |
(frDesc := self _frameDescrAt: aLevel ) ifNil:[ ^ nil ].
fpIdx := frDesc at: 2 .
theMeth := self _fetchMethod:( arStack at:( fpIdx + FP_codePtr_OFS)) .
theMeth ifNil:[ ^  nil  ].
aLevel > 1 ifTrue:[ | calleeFpIdx |
  calleeFpIdx :=  frDesc at: 3 .
  ^ calleeFpIdx + FP_rtnAddr_OFS .
] ifFalse:[
  ^ 1 "savedIP at TOS"  .
]
% 

category: 'Private Debugging Support'
method:
_clearStackBreakAt: aLevel

"Clear stack breakpoint at specified level.

 Debugger implementations should use GciStep rather than invoking 
 this method directly."

| ipOffset dp |
self _nativeStack ifTrue:[
  ^ self _error: #rtErrNativeCodeNoBreakpts
].
(aLevel < 1 or:[ aLevel > (dp := self stackDepth) ]) ifTrue:[
  aLevel _error: #rtErrArgOutOfRange args:{ 1 . dp } .
  ].
(self __returnAddrIdxAt: aLevel) ifNotNil:[ :rtnAddrIdx |
  ipOffset := arStack at: rtnAddrIdx .
  ipOffset < 0 ifTrue:[
    arStack at: rtnAddrIdx put: ipOffset abs 
  ].
].
%
category: 'Private Debugging Support'
method:
_clearAllStackBreaks

  1 to: self stackDepth - 1 do:[:n | self _clearStackBreakAt: n ]
%

category: 'Private Debugging Support'
method:
_clearAllStepBreaksAt: aLevel

"Legacy code, not used in the current debugging implementation.
 Clears single step breakpoints at the specified level on the stack.  If
 aLevel is 1, clears the step-into bit in the virtual machine flag word, which
 could be left over from a step-into flag argument to a GciPerform* or
 GciExecute* call."

| meth |
aLevel == 1 ifTrue:[  | flags |
  flags := self interruptFlag .
  flags := flags bitAnd:( 16r200 bitInvert) .
  self setInterruptFlag: flags  .
].
meth := (self == GsProcess _current) ifTrue:[ GsProcess _frameContentsAt: 1 ]
                                     ifFalse:[ self localMethodAt: aLevel ].
meth ifNotNil:[ meth _clearAllStepBreaks ]
%


category: 'Private Debugging Support'
method:
_setStackBreakAt: aLevel

"Sets a stack breakpoint that is hit when a return from a
 block would return into or across the specified level on the stack.
 When a stack breakpoint is hit, the breakpoint is automatically cleared,
 the return from block is executed, and execution stops with a breakpoint
 error.  If both an unwind block (as installed by ensure:)
 and a stack breakpoint are present at the same level in the stack
 the stack breakpoint will be cleared and execution will stop with a
 breakpoint error before executing the unwind block."

| ipOffset dp |
self convertToPortableStack .

" disallow stack break at aLevel == self stackDepth
  to prevent setting set stack break in _gsReturnToC"

(aLevel < 1 or:[ aLevel >= (dp := self stackDepth) ]) ifTrue:[
  aLevel _error: #rtErrArgOutOfRange args:{ 1 . dp - 1 } .
  ].
(self __returnAddrIdxAt: aLevel) ifNotNil:[ :rtnAddrIdx |
  ipOffset := arStack at: rtnAddrIdx .
  arStack at: rtnAddrIdx put: ipOffset abs negated
].
%

! Gs64 v3.0  methods remoteProcess , _removeRemoteProcess  removed
!   GemEnterprise installation must patch the base image if remoteProcess
!   support is desired.


! changed clientData to public for 41533
category: 'Process Properties'
method:
clientData
  "answer the client data associated with the receiver.
   See also environmentAt:  "
  
  ^clientData
%
category: 'Process Properties'
method:
clientData: anobject
  "set the client data object associated with the receiver.
   See also environmentAt:put:  "
  
  clientData := anobject
%
category: 'Private'
method:
_stackSerialNum
  ^ stackSerialNum
%

category: 'Accessing'
method:
createdByApplication
  "Returns true if the receiver was created by the application.
   Returns false if the receiver was created by a Smalltalk method."

  ^(args == nil)
%
category: 'Private'
method:
_isTerminated

  "returns true if the receiver has been terminated."

  ^ (modeInfo bitAnd: ModeInfo_terminated ) ~~ 0
%
method:
_isRubyThread
  ^ (modeInfo bitAnd: ModeInfo_isRubyThread ) ~~ 0
%

category: 'Private'
method:
_terminationStarted

  "returns true if receiver has started executing ensure: blocks"
  ^ (modeInfo bitAnd: ModeInfo_terminationStarted ) ~~ 0
%


category: 'Private' 
method:
_nativeStack

  "returns true if the receiver represents execution in native code.

   Only usable on instances created by a VM stack save. "

  ^ ((modeInfo bitAnd: ModeInfo_nativeStack_mask ) bitShift: 0 - ModeInfo_nativeStack_shift) > 1"GC_GSPROCESS_cpuKind_interp"
%


category: 'Private'
method:
_setInterpretedStack

  | mask val |
  mask := ModeInfo_nativeStack_mask bitInvert .
  val :=  GC_GSPROCESS_cpuKind_interp bitShift: ModeInfo_nativeStack_shift  .
  modeInfo := (modeInfo bitAnd: mask) bitOr: val  .
%

category: 'Accessing'
method:
priority
  "Answer the scheduling priority of the receiver"
  
  | p_u |
  (p_u := priority_use) ifNil:[ ^ 15 "inline userSchedulingPriority"].
  ^ p_u bitShift: -40
%

category: 'Private'
method:
_setPriority: anInt
  "Update priority of the receiver, and of receiver's in-memory
   stack if the receiver  has one.
  "
  <primitive: 706>
  anInt _validateClass: SmallInteger .
  (anInt < 1 or:[ anInt > 40]) ifTrue:[ 
     anInt _error: #rtErrArgOutOfRange args:{ 1 . 40 } ].
  self _primitiveFailed: #_setPriority: args: { anInt }
%
category: 'Private'
method:
_setModeinfoBit: aMask value: aValue

  "Sets bit specified by aSmallInteger in  the modeInfo instVar.
   aMask must be a SmallInteger, one of
     ModeInfo_terminated - set terminated and terminationStarted bits ; 
       if the process has an in-memory stack, unlocks it and clears its use count;
       stores nil in instVars arStack,clientData,ioSelectResult . 
       if isRubyThread bit == 0, stores nil in  environment instVar.  
     ModeInfo_terminationStarted - sets specified bit
     ModeInfo_isRubyThread  - sets specified bit
     ModeInfo_breakptsIgnore_mask - sets specified bit field.
     ModeInfo_debugEnable - sets specified bit field

   aValue must be a SmallInteger >= 0 , and not larger than the specified bit field.
  "
  <primitive: 708>
  aMask _validateClass: SmallInteger .
  aValue _validateClass: SmallInteger .
  self _primitiveFailed: #_setModeinfoBit:value: args: { aMask . aValue }
%

! fixed 43699
category: 'Debugging Support'
method:
ignoreNextBreakpoint
  "Use this prior to resuming from a breakpoint if your  
   debugger is not using GciContinue to resume execution."

  ^ self _setModeinfoBit: ModeInfo_breakptsIgnore_mask value: 1 .
%
method:
disableDebugging
  "causes the receiver to ignore any breakpoints which may be set."
  ^ self _setModeinfoBit: ModeInfo_debugEnabled value: 0
%
method:
enableDebugging
  "causes the receiver to ignore any breakpoints which may be set."
  ^ self _setModeinfoBit: ModeInfo_debugEnabled value: 1 .
%

category: 'Private'
method:
_terminateCurrentWithResult: aResult

"Sets terminated and terminationStarted bits in the currently running 
  process (which may be different from the receiver),
  stores the result, and clears its waitingOn, and block.
  For the current in-memory stack, unlocks it and clears its use count.
  Stores nil in instVars arStack,clientData,ioSelectResult . 
  If modeInfo.isRubyThread bit == 0, stores nil in  environment instVar.

  Returns currently running process.
"  

  <primitive: 720>
  self _primitiveFailed: #_terminateCurrentWithResult: args: { aResult }
%

category: 'Private'
method:
_storeBit: bitNum value: aValue
  "  aValue must be true,false, or nil(bitNum 0 only) 
   bitNum 0 ==  set/clear  lockedInMemory returns previous value ;
               aValue == nil means no change and return current value.
   bitNum 1 ==  store  debuggingInProgress, per aValue 
                returns receiver ;
   bitNum 2 == set/clear THREAD_SIGNAL_EXC interrupt bit 
        returns previous value, for non-current process only.
   bitNum 3 ==  set PROC_TERMINATE_MASK interrupt bit for non-current process

   The primitive accesses/updates either C memory state or object memory
   as required by whether process is in stack memory or not.
 "
  <primitive: 707>

  self _primitiveFailed: #_storeBit:value: args: { bitNum . aValue }
%

category: 'Accessing'
method:
stackId

  "returns -1 if not in a stack area for execution.
   Only the VM can update stackId ."

  ^ stackId
%

category: 'Accessing'
method:
lockedInMemory

  ^ self _storeBit: 0 value: nil .
%

category: 'Updating'
method:
lockInMemory

 "Returns true if successful.
  Returns false if too many processes would be locked in stack memory,
  or if the receiver is not currently in stack memory."

 ^ self _storeBit: 0 value: true
%

category: 'Updating'
method:
unlockInMemory

 "If receiver is locked in stack memory, allow it to be preempted.
  Returns true if receiver was locked in stack memory."

  ^ self _storeBit:0 value: false
%

category: 'Updating'
method:
priority: anInt
  | oldPriority p_u |
      "inline _priorityRangeCheck"
  (anInt < 1 or:[ anInt > 40])  "inline lowestPriority, highestPriority" ifTrue:[
    anInt _error: #rtErrArgOutOfRange args:{ 1 . 40 } .
  ].
  (p_u := priority_use) ifNil:[ 
    oldPriority := 15 "inline userSchedulingPriority" .
  ] ifNotNil:[
    oldPriority :=  p_u bitShift: -40 .
  ].
  anInt ~~ oldPriority ifTrue:[
    self _setPriority: anInt.
    self _scheduler _changePriority: self from: oldPriority  .
    waitingOn ifNotNil:[
      waitingOn _changePriority: self from: oldPriority .
    ]
  ].
%

method:
_raisePriority
  "Raises priority of receiver to  ((priority +2) max: 40) . 
   Returns the previous priority of receiver.  "
  | old p won |
  old := self priority .
  old <= 20 ifTrue:[   "inline highestPriority logic"
    p := old + 20
  ] ifFalse:[
    p := 40
  ].
  p ~~ old ifTrue:[
    self _setPriority: p .
    self _scheduler _changePriority: self from: p .
    (won := waitingOn) ifNotNil:[
      won _changePriority: self from: old .
    ]
 ].
 ^ old
%

category: 'Accessing'
method:
waitingOn
  ^ waitingOn
%
category: 'Private'
method:
_waitingOn: anObj
  " for use by ProcessorScheduler only"
  waitingOn := anObj 
%
category: 'Accessing'
method:
onQueue
  ^ onQueue
%
category: 'Private'
method:
_onQueue: anObj
  " for use by ProcessorScheduler only"
  onQueue := anObj 
%

! deleted _nextToRun

category: 'Changing Process State'
method:
resume
  "Puts the receiver in the queue of ready processes at its priority"
  (modeInfo bitAnd: ModeInfo_continuationMask) ~~ 0 ifTrue:[
    self error:'cannot resume an instance that is a continuation' 
  ].
  self _scheduler _resumeProcess: self .
%

category: 'Debugging Support'
method:
_stackReport
  "For use with topaz"
  ^ self stackReportToLevel: 1000 withArgsAndTemps: false andMethods: true
		includeSource: false
%

category: 'Debugging Support'
method: 
stackReportToLevel: aLevel withArgsAndTemps: includeArgsAndTemps andMethods: includeMethods
  "Returns a String describing the stack of the receiver.
   The report includes frames from level 1 (top of stack) through the lesser of aLevel and 
   the stack depth.

   The format of the result is subject to change with each release of GemStone.
   aLevel should be a SmallInteger >= 1 .
   includeArgsAndTemps and includeMethods should be Booleans."

^ self stackReportToLevel: aLevel withArgsAndTemps: includeArgsAndTemps 
 	  andMethods: includeMethods includeSource: false
%

category: 'Debugging Support'
method:
stackReportToLevel: aLevel withArgsAndTemps: includeArgsAndTemps andMethods: includeMethods
includeSource: includeSource 

^ self stackReportToLevel: aLevel withArgsAndTemps: includeArgsAndTemps 
      andMethods: includeMethods includeSource: includeSource 
      lineLimit: 100"characters per line"
%

! fixed 46892
category: 'Debugging Support'
method:
_stackReportIncludeSourceFor: gsMethod inclSrc: inclSrc

  "inclSrc is a SmallInteger produced by
     stackReportToLevel:withArgsAndTemps:andMethods:includeSource:lineLimit:  .
   Returns true if source should be displayed."

  | cls sel |
  inclSrc <= 0 ifTrue:[ ^ false "none" ].
  inclSrc >= 3 ifTrue:[ ^ true  "all"  ].
  inclSrc == 1 ifTrue:[ ^  gsMethod inClass == nil  "anonymous methods"].

  "inclSrc == 2 "
  (cls := gsMethod inClass ) ifNil:[ ^ true "an anonymous method"].
  sel := gsMethod isMethodForBlock ifTrue:[ gsMethod homeMethod selector ]
                                ifFalse:[ gsMethod selector ].
  sel ifNil:[ ^ true "failsafe" ].
  (cls compiledMethodAt: sel environmentId: gsMethod environmentId otherwise: nil usePackages: true) 
      ifNotNil:[ ^ false "exists in a method dict, don't report sources"].
  ^ true  "block from block execution, etc"
%

! fixed 45218 , 45731
category: 'Debugging Support'
method: 
stackReportToLevel: aLevel withArgsAndTemps: includeArgsAndTemps andMethods: includeMethods
includeSource: includeSource lineLimit: lineLim
  "Returns a String describing the stack of the receiver.
   The report includes frames from level 1 (top of stack) through the lesser of aLevel and 
   the stack depth.

   The format of the result is subject to change with each release of GemStone.
   aLevel should be a SmallInteger >= 1 .
   includeArgsAndTemps and includeMethods should be Booleans.
   includeSource may be: false or #none   no source , 
                        true or #execution  Execution sources,
                        #blocks    Sources for blocks and methods not in method dictionary
			2  or #all  method sources. 
   lineLim is a SmallInteger specifing the maximum length in Characters 
      of each line of the report."

     | framesArr aFrame report gsMethod level inclSrc rptBlk tmpBlk |
     rptBlk := [:str | str size > lineLim 
                   ifTrue:[ report add: (str copyFrom: 1 to: lineLim) ]
                   ifFalse:[ report add: str ]] .
     tmpBlk := [:anObj | | s | s := String new .
          anObj isSpecial ifFalse:[ 
            anObj isCommitted ifTrue:[ s add: 'oop:'; add: anObj asOop asString; add:' ' ].
            anObj class isIndexable ifTrue:[ s add: 'size:'; add: anObj size asString; add:' '].
          ].
          s size > 0 ifTrue:[ s add:' '].
          s add: anObj describe.  
          s  ].
     inclSrc := includeSource .
     (inclSrc == true or:[ inclSrc == #execution]) ifTrue:[ inclSrc := 1 ] ifFalse:[ 
     (inclSrc == false or:[ inclSrc == #none]) ifTrue:[ inclSrc := 0 ] ifFalse:[
     (inclSrc == #blocks)                ifTrue:[ inclSrc := 2 ] ifFalse:[
     (inclSrc == #all or:[ inclSrc == 2 ]) ifTrue:[ inclSrc := 3 ] ifFalse:[ 
         ArgumentError signal: 'unrecognized includeSource arg ', includeSource asString]]]].
     framesArr := { } .
     level := 1.
     [ level <= aLevel and: [(aFrame := self _frameContentsAt: level) ~~ nil]] whileTrue:[
       framesArr at: level put: aFrame.
       level := level + 1
     ].
     report := String new.
     1 to: framesArr size do: [:j |
       report add: j asString; add: ' '.
       aFrame := framesArr at: j.
       (gsMethod := aFrame at: 1) ifNil: [report add: '<Reenter marker>'; lf 
       ] ifNotNil: [ 
         | stepPoint |
         report add: gsMethod _descrForStack.
         stepPoint := gsMethod _stepPointForIp: (aFrame at: 2) level: j 
             useNext: (self _nativeStack or:[ self _calleeIsAsync: j]) .
         report
           add: ' @'; add: stepPoint asString; add: ' line ';
           add: (gsMethod _lineNumberForStep: stepPoint) asString.
         includeMethods ifTrue:[
          report add: '  [GsNMethod '; add: gsMethod asOop asString; add: $]  
         ].
         report lf.
         includeArgsAndTemps ifTrue:[ | argsAndTempsNames argsAndTempsValues |
          report add: '    receiver '.
             rptBlk value: (tmpBlk value: (aFrame at: 10)) .  report lf .
           argsAndTempsNames := aFrame at: 9.
           argsAndTempsValues := aFrame copyFrom: 11 to: aFrame size.
           [ argsAndTempsNames with: argsAndTempsValues do:[:name :value |
                report add: '    '; add: name; add: ' '.  
                rptBlk value: ( tmpBlk value: value ) . 
                report  lf 
             ]
           ] on: ArgumentError do: [:ex |
             report
               add: '    <<<args and temps mismatch (';
               add: argsAndTempsNames size printString;
               add: ' versus ';
               add: argsAndTempsValues size printString;
               add: ', frame size = ';
               add: aFrame size printString;
               add: ') - ';
               add: '    names: ';
               add: argsAndTempsNames printString;
               add: '    values: '.
             rptBlk value: argsAndTempsValues describe .
             report add: '>>>'; lf .
            ]
          ].
          (self _stackReportIncludeSourceFor: gsMethod inclSrc: inclSrc) ifTrue:[
            gsMethod isMethodForBlock ifTrue:[
              report add: '  Block source:'; add:  (aFrame at: 10) _sourceString; lf
            ] ifFalse:[
              report add: (gsMethod inClass ifNil:[ '  Executed'] ifNotNil:['  Method']);
                     add: ' source: ';
                     add: gsMethod sourceString ; lf .
            ]
          ]
       ].
     ].
     ^report
%

category: 'Changing Process State'
method:
suspend
  "Suspends the receiver from processing and does not
   schedule it for further processing."

  (modeInfo bitAnd: ModeInfo_continuationMask) ~~ 0 ifTrue:[
    self error:'cannot suspend an instance that is a continuation' 
  ].
  self _scheduler _suspendProcess: self .
%

category: 'Changing Process State'

! disabled terminate9 logic for fix 42681 
! fixed 47161
method: GsProcess
terminate
 "Terminates the receiver. If receiver is not the current process,
  it will be raised to normal priority and sent a termination interrupt.
 
  If receiver is a forked process, i.e. not the GsProcess of the
  current execute or perform, returns the receiver after receiver
  has been terminated.

  If receiver is a non-forked process , then
  after completion of the termination,  a not-trappable
  an error is returned to the GCI, containing
  a not-trappable instance of TerminateProcess,
  The   instance of TerminateProcess may have a stack captured,
  but   the GsProcess associated with the error has no stack
  and is terminated."

  ^ self terminateTimeoutMs: 30000"wait up to 30 seconds for ensure blocks to complete"
%

method: GsProcess
terminateTimeoutMs: timeoutMs
 "Terminates the receiver. If receiver is not the current process,
  it will be raised to normal priority and sent a termination interrupt.

  The argument timeoutMs is a SmallInteger specifying maximum time 
  in milliseconds to wait for ensure blocks of the receiver to finish.
  If termination is not complete after this time, an Error is signalled.

  If receiver is a forked process, i.e. not the GsProcess of the
  current execute or perform, returns the receiver after receiver
  has been terminated.

  If receiver is a non-forked process , then
  after completion of the termination,  a not-trappable
  an error is returned to the GCI, containing
  a not-trappable instance of TerminateProcess,
  The   instance of TerminateProcess may have a stack captured,
  but   the GsProcess associated with the error has no stack
  and is terminated .
  "
  | minfo theEx |
  ((minfo := modeInfo) bitAnd: ModeInfo_continuationMask) ~~ 0 ifTrue:[
     ImproperOperation new object: self;
           signal:'cannot terminate an instance that is a continuation' .
  ].
  [  Notification signal ] onSynchronous: AbstractException do:[ :ex |
    "capture stack in case GemExceptionSignalCapturesStack is true"
    (theEx := TerminateProcess new) _gsStack: ex _gsStack .
  ].
  (minfo bitAnd: ModeInfo_terminationStarted) ~~ 0 ifTrue:[ "termination already started"
      "GsFile gciLogServer:'-- termination already started'."
      self _finishTermination .
  ] ifFalse:[ | currentProc |
      self _setModeinfoBit: ModeInfo_terminationStarted value: 1 .
      (self == (currentProc := self _current)) ifTrue:[ "receiver is the active process"
        "GsFile gciLogServer:'-- _serviceTerminationInterrupt' ."
        self _serviceTerminationInterrupt .
      ] ifFalse:[ | sched cPrio slept |
        "GsFile gciLogServer:'-- beforked ' ."
        self beForked .
        sched := self _scheduler _enterCritical .
        self _storeBit: 3 value: true . "set INT_GSPROC_TERMINATE_MASK bit"
        self _unscheduleProcess . "undo waiting on delays, semaphores, etc"
        self priority <= (cPrio := currentProc priority) ifTrue:[
          self _setPriority: cPrio + 1 .
        ].
        sched _scheduleProcess: self "put on readyQ";
              _exitCritical .
        slept := 0 .
        [ self _isTerminated ] whileFalse:[
          slept > timeoutMs ifTrue:[ ThreadError signal: 'Termination waited too long'].
          Delay waitForMilliseconds: 10.
          slept := slept + 10 .
        ].
      ]
  ] .
  "self _isTerminated ifFalse:[
    GsFile gciLogServer:'-- termination not complete, skip terminate9'.
    self terminate9
  ]."
  (minfo bitAnd: ModeInfo_forked) == 0 ifTrue:[
    "GsFile gciLogServer:'-- terminatedReturnToGci'."
    self _isTerminated ifFalse:[
      Error signal:'Termination failed to complete.'
    ].
    self _terminatedReturnToGci: theEx
  ].
%



category: 'Private'
method:
_terminatedReturnToGci: anException

"If receiver was not the most recently started main GsProcess,
 i.e. not from most recent IntSendMsg or IntExecute ,
 return the receiver.

 Otherwise return control to the GCI with a GciErrorSType encapsulating
 anException .  The receiver will be the GsProcess associated with that 
 GciErrorSType , and will have no stack."

<primitive: 543>
self _primitiveFailed: #_terminatedReturnToGci args: { anException } .
self _uncontinuableError  "should never reach here"
%

! if method terminate9 renamed, must also defn/uses of o_sym_terminate9 in VM C code
category: 'Changing Process State'
method:
terminate9
  "agressive termination that executes in the sender's green thread
   without yielding and without using the receiver's stack to execute
   the ensure blocks ."
  | minfo |
  ((minfo := modeInfo) bitAnd: ModeInfo_continuationMask) ~~ 0 ifTrue:[
    self error:'cannot terminate an instance that is a continuation'
  ].
  (minfo bitAnd: ModeInfo_terminationStarted) == 0 ifTrue:[ 
    "ensure blocks not attempted yet"
    [ 
      self _executeEnsureBlocks .
    ] on: Error do:[ :ex |
      "GsFile gciLogServer: '-- Failure during ensure blocks, ' , ex asString ."
      ex return: nil  "abandon further execution of ensure blocks" 
    ]
  ].
  self _finishTermination .
%

! fix 48016
category: 'Private'
method:
_serviceTerminationInterrupt
  [
    self _executeEnsureBlocks . 
  ] on: Error do:[:ex |
    "GsFile gciLogServer: '-- Failure during ensure blocks, ' , ex asString ."
    ex return: nil ."ignore exception"
  ].
  [
    ^ self _finishTermination
  ] on: Error do:[:ex |
    GsFile gciLogServer: ('-- Failure during _finishTermination, ' , ex asString) lf ;
          gciLogServer: ex stackReport .
    ex return: nil ."ignore exception"
  ]
%

category: 'Private'
method:
_finishTermination
  "returns receiver"
  <primitive: 736> "enter critical region, prim always fails"
  block ifNil: [
    uaCount_depth := 0.
    arStack := nil.
    interruptFlag_stEntryCount := 0 .
  ] ifNotNil:[ 
    block := nil 
  ].
  "remoteProcess := nil. "
  self _setModeinfoBit: ModeInfo_terminated value: 1 .  
  self _signalJoiners .
  self _scheduler _exitCritical; _terminateScheduledProcess: self.
%

method:
_executeEnsureBlocks
  | pairs |
  pairs := self _getEnsureBlocks .
  1 to: pairs size by: 2 do: [:k | 
    self _removeEnsureAtFP:( pairs at: k) .
    (pairs at: k + 1 ) value .
  ].
%

category: 'Changing Process State'
method:
signalException: anException

  | prevBit |
  "signal an exception to the receiver."

  self == self _current ifTrue:[ 
    "this path not expected to be used by Ruby"
    ^ anException signal 
  ].
  signalledException := anException .
  prevBit := self _storeBit: 2  value: true . "set INT_THREAD_SIGNAL_EXC_MASK bit"
  self resume .  
%

category: 'Private
method:
_addJoiner: aDelay

| js |
(js := joiners ) ifNil:[ 
  joiners := { aDelay }.  
  ^ self .
].
js at: js size + 1 put: aDelay
%

category: 'Private
method:
_removeJoiner: aDelay

| js ofs |
(js := joiners ) ifNotNil:[
  ofs := (js := joiners ) indexOfIdentical: aDelay .
  ofs ~~ 0 ifTrue:[ js at: ofs put: nil ].
].
%

category: 'Private'
method:
_signalJoiners
| js sched |
js := joiners .  
js ifNotNil:[
  joiners := nil .
  sched := self _scheduler _enterCritical .
  1 to: js size do:[:n | | aDelay |
    aDelay := js at: n .
    aDelay ifNotNil:[ 
      sched _delayUnschedule: aDelay .  "Inline Delay>>signal"
      aDelay _activate
    ]. 
  ].
  sched _exitCritical .
].
%

category: 'Changing Process State'
method: 
join: limitSeconds

  "Waits up to limitSeconds for receiver to terminate.
   Returns self if thread already terminated, nil
   if the time limit expires before receiver terminates"

  | curr delay mi |
  limitSeconds _isSmallInteger ifFalse:[
    ArgumentError signal:'expected a SmallInteger' .
  ].
  limitSeconds < 0 ifTrue:[
    ArgumentError signal:'argument must be >= zero ' .
  ].
  self == (curr := self _current) ifTrue:[
    ThreadError signal:'tried to join itself' .
    ^ nil 
  ].
  mi := modeInfo .
  (mi bitAnd: ModeInfo_terminated ) ~~ 0 ifTrue:[ ^ self ].
  limitSeconds ~~ 0 ifTrue:[
    (mi bitAnd: ModeInfo_terminationStarted) ~~ 0 ifTrue:[ | t limMs |
      limMs := 1000 * limitSeconds . 
      t := 0 .
      [ t < limMs ] whileTrue:[
        delay := Delay forMilliseconds: 20.
        delay highPriorityWait  .
        self _isTerminated ifTrue:[ ^ self ].
        t := t + 20 .
      ]
    ] ifFalse:[
      delay := Delay forMilliseconds: 1000 * limitSeconds .
      self _addJoiner: delay .
      delay highPriorityWait  .
      self _removeJoiner: delay .
      self _isTerminated ifTrue:[ ^ self ].
    ]
  ].
  ^ nil
%

category: 'Accessing'
method:
result
  "Retrieve the result from a completed thread. Returns nil
  if receiver still running, or did not complete normally."
  | res |
  self _isTerminated ifFalse:[ ^ nil ].
  (res := blockResult) == _remoteNil ifTrue:[ ^ nil ].
  ^ res
%


category: 'Private'
method:
_statusString
  "Returns a string that describes the receiver's status as a thread."

  ^ self _scheduler _statusString: self.
%

category: 'Private'
method:
_signalTime
  ^ signalTime
%
category: 'Private'
method:
_signalTime: time

  signalTime := time.
%
category: 'Private'
method:
_activate
  "Wake up the receiver"
  | sched |
  sched := self _scheduler _enterCritical .
  signalTime := 0 .  
  sched _scheduleProcess: self. 
  sched _exitCritical .
%
category: 'Private'
method:
_wait
  "Suspend the receiver who should be the active process"
  self _scheduler _reschedule: true .
  ^ self
%
category: 'Private'
method:
_reapSignal: signalSource
  "Make the receiver ready to run."
  "caller enters critical region"

  waitingOn := nil . 
  self _scheduler _scheduleProcess: self.
%

category: 'Private'
method:
_canWaitOnSocket

  ^ true
%

category: 'Private'
method:
_signalAll
  "Wake up the receiver"  "caller is in critical region"

  waitingOn := nil .
  self _scheduler  _scheduleProcess: self.
%

!      extent0.ruby.dbf uses   _start from   image/ruby/GsProcess_ruby.gs
category: 'Private'
method:
_start
  "Called from C to start a new process. 
   NOTE, exception handling differs in extent0.dbf and extent0.ruby.dbf .
   This version of _start from extent0.dbf .

   This method will never return. Once the receiver completes it should
   find another process to run. 
   This method is preloaded during VM startup.  You must logout/login
   to have any changes to this method take effect."

  | res curr |
  (modeInfo bitAnd: ModeInfo_continuationMask) ~~ 0 ifTrue:[
    self error:'cannot start an instance that is a continuation' 
  ].
  block ifNotNil:[
    block valueWithArguments: args  
  ].
  "if this process resumed a continuation, then current process may
   no longer be self at this point."
  curr := self _terminateCurrentWithResult: res . "terminate currently running process"
  joiners ifNotNil:[ 
    curr == self ifTrue:[ self _signalJoiners ]
               ifFalse:[ self error:'cannot _signalJoiners after resuming continuation'].
  ].
  self _scheduler _runNextProcess .
%

! added  status = 'suspended'  term for Ruby debugger support
category: 'Private'
method:
_checkIfDebuggable
  "Check to make sure the receiver is debuggable. Currently this
   means that it is in the debug or ready states.
   If it is not debuggable then raise an error."

  | status |

  status := self _statusString.
  (status = 'ready' or:[status = 'debug' or:[ status = 'active' ]]) ifFalse: [
    ImproperOperation new _number: 2376 ; reason: #rtErrGsProcessNotDebuggable;
  object: self ; signal
  ].
%

! used by GciStep implementation
category: 'Private'
method:
_stepCleanup: saveArray
  "Legacy code, no longer used."
  | newStackDepth flags |

  self _storeBit: 1 value: false . "debuggingInProgress:=0"

  newStackDepth := self stackDepth.

  self _clearAllStackBreaks .
  GsNMethod _clearAllStepBreaks .

  "clear INT_STEP_INTO_FROM_TOS_MASK "
  flags := self interruptFlag .
  flags := flags bitAnd: (16r200 bitInvert).
  self setInterruptFlag: flags .
%

category: 'Debugging Support'
method:
gciStepOverFromLevel: aLevel
  "For use by a Gci application.  
   Breakpoint and step point errors will be returned as errors to the GCI.
   Do not use when implementing in-process debugging. 

  Step execution so that it stops after the next message send
  in the specified stack level.  If a return causes the
  context at the specified level to be removed from the
  stack, execution will stop immediately after that return.
  If the receiver completes result of the completion is returned."
  aLevel < 1 ifTrue:[ ArgumentError signal:'level argument must be >= 1' ].
  ^ self _stepOverInFrame: aLevel  mode: true replace: false tos: nil
%

category: 'Debugging Support'
method:
gciStepIntoFromLevel: aLevel
  "For use by a Gci application.  
   Breakpoint and step point errors will be returned as errors to the GCI.
   Do not use when implementing in-process debugging. 

  Step execution so that it stops at the beginning of the next method
  invoked.  If a return causes the
  context at the specified level to be removed from the
  stack, execution will stop immediately after that return.
  If the receiver completes result of the completion is returned."
  aLevel < 1 ifTrue:[ ArgumentError signal:'level argument must be >= 1' ].
  ^ self _stepOverInFrame: aLevel - 1 mode: true replace: false tos: nil
%

category: 'Debugging Support'
method:
gciStepThruFromLevel: aLevel
  "For use by a Gci application.  
   Breakpoint and step point errors will be returned as errors to the GCI.
   Do not use when implementing in-process debugging. 

  Step execution so that it stops after the next message send
  in the specified stack level.  If a return causes the
  context at the specified level to be removed from the
  stack, execution will stop immediately after that return.
  If the receiver completes result of the completion is returned."

  | flags |
  aLevel < 1 ifTrue:[ ArgumentError signal:'level argument must be >= 1' ].
  flags := (aLevel bitAnd: 16rFFFFFFFF) bitOr: 16r100000000"thru flag" .
  ^ self _stepOverInFrame: flags mode: true replace: false tos: nil
%

category: 'Private Debugging Support'
method:
_topazStepOverInFrame: flags
  "Used by topaz which 
   expects breakpoint and step point errors to be returned to the GCI .
   Do not use when implementing in-process debugging.

   Step execution so that it stops after the next message send
   in the specified stack level.  If a return causes the
   context at the specified level to be removed from the
   stack, execution will stop immediately after that return.
   Exceptions for step-point and breakpoint errors are NOT suppressed.
   If the receiver completes result of the completion is returned.

   (flags bitAnd:  16rFFFFFFFF) is the stack level to step from .
    (flags bitAnd: 16r100000000) is the step through boolean."

  ^ self _stepOverInFrame: flags mode: true replace: false tos: nil
%

category: 'Private Debugging Support'
method:
_stepOverInFrame: flags
    "Used by GBS , which expects breakpoint and step point errors 
     to be returned to the GCI.
     Do not use when implementing in-process debugging.
     Step execution so that it stops after the next message send
     in the specified stack level.  If a return causes the
     context at the specified level to be removed from the
     stack, execution will stop immediately after that return.
     Exceptions for step-point and breakpoint errors are suppressed.
     Returns the receiver unless the receiver completes in which
     case the result of the completion is returned.

     (flags bitAnd:  16rFFFFFFFF) is the stack level to step from .
     (flags bitAnd: 16r100000000) is the step through boolean .
    "

  ^ self _stepOverInFrame: flags mode: false replace: false tos: nil
%

category: 'Private Debugging Support'
method:
_stepOverInFrame: flags return: anObject
    "Used by GBS , which expects breakpoint and step point errors 
     to be returned to the GCI.
     Do not use when implementing in-process debugging.
     Step execution so that it stops after the next message send
     in the specified stack frame.  If a return causes the
     context at the specified frame to be removed from the
     stack, execution will stop immediately after that return.
     The argument anObject will be the return value of the
     current top of stack - useful for passing back the return
     value of a client forwarder send.
     Returns the receiver unless the receiver completes in which
     case the result of the completion is returned.
     Execution of this method must be in interpreted mode.

     (flags bitAnd:  16rFFFFFFFF) is the stack level to step from .
     (flags bitAnd: 16r100000000) is the step through boolean .
    "
  ^ self _stepOverInFrame: flags mode: false replace: true tos: anObject
%

! fixed 43123
category: 'Private'
method:
_stepOverInFrame: flags mode: raiseException replace: replaceTosBool tos: newTosValue
    "Used in the implementation of GciStep and within the server
     image. topaz no longer uses this directly.
     Do not use when implementing in-process debugging.
     Step execution so that it stops after the next message send
     in the specified stack level.  If a return causes the
     context at the specified level to be removed from the
     stack, execution will stop immediately after that return.
     If 'raiseException' then debugger exceptions are raised
     so that a GCI app can catch them.
     Returns the receiver unless the receiver completes in which
     case the result of the completion is returned.
     Execution of this method must be in interpreted mode.

     flags == 0 is stepInto semantics.
     flags == 1 is typical step over in top frame .
     flags == 16r100000001 is typical step through (stopping in blocks of the method)
     (flags bitAnd:  16rFFFFFFFF) is the stack level to step from .
     (flags bitAnd: 16r100000000) is the step through boolean .
    "

  | actualLevel result saveArray breakpointsToIgnore 
    mySaveArray saveStackDepth status sched dp currProc doTrace |
  self isContinuation ifTrue:[ 
    Error signal:'cannot use stepping to resume a continuation' .
  ].
  self _checkIfDebuggable .

  self convertToPortableStack .
  self _nativeStack ifTrue:[
    Error signal:'unable to convert GsProcess to be stepped to portable stack'
  ].
  (currProc := GsProcess _current) _nativeStack ifTrue:[ | cvtProc |
    cvtProc := [ currProc convertToPortableStack ] fork .
    [ cvtProc _isTerminated ] whileFalse:[
      self _scheduler yield .
    ].
    currProc _nativeStack ifTrue:[
      Error signal:'unable to convert current GsProcess to portable for stepping another GsProcess'.
    ].
  ].
  sched := self _scheduler _enterCritical .
  status := self _statusString.
  (status = 'ready') ifTrue: [
    self _unscheduleProcess .
  ].
  dp := self localStackDepth .
  flags < 0 ifTrue:[
    flags _error: #rtErrArgOutOfRange args:{ 0 . dp } .
  ].
  actualLevel := flags bitAnd:  16rFFFFFFFF .
  (actualLevel > dp) ifTrue:[
    actualLevel _error: #rtErrArgOutOfRange args:{ 0 . dp } .
  ].
  saveStackDepth := dp .
  (actualLevel == 0) ifTrue: [  | interrFlags |
    saveArray := self _setBreaksForStepInto.  
    breakpointsToIgnore := (saveArray at: 3).
    "set INT_STEP_INTO_FROM_TOS_MASK bit"
    interrFlags := self interruptFlag .
    interrFlags := flags bitOr: 16r200.
    self setInterruptFlag: interrFlags .
  ] ifFalse: [
    saveArray := self _setBreaksForStepOverFromLevel: flags .
    breakpointsToIgnore := (saveArray at: 3).
        	" deleted INT_STEP_OVER_FROM_TOS_MASK logic "
  ].

  (replaceTosBool) ifTrue: [
    mySaveArray := { (saveArray at: 1) . (saveArray at: 2) . saveStackDepth . 
                       newTosValue }.
  ] ifFalse: [
    mySaveArray := { (saveArray at: 1) . (saveArray at: 2) . saveStackDepth }.
  ].

  self _storeBit: 1 value: true . "debuggingInProgress:=1"
  sched _exitCritical .

  doTrace ifNotNil:[
    GsFile gciLogServer:'breakpointsToIgnore ' , breakpointsToIgnore asString ].
  result := self _primStep: breakpointsToIgnore
                 from: nil with: mySaveArray mode: raiseException.

  sched _enterCritical .
  (status = 'ready') ifTrue: [
    (result == self) ifTrue: [
      (self _isTerminated) ifFalse: [
        sched _scheduleProcess: self .
      ].
    ].
  ].
  sched _exitCritical .
  "_stepCleanup may run a method that was stepped into (Object>>at:), so run with
   #_performNoDebug:with: to ensure that the cleanup is performed"
  self _performNoDebug: #_stepCleanup:  env: 0  with: mySaveArray.

  ^result
%

category: 'Private'
method:
_primStep: breakpointsToIgnore from: argUnused with: saveArray mode: raiseExcept

<primitive: 2011>
^ self  "normal execution path"
%


category: 'Debugger Support'
method: 
_continue
    "Continue execution. Return the result of execution if the receiver
     completes. Otherwise if a breakpoint is hit returns the receiver."

  | result status sched |

  self _checkIfDebuggable.
  sched := self _scheduler _enterCritical .
  status := self _statusString.
  (status = 'ready') ifTrue: [
    self _unscheduleProcess .
  ].
  self _storeBit: 1 value: true . "debuggingInProgress:=1"
  sched _exitCritical .

  result := self _primContinue: nil .

  sched _enterCritical .
  (status = 'ready') ifTrue: [
    (result == self) ifTrue: [
      (self _isTerminated) ifFalse: [
        sched _scheduleProcess: self .
      ].
    ].
  ].
  self _storeBit: 1 value: false . "debuggingInProgress:=0"
  sched _exitCritical .
  ^result
%

category: 'Private'
method:
_primContinue: argUnused

<primitive: 2012>
"no image code here, compiler emits Bc_RETURN_TOS here"
%

category: 'Private'
method:
_unscheduleProcess

  "remove the given process from the queues it is in.
   caller puts scheduler in critical region."
  | sched won st |
  sched := self _scheduler .
  (st := signalTime) ifNotNil: [ 
     sched _delayUnschedule: self.
  ].
  (won := waitingOn) ifNotNil:[
    won _unscheduleProcess: self .
  ].
  sched _unschedule: self.
  st ifNotNil: [
    signalTime := nil
  ].
  won ifNotNil:[
    waitingOn := nil
  ].
%

category: 'Private'
method:
_targetProcess
  "Returns the GsProcess that is waiting for the receiver."

  ^ self
%

category: 'Process Groups'
method:
_newGroup
  "Add the receiver to a new Smalltalk process group. Return the group value."
  | g |
  g := self _scheduler _newGroup.
  group := g .
  ^ g
%

category: 'Process Groups'
method:
_group
  "Return the Smalltalk process group (a SmallInteger) the receiver is in,
   assigning to a new group if needed ."
  | g |
  (g := group) ifNil:[ ^ self _newGroup].
  ^ g
%

method:
_groupOrNil
  "Return the process group the receiver is in, 
   or return nil if receiver has not been assigned to a group."

  ^ group
%

category: 'Process Groups'
method:
_joinGroup: aGroup
  "Have the receiver join the specified processor group. Return self.
   For Smalltalk,  aGroup is an Integer,
   For Ruby aGroup is a RubyThreadGroup .
  "
  group := aGroup .
%

category: 'Process Groups'
method:
_groupSameAs: anInteger
  "Return true if the receiver's group is the same as anInteger.
   Otherwise return false."

  "Note we don't use 'self _group' because that would initialize
   group which we don't need done for a same as test."

  ^group = anInteger
%

category: 'Accessing'
method:
threadRubyEnvId

 ^ modeInfo bitAnd: ModeInfo_threadEnvMask
%

category: 'Continuations'
method:
isContinuation

"Return true if the receiver is a continuation."

^ (modeInfo bitAnd: ModeInfo_continuationMask) == ModeInfo_isContinuation
%
category: 'Continuations'
method:
isPartialContinuation

"Return true if the receiver is a continuation."

^ (modeInfo bitAnd: ModeInfo_continuationMask) == ModeInfo_isPartialContinuation
%

category: 'Continuations'
method:
value: anArg

"The receiver must be a continuation, i.e. a result
 from continuationFromLevel: , or an error will be generated.

 Resumes execution of the continuation with top-of-stack
 replaced by anArg . "

<primitive: 2016>
self _uncontinuableError  "should never reach here"
%

category: 'Continuations'
method:
value

"See value: for documentation "

^ self value: nil

%
category: 'Private'
classmethod:
_current
  "Returns currently active process .
   The result should be used for accessing thread-local data only.
   Instance methods related to accessing the stack frames do not work
   for the active process." 

<primitive: 693>
self _primitiveFailed: #_current
%

classmethod:
_maxProcessStacks
 
  ^ MaxProcessStacks "from C constant OM_MAX_PROCESS_STACKS"
%

method: 
_current
  "Returns currently active process .
   The result should be used for accessing thread-local data only.
   Instance methods related to accessing the stack frames do not work
   for the active process." 

<primitive: 693>
self _primitiveFailed: #_current
%

category: 'Process Properties'
method:
_environment

  ^ environment
%

category: 'Process Properties'
method:
environment
  | env |
  (env := environment) ifNil:[
     env := Dictionary new .
     environment := env
  ].
  ^ env
%

category: 'Process Properties'
method:
environment: aDictionary

  environment := aDictionary
%

! environmentAt:  part of fix 41533
category: 'Process Properties'
method:
environmentAt: key

  | env |
 (env := environment) ifNil:[ ^ nil ].
 ^ env at: key otherwise:  nil
%

category: 'Process Properties'
method:
environmentAt: key ifAbsent: aBlock
  | env |
  (env := environment) ifNil:[ ^ aBlock value ].
  ^ env at: key ifAbsent:[ aBlock value ]
%

category: 'Process Properties'
method:
environmentAt: key put: value

  ^ (environment ifNil:[ self environment]) at: key put: value
%

category: 'Private'
method:
_convertToPortableStack: forTrimToLevel
  "convert receiver's stack from native code to portable code format.
   Receiver must be suspended and stack in object memory.
  forTrimToLevel should be -1 if not being called for stack trim,
  otherwise a positive stack level. "
   
  self _nativeStack ifTrue:[
    [
      | newIps ipOfs fpOfs ar_size lev |
      newIps := { }  .
      ipOfs := 1 .
      fpOfs := topFpOffset .
      ar_size := arStack size .
      lev := 1 .
      [ fpOfs < 0 ] whileTrue:[ | fpIdx codePtr meth natIp portIp |

        fpIdx := ar_size  + (self _toWordOfs: fpOfs) + 1 .
        codePtr := arStack at:( fpIdx + FP_codePtr_OFS ).
        meth := self _fetchMethod:  codePtr  .
        natIp := arStack  at: ipOfs .
        portIp := meth _nativeIpOffsetToPortable: natIp asReturn: true .
        portIp <= 0 ifTrue:[
          (forTrimToLevel >= 1 and:[ lev <= forTrimToLevel ]) ifTrue:[
            portIp := GsNMethod _firstPortableIpoffset .
          ] ifFalse:[
            "examples: error 2085 from an ifTrue bytecode..."
            self error:'native to portable IP not exact in frame ', lev asString . 
          ]
        ].
        arStack  at: ipOfs put: portIp .
        ipOfs := fpIdx + FP_rtnAddr_OFS .
   
        fpOfs := arStack at: fpIdx"+ FP_savedFP_OFS==0" . "get callerFpOfs"
        lev := lev + 1 .
      ].
      self _setInterpretedStack .
    ] onSynchronous: Error do:[ :ex |
      InternalError new details: ex description ; 
           reason: 'nativeStackNotConvertableToInterpreted' ; signal
    ].
  ].  
%

category: 'Debugging Support'
method:
convertToPortableStack
  "Ensure that the receiver is executing in interpreted mode,
   used so that subsequent breakpoint operations will work."

  self == GsProcess _current ifTrue:[ 
    GsProcess usingNativeCode ifTrue:[
      ^ Error signal:'cannot alter stack of the currently running GsProcess'
    ]
  ] ifFalse:[
    self localStackDepth "ensure stack pushed to object memory".
    self _nativeStack ifTrue:[
      self _convertToPortableStack: -1 .
    ].
  ]
%

method:
convertToPortableStack: forTrimToLevel

  "Ensure that the receiver is executing in interpreted mode,
   used so that subsequent breakpoint operations and execution
   restart will work.
   forTrimToLevel should be -1 if not being called for stack trim,
   otherwise a positive stack level. "

  self == GsProcess _current ifTrue:[ 
    GsProcess usingNativeCode ifTrue:[
      ^ Error signal:'cannot alter stack of the currently running GsProcess'
    ]
  ] ifFalse:[
    self localStackDepth "ensure stack pushed to object memory".
    self _nativeStack ifTrue:[
      self _convertToPortableStack: forTrimToLevel .
    ].
  ]
%

category: 'Debugging Support'
classmethod:
usingNativeCode

  "returns true if the currently executing process is using native code"
<primitive: 855>
self _primitiveFailed: #usingNativeCode
%

classmethod:
_topazAsString: anObject
  ^ [ anObject asString 
    ] onSynchronous: AbstractException do:[:ex | 
      [  | cls | 
         cls := Reflection classOf: anObject .
         cls == UnauthorizedObjectStub ifTrue:[ anObject __asString]
                           ifFalse:[ 'a ', cls name ]
      ] onSynchronous: AbstractException do:[:exb |
        '<error during asString>'
      ]
    ]
%

classmethod:
_topazExceptionName: anException
  ^ [ | ecls |
      (ecls := anException class) _isExceptionClass ifFalse:[ ^ nil ].
      ecls name  .
    ] onSynchronous: AbstractException do:[:ex |
      ex return: '<error during _topazExceptionName:>'
    ].
%
method:
_topazMethodAt: aLevel

  "Returns nil if aLevel is out of range"

| fpIdx dp |
(aLevel < 1 or:[ aLevel > (dp := self localStackDepth) ]) ifTrue:[
  ^ nil
  ].
fpIdx := self _frameOffsetAt: aLevel .
fpIdx ifNil:[ ^ nil].
^ self _fetchMethod:( arStack at: (fpIdx + FP_codePtr_OFS )) .
%

method:
isForked
  ^ (modeInfo bitAnd: ModeInfo_forked) ~~ 0
%

! ! fixed 45091
category: 'Private'
method: GsProcess
beForked
  "Private.  
   Prevents termination of the receiver from causing a return to the GCI.
   if the scheduler has something else to run."

  modeInfo := modeInfo bitOr: ModeInfo_forked
%

category: 'Private Debugging Support'
method: 
_stepOverFromLevel: flags
  "  (flags bitAnd:  16rFFFFFFFF) is the stack level to step from .
     (flags bitAnd: 16r100000000) is the step through boolean ."
  self == GsProcess _current ifTrue:[
    ^ Error signal:'cannot single-step the currently running GsProcess'
  ].
  self localStackDepth . "force to object memory"
  self convertToPortableStack .
  
  "No longer needed, VM cleared stack and step BPs before signalling
   a single step Breakpoint."
    "self _clearAllStackBreaks . " 
    "GsNMethod _clearAllStepBreaks ."  

  "clear any leftover INT_STEP_INTO_FROM_TOS_MASK"
  self setInterruptFlag: (self interruptFlag bitAnd:(16r200 bitInvert)) .

  self _setBreaksForStepOverFromLevel: flags .
%

category: 'In-Process Debugger Support'
method:
stepOverFromLevel: aLevel
    "Step execution over next message in stack frame specified by aLevel.
     To be used within implementation of an in-process debugger.
     After executing this method,
     the sender must yield  so the receiver can run.
     Receiver needs to have an on:do: handler for Breakpoint which
     yields to the debuggger process and then resumes from the Breakpoint."

  | dp |
  self == GsProcess _current ifTrue:[
    ^ Error signal:'cannot single-step the currently running GsProcess'
  ].
  dp := self localStackDepth . "force to object memory"
  aLevel _validateClass: SmallInteger .
  (aLevel < 1 or:[ aLevel > dp]) ifTrue:[
     ^ ArgumentError new name:'alevel' min: 1 max: dp actual: aLevel ; signal.
  ].
  self _stepOverFromLevel: aLevel 
%

category: 'In-Process Debugger Support'
method:
stepIntoFromLevel: aLevel

    "Step execution over next message in stack frame specified by aLevel.
     To be used within implementation of an in-process debugger,
     after the receiver has signaled a Breakpoint from a previous single step
     or from a method breakpoint. 
     After executing this method,
     the sender must yield  so the receiver can run.
     Receiver needs to have an on:do: handler for Breakpoint which
     yields to the debuggger process and then resumes from the Breakpoint.

     The step_into interrupt bit in the VM will be set by the returnNothing
     bytecode when the receiver returns from AbstractException>>_signalAsync."

  self stepOverFromLevel: aLevel .
  self _setModeinfoBit: ModeInfo_rtnNothingStepInto value: 1 .
%

category: 'In-Process Debugger Support'
method:
stepThroughFromLevel: aLevel
  "Step execution over the next message send in stack frame specified 
   by aLevel, stopping in blocks which share a home method with the
   specified frame.
   To be used within implementation of an in-process debugger.
   After executing this method,
   the sender must yield  so the receiver can run.
   Receiver needs to have an on:do: handler for Breakpoint which
   yields to the debuggger process and then resumes from the Breakpoint."

  | dp |
  self == GsProcess _current ifTrue:[
    ^ Error signal:'cannot single-step the currently running GsProcess'
  ].
  dp := self localStackDepth . "force to object memory"
  aLevel _validateClass: SmallInteger .
  (aLevel < 1 or:[ aLevel > dp]) ifTrue:[
     ^ ArgumentError new name:'alevel' min: 1 max: dp actual: aLevel ; signal.
  ].
  self _stepOverFromLevel: (16r100000000 bitOr: aLevel)
%

category: 'In-Process Debugger Support'
method:
clearLastBreakpointHistory
  "If a Breakpoint is handled by handler block installed with on:do: or
   onException:do:, and neither #return nor #resume is sent to the instance
   of Breakpoint, then 
      GsProcess _current clearLastBreakpointHistory 
   is needed in order for the next Breakpoint to be signalled properly."

  lastBreakpt := nil
%

   
