"
Object
  Cidr( ipAddr netmask maskComplement)

a Cidr represents an IPv4 classless internet domain routing subnet.
"
Class {
	#name : 'Cidr',
	#superclass : 'Object',
	#instVars : Array [
		'ipAddr',
		'maskComplement',
		'netmask'
	],
	#category : 'X509-Core'
}

{ #category : 'instance creation' }
Cidr class >> fromIP: ipAddrString [
  ^ CidrParser parse: ipAddrString , '/32' .
]

{ #category : 'instance creation' }
Cidr class >> fromString: aString [
  "aString must include a /nn  suffix"
  ^ CidrParser parse: aString .
]

{ #category : 'instance creation' }
Cidr class >> ip: ipInteger mask: maskInteger [
  | res |
	(res := self new)
		initializeForIp: ipInteger mask: maskInteger.
  ^ res
]

{ #category : 'comparing' }
Cidr >> <= aCidr [
  ^ ipAddr <= aCidr ipAddr
]

{ #category : 'comparing' }
Cidr >> = aCidr [
  ^ ipAddr = aCidr ipAddr and:[ netmask = aCidr netmask ]
]

{ #category : 'printing' }
Cidr >> asString [
  ^ self printString
]

{ #category : 'query' }
Cidr >> containsCidr: aCidr [
  ^ (self containsIp: aCidr ipAddr) and:[ maskComplement >= aCidr maskComplement ]
]

{ #category : 'queries' }
Cidr >> containsIp: ipInteger [
	"Answer true iff the given ipInteger falls within the subnet I represent."

	^(ipInteger bitAnd: netmask) = ipAddr
]

{ #category : 'private' }
Cidr >> error: aMessage [

	Error signal: aMessage
]

{ #category : 'comparing' }
Cidr >> hash [
  ^ (ipAddr + (netmask bitShift: 32)) hash 
]

{ #category : 'initialization' }
Cidr >> initializeForIp: ipInteger mask: maskInteger [

	ipAddr := ipInteger.
	maskComplement := self maskComplementFor: maskInteger.
	netmask := maskComplement bitXor: 16rFFFFFFFF.
	self validate
]

{ #category : 'accessing' }
Cidr >> ipAddr [
  ^ ipAddr
]

{ #category : 'accessing' }
Cidr >> maskComplement [
  ^ maskComplement
]

{ #category : 'private' }
Cidr >> maskComplementFor: maskInteger [

	(maskInteger between: 0 and: 32)
		ifFalse: 
			[self error: 'Netmask of ' , maskInteger asString
						, ' bits is not valid for IPv4'].
	^(2 raisedToInteger: 32 - maskInteger) - 1
]

{ #category : 'accessing' }
Cidr >> netmask [
  ^ netmask
]

{ #category : 'printing' }
Cidr >> printString [
  | str shiftCount byte |
  str := String new .
  shiftCount := -24 .
  [ shiftCount < 0 ] whileTrue:[ 
    byte := (ipAddr bitShift: shiftCount) bitAnd: 16rFF .
    str addAll: byte asString ; add: $. .
    shiftCount := shiftCount + 8 .
  ].
  byte := ipAddr bitAnd: 16rFF .
  str addAll: byte asString ; add: $/ .
  str addAll: (32 - (maskComplement == 0 ifTrue:[ 0 ] ifFalse:[ maskComplement highBit])) asString.
  ^ str
]

{ #category : 'private' }
Cidr >> validate [
  | bits |
  (bits := ipAddr bitAnd: maskComplement) == 0
     ifFalse: [self error: 'Local (low-order) bits should be zero']
]


