!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   Symbol, String, CharacterCollection, SequenceableCollection,
!   Collection, Object.
!
!=========================================================================

removeallmethods Symbol
removeallclassmethods Symbol
set class Symbol

category: 'For Documentation Installation only'
classmethod:
installDocumentation

self comment:
'A Symbol is an invariant String for which all comparisons are case-sensitive.
 Symbols are used internally to represent variable names and selectors.  Symbols
 are always invariant and cannot be modified at any time after they are created.
 Hence, the new and new: methods are disallowed.

 All Symbols and DoubleByteSymbols are canonical, which means that it is not
 possible to create two of them that have the same value.  If two canonical
 symbols compare as equal, then they are the same (identical) object.  Every
 instance of DoubleByteSymbol will contain at least one Character whose value is
 greater than 255.  A Symbol whose Character values are all less than 256 is
 always an instance of Symbol.

 GemStone places all canonical symbols in the DataCuratorObjectSecurityPolicy.  However,
 GemStone does permit you to commit a canonical Symbol, even if you have no
 explicit write authorization for the DataCuratorObjectSecurityPolicy.  GemStone also gathers
 all canonical symbols into one collection (a CanonicalStringDictionary) called
 AllSymbols, which it also places in the DataCuratorObjectSecurityPolicy.

 Since canonical symbols are universally visible, it is not recommended that 
 they be used for names that should remain private or secure.  Such objects
 should be instances of InvariantString instead.

 Since canonical symbols must be universally available, you cannot lock a
 Symbol, DoubleByteSymbol, or QuadByteSymbol..

 Since each canonical symbol has a unique value, you cannot copy a Symbol or
 DoubleByteSymbol.  In addition, to guarantee canonicalization, you cannot send
 the become: or changeClassTo: messages to a Symbol, DoubleByteSymbol, or 
 QuadByteSymbol..

 DoubleByteSymbol is in the ClassHistory of Symbol, so instances of
 DoubleByteSymbol may be stored into instance variables that are constrained to
 hold instances of Symbol.  The inverse is not true, so you should always
 express symbol constraints as Symbol.

 EUCSymbols are not canonicalized and cannot be used interchangeably with
 canonical symbols.  They do not satisfy a constraint of Symbol, and are not
 accepted by the virtual machine as message selectors.' .
%

category 'Instance Creation'
classmethod:
_newString

"Returns a new instance of String."

^ String new
%

category 'Instance Creation'
classmethod:
_newString: aSize

"Returns a new instance of String of the specified size."

^ String new: aSize
%

category: 'Instance Creation'
classmethod:
new

"Disallowed.  To create a new Symbol, use the class method withAll: instead."

self shouldNotImplement: #new
%

category: 'Instance Creation'
classmethod:
migrateNew

"Override default migrateNew behavior with #_basicNew."

^ self _basicNew
%

category: 'Instance Creation'
classmethod:
new: size

"Disallowed.  To create a new Symbol, use the class method withAll: instead."

self shouldNotImplement: #new:
%
classmethod:
_basicNew
^ self shouldNotImplement: #_basicNew
%
classmethod:
_basicNew: aSize
^ self shouldNotImplement: #_basicNew:
%


category: 'Instance Creation'
classmethod:
_existingWithAll: aString

"Returns an existing canonical symbol that has the same value as 'aString'.  
 If no such Symbol, DoubleByteSymbol or QuadByteSymbol already exists, returns nil."

"This method is useful in processing Strings that are supposed to be names
 of already existing instance variables, such as in indexing and sorting."

<primitive: 310>
aString _validateClass: CharacterCollection .
^ self _existingWithAll: aString asString .
%

! edits for 41699
category: 'Instance Creation'
classmethod:
withAll: aString

"Returns a canonical symbol that has the same Characters as aString.  Returns
 an existing canonical symbol if it is already in AllSymbols, or if it was
 created earlier in the current session.  Otherwise, creates and returns a new
 canonical symbol.

 The result is a Symbol, DoubleByteSymbol, or QuadByteSymbol as required
 to hold the largest code point in aString ."

<primitive: 300>
aString _isOneByteString ifTrue:[  "charSize 1 and not a ByteArray"
  ^ self _primitiveFailed: #withAll: args: { aString } .
].
aString stringCharSize == 0 ifTrue:[ "not charSize 2 or 4"
  aString _validateClass: CharacterCollection .
  ^ self withAll: aString asString
].
aString _error: #rtErrBadArgKind2 args: { String . DoubleByteString . QuadByteString }.
%

category: 'Storing and Loading'
classmethod:
loadFrom: passiveObj

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| result inst theSize marker |

"since Symbols can't refer to other objects, the 'hasRead:' message
 may be sent after values have been filled in."

theSize := passiveObj readSize . 
inst := String new: theSize .
marker := passiveObj objectPositionMarker .
passiveObj next: theSize bytesTo: inst .
passiveObj next .
result := inst asSymbol .
passiveObj hasRead: result marker: marker .
^ result
%

! removed isSymbol, to fix 32218

category: 'Converting'
method:
asSymbol

"Returns the receiver."

^ self
%

category: 'Converting'
method:
asSymbolKind

"Equivalent to asSymbol."

 ^self
%

category: 'Copying'
method:
shallowCopy

"Returns self, copies of canonical symbols are not allowed."

^ self
%

category: 'Copying'
method:
copy

"Returns self, copies of canonical symbols are not allowed."

^ self
%

set compile_env: 0
category: 'Copying'
method:
copyReplacing: oldObject withObject: newObject
	"Returns a String comprising a copy of the receiver in which all occurrences
	 of objects equal to oldObject have been replaced by newObject."

	^self asString copyReplacing: oldObject withObject: newObject
%

category: 'Clustering'
method:
cluster

"Has no effect.  Clustering of Symbols is only performed by
 the clusterAllSymbols method in class System . "

^ true 
%

! fix 32711
category: 'Clustering'
method:
clusterInBucket: aClusterBucketOrId

"Has no effect.  Clustering of Symbols is only performed by
 the clusterAllSymbols method in class System . "

^ true
%

category: 'Converting'
method:
keywords

"Returns an Array of the keywords in the receiver, treating
 any colon-delimited segment as if it was a legal keyword."

^ Symbol _keywords: self 
%

! fix 46552
category: 'Converting'
classmethod:
_keywords: aSymbol 
| result str |
result := { } .
str := String new .
1 to: aSymbol size do:[:n | | ch |
  ch := aSymbol at: n .
  ch == $:  ifTrue:[
    str add: $: .  result add: str . str := String new .
  ] ifFalse:[
    str add: ch .
  ].
].
str size ~~ 0 ifTrue:[ result add: str ].
^ result
%


category: 'Compatiblity'
method:
argumentCount

 ^ self numArgs
%

category: 'Testing'
method:
precedence

"Returns the precedence of the receiver, were it a message selector, with
 1=unary, 2=binary and 3=keyword."

^self isInfix
    ifTrue: [ 2 ]
    ifFalse: [ self isKeyword ifTrue: [ 3 ]
                              ifFalse: [ 1 ] ]
%

! changed to fix 34788
category: 'Formatting'
method:
printOn: aStream

"Puts a displayable representation of the receiver on the given stream.
 That representation conforms to GemStone Smalltalk parsing rules."

aStream nextPut: $# .
super printOn: aStream 
%

category: 'Formatting'
method:
withNoColons

"Returns a String containing the value of the receiver with all colons removed.

 A String is returned rather than a Symbol to avoid the expense of unnecessary 
 creation and canonicalization of Symbols."

^ self copyWithout: $:   .
%

category: 'Decompiling without Sources'
method:
_asSource

| result |
result := String new .
result add: $#  .
result addAll: super _asSource .
^ result
%

category: 'Formatting'
method:
asString

"Returns a copy of the receiver as an instance of class String."

^ String withAll: self
%

category: 'Concatenating'
method:
, aCharOrCharCollection

"Returns a new instance of String that contains the elements of the receiver
 followed by the elements of aCharOrCharCollection.  A String is returned rather
 than a Symbol to avoid the expense of unnecessary creation and canonicalization
 of Symbols."

| result |

result := String withAll: self .
result  addAll: aCharOrCharCollection .
^ result
%

category: 'Comparing'
method:
= aCharCollection

"Returns true if and only if the receiver and aCharCollection are identical,
 unless aCharCollection is an instance of ObsoleteSymbol.  In that case, this
 method returns true if the receiver and aCharCollection are equivalent (in the
 sense of the String | = method).  Returns false otherwise."

<primitive: 295>
self _primitiveFailed: #= .
self _uncontinuableError
%

category: 'Comparing'
method:
~= aCharCollection

"This method can be optimized for Symbols since they are canonical."

^ self ~~ aCharCollection
%

category: 'Comparing'
method:
hash

"Returns a numeric hash key for the receiver.
 This implementation for new databases only and is
 replaced in the upgrade scripts."

^ self identityHash
%


category: 'Converting'
method:
-> anObject

"Returns a SymbolAssociation with the receiver as the key and the given object
 as the value."

^ SymbolAssociation newWithKey: self value: anObject
%

category: 'Private'
method:
speciesForConversion

"Returns String."

^ String.
%

category: 'Class Membership'
method:
species

"Returns a class similar to, or the same as, the receiver's class which
 can be used for containing derived copies of the receiver."

^ String
%

! numArgs inherited from CharacterCollection

category: 'New Indexing Comparison'
method:
_idxForCompareEqualTo: arg

  ^arg _idxForCompareEqualToSymbol: self
%
! inherit _decompress from Object

category: 'Private'
classmethod:
_validateNewSymbols: newSymbols  allSymbols: allSymbols

^ true  "real implementation later in filein"
%
category: 'New Indexing Comparison'
set compile_env: 0
method:
_idxForCompareEqualTo: aCharacterCollection collator: anIcuCollator
  ""

  ^ aCharacterCollection _idxForCompareEqualToSymbol: self
%
category: 'New Indexing Comparison'
set compile_env: 0
method:
_idxForSortEqualTo: aCharacterCollection collator: anIcuCollator
  ""

  ^ self asString
    _idxForSortEqualTo: aCharacterCollection
    collator: anIcuCollator
%
category: 'New Indexing Comparison'
set compile_env: 0
method:
_idxForSortNotEqualTo: aCharacterCollection collator: anIcuCollator
  ""

  ^ (self _idxForSortEqualTo: aCharacterCollection collator: anIcuCollator) not
%
category: 'New Indexing Comparison - for Compare'
method:
_idxForCompareEqualToUnicode: aUnicodeString collator: aCollator
"second half of a double dispatch call from CharacterCollection>>_idxForCompareEqualTo:collator:."

  ^ false
%
category: 'New Indexing Comparison'
method:
_idxForSortEqualTo: arg
  ""

  ^ arg _idxForSortEqualToSymbol: self
%
category: 'New Indexing Comparison'
method:
_idxForSortEqualToSymbol: aSymbol
  "second half of a double dispatch call from Symbol>>_idxForSortEqualTo:."

  ^ self asString _idxPrimCompareEqualTo: aSymbol asString
%
category: 'New Indexing Comparison - for Compare'
method:
_idxForCompareEqualToCharacterCollection: aCharacterCollection
  "second half of a double dispatch call from CharacterCollection>>_idxForCompareEqualTo:."

  ^ false
%
category: 'Class Membership'
classmethod:
_classHistoryIncludesIdentical: aClass
  ^ aClass == Symbol or:[ aClass == DoubleByteSymbol or:[ aClass == QuadByteSymbol]]
%
category: 'Class Membership'
classmethod:
isSubclassOf: aClassHistoryOrClass

  (self _classHistoryIncludesIdentical: aClassHistoryOrClass) ifTrue:[ ^ true].
  ^ super isSubclassOf: aClassHistoryOrClass
%

