!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   UnorderedCollection, Collection, Object.
!
!========================================================================

category: 'Indexing Support'
method: UnorderedCollection
_getIndexReferencesInto: refsToRcvr
  "Returns an Array of information about references to the receiver due to the
 receiver's participation in an index as a set-valued instance variable.
 The Array consists of pairs:

 1) An object that references the receiver.
 2) The offset of the instance variable in that object that references the
  receiver.

 This is used only in support of the 'become:' method; therefore if
 the receiver is a root NSC with an index, an error will be raised."

  | indexList pathTerm index vals obj pathTerms |
  indexList := self _indexedPaths.
  pathTerms := IdentitySet new.
  2 to: indexList size by: 2 do: [ :i | 
    " for each index that the receiver participates in ... "
    (indexList at: i) == 1
      ifTrue: [ 
        " check if it is the root NSC or a set-valued instance variable "
        " it is the root, 'become:' not allowed "
        self _error: #'rtErrCantBecomeIndexedNsc' ]
      ifFalse: [ 
        (indexList at: i) > 1
          ifTrue: [ 
            index := indexList at: i - 1.
            pathTerm := index at: (indexList at: i) - 2.
            (pathTerms includesIdentical: pathTerm)
              ifFalse: [ 
                " only want to check each unique path term "
                " get the objects that refer to the receiver "
                vals := index indexDictionary
                  at: self
                  term: pathTerm
                  otherwise: nil.
                nil == vals
                  ifTrue: [ index indexDictionary _errorNoEntryForKey: self term: pathTerm value: nil ].
                (BucketValueBag _hasInstance: vals)
                  ifTrue: [ 
                    1 to: vals size do: [ :j | 
                      obj := vals _at: j.
                      pathTerm indicatesMultiValue
                        ifTrue: [ 
                          self
                            _addToReferences: obj
                            offset: 0
                            occurrences: (obj occurrencesOf: self)
                            into: refsToRcvr ]
                        ifFalse: [ 
                          self
                            _addToReferences: obj
                            offset: (pathTerm _ivOffsetFor: obj)
                            occurrences: 1
                            into: refsToRcvr ] ] ]
                  ifFalse: [ 
                    pathTerm indicatesMultiValue
                      ifTrue: [ 
                        self
                          _addToReferences: vals
                          offset: 0
                          occurrences: (vals occurrencesOf: self)
                          into: refsToRcvr ]
                      ifFalse: [ 
                        self
                          _addToReferences: vals
                          offset: (pathTerm _ivOffsetFor: vals)
                          occurrences: 1
                          into: refsToRcvr ] ].
                pathTerms add: pathTerm ] ] ] ].
  ^ super _getIndexReferencesInto: refsToRcvr	" since NSCs may have named instance variables, invoke this on super "
%

! deleted UnorderedCollection>>_explicitEqualityIndexedPaths v2.0

! deleted UnorderedCollection>>_explicitIdentityIndexedPaths v2.0

! deleted UnorderedCollection>>_explicitEqualityIndexedPathsAndConstraints v2.0

! deleted UnorderedCollection>>_deadNscBeingGarbageCollected v2.0

category: 'Clustering'
method: UnorderedCollection
_clusterIndexes

"Clusters indexing objects.  This may cause concurrency conflicts on
 internal indexing objects.  Returns the receiver."

| iList |
(iList := self _indexedPaths) == nil
  ifTrue: [ 
    ^ self 
  ].

" cluster the index dictionary "
self _indexDictionary ~~ nil
  ifTrue: [ self _indexDictionary clusterDepthFirst ].

" cluster each index object "
1 to: iList size by: 2 do: [ :j |
  (iList at: j) _isIndexObject
    ifTrue: [ (iList at: j) clusterDepthFirst ]
].
%

category: 'Indexing Support'
method: UnorderedCollection
_indexStatistics

"Return a dictionary containing statistics that can be useful in determining the
 performance of indexes.  This information is useful for system administrators
 and customer support in determining how to optimize index usage."

| dict |
dict := SymbolDictionary new.
self _indexedPaths == nil
  ifTrue: [
    dict at: #NoIndexesPresent put: true.
    ^ dict
  ].

self _indexedPaths _statisticsInto: dict.
self _indexDictionary ~~ nil
  ifTrue: [ self _indexDictionary _statisticsInto: dict ].

^ dict
%

category: 'Indexing Support'
method: UnorderedCollection
_refreshIndexCaches

"This method is currently unsupported, but is provided for customer support.

 Verifies that the index dictionary bucket caches are correct.  If an invalid
 entry is found, fixes it and reports where the fix occurred."

| result |
self _indexDictionary ~~ nil
  ifTrue: [ result := self _indexDictionary _refreshBucketCaches ].
( result == nil or: [ result isEmpty ] )
  ifTrue: [ result := 'No problems found with index caches' ].
^ result
%


category: 'Indexing Support'
method: UnorderedCollection
_setIndexDictionaryWorkingSetInterval: aNumber

"Sets the working set interval for the index dictionary when index creation
 is in progress.  This will only be in effect for the life of the transaction."

aNumber _validateClass: Number.
System
  rcValueCacheAt: #indexDictionaryWorkingSetInterval
  put: aNumber
  for: self.
^ aNumber
%

category: 'Deprecated'
set compile_env: 0
method: UnorderedCollection
_auditAndRepairDepListsLogging: aString
  "Get any dependency lists we can find and repair them if they need it.
Report any repairs on the given string."

  | depLists sharedDepLists depList cnt |
  self deprecated: 'UnorderedCollection>>_auditAndRepairDepListsLogging: deprecated v3.3.
     Use DepListTable>>_auditAndRepair: to find and repair dep lists'.	"bug 43886"
  sharedDepLists := SharedDependencyLists.
  depLists := self _getAllDependencyLists.
  cnt := 0.
  1 to: depLists size do: [ :i | 
    (depList := depLists at: i) _needsSorting
      ifTrue: [ 
        cnt := cnt + 1.
        depList _sortEntries: (sharedDepLists depListBucketFor: depList) ] ].
  cnt ~~ 0
    ifTrue: [ 
      aString
        add: 'For nsc [';
        add: self identityHash asString;
        add: '] repaired ';
        add: cnt asString;
        add: ' dependency list(s)';
        add: Character lf ].
  ^ aString
%

category: 'Deprecated'
set compile_env: 0
method: UnorderedCollection
_getAllDependencyLists
  "Return a set of all dependency lists that we can find from the receiver."

  | all iList rootTerms obj sz prevObj bag |
  self deprecated: 'UnorderedCollection>>_getAllDependencyLists deprecated v3.3.
     Use DependencyList class>>depMapValues to collect dependency lists. (3.3)'.  "bug 43886"

  all := IdentitySet new.
  iList := self _indexedPaths.
  bag := self _asIdentityBag.
  iList ~~ nil
    ifTrue: [ 
      prevObj := #'_incompletePathTraversal'.
      rootTerms := iList rootTerms.
      sz := rootTerms size.
      1 to: bag size do: [ :i | 
        obj := bag _at: i.
        obj == prevObj
          ifFalse: [ 1 to: sz do: [ :j | (rootTerms at: j) allDependencyListsFor: obj into: all ] ].
        prevObj := obj ] ].
  ^ all
%

category: 'Updating Indexes - Private'
method: UnorderedCollection
_doUpdateIndexesForAdditionOf: anObject iList: iList logging: doLogging
  "anObject is being added to the receiver.  Update any indexes if necessary.
 Returns true if the index objects were modified correctly; otherwise returns
 an Array containing error information."

  | mapInfo rootTerms i sz num val prevTerm aSet rootObjects |
  rootTerms := iList rootTerms.
  sz := rootTerms size.
  1 to: iList size by: 2 do: [ :j | 
    | indexObj offset indexDict pathTerm |
    "Make a pass through the indexedPaths to see if the receiver participates as a
 set-valued instance variable.  If so, then must update the index dictionary."
    indexObj := iList at: j.
    offset := iList at: j + 1.
    (offset > 1
      _and: [ 
        indexObj isComplete
          _and: [ 
            " get the previous path term (the one with an asterisk) "
            pathTerm := indexObj at: offset - 1.
            pathTerm hasIndexDictionary ] ])
      ifTrue: [ 
        " lazy initialization of aSet (in case we don't ever need it) "
        aSet == nil
          ifTrue: [ aSet := IdentitySet new ].	" only update the dictionary if we haven't already done it "
        (aSet includesIdentical: pathTerm)
          ifFalse: [ 
            aSet add: pathTerm.
            indexDict := indexObj indexDictionary.	" see how many entries exist "
            val := indexDict
              at: self
              term: pathTerm getParentTerm
              otherwise: nil.
            nil == val
              ifTrue: [ indexDict _errorNoEntryForKey: self term: pathTerm getParentTerm value: nil ]
              ifFalse: [ 
                (BucketValueBag _hasInstance: val)
                  ifTrue: [ num := val size ]
                  ifFalse: [ num := 1 ] ].
            num
              timesRepeat: [ 
                " add a dictionary entry for anObject -> NSC "
                indexDict
                  _at: anObject
                  put: self
                  term: pathTerm
                  logging: doLogging ] ] ] ].
  i := 1.
  [ i <= sz and: [ i <= rootTerms size ] ]
    whileTrue: [ 
      | pathTerm |
      " now make a pass through the root path terms "
      (pathTerm := rootTerms at: i) _isObsoletePathTerm
        ifFalse: [ 
          " see if the receiver is participating as a set-valued instance variable "
          pathTerm offset == 1
            ifTrue: [ num := 1 ]
            ifFalse: [ 
              pathTerm hasIndexDictionary
                ifTrue: [ 
                  " get the path term before the set-valued path term "
                  prevTerm := pathTerm getParentTerm.
                  pathTerm indicatesIndexOnNscElements
                    ifFalse: [ prevTerm := prevTerm getParentTerm ].
                  val := pathTerm getIndexDictionary
                    at: self
                    term: prevTerm
                    otherwise: nil.
                  nil == val
                    ifTrue: [ 
                      " if not found, then mappings have already been added "
                      num := 0 ]
                    ifFalse: [ 
                      (BucketValueBag _hasInstance: val)
                        ifTrue: [ num := val size ]
                        ifFalse: [ num := 1 ] ] ]
                ifFalse: [ 
                  "receiver is the '*' for a set valued term"
                  prevTerm := pathTerm indicatesIndexOnNscElements
                    ifTrue: [ pathTerm ]
                    ifFalse: [ pathTerm getParentTerm ].
                  rootObjects := prevTerm findReachableRootsFor: self.
                  pathTerm indicatesIndexOnNscElements
                    ifFalse: [ rootObjects := rootObjects asIdentitySet ] ] ].
          pathTerm indicatesIndexOnNscElements
            ifTrue: [ 
              " if the index is on elements of the NSC, go ahead and update the indexes "
              pathTerm updateBtree
                ifNotNil: [ 
                  (pathTerm _checkBtreeComparisonWith: anObject)
                    ifFalse: [ 
                      | exa |
                      (exa := ImproperOperation new)
                        _number:
                            (ErrorSymbols at: #'rtErrRangeEqualityIndexInvalidClassKindForBtree');
                        args: { anObject class. pathTerm name. pathTerm updateBtree lastElementClassDescription}.
                      i > 1
                        ifTrue: [ 
                          " indexing objects have been modified, prevent commits "
                          exa signal ]
                        ifFalse: [ 
                          ^ {false.	" indicating no indexing objects were modified "
                          exa} ] ] ].
              pathTerm hasIndexDictionary
                ifTrue: [ num timesRepeat: [ pathTerm addDirectMappingFor: anObject logging: doLogging ] ]
                ifFalse: [ 
                  rootObjects == nil
                    ifTrue: [ 
                      pathTerm
                        addDirectMappingFor: anObject
                        root: anObject
                        nsc: self
                        logging: doLogging ]
                    ifFalse: [ 
                      rootObjects
                        do: [ :root | 
                          pathTerm
                            addDirectMappingFor: anObject
                            root: root
                            nsc: self
                            logging: doLogging ] ] ] ]
            ifFalse: [ 
              (pathTerm _doNotPerformPretraversalFor: anObject)
                ifTrue: [ 
                  " if the path has too many set-valued terms or we're using GsPathTerms, go ahead and update the indexes ... potential for too much or incorrect MappingInfo."
                  rootObjects
                    ifNil: [ num timesRepeat: [ pathTerm addMappingsForObject: anObject logging: doLogging ] ]
                    ifNotNil: [ 
                      rootObjects
                        do: [ :root | pathTerm addMappingsForObject: anObject root: root logging: doLogging ] ] ]
                ifFalse: [ 
                  " get all the mapping info first (this detects any errors
                      along the path before any changes are made to the indexes) "
                  mapInfo := pathTerm
                    getMappingInfoFor: anObject
                    ifObject: nil
                    atOffset: 0
                    replaceWith: nil.	" if the result is not a map info object, it is an Array
                      used for error information "
                  mapInfo class == Array
                    ifTrue: [ 
                      i > 1
                        ifTrue: [ 
                          " indexing objects have been modified, prevent commits "
                          (mapInfo at: 2) signal ].
                      ^ mapInfo ].
                  " now update dependency lists, index dictionary, B-trees "
                  num
                    timesRepeat: [ mapInfo pathTerm addMappingsUsing: mapInfo logging: doLogging ] ] ] ].
      i := i + 1 ].
  ^ true
%
category: 'Updating Indexes - Private'
method: UnorderedCollection
_doUpdateIndexesForRemovalOf: anObject iList: iList

"anObject is being removed from the receiver.  Update the indexes if
 necessary.  This method is invoked prior to removal of anObject
 from the receiver. "

  | depList lastOne doLogging origSize rootTerms sz i pathTerm num prevTerm val rootObjects currSize indexObj offset aSet  |
 	  " get the removed object's dependency list "
	  depList := DependencyList for: anObject.

	  " see if another occurrence of anObject is in the index objects "
	  lastOne := self _isLastOccurrenceInIndexObjects: anObject.

	  doLogging := UnorderedCollection _isRcIndexLoggingEnabled.
	  origSize := iList size.

	  rootTerms := iList rootTerms.
	  sz := rootTerms size.
	  i := 1.
	  " for each unique path on which there is an index "
	  [ i <= sz and: [ i <= rootTerms size ] ] whileTrue: [
      num := nil.
		pathTerm := rootTerms at: i.
		" see if the path term is still a root term and is not obsolete "
		((iList rootTerms includesIdentical: pathTerm) and:
		       [ pathTerm _isObsoletePathTerm not ]) ifTrue: [
			" see if the receiver is participating as a set-valued instance variable "
			pathTerm offset == 1
			  ifTrue: [ num := 1 ]
			  ifFalse: [
				pathTerm hasIndexDictionary
               ifTrue: [" get the path term before the set-valued path term "
				    prevTerm := pathTerm getParentTerm.
				    pathTerm indicatesIndexOnNscElements ifFalse: [ prevTerm := prevTerm getParentTerm ].

				    val := pathTerm getIndexDictionary at: self term: prevTerm otherwise: nil.
				    " if not found, then mappings have already been removed "
				    val ifNil: [ num := 0 ]
					   ifNotNil: [
					     (BucketValueBag _hasInstance: val)
						    ifTrue: [ num := val size ]
						    ifFalse: [ num := 1 ]
					   ] ]
                ifFalse: [ 
                  "receiver is the '*' for a set valued term"
                  prevTerm := pathTerm indicatesIndexOnNscElements
                    ifTrue: [ pathTerm ]
                    ifFalse: [ pathTerm getParentTerm ]. 
                  rootObjects := (prevTerm findReachableRootsFor: self) asIdentitySet ] ].

			" see if this index is on the elements of the NSC "
			pathTerm indicatesIndexOnNscElements ifTrue: [
              (pathTerm hasIndexDictionary or: [ rootObjects == nil  ])
                ifTrue: [ num timesRepeat: [ pathTerm removeDirectMappingFor: anObject logging: doLogging ] ]
                ifFalse: [ 
                  rootObjects do: [:root | 
                    pathTerm removeDirectMappingFor: anObject root: root nsc: self logging: doLogging ] ].

				 ( depList ~~ nil and: [ pathTerm needsDepList ] )
				   ifTrue: [ DependencyList removePathTerm: pathTerm for: anObject ]
			  ] ifFalse: [
				anObject ifNotNil: [
                (pathTerm hasIndexDictionary or: [ num ~~ nil ])
                  ifTrue: [ 
					  num - 1 timesRepeat: [
					    pathTerm removeMappingsFor: anObject
					      lastOne: false
					     logging: doLogging ].
					  pathTerm removeMappingsFor: anObject
					    lastOne: lastOne
					    logging: doLogging ]
                 ifFalse: [
                   rootObjects do: [:root | 
                     pathTerm removeMappingsFor: anObject
                        root: root
					      lastOne: lastOne
					      logging: doLogging ] ] .
				( depList ~~ nil and: [ lastOne ] ) ifTrue: [ 
					DependencyList removePathTerm: pathTerm for: anObject 
				]
			  ] ].
		  ].
		i := i + 1
	  ].

     pathTerm hasIndexDictionary
       ifTrue: [
	      currSize := iList size.
	      i := 1.
	      [ i <= origSize and: [ i <= currSize ] ] whileTrue: [
		    indexObj := iList at: i.
		    offset := iList at: i + 1.
		    " see if receiver participates as a set-valued instance variable "
		    (offset > 1 and: [ indexObj isComplete ]) ifTrue: [
			   " lazy initialization of aSet (in case we don't ever need it) "
			   aSet ifNil: [ aSet := IdentitySet new ].
			   " get the previous path term (the one with an asterisk) "
			   pathTerm := indexObj at: offset - 1.
			   " only update the dictionary if we haven't already done it "
			   (aSet includesIdentical: pathTerm) ifFalse: [
				   aSet add: pathTerm.
				   " see how many entries exist "
				   val := indexObj indexDictionary at: self term: pathTerm getParentTerm otherwise: nil.

				   " if not found, then mappings have already been removed "
				   val ifNil: [ num := 0 ]
				     ifNotNil: [
					    (BucketValueBag _hasInstance: val)
				         ifTrue: [ num := val size ]
					  	  ifFalse: [ num := 1 ]
				  	   ].

				   num timesRepeat: [
				     " remove dictionary entry for anObject -> NSC "
				     indexObj indexDictionary removeKey: anObject value: self 
									term: pathTerm logging: doLogging
				   ] ] ].
		    i := i + 2
	      ] ].
%
category: 'Indexing Support'
classmethod: UnorderedCollection
_canCreateQueryOnInstances
  "GsQuery may be created on most subclasses of Collection. Answer false if a GsQuery is not appropriate
   for the receiver."

  ^ true
%

category: 'Reduced Conflict Support'
method: UnorderedCollection
_validateLegacyRcIndexSupport
  "legacy indexes are not supported on RcIndentitySet or RcLowMaintenanceIdentityBag (bug47179)"

%
