!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   BlockSorter, Array, SequenceableCollection, Collection, Object.
!
!=========================================================================

expectvalue %String
run
| oldCls newCls |
oldCls := Globals at:#BlockSorter otherwise: nil .
oldCls == nil ifTrue:[
  Array _newKernelSubclass: #BlockSorter
    instVarNames: #( #sortBlock #sortNodes )
    classVars: #()
    classInstVars: #()
    poolDictionaries: #()
    inDictionary: Globals
    options: #() 
    reservedOop: nil.
  newCls := (Globals at:#BlockSorter) .
  ^ 'created new class: ' , newCls definition
  ]
ifFalse:[
  ^ 'existing class: ' , oldCls definition
  ]
%

! Remove existing behavior from BlockSorter
removeallmethods BlockSorter 
removeallclassmethods BlockSorter 

category: 'For Documentation Installation only'
classmethod: BlockSorter
installDocumentation

self comment:
'BlockSorter is a concrete class that along with the class SortBlockNode implements the 
 behavior used to sort collections efficiently. It is an interface to an algorithm based 
 on an efficient merge-sort.  

 To use BlockSorter explicitly outside of the indexing subsystem, 
 the recommended way is to send  sortWithBlock:  or  sortWithBlock:persistentRoot:
 to a kind of Collection .

Constraints:
	sortBlock: Object
	sortNodes: Array' .
%
! documentation edits to fix 37111 

! ------------------- Class methods for BlockSorter
category: 'Instance Creation'
classmethod: BlockSorter
on: anNsc with: sortBlock

^ self on: anNsc with: sortBlock collator: nil
%

category: 'Instance Creation'
classmethod: BlockSorter
on: anNsc with: sortBlock collator: anIcuCollator

 "Returns an initialized instance of BlockSorter.

 To use BlockSorter explicitly outside of the indexing subsystem, 
 the recommended way is to send  sortWithBlock:  or  sortWithBlock:persistentRoot:
 to a kind of Collection."

| inst sortNode  |
inst := self new.
inst sortBlock: sortBlock collator: anIcuCollator .
sortNode := inst sortNodeClassForSort new: anNsc size.
sortNode blockSorter: inst.

inst sortNodes: (SortNodeArray with: sortNode).
^ inst
%

! ------------------- Instance methods for BlockSorter
category: 'Accessing'
method: BlockSorter
sortBlock

^sortBlock
%
category: 'Updating'
method: BlockSorter
sortBlock: aBlock 

sortBlock := aBlock
%

category: 'Updating'
method: BlockSorter
sortBlock: aBlock  collator: anIcuCollator

sortBlock := aBlock .
anIcuCollator ifNotNil:[
  self dynamicInstVarAt: #collator put: anIcuCollator .
].
%

category: 'Private'
method: BlockSorter
sortInto: anArray startingAt: index

"Returns anArray filled with objects that have been added to the receiver, in
 sorted order."
sortNodes size == 1
  ifTrue: [ (sortNodes at: 1) sortInto: anArray startingAt: index ]
  ifFalse: [
    sortNodes size == 2
      ifTrue: [ sortNodes binaryMergeAscendingInto: anArray startingAt: index  ]
      ifFalse: [  sortNodes nwayMergeAscendingInto: anArray startingAt: index ].
  ].
^ anArray
%
category: 'Accessing'
method: BlockSorter
sortNodeClassForSort

"Returns the class of SortBlockNode to use."
^ (self dynamicInstVarAt: #collator) 
    ifNil:[  SortBlockNode ] 
    ifNotNil:[ SortBlockUnicodeNode ]
%
category: 'Accessing'
method: BlockSorter
sortNodes

"Returns the value of the instance variable 'sortNodes'."

^sortNodes
%
category: 'Updating'
method: BlockSorter
sortNodes: anArray

sortNodes := anArray
%
category: 'Updating'
method: BlockSorter
_addObject: anObject
inNodes: nodeArray

  | lastNode |
    lastNode := nodeArray at: nodeArray size.
    " if the last sort node is full, create a new one "
    lastNode isFull
      ifTrue: [
        lastNode := lastNode class new: lastNode numElements.
        lastNode blockSorter: self.
        nodeArray add: lastNode
      ].
  " add an entry to the sort node "
  lastNode _at: anObject
    put: anObject
    forBtree: false
%

! fix 46903,  use SortedCollection  for sizes <= 2000 
category: 'Sorting'
method: Collection
sortWithBlock: sortBlock

"Returns an Array containing the elements of the receiver, sorted based on the
 sortBlock. The block must take two arguments, and it should return true if the 
 first argument should precede the second argument, and false if not. The 
 expressions within the block are expected to by symmetrical (i.e., if the first
 arg is _less_ than the second, then the block will return false when the arguments
 are reversed and if the first arg is _equal_ to the second arg, then the block 
 will return false regardless of argument order)."

  self size <= 2000 ifTrue:[ | sc |
    sc := SortedCollection sortBlock: sortBlock .
    self do:[:elem | sc add: elem ]  .
    ^ Array withAll: sc . 
  ] ifFalse:[
    ^ self sortWithBlock: sortBlock  persistentRoot: nil collator: nil .
  ]
%

category: 'Sorting'
method: Collection
sortWithBlock: sortBlock collator: anIcuCollator

^ self sortWithBlock: sortBlock  persistentRoot: nil collator: anIcuCollator 
%

category: 'Sorting'
method: Collection
sortWithBlock: sortBlock  persistentRoot: persistentArrayOrNil

^ self sortWithBlock: sortBlock  persistentRoot: persistentArrayOrNil collator: nil 
%

category: 'Sorting'
method: Collection
sortWithBlock: sortBlock  persistentRoot: persistentArrayOrNil collator: anIcuCollator

"Returns an Array containing the elements of the receiver, sorted based on the
 sortBlock. 

 If anIcuCollator == nil, the sortBlock must take two arguments.
 If anIcuCollator is an IcuCollator, then the block must take a 3rd argument which
 is an IcuCollator, as used in the blocks for IcuSortedCollection.  

 The sortBlock should return true if the first argument should precede the second argument, 
 and false if not. The expressions within the block are expected to by symmetrical 
 (i.e., if the first arg is _less_ than the second, then the block will return false 
 when the arguments are reversed and if the first arg is _equal_ to the second arg, 
 then the block will return false regardless of argument order).

 If persistentArrayOrNil is notNil, then it is expected to be an empty persistent array and
 the array will be used to persist large temporary data structures created during
 the sorting operation. IndexManager>>autoCommit must be true in order for periodic
 commits to  be made during the sorting operation. When the sort operation is complete
 the persistent array will be emptied and a final commit performed. The persistentArrayOrNil 
 and  IndexManager>>autoCommit should be used when a collection is so large that it 
 isn't practical to allocate enough temporary memory.
"

 | sorter aColl sortNodes |
sorter := BlockSorter on: self with: sortBlock collator: anIcuCollator .
sortNodes := sorter sortNodes.
aColl := self _asCollectionForSorting.
(IndexManager autoCommit and: [ persistentArrayOrNil ~~ nil ])
  ifTrue: [
    | result obj mgr |
    (mgr := IndexManager current) executeStartingIndexMaintenance: [
      persistentArrayOrNil add: sorter.
      persistentArrayOrNil add: aColl.
      mgr _doCommit.
      1 to: aColl size do: [ :i |
        obj := aColl _at: i.
         sorter _addObject: obj
                  inNodes: sortNodes.
      ].
      result := sorter sortInto: (Array new: self size) startingAt: 1.
      persistentArrayOrNil size: 0 ].
    ^result ]
  ifFalse: [ | result obj |
    1 to: aColl size do: [ :i |
      obj := aColl _at: i.
       sorter _addObject: obj
                inNodes: sortNodes.
    ].
    result := sorter sortInto: (Array new: self size) startingAt: 1.
    ^ result
].
%

! deletion
