!=========================================================================
! Copyright (C) GemTalk Systems 2017-2020.  All Rights Reserved.
!
! $Id: Btree-Stream-Core.gs 38384 2016-01-08 18:22:36Z lalmarod $
!
! Btree-Stream-Core.gs  -  source code for the gs stream classes
!
!========================================================================

! class created in btreeplusclasses.gs

! Class Implementation for BtreePlusReadStream

! Remove existing behavior from BtreePlusReadStream
removeallmethods BtreePlusReadStream
removeallclassmethods BtreePlusReadStream

! ------------------- Class methods for BtreePlusReadStream

category: 'Instance Creation'
classmethod: BtreePlusReadStream
on: aBtreeNode

"Create a stream that can access the entire contents of the B-tree whose
 root node is aBtreeNode."

| arr1 arr2 newStream |
arr1 := { } .
arr2 := { } .

aBtreeNode _putFirstIndexOfFirstChildInto: arr1.
aBtreeNode _putLastIndexOfLastChildInto: arr2.

newStream := self new.
newStream currentIndex: (arr1 at: 1).
newStream currentNode: (arr1 at: 2).
newStream endIndex: (arr2 at: 1).
newStream endNode: (arr2 at: 2).
^ newStream
%

! ------------------- Instance methods for BtreePlusReadStream

category: 'Testing'
method: BtreePlusReadStream
_atEnd
  "Returns true if the receiver is positioned at the end of the stream, and
 false otherwise."

  ^ currentIndex == 0
%

category: 'Testing'
method: BtreePlusReadStream
_btreeAtEnd

""

^ currentIndex == 0
%

category: 'Accessing'
method: BtreePlusReadStream
_btreeNext

"Returns the next value on a stream of B-tree values and root objects.  Updates the current
 stack for a subsequent 'next'."

^ self _btreeNextValue
%

category: 'Accessing'
method: BtreePlusReadStream
_btreeNextRootObject
  "Returns the next root object on a stream of B-tree values and root objects.  Updates the current
 stack for a subsequent 'next'."

  | root |
  " get the root value within the leaf "
  root := currentNode rootObjectAt: currentIndex.
  " if this is the end of the stream, sets the top currentIndex to zero and returns "
  (currentNode == endNode and: [ endIndex == currentIndex ])
    ifTrue: [ currentIndex := 0.
      ^ root ].
  " see if currentIndex refers to last entry in this leaf "
  currentIndex == currentLastValIndex
    ifTrue: [ self _nextLeaf ]
    ifFalse: [ currentIndex := currentIndex + currentEntrySize ].
  ^ root
%

category: 'Accessing'
method: BtreePlusReadStream
_btreeNextValue

"Returns the next value on a stream of B-tree values and root objects.  Updates the current
 stack for a subsequent 'next'."

| val    |
" get the index into the leaf node and see if it has reached the end "
currentIndex == 0
    ifTrue: [ ^ self _errorEndOfStream ].

" get the leaf and the value within the leaf "
val := currentNode at: currentIndex.

" if this is the end of the stream, sets the top index to zero and returns "
( currentNode == endNode and: [ endIndex == currentIndex ] )
    ifTrue: [
        currentIndex := 0.
        ^ val
    ].

" see if index refers to last entry in this leaf "
currentIndex == currentLastValIndex
    ifTrue: [
        " must look down the stack for the next leaf node "
        self _nextLeaf
    ]
    ifFalse: [ currentIndex := (currentIndex + currentEntrySize) ].

^ val
%

category: 'Error Handling'
method: BtreePlusReadStream
_errorEndOfStream

"Raises an error due to an attempt to read beyond the end of the stream."

^ self _error: #rtErrBtreeReadStreamEndOfStream .
%

category: 'Query Support'
method: BtreePlusReadStream
_hasSameEndAs: otherStream

"Returns true if the receiver has the same end index and end node as the
 otherStream."

^ endIndex == otherStream endIndex and: [
   endNode == otherStream endNode ]
%

category: 'Accessing'
method: BtreePlusReadStream
_nextLeaf
  "Get the next leaf node and make it the current node."

  self currentNode: currentNode nextLeaf.
  currentNode ifNil: [ ^ self _errorEndOfStream ].
  currentIndex := 1
%

category: 'Private'
method: BtreePlusReadStream
_peekKey
  "Returns the key of the current entry in the stream."

  currentIndex == 0
    ifTrue: [ ^ self _errorEndOfStream ].
  ^ currentNode keyAt: currentIndex
%

category: 'Private'
method: BtreePlusReadStream
_peekRoot
  "Returns the root of the current entry in the stream."


  currentIndex == 0
    ifTrue: [ ^ self _errorEndOfStream ].
  ^ currentNode rootObjectAt: currentIndex
%

category: 'Private'
method: BtreePlusReadStream
_peekValue
  "Returns the value of the current entry in the stream."

  currentIndex == 0
    ifTrue: [ ^ self _errorEndOfStream ].
  ^ currentNode at: currentIndex
%

category: 'Query Support'
method: BtreePlusReadStream
_positionToEnd

"Sets the current stack to the end index and end node.  This
 has the effect of advancing the stream to the last object."

currentIndex := endIndex.
self currentNode: endNode
%

category: 'Testing'
method: BtreePlusReadStream
atEnd
  "Returns true if the receiver is positioned at the end of the stream, and
 false otherwise."

  ^ self _atEnd
%

category: 'Accessing'
method: BtreePlusReadStream
currentIndex

"Returns the value of the instance variable 'currentIndex'."

^currentIndex
%

category: 'Updating'
method: BtreePlusReadStream
currentIndex: newValue

"Modify the value of the instance variable 'currentIndex'."

currentIndex := newValue
%

category: 'Accessing'
method: BtreePlusReadStream
currentNode

"Returns the value of the instance variable 'currentNode'."

^currentNode
%

category: 'Updating'
method: BtreePlusReadStream
currentNode: newValue
  "Modify the value of the instance variable 'currentNode'."

  currentNode := newValue.
  currentLastValIndex := currentNode _lastEntryIndex.
  currentEntrySize := currentNode entrySize.
%

category: 'Accessing'
method: BtreePlusReadStream
currentStack: array1

"Set the currentIndex and currentNode from array1"

currentIndex := array1 at: 1.
currentIndex ~~ 0
  ifTrue: [ self currentNode: ( array1 at: 2) ]
%

category: 'Accessing'
method: BtreePlusReadStream
endIndex

"Returns the value of the instance variable 'endIndex'."

^endIndex
%

category: 'Updating'
method: BtreePlusReadStream
endIndex: newValue

"Modify the value of the instance variable 'endIndex'."

endIndex := newValue
%

category: 'Accessing'
method: BtreePlusReadStream
endNode

"Returns the value of the instance variable 'endNode'."

^endNode
%

category: 'Updating'
method: BtreePlusReadStream
endNode: newValue

"Modify the value of the instance variable 'endNode'."

endNode := newValue
%

category: 'Accessing'
method: BtreePlusReadStream
next
  "Returns the next root object on a stream of B-tree values and root objects.  Updates the current
 stack for a subsequent 'next'."

  | root |
  currentIndex == 0
    ifTrue: [ ^ self _errorEndOfStream ].	" get the root value within the leaf "
  root := currentNode rootObjectAt: currentIndex.	" if this is the end of the stream, sets the top currentIndex to zero and returns "
  (currentNode == endNode and: [ endIndex == currentIndex ])
    ifTrue: [ currentIndex := 0.
      ^ root ].	" see if currentIndex refers to last entry in this leaf "
  currentIndex == currentLastValIndex
    ifTrue: [ self _nextLeaf ]
    ifFalse: [ currentIndex := currentIndex + currentEntrySize ].
  ^ root
%

category: 'Adding'
method: BtreePlusReadStream
nextPut: anObject

"Disallowed.  You cannot write to a BtreeReadStream."

self shouldNotImplement: #nextPut:
%

category: 'Accessing'
method: BtreePlusReadStream
nextRoot
  "Returns the next root object on a stream of B-tree values and root objects.  Update the current
 stack for a subsequent 'next'."

  currentIndex == 0
    ifTrue: [ ^ self _errorEndOfStream ].
  ^ self _btreeNextRootObject
%

category: 'Accessing'
method: BtreePlusReadStream
size

"Returns the number of elements contained in the receiver (that is, how many
successful 'next' operations can be performed)."

| total theNode theIndex lastNode |
self atEnd ifTrue: [ 
  ^ 0 
  ].
total := 0.
theNode := currentNode.
theIndex := currentIndex.
lastNode := false.
[ lastNode ] whileFalse: [ 
  lastNode := theNode == endNode.
  total := total + (theNode
    _totalElementsFrom: theIndex
    toEndNode: endNode
    endIndex: endIndex).
  theNode := theNode nextLeaf ].
^ total
%

! Class Implementation for BtreePlusGsIndexReadStream

! Remove existing behavior from BtreePlusGsIndexReadStream
removeallmethods BtreePlusGsIndexReadStream
removeallclassmethods BtreePlusGsIndexReadStream

! ------------------- Class methods for BtreePlusGsIndexReadStream

category: 'Instance Creation'
classmethod: BtreePlusGsIndexReadStream
on: aGsRangeIndex
  "Create a stream that can access the entire contents of the given RangeEqualityIndex."

  | newStream |
  newStream := super on: aGsRangeIndex btreeRoot.
  newStream rangeIndex: aGsRangeIndex.
  ^ newStream
%

! ------------------- Instance methods for BtreePlusGsIndexReadStream

category: 'Private'
method: BtreePlusGsIndexReadStream
_advance
   "Advances the B-tree portion of the stream without advancing any of the set
     iteration indexes."
 
   ^ self next
%

category: 'Query Support'
method: BtreePlusGsIndexReadStream
_valuesAddInto: collection

"Adds each entry value into the given NSC."

| querySpec |
querySpec := BtreeQuerySpec new.
^self _valuesAddInto: collection spec: querySpec
%

category: 'Query Support'
method: BtreePlusGsIndexReadStream
_valuesAddInto: collection comparing: aSelector key: aKey
  "Adds each entry value into the given NSC."

  | querySpec |
  querySpec := self btreeComparisonQuerySpec
    key: aKey selector: aSelector;
    yourself.
  ^ self _valuesAddInto: collection spec: querySpec
%

category: 'Query Support'
method: BtreePlusGsIndexReadStream
_valuesAddInto: collection comparing: aSelector1 key: aKey1 and: aSelector2 key: aKey2
  "Adds each entry value into the given NSC."

  | querySpec |
  querySpec := self btreeRangeComparisonQuerySpec
    key: aKey1 selector: aSelector1;
    key2: aKey2 selector2: aSelector2;
    yourself.
  ^ self _valuesAddInto: collection spec: querySpec
%

category: 'Query Support'
method: BtreePlusGsIndexReadStream
_valuesAddInto: collection identicalTo: aKey
  "For each entry whose key is identical to the given key, add the value into
 the given collection."

  | querySpec |
  querySpec := self btreeComparisonQuerySpec
    key: aKey selector: #'==';
    yourself.
  ^ self _valuesAddInto: collection spec: querySpec
%

category: 'Query Support'
method: BtreePlusGsIndexReadStream
_valuesAddInto: collection notEqualTo: aKey
  "For each entry whose key is not identical to the given key, add the value
 into the given collection."

  | querySpec |
  querySpec := self btreeComparisonQuerySpec
    key: aKey selector: #'~=';
    yourself.
  ^ self _valuesAddInto: collection spec: querySpec
%

category: 'Query Support'
method: BtreePlusGsIndexReadStream
_valuesAddInto: collection notIdenticalTo: aKey
  "For each entry whose key is not identical to the given key, add the value
 into the given collection."

  | querySpec |
  querySpec := self btreeComparisonQuerySpec
    key: aKey selector: #'~~';
    yourself.
  ^ self _valuesAddInto: collection spec: querySpec
%

category: 'Query Support'
method: BtreePlusGsIndexReadStream
_valuesAddInto: collection spec: querySpec
  "Adds each entry value into the given NSC."

  | eSize |
  " get the index into the leaf node and see if it has reached the end "
  currentIndex == 0
    ifTrue: [ ^ self ].
  eSize := endNode entrySize.
  " while current node is not the end node "
  [ endNode ~~ currentNode ]
    whileTrue: [ " put the values for the current node in the collection "
      querySpec
        addRootObjectsFrom: currentNode
        start: currentIndex
        end: currentNode numElements * eSize
        into: collection.
      " get the next leaf node "
      self _nextLeaf ].
  " put values of end node in the collection "
  querySpec
    addRootObjectsFrom: currentNode
    start: currentIndex
    end: endIndex
    into: collection
%

category: 'Testing'
method: BtreePlusGsIndexReadStream
atEnd
  "Returns true if there are no more elements to return through the logical
 iteration of the stream."

  mustCompare
    ifTrue: [ "bug 44079: need to skip key/value pairs that do not satisfy the querySpec"
      [ | aKey |
      currentIndex == 0
        ifTrue: [ ^ true ].
      aKey := currentNode keyAt: currentIndex.
      streamQuerySpec compareKey: aKey ] whileFalse: [ 
          self _btreeNextRootObject.
          currentIndex == 0
            ifTrue: [ ^ true ] ] ].
  ^ currentIndex == 0
%

category: 'Accessing'
method: BtreePlusGsIndexReadStream
btreeComparisonQuerySpec
  ^ self gsIndex btreeComparisonQuerySpec
%

category: 'Accessing'
method: BtreePlusGsIndexReadStream
btreeRangeComparisonQuerySpec
  ^ self gsIndex btreeRangeComparisonQuerySpec
%

category: 'Updating'
method: BtreePlusGsIndexReadStream
gsIndex
  "Answer the value of the instance variable 'gsIndex'."

  ^gsIndex
%

category: 'Updating'
method: BtreePlusGsIndexReadStream
identityIndex: aGsIndex
  "Modify the value of the instance variable 'gsIndex'."

  gsIndex := aGsIndex
%

category: 'Query Support'
method: BtreePlusGsIndexReadStream
makeNscComparing: aSelector key: aKey

"Returns a new NSC that contains all the elements from the receiver."

| tmpHolder |
tmpHolder := NscBuilder for: gsIndex nscRoot species new max: gsIndex nscRoot size.

self _valuesAddInto: tmpHolder comparing: aSelector key: aKey.

" make the stream be at the end "
currentIndex := 0.

^ tmpHolder completeBag
%

category: 'Query Support'
method: BtreePlusGsIndexReadStream
makeNscComparing: aSelector1 key: aKey1 and: aSelector2 key: aKey2

"Returns a new NSC that contains all the elements from the receiver."

| tmpHolder |
tmpHolder := NscBuilder for: gsIndex nscRoot species new max: gsIndex nscRoot size.

self _valuesAddInto: tmpHolder comparing: aSelector1 key: aKey1 and: aSelector2 key: aKey2.

" make the stream be at the end "
currentIndex := 0.
^ tmpHolder completeBag
%

category: 'Accessing'
method: BtreePlusGsIndexReadStream
next
  "Returns the next root object on a stream of B-tree values and root objects.  Update the current
 stack for a subsequent 'next'."

  optimizedComparison
    ifTrue: [
      currentIndex == 0
        ifTrue: [ ^ self _errorEndOfStream ].
     ^ self _btreeNextRootObject ].
  "bug 44079: use #atEnd to skip key/value pairs that do not satisfy the querySpec ... "
  self atEnd
    ifTrue: [ ^ self _errorEndOfStream ].
  ^ self _btreeNextRootObject
%

category: 'Updating'
method: BtreePlusGsIndexReadStream
rangeIndex: aGsIndex
  "Modify the value of the instance variable 'gsIndex'."

  gsIndex := aGsIndex.
  optimizedComparison := gsIndex optimizingComparison.
  mustCompare := false
%

category: 'Accessing'
method: BtreePlusGsIndexReadStream
streamQuerySpec
  ^ streamQuerySpec
%

category: 'Accessing'
method: BtreePlusGsIndexReadStream
streamQuerySpec: aBtreeQuerySpec
  streamQuerySpec := aBtreeQuerySpec.
  (optimizedComparison
    and: [ aBtreeQuerySpec == nil or: [ aBtreeQuerySpec canUseOptimizedComparison ] ])
    ifTrue: [ mustCompare := false ]
    ifFalse: [ 
      "bug 44079: need to skip key/value pairs that do not satisfy the streamQuerySpec.
       If optimizedComparison is true, then not exposed to bug 44079. "
      mustCompare := aBtreeQuerySpec notNil ]
%

! Class Implementation for BtreePlusGsReversedIndexReadStream

! Remove existing behavior from BtreePlusGsReversedIndexReadStream
removeallmethods BtreePlusGsReversedIndexReadStream
removeallclassmethods BtreePlusGsReversedIndexReadStream

! ------------------- Class methods for BtreePlusGsReversedIndexReadStream

category: 'Instance Creation'
classmethod: BtreePlusGsReversedIndexReadStream
on: aGsRangeIndex
  "Create a stream that can access the entire contents of the B-tree whose
 root node is aBtreeNode."

  | arr1 arr2 newStream btreeNode |
  arr1 := {}.
  arr2 := {}.
  btreeNode := aGsRangeIndex btreeRoot.
  btreeNode _putFirstIndexOfFirstChildInto: arr2.
  btreeNode _putLastIndexOfLastChildInto: arr1.
  newStream := self new.
  newStream endIndex: (arr2 at: 1).
  newStream endNode: (arr2 at: 2).
  newStream currentStack: arr1.
  newStream rangeIndex: aGsRangeIndex.
  ^ newStream
%
! ------------------- Instance methods for BtreePlusGsReversedIndexReadStream
category: 'Accessing'
method: BtreePlusGsReversedIndexReadStream
_btreeNextRootObject
  "Returns the previous root object on a stream of B-tree values and root objects.  Updates the current
 stack for a subsequent 'previous'."

  ^self _btreePreviousRootObject
%
category: 'Accessing'
method: BtreePlusGsReversedIndexReadStream
_btreePreviousNoValue
  "Returns the next value on a stream of B-tree values and root objects.  Updates the current
 stack for a subsequent 'next'."

  | val |
  " get the index into the leaf node and see if it has reached the end "
  currentIndex == 0
    ifTrue: [ ^ self _errorEndOfStream ].	" get the leaf and the value within the leaf "
  (currentNode == endNode and: [ endIndex == currentIndex ])
    ifTrue: [ 
      currentIndex := 0.
      ^ self ].	" see if index refers to first entry in this leaf "
  currentIndex == 1
    ifTrue: [ 
      " must look down the stack for the next leaf node "
      self _previousLeaf ]
    ifFalse: [ currentIndex := currentIndex - currentEntrySize ].
%
category: 'Accessing'
method: BtreePlusGsReversedIndexReadStream
_btreeNextValue

"Returns the previous value on a stream of B-tree values and root objects.  Updates the current
 stack for a subsequent 'previous'."

^ self _btreePreviousValue
%
category: 'Accessing'
method: BtreePlusGsReversedIndexReadStream
_btreePreviousRootObject
  "Returns the next root object on a stream of B-tree values and root objects.  Updates the current
 stack for a subsequent 'next'."

  | root |
  " get the root value within the leaf "
  root := currentNode rootObjectAt: currentIndex.
  " if this is the end of the stream, sets the top currentIndex to zero and returns "
  (currentNode == endNode and: [ endIndex == currentIndex ])
    ifTrue: [ currentIndex := 0.
      ^ root ].
  " see if currentIndex refers to last entry in this leaf "
  currentIndex == 1
    ifTrue: [ self _previousLeaf ]
    ifFalse: [ currentIndex := currentIndex - currentEntrySize ].
  ^ root
%
category: 'Accessing'
method: BtreePlusGsReversedIndexReadStream
_btreePreviousValue

"Returns the next value on a stream of B-tree values and root objects.  Updates the current
 stack for a subsequent 'next'."

| val    |
" get the index into the leaf node and see if it has reached the end "
currentIndex == 0
    ifTrue: [ ^ self _errorEndOfStream ].

" get the leaf and the value within the leaf "
val := currentNode at: currentIndex.

" if this is the end of the stream, sets the top index to zero and returns "
( currentNode == endNode and: [ endIndex == currentIndex ] )
    ifTrue: [
        currentIndex := 0.
        ^ val
    ].

" see if index refers to first entry in this leaf "
currentIndex == 1
    ifTrue: [
        " must look down the stack for the next leaf node "
        self _previousLeaf
    ]
    ifFalse: [ currentIndex := (currentIndex - currentEntrySize) ].

^ val
%
category: 'Accessing'
method: BtreePlusGsReversedIndexReadStream
_previousLeaf
  "Get the previous leaf node and make it the current node."

  self currentNode: currentNode previousLeaf.
  currentNode ifNil: [ ^ self _errorEndOfStream ].
  currentIndex := currentLastValIndex
%
category: 'Private'
method: BtreePlusGsReversedIndexReadStream
reversedRangeIndexClass
  ^ BtreePlusGsIndexReadStream
%

! Class Extensions

! Class initializers 

doit
true.
%



! End of Package: Btree-Stream-Core


category: 'Query Support'
method: BtreePlusGsIndexReadStream
approxNumElements

"Returns the number of leaf node entries represented by the receiver.  This is
 approximately the number of elements because objects may share sub-objects
 which converge to the same leaf node entry."

^ self size
%
category: 'Conversion'
method: BtreePlusGsIndexReadStream
reversed

  | reversed |
  reversed := (self reversedRangeIndexClass on: self gsIndex).
  
  self atEnd
    ifTrue: [ reversed currentIndex: 0 ]
    ifFalse: [ 
      reversed
        currentIndex: endIndex;
        currentNode: endNode;
        endIndex: currentIndex;
        endNode: currentNode;
        streamQuerySpec: self streamQuerySpec;
        yourself ].
  ^ reversed 
%
category: 'Private'
method: BtreePlusGsIndexReadStream
reversedRangeIndexClass
  ^ BtreePlusGsReversedIndexReadStream
%
category: 'Accessing'
method: BtreePlusGsReversedIndexReadStream
size

"Returns the number of elements contained in the receiver (that is, how many
successful 'previous' operations can be performed)."

| total theNode theIndex lastNode |
self atEnd ifTrue: [ 
  ^ 0 
  ].
total := 0.
theNode := currentNode.
theIndex := currentIndex.
lastNode := false.
[ lastNode ] whileFalse: [ 
  lastNode := theNode == endNode.
  total := total + (theNode
    _totalElementsReversedFrom: theIndex
    toEndNode: endNode
    endIndex: endIndex).
  lastNode 
    ifFalse: [ 
      theNode := theNode previousLeaf.
      theIndex := theNode _lastIndex ] ].
^ total
%
category: 'Query Support'
method: BtreePlusGsIndexReadStream
nextRootComparing: aSelector key: aKey
  " iterate through the stream, return first value that satisfies query "

  [ self _atEnd not ]
    whileTrue: [ 
      (self _peekKey perform: aSelector with: aKey)
        ifTrue: [ ^ self next ].
      self _btreeNext ].
  ^ #'_incompletePathTraversal'
%
category: 'Query Support'
method: BtreePlusGsIndexReadStream
nextRootComparing: aSelector1 key: aKey1 and: aSelector2 key: aKey2
  " iterate through the stream, return first value that satisfies query "

  [ self _atEnd not ]
    whileTrue: [ 
      | key |
      key := self _peekKey.
      ((key perform: aSelector1 with: aKey1)
        and: [ key perform: aSelector2 with: aKey2 ])
        ifTrue: [ ^ self next ].
      self _btreeNext ].
  ^ #'_incompletePathTraversal'
%
category: 'Private'
method: BtreePlusGsIndexReadStream
getNext
  "Returns the next value on a stream of range index values."

  "QueryExecutor compat"

  ^ self next
%
category: 'Query Support'
method: BtreePlusGsIndexReadStream
makeNsc

"Returns a new NSC that contains all the elements from the receiver."

| tmpHolder |
tmpHolder := NscBuilder for: gsIndex nscRoot species new max: gsIndex nscRoot size.

streamQuerySpec 
  ifNil: [
    self _valuesAddInto: tmpHolder ]
  ifNotNil: [
    self _valuesAddInto: tmpHolder spec: streamQuerySpec ].

" make the stream be at the end "
currentIndex := 0.

^ tmpHolder completeBag
%
category: 'Query Support'
method: BtreePlusGsReversedIndexReadStream
_valuesAddInto: collection spec: querySpec
  "Adds each entry value into the given NSC."

  | eSize |
  " get the index into the leaf node and see if it has reached the end "
  currentIndex == 0
    ifTrue: [ ^ self ].
  eSize := endNode entrySize.
  " while current node is not the end node "
  [ endNode ~~ currentNode ]
    whileTrue: [ " put root objects for the current node in the collection "
      querySpec
        addRootObjectsFrom: currentNode
        start: 1
        end: currentIndex
        into: collection.
      " get the previous leaf node "
      self _previousLeaf ].
  " put root objects of end node in the collection "
  querySpec
    addRootObjectsFrom: currentNode
    start: endIndex
    end: currentIndex
    into: collection 
%
category: 'Accessing'
method: BtreePlusReadStream
_btreeNextNoValue

"Returns the next value on a stream of B-tree values and root objects.  Updates the current
 stack for a subsequent 'next'."

" get the index into the leaf node and see if it has reached the end "
currentIndex == 0
    ifTrue: [ ^ self _errorEndOfStream ].

" if this is the end of the stream, sets the top index to zero and returns "
( currentNode == endNode and: [ endIndex == currentIndex ] )
    ifTrue: [
        currentIndex := 0.
        ^ self
    ].

" see if index refers to last entry in this leaf "
currentIndex == currentLastValIndex
    ifTrue: [
        " must look down the stack for the next leaf node "
        self _nextLeaf
    ]
    ifFalse: [ currentIndex := (currentIndex + currentEntrySize) ].
%
category: 'Accessing'
method: BtreePlusGsIndexReadStream
skip: amount
  "advance the receiver's position by amount elements"

  | count |
  count := 0.
  [ self atEnd not and: [ count < amount ] ]
    whileTrue: [ 
      self _btreeNextNoValue.
      count := count + 1 ]
%
category: 'Accessing'
method: BtreePlusGsReversedIndexReadStream
skip: amount
  "advance the receiver's position by amount elements"

  | count |
  count := 0.
  [ self atEnd not and: [ count < amount ] ]
    whileTrue: [ 
      self _btreePreviousNoValue.
      count := count + 1 ]
%
category: 'Query Support'
method: BtreePlusGsIndexReadStream
_scanForRootObjectMaps: aKey value: aValue valueOop: valueOop using: comparisonForCompare do: aBlock
  "Adds each entry value into the given NSC."

  | eSize |
  " get the index into the leaf node and see if it has reached the end "
  currentIndex == 0
    ifTrue: [ ^ self ].
  eSize := endNode entrySize.
  " while current node is not the end node "
  [ endNode ~~ currentNode ]
    whileTrue: [ 
      " scan key and values of currentNode "
      currentNode 
        _scanForRootObjectMapsStartingAt: currentIndex
        end: currentLastValIndex
        key: aKey 
        value: aValue 
        valueOop: valueOop
        using: comparisonForCompare
        do: aBlock.
      " get the next leaf node "
      self _nextLeaf ].
  " scan key and values of endNode "
  currentNode 
    _scanForRootObjectMapsStartingAt: currentIndex
    end: endIndex
    key: aKey 
    value: aValue 
    valueOop: valueOop
    using: comparisonForCompare
    do: aBlock.
%
