!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
! Superclass Hierarchy:
!   GsReferencePathResult, Array
!
!=========================================================================

expectvalue /Class
run
GsSingleRefPathResult comment: 
'GsSingleRefPathResult is used to represent the results of a reference path scan.
It is a subclass of Array and contains the following instance variables:

searchOop - the object for which the reference path was found.

isDead - a Boolean indicating if the search object was found to be dead
(disconnected from the repository).

The first elment of a GsSingleRefPathResult will be the limit object at the top
of the reference path.  The last element will be searchOop.  The other elements
represent intermediate objects which are traversered to go from the limit object
to the search object.'
%

! ------------------- Remove existing behavior from GsSingleRefPathResult
removeallmethods GsSingleRefPathResult
removeallclassmethods GsSingleRefPathResult

! ------------------- Class methods for GsSingleRefPathResult
set compile_env: 0
category: 'Instance Creation'
classmethod: GsSingleRefPathResult
newForSearchObject: anObj isDead: aBoolean path: anArray

"Creates a new GsSingleRefPathResult object for the given object. The 
complete reference path is contained in anArray.  The first element of
anArray must be a member of the limit set and the last element must
be the search object.

Raises an error if the reference path is not valid for the given search oop."

|result|
result := self new.
result searchOop: anObj.
result isDead: aBoolean.
result addAll: anArray.
result validate.
^result
%
! ------------------- Instance methods for GsSingleRefPathResult

category: 'Accessing'
method: GsSingleRefPathResult
isDead
^isDead
%
category: 'Accessing'
method: GsSingleRefPathResult
searchOop
^searchOop
%

category: 'Printing'
method: GsSingleRefPathResult
logReferencePath

"Writes the complete reference path to stdout.  For an RPC gem, the path is 
written to the gem log file.  For linked topaz session, the path is written 
to the screen."

GsSingleRefPathFinder printMessageToLog: (self resultStringWithTabs: 1)
                      includeTime: false .
^ self
%
category: 'Printing'
method: GsSingleRefPathResult
printHeaderOn: ws

^self printHeaderOn: ws tabs: 0
%
category: 'Printing'
method: GsSingleRefPathResult
printHeaderOn: ws tabs: anInt

	anInt timesRepeat: [ws tab].
	ws
		nextPutAll: 'Reference path for search oop ';
		nextPutAll: searchOop asOop asString;
		space;
		nextPut: $(;
		nextPutAll: searchOop class name asString;
		nextPut: $);
		lf.
	^ws
%
category: 'Printing'
method: GsSingleRefPathResult
printReferencePathOn: ws

^ self printReferencePathOn: ws tabs: 0
%
category: 'Printing'
method: GsSingleRefPathResult
printReferencePathOn: ws tabs: anInt

	isDead
		ifTrue: 
			[anInt + 1 timesRepeat: [ws tab].
			ws
				nextPutAll: 'Object is dead.  No reference path found';
				lf]
		ifFalse: 
			[1 to: self size
				do: 
					[:n |
					| obj |
					obj := self at: n.
					anInt + 1 timesRepeat: [ws tab].
					ws
						nextPutAll: n asString;
						space: 3;
						nextPutAll: obj asOop asString;
						space;
						nextPut: $(;
						nextPutAll: obj class name asString;
						nextPut: $);
						lf]].
	^ws
%
category: 'Printing'
method: GsSingleRefPathResult
printResultsToGsFile: aGsFile

aGsFile nextPutAll: self resultString
%
category: 'Printing'
method: GsSingleRefPathResult
resultString

^self resultStringWithTabs: 0
%
category: 'Printing'
method: GsSingleRefPathResult
resultStringWithTabs: anInt

|ws|
ws := AppendStream on: String new.
self printHeaderOn: ws tabs: anInt ;
printReferencePathOn: ws tabs: anInt.
^ws contents
%

category: 'Updating'
method: GsSingleRefPathResult
isDead: newValue
isDead := newValue
%

category: 'Updating'
method: GsSingleRefPathResult
searchOop: newValue
searchOop := newValue
%

category: 'Validating'
method: GsSingleRefPathResult
validate

^ self validateWithLimitSet: nil
%
category: 'Validating'
method: GsSingleRefPathResult
validateWithLimitSet: aLimitSet
"Validates the receiver contains a complete and correct reference path for 
searchOop using the given limit set.  If aLimitSet is nil, then the default
limit set is used.

Raises an error if a problem is found, otherwise returns the receiver."

| myLimitSet |
isDead ifTrue: [^self].
myLimitSet := aLimitSet
  ifNil: [SystemRepository buildLimitSetForRefPathScan]
  ifNotNil: [aLimitSet].
self size == 0 ifTrue: [self halt: 'Receiver is empty'].
(myLimitSet includes: self first)
  ifFalse: [self halt: 'First object is not a limit object'].
self last == searchOop
  ifFalse: [self halt: 'Last element of reference path is not the search object'].
2 to: self size do:[:n | | child parent |
  parent := self at: n - 1.
  child := self at: n.
  (((GsBitmap with: parent) primReferencedObjects) includes: child)
    ifFalse: [self halt: 'parent object does not reference child object']].
^self
%
