! =========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id: ProfMonitorTree.gs 19139 2009-06-15 23:48:14Z lalmarod $
!
! Superclass Hierarchy:
!   ProfMonitorTree, ProfMonitor, Object.
!
!=========================================================================

expectvalue %String
run
| oldClass expectedInstVars newClass |
expectedInstVars := #(rawArray rootPME objRootPMEDict).
oldClass := Globals at: #ProfMonitorTree otherwise: nil.
(oldClass ~~ nil and: [oldClass instVarNames = expectedInstVars]) ifTrue: [
	^'Found expected class'.
].
newClass := ProfMonitor
  subclass: 'ProfMonitorTree'
  instVarNames: expectedInstVars
  classVars: #()
  classInstVars: #()
  poolDictionaries: { }
  inDictionary: Globals .
^(oldClass ifNil: ['Created new class:'] 
	ifNotNil: ['Replaced old class with:']) , '
' , newClass definition.
%

expectvalue %String
run
| curr n |
n := 0 .  "remove old methods for fix 44744"
(curr := ProfMonitorTree) classHistory size > 1 ifTrue:[
  curr classHistory do:[:cls |
    cls ~~ curr ifTrue:[
      cls removeAllMethods . cls class removeAllMethods .
      n := n + 1
    ].
  ].
].
^ 'found ', n asString , ' old versions'
%

removeallmethods ProfMonitorTree
removeallclassmethods ProfMonitorTree

category: 'For Documentation Installation only'
classmethod: ProfMonitorTree
installDocumentation

self comment: 'ProfMonitorTree is an enhancement of ProfMonitor, adding a tree based
display of the profiling results.' .

%

category: 'Reporting'
classmethod: ProfMonitorTree
defaultReports
  ^super defaultReports, {#tree . #objCreationTree}
%


! Instance methods

category: 'Private'
method: ProfMonitorTree
_methodTreeReport
  "Return a report of the method call stack in tree form"

  | rpt thresholdArg |

  rpt := String new.
 " If there's no samples, bail"
  rootPME == nil  ifTrue: [ ^rpt ].

  thresholdArg := self dynamicInstVarAt: #reportThreshold.

  rpt add: (self _reportHeader: 'STACK SAMPLING TREE RESULTS'); lf.

  self _treeReportDownTo: thresholdArg on: rootPME to: rpt.

^ rpt
%

category: 'Private'
method: ProfMonitorTree
_objectCreationTreeReport
  "report on object creation, in tree form."

  | rpt croots tally |

  tally := self dynamicInstVarAt: #reportThreshold.

  rpt := String new.
  rpt add: (self _reportHeader: 'OBJECT CREATION TREE REPORT'); lf.

  " If there's no samples, bail "
  (rootPME == nil  or: [objRootPMEDict keys size < 1]) ifTrue: [ 
     rpt add: ' - - No results to report - -' ; lf.
     ^ rpt.
     ].

  croots := Array new.
  objRootPMEDict keysAndValuesDo: [ :cls :root |
    croots add: (Array with: cls with: root)].
  croots := croots 
    asSortedCollection: [:a :b | ((a at: 2) tally) > ((b at: 2) tally)].
  croots do: [:r | | cls root |
    cls := r at: 1.
    root := r at: 2.
	rpt add: 'root tally is ', root tally asString; lf.
    root tally < tally ifFalse: [
  		rpt add:'----------------'; lf ; add: cls name ; lf ; lf.
     		self _treeReportDownTo: tally on: root to: rpt ]].
^ rpt
%

category: 'Private'
method: ProfMonitorTree
gatherPMEsForTree

  | res rawArraySiz rawArrayPointer sampleStart sampleEnd reposCls fileData |

  res := OrderedCollection new. 
  fileData := self _readSampleFile .  
  rawArray := fileData at: 1 .
  (rawArraySiz := rawArray size) < 1 ifTrue:[ ^ self ].
  rawArrayPointer := 1 .
  rootPME := self readFirstPME.
  objRootPMEDict := IdentityKeyValueDictionary new.
  reposCls := SystemRepository.
  [ rawArrayPointer < rawArraySiz ] whileTrue:[
      (rawArray at: rawArrayPointer) == reposCls ifFalse:[
         Error signal: 'data order problem'
      ].
      sampleStart := rawArrayPointer + 3.   "skip count and statsWord"
      sampleEnd := sampleStart + ((rawArray at: rawArrayPointer + 1) * 2) - 1.
      ((rawArray at: sampleStart) == true) 
      ifTrue: [
        "handle object trace report"
        sampleEnd := sampleEnd + 2.
        self tallyObjPME: (rawArray at: sampleStart + 1)
          position: sampleEnd 
          startIndex: sampleStart + 2
          endIndex: sampleEnd ]
      ifFalse: [
        "handle method trace report"
        self tallyPME: rootPME 
          position: sampleEnd 
          startIndex: sampleStart 
          endIndex: sampleEnd 
         ].
      rawArrayPointer := sampleEnd + 1.
  ].
%
category: 'Reporting'
method: ProfMonitorTree
gatherResults
  "See superclass comment"
  super gatherResults.
  self gatherPMEsForTree.
%
category: 'Private'
method: ProfMonitorTree
printablePercentFor: aFractionOrOtherNumber
  
  ^((aFractionOrOtherNumber asFloat * 100) 
    asStringUsingFormat: #(-6 1 false)) , '%'
%

category: 'Private'
method: ProfMonitorTree
printPME: aPME on: str total: theTotal
  
  str add: (self printablePercentFor: (aPME tally / theTotal)); space.
  str add: '('; add: aPME tally asString; add: ')'; space.

  str add: (aPME asStringWidth:20) ; lf
%

category: 'Private'
method: ProfMonitorTree
readFirstPME
  "trace consistes of SystemRepository, count, a stats word, and (count*2) slots with method and class.
  The highest point in context is at end, create a PME based on that."

| index count |
index := 1.
(rawArray at: index) == SystemRepository 
  ifFalse:[ Error signal: 'Cannot read first PME'].
count := (rawArray at: index + 1).
count _isSmallInteger ifFalse:[ Error signal: 'corrupt prof monitor file'].

index := index + 1. "Allow for statsword"

^ ProfMonitorEntry newForMethod: (rawArray at: (index + (count*2)))  
    receiverClass: (rawArray at: (index + (count*2) + 1)).
%

category: 'Private'
method: ProfMonitorTree
tallyPathPME: aPME position: index startIndex: startIndex endIndex: endIndex

  "Add a tally point for the PME's child, the one that matches the stack at 
  the given index.  Create child if does not exist. "

  | aMethod aClass matchingPME |
  aMethod := rawArray at: index - 1.
  aMethod class == GsNMethod ifFalse:[ aMethod := nil ].
  aClass := rawArray at: index .
  aPME ifNotNil:[
    aPME children do: [:aChildPME | 
      aChildPME cmethod == aMethod ifTrue:[ matchingPME := aChildPME ].
    ].
    aMethod ifNotNil:[ aClass ifNotNil:[
      matchingPME ifNil:[
        matchingPME := ProfMonitorEntry newForMethod: aMethod receiverClass: aClass. 
        aPME children add: matchingPME.
      ].
    ]].
  ].
  ^ matchingPME
%
category: 'Private'
method: ProfMonitorTree
tallyPME: aPME position: index startIndex: startIndex endIndex: endIndex
  " go backwards, top of stack is at end"

  | childPME idxMinus1 idxMinus2 mth |

  idxMinus1 := index - 1 .
  mth := rawArray at: idxMinus1 .
  mth ifNotNil:[ mth class == GsNMethod ifTrue:[ aPME ifNotNil:[
    mth == aPME cmethod ifTrue: [aPME tally: aPME tally + 1]
  ]]].

  idxMinus1 == startIndex ifTrue: [^self "end of path" ] .

  idxMinus2 := index - 2 .
  childPME := self tallyPathPME: aPME position: idxMinus2
            startIndex: startIndex endIndex: endIndex.

  self tallyPME: childPME 
      position: idxMinus2
      startIndex: startIndex endIndex: endIndex.
%

category: 'Private'
method: ProfMonitorTree
tallyObjPME: aClass position: index startIndex: startIndex endIndex: endIndex

  | aPME |
  (aPME := objRootPMEDict at: aClass otherwise: nil) ifNil: [
    aPME := self readFirstPME.
    objRootPMEDict at: aClass put: aPME ].
  ^ self tallyPME: aPME position: index startIndex: startIndex endIndex: endIndex
%

category: 'Private'
method: ProfMonitorTree
treePrintPME: aPME on: str tabs: tabs thisTab: myTab total: theTotal downTo: limit
  | children childrenTab |
  
  tabs do: [:tab | str add: tab].
  self printPME: aPME on: str total: theTotal.
  children := aPME children.
  children isEmpty ifFalse: [
      tabs addLast: myTab.
      children := children asSortedCollection: [:a :b | a tally > b tally].
      1 to: children size do: [:i | | child |
          child := children at: i.
          (child tally >= limit) ifTrue: [
            childrenTab := i < children size ifTrue: ['   |'] ifFalse: ['   '].
            self
              treePrintPME: child
              on: str
              tabs: tabs 
              thisTab: childrenTab
              total: theTotal
              downTo: limit
          ].
      ].
      tabs removeLast
    ]
%
category: 'Private'
method: ProfMonitorTree
_treeReportDownTo: tally on: root to: rpt

root children do: [:aPME |
  self 
    treePrintPME: aPME 
    on: rpt 
    tabs: OrderedCollection new  
    thisTab: '  ' 
    total: aPME tally
    downTo: tally
  ].
rpt lf; lf.
%

! fix 47148
category: 'Reporting'
method: ProfMonitorTree
reportDownTo: thresholdArg reports: arrayOfReports

"Formats and returns a string holding a report of the receiver's most recent
 profile run.  Stops reporting when a tally smaller than tally is encountered.

 thresholdArg should be a SmallInteger or a Float > 0 .
 A thresholdArg < 1.0 is interpreted as a percentage of the total
 samples, otherwise thresholdArg is intepreted as an absolute value.
 Report elements whose sample count is less than the threshold are
 omitted from the report."

  | saveFpe |
  results ifNil:[ ^'No profiling data are available.'].
  ^ [ | rpt |
    saveFpe := FloatingPointError enabledExceptions .
    FloatingPointError enableExceptions: nil .

    self prepareForReportWithThreshold: thresholdArg.

    rpt := String new .
    (self dynamicInstVarAt: #overrunStr) ifNotNil:[:str | rpt add: str ; lf].

    arrayOfReports do: [:sym |
      sym == #samples ifTrue: [
	 rpt addAll: (self _samplingReport); lf; lf ].
      sym == #stackSamples ifTrue: [
	 rpt addAll: (self _stackReport); lf; lf ].
      (sym == #senders and: [(self dynamicInstVarAt: #totalTallies) > 0])  ifTrue: [
	 rpt addAll: (self _sendersReport); lf; lf ].
      (sym == #objCreation and: [ (traceObjCreation bitAnd: 1) ~~ 0 and: 
	[(self dynamicInstVarAt: #totalTallies) > 0]]) ifTrue: [
	  rpt addAll: (self _objCreationReport); lf; lf.].
      sym == #tree ifTrue: [
	 rpt addAll: (self _methodTreeReport); lf; lf ].
      (sym == #objCreationTree and: [ (traceObjCreation bitAnd: 1) ~~ 0 and:
	[(self dynamicInstVarAt: #totalTallies) > 0]]) ifTrue: [
	 rpt addAll: (self _objectCreationTreeReport); lf; lf ].
     ].
     rpt
   ] ensure:[
     saveFpe size > 0 ifTrue:[ FloatingPointError enableExceptions: saveFpe].
   ]
%
 
