!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id: fractio2.gs 24812 2010-12-07 18:27:47Z lalmarod $
!
! Superclass Hierarchy:
!   Fraction, Number, Magnitude, Object.
!
!=========================================================================

category: 'For Documentation Installation only'
classmethod: AbstractFraction
installDocumentation

self comment:
'AbstractFraction is the superclass of Fraction and SmallFraction.'
%

category: 'For Documentation Installation only'
classmethod: Fraction
installDocumentation

self comment:
'A Fraction is a Number represented as the ratio of two Integers.

 Constraints:
	numerator: Integer
	denominator: Integer

 Do not send class Fraction the message new.  Fractions created in that way are
 meaningless and cannot be handled properly by GemStone''s associative access
 mechanism.  To create a new Fraction, use one of the specific instance creation 
 methods.'
%

category: 'For Documentation Installation only'
classmethod: SmallFraction
installDocumentation

self comment:
'SmallFractions are special objects that can represent Fractions
  with   -268435456 <= numerator   <= 268435455
  and             0 <  denominator <= 134217727 . 

 A SmallFraction contains the bits
    snnnnnnn|nnnnnnnn|nnnnnnnn|nnnnnddd|dddddddd|dddddddd|dddddddd|tttttttt
 where bits are shown with least-significant on the right.
 The bits  sn...n  are a 29 bit signed twos-complement numerator ,
 The bits  d...d   are a 27 bit unsigned denominator .
 The 8 tag bits have the constant value  16r2C . '
%

!-----------------------------------------------

category: 'Instance Creation'
classmethod: AbstractFraction
fromStream: aStream

"Returns a Fraction from the stream.  The stream must contain two Integers
 separated by a slash.  (There may be blanks around the slash.)  Generates an
 error if the stream contains anything else, or if an attempt is made to read
 beyond the end of the stream."

| ch n d |
self _checkReadStream: aStream forClass: CharacterCollection.
ch := aStream next.
[ ch == $  ] whileTrue:[ ch := aStream next ].
aStream skip: -1.
n := Integer fromStream: aStream.
ch := aStream next.
[ ch == $  ] whileTrue:[ ch := aStream next ].
aStream skip: -1.
(aStream next == $/ ) ifFalse:[ ^ self _errIncorrectFormat: aStream ].
ch := aStream next.
[ ch == $  ] whileTrue:[ ch := aStream next ].
aStream skip: -1 .
d := Integer fromStream: aStream.
^ self numerator: n denominator: d
%

category: 'Instance Creation'
classmethod: AbstractFraction
numerator: n denominator: d
"Returns an instance of Fraction with numerator numInt and denominator
 denomInt.  If that Fraction can be reduced, this method returns the
 corresponding Integer instead.  The result is made invariant.

 If either argument (numerator or denominator) is not an Integer, that
 argument is truncated to the corresponding Integer."

 (self _newSmallFraction: n denom: d reduce: true) ifNotNil:[:res | 
    ^ res 
 ].
 ^ self _reduce: n denom: d
%

! ------------------ private instance creation code
category: 'Private'
classmethod: AbstractFraction
_reduce: numer denom: denom
  | n d gcd |
  n := numer truncated . 
  n == 0 ifTrue:[ ^ 0 ].
  d := denom truncated .
  (d == 0) ifTrue: [ ^ n _errorDivideByZero ].
  d < 0 ifTrue:[
    d := 0 - d .
    n := 0 - n .
  ].
  gcd := n gcd: d .
  n := n // gcd.
  d := d // gcd.
  (d == 1) ifTrue:[ ^ n ].
  (self _newSmallFraction: n denom: d reduce: false) ifNotNil:[:res |
     ^ res
  ]. 
  ^ Fraction _basicNew _numerator: n denom: d
%

category: 'Private'
classmethod: AbstractFraction
_noreduce_numerator: n denominator: d
  (self _newSmallFraction: n denom: d reduce: false) ifNotNil:[:res |
     ^ res
  ].
  ^ Fraction _basicNew _numerator: n denom: d
%

category: 'Private'
classmethod: AbstractFraction
_newFraction: n denominator: d

  "Returns an instance of Fraction , never reduced to a SmallFraction."

  ^ Fraction _basicNew _numerator: n denom: d
%

category: 'Private'
classmethod: AbstractFraction
_newSmallFraction: n denom: d reduce: reduceBoolean

"Returns nil or a reduced instance of SmallFraction ,
 or a Fraction containing reduced SmallInteger values for n and d.
 A SmallFraction is returned if, after moving overall sign to n
 and reducing n and d, n and d satisfy
                0 <  d <= 134217726  
   and -536870912 <= n <= 536870911 

 Signals an Error if d == 0 ."

<primitive: 996>
reduceBoolean _validateClass: Boolean .
^ self _primitiveFailed: #_newSmallFraction:denom: args: { n . d }
%

category: 'Private'
method: Fraction
_numerator: n denom: d
  numerator := n .
  denominator := d .
  ^ self immediateInvariant
%

!----------------- 

category: 'Storing and Loading'
classmethod: Fraction
loadFrom: passiveObj

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| inst num den |

passiveObj readNamedIV.
num := passiveObj ivValue.
passiveObj readNamedIV.
den := passiveObj ivValue.
  
passiveObj skipNamedInstVars.

inst := self _newFraction: num denominator: den.
passiveObj hasRead: inst.  "arg to hasRead must not be a SmallFraction."
^inst.
%

category: 'Storing and Loading'
method: Fraction
writeTo: passiveObj

"Converts the receiver to its passive form and writes that information on
 passiveObj."

"Reimplemented from Number since the receiver has a non-literal representation."

^super basicWriteTo: passiveObj
%

category: 'Testing'
method: SmallFraction
isSpecial
  ^ true
%
category: 'Storing and Loading'
method: SmallFraction
containsIdentity
  ^ true
%

category: 'Storing and Loading'
method: SmallFraction
writeTo: passiveObj
  "reimplement basicWriteTo: to synthesize output of numerator and denominator"
  passiveObj writeClass: self class ;
     writeObject: self numerator named: #numerator ;
     writeObject: self denominator named: #denominator ;
     endNamedInstVars ;
     cr .
%

category: 'Storing and Loading'
classmethod: SmallFraction
loadFrom: passiveObj
  | n d |
  passiveObj readNamedIV.
  n := passiveObj ivValue.
  passiveObj readNamedIV.
  d := passiveObj ivValue.
  passiveObj skipNamedInstVars.

  ^ self numerator: n denominator: d
%

!-----------------------------------------------
! instvar access

category: 'Accessing'
method: SmallFraction
denominator
<primitive: 1000>

^ self _primitiveFailed: #denominator
%
method: SmallFraction
numerator
<primitive: 999>

^ self _primitiveFailed: #numerator
%

category: 'Accessing'
method: Fraction
denominator

"Returns the denominator of the receiver."

^ denominator
%

category: 'Accessing'
method: Fraction
numerator

"Returns the numerator of the receiver."

^ numerator
%

category: 'Accessing'
method: Fraction
instVarAt: anIndex put: aValue

"Disallowed.  You may not change the value of a Fraction."

self shouldNotImplement: #instVarAt:put:
%

! fix 46790
category: 'Copying'
method: Fraction
postCopy
  ^ self immediateInvariant
%

!----------------- instance methods

category: 'Formatting'
method: AbstractFraction
asString

"Returns a String of the form 'numerator/denominator'."

| result |
result := self numerator asString .
result add: $/ ; addAll: self denominator asString .
^ result
%

category: 'Converting'
method: AbstractFraction
asScaledDecimal: scale

"Returns a ScaledDecimal representation of the receiver."

^ ScaledDecimal numerator: self numerator denominator: self denominator scale: scale.
%

! asFloat rewritten for fix 36173
category: 'Converting'
method: AbstractFraction
asFloat

"Returns an instance of SmallDouble or Float that has the value of the receiver."
| num denom nParts dParts mantiss exp |

(num := self numerator) ifNil:[ ^ PlusSignalingNaN ] .
(denom := self denominator) ifNil:[ ^ PlusSignalingNaN ] .

"avoid loss of precision by handling mantissa and exponents separately"
nParts := num _floatParts .
dParts := denom _floatParts .
mantiss := (nParts at:1) / (dParts at:1) .
exp := (nParts at:2) - (dParts at:2) .
^ mantiss * (2.0 raisedToInteger: exp ) 
%


category: 'Converting'
method: AbstractFraction
asDecimalFloat

"Returns an instance of DecimalFloat that has the value of the receiver."
| num denom |
(num := self numerator) ifNil:[ ^ PlusSignalingNaN ] .
(denom := self denominator) ifNil:[ ^ PlusSignalingNaN ] .
^ num asDecimalFloat / denom asDecimalFloat
%

category: 'Converting'
method: AbstractFraction
asFraction

"Returns the receiver."

^self
% 

category: 'Converting'
method: AbstractFraction
asFixedPoint: scale

^ FixedPoint numerator: self numerator denominator: self denominator scale: scale
%

category: 'Converting'
method: AbstractFraction
_coerce: aNumber

"Reimplemented from Number."
| f |
f := aNumber asFraction .
f _isInteger ifTrue:[ 
  ^ AbstractFraction _noreduce_numerator: f denominator: 1 
].
^ f
%

category: 'Converting'
method: AbstractFraction
_generality

"Reimplemented from Number."

^70
%

category: 'Accessing'
method: AbstractFraction
_getKind

| num denom |
(num := self numerator) ifNil:[ ^ 5 "nan"].
(denom := self denominator) ifNil:[ ^ 5 ].
denom == 0 ifTrue:[ ^ 3 "infinity" ].
num == 0 ifTrue:[ ^ 4 "zero"].
^ 1 "normal"
%

method: AbstractFraction
sign
"Returns 1 if the receiver is greater than zero, -1 if the receiver is less
 than zero, and zero if the receiver is zero."

  ^ self numerator sign
%

category: 'Comparing'
method: AbstractFraction
<= aNumber

"Returns true if the receiver is less than or equal to aNumber; returns false
 otherwise."

aNumber _isNumber ifTrue:[ 
  (aNumber isKindOf: AbstractFraction) ifTrue: [ | otherNum num |
    (otherNum := aNumber numerator) ifNil:[ ^ false ].
    (num := self numerator) ifNil:[ ^ false ].
    otherNum == 0 ifTrue: [ 
      ^ num <= 0
    ].
    ^ (num * aNumber denominator) <=  (self denominator * otherNum)
  ].
].
^self _retry: #<= coercing: aNumber
%

category: 'Comparing'
method: AbstractFraction
< aNumber

"Returns true if the receiver is less than aNumber; returns false otherwise."

aNumber _isNumber ifTrue:[ 
  (aNumber isKindOf: AbstractFraction) ifTrue: [ | otherNum num |
    (otherNum := aNumber numerator) ifNil:[ ^ false ].
    (num := self numerator) ifNil:[ ^ false ].
    otherNum == 0 ifTrue: [  
      ^ num < 0
    ].
    ^ (num * aNumber denominator) < (self denominator * otherNum)
  ].
].
^self _retry: #< coercing: aNumber
%

category: 'Comparing'
method: AbstractFraction
= aNumber

"Returns true if the receiver is equal to aNumber; returns false otherwise."
aNumber _isNumber ifFalse:[ ^ false ].
(aNumber isKindOf: AbstractFraction ) ifTrue: [ | otherNum num |
  (otherNum := aNumber numerator) ifNil:[ ^ false ].
  (num := self numerator) ifNil:[ ^ false ].
  otherNum == 0 ifTrue: [
    ^ num == 0
  ].
  ^ otherNum = num and:[ aNumber denominator = self denominator ]
].
^ self _retry: #= coercing: aNumber
%

category: 'Comparing'
method: AbstractFraction
> aNumber

aNumber _isNumber ifTrue:[ 
  (aNumber isKindOf: AbstractFraction) ifTrue: [ | otherNum num |
    (otherNum := aNumber numerator) ifNil:[ ^ false ].
    (num := self numerator) ifNil:[ ^ false ].
    otherNum == 0 ifTrue: [  
      ^ num > 0
    ].
    ^ (num * aNumber denominator) > (self denominator * otherNum)
  ].
].
^self _retry: #> coercing: aNumber
%

category: 'Comparing'
method: AbstractFraction
>= aNumber

aNumber _isNumber ifTrue:[ 
  (aNumber isKindOf: AbstractFraction) ifTrue: [ | otherNum num |
    (otherNum := aNumber numerator) ifNil:[ ^ false ].
    (num := self numerator) ifNil:[ ^ false ].
    otherNum == 0 ifTrue: [  
      ^ num >= 0
    ].
    ^ (num * aNumber denominator) >= (self denominator * otherNum)
  ].
].
^self _retry: #>= coercing: aNumber
%

! hash inherited from Number

category: 'Truncation and Rounding'
method: AbstractFraction
truncated

"Returns the integer that is closest to the receiver, on the same side
 of the receiver as zero is located."

^ self numerator quo: self denominator
%

method: AbstractFraction
roundTo: aNumber

"Returns the multiple of aNumber that is nearest in value to the receiver."
| r |
aNumber = 0 ifTrue:[ ^ 0 ] .
r := (self / aNumber) rounded * aNumber .
aNumber _isFloat ifTrue:[ ^ r asFloat ].
^ r asFraction 
%

method: AbstractFraction
roundedHalfToEven

"Returns the integer nearest in value to the receiver. If the receiver is 
exactly halfway between two integers, return the even one."

| remainder |
remainder := (self numerator abs \\ self denominator).
^ remainder * 2 = self denominator
	ifFalse: [self rounded]
	ifTrue: [| truncated | truncated := self truncated.
			truncated even 
				ifTrue: [truncated]
				ifFalse: [truncated + self sign]]
%

method: AbstractFraction
truncateTo: aNumber

"Returns the multiple of aNumber that is closest to the receiver, on
 the same side of the receiver as zero is located.  In particular,
 returns the receiver if the receiver is a multiple of aNumber."

| t |
aNumber = 0 ifTrue:[ ^ 0 ].
t := (self quo: aNumber) * aNumber .
aNumber _isFloat ifTrue:[ ^ t asFloat ].
^ t asFraction
%


category: 'Arithmetic'
method: AbstractFraction
* aNumber

"Returns the result of multiplying the receiver by aNumber."

(aNumber isKindOf: AbstractFraction) ifTrue: [
  ^ AbstractFraction numerator: (self numerator * aNumber numerator)
           denominator: (self denominator * aNumber denominator) 
] ifFalse: [
  (aNumber _isInteger and:[ aNumber = self denominator ]) ifTrue:[
     ^ self numerator
  ].
  ^ self _retry: #* coercing: aNumber
]
%


category: 'Arithmetic'
method: AbstractFraction
+ aFraction

"Returns the sum of the receiver and aFraction."

(aFraction isKindOf: AbstractFraction) ifTrue:[ 
  | myDenom otherDenom commonDenominator newNumerator 
    gcd myDemon_div_gcd |
  (myDenom := self denominator) = (otherDenom := aFraction denominator) ifTrue: [
    | myNumer otherNumer |
    (myNumer := self numerator) = (otherNumer := aFraction numerator) ifTrue:[
      myDenom _bottomBit == 0 ifTrue:[
        ^ AbstractFraction numerator: myNumer denominator: (myDenom bitShift: -1)
      ]
    ].
    ^ AbstractFraction numerator: (myNumer + otherNumer ) denominator: myDenom 
  ].
  gcd := myDenom gcd: otherDenom .
  myDemon_div_gcd := myDenom // gcd .
  commonDenominator := myDemon_div_gcd * otherDenom .
  newNumerator := (self numerator * (otherDenom // gcd ))
                 + (aFraction numerator *  myDemon_div_gcd) .
  ^ AbstractFraction numerator: newNumerator denominator: commonDenominator
] ifFalse: [
  ^self _retry: #+ coercing: aFraction
]
%

category: 'Arithmetic'
method: AbstractFraction
- aFraction

"Returns the difference between the receiver and aFraction."
(aFraction isKindOf: AbstractFraction) ifTrue: [
  ^ self + (Fraction _noreduce_numerator:  0 - aFraction numerator 
		 denominator: aFraction denominator) 
] ifFalse: [
  ^ self _retry: #- coercing: aFraction
]
%

category: 'Arithmetic'
method: AbstractFraction
/ aNumber

"Returns the result of dividing the receiver by aFraction."

aNumber _isInteger ifTrue:[
  aNumber = 0 ifTrue: [ ^ self _errorDivideByZero ].
  ^ AbstractFraction numerator: self numerator denominator: (self denominator * aNumber)
].
(aNumber isKindOf: AbstractFraction) ifTrue:[
  ^ self * aNumber _reciprocal
].
^ self _retry: #/ coercing: aNumber
%

category: 'Arithmetic'
method: AbstractFraction
negated

"Returns a Number that is the negation of the receiver."

^ AbstractFraction _noreduce_numerator:  0 - self numerator 
			denominator: self denominator
%

category: 'Private'
method: AbstractFraction
_reciprocal

"Returns 1 divided by the receiver.  Generates an error if the receiver is 0.
 Result is not reduced."
| numer denom |
((numer := self numerator) == 0 or:[ numer == nil]) ifTrue: [ ^ self _errorDivideByZero].
denom := self denominator .
numer < 0 ifTrue:[ denom := 0 - denom . numer := 0 - numer  ].

^  AbstractFraction _noreduce_numerator: denom denominator: numer 
%

category: 'Arithmetic'
method: AbstractFraction
reciprocal

"Returns 1 divided by the receiver.  Generates an error if the receiver is 0.
 Result is reduced."
| numer denom |
((numer := self numerator) == 0 or:[ numer == nil]) ifTrue: [ ^ self _errorDivideByZero].
denom := self denominator .
(numer == 1) ifTrue:[ ^ denom ].
numer < 0 ifTrue:[ 
  (numer == -1) ifTrue:[ ^ 0 - denom ].
  denom := 0 - denom . numer := 0 - numer .
].
^  AbstractFraction numerator: denom denominator: numer 
%


category: 'Testing'
method: AbstractFraction
even

"Returns true if the receiver is an even integer, false otherwise."

 self denominator = 1 ifFalse: [ ^ false ].
 ^ self numerator even

%

category: 'Testing'
method: AbstractFraction
odd

"Returns true if the receiver is an odd integer, false otherwise."

 self denominator = 1 ifFalse: [ ^ false ].
 ^ self numerator odd
%

category: 'Indexing Support'
method: AbstractFraction
_isNaN

"Returns whether the receiver is quiet NaN or signaling NaN.
 This method is only to be used by the indexing subsystem."

^ (self numerator == nil) or: [self denominator == nil]
%
category: 'Testing'
method: AbstractFraction
isZero

"Returns true if the receiver is zero."

^ self numerator = 0 .
%

