!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   Object
!     Random
!       HostRandom
!       SeededRandom
!         Lag1MwcRandom
!         Lag25000CmwcRandom
!
!=========================================================================

expectvalue %String
run
(Object _newKernelSubclass: 'Random'  
  instVarNames: #() 
  inDictionary: Globals)
.
%
expectvalue %String
run
(Random _newKernelSubclass: 'HostRandom'  
  instVarNames: #()  
  classVars: #()  
  classInstVars: #( singleton)  
  poolDictionaries: #()  
  inDictionary: Globals  
  options: #() 
  reservedOop: nil)
.
%
expectvalue %String
run
(Random _newKernelSubclass: 'SeededRandom'  
  instVarNames: #( multiplier lag carry seeds index) 
  inDictionary: Globals)
.
%
expectvalue %String
run
(SeededRandom _newKernelSubclass: 'Lag1MwcRandom'  
  instVarNames: #()  
  inDictionary: Globals)
.
%
expectvalue %String
run
(SeededRandom _newKernelSubclass: 'Lag25000CmwcRandom'  
  instVarNames: #()
  inDictionary: Globals)
.
%
run
Random immediateInvariant.
Random isInvariant.
%
run
HostRandom immediateInvariant.
HostRandom isInvariant.
%
run
SeededRandom immediateInvariant.
SeededRandom isInvariant.
%
run
Lag1MwcRandom immediateInvariant.
Lag1MwcRandom isInvariant.
%
run
Lag25000CmwcRandom immediateInvariant.
Lag25000CmwcRandom isInvariant.
%

! Remove existing behavior from Random
removeallmethods Random
removeallclassmethods Random

! ------------------- Class methods for Random
category: 'Documentation'
set compile_env: 0
classmethod: Random
comment
^'Random is an abstract superclass of generators of random numbers, that is, 
a sequence of numbers that passes statistical tests of randomness. 

See the comment of each subclass for its properties and recommendations for its use. 
For short non-repeatable sequences with high-quality randomness, use HostRandom.
For repeatable (seeded) sequences where randomness requirements are not stringent, 
use Lag1MwcRandom. For long sequences of random numbers (repeatable or not), 
or when you need repeatable random n-tuples, use Lag25000CmwcRandom. 

For convenience, Random class accepts these messages to create instances of subclases:

    new
        Answers the singleton HostRandom. 

    seed: aSmallInteger
       Answers a Lag1MwcRandom with an initial seed generated from the 
       given SmallInteger


Once you have an instance of any subclass of Random, you can generate random numbers 
with these messages:

    float
        Answers a random Float in the range [0,1)

    floats: n
        Answers a collection of n random floats in the range [0,1)

    integer
        Answers a random non-negative 32-bit integer

    integers: n
        Answers a collection of n random non-negative 32-bit integers

    integerBetween: l and: h
        Answers a random integer in the given range

    integers: n between: l and: h
        Answers a collection of n random integers in the given range

    smallInteger
        Answer a random integer in the full SmallInteger range, [-2**60..2**60)

In most cases, you will want to create an instance of a Random subclass and use that 
instance to generate many random numbers. Creating multiple instances of Random 
subclasses and using each one only a few times is much more expensive. Care is needed 
when storing persistent instances of Random subclasses; see the documentation of each 
subclass for details.'
%
category: 'instance creation'
classmethod: Random
new
	"Answer the host random singleton"

	^HostRandom new
%
category: 'instance creation'
classmethod: Random
seed: aSmallInteger
	"Answer a seeded random number generator seeded from the given SmallInteger"

	^Lag1MwcRandom seed: aSmallInteger
%
! ------------------- Instance methods for Random
category: 'compatibility'
set compile_env: 0
method: Random
next
	"A synonym for #float, provided for compatibility with other Smalltalk implementations."

	^ self float.
%
category: 'public'
set compile_env: 0
method: Random
float
	"Answer a random Float in the range [0,1); or equivalently [0, 1[
	There are 2**53 possible results, equally spaced in the interval and 
	generated with equal probability.

	To compute a float in this range, we need 53 random bits. Since bits come in chunks of 32,
	we grab 64 bits from the raw generator and discard the excess. More than 53 bits would result
	in floating point rounding which could improperly push the result to 1.0."

	^ (self integer bitAnd: 16rfffff800) * 2.3283064365386963E-10 + self integer * 2.3283064365386963E-10

	"2.3283064365386963E-10, expressed as a binary float, is exactly 2.0 raisedToInteger: -32, so 
	multiplying by it is equivalent to a 32-bit right shift."
%
category: 'public'
set compile_env: 0
method: Random
floats: n
	"Answer a collection of n random floats. N must be a positive integer."

	| result |
	result := Array new: n.
	1 to: n do: [:i | result at: i put: self float].
	^result
%
category: 'public'
set compile_env: 0
method: Random
integer
	"Answer a random nonnegative 32-bit integer."

	^self subclassResponsibility: #integer
%
category: 'public'
set compile_env: 0
method: Random
integerBetween: l and: h
	"Answer a random integer in the given interval. Both l and h must be numbers, 
	h must be greater than or equal to l, and there must be at least one integer 
  in the range from l to h.
	Examples:
		| random x y z |
		random := Random new.
		x := random integerBetween: 1 and: 20.
		y := random integerBetween: -5 and: 5.
		z := random integerBetween: 2.4 and: 257/3
  "
	|  low high modulus bitValues randomBits maxBits |
	low := l ceiling.
	high := h floor.
	modulus := high - low + 1.
	modulus > 0 ifFalse: [ArgumentError signal: 'range includes no integers']. 
	bitValues := 16r100000000. "How many possible values of randomBits"
	modulus > bitValues ifTrue: [ArgumentError signal: 'Intervals > 2**32 not yet supported'].
	maxBits := bitValues - (bitValues \\ modulus).
	[randomBits := self integer. "[0, 16rFFFFFFFF]"
	randomBits > maxBits] whileTrue. 
	^randomBits \\ modulus + low.
%
category: 'public'
set compile_env: 0
method: Random
integers: n
	"Answer a collection of n random nonnegative 32-bit integers. N must be a positive integer."

	| result |
	result := Array new: n.
	1 to: n do: [:i | result at: i put: self integer].
	^result
%
category: 'public'
set compile_env: 0
method: Random
integers: n between: l and: h
	"Answer a collection of n random integers between l and h.  N must be a positive integer."

	| result |
	result := Array new: n.
	1 to: n do: [:i | result at: i put: (self integerBetween: l and: h)].
	^result
%
category: 'public'
set compile_env: 0
method: Random
smallInteger
	"Answer a random SmallInteger, [-2**60..2**60).
	This requires 61 bits including the sign bit.  We get 64 bits from the underlying generator, 
  use 60 bits to compute a full-range positive SmallInteger, then conditionally negate 
  it based on one of the extra bits."

	| highOrder lowOrder result |
	highOrder := self integer.
	lowOrder := self integer.
	result := ((highOrder bitAnd: 16rFFFFFFF) bitShift: 32) + lowOrder.
	^(highOrder bitAnd: 16r80000000) = 0 
		ifTrue: [result] 
		ifFalse: [result negated - 1]
%
category: 'public'
set compile_env: 0
method: Random
smallIntegers: n
	"Answer a collection of n random SmallIntegers. N must be a positive integer."

	| result |
	result := Array new: n.
	1 to: n do: [:i | result at: i put: self smallInteger].
	^result
%

! Remove existing behavior from HostRandom
removeallmethods HostRandom
removeallclassmethods HostRandom

! ------------------- Class methods for HostRandom
category: 'Documentation'
classmethod: HostRandom
comment
^'HostRandom allows access to the host operating system''s /dev/urandom random number 
generator.

See also the class comment of Random.

HostRandom is fairly fast to initialize, but after initialization is considerably slower 
than the other subclasses of Random. However, /dev/urandom 
on some platforms may be intended to be a cryptographically secure random number 
generator, which none of the other subclasses are. It also has the advantage of 
not needing an initial seed, and so is good for generating random seeds for the 
faster Random subclasses.

HostRandom uses a shared singleton instance, which is accessed by sending #new to 
the class HostRandom. Sending #new has the side effect of opening the underlying file 
/dev/urandom. This file normally remains open for the life of the session, but if 
you wish to close it you can send #close to the instance, and later send #open to 
reopen it. If you store a persistent reference to the singleton instance the underlying
file will not be open in a new session and you must send #open to the instance before 
asking for a random number. Since the singleton HostRandom instance has no state, 
it may be shared by multiple sessions without conflict.

Since HostRandom is a service from the operating system, it cannot be seeded, and 
should not be used when a repeatable random sequence of numbers is needed.'
%
category: 'initialize-release'
set compile_env: 0
classmethod: HostRandom
initialize
	singleton := self basicNew
%
category: 'instance creation'
set compile_env: 0
classmethod: HostRandom
new
	"Answer the singleton instance, opening /dev/urandom if not already open."

	singleton open.
	^singleton
%
category: 'instance creation'
set compile_env: 0
classmethod: HostRandom
seed: anInteger
	"Answer a seeded random number generator seeded from the given SmallInteger"

	ImproperOperation 
		signal: 'HostRandom uses the random number generator of the host random number generator, and cannot be seeded.'
%
! ------------------- Instance methods for HostRandom
category: 'private'
set compile_env: 0
method: HostRandom
file
	"Answer my file (/dev/urandom), if I have one, otherwise nil"

	^ SessionTemps current at: #GsHostRandomFile otherwise: nil 
%
category: 'public'
set compile_env: 0
method: HostRandom
close
	"Close /dev/urandom if open."

	| temps name file success |
	temps := SessionTemps current.
	name := #GsHostRandomFile.
	file := self file.
	file == nil 
		ifFalse: [success := file close.
				success == nil 
					ifTrue: [IOError signal: GsFile serverErrorString] 
					ifFalse: [temps removeKey: name]]

%
category: 'public'
set compile_env: 0
method: HostRandom
integer
	"Answer a random nonnegative 32-bit integer."

	| int file |
	file := self file.
	(file ~~ nil and: [file isOpen]) 
		ifFalse: [ImproperOperation signal: 'This HostRandom is closed'].
	int := 0.
	4 timesRepeat: [int := (int bitShift: 8) + file nextByte].
	^int
					
%
category: 'public'
set compile_env: 0
method: HostRandom
isOpen
	"Answer whether /dev/urandom is open"

	^ self file ifNil:[ false ] ifNotNil:[:f | f isOpen]
%
category: 'public'
set compile_env: 0
method: HostRandom
open
	"Open /dev/urandom if not already open."

	| temps name file |
	temps := SessionTemps current.
	name := #GsHostRandomFile.
	file := temps at: name otherwise: nil .
        file ifNil: [
           file := temps at: name put: (GsFile openOnServer: '/dev/urandom' mode: 'r').
	   file ifNil:[ IOError signal: GsFile serverErrorString ]
        ]
%

! Remove existing behavior from SeededRandom
removeallmethods SeededRandom
removeallclassmethods SeededRandom

! ------------------- Class methods for SeededRandom
category: 'Documentation'
set compile_env: 0
classmethod: SeededRandom
comment
^'SeededRandom is an abstract superclass of classes of generators of sequences of 
random numbers, with the property that a sequence of random numbers may be generated 
repeatedly by giving the same initial seed to the generator.

See also the class comments of Random and of my subclasses.

Instance creation methods to be sent to subclasses:

    new
        Answers a new generator which will by default be seeded from the host OS /dev/urandom

    seed: aSmallInteger
        Answers a new generator seeded with the given seed, which can be any SmallInteger. 
        Another generator of the same class created with the same seed will produce the 
        same sequence of random numbers.

Instance methods:

    seed: aSmallInteger
        Seeds the receiver from the given seed, which can be any SmallInteger.
        The subsequent random number sequence generated will be the same as if this 
        generator had been created with this seed.

    fullState
    fullState:
        The internal state of a generator is more than can be represented by a single SmallInteger. 
        These messages allow you to retrieve the full state of a generator at any time, and 
        to restore that state later.
        The random number sequence generated after the restoration of the state will be the same 
        as that generated after the retrieval of the state. You might, for instance, allow a 
        generator to get its initial state from /dev/urandom, then save this state so the 
        random sequence can be repeated later.'
%
category: 'instance creation'
set compile_env: 0
classmethod: SeededRandom
new
	^ self basicNew initialize
%
category: 'instance creation'
set compile_env: 0
classmethod: SeededRandom
seed: anInteger

	^ self new seed: anInteger
%
! ------------------- Instance methods for SeededRandom
category: 'public'
set compile_env: 0
method: SeededRandom
fullState

	"Answers the current state of the generator. The state represents in what sequence random 
  numbers will be generated, and where in that sequence the generator is. This generator, or 
  another generator of the same class, may be later set to this state by giving this state 
  as an argument to the fullState: method, after which it will generate random numbers in 
  the same sequence as the receiver will after receiving #fullState."

	"If I have no state yet, initialize myself from HostRandom"
	index == nil ifTrue: [self setSeedFromHost].
	^{index.
	carry.
	seeds copy}
%
category: 'public'
set compile_env: 0
method: SeededRandom
fullState: fullState

	"Resets the state of the generator to some point in its sequence; the point at which fullState was 
	obtained by sending #fullState to the generator or another generator of the same class. 
	After receiving this message, the sequence of random numbers generated will match the 
	ones generated after #fullState was received.

	The given fullState must be an Array of size 3. The first element is an integer, the index,
	which must be in the range [1..lag].
	The second element is the carry, which must be an integer in the range [0..multiplier-1].
	The third element is the seedArray. The size of seedArray mut be lag. Each seed in the 
	array must be a 32-bit nonnegative integer.
	Depending on the subclass, it may be an error if *all* of the seeds and the carry to be zero, 
	or for all of them to be their maximum legal value; MWC requires that at least one must be of 
	another value. CMWC does not have this requirement."

	self validateFullState: fullState.
	index := fullState at: 1.
	carry := fullState at: 2.
	seeds := (fullState at: 3) copy
%

! Remove existing behavior from Lag1MwcRandom
removeallmethods Lag1MwcRandom
removeallclassmethods Lag1MwcRandom

! ------------------- Class methods for Lag1MwcRandom
category: 'Documentation'
set compile_env: 0
classmethod: Lag1MwcRandom
comment 
^'Lag1MwcRandom is a seedable random number generator with a period of about 10**18.

See also the class comments of Random and SeededRandom.

Lag1MwcRandom is a lag-1 generator using the multiply-with-carry algorithm to generate random 
numbers. Compared to Lag25000CmwcRandom, its period is much shorter, it is of limited 
use in generating random n-tuples, and it has been shown that the random bits of the MWC algorithm 
are not quite perfectly fair, so for some uses Lag25000CmwcRandom is superior to this class. 
The primary advantage of Lag1MwcRandom is that it can be seeded by a single 61-bit SmallInteger, 
and initialized quickly, whereas Lag25000CmwcRandom requires a seed of more than 800000 bits, 
and therefore takes considerably longer to initialize, though it''s faster once initialized.

Lag25000CmwcRandom>>seed: uses a Lag1MwcRandom to generate all the seed bits from the 
given SmallInteger seed. '
%
! ------------------- Instance methods for Lag1MwcRandom
category: 'initialization'
set compile_env: 0
method: Lag1MwcRandom
initialize

"The exact choice of these constants is crucial to obtaining the desired result. 
Refer to literature on the multiply-with-carry algorithm for the relationship between the constants. 
Note that the implementation of this class also implies the constant b := 2**32."

	multiplier := 698769069.
	lag := 1.
%
category: 'private'
set compile_env: 0
method: Lag1MwcRandom
setSeedFromHost
	
	| random |
	random := HostRandom new.
	seeds := {random integer}.
	carry := random integer \\ multiplier.
	index := 1
%
category: 'private'
set compile_env: 0
method: Lag1MwcRandom
validateFullState: fullState

	| newIndex newCarry seedArray haveNonZero haveNonMaximal |
	fullState size = 3 ifFalse: [ArgumentError signal: 'expected an Array of size 3'].
	newIndex := fullState at: 1.
	newCarry := fullState at: 2.
	seedArray := fullState at: 3.
	haveNonZero := newCarry ~= 0.
	haveNonMaximal := newCarry ~= (multiplier - 1).
	newIndex _isSmallInteger ifFalse: [ArgumentError signal: 'Index must be a SmallInteger'].
	(newIndex between: 1 and: lag) ifFalse:  [OutOfRange new 
												name: 'index' min: 1 max: lag actual: newIndex;
															signal: 'out of range'].
	newCarry _isSmallInteger ifFalse: [ArgumentError signal: 'Carry must be a SmallInteger'].
	(newCarry between: 0 and: multiplier - 1) ifFalse: [OutOfRange new 
															name: 'carry' min: 0 max: multiplier - 1 actual: newCarry;
															signal: 'out of range'].
	seedArray size = lag ifFalse: [ArgumentError signal: 'Expected seed array of size ' , lag printString].
	seedArray do: [:seed | 
					seed _isSmallInteger ifFalse: [ArgumentError signal: 'Each seed must be a SmallInteger'].
					seed = 0 ifFalse: [haveNonZero := true].
					seed = 16rFFFFFFFF ifFalse: [haveNonMaximal := true].
					(seed between: 0 and: 16rFFFFFFFF) ifFalse: [OutOfRange new
																		name: 'seed' min: 0 max: 16rFFFFFFFF actual: seed;
																		signal: 'out of range']].
	haveNonZero ifFalse: [ArgumentError signal: 'Carry and seeds should not all be zero.'].
	haveNonMaximal ifFalse: [ArgumentError signal: 'Carry and seeds should not all be their maximal legal value.']
%
category: 'public'
set compile_env: 0
method: Lag1MwcRandom
integer
	"Answer a random nonnegative 32-bit integer."

	| newCarryAndSeed newSeed |
	index == nil ifTrue: [self setSeedFromHost].
	newCarryAndSeed := (seeds at: index) * multiplier + carry.
	newSeed := newCarryAndSeed bitAnd: 16rFFFFFFFF.
	carry := newCarryAndSeed bitShift: -32.
	index == lag ifTrue: [index := 0].
	index := index + 1.
	seeds at: index put: newSeed.
	^newSeed
	
%
category: 'public'
set compile_env: 0
method: Lag1MwcRandom
seed: newSeed

	"Sets the seed from the given SmallInteger."

	| upperBits lowerBits |
	newSeed _isSmallInteger ifFalse: [ArgumentError signal: 'The seed must be a SmallInteger'].
	
	"A SmallInteger has 61 significant bits, including the sign bit. We will use the lower 32 bits 
  for the initial seed and the upper 29, including the sign, to generate the initial carry. 
  The legal range for carry is [0, multiplier), which in this class is more than 2**29 but 
  less than 2**30. Carry and seed cannot both be zero, so we ensure that the initial carry 
  is non-zero by inverting it within its range."

	upperBits := (newSeed bitAnd: 16r1FFFFFFFFFFFFFFF) bitShift: -32.
	lowerBits := newSeed bitAnd: 16rFFFFFFFF.
	seeds := {lowerBits} .
	carry := multiplier - 1 - upperBits.
	index := 1
%

! Remove existing behavior from Lag25000CmwcRandom
removeallmethods Lag25000CmwcRandom
removeallclassmethods Lag25000CmwcRandom

! ------------------- Class methods for Lag25000CmwcRandom
category: 'Documentation'
set compile_env: 0
classmethod: Lag25000CmwcRandom
comment
^'Lag25000CmwcRandom is a seedable random generator with a period of over 10**240833. 

See also the class comments of Random and SeededRandom.

Lag25000CmwcRandom is a lag-25000 generator using the complementary multiply-with-carry algorithm 
to generate random numbers. It is slow to initialize, but once initialized it is the fastest 
subclass of Random.  Its period is so long that every possible sequence of 24994 successive 
32-bit integers appears somewhere in its output, making it suitable for generating random 
n-tuples where n<24994. Its output is fair in that the number of 0 bits and 1 bits in the 
full sequence are equal.

This generator is recommended for most uses where its initialization time can be tolerated, 
but it is *not* a cryptographically secure generator, so for applications like key generation 
you should consider using HostRandom, once you satisfy yourself that HostRandom is secure 
enough on your operating system.

When you do not need a repeatable sequence of random numbers, an instance of Lag25000CmwcRandom 
can be created with

    Lag25000CmwcRandom new

which will obtain the needed seed bits from the HostRandom.

When you need a repeatable sequence of random numbers, an instance of Lag25000CmwcRandom can 
be created with

    Lag25000CmwcRandom seed: aSmallInteger

which will use the given SmallInteger to seed a Lag1MwcRandom, which will then provide the seeds 
for the Lag25000CmwcRandom.

You can also allow the seed bits to be initialized from the HostRandom, then retrive that state 
by sending #fullState. That state can later be restored by sending the retrieved state as 
an argument to #fullState:.'
%
! ------------------- Instance methods for Lag25000CmwcRandom
category: 'initialization'
set compile_env: 0
method: Lag25000CmwcRandom
initialize

    "The exact choice of these constants is crucial to obtaining the desired result. 
  Refer to literature on the complementary multiply-with-carry algorithm for the 
  relationship between the constants. 
  Note that the implementation of this class also implies the constant b := 2**32, and none of 
  the constants can be changed without changing the others."

	multiplier := 2169967.
	lag := 25000.
%
category: 'private'
set compile_env: 0
method: Lag25000CmwcRandom
setSeedFromGenerator: random
	
	seeds := Array new: lag.
	1 to: lag do: [:i | seeds at: i put: random integer].
	carry := random integer \\ multiplier.
	index := 1
%
category: 'private'
set compile_env: 0
method: Lag25000CmwcRandom
setSeedFromHost
	
	self setSeedFromGenerator: HostRandom new.
%
category: 'private'
set compile_env: 0
method: Lag25000CmwcRandom
validateFullState: fullState

	| newIndex newCarry seedArray |
	fullState size = 3 ifFalse: [ArgumentError signal: 'expected an Array of size 3'].
	newIndex := fullState at: 1.
	newCarry := fullState at: 2.
	seedArray := fullState at: 3.
	newIndex _isSmallInteger ifFalse: [ArgumentError signal: 'Index must be a SmallInteger'].
	(newIndex between: 1 and: lag) ifFalse:  [OutOfRange new 
												name: 'index' min: 1 max: lag actual: newIndex;
															signal: 'out of range'].
	newCarry _isSmallInteger ifFalse: [ArgumentError signal: 'Carry must be a SmallInteger'].
	(newCarry between: 0 and: multiplier - 1) ifFalse: [OutOfRange new 
															name: 'carry' min: 0 max: multiplier - 1 actual: newCarry;
															signal: 'out of range'].
	seedArray size = lag ifFalse: [ArgumentError signal: 'Expected seed array of size ' , lag printString].
	seedArray do: [:seed | 
					seed _isSmallInteger ifFalse: [ArgumentError signal: 'Each seed must be a SmallInteger'].
					(seed between: 0 and: 16rFFFFFFFF) ifFalse: [OutOfRange new
																		name: 'seed' min: 0 max: 16rFFFFFFFF actual: seed;
																		signal: 'out of range']].
%
category: 'public'
set compile_env: 0
method: Lag25000CmwcRandom
integer
	"Answer a random nonnegative 32-bit integer."

	| newCarryAndSeed newSeed |
	index == nil ifTrue: [self setSeedFromHost].
	newCarryAndSeed := (seeds at: index) * multiplier + carry.
	newSeed := 16rFFFFFFFF - (newCarryAndSeed bitAnd: 16rFFFFFFFF).
	carry := newCarryAndSeed bitShift: -32.
	index == lag ifTrue: [index := 0].
	index := index + 1.
	seeds at: index put: newSeed.
	^newSeed
%
category: 'public'
set compile_env: 0
method: Lag25000CmwcRandom
seed: newSeed

"Sets the seed from the given SmallInteger. 
 Since the seed of this class is much larger than a SmallInteger, we
 use this seed to seed a lag 1 generator, then generate our seeds with that generator."

	self setSeedFromGenerator: (Lag1MwcRandom seed: newSeed).
%
run
HostRandom initialize.
true
%
