!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
! 
! UnorderedColl.gs
!
! Superclass Hierarchy:
!    UnorderedCollection, Collection, Object.
!
!=========================================================================

removeallmethods UnorderedCollection
removeallclassmethods UnorderedCollection

category: 'For Documentation Installation only'
classmethod: UnorderedCollection
installDocumentation

self comment:
'UnorderedCollection is an abstract class for collections of objects whose
 elements are not logically arranged in any particular order.  The elements are
 also not physically stored in any fixed order.  Any implied ordering at any
 given time is independent of the order in which the elements were added to the
 collection and cannot be relied upon to persist.

 The elements of unordered collections are all of the same kind.  Unless
 restricted further by a subclass, the kind of elements in unordered collections
 is Object.  That is, the class of each element must simply be some kind of
 Object.

 You cannot add nil to any kind of unordered collection.  Attempts to do so have
 no effect.

 UnorderedCollection provides for fast associative access of collection elements
 in searches by means of the use of indexes with selection blocks.
 UnorderedCollection creates each index for an individual instance, where
 specified, and maintains that index thereafter unless it is removed
 explicitly.

 Indexing is done on instance variables, not on values returned by messages.
 When an index path is used as an argument to a method, it is specified by a
 String that consists of instance variable names separated by periods (such as
 the String ''instvar1.instvar2.instvar3'').  The ith name in the String
 corresponds to the ith position in the path.  A path String may include up to
 16 names and is limited to a total of 1024 Characters.

 If aPathString is an empty path (that is, a zero-length String), the method
 operates upon the elements of the receiver itself rather than upon the instance
 variables of those elements.

 For more information about index structures and path expressions, see the
 GemStone Programming Guide.

All instance variables start with an underscore and are for GemStone internal use.
'.
%

category: 'Private'
classmethod: UnorderedCollection
_isRcIndexLoggingEnabled

""

^ System rcValueCacheAt: #rcIndexLogging otherwise: true.
%

category: 'Private'
classmethod: UnorderedCollection
_enableRcIndexLogging

"System-wide enabling of logging for RC indexes."

System rcValueCache at: #rcIndexLogging put: true
%

category: 'Private'
classmethod: UnorderedCollection
_disableRcIndexLogging

"System-wide disabling of logging for RC indexes
 (only for the life of the transaction)."

System rcValueCache at: #rcIndexLogging put: false
%

category: 'Private'
method: UnorderedCollection
_asCollectionForSorting

"Used by the sorting subsystem."

^ self _asIdentityBag.
%

category: 'Searching'
method: UnorderedCollection
detect: aBlock

"Evaluates aBlock repeatedly, with elements of the receiver as the argument.
 Returns the first element for which aBlock evaluates to true.  If none of the
 receiver's elements evaluates to true, generates an error.  The argument
 aBlock must be a one-argument block.  Uses associative access when the
 argument is a SelectionBlock."

| result |
(aBlock class == SelectBlock)
    ifTrue:[
        "send value: to the block to create the 4 element
        associative access Array; the argument nil is ignored to
        create the Array, but all SelectBlocks have one argument"
        result := (QueryExecuter on: self)
            _boundQueryDetect: (aBlock queryBlock value: nil).
        result == #_incompletePathTraversal
            ifTrue: [
                ^ self _error: #assocErrNoElementsDetected args: { aBlock }
            ].
        ^ result
       ].
^ self _detect: aBlock
%
category: 'Searching'
method: UnorderedCollection
_detect: aBlock

| each bag |
bag := self _asIdentityBag.
1 to: bag size do:[:i |
    each := (bag _at: i) .
    (aBlock value: each) ifTrue:[ ^ each ].
].
^ self _error: #assocErrNoElementsDetected args: { aBlock } .
%


category: 'Searching'
method: UnorderedCollection
detect: aBlock ifNone: exceptionBlock

"Evaluates aBlock repeatedly, with elements of the receiver as the argument.
 Returns the first element for which aBlock has the value true.  If none of the
 receiver's elements has the value true, this evaluates the argument
 exceptionBlock and returns its value.  The argument aBlock must be a
 one-argument block, and exceptionBlock must be a zero-argument block.  Uses
 associative access when the argument is a SelectionBlock."

| each bag result |
(aBlock class == SelectBlock)
    ifTrue:[
        "send value: to the block to create the 4 element
        associative access Array; the argument nil is ignored to
        create the Array, but all SelectBlocks have one argument"
        result := (QueryExecuter on: self)
            _boundQueryDetect: (aBlock queryBlock value: nil).
        result == #_incompletePathTraversal
            ifTrue: [ ^ exceptionBlock value ].
        ^ result
    ].
bag := self _asIdentityBag.
1 to: bag size do:[:i |
    each := (bag _at: i).
    (aBlock value: each)
        ifTrue:[ ^ each ].
].
^ exceptionBlock value
%

category: 'Searching'
method: UnorderedCollection
reject: aBlock

"Evaluates aBlock with each of the receiver's elements as the argument.
 Stores the values for which aBlock is false into a collection of the same
 class as the receiver, and returns the new collection.  The argument aBlock
 must be a one-argument block.  Uses associative access when the argument is
 SelectionBlock."

| result |
(aBlock class == SelectBlock)
    ifTrue:[
       "send value: to the block to create the 4 element
        associative access Array; the argument nil is ignored to
        create the Array, but all SelectBlocks have one argument"
        result := (QueryExecuter on: self)
            _boundQueryReject: (aBlock queryBlock value: nil).
        ^ result
    ].
^ self _reject: aBlock
%
category: 'Searching'
method: UnorderedCollection
_reject: aBlock

| each bag result |
bag := self _asIdentityBag.
result:= NscBuilder for: self species new max: bag size.
1 to: bag size do:[:i |
    each := (bag _at: i).
    (aBlock value: each)
        ifFalse:[ result add: each ].
].
^ result completeBag
%


category: 'Searching'
method: UnorderedCollection
select: aBlock

"Evaluates aBlock with each of the receiver's elements as the argument.
 Stores the values for which aBlock is true into a collection of the same
 class as the receiver, and returns the new collection.  The argument aBlock
 must be a one-argument block.  Uses associative access when the argument is a
 SelectionBlock.

 The new collection returned by this method will not retain any indexes of
 the receiver.  If you want to perform indexed selections on the new
 collection, you must build all of the necessary indexes.  The discussion of
 'Transferring Indexes' in the 'Indexed Associative Access' chapter of the
 GemStone Programming Guide describes a technique for doing this."

| result |
(aBlock class == SelectBlock)
    ifTrue:[
       "send value: to the block to create the 4 element
        associative access Array; the argument nil is ignored to
        create the Array, but all SelectBlocks have one argument"
        result := (QueryExecuter on: self)
            _boundQuerySelect: (aBlock queryBlock value: nil).
        ^ result
    ].
^ self _select: aBlock
%
category: 'Searching'
method: UnorderedCollection
_select: aBlock

| each bag result |
bag := self _asIdentityBag.
result:= NscBuilder for: self species new max: bag size.
1 to: bag size do:[:i |
    each := (bag _at: i) .
    (aBlock value: each) ifTrue:[ result add: each ].
].
^result completeBag
%

category: 'Indexing Support'
method: UnorderedCollection
_hasIndexes

"Returns whether the receiver has any indexes on it."

| result iList |
iList := self _indexedPaths.
result := false.
iList == nil
  ifFalse: [
    2 to: iList size by: 2 do: [ :i |
      (iList at: i) > 0 ifTrue: [ result := true ]
    ]
  ].
^ result
%

category: 'Indexing Support'
method: UnorderedCollection
_hasIncompleteIndexes

"Returns whether the receiver has any incomplete indexes on it.
 If the receiver has incomplete indexes use IndexManager>>removeAllIncompleteIndexesOn:
 to remove the incomplete indexes."

 | iList |
iList := self _indexedPaths.
iList == nil ifTrue: [ ^ false ].
1 to: iList size by: 2 do: [ :i |
  " if it is an incomplete root index ... "
  (((iList at: i + 1) == 1) and: [ (iList at: i) isComplete not])
      ifTrue: [ ^ true ]
].

^ false
%

category: 'Modification Tracking'
method: UnorderedCollection
_hasTrackingObjects

"Returns true if the receiver is being tracked."

| iList |
iList := self _indexedPaths.
iList == nil
  ifFalse: [
    ^ iList hasTrackingObjects
  ].
^ false
%
category: 'Searching'
method: UnorderedCollection
_idxOccurrencesOf: aValue

"Returns the number of occurrences of the given value in the receiver."

^ self occurrencesOf: aValue
%

category: 'Private'
method: UnorderedCollection
_asIdentityBag

"Returns the receiver."

^ self
%

category: 'Adding'
method: UnorderedCollection
add: anObject withOccurrences: anInteger

"Includes anObject as an element of the receiver anInteger number of times.
 Generates an error if anObject is not a kind of the Bag's element kind."

self subclassResponsibility: #add:withOccurrences:
%

category: 'Searching'
method: UnorderedCollection
includes: anObject

"Returns true if anObject is equal to one of the elements of the receiver. 
 Returns false otherwise."

self subclassResponsibility: #includes:
%

category: 'Searching'
method: UnorderedCollection
includesIdentical: anObject

"Returns true if anObject is identical to one of the elements of the receiver. 
 Returns false otherwise."

self subclassResponsibility: #includesIdentical:
%

category: 'Searching'
method: UnorderedCollection
includesValue: anObject

"Returns true if the receiver contains an object of the same value as the
 argument, anObject.  Returns false otherwise.  (Compare with includes:,
 which is based on identity.)"

self subclassResponsibility: #includesValue:
%

category: 'Searching'
method: UnorderedCollection
occurrencesOf: anObject

"Returns the number of the receiver's elements that are identical (==) to
 anObject."

self subclassResponsibility: #occurrencesOf:
%

category: 'Removing'
method: UnorderedCollection
removeIfPresent: anObject

"Removes anObject from the receiver and returns anObject.  If anObject is
 present several times in the receiver, only one occurrence is removed.
 Returns nil if anObject is missing from the receiver."

self remove: anObject ifAbsent: [ ^ nil ].
^ anObject.
%

category: 'Clustering'
method: UnorderedCollection
clusterDepthFirst

"Clusters the receiver and its named and unnamed instance variables in
 depth-first order. Returns true if the receiver has already been clustered
 during the current transaction; returns false otherwise."

self clusterIndexes.
^ super clusterDepthFirst
%

category: 'Indexing Support'
method: UnorderedCollection
_indexedPaths

"Private.  For use only by the indexing subsystem.

 Returns the indexed paths for the receiver."

^ _indexedPaths
%

category: 'Indexing Support'
method: UnorderedCollection
_clearIndexList

"Private.  Sets the indexed paths for the receiver to nil."

self _indexedPaths: nil
%

category: 'Storing and Loading'
classmethod: UnorderedCollection
loadFrom: passiveObj

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| size inst |
size := passiveObj readSize.
inst := self new.
inst loadFrom: passiveObj size: size.
^inst
%

category: 'Storing and Loading'
method: UnorderedCollection
basicLoadFrom: passiveObj size: varyingSize

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into the receiver."

"This method is similar to basicLoadFrom:, but is used for objects whose size
 is not known when they are first instantiated (such as an IdentitySet)."

passiveObj hasRead: self.
^ self basicLoadFromNoRead: passiveObj size: varyingSize .
%

category: 'Storing and Loading'
method: UnorderedCollection
basicLoadFromNoRead: passiveObj size: varyingSize

"Private."

varyingSize == 0 ifTrue: [
  "Old NSC format had no named instance variable section.  A zero-length NSC in
   the old format might read instance variables from an enclosing object if
   there were not a special delimiter that could be reliably found."
  passiveObj checkForBagMark ifFalse: [
    passiveObj checkForInstVarMark ifFalse: [
      ^self
    ].
  ].
].
(passiveObj readNamedIV) ifFalse: [
  "old NSC format with no named instance variables"
  ^self loadVaryingFrom: passiveObj size: varyingSize
].
self loadNamedIVsFrom: passiveObj.
self loadVaryingFrom: passiveObj size: varyingSize.
%

category: 'Storing and Loading'
method: UnorderedCollection
basicWriteTo: passiveObj

"Converts the receiver to its passive form and writes that information on
 passiveObj."

| s cls c |
  "store my structure on the given passiveObj"
  cls := self class.
  passiveObj writeClass: cls.

  passiveObj writeSize: (s := self size) .

  passiveObj writeNamedIvsFrom: self class: cls .
  passiveObj writeBagMark.
  passiveObj endNamedInstVars.

  c := 0.
  self do: [ :x |
    passiveObj writeObject: x.
    c := c + 1.
    c > 99 ifTrue: [
      passiveObj lf.
      c := 0.
    ].
  ].
  passiveObj cr
%

category: 'Storing and Loading'
method: UnorderedCollection
loadFrom: passiveObj size: varyingSize

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into the receiver."

"This method is similar to loadFrom:, but is used for objects whose size
 is not known when they are first instantiated (such as an IdentitySet)."

^self basicLoadFrom: passiveObj size: varyingSize
%

category: 'Storing and Loading'
method: UnorderedCollection
loadNamedIVsFrom: passiveObj

"Reads named instance variables from the given passive object.  The first
 instance variable should already have been parsed and be available in the
 passiveObj argument."

| name offset nameSym |

[ name := passiveObj ivName.
  name ifNotNil: [
    nameSym := Symbol _existingWithAll: name .
    nameSym ifNotNil:[
      offset := self class _ivOffsetOf: nameSym.
      offset ifNotNil:[ self instVarAt: offset put: passiveObj ivValue ]
             ifNil:[ self dynamicInstVarAt: nameSym put: passiveObj ivValue].
    ] ifNil:[
      self dynamicInstVarAt: name asSymbol put: passiveObj ivValue
    ].
    passiveObj readNamedIV
  ] ifNil: [
    false
  ]
] untilFalse.

passiveObj checkForBagMark.
passiveObj skipNamedInstVars.
%

category: 'Storing and Loading'
method: UnorderedCollection
loadVaryingFrom: passiveObj size: varyingSize

"Reads the varying part of the receiver from the given passive object.
 Does not record the receiver as having been read.  Does not read the
 receiver's named instance variables, if any."

1 to: varyingSize do: [:i |
  self add: (passiveObj readObject)
].
%

category: 'Error Handling'
method: UnorderedCollection
_raiseIndexingError: errArray
  "Raises an error that was due to updating indexes.  Information
 about the error is contained in the given error Array."

  | anException |
  anException := errArray at: 2.
  (errArray at: 1)
    ifTrue: [ 
      "indexed were modified"
      ^ IndexingErrorPreventingCommit new
        args: {anException};
        details: anException description;
        signal ]
    ifFalse: [ 
      "can't resignal without copy ; may have already been handled"
      ^ anException copy signal ]
%

category: 'Repository Conversion'
method: UnorderedCollection
getIndexInfo

"Returns indexing information for the receiver."

| iList indexObj path kind lastConstraint result |
  result := { } .
  iList := self _indexedPaths.
  1 to: iList size by: 2 do: [ :i |
    indexObj := iList at: i.
    (iList at: i + 1) == 1 ifTrue: [
       " path is first arg "
       path := indexObj pathComponentsString.

       " kind is true if identity index, false if equality index "
       kind := indexObj isIdentityIndex.

       " for equality indexes, lastConstraint is second argument "
       kind ifFalse: [ 
          lastConstraint := indexObj lastElementClass.  
          (lastConstraint == SmallInteger) ifTrue: [ lastConstraint := Integer ].
       ].
       result add: { self . path . kind . lastConstraint } .
    ].
  ].
  ^ result.
%

category: 'Repository Conversion'
method: UnorderedCollection
rebuildIndexes: indexInfo

"Rebuilds the indexes."

" Interim conversion: need method declaration, but commented out..
 
| path kind lastConstraint newLastConstraint |

  path := indexInfo at: 2.
  kind := indexInfo at: 3.
  lastConstraint := indexInfo at: 4.

  kind
    ifTrue: [
      self createIdentityIndexOn: path
      ]
    ifFalse: [
      newLastConstraint := lastConstraint _correspondingNewClass.
      self createEqualityIndexOn: path withLastElementClass: newLastConstraint
      ].
"
%
category: 'Repository Conversion'
method: UnorderedCollection
convRemoveIndexes

"Private. Removes the indexes for the receiver. This method is to be used
 only during Repository Conversion."

self _clearIndexList.
%

category: 'Copying'
method: UnorderedCollection
postCopy

"Cleanup new copy."

super postCopy.
_indexedPaths := nil
%

category: 'Copying'
method: UnorderedCollection
shallowCopy

"Returns a copy of the receiver which shares the receiver's instance
 variables."

<primitive: 746>
self _primitiveFailed: #shallowCopy .
self _uncontinuableError
%

category: 'Accessing'
method: UnorderedCollection
instVarAt: aSmallInteger

"If the receiver has a publicly accessible named instance variable at
 index aSmallInteger, this returns its value.  Generates an error if
 aSmallInteger is not a SmallInteger or is out of bounds, or if the
 receiver has no publicly accessible named instance variables."

(UnorderedCollection instSize >= aSmallInteger and:[ 1 <= aSmallInteger ])
  ifTrue:[ ^ nil ].

^ super instVarAt: aSmallInteger
%

category: 'Updating'
method: UnorderedCollection
instVarAt: anIndex put: aValue

"Stores the argument aValue in the instance variable indicated by anIndex and
 returns aValue. Generates an error if (1) anIndex is not a SmallInteger,
 (2) anIndex is out of bounds or (3) if the receiver has no publicly accessible
 named instance variables."

(UnorderedCollection instSize < anIndex) ifTrue:[
  ^ super instVarAt: anIndex put: aValue	"skip structural access check"
].
^ self _errorIndexOutOfRange: anIndex
%

category: 'Testing'
method: UnorderedCollection
_isIdentityBag

"Return whether the receiver is an identity-based collection."

^ false
%

category: 'Instance Migration'
method: UnorderedCollection
migrateFrom: anotherObject instVarMap: otherivi

"Takes information from the given object and puts it in the receiver.  This
 message is sent to an object when its class is being migrated to another class
 to account for changes in a schema.  The otherivi argument is a precalculated
 indirection table associating the receiver's instance variables with instance
 variables in the other object.  If a table entry is 0, the other object is
 assumed not to have that instance variable.

 This method should be augmented to perform other necessary initializations in
 the receiver."

| otherClass |

super migrateFrom: anotherObject instVarMap: otherivi.

"if no elements in anotherObject, no need to do anything "
anotherObject size == 0
  ifTrue: [ ^ self ].

otherClass := anotherObject class.

otherClass isNonByteVarying
  ifTrue: [
    anotherObject accompaniedBy: self do: [ :me :anElement | me add: anElement ].
  ].

^ self.
%

category: 'Auditing'
method: UnorderedCollection
repairInternalStructures
"For an explanation of this method, refer to the comments in the method
 UnorderedCollection>>auditInternalStructuresWithRepair:"

^self auditInternalStructuresWithRepair: true
%

category: 'Auditing'
method: UnorderedCollection
auditInternalStructuresWithRepair: aBoolean
"Audit and optionally repair the internal structures of the receiver.
 If aBoolean is true, errors will be fixed as they are detected, but
 the transaction is not automatically committed and a commit after this
 method returns is required.  If aBoolean is false, the audit is 
 read-only operation and no objects are modified.  

 Answers a Boolean indicating if the collection is free of errors. 
 true means the collection passed the audit, false means the audit
 detected one or more errors.

 This primitive prints many messages to stdout and is intended to
 be run from a linked topaz session."

<primitive: 574>
^self _primitiveFailed: #auditInternalStructuresWithRepair: args: { aBoolean }
%

category: 'Auditing'
method: UnorderedCollection
auditInternalStructures
"See the method auditInternalStructuresWithRepair: aBoolean for
 more information on this method."

^self auditInternalStructuresWithRepair: false
%

category: 'Accessing Indexes'
method: UnorderedCollection
identityIndexedPaths
  "Returns an Array of Strings, each of which represents a path for which an
 identity index exists in the receiver.  Each path originates with the elements
 of the receiver."

  | anArray |
  anArray := {}.
  self _indexedPaths == nil
    ifTrue: [ ^ anArray ].
  self _indexedPaths
    indexObjectsAndOffsetsDo: [ :indexObj :offset | 
      | pathString |
      indexObj isIdentityIndex
        ifTrue: [ 
          pathString := indexObj pathComponentsStringStartingAt: offset.
          (anArray includesValue: pathString)
            ifFalse: [ anArray addLast: pathString ] ]
        ifFalse: [ 
          (self _isRangeable: indexObj lastElementClass)
            ifTrue: [ 
              pathString := indexObj pathComponentsStringStartingAt: offset.
              indexObj isComplete
                ifFalse: [ pathString add: ' (incomplete)' ].
              (anArray includesValue: pathString)
                ifFalse: [ anArray addLast: pathString ] ] ] ].
  ^ anArray
%

! deleted createIdentityIndexOn:commitInterval: 

category: 'Indexing Support'
method: UnorderedCollection
_checkIndexPathExpression: aPathString
  "Raises an error if the given path expression (an Array of Strings), is
 not valid; otherwise returns the receiver."

  ^ aPathString asArrayOfPathTerms
%
category: 'Updating Indexes'
method: UnorderedCollection
removeIncompleteIndex
  "If there is an incomplete index, clean it up. In general this method should not be used, a better way to remove
 incomplete indexes is to use IndexManager>>removeAllIncompleteIndexesOn:"

  [ 
    System _zeroArgPrim: 51 . "_enteringIncompleteIndexRemoval"
    [ | systm indexObj |
      systm := System.
      systm abortTransaction.
      indexObj := self _findIncompleteIndex.
      indexObj ifNil: [ ^ self ].
      self
        _undoIndexCreation: indexObj
        pathTerm: indexObj _findFirstUnsharedPathTerm.
      systm _commitPrintingDiagnostics 
    ]
    untilTrue
  ] ensure:[
    System _zeroArgPrim: 52 . "_leavingIncompleteIndexRemoval"
  ]
%

category: 'Updating Indexes - Private'
method: UnorderedCollection
_findIncompleteIndex

"Returns an index that is incomplete, or nil if not found."

| iList |
iList := self _indexedPaths.
iList == nil
  ifTrue: [ ^ nil ].

" for each index on the NSC ... "
1 to: iList size by: 2 do: [ :i |
  " only look at the index if the receiver is the root "
  (iList at: i + 1) == 1
    ifTrue: [
      " see if the index is complete "
      (iList at: i) isComplete
        ifFalse: [ ^ iList at: i ]
    ]
].
^ nil
%

category: 'Indexing Support'
method: UnorderedCollection
_isRangeable: aClass
 
"Returns whether the given class should have an equality index created when an
 identity index is requested."

^ aClass _isSpecial and: [ (aClass isSubclassOf: AbstractCharacter) not ]
%


category: 'Updating Indexes - Private'
method: UnorderedCollection
_calculateIndexDictionarySize: indexPathLength

"Heuristic for determining size of index dictionary root."

| sz  |
"One entry in dictionary for each element/pathTerm key and a bucket load factor of 2/3."
sz := (self size * indexPathLength) // 
       ((RcIndexDictionary defaultBasicSize * 2) // 
        (RcIndexBucket entrySize * 3)).
sz := sz max: RcIndexDictionary defaultBasicSize.
self _setIndexDictionaryCreationSize: (Integer _selectedPrimeGreaterThan: sz).
%

category: 'Indexing Support'
method: UnorderedCollection
_getIndexDictionaryCreationSize

"Returns the basic size of an index dictionary when a new one is
 being created."

^ System
  rcValueCacheAt: #indexDictionarySize
  for: self
  otherwise: RcIndexDictionary defaultBasicSize.
%

! fix statement with no effect with fix 34213
category: 'Indexing Support'
method: UnorderedCollection
_setIndexDictionaryCreationSize: aNumber

"Sets the basic size of an index dictionary when a new one is being created.
 This will only be in effect for the life of the transaction.
 Returns self.  "

aNumber _validateClass: Number.
System
  rcValueCacheAt: #indexDictionarySize
  put: aNumber
  for: self.

%

category: 'Indexing Support'
method: UnorderedCollection
_indexedPaths: anIndexListOrNil
"used along with SimpleDomainTest to be able to efficiently switch between indexed and non-indexed queries"

_indexedPaths := anIndexListOrNil
%

! edited for 43084
category: 'Adding'
method: UnorderedCollection
_updateIndexesForAdditionOf: anObject logging: aBoolean
  "anObject is being added to the receiver.  Update any indexes if necessary.
 Returns true if the index objects were modified correctly; otherwise returns
 an Array containing error information."

  <primitive: 2001>
  | doLogging iList hasIndex prot |
  prot := System _protectedMode.
  [ 
  iList := self _indexedPaths.
  hasIndex := false.
  1 to: iList size by: 2 do: [ :j | 
    " first handle modification tracking "
    (iList at: j + 1) > 0
      ifTrue: [ hasIndex := true ]
      ifFalse: [ 
        (iList at: j + 1) == 0
          ifTrue: [ (iList at: j) adding: anObject to: self ] ] ].
  hasIndex
    ifFalse: [ ^ true ].
  System _bypassReadAuth.
  doLogging := iList isCommitted
    and: [ UnorderedCollection _isRcIndexLoggingEnabled ].	" if the indexes are not committed, do not need to log "
  [ ^ self _doUpdateIndexesForAdditionOf: anObject iList: iList logging: aBoolean ]
    onSynchronous: Error
    do: [ :ex | 
      " returns Array containing err info "
      ^ {true.	" indicating indexing objects may have been modified "
      ex} ] ]
    ensure: [ 
      System _exitBypassReadAuth.
      prot _leaveProtectedMode ].
^ true
%


category: 'Removing'
method: UnorderedCollection
_updateIndexesForRemovalOf: anObject

"anObject is being removed from the receiver.  Update the indexes if
 necessary.  This method is invoked prior to removal of anObject
 from the receiver. "

<primitive: 2001>
| iList hasIndex prot |
prot := System _protectedMode .
[ 	" if it is not an RcIdentityBag, anObject may not be in the receiver "
	( self class isNsc and: [ (self includes: anObject) not ] ) ifTrue: [
		^ true
	].
	iList := self _indexedPaths.
	" first handle modification tracking "
	hasIndex := false.
	1 to: iList size by: 2 do: [ :j |
		(iList at: j + 1) > 0
			ifTrue: [ hasIndex := true ]
			ifFalse: [
				(iList at: j + 1) == 0 ifTrue: [ (iList at: j) removing: anObject from: self ]
			]
	].
	hasIndex ifFalse: [ ^ true ].

	System _bypassReadAuth .
    [self _doUpdateIndexesForRemovalOf: anObject iList: iList ] onSynchronous: Error do:[ :ex | 
      " returns Array containing err info "
      ^ {true.	" indicating indexing objects may have been modified "
      ex}	]
] ensure: [ 
	System _exitBypassReadAuth .
	prot _leaveProtectedMode
].
  ^ true
%

category: 'Updating Indexes - Private'
method: UnorderedCollection
_lockForIndexCreation

"Locks the receiver to prevent concurrent users from invalidating index
 creation."

| lockList |
lockList := { self } .
self _indexDictionary ifNotNil:[
    self _indexDictionary _doCollisionBuckets: [ :cb | lockList add: cb ]
  ].
System writeLockAll: lockList ;
   addAllToCommitOrAbortReleaseLocksSet: lockList.
%

category: 'Indexing Support'
method: UnorderedCollection
_findIndexesWithPath: pathArray

"Returns an Array of index objects whose path components are the same as
 represented by the path string.  This method only considers those indexes that
 are defined with the receiver as the root NSC."

| anArray iList |
iList := self _indexedPaths.
iList == nil
  ifTrue: [ ^ { }  ].

anArray := { } .
" for each index on the NSC ... "
1 to: iList size by: 2 do: [ :i |
  " only look at the index if the receiver is the root "
  (iList at: i + 1) == 1
    ifTrue: [
      " if the index is on the same path, add it to the Array "
      ((iList at: i) hasIndexOnPath: pathArray)
        ifTrue: [ anArray addLast: (iList at: i) ]
    ]
].
^ anArray
%

category: 'Modification Tracking'
method: UnorderedCollection
_setModificationTrackingTo: tracker

"Adds the given tracker to the receiver's indexedPaths and dependency lists."

"Must set indexedPaths first so tracker is not invoked for instance variable
 modification."
self _getIndexList addTracker: tracker.

super _setModificationTrackingTo: tracker.
%

category: 'Modification Tracking'
method: UnorderedCollection
_clearModificationTrackingTo: tracker

"Remove the given tracker from the receiver's indexedPaths and dependency lists."

"Must clear depmap entry first, so we don't get notified about IV changes"
super _clearModificationTrackingTo: tracker.

self _getIndexList removeTracker: tracker for: self.

%

category: 'Updating Indexes - Private'
method: UnorderedCollection
_removeAllRootIndexes: rootIndexes hasNonRootIndex: hasNonRootIndex

"Removes all root indexes on the receiver."

| array roots |

hasNonRootIndex ifTrue: [
    " only remove root indexes "
    1 to: rootIndexes size do: [ :i |
      self _removeIndex: (rootIndexes at: i)
    ].
    " if the receiver still has implicit indexes ... "
    self _indexedPaths ifNotNil: [ | iList |
        iList := self _indexedPaths.
        1 to: iList size by:2 do: [:i | | iObj |
           iObj := iList at: i.
           (iObj isComplete and: [ (iList at: i + 1) > 0 ]) ifTrue: [ 
               " and they are complete...."
               ^ self _indexParticipationInfo 
             ].
         ].
      ]
  ] ifFalse: [
    array := { } .
    roots := self _indexedPaths rootTerms.
    1 to: roots size do: [ :i |
      (roots at: i) _thisAndAllChildTermsInto: array
    ].

    " this will clean up dependency tags and nil out the index list "
    self _cleanUpDependencies.

    " remove dependency lists from global table that are no longer needed "
    SharedDependencyLists removeEntriesContaining: array.
  ].
%

category: 'Indexing Support'
method: UnorderedCollection
_indexParticipationInfoInto: array

"Returns an Array of pairs:  the root NSC and the path string that describes
 the path traversed from the root to reach the receiver."

| iList indexObj pathString included |
iList := self _indexedPaths.
1 to: iList size by: 2 do: [ :i |
  (iList at: i + 1) > 1 ifTrue: [
      indexObj := iList at: i.
      pathString := indexObj _partialPathComponentsStringUpTo: (iList at: i + 1).
      included := false.
      1 to: array size by: 2 do: [ :j |
        ( indexObj nscRoot == (array at: j) and:
           [ (array at: j + 1) = pathString ] )
               ifTrue: [ included := true ]
      ].
      included ifFalse: [
          array add: indexObj nscRoot.
          array add: pathString
        ]
    ]
].
^ super _indexParticipationInfoInto: array
%

category: 'Updating Indexes - Private'
method: UnorderedCollection
_cleanUpDependencies

"The receiver is having all of its indexes removed.  Remove dependency
 list entries for objects along the path of the indexes."

| rootTerms bag prevObj obj iList |
iList := self _indexedPaths.
rootTerms := iList rootTerms.

prevObj := #_incompletePathTraversal.
bag := self _asIdentityBag.
1 to: bag size do: [ :i |
  obj := bag _at: i.
  (obj == prevObj) ifFalse: [
      1 to: rootTerms size do: [ :j |
        (rootTerms at: j) cleanupDependencyListFor: obj
      ]
    ].
  prevObj := obj
].

iList hasTrackingObjects
  ifTrue: [ iList removeAllIndexesFor: self ]
  ifFalse: [ self _indexedPaths: nil ].

" resize the Array of path terms to avoid object audit errors "
rootTerms size: 0.
%

category: 'Indexing Support'
method: UnorderedCollection
_putInWriteSet

"In some cases, it is necessary to put an NSC with indexes into the write set
 explicitly.  For example, when NO_RW_CHECKS concurrency mode is enabled, index
 creation should put the NSC in the write set to ensure that other transactions
 do not commit additions to the NSC successfully (and thus corrupting the
 internal indexing objects).  This method puts the receiver in the write set by
 writing the _indexedPaths."

self _indexedPaths: self _indexedPaths
%

category: 'Indexing Support'
method: UnorderedCollection
_isLastOccurrenceInIndexObjects: anObject
  "Returns true if the given object is maintained in the indexing objects for one
 occurrence."

  | rootTerms pathTerm key val val2 num parentTerm |
  rootTerms := self _indexedPaths rootTerms.	" find a path term with a mapping in the index dictionary "
  1 to: rootTerms size do: [ :i | 
    pathTerm := rootTerms at: i.
    (pathTerm _isObsoletePathTerm)
      ifFalse: [ 
        pathTerm isRangeEqualityIndexLastPathTerm
          ifTrue: [ 
            pathTerm offset == 1
              ifTrue: [ ^ pathTerm _findAllValuesForIdenticalRootObject: anObject ]
              ifFalse: [ 
                pathTerm hasIndexDictionary
                  ifTrue: [
                    pathTerm := pathTerm getParentTerm.
                    val := pathTerm updateDict
                      at: anObject
                      term: pathTerm
                      otherwise: nil.
                    (BucketValueBag _hasInstance: val)
                      ifTrue: [     
                        " see if more than one mapping "
                        pathTerm indicatesNsc
                          ifTrue: [ 
                            num := val occurrencesOf: self.
                            num < val size
                              ifTrue: [ 
                                " if anObject is contained in other NSCs "
                                ^ false ].
                            pathTerm := pathTerm getParentTerm.	" get path term before this one "
                            val2 := pathTerm updateDict
                              at: self
                              term: pathTerm
                              otherwise: nil.
                            (BucketValueBag _hasInstance: val2)
                              ifTrue: [ ^ val2 size == num ]
                              ifFalse: [ ^ true ] ]
                          ifFalse: [ ^ (val occurrencesOf: self) <= 1 ] ]
                      ifFalse: [ ^ true ] ]
                  ifFalse: [ ^ (self occurrencesOf: anObject) <= 1] ] ]
          ifFalse: [ 
            pathTerm hasIndexDictionary
              ifTrue: [ 
                " get key to look up in index dictionary "
                (pathTerm indicatesIndexOnNscElements or: [ nil == anObject ])
                  ifTrue: [ key := anObject ]
                  ifFalse: [ 
                    " see if a path with '*.*' in it "
                    pathTerm indicatesNsc
                      ifTrue: [ 
                        val := pathTerm updateDict
                          at: anObject
                          term: pathTerm getParentTerm
                          otherwise: nil.
                        ^ (BucketValueBag _hasInstance: val) not ]
                      ifFalse: [ key := pathTerm _nextObjectFor: anObject ] ].
                val := pathTerm updateDict at: key term: pathTerm otherwise: nil.	" look up the mapping in the index dictionary "
                (BucketValueBag _hasInstance: val)
                  ifTrue: [ 
                    " see if more than one mapping "
                    (num := val occurrencesOf: anObject) <= 1
                      ifTrue: [ ^ true ]
                      ifFalse: [ 
                        " see if multiple occurrences are due to more than one object
                        referencing self "
                        " if there is a parent term, it is a SetValuedPathTerm "
                        parentTerm := pathTerm getParentTerm.
                        parentTerm == nil
                          ifTrue: [ ^ false ].
                        val2 := pathTerm updateDict
                          at: anObject
                          term: parentTerm
                          otherwise: nil.
                        (BucketValueBag _hasInstance: val2)
                          ifTrue: [ ^ (val2 occurrencesOf: self) == num ]
                          ifFalse: [ ^ false ] ] ]
                  ifFalse: [ ^ true ] ] 
              ifFalse: [ ^ (self occurrencesOf: anObject) <= 1] ] ] ].
  ^ true	" if we get this far, there are only incomplete indexes "
%

category: 'Indexing Support'
method: UnorderedCollection
_indexDictionary

"Returns the index dictionary that is shared by all indexes.  If there are no
 indexes, returns nil."

| iList |
iList := self _indexedPaths.
iList 
  ifNil: [ ^ nil ]
  ifNotNil: [
    1 to: iList size by: 2 do: [ :i |
      ( (iList at: i + 1) == 1 and:
        [ (iList at: i) indexDictionary ~~ nil ] )
        	ifTrue: [ ^ (iList at: i) indexDictionary ]
    ].
    ^ nil
  ]
%

! deleted createEqualityIndexOn:commitInterval:
! deleted createEqualityIndexOn:withLastElementClass:commitInterval: 

category: 'Accessing Indexes'
method: UnorderedCollection
equalityIndexedPaths

"Returns an Array of Strings, each of which represents a path for which an
 equality index exists in the receiver.  Each path originates with the elements
 of the receiver."

| anArray |
anArray := { } .
self _indexedPaths ifNil: [ ^ anArray ].

self _indexedPaths indexObjectsAndOffsetsDo: [ :indexObj :offset | | str |
  indexObj isRangeEqualityIndex
    ifTrue: [
      str := indexObj pathComponentsStringStartingAt: offset.
      indexObj isComplete
        ifFalse: [ str add: ' (incomplete)' ].
      (anArray includesValue: str)
        ifFalse: [ anArray addLast: str ]
    ]
].
^ anArray
%
category: 'Accessing Indexes'
method: UnorderedCollection
equalityIndexedPathsAndConstraints

"Returns an Array containing info about equality indexes.  The Array consists of
 String/Class pairs.  The string represents a path of the receiver's element
 kind for which an equality index exists in the receiver.  The class is the
 constraint on the last element in the path."

| anArray |
anArray := { } .
self _indexedPaths ifNil: [ ^ anArray ].

self _indexedPaths indexObjectsAndOffsetsDo: [ :indexObj :offset |
    indexObj isRangeEqualityIndex
        ifTrue: [
            anArray addLast: (indexObj pathComponentsStringStartingAt: offset).
            anArray addLast: indexObj lastElementClass
        ]
].
^ anArray
%
category: 'Accessing Indexes'
method: UnorderedCollection
kindsOfIndexOn: aPathString
  "Returns a Symbol that indicates the kinds of indexes into the receiver that
 exist on aPathString: #identity, #equality, #equalityAndIdentity, or #none
 (either aPathString is not a path for the element kind of the receiver, or no
 indexes into the receiver exist on aPathString)."

  | pathList pathArray |
  self _indexedPaths ifNil: [ ^ #'none' ].
  pathList := {}.
  pathArray := aPathString asArrayOfPathTerms.
  self _indexedPaths
    indexObjectsAndOffsetsDo: [ :indexObj :offset | 
      (offset <= indexObj size
        and: [ 
          pathArray size > (indexObj size - offset)
            and: [ indexObj hasIndexOnPath: pathArray startingAt: offset ] ])
        ifTrue: [ pathList add: indexObj ] ].
  pathList size == 2
    ifTrue: [ ^ #'equalityAndIdentity' ].
  pathList size == 1
    ifTrue: [ 
      pathList first isRangeEqualityIndex
        ifTrue: [ 
          (self _isRangeable: pathList first lastElementClass)
            ifTrue: [ ^ #'equalityAndIdentity' ]
            ifFalse: [ ^ #'equality' ] ]
        ifFalse: [ ^ #'identity' ] ].
  self _indexedPaths
    ifNotNil: [ 
      (self _indexedPaths _numberOfCommonPathTermsForPathArray: pathArray)
        == pathArray size
        ifTrue: [ ^ #'identity' ]
        ifFalse: [ ^ #'none' ] ].
  ^ #'none'
%
category: 'Clustering'
method: UnorderedCollection
clusterIndexes

"Clusters internal indexing objects using the current default ClusterBucket."

self _clusterIndexes
%

category: 'Clustering'
method: UnorderedCollection
_clusterIndexes

"Clusters indexing objects.  This may cause concurrency conflicts on
 internal indexing objects.  Returns the receiver."

| iList |
(iList := self _indexedPaths) ifNil: [ ^ self ].

" cluster the index dictionary "
self _indexDictionary ifNotNil: [ self _indexDictionary clusterDepthFirst ].

" cluster each index object "
1 to: iList size by: 2 do: [ :j |
  (iList at: j) _isIndexObject
    ifTrue: [ (iList at: j) clusterDepthFirst ]
].
%

category: 'Indexing Audit'
method: UnorderedCollection
_quickCheckIndexes
  "Verifies that the index objects are present. This method is does 
 not catch most types of index corruption. The return values are 
 messages that no indexes are present, or that indexes are okay. 
 Any errors found will result in an unhandled exception. 
 
 This method is unsupported, but is provided for customer support."

  | prevObj |
  self _indexedPaths ifNil: [ ^ 'No indexes are present.' ].	"execute minor code over all the path terms"
  self _indexedPaths rootTerms
    do: [ :rootTerm | 
      rootTerm _thisAndAllChildTerms
        do: [ :aTerm | 
          aTerm indicatesNsc
            ifFalse: [ aTerm isRangeEqualityIndexLastPathTerm ] ] ].
  prevObj := #'_incompletePathTraversal'.
  self
    do: [ :obj | 
      obj ~~ prevObj
        ifTrue: [ obj _indexParticipationInfo ].
      prevObj := obj ].
  ^ 'Indexes are OK'
%

category: 'Updating Indexes'
method: UnorderedCollection
progressOfIndexCreation

"Returns a String that describes the progress of an index creation that is
 underway."

| indexObj cnt str lf |

indexObj := self _findIncompleteIndex.
( indexObj == nil or:
" get the current offset of enumeration through the NSC "
  [ (cnt := indexObj progress) == nil ] )
     ifTrue: [ 
    ^ 'No index creation is in progress.' 
].

str := String new.
lf := Character lf.
str add: lf; add: 'Creating index on <';
  add: indexObj pathComponentsString; add: $>; add: lf;
  add: 'During enumeration of collection, finished ';
  add: cnt asString; add: ' out of ';
  add: self size asString; add: ' total elements.'.

indexObj isRangeEqualityIndex ifTrue: [
    " get the current offset of enumeration through the n-way merge "
    cnt := indexObj nwayMergeProgress.
    cnt ifNotNil: [
        str add: lf; add: 'During n-way merge, finished ';
          add: cnt asString; add: ' elements'; add: lf.
	]
].

^ str
%

category: 'Deprecated'
method: UnorderedCollection
_incrementalAuditIndexes: start to: end
  "Verifies that the index objects are consistent. Scans only the objects in the range 
 from start to end, inclusive. This may take a while. A better option is to use
 _fastAuditIndexes. This method is not supported. "

  | roots sz aString prevObj nscBuilder index |
  self deprecated: 'UnorderedCollection>>_incrementalAuditIndexes:to: deprecated v3.2, Constraints are no longer supported'.
  self _indexedPaths ifNil: [ ^ 'No indexes are present.' ].
  roots := self _indexedPaths rootTerms.
  sz := roots size.
  aString := String new.	" audit by traversing the tree of path terms for each element "
  prevObj := #'_incompletePathTraversal'.
  nscBuilder := NscBuilder for: IdentityBag new max: NscBuilder maxSize.
  index := 1.
  self
    do: [ :obj | 
      (index >= start and: [ index <= end ])
        ifTrue: [ 
          obj ~~ prevObj
            ifTrue: [ 
              | oCount |
              oCount := self occurrencesOf: obj.
              1 to: sz do: [ :j | 
                | rootTerm |
                rootTerm := roots at: j.
                rootTerm
                  auditObject: obj
                  occurrences: oCount
                  on: aString
                  builder: nscBuilder.
                nscBuilder _resetForAudit ] ].
          prevObj := obj ].
      index := index + 1.
      index > end
        ifTrue: [ 
          aString isEmpty
            ifTrue: [ aString := 'Indexes are OK' ].
          ^ aString ] ].	"0 element nsc case"
  aString isEmpty
    ifTrue: [ aString := 'Indexes are OK' ].
  ^ aString
%

category: 'Indexing Audit'
method: UnorderedCollection
_fastAuditIndexes
  "Verifies that the index objects are consistent.
 Returns a string that describes any inconsistencies found."

  | str1 str2 success |
  self _indexedPaths ifNil: [ ^ 'No indexes are present.' ].	"IndexProgressCount stage 1: audit identity indexes"
  System setProgressCountTo: 0.
  self _hasIncompleteIndexes
    ifTrue: [ 
      | str0 |
      str0 := 'Incomplete indexes are present. Use IndexManager>>removeAllIncompleteIndexesOn: 
to remove the incomplete indexes. After incomplete indexes have been removed, rerun the audit.
'
        copy.
      self _indexedPaths
        indexObjectsDo: [ :index | 
          index isComplete
            ifFalse: [ 
              str0
                add:
                  '  -- The index ' , index printString , ' [' , index asOop printString
                    ,
                      '] is incomplete.
' ] ].
      ^ str0 ].
  str1 := self _fastAuditIdentityIndexes.
  System setProgressCountTo: 0.
  str2 := self _fastAuditEqualityIndexes.
  System
    setProgressCountTo: 0;
    setIndexProgressCountTo: 0.
  success := 'Indexes are OK'.
  str1 = success
    ifTrue: [ 
      str2 = success
        ifTrue: [ ^ success ].
      ^ str2 ].
  str2 = success
    ifTrue: [ ^ str1 ].
  ^ str1 , str2
%

category: 'Indexing Audit'
method: UnorderedCollection
_fastAuditIdentityIndexes

"Verifies that the identity index objects are consistent. When run in
 conjunction with _fastAuditEqualityIndexes a complete audit is performed.
 Returns a string that describes any inconsistencies found."

| aString sys |

self _indexedPaths ifNil: [ ^ 'No indexes are present.' ].

sys := System .
aString := String new .

[ | roots sz |
  roots := self _indexedPaths rootTerms.
  sz := roots size.
  sys setIndexProgressCountTo: self class statValueForAuditingIdentityIndexes ; "1"
      setProgressCountTo: sz .

  1 to: sz do: [ :j |
    | rootTerm indexDictionary |
    rootTerm := roots at: j.
    indexDictionary := rootTerm getIndexDictionary.
    indexDictionary ~~ nil
      ifTrue: [ 
        indexDictionary auditNsc: self for: rootTerm on: aString.
        indexDictionary auditEntriesForNsc: self for: rootTerm on: aString.
      ].
      sys decrementProgressCountBy: 1 . "dec the progress count after each term"
  ].
  aString isEmpty
    ifTrue: [ aString := 'Indexes are OK' ].
] ensure:[ sys setIndexProgressCountTo: 0; setProgressCountTo: 0 ] .

^ aString
%

category: 'Indexing Audit'
method: UnorderedCollection
_fastAuditEqualityIndexes
  "Verifies that the equality index objects are consistent. When run in
 conjunction with _fastAuditIdentityIndexes a complete audit is performed.
 Returns a string that describes any inconsistencies found."

  | aString sys participatesInCollectionValuedIndex participatesInIndex |
  self _indexedPaths ifNil: [ ^ 'No indexes are present.' ].
  aString := String new.
  sys := System.
  [ 
  | roots sz btreeCounts rootTerm countArg |
  roots := self _indexedPaths rootTerms.
  sz := roots size.
  btreeCounts := Array new.
  sys setIndexProgressCountTo: self class statValueForAuditingRootTerms.	"2"
  sys setProgressCountTo: sz.
  participatesInIndex := participatesInCollectionValuedIndex := false.
  1 to: sz do: [ :j | 
    | resultArray |
    rootTerm := roots at: j.
    rootTerm offset == 1
      ifTrue: [ 
        "index on receiver"
        participatesInIndex := true.
        resultArray := rootTerm auditNsc: self on: aString level: 1.
        btreeCounts add: resultArray ]
      ifFalse: [ 
        "participates in index with collection-valued path term"
        participatesInCollectionValuedIndex := true.
        btreeCounts add: {} ].
    sys decrementProgressCountBy: 1	"dec progress count after each term" ].
  sys setIndexProgressCountTo: self class statValueForAuditingNscCounts.	"3"
  sys setProgressCountTo: sz.	"Reset progressCount"
  1 to: sz do: [ :j | 
    rootTerm := roots at: j.
    rootTerm offset == 1
      ifTrue: [ 
        countArg := btreeCounts at: j.
        self
          do: [ :obj | rootTerm auditNscCountsFor: obj on: aString count: countArg ] ].
    sys decrementProgressCountBy: 1	"dec progress count after each pass through the NSC" ].
  sys
    setIndexProgressCountTo: self class statValueForAuditingBtreeCounts;
    setProgressCountTo: 0.	"4"
  true ifTrue: [ 
    "bypass this audit step if using GsPathTerm>>auditNscForRootObj:rootObj:using: to get
     better information about 
     BtreePlusNodeAuditor>>pathTermIncorrectNumberOfBtreeEntries:index:offset: failures."
    self _auditBtreeCounts: btreeCounts on: aString]  ]
    ensure: [ 
      sys
        setIndexProgressCountTo: 0;
        setProgressCountTo: 0 ].
  aString isEmpty
    ifTrue: [ 
      participatesInIndex
        ifTrue: [ 
          aString := 'Indexes are OK'.
          participatesInCollectionValuedIndex
            ifTrue: [ 
              aString := aString
                ,
                  ' and the receiver participates in one or more indexes with collection-valued path terms' ] ]
        ifFalse: [ 
          aString := 'No indexes are present'.
          participatesInCollectionValuedIndex
            ifTrue: [ 
              aString := aString
                ,
                  ', but the receiver participates in one or more indexes with collection-valued path terms' ] ] ].
  ^ aString
%

category: 'Indexing Audit'
method: UnorderedCollection
_auditBtreeCounts: btreeCounts on: aString

"Private.  Unsupported method for GemStone Technical Support."

btreeCounts do: [:each |
  (each class == Array) ifTrue: [
      self _auditBtreeCounts: each on: aString.
    ] ifFalse: [
      "must be a 0, or there is a problem with index structure"
      each ~~ 0 ifTrue: [
              aString add: Character lf;
                  add: ' -- The number of entries in a Btree does not match the number of entries in the base collection (extra elements in either the base collection or btree).';
                  add: Character lf
        ].
    ].
].
%

category: 'Indexing Audit'
method: UnorderedCollection
auditIndexes

"Verifies that the index objects are consistent.
 Returns a string that describes any inconsistencies found.

 Since #_fastAuditIndexes provides a significant improvement 
 in speed over tthe older (pre 2.1) algorithms, it is recommended
 that this method be used on a regular basis to ensure index structure
 integrity. 

 If the audit returns errors, the indexes should be dropped and rebuilt 
 and the incident should be reported to Gemstone support for analysis.
" 

^self _fastAuditIndexes
%

category: 'Indexing Audit'
method: UnorderedCollection
_auditIndexes

"Verifies that the index objects are consistent.
 Returns a string that describes any inconsistencies found." 

^self _fastAuditIndexes
%

category: 'Deprecated'
method: UnorderedCollection
_oldAuditIndexes
  "Verifies that the index objects are consistent. _fastAuditIndexes is 
 recommended over _oldAuditIndexes, because it is much, much faster and
 just as accurate.  This method is not supported. "

  self deprecated: 'UnorderedCollection>>_oldAuditIndexes deprecated v3.2, Constraints are no longer supported'.
  ^ self _incrementalAuditIndexes: 1 to: self size
%

category: 'Indexing Audit'
method: UnorderedCollection
_auditIndexDictionary

"Verifies that the index dictionary component of the indexing structure 
 for this collection is consistent.  Returns a string that describes any 
 inconsistencies found.  This method is intended to be used for audit when 
 problems specific to the index dictionary are suspected; for general
 audit on the indexes use _auditIndexes" 

^ self _fastAuditIdentityIndexes
%

category: 'Indexing Audit'
method: UnorderedCollection
_auditIndexBtree

"Verifies that the B-tree component of the indexing structure for 
 this collection is consistent.  Returns a string that describes any 
 inconsistencies found.  This method is intended to be used for audit 
 when problems specific to the btrees are suspected; for general audit 
 on the indexes use _auditIndexes" 

^ self _fastAuditEqualityIndexes
%

category: 'Searching'
method: UnorderedCollection
selectAsStream: aBlock

"Same functionality as select: except that the result is returned as a
 RangeIndexReadStream rather than an IdentitySet. The stream will contain 
 the results sorted in the order specified by the equality index.

 The select block is limited in the following ways:

 * The select block may only contain a single predicate.
 * The predicate must contain one path expression.
 * An equality index must exist for the path expression.

 To use the stream that this method returns most effectively, avoid modifying
 both the receiver of this message and the selected objects returned by the
 stream as long as the stream is being accessed.  Changes that alter the
 equality index can cause stream access failures."

| result |
(aBlock class == SelectBlock)
    ifTrue:[
       "send value: to the block to create the 4 element
        associative access Array; the argument nil is ignored to
        create the Array, but all SelectBlocks have one argument"
        result := (QueryExecuter on: self)
            _boundQuerySelectAsStream: (aBlock queryBlock value: nil).
        ^ result
    ].
^ self _error: #rtErrBagOnlySelectBlockAllowed
%

category: 'Testing'
method: UnorderedCollection
_isLarge

"Returns true if the object is implemented as a tree of private smaller objects"

^ _levels ~~ 0
%

category: 'Testing'
method: UnorderedCollection
_levels

"If the object is implemented as a tree of private smaller objects, returns
 the depth of the tree not including the leaf nodes, otherwise returns 0.

 The result will be a SmallInteger >= 0"

^ _levels 
%

category: 'Indexing Support'
method: UnorderedCollection
_findRangeIndexWithPath: pathArray

"Returns a range equality index whose path components are the same as
 represented by the pathArray (Array of strings)."

| index iList |
iList := self _indexedPaths.
iList ifNil: [ ^ nil ].

" for each index on the NSC ... "
1 to: iList size by: 2 do: [ :i |
  index := iList at: i.
  ( index _isIndexObject and:
    [ index isRangeEqualityIndex and:
     [ index hasIndexOnPath: pathArray ] ] )
       ifTrue: [ ^ index ]
].
^ nil
%

category: 'Parallel Index Creation Support'
method: UnorderedCollection
createParallelIndexes: anArrayOfIndexSpecifications

| indexes |
self _hasIncompleteIndexes ifTrue: [ ^self _error: #rtErrCollectionWithIncompleteIndex ].
self _hasIndexes ifTrue: [ ImproperOperation signal: 'createParallelIndexes: may only be used on a collection with no pre-existing indexes.' ].
[
  indexes := OrderedCollection new.
  IndexManager current executeStartingIndexMaintenance: [
	  anArrayOfIndexSpecifications do: [:spec | | index |
		  (index := self _createParallelIndexOn: spec) 
			  ifNotNil: [ indexes add: (index -> index preIndexCreation) ]
	  ].
	  self _putInWriteSet.
	  self do: [:element | | indexResult |
		  indexResult := self _updateIndexesForAdditionOf: element logging: false.
		  indexResult == true ifFalse: [ self _raiseIndexingError: indexResult ]
	  ].
	  indexes do: [:assoc | assoc key postIndexCreation: assoc value ]
  ].
] onSynchronous: Error do:[:ex | 
	IndexingErrorPreventingCommit resignal: ex objWithIndex: self .
].
%

category: 'Parallel Index Creation Support'
method: UnorderedCollection
_createParallelIndexOn: anIndexSpecification
  | indexManager pathArray index indexes |
  indexManager := IndexManager current.
  pathArray := anIndexSpecification path asArrayOfPathTerms.
  indexes := self _findIndexesWithPath: pathArray.
  indexes
    do: [ :idxObj | 
      " check if an index already exists for the path "
      anIndexSpecification indexType == indexManager _identityIndex
        ifTrue: [ 
          idxObj isIdentityIndex
            ifTrue: [ ^ nil ] ]
        ifFalse: [ 
          idxObj isRangeEqualityIndex
            ifTrue: [ ^ nil ] ] ].
  index := anIndexSpecification _createIndex.
  index _preIndexCreationBuildIndexDictionaryFor: pathArray for: self.
  index nscRoot: self.
  self _getIndexList buildPathTermsFor: index with: pathArray fromSpec: anIndexSpecification.
  indexManager _addIndex: index.
  ^ index
%

category: 'Index Cache Statistics'
classmethod: UnorderedCollection
statValueForAuditingIdentityIndexes
"Answer the value of the IndexProgressCount cache statistic when the
 identity index audit phase of the index audit is in progress."
^ 1
%

category: 'Index Cache Statistics'
classmethod: UnorderedCollection
statValueForAuditingRootTerms
"Answer the value of the IndexProgressCount cache statistic when the
 root term audit phase of the index audit is in progress."
^ 2
%

category: 'Index Cache Statistics'
classmethod: UnorderedCollection
statValueForAuditingNscCounts
"Answer the value of the IndexProgressCount cache statistic when the
 NSC counts audit phase of the index audit is in progress."
^ 3
%

category: 'Index Cache Statistics'
classmethod: UnorderedCollection
statValueForAuditingBtreeCounts
"Answer the value of the IndexProgressCount cache statistic when the
 btree counts audit phase of the index audit is in progress."
^ 4
%

! Request 41117
category: 'Index Cache Statistics'
classmethod: UnorderedCollection
statValueForRemovingAllIndexes
"Answer the value of the IndexProgressCount cache statistic when the
 IndexManager>>removeAllIndexes method is in progress."
^ 5
%

category: 'Repository Conversion'
method: UnorderedCollection
fixRefsAfterConversion

"Default UnorderedCollection method for fixing references
 ObsLargePositiveInteger and ObsLargeNegativeInteger instances that can
 now be represented as a SmallInteger and Floats and SmallFloats
 which can now be represented as a SmallDouble."

| aBagOrSet convBm |
convBm := (GsBitmap newForHiddenSet: #Conversion).
(convBm includes: self)
        ifTrue:[^false]. "already fixed this one"

"Fix inst var refs first"
self fixInstVarRefsAfterConversion.

"now handle the rest of it"
aBagOrSet := self select:[:e| e needsFixingAfterConversion].
1 to: aBagOrSet _basicSize do:[:n| |obj|
        obj := aBagOrSet _at: n.
        self remove: obj.
        obj := obj + 0.
        self add: obj.
].
convBm add: self.
^true
%

