!=========================================================================
! Copyright (C) GemTalk Systems 2017-2020.  All Rights Reserved.
!
! $Id: Btree-Index-Core.gs 38384 2016-01-08 18:22:36Z lalmarod $
!
! Btree-Index-Core.gs  -  source code for the gs index classes
!
!========================================================================

! class created in btreeplusclasses.gs

! Class Implementation for GsAbstractIndex

! Remove existing behavior from GsAbstractIndex
removeallmethods GsAbstractIndex
removeallclassmethods GsAbstractIndex

! ------------------- Instance methods for GsAbstractIndex

category: 'Testing'
method: GsAbstractIndex
_canCompare: aKey withClass: aClass
  "Returns whether the receiver can make B-tree comparisons with the given key."


  ^ self btreeRoot _canCompare: aKey withClass: aClass
%

category: 'Testing'
method: GsAbstractIndex
_checkSameIndexOptionsAs: anIndexSpec
  ^ self termsRequired == anIndexSpec requirePathTerms
%

category: 'Testing'
method: GsAbstractIndex
_checkSameLastElementClassAs: indexObj
  ^ indexObj _checkSameLastElementClassAsEqualityIndex: self
%

category: 'Updating'
method: GsAbstractIndex
_clear
  "Assigns nil to important instance variables."

  btreeRoot := nil.
  nscRoot := nil.
  self size: 0
%

category: 'Searching'
method: GsAbstractIndex
_findFirstUnsharedPathTerm
  "Returns the first path term that is not shared by any other indexes.  This is
 the first path term whose only index is the receiver - which should always be the last term. "

  | pathTerm sz |
  sz := self size.
  1 to: sz do: [ :i | 
    pathTerm := self at: i.
    (pathTerm size == 1 and: [ self == (pathTerm at: 1) ])
      ifTrue: [ ^ pathTerm ] ].
  self error: 'no unshared path term found' "should never get here"
%

category: 'Support'
method: GsAbstractIndex
_findSetValuedPathTermOffset
  "Returns the offset of the set-valued path term in the receiver.  If none
 exists, returns 0."

  1 to: self size do: [ :i | 
    (self at: i) indicatesMultiValue
      ifTrue: [ ^ i ] ].
  ^ 0
%

category: 'Testing'
method: GsAbstractIndex
_isIndexObject

"Returns true if the receiver is an index object; returns false otherwise."

^ true
%

category: 'Index Creation'
method: GsAbstractIndex
_preIndexCreationBuildIndexDictionaryFor: pathArray for: anNsc
  "afford an opportunity to resize dictionary before adding new elements during index creation"

  "noop"

%

category: 'Index Creation'
method: GsAbstractIndex
_preIndexCreationSort
  | btreeRootNode |
  true
    ifTrue: [ 
      "skip pre-sort for now"
      ^ self btreeRoot ].
  btreeRootNode := (PathSorter on: {self} directions: #(true))
    lastNodeOffset: 1;
    yourself.
  ^ self btreeRoot
    btreeRootNode: btreeRootNode;
    yourself
%

category: 'Removing'
method: GsAbstractIndex
_removeBtreeEntriesForKey: aKey

"Removes all entries in the B-tree that have a key identical to aKey.
 Returns an Array of values corresponding to each entry that was removed."

| stream vals |
" first we need to find all values that have aKey as the key "
stream := self asQueryEvaluator _findAllValuesGreaterThanKey: aKey andEquals: true.
vals := { } .

[ stream _btreeAtEnd not and:
[ stream _peekKey _idxForSortEqualTo: aKey ] ] whileTrue: [
    aKey == stream _peekKey
        ifTrue: [ | val root |
          "remove the entry for the value"
          val := stream _peekValue.
          root := stream _peekRoot.
          vals add: {val. root} ].
    stream _btreeNext 
].
" now remove the entry for each value "
vals do: [ :ar |
  (self btreeRemoveKey: aKey value: (ar at: 1) root: (ar at: 2))
].

^ vals
%

category: 'Updating'
method: GsAbstractIndex
_setPathTermState
  "For the last path term, indicate the need to update the B-tree.
   Indicate if the last object along the path needs a dependency list."

  | lastPathTerm |
  lastPathTerm := self at: self size.
  lastPathTerm updateBtree: self
%

category: 'Testing'
method: GsAbstractIndex
_validateCanCompareWith: aKey
  "noop"

%

category: 'Updating Indexes'
method: GsAbstractIndex
addDirectMappingsFor: aBag indexList: iList
  "Add an entry for an index on the elements of the NSC."

  | anObject firstPathTerm indexMgr |
  indexMgr := self indexManager.
  firstPathTerm := self firstPathTerm.
  indexMgr autoCommit
    ifTrue: [ nscRoot _lockForIndexCreation ].
  self lastPathTerm needsDepList
    ifTrue: [ 1 to: aBag size do: [ :i | " Add an entry for an index on the elements of the NSC. "
        nil ~~ (anObject := aBag _at: i)
          ifTrue: [ " add an entry in the dependency list "
            anObject
              getDepListAndAddLastElementPathTerm: firstPathTerm
              logging: false ].
        self btreeAt: anObject put: anObject root: anObject.
        i \\ 100 == 0
          ifTrue: [ indexMgr commitIndexMaintenance: self at: i ] ] ]
    ifFalse: [ 1 to: aBag size do: [ :i | anObject := aBag _at: i.
        self btreeAt: anObject put: anObject root: anObject.
        i \\ 100 == 0
          ifTrue: [ indexMgr commitIndexMaintenance: self at: i ] ] ]
%

category: 'Converting'
method: GsAbstractIndex
asIndexSpecification
  ^ self subclassResponsibility: #'asIndexSpecification'
%

category: 'Support'
method: GsAbstractIndex
asPathEvaluator
  "Returns a PathEvaluator over the same path as the receiver."

  | pathEval holder |
  pathEval := PathEvaluator new.
  1 to: self size do: [ :i | pathEval add: (self at: i) name ].
  pathEval nsc: nscRoot.
  holder := { pathEval } .
  pathEval := nil .
  ^ PathEvaluator asMostSpecificType: holder .
%

category: 'Updating'
method: GsAbstractIndex
btreeAt: aKey put: aValue root: rootObject
  "Insert the key/value pair into the root B-tree node, validate that aKey can be stored in btree."

  self _validateCanCompareWith: aKey.	"signal error if comparison not valid"
  btreeRoot btreeAt: aKey put: aValue root: rootObject
%

category: 'Stream Accessing'
method: GsAbstractIndex
btreeComparisonQuerySpec
  ^ BtreePlusComparisonQuerySpec new
    rangeIndex: self;
    yourself
%

category: 'Btree Accessing'
method: GsAbstractIndex
btreePlusLeafNodeClass
  ^ self isIndexOnRootNsc
      ifTrue: [ BtreePlusLeafKeyNode ]
      ifFalse: [ self isIndexOnFirstTerm
          ifTrue: [ BtreePlusLeafKeyValueNode ]
          ifFalse: [ BtreePlusLeafKeyValueRootObjectNode ] ]
%

category: 'Stream Accessing'
method: GsAbstractIndex
btreeRangeComparisonQuerySpec
  ^ BtreePlusRangeComparisonQuerySpec new
%

category: 'Stream Accessing'
method: GsAbstractIndex
btreeReadStreamClass
  "Returns the class of btree read stream to create for query results."

  ^ BtreePlusReadStream
%

category: 'Updating'
method: GsAbstractIndex
btreeRemoveKey: aKey value: aValue root: rootObject
  "Removes the key and value from the B-tree root node.  Must check to see if
 a merge occurred in the root such that the root only contains a single
 entry.  If so, make the single entry the new root.  Returns whether the
 removal occurred."

  ^ btreeRoot removeKey: aKey value: aValue root: rootObject
%

category: 'Accessing'
method: GsAbstractIndex
btreeRoot
  ^ btreeRoot
    ifNil: [ 
      btreeRoot := self options reducedConflict
        ifTrue: [ RcBtreePlusRoot new ]
        ifFalse: [ BtreePlusRoot new ].
      btreeRoot
        indexObject: self;
        yourself ]
%

category: 'Accessing'
method: GsAbstractIndex
collator
  "No IcuCollator associated with this index."

  ^ nil
%

category: 'Comparison Operators'
method: GsAbstractIndex
comparisonForCompare
  self subclassResponsibility: #comparisonForCompare
%

category: 'Comparison Operators'
method: GsAbstractIndex
comparisonForSort
  "called during creation of receiver, so initialize unset instance variables"

  self subclassResponsibility: #comparisonForSort
%

category: 'Traversing'
method: GsAbstractIndex
cumulativeFactorFor: anObject upTo: aPathTerm startingAt: anOffset 
  "Traverse the sub-objects of the given object using the path terms of the
 receiver up to the given path term. Return an Array of objects. The index 
 may be a collection based index."

  | nextObj sz i thePathTerm theResult factor |
  nil == anObject
    ifTrue: [ ^ 0 ].
  nextObj := anObject.
  sz := self size.
  i := anOffset.
  theResult := { nextObj }.
  factor := 0.
  [ thePathTerm := self at: i.
  thePathTerm ~~ aPathTerm and: [ i <= sz ] ]
    whileTrue: [
      theResult do: [ :nextTraversalObj | 
        | enumeratedValues terms |
        thePathTerm indicatesMultiValue
          ifTrue: [
            theResult := {}.
            enumeratedValues := IdentityBag new.
terms := 0.
            thePathTerm
              nextObj: nextTraversalObj
              do: [ :obj :ivOffset |
                  enumeratedValues add: obj. 
                  factor := factor + (self
                    cumulativeFactorFor: obj
                    upTo: aPathTerm
                    startingAt: i + 1).
                  terms := terms + 1.
                 ].
            factor := factor + (enumeratedValues size / terms).
            ^ factor ]
          ifFalse: [  
            nextObj := thePathTerm _nextObjectFor: nextTraversalObj.
            nil == nextObj
              ifTrue: [ ^ factor  ].
            theResult := { nextObj } ] ].
      i := i + 1 ].
  ^ factor
%

category: 'Accessing'
method: GsAbstractIndex
firstPathTerm

"Returns the first path term in the path components list."

^ self at: 1
%

category: 'Testing'
method: GsAbstractIndex
hasCollectionBasedTerm
  "Returns true if the path has an enumerated path term or a term that indicates a 
   set-valued instance variable."

  ^ self hasEnumeratedTerm or: [ self hasSetValuedTerm ]
%

category: 'Testing'
method: GsAbstractIndex
hasEnumeratedTerm
  "Returns true if the path has a term that indicates an enumerated path term."

  1 to: self size do: [ :i | 
    (self at: i) isEnumeratedTerm
      ifTrue: [ ^ true ] ].
  ^ false
%

category: 'Testing'
method: GsAbstractIndex
hasIndexDictionary
  "Answer if receiver has an indexDictionary."

  "indexDictionary not used by any of receiver's subclasses"

  ^ false
%

category: 'Testing'
method: GsAbstractIndex
hasIndexOnPath: pathArray
  "Returns whether the receiver's path components match the path represented by
 pathArray."

  self size == pathArray size
    ifFalse: [ ^ false ].
  1 to: self size do: [ :i | 
    ((self at: i) hasMatchingTerm: (pathArray at: i))
      ifFalse: [ ^ false ] ].
  ^ isComplete
%

category: 'Testing'
method: GsAbstractIndex
hasSetValuedTerm
  "Returns true if the receiver has any path term that indicates a set-valued
 instance variable."

  1 to: self size do: [ :i | 
    (self at: i) isSetValuedTerm
      ifTrue: [ ^ true ] ].
  ^ false
%

category: 'Accessing'
method: GsAbstractIndex
indexDictionary

"GsAbstractIndexes do not have an indexDictionary."

^nil
%

category: 'Accessing'
method: GsAbstractIndex
indexManager

^IndexManager current
%

category: 'Accessing'
method: GsAbstractIndex
indexType
  ^ self subclassResponsibility: #'indexType'
%

category: 'Testing'
method: GsAbstractIndex
isComplete

"Returns the value of the instance variable 'isComplete'."

isComplete == nil
  ifTrue: [ ^ true ].
^ isComplete
%

category: 'Testing'
method: GsAbstractIndex
isIdentityIndex

"Returns false, subclasses will override as needed."

^ false
%

category: 'Testing'
method: GsAbstractIndex
isIndexOnFirstTerm

"Returns whether the receiver is an index on the first and only path term and is not on 
 a root element of the nsc."

^ self lastPathTerm indicatesIndexOnRootNsc 
  ifTrue: [ ^false ]
  ifFalse:[ self size == 1 ]
%

category: 'Testing'
method: GsAbstractIndex
isIndexOnNscElements

"Returns whether the receiver is an index directly on the elements of an NSC
 (either the root NSC or a set-valued instance variable on the path)."

^ self lastPathTerm indicatesIndexOnNscElements
%

category: 'Testing'
method: GsAbstractIndex
isIndexOnRootNsc

"Returns whether the receiver is an index directly on the elements of
 the root NSC."

^ self lastPathTerm indicatesIndexOnRootNsc
%

category: 'Testing'
method: GsAbstractIndex
isRangeEqualityIndex

"Returns false, subclasses will override as needed."

^ false
%

category: 'Testing'
method: GsAbstractIndex
isStreamable

"Returns true, subclasses will override as needed."

^ true
%

category: 'Accessing'
method: GsAbstractIndex
lastPathTerm

"Returns the last path term in the path components list."

^ self at: self size
%

category: 'Modification Tracking Support'
method: GsAbstractIndex
modifiedObject: aKey userData: valArray
  "Notification that modification of aKey is completed and that the receiver 
   should update the btree using value and root from valArray."

  ^ self btreeAt: aKey put: (valArray at: 1) root: (valArray at: 2)
%

category: 'Accessing'
method: GsAbstractIndex
nscRoot

"Returns the value of the instance variable 'nscRoot'."

^nscRoot
%

category: 'Updating'
method: GsAbstractIndex
nscRoot: newValue

"Modify the value of the instance variable 'nscRoot'."

nscRoot := newValue.
"force lazy initialization of comparisonForSort iv to avoid commit conflicts later... make it invariant, for good measure."
self comparisonForSort immediateInvariant
%

category: 'Testing'
method: GsAbstractIndex
optimizingComparison
  "Answer true if comparison operations can be optimized in Btree operations and elsewhere. 
   Controlled by optimizedComparison index option."

  ^ self options optimizedComparison
%

category: 'Accessing'
method: GsAbstractIndex
options
  "index options include: optionalPathTerms, reducedConflict, legacyIndex"

  ^ options
%

category: 'Accessing'
method: GsAbstractIndex
options: aGsIndexOptions
  "make copy to perserve integrity of the incoming options object"

  options := aGsIndexOptions copy _reify immediateInvariant
%

category: 'Accessing'
method: GsAbstractIndex
pathComponentsString

"Returns a string of the path components."

^ self pathComponentsStringStartingAt: 1
%

category: 'Accessing'
method: GsAbstractIndex
pathComponentsStringStartingAt: offset

"Returns a string of the path components."

| str sz |

sz := self size.
str := String new.
offset to: sz do: [ :i |
    str addAll: (self at: i) name.
    i == sz
        ifFalse: [ str add: $.  ]
].
^ str
%

category: 'Updating Indexes'
method: GsAbstractIndex
postIndexCreation: originalBtreeRoot
  btreeRoot btreeRootNode isBtreePlusNode
    ifFalse: [ 
      | leafNodes pathSorter |
      "Use a merge sort to create a complete B-tree from a PathSorter
       stored in the btreeRoot instance variable."
      leafNodes := {(originalBtreeRoot btreeRootNode)}.	" create an Array containing an empty B-tree node "
      pathSorter := btreeRoot btreeRootNode.
      btreeRoot := {leafNodes.
      pathSorter}.	"per bug 36147, we need the leafNodes array to be persistent to avoid some out-of-memory conditions during index creation"
      pathSorter sortIntoBtreeNodes: leafNodes.	" produce an Array containing many B-tree leaf nodes, all sorted "
      btreeRoot btreeRootNode: (pathSorter createInteriorNodesFor: leafNodes)	" now build up interior nodes for the leaves " ].
  isComplete := true.
  self indexManager autoCommit
    ifTrue: [ | systm |
      systm := System.
      systm commitTransaction
        ifFalse: [ nscRoot _errorCouldNotCommitDuringIndexCreation ].
      systm transactionMode == #'manualBegin'
        ifTrue: [ systm beginTransaction ] ]
%

category: 'Index Creation'
method: GsAbstractIndex
preIndexCreation
  "record that index creation has started and "

  isComplete := false.
  progress := 0.
  nscRoot size > self btreeRoot maxNumberOfElements
    ifTrue: [ ^ self _preIndexCreationSort ].
  ^ btreeRoot
%

category: 'Updating'
method: GsAbstractIndex
progress: newValue

"Modifies the value of the instance variable 'progress'."

progress := newValue
%

category: 'Stream Accessing'
method: GsAbstractIndex
readStreamClass
  "Returns the class of read stream to create for query results."

  ^ BtreePlusGsIndexReadStream
%

category: 'Accessing'
method: GsAbstractIndex
sizeForNscBuilder

"Returns the size to be used when an NscBuilder is collecting objects."

^ nscRoot size
%

category: 'Testing'
method: GsAbstractIndex
termsRequired
  "Returns true if the receiver has any path term that has termsRequired set."

  1 to: self size do: [ :i | 
    (self at: i) termsRequired
      ifTrue: [ ^ true ] ].
  ^ false
%

category: 'Traversing'
method: GsAbstractIndex
traverse: anObject upTo: aPathTerm
  "Traverse the sub-objects of the given object using the path terms of the
 receiver up to the given path term.  This method assumes that the index is not
 over a set-valued instance variable.  For that kind of index, use the
 traverse:upTo:startingAt:into: method."

  | nextObj sz i |
  nil == anObject
    ifTrue: [ ^ anObject ].
  nextObj := anObject.
  sz := self size.
  i := 1.
  [ (self at: i) ~~ aPathTerm and: [ i <= sz ] ]
    whileTrue: [ 
      nextObj := (self at: i) _getNextObjectForTraversal: nextObj.
      nil == nextObj
        ifTrue: [ ^ nextObj ].
      i := i + 1 ].
  ^ nextObj
%

category: 'Traversing'
method: GsAbstractIndex
traverseAllWithParents: anObject upTo: aPathTerm startingAt: anOffset
  "Traverse the sub-objects of the given object using the path terms of the
 receiver up to the given path term. Return an Array of objects. The index 
 may be a collection based index."

 ^ self traverseAllWithParents: anObject parent: nil upTo: aPathTerm startingAt: anOffset
%

! Class Implementation for GsIdentityIndex

! Remove existing behavior from GsIdentityIndex
removeallmethods GsIdentityIndex
removeallclassmethods GsIdentityIndex

! ------------------- Instance methods for GsIdentityIndex

category: 'Testing'
method: GsIdentityIndex
_canCompare: aKey withClass: aClass
  "Returns whether the receiver can make B-tree comparisons with the given key."
  ^ true
%

category: 'Testing'
method: GsIdentityIndex
_canCompareWith: aKey
  "Returns whether the receiver can make B-tree comparisons with the given key."

  ^ true
%

category: 'Testing'
method: GsIdentityIndex
_checkSameLastElementClassAsEqualityIndex: equalityIndexObj
  ^ false
%

category: 'Converting'
method: GsIdentityIndex
asIndexSpecification
  ^ (IdentityIndexSpecification path: self pathComponentsString)
    options: self options;
    requirePathTerms: self termsRequired;
    yourself
%

category: 'Converting'
method: GsIdentityIndex
asQueryEvaluator
  ^ GsIdentityIndexQueryEvaluator on: self
%

category: 'Comparison Operators'
method: GsIdentityIndex
comparisonForCompare
  ^ BtreeComparisonForIdentity new
%

category: 'Comparison Operators'
method: GsIdentityIndex
comparisonForSort
  "called during creation of receiver, so initialize unset instance variables"

  ^ comparisonForSort ifNil: [ comparisonForSort := BtreeComparisonForIdentity new ]
%

category: 'Accessing'
method: GsIdentityIndex
indexType
  ^ #'identity'
%

category: 'Testing'
method: GsIdentityIndex
isIdentityIndex

"Returns true."

^ true
%

! Class Implementation for GsRangeEqualityIndex

! Remove existing behavior from GsRangeEqualityIndex
removeallmethods GsRangeEqualityIndex
removeallclassmethods GsRangeEqualityIndex

! ------------------- Class methods for GsRangeEqualityIndex

category: 'Constants'
classmethod: GsRangeEqualityIndex
_isUnicodeLastElementClass: aClass
  "returns true if <aClass> is a Unicode class"

  self _validateLastElementClass: aClass.
  Unicode16 usingUnicodeCompares
    ifTrue: [ 
      (self isBasicClass: aClass)
        ifTrue: [ 
          (aClass _subclassOf: CharacterCollection)
            ifTrue: [ ^ true ] ] ]
    ifFalse: [ 
      (aClass _subclassOf: Unicode7)
        ifTrue: [ ^ true ].
      (aClass _subclassOf: Unicode16)
        ifTrue: [ ^ true ].
      (aClass _subclassOf: Unicode32)
        ifTrue: [ ^ true ] ].
  ^ false
%

category: 'Private'
classmethod: GsRangeEqualityIndex
_validateLastElementClass: aClass
  aClass isBehavior
    ifFalse: [ 
      ArgumentTypeError
        signal: 'LastElementClass (' , aClass printString , ') must be a class' ]
%

category: 'Constants'
classmethod: GsRangeEqualityIndex
isBasicClass: aClass
  "Returns whether the given class should use BtreePlusBasic*Nodes to store
 B-tree mappings."

  | characterCollectionBasicClasses |
  aClass == nil
    ifTrue: [ ^ false ].
  aClass isSpecial
    ifTrue: [ "specials do not benefit from basic btree nodes"
      ^ false ].
  self _validateLastElementClass: aClass.
  Unicode16 usingUnicodeCompares
    ifTrue: [ characterCollectionBasicClasses := #().
      (aClass _subclassOf: CharacterCollection)
        ifTrue: [ aClass == CharacterCollection
            ifTrue: [ ^ true ].
          (aClass _subclassOf: String)
            ifTrue: [ ^ true ].
          (aClass _subclassOf: MultiByteString)
            ifTrue: [ ^ true ] ] ]
    ifFalse: [ characterCollectionBasicClasses := {String.
      DoubleByteString.
      QuadByteString.
      Unicode7.
      Unicode16.
      Unicode32} ].
  characterCollectionBasicClasses
    , {Number.	"all non-special subclasses of Number"
      DateTime.
      DateAndTime.
      Date.
      Time} do: [ :basicClass | (aClass _subclassOf: basicClass)
        ifTrue: [ ^ true ] ].
  ^ false
%

category: 'Instance Creation'
classmethod: GsRangeEqualityIndex
newWithLastElementClass: aClass
  "Create a new instance and initialize its B-tree root."

  | newOne |
  newOne := super new.
  newOne lastElementClass: aClass.
  ^ newOne
%

! ------------------- Instance methods for GsRangeEqualityIndex

category: 'Testing'
method: GsRangeEqualityIndex
_canCompareWith: aKey
  "Returns whether the receiver can make B-tree comparisons with the given key."

  ^ btreeRoot _canCompare: aKey withClass: lastElementClass
%

category: 'Testing'
method: GsRangeEqualityIndex
_checkSameLastElementClassAsEqualityIndex: equalityIndexObj
  ^ equalityIndexObj lastElementClass == self lastElementClass
%

category: 'Updating'
method: GsRangeEqualityIndex
_clear
  "Assigns nil to important instance variables."

  lastElementClass := nil.
  super _clear
%

category: 'Updating'
method: GsRangeEqualityIndex
_setLastPathTermState
  "Indicate if the last object along the path needs a dependency list."

  | lastPathTerm |
  lastPathTerm := self at: self size.
  lastElementClass allSubclasses , {lastElementClass}
    do: [ :cl | cl instancesInvariant not
        ifTrue: [ "bug 42640 - depList needed unless instances invariant"
          lastPathTerm needsDepList: true.
          ^ self ] ]
%

category: 'Updating'
method: GsRangeEqualityIndex
_setPathTermState
  "For the last path term, indicate the need to update the B-tree.
   Indicate if the last object along the path needs a dependency list."

  super _setPathTermState.
  self _setLastPathTermState
%

category: 'Testing'
method: GsRangeEqualityIndex
_validateCanCompareWith: aKey
  (self _canCompareWith: aKey)
    ifFalse: [ ^ (ImproperOperation new
        _number:
            (ErrorSymbols at: #'rtErrRangeEqualityIndexInvalidClassKindForBtree');
        args: { aKey class. self lastPathTerm name. self lastElementClassDescription }) signal ]
%

category: 'Private'
method: GsRangeEqualityIndex
_validateLastElementClass: aClass
  self class _validateLastElementClass: aClass
%

category: 'Converting'
method: GsRangeEqualityIndex
asIndexSpecification
  (self class _isUnicodeLastElementClass: self lastElementClass)
    ifTrue: [ ^ self asUnicodeIndexSpecification ].
  ^ (EqualityIndexSpecification
    path: self pathComponentsString
    lastElementClass: self lastElementClass)
    options: self options;
    yourself
%

category: 'Converting'
method: GsRangeEqualityIndex
asQueryEvaluator

  ^GsEqualityIndexQueryEvaluator on: self
%

category: 'Converting'
method: GsRangeEqualityIndex
asUnicodeIndexSpecification
  ^ (UnicodeIndexSpecification
    path: self pathComponentsString
    collator: self collator)
    options: self options;
    yourself
%

category: 'Stream Accessing'
method: GsRangeEqualityIndex
btreeRangeComparisonQuerySpec
  ^ BtreePlusRangeComparisonQuerySpec new
    rangeIndex: self;
    yourself
%

category: 'Comparison Operators'
method: GsRangeEqualityIndex
comparisonForCompare
  ^ self optimizingComparison
    ifTrue: [ BtreeOptimizedComparison new ]
    ifFalse: [ BtreeComparisonForCompare newForComparison ]
%

category: 'Comparison Operators'
method: GsRangeEqualityIndex
comparisonForSort
  "called during creation of receiver, so initialize unset instance variables"

  ^ comparisonForSort ifNil: [ 
      comparisonForSort := self optimizingComparison
        ifTrue: [ BtreeOptimizedComparison new ]
        ifFalse: [ BtreeComparisonForCompare newForSort ] ]
%

category: 'Accessing'
method: GsRangeEqualityIndex
indexType
  ^ #'equality'
%

category: 'Testing'
method: GsRangeEqualityIndex
isRangeEqualityIndex

"Returns true."

^ true
%

category: 'Accessing'
method: GsRangeEqualityIndex
lastElementClass

"Returns the value of the instance variable 'lastElementClass'."

^lastElementClass
%

category: 'Updating'
method: GsRangeEqualityIndex
lastElementClass: aClass
  self _validateLastElementClass: aClass.
  lastElementClass := aClass
%

category: 'Accessing'
method: GsRangeEqualityIndex
lastElementClassDescription
  "answer a description of the lastElementClass of the receiver, 
   suitable for use in an error message"

  | cl qualifier |
  qualifier := ''.
  self optimizingComparison ifTrue: [ qualifier := ' [ with optimized comparison ]'. ].
  cl := self lastElementClass.
  (cl isSubclassOf: CharacterCollection)
    ifTrue: [ 
      Unicode16 usingUnicodeCompares
        ifTrue: [ ^ 'CharacterCollection', qualifier ]
        ifFalse: [ ^ 'CharacterCollection excluding Unicode String classes', qualifier ] ].
  ^ cl name asString, qualifier
%

category: 'Updating'
method: GsRangeEqualityIndex
objectSecurityPolicy: anObjectSecurityPolicy
  "Assign the receiver and its B-tree to the given security policy."

  super objectSecurityPolicy: anObjectSecurityPolicy.
  self btreeRoot objectSecurityPolicy: anObjectSecurityPolicy
%

category: 'Testing'
method: GsRangeEqualityIndex
optimizingComparison
  "if option is set, the ensure that the lastElementClass support optimized comparisons"

  ^optimizingComparison ifNil: [
    optimizingComparison := super optimizingComparison
      ifTrue: [ self lastElementClass _idxCanOptimizeComparison ]
      ifFalse: [ false ] ].
%

category: 'Accessing'
method: GsRangeEqualityIndex
sortNodeClass
  "Returns the class of SortNode to use."

  ^ self lastElementClass sortNodeClass
%

! Class Implementation for GsUnicodeRangeEqualityIndex

! Remove existing behavior from GsUnicodeRangeEqualityIndex
removeallmethods GsUnicodeRangeEqualityIndex
removeallclassmethods GsUnicodeRangeEqualityIndex

! ------------------- Class methods for GsUnicodeRangeEqualityIndex

category: 'instance creation'
classmethod: GsUnicodeRangeEqualityIndex
newWithCollator: anIcuCollator
  "Create a new instance and initialize its B-tree root."

  | newOne |
  newOne := super new.
  newOne collator: anIcuCollator.
  ^ newOne
%

category: 'instance creation'
classmethod: GsUnicodeRangeEqualityIndex
newWithLastElementClass: aClass
  self shouldNotImplement: #'newWithLastElementClass:'
%

! ------------------- Instance methods for GsUnicodeRangeEqualityIndex

category: 'Testing'
method: GsUnicodeRangeEqualityIndex
_canCompareWith: aKey
  "Returns whether the receiver can make B-tree comparisons with the given key."

  self constraintType == #unicodeString
    ifTrue: [ ^ super _canCompareWith: aKey ].
  ^ btreeRoot _canCompare: aKey withClass: Unicode7
%

category: 'Index Creation'
method: GsUnicodeRangeEqualityIndex
_preIndexCreationSort
  | btreeRootNode |
  true
    ifTrue: [ 
      "skip pre-sort for now"
      ^ self btreeRoot ].
  btreeRootNode := (PathSorter
    on: {self}
    directions: #(true)
    collator: self collator)
    lastNodeOffset: 1;
    yourself.
  ^ self btreeRoot
    btreeRootNode: btreeRootNode;
    collator: self collator;
    yourself
%

category: 'Stream Accessing'
method: GsUnicodeRangeEqualityIndex
btreeComparisonQuerySpec
  ^ BtreePlusUnicodeComparisonQuerySpec new
    collator: self collator;
    rangeIndex: self;
    yourself
%

category: 'Stream Accessing'
method: GsUnicodeRangeEqualityIndex
btreeRangeComparisonQuerySpec
  ^ BtreePlusUnicodeRangeComparisonQuerySpec new 
    rangeIndex: self;
    collator: self collator;
    yourself
%

category: 'Accessing'
method: GsUnicodeRangeEqualityIndex
collator
  "Returns IcuCollator to be used when comparing Unicode strings"

  ^ collator ifNil: [ collator := IcuCollator default copy immediateInvariant ]
%

category: 'Accessing'
method: GsUnicodeRangeEqualityIndex
collator: anIcuCollator
  "Set the receiver's collator. Use a copy of anIcuCollator to disallow 
   changes to strength, etc, that might affect the sort ordering. "

  collator := anIcuCollator copy immediateInvariant
%

category: 'Accessing'
method: GsUnicodeRangeEqualityIndex
lastElementClassDescription
  "answer a description of the lastElementClass of the receiver, 
   suitable for use in an error message"

  | qualifier |
  qualifier := ''.
  self optimizingComparison ifTrue: [ qualifier := ' [ with optimized comparison ]'. ].
  ^ 'CharacterCollection including Unicode String classes', qualifier
%

category: 'Testing'
method: GsUnicodeRangeEqualityIndex
optimizingComparison
  "Unicode classes can be optimized, just check that option has been specified"

  ^ optimizingComparison ifNil: [ optimizingComparison := self options optimizedComparison ]
%

! Class Extensions

! Class initializers 

doit
true.
%

category: 'Sorting'
method: GsRangeEqualityIndex
_getSecondarySorterOn: indexObjs directions: theBooleans

"Returns a PathSorter if one is needed to perform secondary sorting;
 otherwise returns nil."

| directions traversers aSize |

(aSize := indexObjs size) > 1 ifTrue: [
    directions := theBooleans copyFrom: 2 to: aSize .
    traversers := indexObjs copyFrom: 2 to: aSize .
    ^ PathSorter on: traversers directions: directions.
].
^ nil  .
%
category: 'Sorting'
method: GsRangeEqualityIndex
_sortAscendingOn: indexObjs directions: theBooleans

""

| array cnt marker sorter prevKey pArr |

" create the Array to hold the objects "
array := Array new: nscRoot size.

" build a sorter to be used for secondary sorts "
sorter := self _getSecondarySorterOn: indexObjs directions: theBooleans.

" start the insertion into the Array at the beginning "
self isIndexOnRootNsc
    ifTrue: [ cnt := 0 ]
    ifFalse: [ cnt := nscRoot _idxOccurrencesOf: nil ].

" gather objects with nil along the path "
pArr := self _getObjectsWithNilOnPath.
1 to: pArr size do: [ :i |
    cnt := cnt + 1.
    array at: cnt put: (pArr at: i)
].
pArr := nil .

" if no secondary sorts "
sorter == nil ifTrue: [
    btreeRoot _putAscendingRootObjectsInto: array startingAt: cnt + 1.
    ^ array
  ].

prevKey := #_incompletePathTraversal.
marker := cnt max: 1.

" Iterate over the key/value/roots in the B-tree leaf nodes "
btreeRoot _leafKeysValuesAndRootsDo: [ :key :value :root |

   cnt := cnt + 1.
   array at: cnt put: root.

    " see if finished a run of duplicate keys "
    key ~= prevKey
      ifTrue: [
        marker < (cnt - 1)
          ifTrue: [ " sort on secondary keys "
            marker to: cnt - 1 do: [ :i | sorter addObject: (array at: i) ].
            sorter sortInto: array startingAt: marker.
            sorter clear
          ].
        marker := cnt
      ].
    prevKey := key.
].
" see if ending with duplicate keys "
( sorter ~~ nil and: [ marker < cnt ] )
    ifTrue: [ " sort on secondary keys "
       marker to: cnt do: [ :i | sorter addObject: (array at: i) ].
       sorter sortInto: array startingAt: marker.
       sorter clear
    ].

^ array
%
category: 'Sorting'
method: GsRangeEqualityIndex
_sortDescendingOn: indexObjs directions: theBooleans

""

| array cnt prevKey marker sorter pArr |

" create the Array to hold the objects "
array := Array new: nscRoot size.

" build a sorter to be used for secondary sorts "
sorter := self _getSecondarySorterOn: indexObjs directions: theBooleans.

" start the insertion into the Array at the beginning "
self isIndexOnRootNsc
    ifTrue: [ cnt := array size  ]
    ifFalse: [ cnt := array size - (nscRoot _idxOccurrencesOf: nil) ].

" gather objects with nil along the path "
pArr := self _getObjectsWithNilOnPath.
1 to: pArr size do: [ :i |
    array at: cnt put: (pArr at: i).
    cnt := cnt - 1.
].
pArr := nil .

" if no secondary sorts "
sorter == nil ifTrue: [
    btreeRoot _putDescendingRootObjectsInto: array startingAt: cnt.
    ^ array
  ].

prevKey := #_incompletePathTraversal.
marker := cnt.

" iterate over the key/values in the B-tree leaf nodes "
btreeRoot _leafKeysValuesAndRootsDo: [ :key :value :root |

      array at: cnt put: root.
      cnt := cnt - 1.

    " see if finished a run of duplicate keys "
    key ~= prevKey
      ifTrue: [
        marker > (cnt + 1)
          ifTrue: [ " sort on secondary keys "
            cnt + 2 to: marker do: [ :i | sorter addObject: (array at: i) ].
            sorter sortInto: array startingAt: cnt + 2.
            sorter clear
          ].
        marker := cnt + 1
      ].
    prevKey := key.
].

" see if ending with duplicate keys "
( sorter ~~ nil and: [ marker > 1 ] )
  ifTrue: [ " sort on secondary keys "
   1 to: marker do: [ :i | sorter addObject: (array at: i) ].
   sorter sortInto: array startingAt: 1.
     sorter clear
    ].
^ array
%
category: 'Sorting'
method: GsRangeEqualityIndex
_sortOn: indexObjs directions: theBooleans persistentRoot: persistentArrayOrNil

"Returns an Array of objects from the root NSC that is sorted according to the
 paths traversed by indexObjs (with the direction of the sort specified in
 theBooleans)."

(theBooleans at: 1)
    ifTrue: [ ^ self _sortAscendingOn: indexObjs directions: theBooleans ]
    ifFalse: [ ^ self _sortDescendingOn: indexObjs directions: theBooleans ]
%
category: 'Traversing'
method: GsAbstractIndex
_traverseObject: anObject incompletesInto: incompleteArray incomplete: incomplete
  "Traverse the sub-objects of the given object using the path terms of the
 receiver.  This method assumes that the index is not over a set-valued
 instance variable.  For that kind of index, use the traverse:startingAt:into:
 method.  If a nil value is reached before the end of the path, place the object
 into the appropriate Array in the incompleteArray of Arrays."

  | nextObj sz pathTerm |
  nextObj := anObject.
  sz := self size.
  1 to: sz do: [ :i | 
    pathTerm := self at: i.
    nextObj := pathTerm _getNextObjectForTraversal: nextObj.
    (nil == nextObj and: [ i ~~ sz ])
      ifTrue: [ 
        incompleteArray == nil
          ifTrue: [ ^ nil ].
        (incompleteArray at: i + 1) add: anObject.
        ^ incomplete ] ].
  ^ nextObj
%
category: 'Testing'
method: GsAbstractIndex
hasIndexOnPath: pathArray startingAt: offset

"Returns whether the receiver's path components match the path represented by
 pathArray."

self size - offset + 1 == pathArray size
  ifFalse: [ ^ false ].

offset to: self size do: [ :i |
  (self at: i) name == (pathArray at: i + 1 - offset)
    ifFalse: [ ^ false ].
].
^ isComplete
%
category: 'Support'
method: GsAbstractIndex
hasSamePathAs: anIndex

"Returns whether the given index is on the same path as the receiver.
 This can only be true if the index is identical."

^ self == anIndex
%
category: 'Updating Indexes'
method: GsAbstractIndex
preIndexRemoval

isComplete := false.
progress := 0.

^ nil
%
category: 'Testing'
method: GsAbstractIndex
isLegacyIndex

"Legacy indexes are kinds of IdentityIndex, so return false for a kind of GsAbstractIndex."

^ false
%
category: 'Testing'
method: GsAbstractIndex
isPathEvaluator

^ false
%
category: 'conversion'
method: GsUnicodeRangeEqualityIndex
asIndexSpecification
  ^ (UnicodeIndexSpecification
    path: self pathComponentsString
    collator: self collator)
    options: self options;
    constraintType: self constraintType;
    yourself
%
category: 'Testing'
method: GsAbstractIndex
_commonPathTermsWith: anEvaluatorOrIndex
  ^ anEvaluatorOrIndex _commonPathTermsWithIndex: self
%
category: 'Testing'
method: GsAbstractIndex
_commonPathTermsWithIndex: anIndex
  | commonSize commonTerms |
  commonSize := self size min: anIndex size.
  commonTerms := {}.
  1 to: commonSize do: [ :index | 
    | term |
    (term := (self at: index) name) = (anIndex at: index) name
      ifFalse: [ ^ commonTerms ].
    commonTerms add: term ].
  ^ commonTerms
%
category: 'Sorting'
method: GsAbstractIndex
_getObjectsWithNilOnPath

"Returns an Array containing each object that has a nil value along the path."

| array |
" gather the root objects with nil on the path "
array := { } .
self _getObjectsWithNilOnPathInto: array.
^ array

%
category: 'Sorting'
method: GsAbstractIndex
_getObjectsWithNilOnPathInto: anNscBuilderOrArray

"Returns an Array containing each object that has a nil value along the path."

1 to: self size - 1 do: [ :i |
    (self at: i) nilOnPath ifNotNil: [:nilOnPath | anNscBuilderOrArray addAll: nilOnPath ]
].

%
category: 'Traversing'
method: GsAbstractIndex
traverseAllWithParents: anObject parent: theParent upTo: aPathTerm startingAt: anOffset
  "Traverse the sub-objects of the given object using the path terms of the
 receiver up to the given path term. Return an Array of objects. The index 
 may be a collection based index."

  | nextObj sz i thePathTerm theResult |
  theResult := {{ theParent. anObject }}.
  nil == anObject
    ifTrue: [ ^ theResult ].
  sz := self size.
  i := anOffset.
  [ thePathTerm := self at: i.
  thePathTerm ~~ aPathTerm and: [ i <= sz ] ]
    whileTrue: [
      theResult do: [ :ar | | nextTraversalObj | 
        nextTraversalObj := ar at: 2.
        thePathTerm indicatesMultiValue
          ifTrue: [
            theResult := {}.
            thePathTerm
              nextObj: nextTraversalObj
              do: [ :obj | | result |
                " for each object in the collection-based instance variable make recursive call "
                result := self
                  traverseAllWithParents: obj
                  parent: nextTraversalObj 
                  upTo: aPathTerm
                  startingAt: i + 1.
                theResult addAll: result ].
            ^ theResult ]
          ifFalse: [  
            nextObj := thePathTerm _nextObjectFor: nextTraversalObj.
            theResult := {{ nextTraversalObj. nextObj }}.
            nil == nextObj
              ifTrue: [ ^ theResult ] ] ].
      i := i + 1 ].
  ^ theResult
%
category: 'Traversing'
method: GsAbstractIndex
traverse: anObject upTo: aPathTerm startingAt: anOffset addingIndex: addingIndex do: aBlock
  "Traverse the sub-objects of anObject, using the path terms of the receiver up
 to the given path term.  Add each object at the end of the traversal to the
 result set.  If addingIndex is true, add an index list entry for the
 receiver to any NSCs encountered along the path; if false, remove an index list entry for the receiver.  Returns the result set."

  self traverse: anObject withRoot: anObject upTo: aPathTerm startingAt: anOffset addingIndex: addingIndex do: aBlock
%
category: 'Traversing'
method: GsAbstractIndex
traverse: anObject withRoot: rootObject upTo: aPathTerm startingAt: anOffset addingIndex: addingIndex do: aBlock
  "Traverse the sub-objects of anObject, using the path terms of the receiver up
 to the given path term.  Add each object at the end of the traversal to the
 result set.  If addingIndex is true, add an index list entry for the
 receiver to any NSCs encountered along the path; if false, remove an index list entry for the receiver.  Returns the result set."

  | nextObj sz i thePathTerm nscOffset |
  nextObj := anObject.
  sz := self size.
  i := anOffset.
  [ (self at: i) ~~ aPathTerm _and: [ i <= sz _and: [ nil ~~ nextObj ] ] ]
    whileTrue: [ 
      thePathTerm := self at: i.
      thePathTerm indicatesMultiValue
        ifTrue: [ 
          " if indexing over a collection-based instance variable "
          thePathTerm
            _updateCollectionBasedIndexFor: self
            on: nextObj
            offset: i
            addingIndex: addingIndex.
          nscOffset := i + 1.
          thePathTerm
            nextObj: nextObj
            do: [ :obj | 
              " for each object in the collection-based instance variable make recursive call "
              self
                traverse: obj
                withRoot: rootObject
                upTo: aPathTerm
                startingAt: nscOffset
                addingIndex: addingIndex
                do: aBlock ].
          ^ self ]
        ifFalse: [ nextObj := (self at: i) _getNextObjectForTraversal: nextObj ].
      i := i + 1 ].
  i == 1
    ifTrue: [
      "no traversal"
      aBlock value: #'_incompletePathTraversal' value: rootObject.
      ^ self ].
  (nil ~~ nextObj _and: [ aBlock ~~ nil ])
    ifTrue: [ aBlock value: nextObj value: rootObject ].
%
category: 'Traversing'
method: GsAbstractIndex
_traverseObject: anObject
  "Traverse the sub-objects of the given object using the path terms of the
 receiver.  This method assumes that the index is not over a set-valued
 instance variable.  For that kind of index, use the traverse:startingAt:into:
 method.  If a nil value is reached before the end of the path, returns the
 incompletePathTraversal object."

  | nextObj sz pathTerm |
  nextObj := anObject.
  sz := self size.
  1 to: sz do: [ :i | 
    pathTerm := self at: i.
    nextObj := pathTerm _getNextObjectForTraversal: nextObj.
    (nil == nextObj and: [ i ~~ sz ])
      ifTrue: [ ^ #'_incompletePathTraversal' ] ].
  ^ nextObj
%
category: 'Traversing'
method: GsAbstractIndex
traverseObject: anObject

"Traverse the sub-objects of the given object using the path terms of the
 receiver.  This method assumes that the index is not over a set-valued
 instance variable.  For that kind of index, use the traverse:startingAt:into:
 method.  If a nil value is reached before the end of the path, returns the
 incompletePathTraversal object."

self isIndexOnNscElements
    ifTrue: [ ^ anObject ].

nil == anObject
    ifTrue: [ ^ #_incompletePathTraversal ].

^ self _traverseObject: anObject
%
category: 'Indexing Support'
method: GsAbstractIndex
_doPutCommonPathTermsForPathArray: pathArray for: anIndexList

"Find the common path terms that already exist for indexes whose path
 components match pathArray (an Array of Strings).  Adds those path terms to
 the end of receiver.  Do not put a common path term into receiver that
 is the last path term for an existing index."

| firstUnique somePathTerms j minSize sz someSz |
firstUnique := 0.
sz := pathArray size.
1 to: anIndexList size by: 2 do: [ :i |
  j := anIndexList at: i + 1.
  " only consider index objects for which the NSC is the root "
  j == 1
    ifTrue: [
      somePathTerms := anIndexList at: i.
      someSz := somePathTerms size.
      minSize := someSz min: sz.

      [ j <= minSize and:
      [ (somePathTerms at: j) name = (pathArray at: j) ] ] whileTrue: [
        ( j > firstUnique _and:
          [ ( j < someSz )_and: [
              j ~~ sz ] ] )
             ifTrue: [
               firstUnique := j.
               self addLast: (somePathTerms at: j) ].
          j := j + 1 ] ] ].
%
category: 'Searching'
method: GsRangeEqualityIndex
findRootObjectsForKey: aKey value: aValue
  | evaluator stream rootObjects |
  evaluator := self asQueryEvaluator.
  stream := evaluator
    _findAllValuesGreaterThan: aKey
    andEquals: true
    andLessThan: aKey 
    andEquals: true 
    using: evaluator comparisonForSort.
  rootObjects := IdentityBag new.
  [ stream _btreeAtEnd ]
    whileFalse: [ 
      | rootObject |
      "collect key/value/root tuples for entries in the btree whose intermediate parent object 
       is reachable from the root object in the entry."
      ((stream _peekValue == aValue) and: [ stream _peekKey == aKey ])
        ifTrue: [ "select value entries that are identically equal to aValue"
          rootObject := stream _peekRoot.
          rootObjects add: rootObject ].
      stream _btreeNext ].
 ^ rootObjects
%
category: 'Searching'
method: GsAbstractIndex
findRootObjectsComparingForKey: aKey value: aValue
  | stream rootObjects theKey |
  stream := self asQueryEvaluator
    _findAllValuesGreaterThanKey: aKey
    andEquals: true.
  rootObjects := IdentityBag new.
  [ stream _btreeAtEnd not and: [ 
    theKey := stream _peekKey.
    theKey _idxForSortEqualTo: aKey ] ]
    whileTrue: [
      | rootObject |
      "collect key/value/root tuples for entries in the btree whose intermediate parent object 
       is reachable from the root object in the entry."
      ((theKey _idxForCompareEqualTo: aKey) and: [ stream _peekValue == aValue ])
        ifTrue: [ "selec value entries that are identically equal to aValue"
          rootObject := stream _peekRoot.
          rootObjects add: rootObject ].
      stream _btreeNext ].
 ^ rootObjects
%
category: 'Searching'
method: GsAbstractIndex
findRootObjectsForKey: aKey value: aValue
  | stream rootObjects theKey |
  stream := self asQueryEvaluator
    _findAllValuesGreaterThanKey: aKey
    andEquals: true.
  rootObjects := IdentityBag new.
  [ 
  stream _btreeAtEnd not
    and: [ 
      theKey := stream _peekKey.
      theKey _idxForSortEqualTo: aKey ] ]
    whileTrue: [ 
      | rootObject |
      "collect key/value/root tuples for entries in the btree whose intermediate parent object 
       is reachable from the root object in the entry."
      (theKey == aKey and: [ stream _peekValue == aValue ])
        ifTrue: [ 
          "select value entries that are identically equal to aValue"
          rootObject := stream _peekRoot.
          rootObjects add: rootObject ].
      stream _btreeNext ].
  ^ rootObjects
%
category: 'Indexing Support'
method: GsAbstractIndex
_partialPathComponentsStringUpTo: offset

"Returns the path components string up to the given offset."

| str |
str := String new.
1 to: (offset min: self size) do: [ :i |
    str add: (self at: i) name.
    i == offset
        ifFalse: [ str add: $. ]
].
^ str
%

category: 'Updating'
method: GsRangeEqualityIndex
nscRoot: newValue
  super nscRoot: newValue.
  "force lazy initialization of optimizingComparison iv to avoid commit conflicts later..."
  self optimizingComparison
%
category: 'Searching'
method: GsRangeEqualityIndex
findRootObjectMaps: rootObjectMap pathTerm: pathTerm key: aKey value: aValue
  | evaluator stream |
  evaluator := self asQueryEvaluator.
  stream := evaluator
    _findAllValuesGreaterThan: aKey
    andEquals: true
    andLessThan: aKey
    andEquals: true
    using: evaluator comparisonForSort.
  stream 
    _scanForRootObjectMaps: aKey 
    value: aValue
    valueOop: aValue asOop
    using: self comparisonForCompare
    do: [:aRoot |
      self 
        findRootObjects: rootObjectMap 
        rootObject: aRoot
        pathTerm: pathTerm 
        key: aKey 
        value: aValue ].
%
category: 'Searching'
method: GsAbstractIndex
findRootObjectMaps: rootObjectMap pathTerm: pathTerm key: aKey value: aValue
  | stream theKey |
  stream := self asQueryEvaluator
    _findAllValuesGreaterThanKey: aKey
    andEquals: true.
  [ 
  stream _btreeAtEnd not
    and: [ 
      theKey := stream _peekKey.
      theKey _idxForSortEqualTo: aKey ] ]
    whileTrue: [ 
       "collect key/value/root tuples for entries in the btree whose intermediate parent object 
       is reachable from the root object in the entry."
      (theKey == aKey and: [ stream _peekValue == aValue ])
        ifTrue: [
          self 
            findRootObjects: rootObjectMap 
            rootObject: stream _peekRoot 
            pathTerm: pathTerm 
            key: aKey 
            value: aValue ].
      stream _btreeNextNoValue "safe as long as we don't start using reversed read streams" ]
%
category: 'Searching'
method: GsAbstractIndex
findRootObjects: rootObjectMap rootObject: rootObject pathTerm: pathTerm key: aKey value: aValue
  | pivotObject pivotPathTerm pivotPathTermOffset rootIsPivot |
  pivotPathTerm := rootObjectMap pivotPathTerm.
  pivotPathTermOffset := pivotPathTerm offset.
  pivotObject := rootObjectMap pivotObject.

  rootIsPivot := false.
  pivotPathTermOffset == 1 
    ifTrue: [ 
      rootIsPivot := pivotObject == rootObject.
      rootIsPivot ifTrue: [ (rootObjectMap traversalMap at: rootObject ifAbsentPut: [ IdentityBag new ]) add: rootObject ] ]
    ifFalse: [ (rootObjectMap roots at: rootObject ifAbsentPut: [ IdentityBag new ]) add: rootObject ].

  pathTerm offset == 1
    ifTrue: [ ^ self ].
  (rootObjectMap traversalMap at: aKey ifAbsentPut: [ IdentityBag new ]) add: aValue.
  pathTerm offset == 2
    ifTrue: [ 
      (rootObjectMap traversalMap at: aValue ifAbsentPut: [ IdentityBag new ]) add: rootObject.
      ^ self ].
  (pivotPathTermOffset ~~ 1 or: [ rootIsPivot ] )
    ifTrue: [
      self
        traverse: rootObject
        parent: nil
        upTo: pathTerm
        startingAt: 1
        do: [ :term :parent :child | (rootObjectMap traversalMap at: child ifAbsentPut: [ IdentityBag new ]) add: parent ] ].
%
category: 'Traversing'
method: GsAbstractIndex
traverse: anObject parent: aParent upTo: aPathTerm startingAt: anOffset do: aBlock
  ""

  | limit |
  nil == anObject
    ifTrue: [ ^ self ].
  limit := (self indexOfIdentical: aPathTerm) - 1.
  ^ self
    traverse: anObject 
    parent: aParent 
    pathTerm: aPathTerm 
    startingAt: anOffset 
    limit: limit 
    do: aBlock
%
category: 'Traversing'
method: GsAbstractIndex
traverse: anObject parent: aParent pathTerm: aPathTerm startingAt: anOffset limit: limit do: aBlock
  ""

  | nextObj thePathTerm nextTraversalObj |
  nil == anObject
    ifTrue: [ ^ self ].
  nextTraversalObj := anObject.
  anOffset to: limit do:
    [:i |
        thePathTerm := self at: i.
        thePathTerm indicatesMultiValue
          ifTrue: [
            thePathTerm
              nextObj: nextTraversalObj
              do: [ :obj | 
                aBlock value: thePathTerm value: nextTraversalObj value: obj.
                self
                  traverse: obj
                  parent: nextTraversalObj 
                  thru: aPathTerm
                  startingAt: i + 1
                  do: aBlock ].
            ^ self ]
          ifFalse: [  
            nextObj := thePathTerm _nextObjectFor: nextTraversalObj.
            aBlock value: thePathTerm value: nextTraversalObj value: nextObj.
            nil == nextObj
              ifTrue: [ ^ self ].
            nextTraversalObj := nextObj ] ]
%
category: 'Traversing'
method: GsAbstractIndex
traverse: anObject parent: aParent thru: aPathTerm startingAt: anOffset do: aBlock
  ""

  | limit |
  nil == anObject
    ifTrue: [ ^ self ].
  limit := (self indexOfIdentical: aPathTerm).
  ^ self
    traverse: anObject 
    parent: aParent 
    pathTerm: aPathTerm 
    startingAt: anOffset 
    limit: limit 
    do: aBlock
%
category: 'Testing'
method: GsAbstractIndex
isReducedConflictIndex
  "answer true if the receiver supports reduced conflict operations"

  ^ self options reducedConflict
%
category: 'Statistics'
method: GsAbstractIndex
_statisticsInto: dict

"Puts statistical information into the given dictionary."

| arr arr1 arr2 |
arr := dict at: #BtreeSpaceUtilization ifAbsent: [
  dict
    at: #BtreeSpaceUtilization
    put:  { { } . { } } 
].

arr1 := arr at: 1.
arr2 := arr at: 2.

btreeRoot _preOrderDo: [ :node |
  arr1 add: node numElements.
  arr2 add: node class maxNumberOfElements.
].
%
category: 'Converting'
method: GsAbstractIndex
asString
  | str |
  str := '('.
  1 to: self size do: [ :index | 
    | pt |
    pt := self at: index.
    str := str , pt name asString.
    index < self size
      ifTrue: [ str := str , '.' ] ].
  ^ super asString , str , ')'
%
category: 'Clustering'
method: GsAbstractIndex
clusterDepthFirst

"Cluster the receiver.  Only need to cluster the path terms."

1 to: self size do: [ :i |
  (self at: i) clusterDepthFirst
]
%
category: 'Formatting'
method: GsAbstractIndex
printOn: aStream

"Puts a displayable representation of the receiver on the given stream."

"Copy the implementation from Object so we don't inherit it from Collection."

aStream nextPutAll: self asString
%
category: 'Accessing'
method: GsAbstractIndex
progress

"Returns the value of the instance variable 'progress'."

^progress
%
category: 'accessing'
method: ConstrainedCharacterCollectionIndexSpecification
_createIndex
  ^ (self equalityIndexClass newWithCollator: self collator)
    options: self options; 
    constraintType: self constraintType;
    yourself
%
category: 'private'
method: ConstrainedCharacterCollectionIndexSpecification
_validateLastElementClassOn: anNsc
  "noop"

  
%
category: 'accessing'
method: ConstrainedCharacterCollectionIndexSpecification
collator
  "Returns IcuCollator to be used when comparing Unicode strings. collator may be nil."

  ^ collator
%
category: 'accessing'
method: ConstrainedCharacterCollectionIndexSpecification
collator: anIcuCollatorOrNil
  "Set the receiver's collator. Use a copy of anIcuCollator to disallow 
   changes to strength, etc, that might affect the sort ordering. "

  anIcuCollatorOrNil
    ifNil: [ collator := nil ]
    ifNotNil: [ collator := anIcuCollatorOrNil copy immediateInvariant ]
%
category: 'accessing'
method: ConstrainedCharacterCollectionIndexSpecification
constraintType
  "Returns the constraintType, either #string or #symbol."

  ^ constraintType
%
category: 'accessing'
method: ConstrainedCharacterCollectionIndexSpecification
constraintType: aSymbol
  "Set the constraintType, either #string or #symbol."

  constraintType := aSymbol
%
category: 'accessing'
method: ConstrainedCharacterCollectionIndexSpecification
equalityIndexClass
  ^ GsConstrainedCharacterCollectionIndex
%
category: 'accessing'
method: ConstrainedCharacterCollectionIndexSpecification
indexTypePrintString
  ^ self constraintType == #string
    ifTrue: [ 'stringOptimizedIndex' ]
    ifFalse: [ 'symbolOptimizedIndex' ]
%
category: 'Testing'
method: GsConstrainedCharacterCollectionIndex
_canCompareWith: aKey
  "Returns whether the receiver can make B-tree comparisons with the given key."

  (self constraintType == #string or: [ self constraintType == #unicodeString ])
    ifTrue: [
      | keyClass |
      keyClass := aKey class.
      (aKey _isSymbol not
        and: [ (keyClass _subclassOf: String)
          or: [ (keyClass _subclassOf: DoubleByteString)
              or: [ keyClass _subclassOf: QuadByteString ] ] ])
        ifFalse: [ ^ false ].
      (CharacterCollection isInUnicodeComparisonMode or: [ self constraintType == #unicodeString ])
        ifTrue: [ ^ true ]
        ifFalse: [
          "self constraintType == #string and unicode comparison mode DISABLED" 
          ^ ((keyClass == Unicode7)
            or: [ (keyClass == Unicode16)
              or: [ keyClass == Unicode32 ] ]) not ] ]
    ifFalse: [ 
      "self constraintType == #symbol"
      ^ aKey _isSymbol ]
%
category: 'accessing'
method: GsConstrainedCharacterCollectionIndex
collator
  "Returns IcuCollator to be used when comparing Unicode strings. collator may be nil."

  ^ collator
%
category: 'accessing'
method: GsConstrainedCharacterCollectionIndex
collator: anIcuCollatorOrNil
  "Set the receiver's collator. Use a copy of anIcuCollator to disallow 
   changes to strength, etc, that might affect the sort ordering. "

  anIcuCollatorOrNil
    ifNil: [ collator := nil ]
    ifNotNil: [ collator := anIcuCollatorOrNil copy immediateInvariant ]
%
category: 'accessing'
method: GsConstrainedCharacterCollectionIndex
constraintType
  "Returns the constraintType, either #string or #symbol."

  ^ constraintType
%
category: 'accessing'
method: GsConstrainedCharacterCollectionIndex
constraintType: aSymbol
  "Set the constraintType, either #string or #symbol."

  (aSymbol ~~ #string and: [ aSymbol ~~ #symbol ])
    ifTrue: [ ArgumentError signal: self class name asString, ' constraintType must #string or #symbol.'].
  constraintType := aSymbol
%
category: 'instance creation'
classmethod: GsConstrainedCharacterCollectionIndex
newWithCollator: anIcuCollatorOrNil
  "Create a new instance and initialize its B-tree root."

  | newOne |
  newOne := super new.
  newOne collator: anIcuCollatorOrNil.
  ^ newOne
%
category: 'instance creation'
classmethod: GsConstrainedCharacterCollectionIndex
newWithLastElementClass: aClass
  self shouldNotImplement: #'newWithLastElementClass:'
%
category: 'Accessing'
method: GsConstrainedCharacterCollectionIndex
lastElementClassDescription
  "answer a description of the lastElementClass of the receiver, 
   suitable for use in an error message"

  ^ self constraintType == #string
    ifTrue: [ 'String, DoubleByeString, or QuadByteString with optimizedComparison' ]
    ifFalse: [ 
      "self constraintType == #symbol"
      'Symbol, DoubleByeSymbol, or QuadByteSymbol with optimizedComparison' ].
%
category: 'Updating'
method: GsConstrainedCharacterCollectionIndex
_setLastPathTermState
  "Last object along the path unconditionally needs a dependency list."

  | lastPathTerm |
  lastPathTerm := self at: self size.
  lastPathTerm needsDepList: true
%
category: 'conversion'
method: GsConstrainedCharacterCollectionIndex
asIndexSpecification
  ^ (ConstrainedCharacterCollectionIndexSpecification
    path: self pathComponentsString
    collator: self collator)
    options: self options;
    constraintType: self constraintType;
    yourself
%
category: 'Comparison Operators'
method: GsConstrainedCharacterCollectionIndex
comparisonForCompare
  collator 
    ifNil: [ ^ super comparisonForCompare ].
  ^ BtreeOptimizedUnicodeComparison new
      collator: self collator;
      yourself
%
category: 'Comparison Operators'
method: GsConstrainedCharacterCollectionIndex
comparisonForSort
  "called during creation of receiver, so initialize unset instance variables"

  ^ comparisonForSort 
      ifNil: [
        (self constraintType == #symbol and: [ collator isNil ])
          ifTrue: [ 
            "use unicode collation for Symbols, so that Symbols with NUL bytes handled
             correctly. See bug 47116."
            self collator: IcuCollator default ].
        collator 
          ifNil: [ comparisonForSort := self comparisonForCompare ]
          ifNotNil: [ 
            comparisonForSort := BtreeOptimizedUnicodeComparison new
            collator: self collator;
            yourself ] ]
%
category: 'accessing'
method: GsConstrainedCharacterCollectionIndex
lastElementClass
  ^ CharacterCollection
%
category: 'Testing'
method: GsConstrainedCharacterCollectionIndex
optimizingComparison
  "optimizedComparison option is uncoditionally true"

  ^ true
%
category: 'instance creation'
classmethod: ConstrainedCharacterCollectionIndexSpecification
path: aString collator: anIcuCollator
  | res |
  (res := self new)
    path: aString;
    collator: anIcuCollator .
  ^ res
%
category: 'instance creation'
classmethod: ConstrainedCharacterCollectionIndexSpecification
path: aString lastElementClass: aClass
  ^ self shouldNotImplement: #'path:lastElementClass:'
%
category: 'accessing'
method: GsUnicodeRangeEqualityIndex
constraintType: aSymbolOrNil
  "Set the constraintType, either #unicodeString or nil."

  (aSymbolOrNil ~~ #'unicodeString' and: [ aSymbolOrNil ~~ nil ])
    ifTrue: [ 
      ArgumentError
        signal:
          self class name asString , ' constraintType must #unicodeString or nil' ].
  constraintType := aSymbolOrNil
%
category: 'Comparison Operators'
method: GsUnicodeRangeEqualityIndex
comparisonForSort
  "called during creation of receiver, so initialize unset instance variables"

  ^ comparisonForSort 
      ifNil: [ comparisonForSort := BtreeComparisonForCompare newForSort: self collator ]
%
category: 'Comparison Operators'
method: GsUnicodeRangeEqualityIndex
comparisonForCompare
  ^ BtreeComparisonForUnicodeCompare new
      collator: self collator;
      yourself
%
