!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Definition of Duration (from TimeZone2007.gs)
!
!=========================================================================

set class Duration

removeallmethods 
removeallclassmethods 

! ------------------- Class methods for Duration
category: 'For Documentation Installation only'
classmethod: Duration
installDocumentation

self comment:
  'Represents a length of time (from ANSI).'.
%

category: 'Duration Factory'
classmethod: Duration
days: days hours: hours minutes: minutes seconds: seconds
	"Synopsis
	Answer a <Duration> of the number of days, hours, minutes, and seconds.
	Definition: <Duration factory>
	Answer a <Duration> of the number of days, hours, minutes, and seconds. If any of the operands
	are negative, the result is smaller by that number of days, hours, minutes, or seconds as
	appropriate.
	Parameters
	days <integer> unspecified
	hours <integer> unspecified
	minutes <integer> unspecified
	seconds <number> unspecified
	Return Values
	<Duration> new"

	^self seconds: days * 24 + hours * 60 + minutes * 60 + seconds.
%
category: 'Duration Factory'
classmethod: Duration
zero
	"Synopsis
	Answer a <Duration> of zero length.
	Definition: <Duration factory>
	Answer a <Duration> representing a length of no time.
	Return Values
	<Duration> unspecified"

	^self seconds: 0.
%
category: 'other'
classmethod: Duration
seconds: seconds
	"Synopsis
	Answer a <Duration> which is seconds in length
	Definition: <Duration factory>
	If seconds is negative, answer a <Duration> that is abs (seconds) less than <Duration factory>
	#zero.
	Parameters
	seconds <number> unspecified
	Return Values
	<Duration> new"
  | res |
  (res := self new) _seconds: seconds.
  ^ res
%
! ------------------- Instance methods for Duration
category: 'Duration'
method: Duration
* operand
	"Synopsis
	Answer the result of multiplying the receiver by operand.
	Definition: <Duration>
	Answer a <Duration> that is the result of multiplying the receiver by operand.
	Parameters
	operand <number> unspecified
	Return Values
	<Duration> new"

	^self class seconds: seconds * operand.
%
category: 'Duration'
method: Duration
+ operand
	"Synopsis
	Answer the result of adding operand to the receiver.
	Definition: <Duration>
	Answer a <Duration> whose value is the result of adding the receiver and operand.
	Parameters
	operand <Duration> unspecified
	Return Values
	<Duration> new"

	^self class seconds: seconds + operand asSeconds.
%
category: 'Duration'
method: Duration
- operand
	"Synopsis
	Answer the result of subtracting the operand from the receiver.
	Definition: <Duration>
	Answer a <Duration> whose value is the result of subtracting operand from the receiver.
	Parameters
	operand <Duration> unspecified
	Return Values
	<Duration> new"

	^self class seconds: seconds - operand asSeconds.
%
category: 'Duration'
method: Duration
/ operand
	"Synopsis
	Answer the result of dividing the receiver by operand.
	Definition: <Duration>
	If operand is a <number> answer a new <Duration> whose value is the result of dividing the
	receiver by operand. If operand equals zero the ZeroDivide exception is signaled.
	If operand is a <Duration> answer a <number> whose value is the result of dividing the receiver
	by operand. If operand is <Duration factory> #zero the ZeroDivide exception is signaled.
	Parameters
	operand<number> | <Duration> unspecified
	Return Values
	<Duration> unspecified
	<number> unspecified"

	^(operand isKindOf: Duration)
		ifTrue: [seconds / operand asSeconds]
		ifFalse: [Duration seconds: seconds / operand].
%
category: 'Duration'
method: Duration
< operand
	"Synopsis
	Answer true if the receiver is less than operand. Answer false otherwise.
	Definition: <magnitude>
	Answer true if the receiver is less than operand with respect to the ordering defined for them.
	Answer false otherwise.
	It is erroneous if the receiver and operand are not comparable.
	The semantics of the natural ordering must be defined by refinement, which may also restrict the
	type of operand.
	Refinement: <Duration>
	Answer true if operand represents a <Duration> that is larger than the receiver. Answer false
	otherwise.
	Parameters
	operand <Duration> unspecified
	Return Values
	<boolean> unspecified"

	^seconds < operand asSeconds.
%
category: 'Duration'
method: Duration
= comparand
	"Synopsis
	Object equivalence test.
	Definition: <Object>
	This message tests whether the receiver and the comparand are equivalent objects at the time the
	message is processed. Return true if the receiver is equivalent to comparand. Otherwise return
	false.
	The meaning of 'equivalent' cannot be precisely defined but the intent is that two objects are
	considered equivalent if they can be used interchangeably. Conforming protocols may choose to
	more precisely define the meaning of 'equivalent'.
	The value of
	receiver = comparand
	is true if and only if the value of
	comparand = receiver
	would also be true. If the value of
	receiver = comparand
	is true then the receiver and comparand must have equivalent hash values. Or more formally:
	receiver = comparand
	receiver hash = comparand hash
	The equivalence of objects need not be temporally invariant. Two independent invocations of #=
	with the same receiver and operand objects may not always yield the same results. Note that a
	collection that uses #= to discriminate objects may only reliably store objects whose hash values
	do not change while the objects are contained in the collection.
	Refinement: <Duration>
	Answer true if the comparand is a <Duration> representing the same length of time as the
	receiver. Answer false otherwise.
	Parameters
	comparand <Object> uncaptured
	Return Values
	<boolean> unspecified"

	^(comparand isKindOf: Duration) and: [seconds = comparand asSeconds].
%
category: 'Duration'
method: Duration
abs
	"Synopsis
	Answer the absolute value of the receiver.
	Definition: <Duration>
	If the receiver is greater than or equal to <Duration Factory> #zero answer a <Duration> which is
	equal to the receiver. Otherwise answer a <Duration> which has the same magnitude as the
	receiver but the opposite sign.
	Return Values
	<Duration> unspecified"

	^0 <= seconds
		ifTrue: [self]
		ifFalse: [self class seconds: seconds negated].
%
category: 'Duration'
method: Duration
asSeconds
	"Synopsis
	Answer the total number of seconds in the length of time represented by the receiver.
	Definition: <Duration>
	Answer the total number of seconds in the length of time represented by the receiver including any
	fractional part of a second. If the receiver is less than <Duration factory> #zero then the result will
	be less than 0.
	Return Values
	<number> unspecified"

	^seconds.
%
category: 'Duration'
method: Duration
days
	"Synopsis
	Answer the number of complete days in the receiver.
	Definition: <Duration>
	Answer the number of complete days in the receiver. If the receiver is less than <Duration factory>
	#zero then the result will be less than or equal to 0.
	Return Values
	<integer> unspecified"

	^ (seconds abs // 60) // 60 // 24 * seconds sign.
%
category: 'Duration'
method: Duration
hours
	"Synopsis
	Answer the number of complete hours in the receiver.
	Definition: <Duration>
	Answer an <integer> between -23 and 23 inclusive that represents the number of complete hours
	in the receiver, after the number of complete days has been removed. If the receiver is less than
	<Duration factory> #zero then the result will be less than or equal to 0.
	Return Values
	<integer>unspecified"

	^ (seconds abs // 60) // 60 \\ 24 * seconds sign.
%
category: 'Duration'
method: Duration
minutes
	"Synopsis
	Answer the number of complete minutes in the receiver.
	Definition: <time>
	Answer an <integer> between -59 and 59 inclusive that represents the number of complete
	minutes in the receiver, after the number of complete days and hours have been removed. If the
	receiver is less than <Duration factory> #zero then the result will be less than or equal to 0.
	Return Values
	<integer>unspecified"

	^ ( seconds abs // 60) \\ 60 * seconds sign.
%
category: 'Duration'
method: Duration
negated
	"Synopsis
	Answer the negation of the receiver.
	Definition: <Duration>
	Answer a <Duration> which is of the same magnitude but opposite sign as the receiver.
	Return Values
	<Duration> unspecified"

	^self class seconds: seconds negated.
%
category: 'Duration'
method: Duration
negative
	"Synopsis
	Answer true if the receiver is less than <Duration factory> #zero.
	Definition: <Duration>
	Answer true if the receiver is less than <Duration factory> #zero, false otherwise.
	Return Values
	<boolean> unspecified"

	^seconds < 0.
%
category: 'Duration'
method: Duration
positive
	"Synopsis
	Answer true if the receiver is greater than or equal to <Duration factory> #zero.
	Definition: <Duration>
	Answer true if the receiver is greater than or equal to the <Duration factory> #zero, false
	otherwise.
	Return Values
	<boolean> unspecified"

	^0 <= seconds.
%
! fixed 42705, 44513
category: 'Duration'
method: Duration
printOn: aStream
	"Synopsis
	Return a string that describes the receiver.
	Definition: <Object>
	A string consisting of a sequence of characters that describe the receiver are returned as the
	result.
	The exact sequence of characters that describe an object is implementation defined.
	Refinement: <Duration>
	Answer a description of the receiver that is formatted as
	[-]D:HH:MM:SS[.S] where
	- is a minus sign if the receiver represents a length of time going from the future into
	the past,
	D is the number of complete days with leading zeros to fill one place,
	HH is the number of complete hours with leading zeros to fill two places,
	MM is the number of complete minutes with leading zeros to fill two places,
	SS is. the number of complete seconds with leading zeros to fill two places, and
	.S is the fractional part of the number of seconds, if any.
	Return Values
	<readableString> unspecified"

  | secs mins hours absSecs |

  seconds < 0 ifTrue: [aStream nextPut: $-].
  self days abs printOn: aStream.
  aStream nextPut: $:.
  (hours := self hours abs) < 10 ifTrue: [aStream nextPut: $0].
  hours printOn: aStream.
  aStream nextPut: $:.
  (mins := self minutes abs) < 10 ifTrue: [aStream nextPut: $0].
  mins printOn: aStream.
  aStream nextPut: $:.
  (absSecs := (secs := self seconds) asInteger abs) < 10 ifTrue: [aStream nextPut: $0].
  absSecs printOn: aStream.
  (secs _isInteger) ifFalse: [
    | float |
    0 < (float := (secs abs - absSecs) asFloat) ifTrue: [ | string | 
      aStream nextPut: $. .
      string := float asStringUsingFormat: self _fractionFormat.
      aStream nextPutAll: (string copyFrom: 3 to: string size).
    ].
  ].
%
method: Duration
_fractionFormat

^ #( 5 3 false )
%

method: Duration
asString
  ^ self printString
%
category: 'Duration'
method: Duration
seconds
	"Synopsis
	Answer the number of seconds in the receiver.
	Definition: <Duration>
	Answer a <number> strictly greater than -60 and strictly less than 60 
	that represents the number of seconds in the receiver, after the complete days, 
	hours, and minutes have been removed. 
	If the receiver is less than <Duration factory> #zero then the result 
	will be less than or equal to 0.
	Return Values
	<number> unspecified"

  | s |
  ^(s := seconds) abs \\ 60 * s sign.
%
category: 'other'
method: Duration
hash

	^seconds hash.
%
category: 'other'
method: Duration
printOnHoursMinutes: aStream

	aStream nextPut: (seconds negative ifTrue: [$-] ifFalse: [$+]).
	self hours abs < 10 ifTrue: [aStream nextPut: $0].
	self hours abs printOn: aStream.
	aStream nextPut: $:.
	self minutes abs < 10 ifTrue: [aStream nextPut: $0].
	self minutes abs printOn: aStream.
%
category: 'other'
method: Duration
_seconds: aNumber

	seconds := aNumber.
%
