!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id: gsbitmap.gs 38462 2016-01-13 04:33:12Z lalmarod $
!
! Superclass Hierarchy:
!   GsBitmap, Object.
!
!=========================================================================

expectvalue %String
run
(Globals at: #GsBitmap otherwise: nil) ifNil:[
  Object _newKernelSubclass: 'GsBitmap'
    subclassOf: Object
    instVarNames: #( hiddenSetId )
    classVars: #()
    classInstVars: #()
    poolDictionaries: #()
    inDictionary: Globals
    options:  #( #instancesNonPersistent )
    reservedOop: 1949 .
  ^ ' created GsBitmap'
].
^ 'exists'
%
run
| o |
(o := GsBitmap asOop) == 249345 ifFalse:[ ^ o ].
^ true
%
expectvalue /String
send GsBitmap definition

removeallmethods GsBitmap
removeallclassmethods GsBitmap
set class GsBitmap

category: 'For Documentation Installation only'
classmethod: GsBitmap
installDocumentation

self comment:

'GsBitmap is a class that implements an in memory bitmap, i.e. a transient
 bitmap that cannot be made persistent. A bitmap is a sparse data structure
 that logically implements a bit array. Each bit in the bitmap represents
 the objectId of a committed non-special object.
 Access to the hiddenSets in the system is implemented by creating 
 a GsBitmap that references a particular hiddenSet (see newForHiddenSet:).  

 Most of the hiddenSets in the system are read only and generate an error
 if the operation is not allowed.  See the code for the hiddenSetSpecifiers 
 method and note the last mutable hiddenSet.  HiddenSets after this point in 
 the  array generate errors if an attempt is made to modify them and the user 
 is not SystemUser.  The NotifySet, PureExportSet, and GciTrackedObjs
 have interfaces defined in System class (categories are NotifySet and
 GcISets) and these interfaces should be used to guarantee correct behavior.
 Updates to these 3 sets generate errors when attempted from GsBitmap.

 An instance of GsBitMap uses C Heap memory to store the bit array. 
 The C Heap memory use is minium of 16KBytes per instance of GsBitMap,
 For an instance with more than 2K elements, worst case memory usage is 
 approximately 8 bytes per element up to 1/64th of the objects in the repository. 
 An instance containing all of the objects in the
 repository will use approximately (System _oopHighWaterMark // 8) bytes of 
 C heap memory.
 The C Heap memory associated with an instance of GsBitmap is automatically 
 freed when the instance is garbage collected.
 Extensive use of large GsBitMaps in a session may require use
 of  ( System _vmMarkSweep )  to force garbage collection of the
 instances sooner than memory pressure on temporary object memory 
 would cause them to be garbage collected, to prevent running out of C heap
 memory.

 Do not retain an instance of GsBitmap from one session to another.
 Instances of GsBitmap are intended to exist only within a given GemStone
 session. 

 Instances may not be committed.

 The objects referenced in a GsBitmap are not protected from garbage collection.
'
%

category: 'Hidden Set Support'
classmethod: GsBitmap
hiddenSetSpecifiers

"Returns a list of the hidden set specifiers.

**************************************************************************************************
***** DO NOT CHANGE THE ORDER OF THE ELEMENTS IN THE LIST WITHOUT EDITING CODE IN gsbitmap.c *****
**************************************************************************************************
"

^#(
 #ListInstances
 #SaveNewPomObjs "See method #_enableTraceNewPomObjs"
 #ObjectsRead    "See method #_enableTraceObjectsRead"
 #SaveWriteSetUnion
 #CommitReleaseLocksSet
 #CommitOrAbortReleaseLocksSet
 #StrongReadSet  
 #GcCandidates 
 #Indexing
 #Conversion
 #WeakReferences
 #ObjInventory
 #Customer1
 #Customer2
 #Customer3
 #Customer4
 #Customer5 "last mutable hiddenSet"

 #RcReadSet
 #DepMapWriteSet
 #PomWriteSet    "Empty except after flush for commit, so only useful
		       after a transaction conflict."
 #RemovedDepMapEntries
 #SaveDepMapChangedUnion
 #ReadWriteConflicts  "StrongRead-Write conflicts "
 #WriteWriteConflicts
 #WriteDependencyConflicts
 #WriteReadLockConflicts
 #WriteWriteLockConflicts
 #AllocatedGciOops
 #ExportedDirtyObjs
 #TrackedDirtyObjs
 #ReferencedSet
 #WriteLockWriteSubset
 #NewDataPages
 #PreviousWsUnion
 #PureExportSet
 #GciTrackedObjs
 #NotifySet          
 )
%

category: 'Private'
method: GsBitmap
_1ArgPrim: opCode

"opCode     function
   1        _basicNew
   2        removeAll
   3        size
   4        asArray
   5        asArrayOfOops
   6        copy
   7        sizeInPages
   8        peekLast
   9        referencedObjects
   10       allReferencedObjects
   11       allValidOops
   12       primReferencedObjects
   13       _asArrayOfBits
"

<primitive: 1026>
opCode _validateClass: SmallInteger.
self _primitiveFailed: #_1ArgPrim: args: { opCode }
%
 
category: 'Instance Creation'
classmethod: GsBitmap
basicNew

"disallowed"

self shouldNotImplement: #basicNew
%

category: 'Private'
method: GsBitmap
_basicNew

"creates an instance registered with VM for finalization of cData"

<primitive: 674>
self _primitiveFailed: #_basicNew
%

category: 'Instance Creation'
classmethod: GsBitmap
new

"Returns a new empty GsBitmap" 

^ self _basicNew _1ArgPrim: 1
%

category: 'Instance Creation'
classmethod: GsBitmap
with: aValue

"Returns an instance of the receiver containing the argument" 

| inst |
inst := self new.
inst add: aValue.
^inst
%

category: 'Instance Creation'
classmethod: GsBitmap
with: aValue with: val2

"Returns an instance of the receiver containing the arguments" 

| inst |
inst := self new.
inst add: aValue; add: val2.
^inst
%

category: 'Instance Creation'
classmethod: GsBitmap
with: aValue with: val2 with: val3

"Returns an instance of the receiver containing the arguments" 
| inst |
inst := self new.
inst add: aValue; add: val2; add: val3.
^inst
%

category: 'Instance Creation'
classmethod: GsBitmap
with: aValue with: val2 with: val3 with: val4

"Returns an instance of the receiver containing the arguments" 
| inst |
inst := self new.
inst add: aValue; add: val2; add: val3; add: val4.
^inst
%

category: 'Instance Creation'
classmethod: GsBitmap
withAll: aCollection

"Returns an instance of the receiver containing the elements of the argument."

| inst | 
inst := GsBitmap new.
aCollection do: [ :each | inst add: each ].
^ inst.
%

category: 'Private'
method: GsBitmap
_hiddenSetId

^hiddenSetId
%

category: 'Private'
method: GsBitmap
_hiddenSetId: aSmallInt

hiddenSetId := aSmallInt
%

category: 'Instance Creation'
classmethod: GsBitmap
newForHiddenSet: hiddenSetSpecifier

"Returns a GsBitmap that references the specified hiddenSet."

| gsBitmap idx | 
gsBitmap := GsBitmap new.
idx := GsBitmap hiddenSetSpecifiers indexOfIdentical: hiddenSetSpecifier.
idx < 1 ifTrue: [ ^ hiddenSetSpecifier _error: #rtErrInvalidArgument 
                              args:{ 'not a valid hiddenSetSpecifier' }] .
gsBitmap _hiddenSetId: idx.
^gsBitmap
%

category: 'Accessing'
method: GsBitmap
hiddenSetName

"Returns the name as a symbol that corresponds to this instance.
 If there is no hidden set reference then it returns the symbol #NotAHiddenSet."

 hiddenSetId == 0 ifTrue: [ ^ #NotAHiddenSet ]
                  ifFalse: [ ^ (GsBitmap hiddenSetSpecifiers) at: hiddenSetId ]
%

category: 'Removing'
method: GsBitmap
removeAll

"Clears all of the bits in the receiver."

^self _1ArgPrim: 2
%

category: 'Testing'
method: GsBitmap
size 

"Returns the number of bits that are set in the GsBitmap."

^self _1ArgPrim: 3
%

category: 'Accessing'
method: GsBitmap
asArray

"Returns the bits that are set in the GsBitmap as an Array of objects.
 The results is an object in temporary object memory."

^self _1ArgPrim: 4
%

category: 'Private'
method: GsBitmap
_asArrayOfOops

"Returns the bits that are set in the GsBitmap as an Array of SmallIntegers
 which correspoind to the oops of the persistent objects.
 This method can be used to return the results of a large GsBitmap in 
 an array without faulting all of the objects into memory like asArray does.
 The oops can be used in the method _objectForOop: to convert the value to an object."

^self _1ArgPrim: 5
%
category: 'Private'
method: GsBitmap
_asArrayOfBits

"Returns the bits that are set in the GsBitmap as an Array of Integers
 which correspoind to the raw bit numbers in the receiver."

^ self _1ArgPrim: 13
%

category: 'Accessing'
method: GsBitmap
copy

"Returns a new GsBitmap that is a copy of the receiver."

^self _1ArgPrim: 6
%

category: 'Accessing'
method: GsBitmap
sizeInPages

"Returns the number of 16 K pages used to implement the receiver."

^self _1ArgPrim: 7
%

category: 'Enumerating'
method: GsBitmap
referencedObjects

"Answer a new GsBitmap that contains the committed, non-special objects
referenced by the objects contained in the receiver.  For each object for
which the current user has read authorization, the objects that are directly
referenced stucturally by that object (from named instance variables, indexed
or NSC contents, and dynamic instance variables) are included."

^self _1ArgPrim: 9
%

category: 'Enumerating'
method: GsBitmap
allReferencedObjects

"Similar to referencedObjects, returning a new instance of GsBitmap containing the objects directly referenced by the objects in the receiver; but in addition, the result includes all of the normally hidden interior nodes for large objects."

^self _1ArgPrim: 10
%

category: 'Enumerating'
classmethod: GsBitmap
allValidOops

"Returns an instance of GsBitmap containing the oops of all valid objects in the repository.  This bitmap only reflects currently committed objects that are not in the process of being garbage collected, i.e, not in the possible dead or deadNotReclaimed sets."

^GsBitmap new _1ArgPrim: 11
%

category: 'Enumerating'
classmethod: GsBitmap
transitiveReferences: aCollection

"Returns an instance of GsBitmap that contains all the objects which are transitively referenced from the objects in <aCollection>."

| newFound allFound diff |

newFound := GsBitmap withAll: aCollection.
allFound := newFound copy.

[true] whileTrue: [ 
  newFound := newFound referencedObjects.
  diff := newFound - allFound.
  allFound addAll: diff.
  newFound := diff.
  diff isEmpty ifTrue: [ ^ allFound ]
] 
% 

category: 'Enumerating'
classmethod: GsBitmap
allTransitiveReferences: aCollection

"Similar to transitiveReferences:, however the result includes all of the normally hidden interior nodes for large objects."

| newFound allFound diff |

newFound := GsBitmap withAll: aCollection.
allFound := newFound copy.

[true] whileTrue: [ 
  newFound := newFound allReferencedObjects.
  diff := newFound - allFound.
  allFound addAll: diff.
  newFound := diff.
  diff isEmpty ifTrue: [ ^ allFound ]
] 
% 

category: 'Enumerating'
classmethod: GsBitmap
allObjectsExcept: aCollection

"Returns an instance of GsBitmap that contains all objects that exist which are not contained in the objects transitively referenced from <aCollection>. The objects in the result may or may not be live objects (reachable from the repository roots)."

| found |

found := GsBitmap withAll: aCollection.
found := GsBitmap allTransitiveReferences: found.
^ (GsBitmap allValidOops) - found
% 

category: 'Private'
method: GsBitmap
_2ArgPrim: opCode with: arg2

" Functions with the receiver readOnly should be first
 opCode     function      
   1        includes:  
   2        writeToFile: 
   3        writeToFileInPageOrder:
   4        equals:
   5        firstObjectThatReferences:
   6        _includesBit:

     First mutable opcode
  10        add:         primitive returns the previous state
  11        addAll:  
  12        remove:      primitive returns the previous state
  13        removeAll: 
  14        union:
  15        difference:
  16        intersect:
  17        truncateToSize:
  18        _removeFirst:
  19        hiddenSetIdAsSymbol:

  20        fileInfo:
  21        readFromFile: 
  22        _auditFile:
  23        _setBits:
  24        _clearBits:
  25        primFirstObjectThatReferences:
"

<primitive: 1027>
opCode _validateClass: SmallInteger.
self _primitiveFailed: #_2ArgPrim:with: args: { opCode . arg2 }
%

category: 'Testing'
method: GsBitmap
includes: anObject

"Returns true if anObject is in the GsBitmap."

^self _2ArgPrim: 1 with: anObject
%

category: 'Private'
method: GsBitmap
_includesBit: aBit

"Returns true if the bit is in the GsBitmap."

^self _2ArgPrim: 6 with: aBit
%

category: 'Private'
method: GsBitmap
_truncateToSize: newSize

"Truncate receiver by removing objects from the end of the set until it reaches a size
 of newSize.  Objects are removed in order, going from highest to lowest object ID.   
 
 Returns the number of objects removed."
 
^self _2ArgPrim: 17 with: newSize
%

category: 'Private'
method: GsBitmap
_removeFirst: count 

"Remove the first count objects from the receiver.  Objects are removed from
 the beginning, going from lowest to highest object ID. 
 
 Returns the number of objects removed from the hidden set."
 
^self _2ArgPrim: 18 with: count
%

category: 'Hidden Set Support'
classmethod: GsBitmap
hiddenSetIdAsSymbol: oldHiddenSetId

"Returns the symbol corresponding to the old hiddenSet specifier" 

^ GsBitmap new _2ArgPrim: 19 with: oldHiddenSetId
%

category: 'Removing'
method: GsBitmap
removeFirst: count 

"Remove the first count objects from the receiver and return them in a new GsBitmap.
 Objects are removed from the beginning, going from lowest to highest object ID.  
"

| copy |
copy := self copy.
self _removeFirst: count.
^copy removeAll: self
%

category: 'Removing'
method: GsBitmap
removeLast: count 

"Remove the last count objects from the receiver and return them in a new GsBitmap.
 Objects are removed from the end, going from higest to the lowest object ID.  
"

| copy |
copy := self copy.
self _truncateToSize: (self size - count).
^copy removeAll: self
%

category: 'Testing'
method: GsBitmap
isEmpty

"Returns true there are no entries in the GsBitmap."

^self size == 0
%

category 'File Operations'
method
writeToFile: aString

"Writes the contents of the receiver to the specified file.
 The file created is approximately 5 bytes for each object plus 24 bytes for 
 the file header.

 If the file exists an error is returned.

 Returns the number of objectIds written to the file.
"

^self _2ArgPrim: 2 with: aString
%

category 'File Operations'
method
writeToFileInPageOrder: aString
"Writes the contents of the receiver to the specified file in pageOrder.

 Operations like instance migration which process a large volume of objects 
 often run significantly faster if the objects are processed in batches that
 are in page order rather than in object ID order.

 The file created is approximately 5 bytes for each object plus 24 bytes for 
 the file header.

 If the file exists an error is returned.

 Returns the number of objectIds written to the file.
"

^self _2ArgPrim: 3 with: aString
%

category: 'Testing'
method: GsBitmap
equals: aGsBitmap

"Returns true if the receiver and argument contain exactly the same elements."

^self _2ArgPrim: 4 with: aGsBitmap
%

category: 'Searching'
method: GsBitmap
firstObjectThatReferences: anObject
"Search the receiver to find and return the first object which references anObject. 
 Returns nil if no such object was found."

^self _2ArgPrim: 5 with: anObject
%

category: 'Adding'
method: GsBitmap
add: anObject

"Adds a non special object to the GsBitmap.  
 Returns anObject."

self _2ArgPrim: 10 with: anObject.
^anObject
%

category: 'Adding'
method: GsBitmap
addAll: anArrayOrGsBitmap

"Adds each element of the Array of non special objects or the contents of a GsBitmap 
 to the receiver. Returns an Integer, the number of bits added to the receiver."

(anArrayOrGsBitmap == nil) ifTrue: [ ^anArrayOrGsBitmap ].
^self _2ArgPrim: 11 with: anArrayOrGsBitmap
%

category: 'Private'
method:
_setBits: anArray

"Add the bits specified by the Integers in anArray to the receiver.
 Returns the number of bits added."

^self _2ArgPrim: 23 with: anArray
%
category: 'Private'
method:
_clearBits: anArray

"Clears the bits specified by the Integers in anArray from the receiver.
 Returns the number of bits cleared."

^self _2ArgPrim: 24 with: anArray
%


category: 'Removing'
method: GsBitmap
remove: anObject

"Removes anObject from the GsBitmap.
 If the object is not present in the GsBitmap it issues #objErrNotInColl,
 otherwise it returns anObject.
"
(self _2ArgPrim: 12 with: anObject) 
    ifFalse: [ self _error: #objErrNotInColl args: { anObject } ].
^anObject
%

category: 'Removing'
method: GsBitmap
remove: anObject ifAbsent: aBlock

"Removes anObject from the receiver.  If anObject is not in the
 receiver, this method evaluates aBlock and returns its value.  
 The argument aBlock must be a zero-argument block.
"

(self _2ArgPrim: 12 with: anObject)
    ifFalse: [ ^ aBlock value ].
^anObject
%

category: 'Removing'
method: GsBitmap
removeIfPresent: anObject

"Remove anObject from the receiver.  Returns nil if anObject is
 missing from the receiver, otherwise returns the argument.
"

(self _2ArgPrim: 12 with: anObject)
    ifFalse: [ ^ nil ].
^anObject
%

category: 'Removing'
method: GsBitmap
removeAll: anArrayOrGsBitmap

"Removes each element of the Array of non special objects or the contents of a GsBitmap 
 from the receiver.  Returns anArrayOrGsBitmap."

^self _2ArgPrim: 13 with: anArrayOrGsBitmap
%

category: 'Private'
method: GsBitmap
_3ArgPrim: opCode with: arg2 with: arg3

" Functions with the receiver readOnly should be first
 opCode     function      
   1        enumerateWithLimit:startingAfter:
   2        enumerateAsOopsWithLimit:startingAfter:
 "

<primitive: 1028>
opCode _validateClass: SmallInteger.
self _primitiveFailed: #_3ArgPrim:with:with: args: { opCode . arg2 . arg3}
%

category: 'Enumerating'
method: GsBitmap
enumerateWithLimit: maxResultSize startingAfter: anObject

"Returns an Array containing the first maxResultSize bits as objects
 starting with the first one after the specified object.  
 To start the enumeration use the SmallInteger zero as anObject.
 To get the next batch use the last element of the array returned.

 This enumeration is non destructive, so no values are cleared.

 If the GsBitmap contains fewer than maxResultSize elements 
 or the maxResultSize has the value 0, then it returns 
 all of the elements after the specified object.
"

^self _3ArgPrim: 1 with: maxResultSize with: anObject
%

category: 'Enumerating'
method: GsBitmap
enumerateAsOopsWithLimit: maxResultSize startingAfter: anOop

"Returns an Array containing the first maxResultSize bits as oops
 starting with the first one after the specified oop.
 To start the enumeration use the SmallInteger zero as anOop. 
 To get the next batch use the last element of the array returned.

 This enumeration is non destructive, so no values are cleared.

 If the GsBitmap contains fewer than maxResultSize elements after the
 specified object or the maxResultSize has the value 0, then it returns 
 all of the elements after the specified object.
"

^self _3ArgPrim: 2 with: maxResultSize with: anOop
%

category: 'Enumerating'
method: GsBitmap
do: aBlock

"Executes the one argument block aBlock for each object in 
 in the receiver.  Does a non destructive enumerate of the GsBitmap. 
 Returns the number of objects enumerated."

 | count arr |
 count := 0.
 arr := Array new.
 arr add: 0.
 [ | n |
   arr := self enumerateWithLimit: 2000 startingAfter: (arr last).
   n := arr size .
   count := count + n .
   1 to: n do:[:j | aBlock value: (arr at: j) ] .
   n ~~ 0 . 
 ] whileTrue.
 ^ count
%

category: 'Private'
method: GsBitmap
_doAsOops: aBlock

"Executes the one argument block aBlock for each oop in 
 in the receiver.  Does a non destructive enumerate of the GsBitmap 
 with the argument to the block being anOop. 
 Returns the number of objects enumerated."

 | count arr |
 count := 0.
 arr := Array new.
 arr add: 0.
 [ | n |
   arr := self enumerateAsOopsWithLimit: 2000 startingAfter: (arr last).
   n := arr size .
   count := count + n .
   1 to: n do:[:j | aBlock value: (arr at: j) ] .
   n ~~ 0 . 
 ] whileTrue.
 ^ count
%


category: 'Set Arithmetic'
method: GsBitmap
+ aGsBitmap

"Returns a new GsBitmap with the elements that are in the receiver or aGsBitmap.
 Same as union:."

^self _2ArgPrim: 14 with: aGsBitmap
%

category: 'Set Arithmetic'
method: GsBitmap
union: aGsBitmap

"Returns a new GsBitmap with the elements that are in the receiver or aGsBitmap.
 Same as +."

^self _2ArgPrim: 14 with: aGsBitmap
%

category: 'Set Arithmetic'
method: GsBitmap
- aGsBitmap

"Returns a new GsBitmap with the elements in the receiver that are not in aGsBitmap.
 Same as difference:."

^self _2ArgPrim: 15 with: aGsBitmap
%

category: 'Set Arithmetic'
method: GsBitmap
difference: aGsBitmap

"Returns a new GsBitmap with the elements in the receiver that are not in aGsBitmap.
 Same as -."

^self _2ArgPrim: 15 with: aGsBitmap
%

category: 'Set Arithmetic'
method: GsBitmap
* aGsBitmap

"Returns a new GsBitmap with the elements that are in both the receiver and in aGsBitmap.
 Same as inersect:."

^self _2ArgPrim: 16 with: aGsBitmap
%

category: 'Set Arithmetic'
method: GsBitmap
intersect: aGsBitmap

"Returns a new GsBitmap with the elements that are in both the receiver and in aGsBitmap.
 Same as *."

^self _2ArgPrim: 16 with: aGsBitmap
%

category: 'File Operations'
classmethod: GsBitmap
fileInfo: bitmapFilePath

" Returns an array which contains the following values:
  1.  The number of oops in the file
  2.  True if the file was written in page order.
  3.  The number of valid oops.
  4.  The number of invalid oops (oops that are not allocated or in the 
      process of being garbage collected).

  Raises an error if the file could not be opened for reading, the file
  is corrupt, or aString is not an instance of String.
"

^(GsBitmap new) _2ArgPrim: 20 with: bitmapFilePath
%

category: 'Enumerating'
method: GsBitmap
readFromFile: aString

"Reads the contents of the file and adds the oops in the file to the 
 receiver.  If the file doesn't exist or is not in the GsBitmap format, 
 then this method returns an error. 

 Subsets of the file can be read into a GsBitmap with 
         readFromFile:withLimit:startingAt:.

 ObjectIds that are in the freelist or in the deadObjs or possibleDead sets 
 are not included. 

 Returns the number of objects that were not included.
"

^self _2ArgPrim: 21 with: aString
%

category: 'Private'
method
_auditFile: aString
 "Reads the file. "

 ^ self _2ArgPrim: 22 with: aString
%

category: 'File Operations'
method: GsBitmap
readFromFile: aString withLimit: maxResultSize startingAt: startIndex

"Adds the contents of the specified file indexed by the subrange: 
 (startingAt, startingAt + maxResultSize).  The resulting GsBitmap 
 may contain fewer than maxResultSize elements if the value of
 startingAt + maxResultSize is an index that is larger than the number of
 oops in the file or if some of the ObjectIds represented are no longer 
 live objects, i.e., they were found to be in the freeList or the 
 deadNotReclaimed or possible dead sets.

 If the file doesn't exist or is not in the GsBitmap format, 
 then this method returns an error. 

 The first object in the file has an index of 1.  It is an error if 
 startIndex is less than 1.

 Returns the number of objects read.
"

<primitive: 1029>
aString _validateClass: String.
maxResultSize _validateClass: SmallInteger.
startIndex _validateClass: SmallInteger.
self _primitiveFailed: #readFromFile:withLimit:startingAt: args: { aString . maxResultSize . startIndex }
%

category: 'Formatting'
method:
printString

"Returns a String with a displayable representation of the receiver."

^ self printStringWithMaxSize: 1000
%


category: 'Formatting'
method: GsBitmap
printStringWithMaxSize: n

"Returns the contents of the GsBitmap as a string with oops of the 
 objects in the bitmap arranged 8 per line.
"
| count result bmSize |

count := 0.
result := String new.
bmSize := self size.
result add: ('GsBitmap contains ' , bmSize asString , ' objects. ' , Character lf).
(bmSize == 0) ifTrue: [ ^ result ].
result add: ('1. ').
self _doAsOops: [ :oop | 
  count := count + 1.
  result add: (' ' , oop asString , ' ').
  ((count rem: 8) = 0 and: [count < bmSize]) ifTrue: [ 
     result add: (' ' , Character lf , (count + 1) asString , '. ').
     (result size > n and: [n ~= 0]) ifTrue: [ 
       result add: (' ........' , Character lf).
       result add: ( bmSize asString , '. ' , self peekLast asString, Character lf).
       ^ result
     ]
  ]
].
^result
%

! Fix 48443 - limit bits to 8
category: 'Formatting'
method: GsBitmap
printOn: aStream

"Puts a displayable representation of the receiver on aStream."

aStream nextPutAll: (self printStringWithMaxSize: 8).
%

category: 'Peeking'
method: GsBitmap
peek

"Answer the first element in the receiver without removing it, or nil if the receiver is empty."
^ self isEmpty ifTrue:[nil] ifFalse:[ (self peekCount: 1) at: 1]
%

category: 'Peeking'
method: GsBitmap
peekLast 

"Answer the last in the receiver without removing it, or nil if the receiver is empty."
^ self isEmpty ifTrue:[nil] ifFalse:[ self _1ArgPrim: 8 ]
%

category: 'Peeking'
method: GsBitmap
peekCount: aCount

"Fetches elements from the receiver without removing them.  Returns an Array that contains
 the minimum of aCount or the receiver's size entries."

^self enumerateWithLimit: aCount startingAfter: 0
%

category: 'Set Arithmetic'
method: GsBitmap
intersectSize: aGsBitmap

"Returns a SmallInteger which is the number of elements present in both the receiver and aGsBitmap."

| result newBm |
newBm := self * aGsBitmap .
result := newBm size.
newBm removeAll .
^ result
%

category: 'Removing'
method: GsBitmap
removeCount: maxToRemove
"Removes entries from the receiver, and returns an Array that contains the
 minimum of maxToRemove or the receiver's size entries."

|result|
result := self peekCount: maxToRemove .
self removeAll: result .
^result
%

category: 'Removing'
method: GsBitmap
removeNext
"Removes the first element from the receiver and returns it.  Returns nil if the receiver is empty."

^ self isEmpty ifTrue:[nil] ifFalse:[ (self removeCount: 1) at: 1]
%
category: 'Converting'
method: GsBitmap
asGsBitmap

^ self
%

! Fix 48445 - use these variants when finding reference paths
category: 'Private'
method: GsBitmap
primReferencedObjects

"Answer a new GsBitmap that contains the committed, non-special objects
referenced by the objects contained in the receiver. Similar to the 
referencedObjects method except this method includes internal objects
in the result, and does not include the entire collection contents
when traversing large objects or NSCs."

^self _1ArgPrim: 12
%

category: 'Searching'
method: GsBitmap
primFirstObjectThatReferences: anObject
"Search the receiver to find and return the first object which references anObject. 
 Returns nil if no such object was found. Similar to the firstObjectThatReferences:
 method except this method may return an internal object as the result"

^self _2ArgPrim: 25 with: anObject
%

method: GsBitmap
_reportString
| str |
str := String new .
str add: self size asString; add: ' oops '.
self asArray do:[:obj |
  str add: ' '; add: obj asOop asString; add:'(a ', obj class name; add: $) .
].
^ str
%
