!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!

! edited for 44362

!------------- session methods support for Behavior ------------------
set class Behavior

category: 'Accessing the Method Dictionary'
method:
_fullMethodDictEnv0

^ self _fullMethodDictEnv: 0
%

category: 'Accessing the Method Dictionary'
method:
_fullMethodDictEnv: envId

"Returns a dictionary of combined base and override methods.
 Does not include the transient method dictionary.
 For envId==0 only,  includes the GsPackagePolicy contents.
 Used by ClassOrganizer."
 | res |
 res := SymbolKeyValueDictionary new .
 (self persistentMethodDictForEnv: envId) ifNotNil:[ :aDict |
    aDict keysAndValuesDo:[:k :v | res at: k put: v ].
 ].
 envId == 0 ifTrue:[
   GsPackagePolicy currentOrNil ifNotNil:[:pp| pp copyMethodDictFor: self into: res ].
 ].
^ res 
%

category: 'Enumerating'

! replaces categorysForEnv:
method:
_unifiedCategorys: envId
  | cats | 
  cats := self _baseCategorys: envId .
  cats ifNil:[ cats := GsMethodDictionary new ] ifNotNil:[ cats copy ].
  envId == 0 ifTrue:[ 
    cats := cats ifNil:[ GsMethodDictionary new ] ifNotNil:[ cats copy ].
    GsPackagePolicy currentOrNil ifNotNil:[:pp| pp copyCategoryDictFor: self into: cats ].
  ].
  ^ cats
%

method:
env: envId categorysDo: aBlock
  "evaluates aBlock for each method category of receiver
   in specified environment. Returns the receiver.

   aBlock should be a two argument block
   expecting the args  categoryNameSymbol ,   selectorsSet.
   If the package manager is active in the current session,
   aBlock may be invoked more than once for each category name.
   The iteration is done directly over the receiver's categories."

  envId == 0 ifTrue:[ 
    GsPackagePolicy currentOrNil ifNotNil:[:pp| pp categorysDo: aBlock for: self ].
  ].
  self env: envId baseCategorysDo: aBlock
%

category: 'Updating Categories'
method:
renameOrMergeCategory: oldName to: newName 
  ^ self renameOrMergeCategory: oldName to: newName environmentId: 0
%

category: 'Updating Categories'
method:
renameOrMergeCategory: oldName to: newName environmentId: envId

"Changes the name of the specified category to newName (a String), and
 returns the receiver.  If oldName is not an existing category in the receiver
 generates an error.  If newName is already in the receiver's
 category list, moves all the methods from the old category to the new
 category, and removes the old category."

| oldsym newsym oldCatSet policy |
self _validatePrivilege ifFalse:[ ^ nil ].
(self includesCategory: oldName environmentId: envId) ifFalse:[
  ^ self _categoryNotFound: oldName 
].
(self includesCategory: newName environmentId: envId) ifFalse:[
  ^ self renameCategory: oldName to: newName environmentId: envId
].
oldsym := oldName asSymbol.
newsym := newName asSymbol.
policy := GsPackagePolicy currentOrNil .
(envId ~~ 0 or:[ self canWriteMethods or: [ policy == nil ]])  ifTrue:[ | catDict |
  catDict := self _baseCategorysForStore: envId . 
  oldCatSet := catDict at: oldsym .
  (catDict at: newsym ) addAll: oldCatSet .
  catDict removeKey: oldsym .
] ifFalse:[ |  pkgCats |
  pkgCats := GsMethodDictionary new  .
  policy copyCategoryDictFor: self into: pkgCats .
  oldCatSet := (pkgCats at: oldsym ) copy .
  oldCatSet do:[ :aSelector |
     policy moveSelector: aSelector toCategory: newsym for: self .
  ].
  policy removeCategory: oldsym for: self.
].
self _needsAnnouncement ifTrue:[
  oldCatSet do: [:sel | 
     self _announceMethodMoved: (self compiledMethodAt: sel) oldCategory: oldsym
  ].
].
%

category: 'Updating Categories'
method:
renameCategory: categoryName to: newCategoryName
  ^ self renameCategory: categoryName to: newCategoryName environmentId: 0
%

method:
renameCategory: categoryName to: newCategoryName environmentId: envId

"Changes the name of the specified category to newCategoryName (a
 String), and returns the receiver.  If categoryName is not in the
 receiver's method dictionary, or if newCategoryName is already in the
 receiver's method dictionary, generates an error."

| catSym newCatSym oldCatSet policy |
self _validatePrivilege ifFalse:[ ^ nil ].
(self includesCategory: newCategoryName environmentId: envId ) ifTrue:[
  ^ self _error: #classErrMethCatExists args: { newCategoryName . self }
].
(self includesCategory: categoryName environmentId: envId ) ifFalse:[
   ^ self _error: #classErrMethCatNotFound args: { categoryName }
].
newCatSym := newCategoryName asSymbol .
catSym := categoryName asSymbol.
policy := GsPackagePolicy currentOrNil .
(envId ~~ 0 or:[ self canWriteMethods or: [ policy == nil ]]) ifTrue:[ | catDict|
   catDict :=  self _baseCategorysForStore: envId .
   catDict at: newCatSym put: (oldCatSet := catDict at: catSym ).
   catDict removeKey: catSym .
] ifFalse: [   | pkgCats catSet |
   pkgCats := GsMethodDictionary new  .
   policy copyCategoryDictFor: self into: pkgCats .
   oldCatSet := pkgCats at: catSym otherwise: { } .
   catSet := self addCategory: newCatSym environmentId: envId .
   catSet addAll: oldCatSet.
   policy removeCategory: catSym for: self.
].
self _needsAnnouncement ifTrue:[
  oldCatSet do: [:sel | 
     self _announceMethodMoved: (self compiledMethodAt: sel) oldCategory: catSym
  ].
].
%

category: 'Updating Categories'
method:
removeCategory: categoryName environmentId: envId

"Removes the specified category and all its methods from the receiver's
 method dictionary.  If categoryName is not in the receiver's categories,
 generates an error.
 Any breakpoints in removed methods are cleared."

| theSelectors catSym allCats |
self _validatePrivilege ifFalse:[ ^ nil ].
(self includesCategory: categoryName environmentId: envId ) ifFalse:[
  ^ self _categoryNotFound: categoryName .
].
catSym := categoryName asSymbol .
allCats := self _unifiedCategorys: envId .
theSelectors := (allCats at: catSym ) copy.
1 to: theSelectors size do: [ :j |
  self _basicRemoveSelector: (theSelectors at: j) environmentId: envId
].
envId == 0 ifTrue:[ | policy |
  policy := GsPackagePolicy currentOrNil .
  policy ifNotNil:[ :pp | pp removeCategory: catSym for: self ].
  (self canBeWritten or: [ policy == nil ]) ifTrue:[
    (self _baseCategorys: envId) removeKey: catSym otherwise: nil .
  ].
] ifFalse: [
  (self _baseCategorys: envId) removeKey: catSym
]. 
self _announceReorganized .
%

category: 'Updating Categories'
method:
addCategory: aString
  ^ self addCategory: aString environmentId: 0
%

method:
addCategory: aString environmentId: envId

"Adds aString as a method category for the receiver.  If aString is already a
 method category, generates an error.
 Returns the newly added category (a SymbolSet)."

| aSymbol canWrite res policy |
self _validatePrivilege ifFalse:[ ^ nil ].
aSymbol := aString asSymbol .
(self includesCategory: aSymbol environmentId: envId ) ifTrue:[ 
  ^ self _error: #classErrMethCatExists args: { aString . self }
].
canWrite := self canWriteMethods .
policy := GsPackagePolicy currentOrNil .
(canWrite or:[ envId ~~ 0 or: [ policy == nil ]]) ifTrue:[ | aSet catDict |
  catDict :=  self _baseCategorysForStore: envId .
  catDict at: aSymbol put: ( aSet := SymbolSet new ) .
  res := aSet
] ifFalse: [ 
  res := policy addCategory: aSymbol for: self 
].
self _announceReorganized .
^ res
%

category: 'Updating Categories'
method:
moveMethod: aSelector toCategory: categoryName
  ^ self moveMethod: aSelector toCategory: categoryName environmentId:0
%
category: 'Updating Categories'
method: Behavior
moveMethod: aSelector toCategory: categoryName environmentId: envId
	"Moves the method aSelector (a String) from its current category to the
 specified category (also a String).  If either aSelector or categoryName is
 not in the receiver's method dictionary, or if aSelector is already in
 categoryName, generates an error..
 This method does not account for selectors inherited from superclasses."

	| selSym catSym oldCat policy method |
	self _validatePrivilege ifFalse: [^nil].
	(self includesCategory: categoryName environmentId: envId)
		ifFalse: [^self _categoryNotFound: categoryName].
	selSym := aSelector asSymbol.
	policy := GsPackagePolicy currentOrNil.
	method := self
				compiledMethodAt: selSym
				environmentId: envId
				otherwise: nil
				usePackages: oldCat notNil.
	method ifNil: [^self _error: #classErrSelectorNotFound args: {aSelector}].
	oldCat := self categoryOfSelector: aSelector environmentId: envId.
	catSym := categoryName asSymbol.
	(envId ~~ 0 or: [self canWriteMethods or: [policy == nil]])
		ifTrue: 
			[| catDict |
			catDict := self _baseCategorysForStore: envId.
			oldCat ifNotNil: [(catDict at: oldCat) remove: selSym].
			(catDict at: catSym) add: selSym]
		ifFalse: [policy moveSelector: selSym toCategory: catSym for: self].
	self _needsAnnouncement
		ifTrue: [self _announceMethodMoved: method oldCategory: oldCat]
%

method:
_announceReorganized
  ^ self
%
method:
_needsAnnouncement
 "Result controls sends of  _announceMethodMoved:oldCategory: .
  This implementation replaced when Seaside or other GUI tools installed. "

  ^ false
%

category: 'Browser Methods'
method:
_selectorWithSource: aString

"Returns the selector for the environment 0 method 
 whose source string is identical to aString."

self _fullMethodDictEnv0 keysAndValuesDo:[:aSelector :aMethod |
  (aMethod _sourceString == aString) ifTrue: [^ aSelector ]
].
^nil
%

method:
compiledMethodAt: aSelector environmentId: envId otherwise: notFoundVal usePackages: usePkg

"Returns the compiled method associated with the argument aSelector (a String).
 The argument must be a selector in the receiver's method dictionary; if it is
 not, this method returns notFoundVal.
 If usePkg is true, looks in GsPackagePolicy, then in persistent method dicts;
 else looks in session method dicts then in persistent method dicts"

  <primitive: 2001>  "enter protected mode"
  | prot |
  prot := System _protectedMode .
  ^ [ | aSym meth md cls fmt |
      (aSym := Symbol _existingWithAll: aSelector) ifNotNil:[
        cls := self . 
        fmt := format .
        [ cls ~~ nil ] whileTrue:[
          (envId == 0 and:[ usePkg]) ifTrue:[
            meth := GsPackagePolicy currentOrNil ifNotNil:[:pp | pp compiledMethodAt: aSym for: cls].
          ] ifFalse:[
            (md := cls transientMethodDictForEnv: envId) ifNotNil:[
              meth := md at: aSym otherwise: notFoundVal .
            ] 
          ].
          meth ifNil:[
            (md := cls persistentMethodDictForEnv: envId ) ifNotNil:[
              meth := md at: aSym otherwise: notFoundVal .
            ].
          ].
          meth ifNotNil:[ cls := nil ] 
                ifNil:[ (fmt bitAnd: 16r14000) ~~ 0 
                           ifTrue:[ cls := primaryCopy "RUBY_VIRTUAL|MODULE_inclSelf" ]
                           ifFalse:[ cls := nil ]].
          fmt := 0 .
        ].
      ].
      meth ifNil:[ notFoundVal]
    ] ensure:[
      prot _leaveProtectedMode
    ].
%

method:
compiledMethodAt: aSelector environmentId: envId otherwise: notFoundVal
  ^ self compiledMethodAt: aSelector environmentId: envId otherwise: notFoundVal 
         usePackages: false
%

method:
compiledMethodAt: aSelector environmentId: envId usePackages: usePkg
| aMeth |
aMeth := self compiledMethodAt: aSelector environmentId: envId otherwise: nil
		usePackages: usePkg .
aMeth == nil ifTrue:[ self _error: #rtErrKeyNotFound args: { aSelector } ].
^ aMeth
%

method:
includesSelector: aString environmentId: envId

^ (self compiledMethodAt: aString environmentId: envId otherwise: nil
	usePackages: false) ~~ nil
%


category: 'Accessing the Method Dictionary'
method:
selectorsForEnvironment: envId usePackages: usePkg

"Returns an Array of Symbols, consisting of all of the message selectors
 defined by the receiver.  (Selectors inherited from superclasses are not
 included.)  For keyword messages, the Symbol includes each of the keywords,
 concatenated together.  
 If usePkg is true, result consists of persistent methods
 plus GsPackagePolicy methods ; if false  persistent methods plus
 session methods."

 <primitive: 2001>  "enter protected mode"
 | set prot |
 prot := System _protectedMode .
 [ | aDict |
   set := IdentitySet new.
   (aDict := self persistentMethodDictForEnv: envId ) ifNotNil:[
     set addAll: aDict keys.
   ].
   (envId == 0 and:[ usePkg]) ifTrue:[
     GsPackagePolicy currentOrNil ifNotNil:[:pp| pp  selectorsFor: self into: set].
   ] ifFalse:[
     (aDict := self transientMethodDictForEnv: envId) ifNotNil:[
       set addAll: aDict keys.
     ].
   ].
 ] ensure:[
   prot _leaveProtectedMode
 ].
 ^ set asArray .
%

category: 'Accessing the Method Dictionary'
method:
selectorsForEnvironment: envId
  ^ self selectorsForEnvironment: envId usePackages: false
%

category: 'Accessing the Method Dictionary'
method:
whichClassIncludesSelector: aString environmentId: envId usePackages: usePkg

"If the selector aString is in the receiver's method dictionary,
 returns the receiver.  Otherwise, returns the most immediate superclass
 of the receiver where aString is found as a message selector.  Returns
 nil if the selector is not in the method dictionary of the receiver or
 any of its superclasses.
 If usePkg is true, result consists of persistent methods
 plus GsPackagePolicy methods ; if false  persistent methods plus
 session methods."

  | currClass aSymbol |
  aSymbol := Symbol _existingWithAll: aString .
  aSymbol ifNil:[ ^ nil ].

  currClass := self.
  envId ~~ 0 ifTrue:[
    [ currClass == nil ] whileFalse:[ 
      (currClass compiledMethodAt: aSymbol environmentId: envId otherwise: nil 
                  usePackages: usePkg) ifNotNil:[
         ^currClass
      ].
      currClass := currClass superclassForEnv: envId 
    ].
  ] ifFalse:[
    [ currClass == nil ] whileFalse:[ 
      (currClass compiledMethodAt: aSymbol environmentId: envId otherwise: nil 
              usePackages: usePkg) ifNotNil:[ ^ currClass ].
      currClass := currClass superClass
    ].
  ].
  ^ nil
%

category: 'Accessing the Method Dictionary'
method:
whichClassIncludesSelector: aString environmentId: envId
  ^ self whichClassIncludesSelector: aString environmentId: envId usePackages: false
%

category: 'Accessing the Method Dictionary'
method:
_topazGeneralSelector: aString env: envId
  "ruby_selector_suffix dependent"
  | sel |
  sel := aString .
  envId == 1 ifTrue:[ "Ruby semantics"  | sz |
    sz := aString size .
    (sz > 3 and:[ (aString at: sz - 3) == $# ]) ifFalse:[
      "default: add most general ruby suffix"
      sel := sel , '#0*&'   "_asSymbolWithRubySuffix: but not asSymbol yet"
    ].
  ].
  ^ sel
%

category: 'Accessing the Method Dictionary'
method: 
_topazMethodLookup: aString env: envId usePackages: usePkgBool
  "Returns an Array   { actualSelector , class } "
  | sel |
  sel := self _topazGeneralSelector: aString env: envId  .
  ^ { sel .
      self whichClassIncludesSelector: sel environmentId: envId 
        usePackages: usePkgBool  
    }
%

category: 'Accessing the Method Dictionary'
method:
_topazMethodAt: aString env: envId 
  "Returns a GsNMethod, or signals an Error."
  ^ [ self compiledMethodAt: (self _topazGeneralSelector: aString env: envId)
           environmentId: envId 
    ] on: LookupError do: [:ex |
      nil
    ]
%


! method Behavior whichClassIncludesSelector:environmentId:usePackages:

!-------------------------------------------------------------------
category: 'Class Timestamps'
method:
commentStamp

  "Return the comment timeStamp.
   Both a Class and its metaclass share a single comment timeStamp"
  | eDict |
  (eDict := self extraDict) ifNil:[ ^ nil ].
  ^ eDict at: #GSClassCommentStamp otherwise: nil
%

method:
commentStamp: aStamp
  "Set the receiver's comment timeStamp to aStamp.
   Both a Class and its metaclass share a single comment stamp"

  | eDict |
  (eDict := self extraDictForStore) ifNotNil:[
    eDict at: #GSClassCommentStamp put: aStamp
  ].
%

method: 
theNonMetaClass
  "Sent to a class or metaclass, always return the class.
   Used by Monticello"

  ^self
%


!-------------------------------------------------------------------
category: 'Method Timestamps'
method:
authorInitials
    ^ GsPackagePolicy authorInitials
%

method:
changeStamp 
  "Answer a string to be pasted into source code to mark who changed it and when."
  ^ self authorInitials , ' ' , (Date today asStringUsingFormat: #(2 1 3 $/ 1 1 $: false )), ' ',
    (Time now asString copyFrom: 1 to: 5)
%

!-------------------------------------------------------------------
category: 'Method Timestamps'
method:
methodStampDictName
  ^ #GSMethodStampDict
%

method:
setStamp: aStamp forMethod: selector
  "Set the timestamp for the given method. 
   This code shared by Classes and metaclasses, 
   parameterized by methodStampDictName."

  | methodStampDict eDict selSym |
  selSym := Symbol _existingWithAll: selector  .
  selSym ifNil:[ self error:'invalid selector'].
  GsPackagePolicy currentOrNil ifNotNil:[:pp | 
    (pp setStamp: aStamp forBehavior: self forMethod: selSym ) ifNotNil:[ ^self].
  ]. 
  (eDict := self extraDictForStore) ifNotNil:[  | dictName |
    dictName := self methodStampDictName .
    methodStampDict := eDict at: dictName otherwise: nil .
    methodStampDict ifNil: [ 
      GsObjectSecurityPolicy setCurrent: self objectSecurityPolicy while:[
        methodStampDict := IdentityKeyValueDictionary new.
      ].
      eDict at: dictName put: methodStampDict.
    ].
    aStamp ifNil: [ methodStampDict removeKey: selSym otherwise: nil ]
        ifNotNil: [ methodStampDict at: selSym put: aStamp ]
  ].
%

method:
stampForMethod: selector
  "Get the timestamp for the given method"

  | stamp methodStampDict eDict selSym |
  selSym := Symbol _existingWithAll: selector  .
  selSym ifNil:[ ^ '' ].
  stamp := GsPackagePolicy currentOrNil ifNotNil:[:pp| pp stampForMethod: selSym in: self].
  stamp ifNotNil: [ ^stamp ].

  (eDict := self extraDictForStore) ifNil:[ ^'' ]. 
  (methodStampDict := eDict at: self methodStampDictName otherwise: nil) ifNil: [ ^'' ].
  ^ methodStampDict at: selSym otherwise: '' 
%
!-------------------------------------------------------------------
category: 'Pragmas'

method:
pragmaDictName

  ^ #GSMethodPragmaDict
%
method:
createPragmaFrom: aPragmaArray for: aGsMethod
  | pragma args pragmas keyword |
  aPragmaArray isEmpty
    ifTrue: [ ^ nil ].
  pragmas := Array new.
  1 to: aPragmaArray size by: 2 do: [ :i | 
    keyword := aPragmaArray at: i.
    args := aPragmaArray at: i + 1.
    args == nil
      ifTrue: [ args := #() ].
    pragma := Pragma keyword: keyword arguments: args.
    pragma setMethod: aGsMethod.
    pragmas add: pragma ].
  ^ pragmas
%

method:
_pragmasForMethod: aGsNMethod
  "Get the legacy Pragmas for a method compiled before 3.5.
   This code shared by Classes and metaclasses, 
   parameterized by pragmaDictName."

  | pragmas methodPragmaDict eDict selSym |
  selSym := aGsNMethod selector .
  pragmas := GsPackagePolicy currentOrNil ifNotNil:[:pp | pp  pragmasForMethod: selSym in: self].
  pragmas ifNotNil: [ ^pragmas ].

  (eDict := self extraDict) ifNil:[ ^#() ].
  (methodPragmaDict := eDict at: self pragmaDictName otherwise: nil) ifNil: [ ^#() ].
  ^ methodPragmaDict at: selSym otherwise: #() 
%

method:
pragmasForMethod: selector
  "Get the Pragmas for the method specified by  selector."

  | pragmas selSym meth |
  selSym := Symbol _existingWithAll: selector  .
  selSym ifNil:[ ^ #() ] .
  meth := self compiledMethodAt: selector otherwise: nil .
  meth ifNotNil:[ 
    pragmas := meth pragmas.
    pragmas ifNotNil:[ ^ pragmas ].
  ]. 
  ^ self _pragmasForMethod: meth .
%

method:
pragmasForMethod: selector env: envId
  "Get the Pragmas for the method specified by selector and envId."

  | meth |
  envId == 0 ifTrue:[ ^ self pragmasForMethod: selector ].
  meth := self compiledMethodAt: selector environmentId: envId otherwise: nil.
  meth ifNil:[ ^ #() ].
  meth pragmas ifNotNil:[ :p | ^ p ].
  ^ #()
%

category: 'Pragmas'
method:
setPragmas: pragmaArray forMethod: selector

  "Deprecated, pragmas are now generated into GsNMethod's debugInfo during compile. 

   Set the pragmas for the given method .
   This code shared by Classes and metaclasses, 
   parameterized by pragmaDictName."

  | methodPragmaDict eDict selSym |
  pragmaArray ifNotNil:[
    self deprecated: 'Behavior >> setPragmas:forMethod: deprecated in 3.5' .
  ].
  selSym := Symbol _existingWithAll: selector  .
  selSym ifNil:[ self error:'invalid selector'].
  GsPackagePolicy currentOrNil ifNotNil:[:pp |
     (pp setPragmas: pragmaArray forBehavior: self forMethod: selSym ) ifNotNil:[ ^self ].
  ].
  (eDict := self extraDictForStore) ifNotNil:[ | dictName |
     dictName := self pragmaDictName .
     methodPragmaDict := eDict at: dictName otherwise: nil .
     methodPragmaDict ifNil: [ 
       GsObjectSecurityPolicy setCurrent: self objectSecurityPolicy while:[
         methodPragmaDict := IdentityKeyValueDictionary new.
       ].
       eDict at: dictName put: methodPragmaDict.
     ].
     pragmaArray ifNil:[ methodPragmaDict removeKey: selSym otherwise: nil  ]
         ifNotNil: [ methodPragmaDict at: selSym put: pragmaArray ]
  ].
%

!-------------------------------------------------------------------
category: 'Updating the Method Dictionary'
method:
removeSelector: aString environmentId: envId

"Removes the method whose selector is aString from the receiver's method
 dictionary.  If the selector is not in the method dictionary, generates an
 error."

| selector removedFromBaseDict |

self _validatePrivilege ifFalse:[ ^ nil ].
selector := Symbol _existingWithAll: aString .
selector ifNotNil:[
  removedFromBaseDict := self _basicRemoveSelector: selector environmentId: envId.
  removedFromBaseDict ifTrue:[ | cDict |
    cDict := self _baseCategorys: envId .
    cDict keysAndValuesDo: [:aKey :setOfSelectors |
        (setOfSelectors remove: selector otherwise: nil ) ifNotNil:[ 
           ^ self "done"
        ]
     ].
    ^ self _error: #classErrSelectorNotFound args: { aString } .
  ].
] ifNil:[ 
  ^ self _error: #classErrSelectorNotFound args: { aString } .
]
%

method:
__basicRemoveSelector: aSymbol environmentId: envId
"returns  true if a method found to remove"
| md pmd removed tmeth pmeth |
self _validatePrivilege ifFalse:[ ^ nil ].
md := self transientMethodDictForEnv: envId .
removed := false .
md ifNotNil:[
  tmeth := md removeKey: aSymbol otherwise: nil .
].
pmd := self persistentMethodDictForEnv: envId .
pmd ifNotNil:[
  pmeth := pmd removeKey: aSymbol otherwise: nil .
].
"Now refresh method lookup caches to account for removal of the selector
 and delete any breakpoints in the removed method(s) ."
tmeth ifNotNil:[
  self _refreshLookupCache: aSymbol oldMethod: tmeth env: envId .
  removed := true
].
pmeth ifNotNil:[
  removed := true .
  self _codeChangedForEnv: envId .
  self _refreshLookupCache: aSymbol oldMethod: pmeth env: envId
].
^ removed
%

! fixed 42720
method:
_basicRemoveSelector: aSymbol environmentId: envId

"Private."  
| oldMeth removedFromSelf |
self _validatePrivilege ifFalse:[ ^ nil ].
removedFromSelf := false.
envId == 0 ifTrue:[
  oldMeth := GsPackagePolicy currentOrNil ifNotNil:[:pp| pp removeMethodAt: aSymbol for: self]. 
  oldMeth ifNil: [
    (self persistentMethodDictForEnv: 0) ifNotNil:[:dict | 
       oldMeth := dict removeKey: aSymbol . "runtime error here if key not found"
    ].
    self setStamp: nil forMethod: aSymbol.
    removedFromSelf := true.
  ].
  self _codeChangedForEnv: envId .
  self _refreshLookupCache: aSymbol oldMethod: oldMeth env: envId .
] ifFalse:[
  removedFromSelf := self __basicRemoveSelector: aSymbol environmentId: envId .
].
^removedFromSelf
%

method:
removeAllMethods
  ^ self removeAllMethods: 0
%

method:
removeAllMethods: envId

"Removes all methods from the receiver.  This should not be done without
 considerable forethought!"
| baseCats baseMeths |
self _validatePrivilege ifFalse:[ ^ nil ].
baseCats := self _baseCategorys: envId .
baseCats ifNotNil: [ baseCats removeAll ].
baseMeths := self persistentMethodDictForEnv: envId .
baseMeths ifNotNil: [ 
  envId == 0 ifTrue:[
    baseMeths keysDo: [:sel | 
      self setStamp: nil forMethod: sel.
    ].
  ].
  baseMeths removeAll .
  baseMeths valueConstraint: GsNMethod . "to handle v2.3 to v3.0 image upgrade"
].
envId == 0 ifTrue:[
  GsPackagePolicy currentOrNil ifNotNil:[:pp| pp removeAllMethodsFor: self].
].
self _clearLookupCaches: envId .
%

!-------------------------------------------------------------------
category: 'Modifying Classes'
method:
recompileAllMethodsInContext: aSymbolList

"Recompiles all methods for the receiver, using the specified symbol list.

 The environmentId of each method is preserved by the recompilation.
 
 This method is designed to allow a user interface to issue GciContinue after
 fixing the source code for a method in error.  GciContinue will reattempt the
 compilation of the method which contained an error, then proceed to the next
 method."
 
| envId |
self _validatePrivilege ifFalse:[ ^ nil ].
envId := 0 .
self env: envId unifiedCategoriesDo:[ :theCategory :selectorList |
    selectorList do: [ :aSelector| | aMeth err theSource |
      [ "This Block supports GciContinue"
        "Make sure the method is there before trying to recompile it.
         An exception handler may have removed the method!"
        err := nil .
        aMeth := self compiledMethodAt: aSelector environmentId: envId otherwise: nil .
        aMeth ifNotNil:[ | mEnvId |
          theSource := aMeth sourceString .
          (mEnvId := aMeth environmentId) == envId ifFalse:[
             self error:'mismatched environmentId'
          ].
          [ | stamp |
            envId == 0 ifTrue:[ stamp := self stampForMethod: aSelector].
            self compileMethod: theSource dictionaries: aSymbolList
                          category: theCategory environmentId: envId .
            envId == 0 ifTrue:[ self setStamp: stamp forMethod: aSelector].
          ] onException: CompileError do:[:ex |
            err := ex .
            ex outer .
          ]
        ].
        err == nil 
      ] untilTrue .
    ] .
].
^ self
%

method:
_removeAllSubclassCode

"Dereference the code objects of all environment 0 GsMethods of the receiver,
 to force recompilation of those methods."

| theSubclasses aDict envId |
self _validatePrivilege ifFalse:[ ^ nil ].
envId := 0 .
theSubclasses := self _subclasses .
theSubclasses ifNotNil:[
  theSubclasses do:[:x| x _removeAllSubclassCode ] .
  ].
GsPackagePolicy currentOrNil ifNotNil:[:pp | pp removeAllSubclassCodeFor: self "envId 0 only"].
aDict := self persistentMethodDictForEnv: envId .
aDict ifNotNil:[
  aDict keysAndValuesDo:[ :aKey :aMethod |
    aDict at: aKey put: ((aDict at: aKey) _copyToForceRecompilation).
  ].
  self _codeChangedForEnv: envId  .
].
self _refreshClassCache: true .
^ self
%

method:
recompileAllSubclassMethodsInContext: aSymbolList

"Recompiles all methods for the receiver and its subclasses, using
 the specified symbol list.  If the receiver is not modifiable,
 then methods in subclasses will not be recompiled, since only
 modifiable classes should have the Subclasses class variable present."

| theSubclasses |
self _validatePrivilege ifFalse:[ ^ nil ].
self _removeAllSubclassCode .
self recompileAllMethodsInContext: aSymbolList .
theSubclasses := self _subclasses .
theSubclasses ifNotNil:[
  theSubclasses do:[:x | x recompileAllSubclassMethodsInContext: aSymbolList ]
  ].
^ self
%
!-------------------------------------------------------------------
category: 'Testing
method:
canWriteMethods

  ^ self canWriteMethodsEnv: 0
%

method:
canWriteMethodsEnv: envId

  | mds dict |
  (mds := methDicts) _isArray ifTrue:[
    dict := mds atOrNil: (envId*4 + 1) .  
    dict ifNotNil:[ ^ dict canBeWritten ].
    ^ mds canBeWritten
  ] ifFalse:[
    mds ifNotNil:[ 
      envId == 0 ifTrue:[ ^ mds canBeWritten ] 
    ].
  ].
  ^ self canBeWritten
%


! big deletion

category: 'Accessing the Method Dictionary'
method:
methodDictForEnv: envId usePackages: usePkg
 | dict md |
 (md := self persistentMethodDictForEnv: envId ) ifNotNil:[
   dict := md copy 
 ].
 dict ifNil:[ dict := GsMethodDictionary new ].
 (envId == 0 and:[ usePkg]) ifTrue:[
   GsPackagePolicy currentOrNil ifNotNil:[:pp| pp copyMethodDictFor: self into: dict] 
 ] ifFalse:[
   (md := self transientMethodDictForEnv: envId ) ifNotNil:[
     dict addAll: md 
   ].
 ].
 ^ dict
%

method:
methodDictForEnv: envId
  ^ self methodDictForEnv: envId usePackages: false
%
category: 'Modifying Classes'
method:
_addClassVar: aSymbol value: aVal 

"Adds the class variable with name aSymbol to the receiver's
 class variables dictionary."
  | dict |
  self _validatePrivilege ifTrue:[
    (dict := classVars) ifNil:[ dict := self _createClassVarsDict ].
    (dict associationAt: aSymbol otherwise: nil) ifNotNil:[ :assoc |
       assoc _value: aVal
    ] ifNil:[ | assocClass |
      assocClass := SessionTemps current at:#ClassVariablesAssociationClass otherwise: SymbolAssociation.
    
      dict addAssociation:( assocClass newWithKey: aSymbol value: aVal)  .
    ]
  ]
%

! 42240
method: Behavior
upgradeComments

  " Upgrade comment representation to GS/64 3.1 standard.
    Prints out info on changes made to log file.
    Prints out WARNING info if any unusual conditions noted.
    Returns an array of two Boolean elements:
      1:  true if any warnings, otherwise false
      2:  true if a change was made to class, otherwise false
  "

  | prefix meta precomment comment descript descriptStr warning changed lf |

  System currentObjectSecurityPolicy: self objectSecurityPolicy.
  prefix := '    Class ' , (self name asString) , ': '.
  comment := nil.
  precomment := self _extraDictAt: #comment.
  meta := self class.
  warning := false.
  changed := false.
  lf := Character lf.
  " Check for comment method "
  (meta includesSelector: #comment) 
  ifTrue: [
    comment := [ self comment ] on: Error do:[:ex | 'old comment not available'].
    (comment isKindOf: CharacterCollection)
    ifFalse: [
      GsFile gciLogServer: prefix , lf ,
      '          WARNING: Non-standard comment method.' , lf ,
      '          No changes made.'.
      warning := true.
      ^ Array with: warning with: changed ]
    ifTrue: [
      self _extraDictAt: #comment put: comment.
      meta removeSelector: #comment.
      changed := true.
      GsFile gciLogServer: prefix , 'Using comment method.'.
      precomment isNil ifFalse: [
        (precomment = comment) ifFalse: [
          GsFile gciLogServer: 
          '          WARNING: Pre-existing #comment field will be lost!'.
          warning := true.
      "fall-through deliberate"
      ]]]].
  " Now check description field"
  descript := self _extraDictAt: #description.
  descript ifNil: [ 
    comment ifNil: [ 
      GsFile gciLogServer: prefix , 'No comment method or #description field.'.
      precomment ifNotNil:[ GsFile gciLogServer: 
        '          Pre-existing #comment field will be used.' ]
      ifNil: [ GsFile gciLogServer: '          No changes made.']].
    ^ Array with: warning with: changed ].

  (descript isKindOf: GsClassDocumentation)
  ifFalse: [
    comment ifNil: [
      GsFile gciLogServer: prefix , lf ,
'          WARNING: No comment method and non-standard #description field.'.
      precomment ifNotNil: [
        GsFile gciLogServer: 
        '          Pre-existing #comment field will be used.' ].
      GsFile gciLogServer: '          No changes made.'.
      warning := true.
      ^ Array with: warning with: changed ]
    ifNotNil: [
      GsFile gciLogServer: 
      '          WARNING: Non-standard #description field will be deleted!'.
      self _extraDictRemoveKey: #description.
      changed := true.
      warning := true.
      ^ Array with: warning with: changed ]].

  descriptStr := descript asString.
  comment ifNil: [
    GsFile gciLogServer: prefix , 'Using #description field.'.
      precomment ifNotNil: [
        (precomment = descriptStr) ifFalse: [
          GsFile gciLogServer: 
      '          WARNING: Pre-existing #comment field will be overridden!'.
          warning := true ]].
    self _extraDictAt: #comment put: descriptStr.
    self _extraDictRemoveKey: #description.
    changed := true.
    ^ Array with: warning with: changed ]
  ifNotNil: [
    self _extraDictRemoveKey: #description.
    changed := true.
    (comment = descriptStr) 
    ifTrue: [  ^ Array with: warning with: changed ]
    ifFalse: [
      GsFile gciLogServer: 
      '          WARNING: Comment method and #description field mismatch.'.
      GsFile gciLogServer: 
      '          WARNING: description field will be deleted!'.
      warning := true.
      ^ Array with: warning with: changed ]].
^ Array with: warning with: changed
%

category: 'Category'
method:
category

"Returns the classCategory instance variable of the receiver.  If the receiver's
 category is nil, returns its superclass's category."

| categ supercls |
categ := self _classCategory .
supercls := self superClass .
categ ifNil: [
  supercls == Object ifTrue:[ 
    self objectSecurityPolicy == SystemObjectSecurityPolicy 
      ifTrue:[  ^ 'Kernel' ]
      ifFalse:[ ^ 'User Classes' ].
  ].
  supercls == nil ifTrue: [ ^'Kernel' ].
  ^ supercls category
].
^ categ
%

!------------- transient session methods support for Behavior ------------------
category: 'Updating the Method Dictionary'
method: Behavior
_transientSessionMethodBehaviorsCacheName
  ^ #'TransientSessionMethod_Behaviors'
%
category: 'Updating the Method Dictionary'
method: Behavior
_transientSessionMethodBehaviorsCache
  | behaviors |
  behaviors := SessionTemps current
    at: self _transientSessionMethodBehaviorsCacheName
    ifAbsent: [ 
      behaviors := IdentitySet new _setNoStubbing.
      SessionTemps current
        at: self _transientSessionMethodBehaviorsCacheName
        put: behaviors ].
  ^behaviors
%
category: 'Updating the Method Dictionary'
method: Behavior
compileTransientMethod: sourceString dictionaries: aSymbolList environmentId: environmentId
  "This method compiles some source code for the receiver and installs the method into
   the transient method dictionary."

  <primitive: 2001>  "enter protected mode"
  | prot |
  self _validatePrivilege
    ifFalse: [ ^nil].
  [
    prot := System _protectedMode.
    ^ self _compileTransientMethod: sourceString dictionaries: aSymbolList environmentId: environmentId  ]
    ensure: [ prot _leaveProtectedMode ]
%
category: 'Updating the Method Dictionary'
method: Behavior
_compileTransientMethod: sourceString dictionaries: aSymbolList environmentId: environmentId
  "This method compiles some source code for the receiver and installs the method into
   the transient method dictionary.  
   The first argument, sourceString, is the string of source code to be compiled.  

   The second argument is a SymbolList to be used in parsing, along with the list of all
   class variables and pool dictionaries for the receiver and all of its
   superclasses.  

   sourceString must be a kind of String or MultiByteString.  Instances of
   JapaneseString are not supported as source strings.  String literals
   ('abc') are generated as instances of the class of sourceString, 
   unless sourceString is a Symbol, in which case 'abc' produces a String.
   If sourceString is a DoubleByteSymbol, 'abc' produces a DoubleByteString.

   anEnvironmentId must be a SmallInteger >= 0 and <= 16rFFFF. 
   0 denotes the base Smalltalk image.  1 is reserved for use by Ruby.

   If a method with the same selector does not already exist an error is signalled.

   If there are no errors, this adds the resulting compiled method to the
   receiver's method dictionary and returns that method,
   otherwise signals a CompileError .
   A CompileWarning may be signaled, after adding the new method
   to a receiver's method dictionary."

  <protected>
  | meth tmd categ preCompiledMethod |
  self objectSecurityPolicy 
    ifNotNil: [:policy |
      policy currentUserCanWrite 
        ifFalse: [ self error: 'You must be able to write to the receiver to add transient session methods' ] ].
   preCompiledMethod := self
        _primitiveCompileMethod: sourceString
        symbolList: aSymbolList
        category: #'xxyzzy'
        oldLitVars: nil
        intoMethodDict: GsMethodDictionary new
        intoCategories: GsMethodDictionary new
        environmentId: environmentId. "pre-compile the method source to extract selector"
  self 
    _checkCompileResult: preCompiledMethod
    source: sourceString
    suppressCompileWarning: true. 
  preCompiledMethod class == GsNMethod
    ifTrue: [
      (self categoryOfSelector: preCompiledMethod selector)
        ifNil: [ 
          self
            error:
              'Transient method: ' , preCompiledMethod selector asString
                , ' must override an existing method.' ]
        ifNotNil: [ :existingCategory | 
          "Use the category of the existing persistent method"
          categ := existingCategory ] ].
  tmd := self transientMethodDictForEnv: environmentId.
  tmd ifNil: [
    tmd := GsMethodDictionary new.
    self _transientSessionMethodBehaviorsCache add: self.
    self transientMethodDictForEnv: environmentId put: tmd.
  ].
  meth := self
    compileMethod: sourceString
    dictionaries: aSymbolList
    category: categ
    intoMethodDict: tmd
    intoCategories: nil
    intoPragmas: nil
    environmentId: environmentId.
  self _clearLookupCaches: environmentId.
  ^ meth 
%
category: 'Updating the Method Dictionary'
method: Behavior
removeTransientSelector: aString environmentId: envId
  "Removes the method whose selector is aString from the receiver's transient
   method dictionary.  If the selector is not in the transient method dictionary, 
   generates an error."

  | selector |
  self _validatePrivilege
    ifFalse: [ ^nil].
  selector := Symbol _existingWithAll: aString.
  selector
    ifNotNil: [ 
      | md tmeth |
      md := self transientMethodDictForEnv: envId.
      md ifNotNil: [ tmeth := md removeKey: selector otherwise: nil ].
      tmeth ifNotNil: [ ^ self _refreshLookupCache: selector oldMethod: tmeth env: envId ] ].
  ^ self _error: #'classErrSelectorNotFound' args: {aString} 
%
category: 'Updating the Method Dictionary'
method: Behavior
removeTransientSelector: aString environmentId: envId ifAbsent: aBlock
  "Removes the method whose selector is aString from the receiver's
   transient method dictionary.  If the selector is not in the 
   transient method dictionary, returns the result of evaluating the
   zero-argument block aBlock.  Otherwise, returns the receiver."

  | aKey tmd meth |
  self _validatePrivilege
    ifFalse: [ ^ nil ].
  aKey := Symbol _existingWithAll: aString.
  aKey
    ifNotNil: [ 
      tmd := self transientMethodDictForEnv: envId.
      tmd ifNotNil: [ meth := tmd at: aKey otherwise: nil ] ].
  meth ifNil: [ ^ aBlock value ].
  self removeTransientSelector: aKey environmentId: envId
%

!-----------------------------------------------
category: 'Fileout'
method:
nameForFileout

"Deprecated. Returns the name to be used for this class for fileout."

| arr sz |
self deprecated: 'Behavior>>nameForFileout deprecated v3.2.'.

arr := GsCurrentSession currentSession symbolList dictionariesAndSymbolsOf: self.
1 to: (sz := arr size) do:[:j | | nm |
  (nm := (arr at: j) at: 2)  == self name ifTrue:[ ^ nm ].
].
sz == 0 ifTrue:[ ^ self thisClass name ].
Error signal:'during fileout, class name does not match name in dictionary(s)'
%

category: 'Fileout'
method:
_fileoutHeaderOn: stream

  stream isEmpty ifTrue:[
    (stream isKindOf: GsFile) ifTrue:[
      stream nextPutAll: 'fileformat utf8' ; lf .
    ].
    "Gs64 v3.3, no SET SOURCESTRINGCLASS directives in fileouts."
  ].
%

category: 'Fileout'
method:
fileOutCategories

"Returns a String or MultiByteString with all the receiver's methods in 
 Topaz Filein format."

| strm |
strm := AppendStream on: String new .
self fileOutCategoriesOn: strm .
^ strm contents
%

category: 'Fileout'
method:
fileOutCategoriesOn: stream

"Writes the receiver's categories and methods onto the given stream
 in Topaz filein format."

| lf cls nm |

self _fileoutHeaderOn: stream  .
cls := self thisClass.
lf := Character lf.
stream nextPut: lf.
nm := cls name.

self fileOutMethodRemovalOn: stream name: nm.

self fileOutPreMethodsOn: stream.

stream nextPutAll: '! ------------------- Class methods for '; _fileOutAll: nm; nextPut: lf.
cls class sortedCategoryNames do: [:category |
  cls class fileOutCategory: category on: stream
].

stream nextPutAll: '! ------------------- Instance methods for '; _fileOutAll: nm; nextPut: lf.
cls sortedCategoryNames do: [:category |
  cls fileOutCategory: category on: stream
].

self fileOutPostMethodsOn: stream.

^stream
%

category: 'Fileout'
method:
fileOutCategory: catName

"Returns a String or MultiByteString containing the methods of the given category in
 Topaz Filein format."

| strm |
strm := AppendStream on: String new .
self fileOutCategory: catName on: strm .
^ strm contents
%
category: 'Fileout'
method:
_topazFileOutCategory: catName header: headerStr asUtf8: utf8Bool

^ self _topazFileOutCategory: catName header: headerStr asUtf8: utf8Bool env: 0
%

category: 'Fileout'
method:
_topazFileOutCategory: catName header: headerStr asUtf8: utf8Bool env: envId
| strm |
strm := AppendStream on: String new  .
strm dynamicInstVarAt: #utf8Bool put: utf8Bool .
strm nextPutAll: headerStr .
self fileOutCategory: catName on: strm environmentId: envId .
^ utf8Bool ifTrue:[ strm contents encodeAsUTF8 ] ifFalse:[ strm contents ].
%

category: 'Fileout'
method:
fileOutCategory: catName on: stream

^ self fileOutCategory: catName on: stream environmentId: 0
%

category: 'Fileout'
method:
fileOutCategory: catName on: stream environmentId: envId

"Files out the given category on the given stream."

| sels |

self _fileoutHeaderOn: stream  .
sels := self sortedSelectorsIn: catName environmentId: envId .
stream nextPutAll:'set compile_env: ' ; nextPutAll: envId asString ; lf .
sels do: [:selector |
  self fileOutMethod: selector environmentId: envId on: stream
]
%

category: 'Fileout'
method:
fileOutClass

"Returns a String or MultiByteString with the receiver's class definition 
 and all the receiver's methods in Topaz Filein format."

| strm |
strm := AppendStream on: String new .
self fileOutClassOn: strm environmentId: 0 .
^ strm contents 
%

category: 'Fileout'
method:
_topazFileoutClass: headerStr asUtf8: utf8Bool env: envId
| strm |
strm := AppendStream on: String new .
strm dynamicInstVarAt: #utf8Bool put: utf8Bool .
strm nextPutAll: headerStr .
self fileOutClassOn: strm environmentId: envId .
^ utf8Bool ifTrue:[ strm contents encodeAsUTF8 ] ifFalse:[ strm contents ].
%
category: 'Fileout'
method:
_topazFileoutClass: headerStr asUtf8: utf8Bool

^ self _topazFileoutClass: headerStr asUtf8: utf8Bool env: 0
% 

category: 'Fileout'
method:
multibyteSources

"Returns an Array of selectors whose source strings contain code points > 255"
| res |
res := { } .
self selectors do:[:sel | | str |
  str := (self compiledMethodAt: sel) sourceString .
  ((str _stringCharSize > 1) and:[ str _asString == nil]) ifTrue:[
    res add: sel .   
  ].
].
^ res
%

category: 'Fileout'
method: 
fileOutClassDefinitionOn: stream 

^ self fileOutClassDefinitionOn: stream environmentId: 0
%

! fixed 44271
category: 'Fileout'
method: 
fileOutClassDefinitionOn: stream environmentId: envId

"Writes the receiver's preclass, class definition, and comment onto 
 the given stream in filein format."

self _fileoutHeaderOn: stream  .
stream nextPutAll:'set compile_env: 0'; lf .
envId == 0 ifTrue:[ 
  stream    nextPutAll: '! ------------------- Class definition for ' ;
       _fileOutAll: self thisClass name; lf .
  self fileOutPreClassOn: stream .
  stream nextPutAll: 'expectvalue /Class'; lf ;
    nextPutAll: 'doit'; lf ;
    _fileOutAll: self definition; lf ;
    nextPut: $% ; lf .
  self fileOutCommentOn: stream .
  self fileOutClassCategoryOn: stream .
  self _fileOutEnvSuperclassesOn: stream .
] ifFalse:[
  self _fileOutEnvSuperclass: envId on: stream .
].
%

category: 'Fileout'
method:
_fileOutEnvSuperclassesOn: stream
  | max |
  max := self _maxSuperclassEnv .
  max downTo: 1 do:[:n | 
    self _fileOutEnvSuperclass: n on: stream
  ]
%
category: 'Fileout'
method:
_fileOutEnvSuperclass: envId on: stream
  | sup |
  sup := self superclassForEnv: envId . 
  sup ~~ self superClass ifTrue:[
    stream nextPutAll: 'doit'; lf ;
       _fileOutAll: self thisClass name ;
       nextPutAll: ' superclassForEnv: '; nextPutAll: envId asString ;
       nextPutAll: ' put: ' ;  _fileOutAll: sup name ; nextPutAll:' . true ' ; lf ;
       nextPut: $% ; lf .
  ].
%


category: 'Fileout'
method: 
fileOutClassCategoryOn: stream
 
"Writes out class category, if there is one for this class."

self _fileoutHeaderOn: stream  .
(self _classCategory) ifNotNil:[ :cat |
  stream nextPutAll: 'expectvalue /Class'; lf ;
         nextPutAll: 'doit'; lf ;
         _fileOutAll: (self name asString);
         nextPutAll: ' category: ';
         _fileOutAll: cat quoted ; lf ;
         nextPut: $% ; lf .
].
%

category: 'Fileout'
method:
fileOutClassByCategoryOn: stream

"Writes the receiver's definition and methods onto the given stream in
 filein format."

self _fileoutHeaderOn: stream  .
self fileOutClassDefinitionOn: stream .
self fileOutCategoriesOn: stream.
%

category: 'Fileout'
method:
fileOutClassOn: stream

^ self fileOutClassOn: stream environmentId: 0
%

category: 'Fileout'
method:
fileOutClassOn: stream environmentId: envId

"Writes the receiver's definition and methods onto the given stream in
 filein format."

self _fileoutHeaderOn: stream  .
self fileOutClassDefinitionOn: stream environmentId: envId .
self fileOutMethodsOn: stream environmentId: envId .
%

! 42240
category: 'Fileout'
method: Behavior
fileOutCommentOn: stream

"Writes code to create class comment onto the given stream in
 filein format."

self _fileoutHeaderOn: stream  .
self commentForFileout ifNotNil:[ :cmt |
  stream nextPutAll: 'expectvalue /Class'; lf ;
    nextPutAll: 'doit'; lf ;
    _fileOutAll: self thisClass name; nextPutAll: ' comment: ' ; lf ;
    _fileOutAll: cmt printString; lf ;
    nextPut:    $%;  lf
].
%

category: 'Fileout'
method:
fileOutMethod: selector
  ^ self fileOutMethod: selector environmentId: 0
%

category: 'Fileout'
method:
_topazFileoutMethod: selector header: headerStr asUtf8: utf8Bool

^ self _topazFileoutMethod: selector header: headerStr asUtf8: utf8Bool env: 0
%

category: 'Fileout'
method:
_topazFileoutMethod: selector header: headerStr asUtf8: utf8Bool env: envId

| strm |
strm := AppendStream on: String new .
strm nextPutAll: headerStr .
strm dynamicInstVarAt: #utf8Bool put: utf8Bool .
self fileOutMethod: selector environmentId: envId on: strm .
^ utf8Bool ifTrue:[ strm contents encodeAsUTF8 ] ifFalse:[ strm contents ].
%

category: 'Private'
method:
_fileOutMethod: selector environmentId: envId

"Returns a string with the given method's category and source in Topaz
 Filein format."

| strm |
strm := AppendStream on: String new. 
self fileOutMethod: selector environmentId: envId on: strm .
^ strm contents
%

category: 'Fileout'
method:
fileOutMethod: selector environmentId: envId

"Returns a String or MultiByteString with the given method's category and  
 source in Topaz Filein format."

| strm |
strm := AppendStream on: String new .
self fileOutMethod: selector environmentId: envId on: strm .
^ strm contents 
%


method:
fileOutMethod: selector on: stream
 ^ self fileOutMethod: selector environmentId: 0 on: stream 
%

method:
fileOutMethod: selector environmentId: envArg on: stream

"Writes the given method's source to the given stream in Topaz Filein format."

| lf cat src envId |
envId := envArg < 0 ifTrue:[ 0 - envArg ] ifFalse:[ envArg ].
self _fileoutHeaderOn: stream  .
src := self sourceCodeAt: selector environmentId: envId .
((src _isOneByteString not) 
    and:[ (stream dynamicInstVarAt: #utf8Bool) == false ]) ifTrue:[
   src _asString ifNotNil:[ :s | src := s "source ok" ]
                 ifNil:[ Error signal: 'source for ' , selector printString, 
           ' contains codePoint > 255  when 8 bit fileout requested' ]
].
lf := Character lf.
cat := self categoryOfSelector: selector environmentId: envId.
cat ifNil:[ 
  ^ self _error: #rtErrKeyNotFound args: { selector }
].
stream nextPutAll: 'category: '''; _fileOutAll: cat; nextPut: $'; nextPut: lf .
envArg > 0 ifTrue:[
  stream nextPutAll:'set compile_env: '; nextPutAll: envId asString ; nextPut: lf .
].
stream
  nextPutAll: (self isMeta ifTrue: [ 'classmethod: ' ]
              ifFalse: [ 'method: ' ]) ;
  _fileOutAll: self thisClass name; nextPut: lf .
stream _fileOutAll: src.
src last == lf ifFalse: [
  stream nextPut: lf
].
stream nextPut:  $% ; nextPut: lf .
envArg > 0 ifTrue:[
  stream nextPutAll:'set compile_env: 0'; nextPut: lf .
].
%

category: 'Fileout'
method:
fileOutMethodRemovalOn: stream name: clsname

^ self fileOutMethodRemovalOn: stream name: clsname environmentId: 0
%

category: 'Fileout'
method:
fileOutMethodRemovalOn: stream name: clsname environmentId: envId

"Writes code to remove all the receiver's methods onto the given stream
 in filein format."

| theSend |
theSend := ' removeAllMethods' , 
          (envId == 0 ifTrue:[ '.' ] ifFalse:[ ': ' , envId asString , '.']) .
self _fileoutHeaderOn: stream  .
stream nextPutAll: '! ------------------- Remove existing behavior from ';
  _fileOutAll: clsname; lf ;
  "keep trailing spaces on next line for diff compatibility with previous releases"
  nextPutAll: 'expectvalue /Metaclass3       
doit
';
  _fileOutAll: clsname;
    nextPutAll: theSend ; lf ;
  _fileOutAll: clsname;
    nextPutAll: ' class' ; nextPutAll: theSend ; lf ;
  nextPut: $% ; lf .
%

category: 'Fileout'
method:
fileOutMethods

"Returns a String or MultiByteString with all the receiver's methods 
 in Topaz Filein format."

| strm |
strm := AppendStream on: String new.
self fileOutMethodsOn: strm .
^ strm contents 
%

category: 'Fileout'
method:
fileOutMethodsOn: stream 

^ self fileOutMethodsOn: stream environmentId: 0 
%

category: 'Fileout'
method:
fileOutMethodsOn: stream environmentId: envId

"File out this class's methods, but sort the selectors alphabetically."

| cls sels nm |
self _fileoutHeaderOn: stream  .
stream ifNil: [^self].
cls := self thisClass.
nm := cls name.

self fileOutMethodRemovalOn: stream name: nm environmentId: envId .
self fileOutPreMethodsOn: stream environmentId: envId .

stream nextPutAll:'set compile_env: '; nextPutAll: envId asString ; lf .
stream nextPutAll: '! ------------------- Class methods for '; _fileOutAll: nm; lf.
sels := SortedCollection withAll: (cls class selectorsForEnvironment: envId ).
1 to: sels size do: [:i |
  cls class fileOutMethod: (sels at: i)  environmentId: 0 - envId on: stream .
].
stream nextPutAll: '! ------------------- Instance methods for '; _fileOutAll: nm; lf.
sels := SortedCollection withAll: (cls selectorsForEnvironment: envId ).
1 to: sels size do: [:i |
  cls fileOutMethod: (sels at: i) environmentId: 0 - envId on: stream .
].
self fileOutPostMethodsOn: stream.
envId ~~ 0 ifTrue:[ stream nextPutAll:'set compile_env: 0' ; lf ].
^stream
%

category: 'Fileout'
method:
fileOutPostMethodsOn: stream 

 ^ self fileOutPostMethodsOn: stream environmentId: 0
%

category: 'Fileout'
method:
fileOutPostMethodsOn: stream environmentId: envId

"This method gives classes an opportunity to file out information not
 normally emitted by the predefined fileout methods.  Classes may override
 this method (as a class method, of course) to add extra code to the
 output stream.  Emitted code should be in Topaz filein format.  It will
 be placed after method creation."

%

category: 'Fileout'
method:
fileOutPreClassOn: stream

"This method gives classes an opportunity to file out information not
 normally emitted by the predefined fileout methods.  Classes may override
 this method (as a class method, of course) to add extra code to the
 output stream.  Emitted code should be in Topaz filein format.  It will
 be placed before any other fileout information for the class."

%

category: 'Fileout'
method:
fileOutPreMethodsOn: stream 

^ self fileOutPreMethodsOn: stream environmentId: 0
%
category: 'Fileout'
method:
fileOutPreMethodsOn: stream environmentId: envId

"This method gives classes an opportunity to file out information not
 normally emitted by the predefined fileout methods.  Classes may override
 this method (as a class method, of course) to add extra code to the
 output stream.  Emitted code should be in Topaz filein format.  It will
 be placed after existing method removal and before method creation."

%

