!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   ByteArray, SequenceableCollection, Collection, Object.
!
!=========================================================================

! class ByteArray is created in bom

set class ByteArray
removeallmethods 
removeallclassmethods 

category: 'For Documentation Installation only'
classmethod:
installDocumentation

self comment: 
'A ByteArray is a SequenceableCollection whose elements are SmallIntegers with
 a value between zero and 255 inclusive.  Uninitialized ByteArray elements are
 zero.' .

%
! fromString:   Gs64 v2.2, changed to allow DoubleByteString arg to fix 36126
category: 'instance creation'
classmethod:
fromString: aString

"aString must be a kind of String or MultiByteString.  For MultiByteString
 arguments, the result will be in big-endian byte order."

<primitive: 653>
aString _validateClasses:{ String }.
self _primitiveFailed: #fromString: args: { aString }
%

category: 'Comparing'
method:
= aByteArray

"Returns true if all of the following conditions are true, otherwise
  returns false.

 1.  The receiver and aByteArray are of the same class.
 2.  The two Arrays are the same size.
 3.  The corresponding elements of the receiver and aByteArray
     are equal."

<primitive: 613>

^ super = aByteArray.
%

set compile_env: 0
category: 'Copying'
method: ByteArray
copyReplaceFrom: startIndex to: stopIndex with: aSequenceableCollection
	"See SequenceableCollection>>copyReplaceFrom:to:with: for details"

	^super
		copyReplaceFrom: startIndex
		to: stopIndex
		with: aSequenceableCollection asByteArray.
%

! fix 46127
category: 'Copying'
method:
replaceFrom: startIndex to: stopIndex with: aSeqCollection startingAt: repIndex

"Replaces the elements of the receiver between the indexes startIndex and
 stopIndex inclusive with the elements of aSeqCollection starting at repIndex.
 If aSeqCollection is identical to the receiver, the source and
 destination blocks may overlap.

 The primitive supports directly the case where 
    (aSeqCollection isKindOfClass: CByteArray) == true , 
 with repIndex being one-based .

 Returns the receiver."

<primitive: 297>
startIndex _isSmallInteger ifFalse:[ startIndex _validateClass: SmallInteger ].
stopIndex _isSmallInteger ifFalse:[ stopIndex _validateClass: SmallInteger ].
repIndex _isSmallInteger ifFalse:[ repIndex _validateClass: SmallInteger ].
(aSeqCollection stringCharSize >= 2) ifTrue:[
  "aSeqCollection is a DoubleByteString or QuadByteString"
  aSeqCollection _error: #rtErrInvalidArgClass args: { String . ByteArray }
].
^ super replaceFrom: startIndex to: stopIndex with: aSeqCollection startingAt: repIndex
%

! fixed 31357
category: 'Copying'
method:
insertAll: aByteArray at: anIndex

"Inserts aByteArray into the receiver beginning at anIndex.  The receiver's
 size is modified to become its old size plus the size of aByteArray.

 The argument anIndex must be greater than or equal to 1.  If anIndex is 1
 greater than the size of the receiver, appends aByteArray to the receiver.  If
 anIndex is more than 1 greater than the size of the receiver, generates an
 error."

<primitive: 230>

aByteArray _validateClass: ByteArray.
(anIndex _isSmallInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
(anIndex < 1 or: [ anIndex > (self size + 1)]) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].

^ super insertAll: aByteArray at: anIndex
%

category: 'Adding'
method:
addAll: aCollection

"Adds all of the elements of aCollection to the receiver and returns
 aCollection."

| collectionSize |

(self == aCollection) ifTrue: [ ^ self addAll: (aCollection copy) ].

(aCollection isKindOf: ByteArray) ifTrue:[
  collectionSize := aCollection size.
  (collectionSize ~~ 0) ifTrue:[
    self insertAll: aCollection at: (self size + 1)
    ].
  ^ aCollection.
  ].
aCollection accompaniedBy: self do: [ :me :each | me add: each ].
^ aCollection
%

category: 'Comparing'
method:
hash

"Returns a positive SmallInteger based on the contents of the receiver."

"Uses a case-sensitive string hash algorithm.
 The algorithm implemented is described in:

 [Pearson 90]
 Pearson, Peter K., Fast Hashing of Variable-Length Text Strings,
 Communications of the ACM 33, 6, (June 1990), 677-680."

<primitive: 31>
self _primitiveFailed: #hash .
self _uncontinuableError
%

category: 'Converting'
method:
asHexString

"Returns a String containing a hexadecimal printed representation of the
 contents of the receiver.  For example, the message 'abc' asHexString
 returns the String '616263'.

 The receiver must be a byte format object."

<primitive: 467>
self _validateByteClass: ByteArray .
self _primitiveFailed: #asHexString .
%
method: 
asUnicodeString

"This will eventually be Deprecated.  
 New code should use bytesIntoUnicode. 
 Return an instance of Unicode7 or Unicode16
 using the class with smallest character size needed to
 represent a copy of the receiver. 
 The receiver is interpreted as an array of 8 bit codePoints."

<primitive: 927>

self _primitiveFailed:#asUnicodeString
%

method:
bytesIntoUnicode
"Return an instance of Unicode7 or Unicode16
 using the class with smallest character size needed to
 represent a copy of the receiver.  
 The receiver is interpreted as an array of 8 bit codePoints."

<primitive: 927>

self _primitiveFailed: #bytesIntoUnicode
%

method: 
_asUnicode7

<primitive: 925>
"Return an instance of Unicode7 if receiver can be represented 
 as a Unicode7 string, else return nil."

self _primitiveFailed:#_asUnicode7
%
method: 
_asUnicode16

<primitive: 926>

"Return an instance of Unicode7 or Unicode16 using the 
minimum bytes per character required to represent the receiver.
Return nil if the receiver is not representable as Unicode7 nor Unicode16."

self _primitiveFailed:#_asUnicode16
%



category: 'Accessing'
method:
charAt: index

"Returns a (single byte) Character located at index in the receiver."

<primitive: 500>

(index _isInteger)
  ifFalse: [^ self _errorNonIntegerIndex: index].

((index <= 0) or: [ (index > self size) ])
  ifTrue: [^ self _errorIndexOutOfRange: index].

self _validateByteClass: ByteArray .
self _primitiveFailed: #charAt: args: { index }
%

category: 'Accessing'
method:
doubleByteCharAt: index

"Returns a double-byte Character located at index in the receiver. The most
 significant byte would be the one located at index, the least significant 
 one located at index + 1."

<primitive: 501>

(index _isInteger)
  ifFalse: [^ self _errorNonIntegerIndex: index].

((index <= 0) or: [ (index >= self size) ])
  ifTrue: [^ self _errorIndexOutOfRange: index].

self _validateByteClass: ByteArray .
self _primitiveFailed: #doubleByteCharAt: args: { index }
%

category: 'Accessing'
method:
quadByteCharAt: index

"Returns a quad-byte Character located at index in the receiver. The most
 significant byte would be the one located at index, the least significant 
 one located at index + 3."

<primitive: 703>

(index _isInteger)
  ifFalse: [^ self _errorNonIntegerIndex: index].

((index <= 0) or: [ (index >= self size) ])
  ifTrue: [^ self _errorIndexOutOfRange: index].

self _validateByteClass: ByteArray .
self _primitiveFailed: #quadByteCharAt: args: { index }
%

category: 'Updating'
method:
at: index putChar: char

"Stores the codepoint of char in the receiver at index. Stores one byte if
 char is a single byte character, two bytes if it is a double byte 
 character. , 4 bytes if it is a quad byte character. Returns char."

<primitive: 502>

(index _isInteger)
  ifFalse: [^ self _errorNonIntegerIndex: index].

((index <= 0) or: [ (index > (self size + 1)) ])
  ifTrue: [^ self _errorIndexOutOfRange: index].

char _validateClass: Character .
self _validateByteClass: ByteArray .
self _primitiveFailed: #at:putChar: args: { index . char }
%

category: 'Updating'
method:
at: anIndex put: aValue

"Store aValue in the receiver at anIndex .
 aValue should be a SmallInteger >= 0 and <= 255 . "
<primitive: 1002>

(anIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
((anIndex < 1) | (anIndex > (self size + 1))) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].
(aValue _isSmallInteger not or: [ (aValue < 0) | (aValue > 255) ])
  ifTrue: [^ aValue _error: #rtErrExpectedByteValue].

self _primitiveFailed: #at:put: args: { anIndex . aValue } .
self _uncontinuableError
%
category: 'Updating'
method:
byteAt: anIndex put: aValue

"Store aValue in the receiver at anIndex .
 aValue should be a SmallInteger >= 0 and <= 255 . "
<primitive: 1002>

(anIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
((anIndex < 1) | (anIndex > (self size + 1))) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].
(aValue _isSmallInteger not or: [ (aValue < 0) | (aValue > 255) ])
  ifTrue: [^ aValue _error: #rtErrExpectedByteValue].

self _primitiveFailed: #at:put: args: { anIndex . aValue } .
self _uncontinuableError
%

category: 'Accessing'
method:
bitAtZ: zeroBasedOffset 
  "Returns a bit of the receiver, as if receiver is an Array of bytes .
   ((zeroBasedOffset bitShift: -3)+1)  defines the byte addressed in the receiver
    Within that byte,  bitAt:((zeroBasedOffset bitAnd: 7) + 1) is modified.
    Returns 0 or 1 ."
  | byte ofs  mask |
  ofs  := (zeroBasedOffset bitShift: -3) + 1 .
  byte := self at: ofs  .
  mask := 1 bitShift:( zeroBasedOffset bitAnd: 7 ).
  ^ (byte bitAnd: mask) == 0 ifTrue:[ 0 ] ifFalse:[ 1]
%

category: 'Updating'
method:
bitAtZ: zeroBasedOffset put: anInteger
  "Stores a bit into the receiver, as if receiver is an Array of bytes .
   ((zeroBasedOffset bitShift: -3)+1)  defines the byte addressed in the receiver.
    Within that byte,  bitAt:((zeroBasedOffset bitAnd: 7) + 1) is modified.
    anInteger must be one of  0, 1, true, false ."
  | byte ofs  mask |
  ofs  := (zeroBasedOffset bitShift: -3) + 1 .
  byte := self at: ofs  .
  mask := 1 bitShift:( zeroBasedOffset bitAnd: 7 ).
  (anInteger == 0 or:[ anInteger == false]) ifTrue:[
    mask := 16rFF bitXor: mask .
    byte := byte bitAnd: mask .      
    self at: ofs put: byte .
    ^ self .
  ].
  (anInteger == 1 or:[ anInteger == true]) ifTrue:[
    byte := byte bitOr: mask . 
    self at: ofs put: byte .
    ^ self .
  ].
  ArgumentError signal: anInteger asString , '  is not one of  0, 1, true, false'
%


category: 'Updating'
method:
_basicAt: anIndex put: aValue

"Store aValue in the receiver at anIndex .
 aValue should be a SmallInteger >= 0 and <= 255 . "
<primitive: 1002>

(anIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
((anIndex < 1) | (anIndex > (self size + 1))) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].
(aValue _isSmallInteger not or: [ (aValue < 0) | (aValue > 255) ])
  ifTrue: [^ aValue _error: #rtErrExpectedByteValue].

self _primitiveFailed: #at:put: args: { anIndex . aValue } .
self _uncontinuableError
%

category: 'Removing'
method:
removeFrom: startIndex to: stopIndex

"Deletes the elements of the receiver from startIndex to stopIndex.  The
 size of the receiver is decreased by stopIndex - startIndex + 1.

 Both startIndex and stopIndex must be positive integers not larger than the
 size of the receiver, with startIndex <= stopIndex."

<primitive: 386>

(stopIndex < startIndex)
ifTrue:
   [ ^ startIndex _error: #rtErrBadCopyFromTo args: { stopIndex }].
((stopIndex > self size) or: [(stopIndex < 1)])
   ifTrue: [ ^ self _errorIndexOutOfRange: stopIndex].
(startIndex < 1)
   ifTrue: [ ^ self _errorIndexOutOfRange: startIndex].
^ self _primitiveFailed: #removeFrom:to: args: { startIndex . stopIndex }
%

category: 'Private'
method:
at: index put: aString fromOffset: stringOffset sizeBytes: numBytes
 "Write a string into the receiver, which must be a byte object (usually
  a ByteArray).  The offset to start writing the receiver is given by the
  at: argument, the object to write (must also be a byte object, usually
  a string) is given by the put: argument.  The fromOffset: argument 
  specifies the offset from within the given string to start copying.
  The sizeBytes argument specifies the width it takes to store the size
  of the string.  The size of the string is stored in the first offset
  of the byte array.  The sizeBytes must one of the following values:
 
  sizeBytes   max string size      Notes
     0        255                  Do not store the size in the byte array
     1        255
     2        65535
     4        4 GB - 1
     8        max legal object size - 8.
  Returns the receiver."

<primitive: 622>
self _primitiveFailed: #at:put:fromOffset:sizeBytes:
     args: { index . aString . stringOffset . numBytes }
%

category: 'Private'
method:
at: index put: aNumber signed: aBool width: aWidthInBytes

"Store the big-endian represention of an aNumber into
 the specified position in the receiver.
 aWidthInBytes is allowed to be 1,2,3,4 or 8 if aNumber is
 an Integer .
 aWidthInBytes is allowed to be 4 , or 8 if aNumber is
 a BinaryFloat  , in which case the absolute value of aWidthInBytes
 specifies the width to be stored and aBool is ignored.
 If representation of an Integer requires more than aWidthInBytes ,
 the primitive will fail. 
 If coercion of a BinaryFloat to a 4 byte C float would produce
 loss of exponent bits, the primitive will fail.   
 If coercion of a BinaryFloat to a 4 byte C float would cause loss
 of precision that would convert a non-zero value to zero , 
 the primitive will fail .
 "

<primitive: 618>
index _validateClass: SmallInteger.
aNumber _validateClass: Number.
aBool _validateClass: Boolean.
aWidthInBytes _validateClass: SmallInteger.
(aWidthInBytes < 1) ifTrue: [ self error: 'aWidthInBytes negative or zero'].
(aWidthInBytes > 4) and: [ (aWidthInBytes ~= 8) 
              ifTrue: [ self error: 'aWidthInBytes > 4 and ~= 8']].
self _primitiveFailed: #at:put:signed:width: 
     args: { index . aNumber . aBool . aWidthInBytes }
%
category: 'Private'
method:
at: index signed: aBool width: aWidthInBytes

"Retrieve a Number stored in  big-endian represention ,
 from the specified position and width in the receiver.
 aWidthInBytes of 1,2,3,4 or 8  retrieves an Integer.
 aWidthInBytes of -4 or -8 retrieves a SmallDouble or Float,
 and aBool is ignored.  "

<primitive: 619>
self _primitiveFailed: #at:signed:width: 
     args: { index . aBool . aWidthInBytes }
%

category: 'Private'
method:
at: index sizeBytes: anInt stringSize: anIntOrNil
 "Read a string from the receiver into a new string and return the new
  string.  The offset to start writing the receiver is given by the
  at: argument. The sizeBytes argument specifies the width (in bytes) it takes to
  store the size of the string.  In this case, it is assumed the size
  of the string is stored in the first sizeBytes of the receiver. The
  stringSize: argument should be nil in this case, which means read the 
  string size from the byte array.  If the stringSize: argument is not nil,
  it means the size is not encoded in the string and the caller is
  passing in the string size.
  Returns the new string object created from reading"

<primitive: 623>
self _primitiveFailed: #at:sizeBytes:stringSize:
     args: { index . anInt . anIntOrNil }
%

category: 'Private-Accessing'
method:
dateTime32At: startIndex

"retrieves a DateTime that was stored with seconds resolution 
 in big-endian byte order."

^ self dateTimeAt: startIndex width: 4 
%
category: 'Private-Accessing'
method:
dateTime32NativeAt: startIndex

"retrieves a DateTime that was stored with seconds resolution
 in the gem process's native byte order.  
 Provided for compatiblity and renamed from dateTimeAsUnsigned32At: "

^ self dateTimeAt: startIndex width: -4 
%

category: 'Private-Accessing'
method:
dateTime64At: startIndex

"retrieves a DateTime that was stored with millisecond resolution.
 in big-endian byte order"
 
^ self dateTimeAt: startIndex width: 8 
%

category: 'Private-Accessing'
method:
dateTimeAt: startIndex width: anInt 

"values for anInt:
   -4   second resolution (4 bytes) in gem process native byte order
    4   second resolution (4 bytes) in big-endian byte order
    8   millisecond resolution (8 bytes) in big-endian byte order"

<primitive: 621>
self _primitiveFailed: #dateTimeAt:width: args: { startIndex . anInt }
%

category: 'Private-Accessing'
method:
shortStringAt: startIndex

^self at: startIndex sizeBytes:  1 stringSize: nil
%
category: 'Private-Accessing'
method:
signed16At: startIndex

^self at: startIndex signed: true width: 2
%
category: 'Private-Accessing'
method:
signed24At: startIndex

^self at: startIndex signed: true width: 3
%
category: 'Private-Accessing'
method:
signed32At: startIndex

^self at: startIndex signed: true width: 4
%
category: 'Private-Accessing'
method:
signed64At: startIndex

^self at: startIndex signed: true width: 8
%
category: 'Private-Accessing'
method:
signed8At: startIndex

^self at: startIndex signed: true width: 1
%
category: 'Private-Accessing'
method:
string4gAt: startIndex

^self at: startIndex sizeBytes: 4 stringSize: nil
%
category: 'Private-Accessing'
method:
string64kAt: startIndex

^self at: startIndex sizeBytes: 2 stringSize: nil
%
category: 'Private-Accessing'
method:
stringOfSize: anInt at: startIndex

^self at: startIndex sizeBytes:  0 stringSize: anInt
%
category: 'Private-Accessing'
method:
unsigned16At: startIndex

^self at: startIndex signed: false width: 2
%
category: 'Private-Accessing'
method:
unsigned24At: startIndex

^self at: startIndex signed: false width: 3
%
category: 'Private-Accessing'
method:
unsigned32At: startIndex

^self at: startIndex signed: false width: 4
%
category: 'Private-Accessing'
method:
unsigned64At: startIndex

^self at: startIndex signed: false width: 8
%
category: 'Private-Accessing'
method:
unsigned8At: startIndex

^self at: startIndex signed: false width: 1
%
category: 'Private-Accessing'
method:
floatAt: startIndex

"extract a 4 byte float from the receiver, the result will 
 be a SmallDouble."

^self at: startIndex signed: false width: -4
%
category: 'Private-Accessing'
method:
doubleAt: startIndex

"extract an 8 byte float from the receiver, the result will 
 be either a SmallDouble or a Float. "

^self at: startIndex signed: false width: -8
%

category: 'Private-Accessing'
method:
unsigned64At: startIndex

^self at: startIndex signed: false width: 8
%

category: 'Private-Updating'
method:
dateTime32At: startIndex put: aDateTime

"Stores the DateTime with seconds resolution in big-endian byte order
 WARNING, this truncates the DateTime to seconds resolution, 
 throwing away the milliseconds.  
 "

^ self dateTimeAt: startIndex put: aDateTime width: 4
%

category: 'Private-Updating'
method:
dateTime32NativeAt: startIndex put: aDateTime

"Stores the DateTime with seconds resolution and gem process native
 byte order.
 WARNING, this truncates the DateTime to seconds resolution,
 throwing away the milliseconds.  Provided for compatibility and
 renamed from dateTimeAsUnsigned32At:put:  "

^ self dateTimeAt: startIndex put: aDateTime width: -4
%


category: 'Private-Updating'
method:
dateTime64At: startIndex put: aDateTime

"Stores the DateTime with full millisecond resolution in big-endian
 byte order."

^ self dateTimeAt: startIndex put: aDateTime width: 8
%

category: 'Private-Updating'
method:
dateTimeAt: startIndex put: aDateTime width: anInt

"values for anInt:
   -4   second resolution (4 bytes) in gem process native byte order
    4   second resolution (4 bytes) in big-endian byte order
    8   millisecond resolution (8 bytes) in big-endian byte order"
<primitive: 620>
| yr |
aDateTime _validateClass: DateTime .
(yr := aDateTime yearGmt) < 1970 ifTrue:[
  OutOfRange new name: 'yearGmt' min: 1970 actual: yr ; 
      details: 'year is less than 1970' ; signal
].
self _primitiveFailed: #dateTimeAt:put:width: 
     args: { startIndex . aDateTime . anInt }
%

category: 'Private-Updating'
method:
shortStringAt: startIndex put: aString fromOffset: stringOffset 
	^self at: startIndex put: aString fromOffset: stringOffset sizeBytes: 1
%
category: 'Private-Updating'
method:
signed16At: startIndex put: anInteger

^self at: startIndex put: anInteger signed: true width: 2
%
category: 'Private-Updating'
method:
signed24At: startIndex put: anInteger

^self at: startIndex put: anInteger signed: true width: 3
%
category: 'Private-Updating'
method:
signed32At: startIndex put: anInteger

^self at: startIndex put: anInteger signed: true width: 4
%
category: 'Private-Updating'
method:
signed64At: startIndex put: anInteger

^self at: startIndex put: anInteger signed: true width: 8
%
category: 'Private-Updating'
method:
signed8At: startIndex put: anInteger

^self at: startIndex put: anInteger signed: true width: 1
%
category: 'Private-Updating'
method:
string4gAt: startIndex put: aString fromOffset: stringOffset 
	^self at: startIndex put: aString fromOffset: stringOffset sizeBytes: 4
%
category: 'Private-Updating'
method:
string64kAt: startIndex put: aString fromOffset: stringOffset 
	^self at: startIndex put: aString fromOffset: stringOffset sizeBytes: 2
%
category: 'Private-Updating'
method:
stringAt: startIndex put: aString fromOffset: stringOffset 
	^self at: startIndex put: aString fromOffset: stringOffset sizeBytes: 0
%
category: 'Private-Updating'
method:
unsigned16At: startIndex put: anUnsignedInt

^self at: startIndex put: anUnsignedInt signed: false width: 2
%
category: 'Private-Updating'
method:
unsigned24At: startIndex put: anUnsignedInt

^self at: startIndex put: anUnsignedInt signed: false width: 3
%
category: 'Private-Updating'
method:
unsigned32At: startIndex put: anUnsignedInt

^self at: startIndex put: anUnsignedInt signed: false width: 4
%
category: 'Private-Updating'
method:
unsigned64At: startIndex put: anUnsignedInt

^self at: startIndex put: anUnsignedInt signed: false width: 8
%
category: 'Private-Updating'
method:
doubleAt: startIndex put: aBinaryFloat

"aBinaryFloat must be a kind of BinaryFloat."

^self at: startIndex put: aBinaryFloat signed: false width: 8
%
category: 'Private-Updating'
method:
floatAt: startIndex put: aBinaryFloat

"aBinaryFloat must be a kind of BinaryFloat, representable as
 a 4 byte IEEE float without loss of exponent bits. 
 If coercion of a BinaryFloat to a 4 byte float would produce
 loss of exponent bits, the primitive will fail.   
 If coercion of a BinaryFloat to a 4 byte float would cause loss
 of precision that would convert a non-zero value to zero , 
 the primitive will fail ."

^self at: startIndex put: aBinaryFloat signed: false width: 4
%

category: 'Private-Updating'
method:
unsigned8At: startIndex put: anUnsignedInt

^self at: startIndex put: anUnsignedInt signed: false width: 1
%
! fixed 31356
category: 'Private-Updating'
method:
_deleteNoShrinkFrom: startIndex to: endIndex anchorTailSize: aSize

"Deletes bytes from startIndex to endIndex by sliding bytes
 from endIndex+1 to aSize-1 to the left and filling with zeros.
 The bytes from  (self size - aSize) to (self size) are not altered. 
 The size of the receiver is not changed.  
 The size of the receiver must be <= 65536 ."

<primitive: 626>

(startIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: startIndex].
(endIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: endIndex].
(aSize _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: aSize].
self size > 65536 ifTrue:[ self error:'receiver size > 65536' ].
startIndex < 1 ifTrue:[ self _errorIndexOutOfRange:startIndex  ] .
(endIndex < startIndex or:[ endIndex > self size] ) ifTrue:[ self _errorIndexOutOfRange: endIndex].
(aSize < 0 or:[ endIndex > (self size - aSize)]) ifTrue:[ aSize _error: #rtErrArgOutOfRange ].
self _primitiveFailed: #_deleteNoShrinkFrom:to:anchorTailSize:
     args: { startIndex . endIndex . aSize }
%

! fixed 31356
category: 'Private-Updating'
method:
_reverseDeleteNoShrinkFrom: startIndex to: endIndex anchorHeadSize: aSize

"Deletes bytes from startIndex to endIndex.  startIndex must be > aSize,
 and the bytes from 1 to aSize  are not altered.
 The bytes from aSize + 1 to startIndex - are shifted to the right by
 the distance (endIndex - startIndex + 1), and the bytes from (aSize + 1)
 to (endIndex - (startIndex - aSize) +1) are zeroed. 
 The size of the receiver is not changed.  
 The size of the receiver must be <= 65536 ."

<primitive: 627>

(startIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: startIndex].
(endIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: endIndex].
(aSize _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: aSize].
self size > 65536 ifTrue:[ self error:'receiver size > 65536' ].
startIndex < 1 ifTrue:[ self _errorIndexOutOfRange:startIndex  ] .
(endIndex < 1 or:[ endIndex > self size]) ifTrue:[ self _errorIndexOutOfRange:endIndex ].
startIndex < aSize ifTrue:[ self error:'startIndex < aSize' ].
(self size - aSize) < endIndex ifTrue:[ aSize _error: #rtErrArgOutOfRange ].
(endIndex - startIndex + 1) < 0 ifTrue:[ self error: 'numToDelete less than 0' ].
((endIndex - startIndex + 1) > ((self size) - startIndex)) ifTrue:[ self error: 'numToDelete > numDeletable' ].
self _primitiveFailed: #_reverseDeleteNoShrinkFrom:to:anchorHeadSize:
     args: { startIndex . endIndex . aSize }
%

category: 'Private-Comparing'
method:
compareStringAt: startIndex to: aString startingAt: stringIndex sizeBytes: numSizeBytes useCase: aBool

"  Compare a string encoded in the receiver (aByteArray) to the given string
  starting at the offset specified in the given string.  If sizeBytes not
  zero, the byte array is assumed to have the size of the encoded string
  encoded in the first numSizeBytes bytes at the startIndex.  If sizeBytes 
  is zero, the size has not been encoded in the byteArray and the entire 
  length of the second string is compared.  numSizeBytes must be 0, 1, 2, 4, 
  or 8.  <aBool> determines if the comparision is case sensitive.
  Returns true if the strings match, false otherwise."
<primitive: 624>
self _primitiveFailed: #compareStringAt:to:startingAt:sizeBytes:useCase:
     args: { startIndex . aString . stringIndex . numSizeBytes . aBool }
%

category: 'Private-Comparing'
method:
compareStringAt: startIndex to: aString startingAt: stringIndex sizeBytes: numSizeBytes

"  Compare a string encoded in the receiver (aByteArray) to the given string
  starting at the offset specified in the given string.  If sizeBytes not
  zero, the byte array is assumed to have the size of the encoded string
  encoded in the first numSizeBytes bytes at the startIndex.  If sizeBytes 
  is zero, the size   has not been encoded in the byteArray and the entire 
  length of the second string is compared.  numSizeBytes must be 0, 1, 2, 4, 
  or 8.
  Returns true if the strings match, false otherwise."
        ^self compareStringAt: startIndex to: aString startingAt: stringIndex sizeBytes: numSizeBytes useCase: true
%

category: 'Private-Comparing'
method:
at: anIndex equals: aString

^self compareStringAt: anIndex to: aString startingAt: 1 sizeBytes: 0 useCase: true
   "size is not encoded, compare entire <aString>"
%

category: 'Private-Comparing'
method:
at: anIndex equals: aString useCase: aBool

^self compareStringAt: anIndex to: aString startingAt: 1 sizeBytes: 0 useCase: aBool
   "size is not encoded, compare entire <aString>"
%

category: 'Private-Comparing'
method:
compareShortStringAt: startIndex to: aString startingAt: stringIndex

^self compareStringAt: startIndex to: aString startingAt: stringIndex sizeBytes: 1 useCase: true
   "size is encoded in first byte of receiver"
%

category: 'Private-Comparing'
method:
compareCaseInsensitiveShortStringAt: startIndex to: aString startingAt: stringIndex
"  Compare a string encoded in the receiver (aByteArray) to the given string
  starting at the offset specified in the given string.  The byte array is assumed
  to have the size of the encoded string encoded in the first 1 byte.
  Returns true if the strings are case insensitive match, false otherwise."

^self compareStringAt: startIndex to: aString startingAt: stringIndex sizeBytes: 1 useCase: false
    "size is encoded in first byte of receiver"
%

category: 'Private-Comparing'
method:
compareShortStringAt: startIndex to: aString startingAt: stringIndex useCase: aBool

^self compareStringAt: startIndex to: aString startingAt: stringIndex sizeBytes: 1 useCase: aBool
   "size is encoded in first byte of receiver"
%

category: 'Private-Comparing'
method:
shortStringAt: anIndex compareWith: aByteObject startingAt: stringOffset opCode: anOpCode
 "Compares, in order starting at anIndex, if the contents of the receiver
  are less than/greater than the contents of aByteObject, starting at stringOffset.
  The first byte of the receiver is assumed to be the size of the string 
  held in the receiver.  GemStone character precedence is adhered to.
  The argument string must be a small object.
op code == 0 means less than
op code == 1 means greater than"

<primitive: 625>
self _primitiveFailed: #shortStringAt:compareWith:startingAt:opCode:
     args: { anIndex . aByteObject . stringOffset . anOpCode }
%

category: 'Private-Comparing'
method:
shortStringAt: anIndex greaterThan: aByteObject startingAt: stringOffset

 " Checks, in order starting at anIndex, if the contents of the receiver
  are greater than the contents of aByteObject, starting at stringOffset.
  The first byte of the receiver is assumed to be the size of the string 
  held in the receiver.  
 GemStone character precedence is adhered to.
  The argument string must be a small object."

^self shortStringAt: anIndex compareWith: aByteObject startingAt: stringOffset opCode: 1
%

category: 'Private-Comparing'
method:
shortStringAt: anIndex lessThan: aByteObject startingAt: stringOffset

 " Checks, in order starting at anIndex, if the contents of the receiver
  are less than the contents of aByteObject, starting at stringOffset.
  The first byte of the receiver is assumed to be the size of the string 
  held in the receiver.  
 GemStone character precedence is adhered to.
  The argument string must be a small object."

^self shortStringAt: anIndex compareWith: aByteObject startingAt: stringOffset opCode: 0
%

category: 'Private-Comparing'
method:
compareString4gAt: startIndex to: aString startingAt: stringIndex

^self compareStringAt: startIndex to: aString startingAt: stringIndex sizeBytes: 4
  "size is encoded in first byte of receiver"
%

category: 'Private-Comparing'
method:
compareString64kAt: startIndex to: aString startingAt: stringIndex

^self compareStringAt: startIndex to: aString startingAt: stringIndex sizeBytes: 2
   "size is encoded in first byte of receiver"
%

category: 'Private-Updating'
method:
deleteIndexKeyAt: anIndex

<primitive: 630>
self _primitiveFailed: #deleteIndexKeyAt: args: { anIndex }

%

!################################################
! Start of new methods for GemStone 2G release
!################################################

category: 'Encoded OOPs'
method:
at: aSmallInt putOopValueOfObject: anObject
"Store the 8 byte OOP of anObject in the receiver.  
 aSmallInt is an offset in the receiver where the bytes are to be stored.
 The bytes are stored in big-endian order.
 The receiver is grown if needed (use aSmallInt one past end to append)"
<primitive: 633>
aSmallInt _validateClass: SmallInteger .
aSmallInt < 1 ifTrue:[self _errorIndexOutOfRange: aSmallInt].
aSmallInt > (self size + 1) ifTrue:[self _errorIndexOutOfRange: aSmallInt].
self _primitiveFailed: #at:putOopValueOfObject: args: { aSmallInt . anObject }
%

category: 'Encoded OOPs'
method:
at: aSmallInt putOldOopValueOfObject: anObject
"Store the 4 byte Gs64 v1.1 OOP of anObject in the receiver.  
 aSmallInt is an offset in the receiver where the bytes are to be stored.
 The bytes are stored in big-endian order.
 The receiver is grown if needed (use aSmallInt one past end to append)
 anObject cannot be a special object, and its oopNumber must be < 2 billion."

<primitive: 634>
aSmallInt _validateClass: SmallInteger .
aSmallInt < 1 ifTrue:[self _errorIndexOutOfRange: aSmallInt].
aSmallInt > (self size + 1) ifTrue:[self _errorIndexOutOfRange: aSmallInt].
self _primitiveFailed: #at:putOldOopValueOfObject:
     args: { aSmallInt . anObject }
%

category: 'Encoded OOPs'
method:
nextPutOopOfObject: anObject
"Extend the receiver by 8 bytes and add the 8 byte OOP of anObject."

^self at: self size + 1 putOopValueOfObject: anObject
%

category: 'Encoded OOPs'
method:
nextPutOldOopOfObject: anObject
"Extend the receiver by 4 bytes and add the 4 byte OOP of anObject.
 anObject cannot be a special object, and its oopNumber must be < 2 billion."

^self at: self size + 1 putOldOopValueOfObject: anObject
%

category: 'Encoded OOPs'
method:
getObjectWithOopValueAt: anOffset
"Answer the object with the OOP stored in 8 bytes of the receiver 
 at the given offset.
 The bytes are expected to have been stored in big-endian order.
 Returns nil if the object identifier
 encoded at anOffset does not exist or is an invalid object identifier."

<primitive: 635>
anOffset _validateClass: SmallInteger.
(anOffset < 1) ifTrue:[self _errorIndexOutOfRange: anOffset].
(anOffset > (self _basicSize - (self class bytesPerOop - 1)))
  ifTrue:[self _errorIndexOutOfRange: anOffset].
self _primitiveFailed: #getObjectWithOopValueAt: args: { anOffset }
%

category: 'Encoded OOPs'
method:
getObjectWithOldOopValueAt: anOffset
"Answer the object with the OOP stored in 4 bytes of the receiver
 at the given offset.
 The bytes are expected to have been stored in big-endian order.
 Returns nil if the object does not exist or is an invalid object identifier."
<primitive: 636>
anOffset _validateClass: SmallInteger.
(anOffset < 1) ifTrue:[self _errorIndexOutOfRange: anOffset].
(anOffset > (self _basicSize - (self class bytesPerOop - 1)))
  ifTrue:[self _errorIndexOutOfRange: anOffset].
self _primitiveFailed: #getObjectWithOldOopValueAt: args: { anOffset }
%


category: 'Encoded OOPs'
method:
put: anInt objectsWithOopValueAt: anOffset into: anArray
"Extract a given number of objects from the OOPs stored in the receiver 
 starting at the given offset. Each OOP extracted occupies 8 bytes
 in the receiver. Store the objects into the specified Array object.  
 A nil is stored in the array for each object which does not exist.
 The specified Array is set to size 0 at the start of the operation."

1 to: anInt do:[:j | | obj |
  obj := self getObjectWithOopValueAt: j .
  anArray at: j put: obj .
  ].
anArray size: anInt .
^ self
%

category: 'Encoded OOPs'
method:
put: anInt objectsWithOldOopValueAt: anOffset into: anArray
"Extract a given number of objects from the OOPs stored in the receiver 
 starting at the given offset. Each OOP extracted occupies 4 bytes
 in the receiver. Store the objects into the specified Array object.  
 A nil is stored in the array for each object which does not exist.
 The specified Array is set to size 0 at the start of the operation."

1 to: anInt do:[:j | | obj |
  obj := self getObjectWithOldOopValueAt: j .
  anArray at: j put: obj .
  ].
anArray size: anInt .
^ self
%

category: 'Encoded OOPs'
method:
putObjectsWithOopValuesAtIndices: anArrayOfOffsets into: anArray
"For each offset stored in anArrayOfOffsets, extract the OOP from the 
 receiver stored at that offset.  Then store the object with that OOP in
 anArray.  A nil is stored in anArray for each object which does not exist.
 Each OOP extracted occupies 8 bytes in the receiver.
 The specified Array is set to size 0 at the start of the operation."

| limit |
limit := anArrayOfOffsets size .
1 to: limit do:[:j | | obj |
   obj := self getObjectWithOopValueAt: (anArrayOfOffsets at: j) .
   anArray at: j put: obj .
  ].
anArray size: limit .
^ self
%

category: 'Encoded OOPs'
method:
putObjectsWithOldOopValuesAtIndices: anArrayOfOffsets into: anArray
"For each offset stored in anArrayOfOffsets, extract the OOP from the 
 receiver stored at that offset.  Then store the object with that OOP in
 anArray.  A nil is stored in anArray for each object which does not exist.
 Each OOP extracted occupies 4 bytes in the receiver.
 The specified Array is set to size 0 at the start of the operation."

| limit |
limit := anArrayOfOffsets size .
1 to: limit do:[:j | | obj |
   obj := self getObjectWithOldOopValueAt: (anArrayOfOffsets at: j) .
   anArray at: j put: obj .
  ].
anArray size: limit .
^ self
%

category: 'Encoded OOPs'
method:
at: startIndex putAllOopsOfObjects: anArray
"Store the OOP of each object contained in anArray into 8 bytes of the receiver
 starting at startIndex.  The receiver is grown as necessary."
| idx | 
idx := startIndex .
1 to: anArray size do:[:j |
  self at: idx putOopValueOfObject: (anArray at: j ).
  idx := idx + 8 .
  ]. 
^ self
%

category: 'Encoded OOPs'
method:
nextPutAllOopsOfObjects: anArray
"For each object in anArray, append the 8 byte OOP to the receiver.
 The receiver is grown by 8 bytes for every object contained in
 anArray."

^ self at: self size + 1 putAllOopsOfObjects: anArray
%

category: 'Encoded OOPs'
method:
at: startIndex putAllOldOopsOfObjects: anArray
"Store the OOP of each object contained in anArray into 4 bytes of the receiver
 starting at startIndex.  The receiver is grown as necessary."
| idx | 
idx := startIndex .
1 to: anArray size do:[:j |
  self at: idx putOldOopValueOfObject: (anArray at: j ).
  idx := idx + 4 .
  ]. 
^ self
%

category: 'Encoded OOPs'
method:
nextPutAllOldOopsOfObjects: anArray
"For each object in anArray, append the 4 byte OOP to the receiver.
 The receiver is grown by 4 bytes for every object contained in
 anArray."

^ self at: self size + 1 putAllOldOopsOfObjects: anArray
%

category: 'Encoded OOPs'
classmethod:
bytesPerOop
"Number of bytes consumed by OopType in C."
^ 8
%
category: 'Encoded OOPs'
classmethod:
oldBytesPerOop
"Number of bytes consumed by Gs64 v1.1 OopType in C."
^ 4
%

category: 'Compression'
method:
_compressBytes

^ self _compress
%

method:
_decompressBytes

^ self _decompress
%

method:
_decompress

"Decompress receiver using zlib.so .
 If receiver contains a dynamic instVar #sourceClass ,
 then the value of that instVar is used as the class of the result.
 otherwise the result is an instance of receiver's class."

  ^self _decompress: 0
%

method:
_computeCRC32: crc
"Compute the crc32 value for the receiver, given the starting value.
 The crc32 value for a given receiver should be the same both before
 and after compression.  
 The argument and result are 32bit unsigned values."

<primitive: 649>
crc _validateInstanceOf: SmallInteger .
(crc < 0 or:[ crc > 16rFFFFFFFF ]) ifTrue:[ 
  OutOfRange new name: 'initialCrc32' min: 0 max: 16rFFFFFFFF actual: crc; signal
].
self _primitiveFailed: #_computeCRC32: args: { crc }
%

category: 'Copying'
method:  
copyFrom: startIndex to: stopIndex

"Returns a new Array containing the elements of the receiver
 between startIndex and stopIndex, inclusive.  The result is of the same class
 as the receiver.

 If startIndex > stopIndex then an empty collection is returned.
 Otherwise both startIndex and stopIndex must be positive integers not larger than the
 size of the receiver, with startIndex <= stopIndex.
"

<primitive: 818>
(startIndex < 1) ifTrue: [ ^ self _errorIndexOutOfRange: startIndex].

((stopIndex > self size) or: [(stopIndex < 0)])
   ifTrue: [ ^ self _errorIndexOutOfRange: stopIndex].

self _primitiveFailed: #copyFrom:to: args: { startIndex . stopIndex }
%
!
! following CopyReplace* methods for 41165
!
category: 'Copying'
method:
copyReplaceAll: oldSubCollection with: newSubCollection

"See SequenceableCollection>>copyReplaceAll:with: for details"

| old new |
old := oldSubCollection.
new := newSubCollection.
(old isKindOf: ByteArray) ifFalse: [ old := old asByteArray ].
(new isKindOf: ByteArray) ifFalse: [ new := new asByteArray ].
^ super copyReplaceAll: old with: new
%
category: 'Copying'
method:
copyReplaceFrom: startIndex to: stopIndex with: aSequenceableCollection

"See SequenceableCollection>>copyReplaceFrom:to:with: for details"

| col |
col := aSequenceableCollection.
(col isKindOf: ByteArray) ifFalse: [ col := col asByteArray ].
^ super copyReplaceFrom: startIndex to: stopIndex with: col
%
category: 'Copying'
method:
copyReplacing: oldObject withObject: newObject

"Returns a String comprising a copy of the receiver in which all occurrences
 of objects equal to oldObject have been replaced by newObject."

^ super copyReplacing: oldObject asInteger withObject: newObject asInteger
%
category: 'Copying'
method:
copyWith: anObject

"See SequenceableCollection>>copyWith: for details"

^ super copyWith: anObject asInteger
%
category: 'Copying'
method:
copyWithout: anObject

"See SequenceableCollection>>copyWithout: for details"

^ super copyWithout: anObject asInteger 
%

category: 'Encoding'
method: 
decodeFromUTF8
"Decode receiver from UTF8 format. 
 Returns either a Unicode7 , Unicode16 or Unicode32 ,
 using the minimum character size needed to represent decoded result."

^ self _decodeUtf8StartingAt: 1 unicodeResult: true maxSize: nil bytesConsumed: nil
%

method:
decodeFromUTF8ToUnicode
"Decode receiver from UTF8 format. 
 Returns either a Unicode7 , Unicode16 or Unicode32 ,
 using the minimum character size needed to represent decoded result."

^ self _decodeUtf8StartingAt: 1 unicodeResult: true maxSize: nil bytesConsumed: nil
% 

method:
decodeFromUTF8ToString
"Decode UTF8 contents of the receiver returning a String, DoubleByteString or
  QuadByteString."

^ self _decodeUtf8StartingAt: 1 unicodeResult: false maxSize: nil bytesConsumed: nil
%

category: 'Converting'
method:
bytesIntoString

"Returns an instance of String containing the bytes of the receiver
 without doing any decoding."

 ^ String withBytes: self
%

method:
_decodeUtf8At: anOffset bytesConsumed: anArray 

"Attempt to decode one Utf8 codePoint starting at anOffset in the receiver.
 anArray should be a temporary Array of size 1 .
 If successful, returns a Character and (anArray at: 1) is a SmallInteger,
 the number of bytes consumed by the decode.
 If  (self size - anOffset ) are not enough bytes for a decode returns nil.
 Signals an Error if the bytes starting at anOffset are not legal UTF8 data .
 Receiver must be a small ByteArray ( self size <= 16272), otherwise an Error is signalled."

<primitive: 1086>
anOffset _validateClass: SmallInteger .
anArray _validateClass: Array .
^ self _primitiveFailed: #_decodeUtf8At:bytesConsumed: args: { anOffset . anArray }
%

method:
_decodeUtf8StartingAt: anOffset unicodeResult: unicodeResultBool 
                 maxSize: aSize bytesConsumed: anArray

"Attempt to decode Utf8 data starting at anOffset in the receiver.
 If unicodeResultBool == true, result is a Unicode7 , Unicode16 or Unicode32.
 If unicodeResultBool == false, result is a String, DoubleByteString or
  QuadByteString.
 If there are insufficient bytes in the receiver and anArray ~~ nil, 
 returns nil .
  If there are insufficient bytes in the receiver and anArray == nil, signals an Error.
 Signals an Error if the bytes starting at anOffset are not legal UTF8 data .
 If anArray ~~ nil, returns in (anArray at: 1) the number of bytes consumed by the decode."
 
<primitive: 493>
unicodeResultBool _validateClass: Boolean .
aSize _validateClass: SmallInteger .
anArray ifNotNil:[ anArray _validateClass: Array  ].
anOffset _validateClass: SmallInteger .

^ self _primitiveFailed: #_decodeUtf8StartingAt:unicodeResult:maxSize:bytesConsumed:
          args: { anOffset . unicodeResultBool . aSize . anArray }
%

category: 'Private'
method: 
_decodeFromUtf8: unicodeResultBool maxSize: aSize
"Decode UTF8 contents of the receiver.
 If unicodeResultBool == true, result is a Unicode7 , Unicode16 or Unicode32.
 If unicodeResultBool == false, result is a String, DoubleByteString or
  QuadByteString.

 aSize should be nil for no limit on result size, or
 a SmallInteger specifying approximate maximum size of the result."

^ self _decodeUtf8StartingAt: 1 unicodeResult: unicodeResultBool maxSize: aSize bytesConsumed: nil
%

method: 
_decodeFromUtf8: unicodeResultBool

"Decode UTF8 contents of the receiver.
 If unicodeResultBool == true, result is a Unicode7 , Unicode16 or Unicode32.
 If unicodeResultBool == false, result is a String, DoubleByteString or
  QuadByteString."

^ self _decodeFromUtf8: unicodeResultBool maxSize: nil
%

category: 'Private-Accessing'
method:
int32LittleEndianAt: startIndex
  "Receiver can be any byte format object.
   Answer a SmallInteger resulting from interpreting the bytes in the 
   receiver from offset to offset+3 as a little-endian signed integer.
  Offset is 1-based and need not be aligned.
  Variant of signed32At:, for little endian data.
  * The given offset is not a SmallInteger (ArgumentTypeError)
  * offset+3 is outside the size of the receiver (OutOfRange)"
<primitive: 945>
startIndex _validateClass: SmallInteger .
(startIndex < 1 or:[ startIndex + 3 > self size]) ifTrue:[
  OutOfRange new
    name:'startIndex' min: 1 max: self size - 3 actual: startIndex ;
    signal
].
self _primitiveFailed: #int32LittleEndianAt: args: { startIndex }
%

category: 'Private-Updating'
method:
_int32LittleEndianAt: startIndex put: anInteger

"Receiver can be any byte format object.
Does not update any indexed dependent on receiver.
Overwrite the four bytes of the receiver from offset to offset+3 with the low-order four bytes of the given SmallInteger value, in little-endian signed format. 
Offset is 1-based and need not be aligned.
Note that if the given value is a SmallInteger outside the range that can be represented in four bytes, the low-order four bytes will silently be stored.
  Returns anInteger
Errors:
  * The given offset is not a SmallInteger (ArgumentTypeError)
  * Some index from offset to offset+3 is outside the size of the receiver (OutOfRange)
  * The given value is not a SmallInteger (ArgumentTypeError)"


<primitive: 946>
startIndex _validateClass: SmallInteger .
anInteger _validateClass: SmallInteger .
(startIndex < 1 or:[ startIndex + 3 > self size]) ifTrue:[
  OutOfRange new 
   name:'startIndex' min: 1 max: self size - 3 actual: startIndex  ;
   signal
].
self _primitiveFailed: #int32LittleEndianAt:put: args: { startIndex . anInteger}
%
category: 'Searching'
method:
indexOf: searchByte startingAt: startIndex stoppingAt: stopIndex

"Receiver is searched for a byte that has the same value
as the low-order byte (searchByte bitAnd: 16rFF) of SmallInteger 
searchByte. The search starts at 1-based startIndex within the receiver 
and stops when a matching byte is found or 1-based stopIndex is reached. 
The index of the first matching byte is answered, or 0 is answered if no 
matching byte is found.

Errors:
  * ArgumentTypeError if any of the arguments is not a SmallInteger.
  * OutOfRange if startIndex or stopIndex is outside the bounds of
    of the receiver."

<primitive: 947>
searchByte _validateClass: SmallInteger .
startIndex _validateClass: SmallInteger .
stopIndex _validateClass: SmallInteger .
(startIndex < 1 or:[ startIndex > self size]) ifTrue:[
  OutOfRange new 
   name:'startIndex' min: 1 max: self size actual: startIndex  ;
   signal
].
(stopIndex < 1 or:[ stopIndex > self size]) ifTrue:[
  OutOfRange new 
   name:'stopIndex' min: 1 max: self size actual: stopIndex  ;
   signal
].
self _primitiveFailed: #indexOf:startingAt:stoppingAt: 
	args: { searchByte . startIndex . stopIndex}
%

category: 'Private'
method:
_primAddRandomBytes: anIntHowMany startingAt: anIntOffset

<primitive: 954>
anIntHowMany _validateClass: SmallInteger .
anIntOffset  _validateClass: SmallInteger .
anIntHowMany < 1 
  ifTrue:[ anIntHowMany _error: #rtErrArgOutOfRange ] .
anIntOffset < 1 
  ifTrue:[ anIntOffset _error: #rtErrArgOutOfRange ] .
self _primitiveFailed: #_primAddRandomBytes:startingAt: .
%

category: 'Random Value Generation'
method:
addRandomBytes: howMany startingAt: offset

"Adds howMany random bytes to the receiver starting at offset.
 This method grows the receiver to accomodate the random data
 if necessary.   Any existing data in the range of offset to 
 (offset + howMany - 1) is overwritten by this method.

 Both arguments are expected to be positive SmallIntegers.

 Returns the receiver."

^ self _primAddRandomBytes: howMany startingAt: offset
%

category: 'instance creation'
classmethod:
withRandomBytes: howMany

"Returns a new instance of the receiver of size howMany containing
 randomly generated data."

^ self new addRandomBytes: howMany startingAt: 1
%

category: 'Accessing'
method: 
at: anIndex

"Returns the value of an indexed variable in the receiver.
 The argument anIndex must not be larger than the size of the
 receiver, and must not be less than 1.

 Generates an error if anIndex is not a SmallInteger or is out of
 bounds, or if the receiver is not indexable."

<primitive: 974>

(anIndex _isInteger)
  ifTrue: [^ self _errorIndexOutOfRange: anIndex]
  ifFalse: [^ self _errorNonIntegerIndex: anIndex].
self _primitiveFailed: #at: args: { anIndex } .
self _uncontinuableError
%

category: 'Accessing'
method: 
byteAt: anIndex

"Returns the value of an indexed variable in the receiver.
 The argument anIndex must not be larger than the size of the
 receiver, and must not be less than 1.

 Generates an error if anIndex is not a SmallInteger or is out of
 bounds, or if the receiver is not indexable."

<primitive: 974>

(anIndex _isInteger)
  ifTrue: [^ self _errorIndexOutOfRange: anIndex]
  ifFalse: [^ self _errorNonIntegerIndex: anIndex].
self _primitiveFailed: #at: args: { anIndex } .
self _uncontinuableError
%


category: 'instance creation'
classmethod:
fromHexString: aHexString leftPadded: padLeft

"Create a new instance of the receiver containing the byte values contained
 in aHexString. aHexString must be single-byte character collection (i.e, 
 it must answer 1 to the message #charSize) and contain characters in the 
 range of 0 - 9 and/or of a-f/A-F. Alphabetic hex characters may be in upper
 or lower case. A C hex prefix of 0x or 0X present in the first two characters
 of aHexString is permitted. A Smalltalk hex prefix of 16r in the first three
 characters is also permitted.

 If aHexString contains an odd number of hex digits, then the result object
 must be padded. The Boolean argument padLeft determines if left or right 
 padding is used. If padLeft is true, the result is left-padded, meaning
 the first four bits of the first byte in the result object will be zero. If
 padLeft is false, the result is right-padded, meaning the last four bits of
 the last byte in the result object will be zero.

 Examples:
   (ByteArray fromHexString: 'abc' leftPadded: true)  => 0a,bc
   (ByteArray fromHexString: 'abc' leftPadded: false) => ab,c0

 Returns a new instance of the receiver on success or raises an exception
 on error."

<primitive: 1061>
aHexString _validateClasses: { String }.
padLeft   _validateClass: Boolean .
self _primitiveFailed: #fromHexString:leftPadded: args: { aHexString . padLeft }
%

category: 'instance creation'
classmethod:
fromHexString: aHexString
"Create a new instance of the receiver containing the byte values contained
 in aHexString. aHexString must be single-byte character collection (i.e, 
 it must answer 1 to the message #charSize) and contain characters in the 
 range of 0 - 9 and/or of a-f/A-F. Alphabetic hex characters may be in upper
 or lower case. A C hex prefix of 0x or 0X present in the first two characters
 of aHexString is permitted. A Smalltalk hex prefix of 16r in the first three
 characters is also permitted.

 If aHexString contains an odd number of hex digits, then the result object
 is left-padded, meaning the first four bits of the first byte in the result 
 object will be zero.

 Example:
   (ByteArray fromHexString: 'abc') => 0a,bc

 Returns a new instance of the receiver on success or raises an exception
 on error."
 
^ self fromHexString: aHexString leftPadded: true
%

category: 'instance creation'
classmethod:
fromBase64String: aString
"Creates an instance of the receiver by decoding aString, which must be an 
 instance of String or another single-byte character collection class. The
 argument must be in valid base64 format. The argument may contain newline
 and other whitespace charcaters, which are ignored.

 Raises an exception if aString is not a valid base64 string."
 
<primitive: 1063>
aString _validateClass: String .
self _primitiveFailed: #fromBase64String: args: { aString }
%


! File in methods common to both ByteArray and CharacterCollection.
! It's a shame they don't share a common byte-format superclass
! SequenceableCollection does not qualify!
!
input $upgradeDir/message_digests.gs

input $upgradeDir/encrypt_decrypt.gs

input $upgradeDir/sign_verify_asym.gs

