! ========================================================================
! Copyright (C) by GemTalk Systems 1991-2020.  All Rights Reserved
!
! $Id$
!
! ========================================================================
removeallmethods DoubleByteString
removeallclassmethods DoubleByteString
set class DoubleByteString

category: 'Documentation'
classmethod:
comment
^ '
 A DoubleByteString is a string for which each Character occupies two bytes.
 Instances are in CPU-native byte order when in memory.

 For historical reasons dealing with the now-deprecated constraints, 
 DoubleByteString is in the ClassHistory of String. 

 Category: Comparision
   Unless noted otherwise, comparisons between DoubleByteStrings are 
   case-sensitive.
   Some of these methods determine whether one strings collates before
   another.  In collation, the Characters of the receiver and aString are
   compared Character-by-Character, from left to right, in case-sensitive
   fashion.  If two strings are of different length, and all Characters in the
   shorter string are equal to their counterparts in the longer strings, the
   shorter string collates before the longer.

   These methods require that the argument be a kind of CharacterCollection, 
   not necessarily a String.'
%

! ------------------- Class methods for DoubleByteString

category 'Formatting'
classmethod:
charSize

"Returns number of bytes that make up a character for this string class"

^ 2
%

category 'Instance Creation'
classmethod:
new: aSize

"Returns a new instance of the receiver, of sufficient size to hold aSize
 Characters of two bytes each."

^ self _basicNew:  aSize + aSize 
%

classmethod:
new

"Returns a new instance of the receiver."

^ self _basicNew
%

category 'Instance Creation'
classmethod:
withAll: aString

"Returns an instance of the receiver or an instance of QuadByteString,
 using the minimum bytes per character required to represent the argument."

<primitive: 456>
^ QuadByteString withAll: aString
%

! ------------------- Instance methods for DoubleByteString
category 'Formatting'
method:
charSize

"Returns number of bytes that make up a character for this string class."

^ 2
%

category: 'Concatenating'
method:
, aCharOrCharColl

"Returns a new instance of the receiver's class that contains the elements of
 the receiver followed by the elements of aCharOrCharColl.  The argument
 must be a String, a DoubleByteString, or an AbstractCharacter.

 The result may not be an instance of the class of the receiver if one of the
 following rules applies:

 1) If the receiver is a DoubleByteSymbol, the result is a DoubleByteString.

 Warning: Creating a new instance and copying the receiver take time.  If you
 can safely modify the receiver, it can be much faster to use the addAll:
 method.  See the documentation of the Concatenating category of class
 SequenceableCollection for more details."

<primitive: 614 >
| result cSize |
cSize := aCharOrCharColl stringCharSize .
cSize == 4 ifTrue:[
  result:= QuadByteString withAll: self.
  result addAll: aCharOrCharColl .
  ^ result .
].
aCharOrCharColl class == Character ifTrue:[
  aCharOrCharColl codePoint <= 16rFFFF ifTrue:[
    result := self copy 
  ] ifFalse:[
    result := QuadByteString withAll: self
  ].
  result at: (result size + 1) put: aCharOrCharColl .
  ^ result
].
aCharOrCharColl _validateClasses: { AbstractCharacter . CharacterCollection }.
^ super , aCharOrCharColl
%

method:
asDoubleByteString
 ^ self
%

! asSymbol, asSymbolKind inherited from MultiByteString

category: 'Accessing'
method:
at: anIndex

"Returns the Character at anIndex."

<primitive: 435>

(anIndex _isSmallInteger)
  ifTrue: [^ self _errorIndexOutOfRange: anIndex]
  ifFalse: [^ self _errorNonIntegerIndex: anIndex].
self _primitiveFailed: #at: args: { anIndex } .
self _uncontinuableError
%

category: 'Accessing'
method:
codePointAt: anIndex

"Returns the integer value of the Character at anIndex in the receiver."

<primitive: 1072>

(anIndex _isSmallInteger)
  ifTrue: [^ self _errorIndexOutOfRange: anIndex]
  ifFalse: [^ self _errorNonIntegerIndex: anIndex].
self _primitiveFailed: #codePointAt: args: { anIndex } .
self _uncontinuableError
%

category: 'Updating'
method:
at: anIndex put: aChar

"Stores aChar at the specified location and returns aChar."

<primitive: 436>

(aChar class == Character) ifTrue:[
  (aChar codePoint) > 16rFFFF ifTrue:[
    ^ self _convertToQuadByte at:anIndex put:aChar
  ].
] ifFalse:[
  aChar _validateClass: AbstractCharacter .
  ^ self at: anIndex put: aChar asCharacter
].
(anIndex _isSmallInteger) ifTrue: [
  ((anIndex > (self size + 1)) or: [anIndex <= 0]) ifTrue: [
    ^ self _errorIndexOutOfRange: anIndex
    ].
] ifFalse:[
  ^ self _errorNonIntegerIndex: anIndex
] .
self _primitiveFailed: #at:put: args: { anIndex . aChar }
%

category: 'Updating'
method: 
codePointAt: anIndex put: aValue

"Stores aValue at the specified location, with autogrow, but
 without auto-conversion of receiver to larger character size.
 Returns aValue.
"
<primitive: 1074>
aValue _validateClass: SmallInteger .
(aValue < 0 or:[ aValue > 65535]) ifTrue:[
  aValue _error: #rtErrArgOutOfRange args:{ 0 . 65535 }  
].
(anIndex _isSmallInteger) ifTrue: [ 
  ((anIndex > (self size + 1)) or: [anIndex < 1]) ifTrue: [
    ^ self _errorIndexOutOfRange: anIndex 
  ]
] ifFalse: [
  ^ self _errorNonIntegerIndex: anIndex
] .
self _primitiveFailed: #codePointAt:put: args: { anIndex . aValue }
%

! numArgs inherited from CharacterCollection


category: 'Adding'
method:
add: aCharOrCharColl

"Appends all of the elements of aCharOrCharColl to the receiver and returns
 aCharOrCharColl."

<primitive: 437>
| cSize |
aCharOrCharColl class == Character ifTrue:[
  aCharOrCharColl codePoint > 16rFFFF ifTrue:[
    ^ self _convertToQuadByte add: aCharOrCharColl
  ].
  ^ self _primitiveFailed: #add: args: { aCharOrCharColl }
].
(cSize := aCharOrCharColl stringCharSize) == 4 ifTrue:[
  aCharOrCharColl _asDoubleByteString ifNotNil:[ :aString |
    ^ self addAll: aString
  ] ifNil:[ 
    ^ self _convertToQuadByte addAll: aCharOrCharColl 
  ]
].
(aCharOrCharColl isKindOf: AbstractCharacter) ifTrue:[ 
  ^ self add: aCharOrCharColl asCharacter .
].
aCharOrCharColl _validateClass: CharacterCollection .
^ self add: aCharOrCharColl asDoubleByteString 
%

category: 'Adding'
method:
addAll: aCharOrCharColl

"Equivalent to add: aCharOrCharColl."

<primitive: 437>
| cSize |
(cSize := aCharOrCharColl stringCharSize) == 4 ifTrue:[
  aCharOrCharColl _asDoubleByteString ifNotNil:[ :aString |
    ^ self addAll: aString
  ] ifNil:[ 
    ^ self _convertToQuadByte addAll: aCharOrCharColl 
  ]
].
aCharOrCharColl class == Character ifTrue:[
  aCharOrCharColl codePoint > 16rFFFF ifTrue:[
    ^ self _convertToQuadByte add: aCharOrCharColl
  ].
  ^ self _primitiveFailed: #add: args: { aCharOrCharColl }
].
(aCharOrCharColl isKindOf: AbstractCharacter) ifTrue:[ 
  ^ self add: aCharOrCharColl asCharacter .
].
(aCharOrCharColl isKindOf: CharacterCollection) ifTrue:[
  ^ self add: aCharOrCharColl asDoubleByteString.
].
aCharOrCharColl do: [:each | self add: each].
^aCharOrCharColl.
%

category: 'Adding'
method:
addLast: aCharOrCharColl

"Equivalent to add: aCharOrCharColl."

<primitive: 437>
| cSize |
aCharOrCharColl class == Character ifTrue:[
  aCharOrCharColl codePoint > 16rFFFF ifTrue:[
    ^ self _convertToQuadByte add: aCharOrCharColl
  ].
  ^ self _primitiveFailed: #add: args: { aCharOrCharColl }
].
(cSize := aCharOrCharColl stringCharSize) == 4 ifTrue:[
  aCharOrCharColl _asDoubleByteString ifNotNil:[ :aString |
    ^ self addAll: aString
  ] ifNil:[ 
    ^ self _convertToQuadByte addAll: aCharOrCharColl 
  ]
].
(aCharOrCharColl isKindOf: AbstractCharacter) ifTrue:[ 
  ^ self add: aCharOrCharColl asCharacter .
].
aCharOrCharColl _validateClass: CharacterCollection .
^ self add: aCharOrCharColl asDoubleByteString 
%

category: 'Adding'
method:
addCodePoint: aSmallInteger

<primitive: 1048>
^ self add: (Character codePoint: aSmallInteger).
%

category: 'Accessing'
method:
size

"Returns the number of Characters in the receiver."

^ self _basicSize // 2.
%

! fix 41693
category: 'Private'
method:
_basicSize: anInteger

"Disallowed"
self shouldNotImplement: #_basicSize:
%
method:
_basicAt: anInteger put: aValue

"Disallowed"
self shouldNotImplement: #_basicAt:put:
%

category: 'Updating'
method:
size: anInteger

"Changes the size of the receiver to anInteger.

 If anInteger is less than the current size of the receiver, the receiver is
 shrunk accordingly.  If anInteger is greater than the current size of the
 receiver, the receiver is extended and new elements are initialized to 
 codePoint zero."

^ super _basicSize: anInteger + anInteger 
%

category: 'Copying'
method:
replaceFrom: startIndex to: stopIndex with: charCollection startingAt: repIndex

"Replaces the elements of the receiver between the indexes startIndex and
 stopIndex inclusive with the elements of aSeqCollection starting at repIndex.
 If aSeqCollection is identical to the receiver, the source and
 destination blocks may overlap.

 Returns the receiver."

<primitive: 439>
| cSize argInfo recInfo src16 |
startIndex _isSmallInteger ifFalse:[ startIndex _validateClass: SmallInteger ].
stopIndex _isSmallInteger ifFalse:[ stopIndex _validateClass: SmallInteger ].
repIndex _isSmallInteger ifFalse:[ repIndex _validateClass: SmallInteger ].
argInfo := charCollection _stringCharSize .
recInfo := self _stringCharSize .
(recInfo bitAnd: 16r8) ~~ 0 ifTrue:[  "receivier is a Utf string"
  (cSize := argInfo bitAnd: 16r7) > 1 ifTrue:[ 
    src16 := charCollection _asUnicode16 .
    (src16 ~~ nil and:[ src16 ~~ charCollection]) ifTrue:[
      ^ self replaceFrom: startIndex to: stopIndex with: src16 startingAt: repIndex
    ].
    ^ self _convertToQuadByte
     replaceFrom: startIndex to: stopIndex with: charCollection startingAt: repIndex
  ].
].
(cSize := argInfo bitAnd: 16r7) == 4 ifTrue:[
   src16 := charCollection _asDoubleByteString .  
   (src16 ~~ nil and:[ src16 ~~ charCollection]) ifTrue:[
      ^ self replaceFrom: startIndex to: stopIndex with: src16 startingAt: repIndex
   ].
   ^ self _convertToQuadByte replaceFrom: startIndex to: stopIndex
                        with: charCollection startingAt: repIndex
].
^ super replaceFrom: startIndex to: stopIndex with: charCollection startingAt: repIndex
%     


! deleted _idxCompareLessThan: v2.0

!deleted _idxCompareGreaterThan: v2.0

category: 'Adding'
method:
insertAll: aCharOrCharColl at: anIndex

"Inserts aCharOrCharColl at the specified index.  Returns aCharOrCharColl."

<primitive: 454>
| cSize |
anIndex _isSmallInteger ifFalse:[ anIndex _validateClass: Integer ].
((anIndex <= 0) or: [anIndex > (self size + 1)])
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].

(cSize := aCharOrCharColl stringCharSize) == 4 ifTrue:[ | aString |
  (aString := aCharOrCharColl _asDoubleByteString ) 
    ifNil:[ ^ self _convertToQuadByte insertAll: aCharOrCharColl at: anIndex]
    ifNotNil:[ ^ self insertAll: aString at: anIndex ].
].
aCharOrCharColl _validateClasses: { AbstractCharacter . CharacterCollection }.
^self insertAll: aCharOrCharColl asDoubleByteString at: anIndex.
%
!
! 39484: _compileInContext* methods moved up to MultiByteString
!

category: 'Converting'
method:
_convertToQuadByte

"Change the receiver from a DoubleByteString to a QuadByteString,
 or from a Unicode16 to Unicode32.  Changes the
 class of the receiver and the receiver's storage structure.
 Returns the receiver."

<primitive: 445>
%
category: 'Deprecated'
method:
asciiLessThan: aString

"Returns true if the receiver collates before the argument using the
 ASCII collating table, which collates AB...Z...ab..z."

self deprecated: 'DoubleByteString>>asciiLessThan: deprecated in 3.2'.

^ self lessThan: aString collatingTable: DoubleByteAsciiCollatingTable
%
category: 'Encoding'
method:
encodeAsUTF8

"Encode receiver in UTF8 format. Returns a Utf8 ." 

<primitive: 468>
self _primitiveFailed: #encodeAsUTF8
%
category: 'Encoding'
method: 
encodeAsUTF16

"Encode receiver in UTF16 format.  Returns a Utf16 ."

<primitive: 1084>
self _primitiveFailed: #encodeAsUTF16
%
method:
encodeAsUTF8IntoString
"Encode receiver in UTF8 format.  Returns a String. For Seaside"

<primitive: 994>
self _primitiveFailed: #encodeAsUTF8IntoString
%
