!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   GsSocket, IO, Object.
!
!=========================================================================

set class GsSocket
removeallmethods
removeallclassmethods 

category: 'For Documentation Installation only'
classmethod:
installDocumentation

self comment:
'GsSocket provides the means for creating and binding TCP sockets through the
 operating system of the machine that is running the session''s Gem process, and
 for communicating across those sockets.  
 When the current GsProcess is suspended until a socket is ready to read or write,
 other GsProcess that are ready to run will be run by the process scheduler.
 Methods that suspend the current GsProcess until the socket operation completes 
 are interruptable by a hard break.
 (You can get a hard break in Topaz by pressing the control-C key twice.  You can get
 a hard break in GemBuilder for C by calling the GciHardBreak function, and in
 GemBuilder for Smalltalk by calling the corresponding hard break method.)

 Beginning with Gs64 v3.0,  instances of GsSocket automatically have
 their C state (including their file descriptor) closed when 
 the instance is garbage collected or when a persistent instance drops out of memory.

                              Warning:
    Do not retain an instance of GsSocket from one session to another.
    Instances of GsSocket are intended to exist only within a given GemStone
    session.  GsSockets that are used across sessions always generate an error.

 All instVars of GsSocket are private, for use by the implementation
 of socket methods, and for use by the ProcessorScheduler only.
 The in-memory state of a committed GsSocket is not changed by a transaction abort.

Constraints:
	fileDescriptor: SmallInteger
	lineNumber: SmallInteger
	readWaiters: Object
	writeWaiters: Object
	readyEvents: SmallInteger
	pollArrayOfs: SmallInteger
	selectWaiters: Object' .
%

! ------------------- Class methods for GsSocket
category: 'Drastic Measures'
classmethod:
closeAll

"Close all instances of GsSocket that are open."

self _zeroArgClassPrim: 6 .
^ self
%

category: 'Deprecated'
classmethod:
finalizeAll

"Returns the number of instances of kinds of GsSocket that are alive
 and not closed."

self deprecated: 'GsSocket class>>finalizeAll deprecated in v3.2; finalization is automatic as of this release.
 Use closeAll to close all instances.'.
^ self _zeroArgClassPrim: 16 
%

category: 'Deprecated'
classmethod:
getServByName: serviceName
	"Returns the port number for the given service.  Returns nil if the service
	is undefined or an error occurs."

self deprecated: 'GsSocket class >> getServByName: deprecated long before v3.0. Use getServicePortByName: instead.'.
^ self getServicePortByName: serviceName
%

category: 'Queries'
classmethod:
getServicePortByName: serviceName

"Returns the port number for the given service.  
 Returns nil if the service is undefined or an error occurs."

^ self _twoArgClassPrim: 1  with: serviceName with: nil"tcp"
%

category: 'Queries'
classmethod:
getServicePortByName: serviceName withProtocol: protocol

"Returns the port number for the given service and protocol.
 Protocol may be nil (interpreted as tcp), 'tcp', or 'udp' .
 Returns nil if the service is undefined or an error occurs."

 | p |
 p := self _protocolToInt: protocol .
 ^ self _twoArgClassPrim: 1  with: serviceName with: p
%

category: 'Queries'
classmethod:
getServiceNameByPort: servicePort withProtocol: protocol

"Returns the service name for the given port number and protocol.
 Protocol may be nil (interpreted as tcp), 'tcp', or 'udp' .
 Returns nil if the service is undefined or an error occurs."

 | p |
 p := self _protocolToInt: protocol .
  ^ self _twoArgClassPrim: 8 with: servicePort with: p
%

category: 'Private'
classmethod:
_protocolToInt: protocol
  | p |
 protocol ifNil:[ 
   p := 0 "tcp"
 ] ifNotNil:[ 
   protocol _isOneByteString ifFalse:[ protocol _validateClass: String ].
   p := protocol asLowercase .
   p = 'tcp' ifTrue:[ p := 0 ] ifFalse:[ 
   p = 'udp' ifTrue:[ p := 1 ]
         ifFalse:[ ArgumentError signal:'protocol neither udp nor tcp' ]].
 ].
 ^ p
%

category: 'Queries'
classmethod:
getServiceNameByPort: servicePort

"Returns the service name for the given port number.
 Returns nil if the service is undefined or an error occurs."

^ self getServiceNameByPort: servicePort withProtocol: nil "tcp"
%

category: 'Queries'
classmethod:
getHostNameByAddress: address

"Returns the host name for the given ip address.
 Returns nil if the host is undefined or an error occurs.

 The address needs to be a String in a form acceptable to 
 either inet_pton(AF_INET6, address, ...)
 or    inet_pton(AF_INET, address, ...) in the host operating  
 system."

^ self _twoArgClassPrim: 7 with: address with: nil
%

category: 'Queries'
classmethod:
getHostAddressByName: aHostName

"aHostName must be a String .
 Returns the ip address for aHostName .

 Signals an ArgumentError if aHostName cannot be resolved.

 The resulting ip address is a String in a form acceptable to 
 inet_pton(AF_INET6, aString, ...)

 If getaddrinfo() returns more than one address for aHostName,
 the first element of the list from getaddrinfo() is returned.
"

| list |
list := self getHostAddressesByName: aHostName .
list ifNotNil:[ ^ list atOrNil: 1].
^ nil
%

! comments edited for 42670
classmethod:
getHostAddressesByName: aHostName

"aHostName must be a String.

 Signals an ArgumentError if aHostName cannot be resolved.

 Returns an Array of Strings. Each element of the array
 is a String containing an ip address in a form acceptable to
 inet_pton(AF_INET6, aString, ...) .
 The array contains the results from getaddrinfo()"

| canonName sz list | 
list := self _twoArgClassPrim: 12 with: aHostName with: nil .
(sz := list size) < 2 ifTrue:[ ^ nil ].
canonName := list at: 1 . "for debugging"
^ list copyFrom: 2 to: sz
%

category: 'Queries'
classmethod:
getLocalHostName

"Returns the name of the local host or nil if an error occurs."

^ self _zeroArgClassPrim: 5
%

category: 'Queries'
classmethod:
isAvailable

"Returns whether the supporting socket actions are available in the
 user's session."

^ true
%

category: 'Queries'
method:
raiseExceptionOnError

"Returns true if a socket error on the receiver will raise an exception.
 Returns false if not. The initial value is false."

^ self _twoArgPrim: 10 with: nil with: nil
%

category: 'Configuration'
method:
raiseExceptionOnError: bool

"Sets the value of raiseExceptionOnError to the Boolean bool for the receiver.
 Returns the receiver."

^ self _twoArgPrim: 10 with: bool with: nil
%

category: 'Instance Creation'
classmethod:
basicNew

"disallowed"

self shouldNotImplement: #basicNew
%

category: 'Private'
classmethod:
_basicNew

"creates an instance registered with VM for finalization of cData"

<primitive: 674>
^ self _primitiveFailed: #_basicNew
%

category: 'Instance Creation'
classmethod:
new

"Returns a new non-blocking TCP socket or nil if unable to create a new socket.

 On Solaris this creates a socket with family AF_INET6.

 On other platforms this creates a socket with family AF_INET.
 which may be bound or connected to IPv4 addresses or to IPv4-mapped IPv6 addresses."

^ self _basicNew _zeroArgPrim: 1
%

classmethod:
newUdp

"Returns a new non-blocking UDP socket or nil if unable to create a new socket.

 On Solaris this creates a socket with family AF_INET6.

 On other platforms this creates a socket with family AF_INET
 which may be bound or connected to IPv4 addresses or to IPv4-mapped IPv6 addresses."

^ self _basicNew _zeroArgPrim: 26
%

category: 'Instance Creation'
classmethod:
newIpv6

"Returns a new non-blocking TCP socket or nil if unable to create a new socket.

 This creates a socket with family AF_INET6
 which may be bound or connected to IPv6 addresses.
 On Linux and solaris the result socket may also 
 be bound or connected to IPv4-mapped IPv6 addresses."

^ self _basicNew _zeroArgPrim: 34
%

classmethod:
newUdpIpv6

"Returns a new non-blocking UDP socket or nil if unable to create a new socket.

 This creates a socket with family AF_INET6
 which may be bound or connected to IPv6 addresses.
 On Linux and solaris the result socket may also 
 be bound or connected to IPv4-mapped IPv6 addresses."

^ self _basicNew _zeroArgPrim: 35
%
classmethod:
fromFileHandle: aSmallInteger

"Create an instance of the receiver which will use the specified file handle.
 The fileHandle must be the file descriptor of
 an open socket that was inherited from the parent when the gem
 or  topaz -l  process was forked, or a socket created by non-Gemstone C code.
 in this gem or  topaz -l  process.  The non-Gemstone C code may be C application 
 code that calls the GCI or C code called from a user action or through FFI .
 The address family of the socket is determined by the process doing the 
 fork when it accepts a connection, or by the non-Gemstone C code.  
 The blocking / non-blocking state of the underlying socket will not be changed.
 The socket will be set to close-on-exec by this method.
 aSmallInteger must be > 0 and < 16r7fffffff .
 If aSmallInteger is not a valid file handle for a socket, a SocketError
 is signalled. 

 The instance of GsSocket produced by fromFileHandle: will not automatically
 close its file descriptor when the instance is garbage collected by in-memory
 GC .

 If the argument is from an instance of GsSocket created with a connect or accept
 method, then be aware that the file descriptor will be closed when the
 instance of GsSocket from connect or accept is garbage collected by in-memory
 GC .
"
  ^ self _basicNew _twoArgPrim: 28 with: aSmallInteger with: nil 
%



category: 'Copying'
method:
copy

"Returns an instance with no associated C state. 
 There are no instVars to copy"

^ self class _basicNew
%

! GsSocket>>newWithId: deleted

category: 'Low Level Access'
classmethod:
getStandardId: index inband: bool
"Returns the socket id for the given index. Index must be >= 0.
 If bool is true then gets the inband socket id. Otherwise gets
 the outband socket id.

 If index is 0 then the socket id to the stone is returned. Starting with Gs64 v2.2,
 the socket id to the stone will be -1 if the session is using shared memory 
 communication to the stone.

 If index is 1 then the socket id to the rpc application is returned.

 Otherwise index represents the GCI session id (see GciGetSessionId) and
 the socket id to that session is returned.
 If a socket id does not exist for the given index then -1 is returned."

^ self _twoArgClassPrim: 15 with: index with: bool
%

category: 'Accessing'
method:
changeable: aFlag
  "Has no effect in this release "
  ^ self
%

category: 'Accessing'
method:
changeable

"Returns true if methods that would change the state of the receiver
 are allowed. Returns false if the state of the receiver can not be
 changed."

^ true
%

category: 'Error Reporting'
classmethod:
lastErrorString

"Returns a String containing information about the last error for 
 GsSocket class methods, or nil if there is no error.
 Clears the error information for the receiver."

^ self _zeroArgClassPrim: 7
%

category: 'Error Reporting'
classmethod:
lastErrorCode

"Returns an integer representing that last operating system error
 for GsSocket class methods. Returns zero if there is no error.
 Does not clear the error information for the receiver."

^ self _zeroArgClassPrim: 8
%

category: 'Private'
classmethod:
_getErrorSymbol: code

"Returns a symbol from SocketErrorSymbols for the given code.
 Returns nil if no error."

"Note: socketprim.c:raiseError does a recur from om to this method."

(code == 0) ifTrue: [^nil].
(code > (SocketErrorSymbols size)) ifTrue: [^SocketErrorSymbols at: 1].

^SocketErrorSymbols at: code
%

category: 'Error Reporting'
classmethod:
lastErrorSymbol

"Returns a Symbol representing that last operating system error
 for GsSocket class methods. Returns nil if there is no error.
 Does not clear the error information for the receiver."

^ self _getErrorSymbol: (self _zeroArgClassPrim: 19)
%

category: 'Error Reporting'
method:
lastErrorString

"Returns the string of the last error on the receiver, or nil if no error has
 occurred.  Clears the error information for the receiver."

^ self _zeroArgPrim: 10
%

category: 'Error Reporting'
method:
lastErrorCode

"Returns an integer representing the last operating system error on the
 receiver. Returns zero if there is no error.
 Does not clear the error information for the receiver."

 ^ self _zeroArgPrim: 11
%

category: 'Error Reporting'
method:
lastErrorSymbol

"Returns a Symbol representing the last operating system error on the
 receiver. Returns nil if there is no error.
 Does not clear the error information for the receiver."

^ self class _getErrorSymbol: (self _zeroArgPrim: 20)
%

category: 'Private'
classmethod:
_zeroArgClassPrim: opcode

"See GsSocket>>_zeroArgPrim: for a description of the opcode."

<primitive: 323>
^ self _primitiveFailed: #_zeroArgClassPrim: args: { opcode }
%

category: 'Private'
method:
_zeroArgPrim: opcode

"opcode  function
   1     instance method: SocketCreate
   2     instance method: SocketClose
   3     instance method: SocketPort
   4     reserved
   5     class method:    GetLocalHostName
   6     class method:    SocketCloseAll
   7     class method:    SocketErrorString
   8     class method:    SocketErrno
   9     instance method: SocketPeerAddress AF_INET
  10     instance method: SocketErrorString
  11     instance method: SocketErrno
  12     instance method: SocketId
  13     instance method: SocketIsConnected
  14     instance method: SocketPeerPort
  15     instance method: SocketAddress
  16     class method:    GsSocketFinalizeAll
  17     instance method: SocketReadWillNotBlock
  18     instance method: SocketWriteWillNotBlock
  19     class method:    _lastErrorSymbolOffset
  20     instance method: _lastErrorSymbolOffset
  23     instance method: shutdownReading
  24     instance method: shutdownWriting
  25     instance method: shutdownReadingAndWriting
  26     instance method: SocketCreate UDP
  27     instance method: SocketCreate UNIX Stream
  28     instance method: SocketCreate UNIX Dgram
  29     class method:    SocketPair UNIX Stream (returns array of fds)
  30     instance method: SocketAddress AF_UNIX
  31     instance method: SocketPeerAddress AF_UNIX
  32     instance method: _peerSockAddr (returning a Ruby sockaddr String)
  33     instance method: _socketLocation (returning a Ruby sockaddr String)
  34     instance method: called by class method newIpv6
  35     instance method: called by class method newUdpIpv6
  36     instance method: isNonBlocking
"

<primitive: 323>

^ self _primitiveFailed: #_zeroArgPrim: args: { opcode }
%

category: 'Private'
method:
_twoArgPrim: opcode with: arg1 with: arg2

"opcode  function
   1     class method:    getServicePortByName
   2     instance method: SocketConnect, arg is a String, struct sockaddr_in6 (see below)
   3     instance method: SocketLinger
   4     instance method: SocketAccept
   5     instance method: SocketKeepAlive
   6     instance method: SocketBind
   7     class method:    GetHostNameByAddress
   8     class method:    GetServiceNameByPort
   9     instance method: GsSocketSetError
  10     instance method: GsSocketRaiseException
  11     instance method: SocketMakeListener
  12     class method:    GetHostAddressByName
  13     instance method  UNIXServer>>bind: aPath
  14     not used
  15     class method:    GetStandardId
  16     instance method  UNIXServer>>connect: aPath 
  17     class method     _existingSocketForFd:
  18     class method     _gethostbyaddr:   (using getnameinfo)
  19     class method     _getsockaddr:host:  (getaddrinfo with hints.ai_family=AF_UNSPEC)
  20     class method     _unpackSockAddr:flags:
  21     class method     _getsockaddrUnix:
  22     class method     _unpackSockAddrUnix:
  23     instance method  GsSocket>>rubyConnect: addrString
  24     instance method  _bindAddr:
  25     instance method _ addrInfo: aHost port: aPort  
	  result is { ai_canonname . struct_sockaddr ... struct_sockaddr }
	   aHost == -1 means <broadcase>
           aHost == nil means IN6ADDR_ANY_INIT
          IPv4 addresses preceed IPv6 addresses in returned list.
  26     instance method  _rubyCreate: type af: af 
  27     class method     addressIsIpv6:
  28     instance method: _setFileDescriptor:
  29     instance method: setCloseOnGc

  For opcode 2, SocketConnect the return values from this primitive are
    self  - connect succeeded
    false -  the connect() call returned EINPROGRESS, and
           a subsequent  poll()  waiting for write-ready with timeout of 500ms
           timedout , thus socket is still in EINPROGRESS state .
           The image then uses writeWillNotBlockWithin:  to wait longer for
           the  connect to complete .
    nil -    connect()  returned EINPROGRESS and the subsequent poll for write-ready
           failed with an error, indicating connect() failed rather than completing.
    aSmallInteger - errno value of a failed connect() , other than EINPROGRESS.
"

<primitive: 322>

^ self _primitiveFailed: #_twoArgPrim:with:with: args: { opcode . arg1 . arg2 }
%

category: 'Private'
classmethod:
_twoArgClassPrim: opcode with: arg1 with: arg2

"See GsSocket>>_twoArgPrim:with:with: for a description of legal opcodes"
<primitive: 322>

^ self _primitiveFailed: #_twoArgClassPrim:with:with:
       args: { opcode . arg1 . arg2 }
%

method:
_readInto: aString startingAt: anOffset maxBytes: numBytes

"Returns nil if an error occurred,
   false if non-blocking receiver would block,
   true if EINTR occurred,
   a SmallInteger number of bytes read.

 numBytes specifies maximum number of bytes to read, they
 are stored starting at (aString at: anOffset).

 Clears bits 16r3 from self.readyEvents if no error generated.
" 

<primitive: 489>
self _primitiveFailed: #_readInto:startingAt:maxBytes:
     args: { aString . anOffset . numBytes }
%

method:
_write: aByteObject startingAt: anOffset ofSize: numBytes

"returns nil if an error occurred,
   false if non-blocking receiver would block,
   true if EINTR occurred,
   a SmallInteger number of bytes written.
 Generates an error if aByteObject is not a String or ByteArray.
 Clears bits 16r4 from self.readyEvents if no error generated."

<primitive: 882>
self _primitiveFailed: #_write:startingAt:ofSize:
     args: { aByteObject . anOffset . numBytes }
%

category: 'Private'
classmethod:
_initSocketErrorSymbols

"Initializes the array of socket error symbols."

SocketErrorSymbols := #(
  "SYSERR_UNKNOWN" #UnknownSystemError
  "SYSERR_HOST_NOT_FOUND" #HostNotFound
  "SYSERR_TRY_AGAIN" #TryAgain
  "SYSERR_NO_RECOVERY" #NoRecovery
  "SYSERR_NO_DATA" #NoData
  "SYSERR_NO_ADDRESS" #NoAddress
  "SYSERR_NETDOWN" #NetworkDown
  "SYSERR_NETUNREACH" #NetworkUnreachable
  "SYSERR_NETRESET" #NetworkReset
  "SYSERR_CONNABORTED" #ConnAborted
  "SYSERR_CONNRESET" #ConnReset
  "SYSERR_NOBUFS" #NoBuffers
  "SYSERR_ISCONN" #AlreadyConnected
  "SYSERR_NOTCONN" #NotConnected
  "SYSERR_SHUTDOWN" #Shutdown
  "SYSERR_TIMEDOUT" #TimedOut
  "SYSERR_CONNREFUSED" #ConnectionRefused
  "SYSERR_NAMETOOLONG" #NameTooLong
  "SYSERR_INTR" #Interrupted
  "SYSERR_INPROGRESS" #InProgress
  "SYSERR_INVAL" #InvalidInput
  "SYSERR_ALREADY" #AlreadyDone
  "SYSERR_NOTSOCK" #NotSocket
  "SYSERR_DESTADDRREQ" #DestinationAddressRequired
  "SYSERR_MSGSIZE" #MsgSize
  "SYSERR_PROTOTYPE" #WrongProtocol
  "SYSERR_NOPROTOOPT" #OptionNotSupported
  "SYSERR_PROTONOSUPPORT" #ProtocolNotSupported
  "SYSERR_SOCKTNOSUPPORT" #TypeNotSupported
  "SYSERR_OPNOTSUPP" #OperationNotSupported
  "SYSERR_NOTSUP" #OperationNotSupported
  "SYSERR_PFNOSUPPORT" #ProtocolFamilyNotSupported
  "SYSERR_AFNOSUPPORT" #FamilyNotSupported
  "SYSERR_ADDRINUSE" #AddressInUse
  "SYSERR_ADDRNOTAVAIL" #AddressNotAvailable
  "SYSERR_WOULDBLOCK" #WouldBlock
  "SYSERR_BADF" #NotSocket
  "SYSERR_AGAIN"  #WouldBlock
  "SYSERR_NOMEM" #NoMemory
  "SYSERR_ACCES" #AccessDenied
  "SYSERR_FAULT" #Fault
  "SYSERR_PIPE" #BrokenPipe
  "SYSERR_HOSTDOWN" #HostDown
  "SYSERR_HOSTUNREACH" #HostUnreachable
  "SYSERR_NONET" #NoNetwork
  "SYSERR_WSANOTINITIALISED" #NotInitialized
  "SYSERR_NOSR" #NoStreams
  "SYSERR_PROTO" #ProtocolError
  "SYSERR_NXIO" #ServerExited
  "SYSERR_WSAEINTR" #BlockingCallCanceled
  "SYSERR_MFILE"    #ProcessOutOfDescriptors
  "SYSERR_NFILE"    #SystemOutOfDescriptors
  "GSSOCKETERR_NOHOSTNAME" #GsSocketNoHostName
).
%

category: 'Private'
method:
_isReadable

"Returns true if the receiver is marked as being ready to read."

^ (readyEvents bitAnd: 3) ~~ 0
%

category: 'Private'
method:
_isReadableOrException

"Returns true if the receiver is marked as being ready to read
 or has an exception."

^ (readyEvents bitAnd:16r3B) ~~ 0
%

category: 'Private'
method:
_isWritable

"Returns true if the receiver is marked as being ready to write."

^ (readyEvents bitAnd: 4) ~~ 0
%

category: 'Private'
method:
_isWritableOrException

"Returns true if the receiver is marked as being ready to write
 or has an exception."

^ (readyEvents bitAnd: 16r3c) ~~ 0
%

category: 'Private'
method:
_hadException

"Returns true if the receiver is marked as having had an exception."

^ (readyEvents bitAnd: 16r38) ~~ 0
%

category: 'Private'
method:
_enableRead

"Enables read detection by ProcessorScheduler>>_doPoll: .
 Returns receiver if successful, nil if socket not open."

<primitive: 731>
self _primitiveFailed: #_enableRead  "socket maybe set to blocking mode"
%

category: 'Private'
method:
_enableWrite

"Enables write detection by ProcessorScheduler>>_doPoll: .
 Returns receiver if successful, nil if socket not open."

<primitive: 732>
self _primitiveFailed: #_enableWrite  "socket maybe set to blocking mode"
%

category: 'Private'
method:
_disableEvent: aKind

"Disables the read or write detection specified by aKind.
 Used to cancel a previous  _enableRead or _enableWrite.
  aKind == true   cancels previous _enableWrite
  aKind == false  cancels previous _enableRead
  aKind == nil    cancels any previous _enableRead or _enableWrite
"

<primitive: 729>
self _primitiveFailed: #_disableEvent: args: { aKind }
%

category: 'Private'
method:
_setError: str

^self _twoArgPrim: 9 with: str with: nil
%


! ------------------- Instance methods for GsSocket
category: 'Deprecated'
method:
bind

self deprecated: 'GsSocket>>bind deprecated long before v3.0. Use bindTo: instead.'.
^self bindTo: nil 
%

category: 'Client Operations'
method:
bindTo: portNumber

"Binds the receiver to the specified port number.  Use the makeServer
 methods to do the bind when creating a server socket.
 If portNumber is nil then a random port is selected.
 This method is provided to bind a client socket to a specific port
 before it is connected.  Returns the port number actually
 bound to (should be the same as the argument unless argument is nil), 
 or nil if not successful."

^ self _twoArgPrim: 6 with: portNumber with: nil
%

category: 'Client Operations'
method:
bindTo: portNumber toAddress: address

"Binds the receiver to the specified port number and address. 
 Use the makeServer methods to do the bind when creating a server socket.
 This method is provided to bind a client socket to a specific port
 and/or address before doing a listen or connect on the socket.
 If portNumber is nil then a random port is selected.
 If address is nil then any network interface(IN6ADDR_ANY_INIT) is used.
 Returns the port number actually bound to (should be the same as 
 the argument unless argument is nil), or nil if not successful.

 address is a String containing a numeric IP address in one of these forms
   IPv4  dotted-decimal format  d.d.d.d  
   IPv6  hex format   x:x:x:x:x:x:x:x , 
   IPv4-mapped IPv6  ::FFFF:d.d.d.d, 
   where d is an 8 bit decimal number and x is a 16 bit hexadecimal number.
   An IPv6 format may contain at most one :: which is a contigous
   group of zeros.

 Gs64 v3.1:  address = -1, denoting IPv4 INADDR_BROADCAST, no longer supported.
 
"

^ self _twoArgPrim: 6 with: portNumber with: address
%

category: 'Socket Operations'
method:
close

"Release any temporary system resources used by the receiver.  This includes
 closing the low level socket.  Returns self if the socket is closes
 successfully or the socket is already closed.  Returns nil if socket cannot
 be closed."
  
^ self _zeroArgPrim: 2
%

category: 'Client Operations'
method:
connectTo: portNumber

"Connect the receiver to the server socket on the local machine
 identified by portNumber.
 portNumber maybe either a SmallInteger, or the String name of a service.
 Returns true if the connection succeeded and false if not.

 If the underlying connect() returns EISCONN(already connected)
 this method will return true.  (Gs64 v3.0 document preexisting behavior)
 "

^ self connectTo: portNumber on: 'localhost'
%

! changes to fix 34574 , 36607
! added send of peerAddress to fix 42049
category: 'Client Operations'
method:
connectTo: portNumber on: aHost

"Connect the receiver to the server socket identified by portNumber and aHost. 
 aHost may be the name of the host or its address, 
 or aHost == -1 for <broadcase> , or aHost == nil for IN6ADDR_ANY_INIT .
 portNumber maybe either a SmallInteger, or the String name of a service.
 Returns true if the connection succeeded and false if not."

^ self connectTo: portNumber on: aHost timeoutMs: -1
%

! HR 8681, fix 42571
category: 'Client Operations'
method:
connectTo: portNumber on: aHost timeoutMs: timeoutMs

"Connect the receiver to the server socket identified by portNumber and aHost.
aHost may be the name of the host or its numeric address,
or aHost == -1 for <broadcase> , or aHost == nil for IN6ADDR_ANY_INIT .
portNumber maybe either a SmallInteger, or the String name of a service.
timeoutMs is a SmallInteger specifying maximum time to wait for the
connection to complete, or -1 to wait indefinitely.
If aHost is the name of a host, connect is attempted on IPv4 first 
if getaddrinfo returns any IPv4 addresses for aHost, then attempted
on IPv6.
Returns true if the connection succeeded and false if not."

| addrList canonName |
(portNumber _isSmallInteger and:[ portNumber < 0]) ifTrue:[
  OutOfRange new name:'portNumber' min: 0 actual: portNumber ; signal
].
addrList := self _twoArgPrim: 25 with: aHost with: portNumber . "calls getaddrinfo"
addrList ifNotNil:[
  "addrList is Array of Strings representing list returned by getaddrinfo
  first element is the entry->ai_canonname if first address .
  subequent elements are addresses, each a struct sockaddr
  "
  canonName := addrList atOrNil: 1 . "for debugging"
  2 to: addrList size do:[:j | | status |
    status := self _twoArgPrim: 2 with: (addrList at: j) with: nil. "non-blocking connect"
    status == self ifTrue:[
      self peerAddress ifNotNil:[ ^ true ] ifNil:[ ^ false ].
    ].
    (status == false) ifTrue: [
      "connect call has blocked, wait for it to complete"
      self writeWillNotBlockWithin: timeoutMs .
      self peerAddress ifNotNil:[ ^ true ] .
    ].
  ].
].
^ false
%


category: 'Accessing'
method:
option: optionName

"Returns the current value of the specified option.
 Returns nil if an error occurred.
 optionName can be any of the following:
   'BROADCAST': value is Boolean; permission to transmit broadcast messages.
   'DEBUG': value is Boolean; records debugging information.
   'DONTROUTE': value is Boolean; routing bypass for outgoing messages.
   'ERROR': value is SmallInt; error code of the socket.
   'KEEPALIVE': value is Boolean; detect broken connections.
   'NODELAY': value is Boolean; disables nagle algorithm for send coalescing.
   'NONBLOCKING': value is Boolean , non-blocking state of socket.
   'OOBINLINE': value is Boolean; reception of out-of-band data inband.
   'REUSEADDR': value is Boolean; allows local address reuse.
   'RCVBUF': value is SmallInt; buffer size for input.
   'SNDBUF': value is SmallInt; buffer size for output.
   'TYPE': value is SmallInt; type of the socket (tcp or udp).
   'USELOOPBACK': value is Boolean; bypass network card if possible.
   'REUSEPORT': value is Boolean, multiple binds to same address, see SO_REUSEPORT in OS man pages .

  If an option name is not supported on a particular platform the
  error string will be 'The requested socket option does not exist'."

^ self _twoArgPrim: 5 with: optionName with: nil
%

category: 'Socket Operations'
method:
option: optionName put: value

"Sets the value of the specified option.
 Returns the receiver or nil if an error occurred.
 optionName can be any of the following:
   'BROADCAST': value is Boolean; permission to transmit broadcast messages.
   'CLOSEONEXEC': value is Boolean; close when system exec() is done.
   'DEBUG': value is Boolean; records debugging information.
   'DONTROUTE': value is Boolean; routing bypass for outgoing messages.
   'KEEPALIVE': value is Boolean; detect broken connections.
   'NODELAY': value is Boolean; disables nagle algorithm for send coalescing.
   'NONBLOCKING': value is Boolean , set state of non-blocking for socket.
   'OOBINLINE': value is Boolean; reception of out-of-band data inband.
   'REUSEADDR': value is Boolean; allows local address reuse.
   'RCVBUF': value is SmallInt; buffer size for input.
   'SNDBUF': value is SmallInt; buffer size for output.
   'USELOOPBACK': value is Boolean; bypass network card if possible.
   'REUSEPORT': value is Boolean, multiple binds to same address, see SO_REUSEPORT in OS man pages .

  If an option name is not supported on a particular platform the
  error string will be 'The requested socket option does not exist'."

^ self _twoArgPrim: 5 with: optionName with: value
%
   
category: 'Socket Operations'
method:
keepAlive: bool

"Sets the receiver to periodically broadcast messages to clients.  If
 a client does not respond to a broadcast, its connection is severed.
 If bool is false, keepAlive is turned off.
 Returns the receiver or nil if an error occurred."

^ self option: 'KEEPALIVE' put: bool
%

category: 'Socket Operations'
method:
linger: bool length: timeOut

"Sets up the receiver so that if unsent data is waiting to be transmitted
 at the time the receiver is closed, the current process will block until
 either the data is transmitted, or the given timeOut expires.  timeOut is
 in units of seconds.

 Returns the receiver or nil if an error occurred."

^ self _twoArgPrim: 3 with: bool with: timeOut
%

category: 'Server Operations'
method:
makeListener: queueLength

"Turns the receiver into a listening socket.
 The queueLength argument specifies the size of the listen backlog queue for
 incoming connections.
 Returns the receiver or nil if an error occurred."

^ self _twoArgPrim: 11 with: queueLength with: nil
%

category: 'Server Operations'
method:
makeServer

"Turns the receiver into a server socket. Binds the receiver to a port and
 makes it a listening socket.  Returns the receiver or nil if an error
 occurred."

^ self makeServer: 5 atPort: nil atAddress: nil
%

category: 'Server Operations'
method:
makeServer: queueLength

"Turns the receiver into a server socket.  The queueLength argument specifies
 the size of the listen backlog queue for incoming connections.  Binds the
 receiver to a random port.  Returns the receiver or nil if an error occurred."

^ self makeServer: queueLength atPort: nil atAddress: nil
%

category: 'Server Operations'
method:
makeServerAtPort: portNum

"Turns the receiver into a server socket.  Binds the receiver to portNum and
 makes it a listening socket.  Returns the receiver or nil if an error
 occurred."

^ self makeServer: 5 atPort: portNum atAddress: nil
%

category: 'Server Operations'
method:
makeServer: queueLength atPort: portNum

"Turns the receiver into a server socket.  The queueLength argument specifies
 the size of the listen backlog queue for incoming connections.  Binds the
 receiver to portNum.  If portNum is nil then a random port is selected.
 Returns the receiver, or nil if an error occurred."

^ self makeServer: queueLength atPort: portNum atAddress: nil
%

category: 'Server Operations'
method:
makeServer: queueLength atPort: portNum atAddress: address

"Turns the receiver into a server socket.  The queueLength argument specifies
 the size of the listen backlog queue for incoming connections.  Binds the
 receiver to portNum and address.
 If portNum is nil then a random port is selected.
 If address is nil then any appropriate network interface is used.
 Returns the receiver, or nil if an error occurred."

((self bindTo: portNum toAddress: address) == nil) ifTrue: [^nil].
^ self makeListener: queueLength
%

category: 'Server Operations'
method:
speciesForAccept
"Returns a class, an instance of which should be used as the result
 of the accept method."

^self class
%

category: 'Server Operations'
method:
accept

"Accept a client request for a connection on the receiver.  
 Returns a socket created for a new connection, 
 or nil if there was some problem.
 The result is an instance of speciesForAccept , and has non-blocking state
 equal to non-blocking state of the receiver.

 For example, the following code does not return until there is a connection:
   
 sock := GsSocket new.
 sock makeServer.
 newsock := sock accept.
 msg := newsock read: 512."

| res aSocket cnt |

aSocket := self speciesForAccept _basicNew .
res := true.
cnt := 1 .
[ res == true or:[ res == false ]] whileTrue: [
  "primitive will wait until data available if C socket is blocking."
  res := self _twoArgPrim: 4 with: aSocket with: nil .
  (res == false) ifTrue: [
    "socket is non-blocking and would have blocked, process scheduler will wait"
    res := self _waitForReadReady .
  ].
  cnt := cnt + 1 "for debugging"
].
(res == nil) ifFalse: [
  res := aSocket.
].
^res
%

category: 'Server Operations'
method:
acceptTimeoutMs: timeoutMs

"Returns a socket created for a new connection,
 or nil if there was an error or a timeout.
 The result is an instance of speciesForAccept , and has non-blocking state
 equal to non-blocking state of the receiver."

| res aSocket cnt |

aSocket := self speciesForAccept _basicNew .
res := true.
cnt := 1 .
[ res == true or:[ res == false ]] whileTrue: [
  "primitive will wait until data available if C socket is blocking."
  res := self _twoArgPrim: 4 with: aSocket with: nil .
  (res == false) ifTrue: [ | status |
    "socket is non-blocking and would have blocked, process scheduler will wait"
    status := self readWillNotBlockWithin: timeoutMs .
    status == true ifFalse:[ ^ nil ].
  ].
  cnt := cnt + 1 "for debugging"
].
(res == nil) ifFalse: [
  res := aSocket.
].
^res
%

! listen:  , listen:acceptingWith:   deleted as of Gs64 v3.0

category: 'Accessing'
method:
peerName

"For a bound socket, returns the hostname of the machine on which the 
 process at the other end of the connection is running.

 If the socket is not bound, or an error occurs, returns nil."

| addr |  

addr := self peerAddress.
(addr == nil) ifTrue: [^nil].
^ self class getHostNameByAddress: addr
%

category: 'Accessing'
method:
peerAddress

"For a bound socket, returns the address of the machine on which the 
 process at the other end of the connection is running.

 If the socket is not bound, or an error occurs, returns nil."
  
^ self _zeroArgPrim: 9
%

category: 'Accessing'
method:
peerPort

"For a bound socket, returns the port being used by the
 the other end of the connection.

 If the socket is not bound, or an error occurs, returns nil."

^ self _zeroArgPrim: 14
%

category: 'Accessing'
method:
port

"Returns the local port number of a socket, or nil if an error occurs.
 On some platforms an error is raised if the receiver is not bound.
 On others the port 0 is returned and no error is raised."
  
^ self _zeroArgPrim: 3
%

category: 'Accessing'
method:
address

"Returns the local ip address of a socket, or nil if an error occurs.
 On some platforms an error is raised if the receiver is not bound.
 On others the address '::' is returned and no error is raised."
  
^ self _zeroArgPrim: 15
%

!"Provided for compatibility with Geode."
category: 'Reading'
method:
read: maxBytes

"This method is equivalent to readString: .
 maxBytes must be a SmallInteger > 0 . "

 | bytes amount |
 bytes := String new.
 amount := self read: maxBytes into: bytes startingAt: 1.
 amount ifNil: [ ^nil ].
 ^ bytes
%

category: 'Reading'
method:
read: maxBytes into: byteObj

"Reads up to the given number of bytes into the given byte object (for
 example, a String) starting at index 1.
 Returns the number of bytes read, or nil if an error occurs, 
 or 0 if EOF on the receiver.
 byteObj is grown as needed but is not shrunk.
 maxBytes must be a SmallInteger > 0 .

 If no data is available for reading, 
 the current GsProcess is suspended until data arrives.
 The readWillNotBlock or readWillNotBlockWithin: methods may be used to
 determine whether or not data is ready for reading before calling this method."

^ self read: maxBytes into: byteObj startingAt: 1  
%

category: 'Private'
method:
_maxReadWaits
  ^ 3
%

category: 'Reading'
method:
read: maxBytes into: byteObj startingAt: index

"Reads up to the given number of bytes into the given byte object (for
 example, a String).  The first byte read will go into the position
 indicated by index. 
 Returns the number of bytes read, or nil if an error, 
 or 0 if EOF on the receiver.

 maxBytes must be a SmallInteger > 0 and <= 500000000 .

 An error will be raised if index is greater than the size of byteObj.
 Note that index can be equal to the size of byteObj plus 1 in which
 case any bytes read will be appended to byteObj.

 If no data is available for reading
 the current GsProcess is suspended until data arrives.
 The readWillNotBlock or readWillNotBlockWithin: methods may be used to
 determine whether or not data is ready for reading before calling this method."

| status waitCnt waitLimit |
[
  "primitive will wait until data available if C socket is blocking."
  status := self _readInto: byteObj startingAt: index maxBytes: maxBytes.
  "status==true from _readInto means got EINTR and must retry"
  (status == false) ifTrue: [
    "socket is non-blocking and would have blocked, process scheduler will wait"
    waitCnt ifNotNil:[
      waitCnt < (waitLimit ifNil:[ waitLimit := self _maxReadWaits])  ifTrue:[
        waitCnt := waitCnt + 1 .
        Delay waitForMilliseconds: 1.
      ] ifFalse:[
        "System _printSocketTrace ."
        SocketError signal:'EWOULDBLOCK from socket read after poll said read-ready'.
      ].
    ] ifNil:[
      waitCnt := 0 .
    ].
    status := self _waitForReadReady . "status true means ready to read"
  ] .
  status == true  
] whileTrue .
^ status
%

category: 'Reading'
method: 
read: maxBytes into: byteObj startingAt: index maxWait: timeMs

"Reads up to the given number of bytes into the given byte object (for
 example, a String).  The first byte read will go into the position
 indicated by index. 
 Returns the number of bytes read, nil if an error, 
 0 if EOF on the receiver, 
 false if receiver is not ready to read within timeMs milliseconds.

 maxBytes must be a SmallInteger > 0 and <= 500000000 .

 An error will be raised if index is greater than the size of byteObj.
 Note that index can be equal to the size of byteObj plus 1 in which
 case any bytes read will be appended to byteObj.

 If no data is available for reading
 the current GsProcess is suspended for up to timeMs until data arrives."

| status waitCnt waitLimit |
[
  "primitive will wait until data available if C socket is blocking."
  status := self _readInto: byteObj startingAt: index maxBytes: maxBytes.
  "status==true from _readInto means got EINTR and must retry"
  (status == false) ifTrue: [
    "socket is non-blocking and would have blocked, process scheduler will wait"
    waitCnt ifNotNil:[
      waitCnt < (waitLimit ifNil:[ waitLimit := self _maxReadWaits])  ifTrue:[
        waitCnt := waitCnt + 1 .
        Delay waitForMilliseconds: 1.
      ] ifFalse:[
        "System _printSocketTrace ."
        SocketError signal:'EWOULDBLOCK from socket read after poll said read-ready'.
      ].
    ] ifNil:[
      waitCnt := 0 .
    ].
    status := self readWillNotBlockWithin: timeMs . "status true means ready to read"
  ] .
  status == true  
] whileTrue .
^ status
%

category: 'Reading'
method:
readString: maxBytes untilFalse: aBlock

"Reads bytes from the receiver into a String which it creates and returns.
 Will keep reading until maxBytes bytes are read or 
 the one argument Block  aBlock  returns false.
 Each time aBlock is evaluated it is given the number of bytes read so far.
 aBlock is only evaluated after a partial read.
 maxBytes must be a SmallInteger > 0 .
 Returns the String or nil if an error occurs."

| bytes |

bytes := String new.
((self read: maxBytes into: bytes untilFalse: aBlock) == nil) ifTrue: [
  ^nil
].

^bytes
%

category: 'Reading'
method:
read: amount untilFalse: aBlock

"Same as readString:untilFalse:."

^ self readString: amount untilFalse: aBlock.
%

category: 'Reading'
method:
read: amount into: byteObj untilFalse: aBlock

"Reads at most amount bytes from the receiver into the given byte object
 starting at index 1.
 Returns when amount bytes have been read, an error occurs, or 
 the one argument Block  aBlock   evaluates to false.

 See read:into:startingAt:untilFalse: for details.

 If aBlock is evaluated its argument is the total number of bytes read
 so far. Returns the number of bytes read, or nil if an error occurs."

^ self read: amount into: byteObj startingAt: 1 untilFalse: aBlock.
%

category: 'Reading'
method:
read: amount into: byteObj startingAt: index untilFalse: aBlock

"Reads at most amount bytes from the receiver into byteObj.
 The first byte read is put in byteObj at position specified by index.
 Returns when amount bytes have been read, an error occurs, or 
 the one argument Block   aBlock   evaluates to false.

 An error will be raised if index is greater than the size of byteObj.
 Note that index can be equal to the size of byteObj plus 1 in which
 case any bytes read will be appended to byteObj.

 Repeatedly does a read followed by an evaluation of aBlock if the
 read returned without amount bytes having been read.

 If the receiver is not ready for reading
 the current GsProcess is suspended until data arrives.
 The readWillNotBlock or readWillNotBlockWithin: methods may be used to
 determine whether or not data is ready for reading before calling this method.
 They can also be used in aBlock to determine when it should return true
 so that another read will be done without blocking.

 If aBlock is evaluated its argument is the total number of bytes read
 so far. Returns the number of bytes read, or nil if an error occurs."

| total result offset numToRead |

total := 0.
offset := index.
numToRead := amount.

[result := self read: numToRead into: byteObj startingAt: offset.
 (result == nil) ifTrue: [^nil]. "Quit due to error"
 total := total + result.
 (total == amount) ifTrue: [^total]. "All done"
 numToRead := numToRead - result.
 offset := offset + result.
 aBlock value: total 
] untilFalse.

^total
%

category: 'Testing'
method:
isActive
"Returns true if the receiver's socket is in a usable state.
 Returns false if the receiver's socket is not usable.
 Sockets become unusable when they are closed."

((self id) == -1) ifTrue: [^false].
^ true
%

category: 'Testing'
method:
isConnected
"Returns true if the socket is connected to a peer.
 Returns false if the socket never was connected or has lost its connection
 to the peer."

^ self _zeroArgPrim: 13.
%

category: 'Testing'
method:
readWillNotBlock

"Returns true if the socket is currently ready to receive input without
 blocking.  Returns false if it is not currently ready.  Returns nil if an error
 occurs.

 The receiver must already be connected for this method to work properly.  If it
 is not connected, then the value that this method returns is indeterminate.
 Use the peerName method to determine if the receiver is connected.

 Call this method to prevent subsequent read or accept operations from hanging.
 If it returns true for a connected socket, then the input operation will not
 hang.  However, a return value of true is no guarantee that the operation
 itself will succeed."
  
^ self _zeroArgPrim: 17
%

! fixed 44336
category: 'Testing'
method:
readWillNotBlockWithin: msToWait

"Returns true if the socket is ready to receive input without blocking within
 msToWait milliseconds from the time that this method is called.  
 Returns false if it is not ready after msToWait milliseconds.  
 Returns nil if an error occurs.

 If msToWait ~~ 0 and the socket is not ready to read, the current
 GsProcess is suspended.  Semantics of suspend are the same as
 for Delay >> wait ; other GsProcess may execute while this GsProcess
 is suspended.

 If msToWait is 0, then this method reports the current readiness of the
 receiver.  If msToWait is -1, then this method never returns false, 
 but suspends the current GsProcess until the receiver is ready to receive 
 input without blocking, and then returns true.

 The receiver must already be connected for this method to work properly.  If it
 is not connected, then the value that this method returns is indeterminate.
 Use the peerName method to determine if the receiver is connected.

 Call this method to prevent subsequent read or accept operations from hanging.
 If it returns true for a connected socket, then the input operation will not
 hang.  However, a return value of true is no guarantee that the operation
 itself will succeed."

 | sched proc |
 msToWait == 0 ifTrue: [^ self readWillNotBlock ].
 self _isReadableOrException ifTrue:[ ^ true ].
 self _enableRead .
 sched := self _scheduler _enterCritical .
 proc := sched activeProcess .
 readWaiters ifNil:[  
   readWaiters := proc 
 ] ifNotNil:[  
   self _addReadWaiter: proc 
 ].
 msToWait == -1 ifTrue:[
   sched _waitForSocket: self .
   ^ true
 ] ifFalse:[
   sched _waitForSocket: self timeout: msToWait forWrite: nil .
   ^ self _isReadableOrException
 ]
%

category: 'Private'
method:
_unscheduleProcess: aProcess
   self _cancelReadEventFor: aProcess .
   self _cancelWriteEventFor: aProcess
%

category: 'Private'
method:
_addReadWaiter: aProcess 
   "assumes that readWaiters is not nil, and aProcess will be second or 
    subsequent waiter."

   | waiters arr |
   (waiters := readWaiters) _isArray ifFalse:[ 
     arr := { waiters } .
     readWaiters := arr .
     waiters := arr .
   ].
   self _add: aProcess toArray: waiters .
%

category: 'Private'
method:
_addWriteWaiter: aProcess 
   "assumes that writeWaiters is not nil, and aProcess will be second or 
    subsequent waiter."

   | waiters arr |
   (waiters := writeWaiters) _isArray ifFalse:[ 
     arr := { waiters } .
     writeWaiters := arr .
     waiters := arr .
   ].
   self _add: aProcess toArray: waiters .
%

category: 'Testing'
method:
_waitForReadReady

"Returns true when socket is ready to receive input without blocking.
 Caller should have already attempted a read that failed because it would block,
 or checked status from _isReadableOrException.
 "
 | sched proc |
 self _enableRead .
 sched := self _scheduler _enterCritical .
 proc := sched activeProcess .
 readWaiters ifNil:[
   readWaiters := proc
 ] ifNotNil:[  
   self _addReadWaiter: proc 
 ].
 sched _waitForSocket: self . 
 ^ true
%
 

category: 'Accessing
method:
readWaiters
  "Return an Array containing the objects waiting on the receiver to be ready
   for reading."

^ self _getWaiters: readWaiters
%

category: 'Accessing
method:
writeWaiters
  "Return an Array containing the objects waiting on the receiver to be ready
   for writing."

^ self _getWaiters: writeWaiters
%


category: 'Private'
method:
_getWaiters: waiters
  waiters ifNotNil:[
    waiters _isArray ifTrue:[
      ^ waiters copy
    ] ifFalse:[
      ^ { waiters } 
    ] 
  ] ifNil:[
    ^ { } 
  ].
%

category: 'Private Scheduling'
method:
_whenReadableNotify: objToNotify

 objToNotify _canWaitOnSocket .  "DNU if waiting not supported on objToNotify"
 self _enableRead .
 readWaiters ifNil:[
   readWaiters := objToNotify
 ] ifNotNil:[  
   self _addReadWaiter: objToNotify
 ]
%  

category: 'Accessing
method:
hasReadWaiter: anObject
  | wtrs |
  (wtrs := readWaiters) ifNotNil:[
    wtrs _isArray ifTrue:[
      ^ wtrs includesIdentical: anObject
    ] ifFalse:[
      ^ wtrs == anObject 
    ].
  ].
  ^ false  
%

category: 'Accessing
method:
hasWriteWaiter: anObject
  | wtrs |
  (wtrs := writeWaiters) ifNotNil:[
    wtrs _isArray ifTrue:[
      ^ wtrs includesIdentical: anObject
    ] ifFalse:[
      ^ wtrs == anObject 
    ].
  ].
  ^ false  
%

category: 'Private Scheduling'
method:
_reapEvents
  "returns true if receiver has no more waiters after reaping events"

  | rwtrs wwtrs res revents |
  rwtrs := readWaiters .
  ((revents := readyEvents) bitAnd:16r3B) ~~ 0 ifTrue:[ "inline _isReadableOrException"
    rwtrs ifNotNil:[
      rwtrs _isArray ifTrue:[
        1 to: rwtrs size do:[:k |
          (rwtrs at: k) _reapSignal: self  .
        ].  
      ] ifFalse:[
        rwtrs _reapSignal: self 
      ].
      "GsFile gciLogServer:' reaped readable' .  "
      readWaiters := nil .
      rwtrs := nil .
    ].
  ].
  wwtrs := writeWaiters .
  (revents bitAnd: 16r3c) ~~ 0 ifTrue:[ "inline _isWritableOrException"
    wwtrs ifNotNil:[
      wwtrs _isArray ifTrue:[
        1 to: wwtrs size do:[:k |
          (wwtrs at: k) _reapSignal: self 
        ].  
      ] ifFalse:[
        wwtrs _reapSignal: self 
      ].
      " GsFile gciLogServer:' reaped writable' . "
      writeWaiters := nil .
      wwtrs := nil .
    ].
    res := rwtrs == nil .
  ] ifFalse:[
    res := rwtrs == nil and:[ wwtrs == nil] .
  ].
  " GsFile gciLogServer:'  returning ' , res asString .   "
  ^ res 
%

category: 'Private Scheduling'
method:
_cancelReadEventFor: objToNotify

  "Returns true if event has already been cancelled or reaped.
   Returns false if event is cancelled by this invocation."

  | wtrs |
  wtrs := readWaiters .
  wtrs _isArray ifTrue:[
    (wtrs removeIdentical: objToNotify otherwise:nil) ifNotNil:[ ^ true ].
    wtrs size == 0 ifTrue:[ readWaiters := nil. wtrs := nil ].
  ] ifFalse:[
    wtrs == objToNotify ifTrue:[ 
      readWaiters := nil.  wtrs := nil .
    ] ifFalse:[
      ^ true 
    ].
  ].
  wtrs ifNil:[
    self _disableEvent: false . "disableRead"
  ].
  ^ false
%

category: 'Private Scheduling'
method:
_cancelWriteEventFor: objToNotify

  "Returns true if event has already been cancelled or reaped.
   Returns false if event is cancelled by this invocation."

  | wtrs |
  wtrs := writeWaiters .
  wtrs _isArray ifTrue:[
    (wtrs removeIdentical: objToNotify otherwise: nil) ifNotNil:[ ^ true ].
    wtrs size == 0 ifTrue:[ writeWaiters := nil. wtrs := nil ].
  ] ifFalse:[
    wtrs == objToNotify ifTrue:[ 
      writeWaiters := nil. wtrs := nil .
    ] ifFalse: [
      ^ true 
    ].
  ].
  wtrs ifNil:[
    self _disableEvent: true . "disableWrite"
  ].
  ^ false

%

category: 'Accessing'
method:
hasWaiters

  ^ readWaiters ~~ nil or:[ writeWaiters ~~ nil ]
%

category: 'Private'
method:
_scheduler

"Returns   ProcessorScheduler scheduler "
<primitive: 457>

self _primitiveFailed: #_scheduler
%

category: 'Private'
method: 
_add: anObject toArray: aProcessArray

"Implements IdentitySet add: semantics for aProcessArray
 Returns true if anObject was already in aProcessArray, false otherwise"

<primitive: 725>
aProcessArray size > 100 ifTrue:[
  self error:'too many processes waiting on socket'.
].
self _primitiveFailed: #_add:toSet: args: { anObject . aProcessArray }
%


category: 'Deprecated'
method:
readReady

self deprecated: 'GsSocket>>readReady deprecated long before v3.0. Use readWillNotBlock or readWillNotBlockWithin: instead.'.
^ self _zeroArgPrim: 17
%

category: 'Reading'
method:
readString: maxBytes

"Reads up to the given number of bytes, returning them in a String whose size is
 between 0 and maxBytes inclusive.  If an error occurs, nil is returned instead.
 If EOF occurs on the receiver before any bytes are read,
 a String of size zero is returned.

 maxBytes must be a SmallInteger > 0 .
 If maxBytes is greater than the size of the operating system's buffer for 
 the socket, the size of the result string may be a function of this
 buffer size, even if more data is available from the sender.  Repeated
 invocation of this method may be necessary to obtain all of the data.

 If no data is available for reading
 the current GsProcess is suspended until data arrives.
 The readWillNotBlock or readWillNotBlockWithin: methods may be used to
 determine whether or not data is ready for reading before calling this method."
 
  | bytes amount |
  bytes := String new.
  amount := self read: maxBytes into: bytes startingAt: 1.
  amount ifNil: [ ^nil ].
  ^ bytes
%

category: 'Writing'
method:
write: byteObj

"Write out the given byte object.  Returns the number of bytes written,
 or nil if an error occurs.
  
 If the receiver is not ready for writing
 the current GsProcess is suspended until data arrives.
 The writeWillNotBlock or writeWillNotBlockWithin: methods may be used to
 determine whether or not data is ready for writing before calling this method."

^ self write: (byteObj size) from: byteObj startingAt: 1
%

category: 'Writing'
method:
write: byteObj startingAt: index

"Write bytes from byteObj to the receiver.
 The first byte written will be from the position indicated by index.
 All of the bytes after it in byteObj will be written.
 Returns the number of bytes written, or nil if an error occurs.
  
 If the receiver is not ready for writing
 the current GsProcess is suspended until data arrives.
 The writeWillNotBlock or writeWillNotBlockWithin: methods may be used to
 determine whether or not data is ready for writing before calling this method."

^ self write: ((byteObj size) - (index - 1)) from: byteObj startingAt: index
%

category: 'Writing'
method:
write: byteObj untilFalse: aBlock

"Writes the entire contents of byteObj to the receiver unless
 an error occurs or aBlock returns false.
 See write:from:startingAt:untilFalse: for details.
 Returns the number of bytes actually written."

^ self write: (byteObj size) from: byteObj startingAt: 1 untilFalse: aBlock.
%

category: 'Writing'
method:
write: byteObj startingAt: index untilFalse: aBlock

"Writes the contents of byteObj to the receiver unless
 an error occurs or aBlock returns false.
 The first byte written from byteObj is at the position specified by index.
 All of the bytes after it in byteObj will be written.
 See write:from:startingAt:untilFalse: for details.
 Returns the number of bytes actually written."

^ self write: ((byteObj size) - (index -1)) from: byteObj startingAt: index
       untilFalse: aBlock.
%

category: 'Writing'
method:
write: amount from: byteObj

"Write the given number of bytes from the given byte object.  Returns the
 number of bytes written, or nil if an error occurs.
  
 If the receiver is not ready for writing
 the current GsProcess is suspended until data arrives.
 The writeWillNotBlock or writeWillNotBlockWithin: methods may be used to
 determine whether or not data is ready for writing before calling this method."

^ self write: amount from: byteObj startingAt: 1
%


category: 'Writing'
method:
write: amount from: byteObj startingAt: index

"Write bytes from byteObj to the receiver.
 The first byte written will be from the position indicated by index.
 The number of bytes written is specified by amount.
 It is in an error if amount bytes are not available.
 Returns the number of bytes written, or nil if an error occurs.
  
 If the receiver is not ready for writing
 the current GsProcess is suspended until data arrives.
 The writeWillNotBlock or writeWillNotBlockWithin: methods may be used to
 determine whether or not data is ready for writing before calling this method."
  
 | offset numToWrite waitCnt |
 offset := index.
 numToWrite := amount.
 [ true ] whileTrue:[ | result |
   [
     result := self _write: byteObj startingAt: offset ofSize: numToWrite .
     result == true 
   ] whileTrue . "loop to handle EINTR "
 
   result _isSmallInteger ifTrue:[
     numToWrite  := numToWrite - result .
     (numToWrite <= 0) ifTrue: [ ^ amount ]. "All done"
     waitCnt := nil .
     offset := offset + result .
   ] ifFalse: [
     result ifNil:[ ^ nil "socket error"].
     waitCnt ifNotNil:[ 
       waitCnt < 3 ifTrue:[
         waitCnt := waitCnt + 1 .
         Delay waitForMilliseconds: 1.
       ] ifFalse:[
         GsFile gciLogServer:'socket EAGAIN fd ', fileDescriptor asString .
         System _printSocketTrace .
         SocketError signal:'EAGAIN from socket write after poll said write-ready'.
       ].
     ] ifNil:[
       waitCnt := 0 .
     ].
     self _waitForWriteReady . 
   ].
 ].
%

category: 'Writing'
method:
nbwrite: amount from: byteObj startingAt: index

"If receiver is a non-blocking socket,
 write at most amount bytes from byteObj to the receiver without blocking.

 If receiver is a blocking socket,
 write at most amount bytes from byteObj to the receiver blocking until
 the socket accepts data, or until the gem process receives SIGTERM.

 The first byte written will be from the position indicated by index.
 The maximum number of bytes to write is specified by amount.
 It is an error if amount bytes are not available.
 Returns the number of bytes written, or nil if an error occurs.
 Returns 0 if non-blocking socket would have blocked."

 | res |
 [
   res := self _write: byteObj startingAt: index ofSize: amount.
   (res == false) ifTrue: [
     "would have blocked"
     ^ 0
   ].
   res == true  "if res==true, socket got EINTR and we retry"
 ] whileTrue .
 ^res
%

! method comments edited for 42379
category: 'Writing'
method:
write: amount from: byteObj startingAt: index untilFalse: aBlock

"Write bytes from byteObj to the receiver until amount bytes have
 been written, an error occurs, or aBlock's value is false.
 The first byte written will be from the position indicated by index.
 The number of bytes written is specified by amount.
 It is in an error if amount bytes are not available.
 Returns the number of bytes written, or nil if an error occurs.
 This method repeatedly tries to write as many bytes as it can
 and then, if more bytes still need to be written, evaluates aBlock.

 If the receiver is not ready for writing the one argument Block
 aBlock is called.
 The writeWillNotBlock or writeWillNotBlockWithin: methods may be used
 in aBlock to determine when it should return true so that another write
 will be done without blocking.
 Every time aBlock is evaluated its argument is the total number of bytes
 written so far.

 Implementation uses nbwrite, so unless aBlock uses 
 writeWillNotBlockWithin:-1 or other logic based on writeWillNotBlock ,
 to wait for the socket to be ready before returning false from
 aBlock, this method will run hot.  

 If the receiver is a GsSecureSocket, aBlock will not be executed,
 since the wait for ready for writing is performed at a lower layer.
"

| total offset numToWrite |

total := 0.
offset := index.
numToWrite := amount.

[ | result |
 result := self nbwrite: numToWrite from: byteObj startingAt: offset.
 (result == nil) ifTrue: [^nil]. "Quit due to error"
 total := total + result.
 (total == amount) ifTrue: [^total]. "All done"
 numToWrite := numToWrite - result.
 offset := offset + result.
 aBlock value: total
] untilFalse.

^total
%

category: 'Testing'
method:
writeWillNotBlock

"Returns true if the socket is currently ready to take output without blocking.
 Returns false if it is not currently ready.  Returns nil if an error occurs.

 The receiver must already be connected for this method to work properly.  If it
 is not connected, then the value that this method returns is indeterminate.
 Use the peerName method to determine if the receiver is connected.

 Call this method to prevent subsequent write operations from hanging.  If it
 returns true for a connected socket, then a subsequent write will not hang.
 However, a return value of true is no guarantee that the write operation itself
 will succeed."
  
^ self _zeroArgPrim: 18    
%

! fixed 44336
category: 'Testing'
method:
writeWillNotBlockWithin: msToWait

"Returns true if the socket is ready to take output without blocking within
 msToWait milliseconds from the time that this method is called.  
 Returns false if it is not ready after msToWait milliseconds.  
 Returns nil if an error occurs.

 If msToWait is 0, then this method reports the current readiness of the receiver.  
 If msToWait is -1, then this method never returns false, but suspends the
 current GsProcess until the receiver is ready to take output without blocking, 
 and then returns true.

 If msToWait ~~ 0 and the socket is not ready to read, the current
 GsProcess is suspended.  Semantics of suspend are the same as
 for Delay >> wait ; other GsProcess may execute while this GsProcess
 is suspended.

 The receiver must already be connected for this method to work properly.  If it
 is not connected, then the value that this method returns is indeterminate.
 Use the peerName method to determine if the receiver is connected.

 Call this method to prevent subsequent write operations from hanging.  If it
 returns true for a connected socket, then a subsequent write will not hang.
 However, a return value of true is no guarantee that the write operation itself
 will succeed."
  
 | sched proc |
 msToWait == 0 ifTrue:[ ^ self writeWillNotBlock].
 self _isWritableOrException ifTrue:[ ^ true ].
 self _enableWrite .
 sched := self _scheduler _enterCritical .
 proc := sched activeProcess .
 writeWaiters ifNil:[  
   writeWaiters := proc 
 ] ifNotNil:[  
   self _addWriteWaiter: proc 
 ].
 msToWait == -1 ifTrue:[
   sched _waitForSocket: self .
   ^ true
 ] ifFalse:[
   sched _waitForSocket: self timeout: msToWait forWrite: true .
   ^ self _isWritableOrException
 ]
%

category: 'Testing'
method:
_waitForWriteReady

"Waits until socket is ready to take output without blocking
 and then returns true. Caller should have already done a write or connect
 which failed because it would have blocked, or checked status with _isWritableOrException."

 | sched proc |
 self _enableWrite .
 sched := self _scheduler _enterCritical .
 proc := sched activeProcess .
 writeWaiters ifNil:[
   writeWaiters := proc
 ] ifNotNil:[  
   self _addWriteWaiter: proc 
 ].
 sched _waitForSocket: self .
 ^ true
%


category: 'Private Scheduling'
method:
_whenWritableNotify: objToNotify

 objToNotify _canWaitOnSocket .  "DNU if waiting not supported on objToNotify"
 self _enableWrite .
 writeWaiters ifNil:[
   writeWaiters := objToNotify
 ] ifNotNil:[  
   self _addWriteWaiter: objToNotify 
 ].
%

!   fix Maglev Trac 806
method:
_changePriority: aGsProcess from: oldPriority

  ^ self  "do nothing"
%


category: 'Deprecated'
method:
writeReady

self deprecated: 'GsSocket>>writeReady deprecated long before v3.0. Use writeWillNotBlock or writeWillNotBlockWithin: instead.'.
^ self _zeroArgPrim: 18
%

category: 'Comparing'
method:
hash

"Returns a SmallInteger related to the value of the receiver.  If two instances
 of GsSocket are equal (as compared by the = method), then they must have the
 same hash value."

^ self id hash
%

category: 'Comparing'
method:
= aSocket

"Returns true if the receiver and aSocket represent the same operating system
 socket.  Returns false otherwise."

| idSelf |
(self == aSocket) ifTrue: [^true].
aSocket ifNil: [^false].
(aSocket isKindOf: self class) ifFalse: [^false].
idSelf := self id.
(idSelf == -1) ifTrue: [^false].
^idSelf == (aSocket id)
%

category: 'Accessing'
method:
id

"Returns the value of the low level socket.  If no low level socket exists,
 returns -1."

^ fileDescriptor ifNil:[ -1 ] ifNotNil:[ fileDescriptor ]
%

category: 'Comparing'
method:
~= aSocket

"Returns false if the receiver and aSocket represent the same operating system
 socket.  Returns true otherwise."

^ (self = aSocket) not
%

category: 'Examples'
classmethod:
clientExample 

^ self clientExample: true usingPort: 57785
%

category: 'Examples'
classmethod:
clientExample: logToGciClient usingPort: portNum 

^ self clientExample: logToGciClient usingPort: portNum address: 'localhost'
%

category: 'Examples'
classmethod:
clientExample: logToGciClient usingPort: portNum address: serverAddress

"logToGciClient is a Boolean, true if GsFile-based logging is to be done to
  the RPC Gci client, false if to the server process's log file .
  Use false if executing via topaz 'nbrun'  .

 This client will connect to a server created with serverExample, and read
 the string object that the server sends.
   
 Creates a socket, connect it to port portNum, read a string, close the
 socket, and check the resulting object.  Returns true if successful.
   
 The server should already be listening for connections when this method is
 invoked."

| socket dataString dataObj chunk firstChunk sleepTime lastStr |

socket := GsSocket new.
(socket connectTo: portNum on: serverAddress) ifFalse:[ 
  socket close .
  Error signal: 'Unable to connect to port ' , portNum asString , '. Wait 30 seconds or so, then ensure serverExample is started first.' 
  ].
dataString := String new .

firstChunk := true .
sleepTime := 0 .
[ socket readWillNotBlockWithin: 5000] whileFalse: [ 
  GsFile gciLog: 'Waiting for server to write...' onClient: logToGciClient .
  sleepTime := sleepTime + 5 .
  sleepTime > 20 ifTrue:[
    GsFile gciLog: 'Not ready to read after 20sec' onClient: logToGciClient.
    Error signal: 'Not ready to read after 20sec' .
  ].
].
GsFile gciLog: 'Got something from the server to read.' onClient: logToGciClient .
[
  chunk := socket readString: 4000 "max bytes".
  firstChunk ifTrue:[
    firstChunk := false .
    chunk size <= 0 ifTrue:[ Error signal: 'readWillNotBlock disagrees with read'].
    ] .
  chunk ifNil:[ Error signal: 'Error in reading from socket' ].
  dataString addAll: chunk .
  (chunk at: chunk size) == (Character withValue: 0)  "until null terminator"
] untilTrue .
lastStr := socket readString: 100 .
lastStr size == 0 ifFalse:[ Error signal:'did not get EOF as expected'].  
socket close.
dataString size: (dataString size - 1) "strip null terminator" .
dataObj := (PassiveObject newWithContents: dataString) activate.
dataObj size == 5000 ifFalse:[ Error signal: 'bad size of dataObj' ].
1 to: 5000 do:[:j | 
  (dataObj at:j) == j ifFalse:[ Error signal: 'invalid contents in dataObj' ].
  ] .
^ true. 
%

category: 'Examples'
classmethod:
serverExample

^ self serverExample: true usingPort: 57785
%

category: 'Examples'
classmethod:
serverExample: logToGciClient usingPort: portNum

^ self serverExample: logToGciClient usingPort: portNum address: nil
%

category: 'Examples'
classmethod:
serverExample: logToGciClient usingPort: portNum address: listeningAddress

"logToGciClient is a Boolean, true if GsFile-based logging is to be done to
  the RPC Gci client, false if to the server process's log file . 
  Use false if executing via topaz 'nbrun'  .

 Creates a socket, binds it to port portNum, and waits for a connection.
 When a connection is established, sends some data to the client, and
 closes the connection.  Returns true if successful.

 You will need two GemStone sessions running from two independent
 interface processes to run both this and the clientExample.  The Gem
 processes for the two sessions must be on the same machine. (For
 example two Topaz sessions.)
   
 Warning: This method will cause your current session to hang until a 
 connection is established."

| server client dataObj dataString numWritten dataStream errStr 
  sleepTime |

server := GsSocket new.

(server makeServer: 5 atPort: portNum atAddress: listeningAddress) ifNil: [
  errStr := server lastErrorString.
  server close.
  Error signal: errStr.
  ].

server port == portNum ifFalse:[ Error signal: 'bad port number' ] .

GsFile gciLog: 'Waiting for GsSocket clientExample to connect...' 
	onClient: logToGciClient .
[server readWillNotBlockWithin: 5000] whileFalse: [
  GsFile gciLog: 'waiting...' 
	onClient: logToGciClient
  ].
GsFile gciLog: 'Client connected, starting accept.' onClient: logToGciClient .

client := server accept .
GsFile gciLog: 'accept finished. client = ', client asString  onClient: logToGciClient .
client ifNil: [
  errStr := server lastErrorString.
  server close.
  GsFile gciLog: 'error from accept, ', errStr onClient: logToGciClient .
  Error signal: errStr.
].
client linger: true length: 10.  "wait after closing until data is processed"

dataObj := Array new:5000 .
1 to: 5000 do:[:j | dataObj at:j put: j ].
dataString := String new .
dataStream := WriteStream on: dataString . 
PassiveObject passivate: dataObj toStream: dataStream .
dataString add: (Character withValue: 0). "null terminate so client knows
                                           when to stop reading"
sleepTime := 0 .
[ client writeWillNotBlock ] whileFalse:[ 
  sleepTime == 0 ifTrue:[ 
     GsFile gciLog: 'waiting because write to client would block' onClient: logToGciClient.
  ].
  sleepTime > 5000 ifTrue:[
    GsFile gciLog: 'ERROR, client would block' onClient: logToGciClient .
    Error signal: 'socket write will block' 
  ].
  System _sleepMs: 20 .
  sleepTime := sleepTime + 20 .
].
GsFile gciLog: 'waited ' , sleepTime asString, 'ms ' onClient: logToGciClient.
GsFile gciLog: 'about to write ', dataString size asString, ' bytes'  onClient: logToGciClient .
numWritten := client write: dataString .
GsFile gciLog: 'wrote ', numWritten asString, ' bytes'  onClient: logToGciClient .
numWritten == dataString size ifFalse:[ Error signal: 'error writing to socket'].
client close.
server close .
				"deleted   'reduce garbage' code"
^ true
%

category: 'Socket Operations'
method:
shutdownReading

"Partially shutdown the socket such that further reads are disallowed.
 The socket is not closed.

 Returns true if successful, false if the method failed, and nil if 
 the receiver is not a valid GsSocket."
  
^ self _zeroArgPrim: 23
%

category: 'Socket Operations'
method:
shutdownWriting

"Partially shutdown the socket such that further writes are disallowed.
 The socket is not closed.

 Returns true if successful, false if the method failed, and nil if 
 the receiver is not a valid GsSocket."
  
^ self _zeroArgPrim: 24
%

category: 'Socket Operations'
method:
shutdownReadingAndWriting

"Shutdown the socket such that further reads and writes are disallowed.
 The socket is not closed.

 Returns true if successful, false if the method failed, and nil if 
 the receiver is not a valid GsSocket."
  
^ self _zeroArgPrim: 25
%
category: 'Private'
method:
_waitingProcessesInto: anIdentitySet
  | waiters |
  1 to: (waiters := selectWaiters) size by: 2 do:[:j |
    anIdentitySet add: (waiters at: j)
  ]
%
method:
_waitingProcessesInto: anIdentitySet inGroup: groupId
  | waiters |
  1 to: (waiters := selectWaiters) size by: 2 do:[:j | | proc |
    proc := waiters at: j .
    proc _groupOrNil == groupId ifTrue:[ anIdentitySet add: proc ].
  ]
%

category: 'Private'
method:
_recvfrom: maxBytes

"calls recvfrom().
 Returns an Array of the form [ aString , senderInfoArray ],
 or false to indicate EWOULDBLOCK, or true to indicate EINTR,
 or a String error message. 
 aString in the result array is the data received from the socket.
 The senderInfoArray represents the struct sockaddr filled in by
  recvfrom() , and is of the form
    { af_string . portNum . nil . ipAddrString } .
  af_string will be one of 'AF_INET' , 'AF_INET6', 'AF_UNIX' . "

<primitive: 884>
self _primitiveFailed: #_recvfrom: args: { maxBytes }
%

method:
_sendUdp: aString startingAt: anOffset to: hostName port: portNum

"Returns a SmallInteger number of bytes written,
  or false to indicate EWOULDBLOCK,
  or true to indicate EINTR,
  or a String error message. 
  Uses getaddrinfo() with hints.ai_socktype == SOCK_DGRAM to translate
  hostName and portNum to a struct sockaddr for use by sendto() ."

<primitive: 883>
self _primitiveFailed: #_sendUdp:to:port:
     args: { aString . anOffset . hostName . portNum }
%


category: 'UDP Support'
method:
recvfrom: maxBytes 

 "Returns an Array of the form [ aString , senderInfoArray ] .
  May be used to receive data from a socket was created by
    GsSocket newUdp  
  and which has been bound using  bindTo:  or bindTo:toAddress: .
  aString in the result array is the data received from the socket.
  The senderInfoArray represents the struct sockaddr filled in by
  recvfrom() , and is of the form
    { af_string . portNum . nil . ipAddrString } .
  af_string will be one of 'AF_INET' , 'AF_INET6', 'AF_UNIX' . 
  Signals an Error if a socket error occurs "

 maxBytes _isSmallInteger ifFalse:[ ArgumentTypeError signal:'argument must be a Fixnum'].
 maxBytes > 0 ifFalse:[ ArgumentError signal:'argument must be > 0'].
 [ true ] whileTrue:[ | result |
   [ result := self _recvfrom: maxBytes .
     result == true
   ] whileTrue . "loop to handle EINTR"
   result _isOneByteString ifTrue:[  SocketError signal:'recvfrom failed, ', result ].
   result == false ifTrue:[  "non-blocking socket would have blocked."
     self _waitForReadReady .
   ] ifFalse:[
     ^ result  "an Array"
   ]
 ].
%

method:
sendUdp: aString flags: flagsInt toHost: hostName port: aPort

" Sends aString using sendto().  Intended for use with receiver being
  a socket obtained from   
     GsSocket newUdp
  aPort must be a SmallInteger or a String specifying a port number or
  service name.
  Uses getaddrinfo() with hints.ai_socktype == SOCK_DGRAM to translate
  hostName and portArray to a struct sockaddr for use by sendto(). 
  Signals a SocketError if an error occurs."

  | idx strSize portStr isReady |
  aPort _isSmallInteger ifTrue:[
    portStr := aPort asString .
  ] ifFalse:[
    aPort _isOneByteString ifFalse:[ ArgumentTypeError signal:'port must be a String or SmallInteger'].
    portStr := aPort .
  ].
  hostName _isOneByteString ifFalse:[ ArgumentTypeError signal:'hostname must be a String'].
  flagsInt == 0 ifFalse:[ ArgumentError signal:'non-zero flags arg not supported yet'].
  aString _isOneByteString ifFalse:[ ArgumentTypeError signal:'first arg(data to send) must be a String'].
  strSize := aString size .
  idx := 1 .
  [ true ] whileTrue:[ | result |
    [ result := self _sendUdp: aString startingAt: idx to: hostName port: portStr .
      result == true
    ] whileTrue . "loop to handle EINTR"
    result _isSmallInteger ifTrue:[
      idx := idx + result .
      idx > strSize ifTrue:[ ^ strSize "done" ]
          ifFalse:[ result == 0 ifTrue:[ SocketError signal:'sendto infinite loop']].
      isReady := nil .
    ] ifFalse:[
      isReady ifNotNil:[ 
        System _printSocketTrace .
        SocketError signal:'EAGAIN from socket write after poll said write-ready'
      ].
      result == false ifTrue:[  "non-blocking socket would have blocked."
        self _waitForWriteReady .
        isReady := true .
      ] ifFalse:[
        SocketError signal:'sendto failed, ', result asString
      ].
   ]
 ]
%

category: 'Accessing'
method:
isNonBlocking

^ self _zeroArgPrim: 36
%

category: 'Accessing'
method:
isBlocking

^ self isNonBlocking == false
%

category: 'Updating'
method:
makeNonBlocking
self isNonBlocking ifFalse:[ self option: 'NONBLOCKING' put: true ].
^ self
%

category: 'Updating'
method:
makeBlocking
self isBlocking ifFalse:[ self option: 'NONBLOCKING' put: false ].
^ self
%

category: 'Queries'
classmethod:
addressIsIpv6: aString

"Returns true if aString is of a form acceptable to 
 inet_pton(AF_INET6, aString, ...)  and is not an IPv4-mapped IPv6 address
 Returns false otherwise."

^ self _twoArgClassPrim: 27 with: aString with: nil
%

expectvalue true
run
GsSocket _initSocketErrorSymbols .
^true
%

category: 'Writing'
method:
writev: numBuffers specs: specArray

"Use the writev system call to write the specified number
 of buffers.  Each buffer is a byte format object described by
 3 elements of specArray in this order:
  * Buffer: A byte object containing bytes to be written
  * StartIndex: The 1-based index within the buffer of the first byte to be written
  * NumBytes: The number of bytes to be written.
 
 This method will block until the write completes, or until a socket error
 occurs.
 Returns receiver if write completes successfully, or signals an Error .
 Signals an ArgumentError if an element of specArray is invalid .
 Signals a SocketError if the underlying writev() fails .
"
| ofs |
ofs := 1 .
 [ true ] whileTrue:[ | result |
   [
     result := self _writev: numBuffers specs: specArray byteOffset: ofs .
     result == true
   ] whileTrue . "loop to handle EINTR "
   result == 0 ifTrue:[ ^ self  "all buffers sent" ].
   result > 0 ifTrue:[ 
     ofs := result .
     self _waitForWriteReady
   ] ifFalse:[
     SocketError signal:'errno = ', (0 - result) asString      
   ]
 ]
%

category: 'Private'
method:
_writev: numBuffers specs: specArray byteOffset: startOffset

"Use the writev system call to write buffers per  GsSocket>>writev:specs:
with startOffset being a one based byte offset into the concatenated
data specified by the first numBuffers*3 elements of specArray .

numBuffers must be <= 20 .

Returns 
  0 if all writes completed, 
  a positive offset if EWOULDBLOCK or EAGAIN was returned by the
    underlying socket,
  true if EINTR was returned by the underlying socket (repeat
    call with same startOffset to resume writing),
  a negated errno value in case of other error.
The positive offset should be used as the next
startOffset value to resume the write .
"

<primitive: 948>
numBuffers _validateClass: SmallInteger .
startOffset _validateClass: SmallInteger .
specArray _validateClass: Array .
numBuffers > 20 ifTrue:[ ArgumentError signal:'numBuffers must be <= 20' ].
(numBuffers * 3) > specArray size ifTrue:[
  ArgumentError signal:'numBuffers larger than specArray'].
startOffset < 1 ifTrue:[ ArgumentError signal:'startOffset must be >= 1'].
self _primitiveFailed:#_writev:specs:byteOffset: 
	args: { numBuffers . specArray . startOffset }
%
category: 'RPC GCI Socket - Accessing'
classmethod:
sendBufferSizeForClient

"Returns the current size in bytes of the socket send buffer used by 
 the RPC GCI client of the current session.  This is the size of the socket 
 buffer used by the client to send data to its gem.

 Raises an exception if the buffer size could not be determined or if the
 session is linked (and therefore does not have socket connection to its client).

 Note: on Linux systems only, the value returned by getsockopt() is larger
 by a factor of 2 than the amount usable memory in the buffer due to overhead.
 This method accounts for this overhead and divides by 2 the result returned
 by getsockopt().  See the man page (socket(7)) for details."

 | result |
result := GsFile classUserAction: #GsfGciGetSocketBufSize 
                 onClient: true with: 'SNDBUF' .
result _isSmallInteger
  ifFalse:[ SocketError signal: result ] .
^ result
%

category: 'RPC GCI Socket - Accessing'
classmethod:
sendBufferSizeForServer

"Returns the current size of the socket send buffer in bytes used by 
 the current session to send data to its client.  This is the size of the socket 
 buffer used by this process to send data to the client.

 Raises an exception if the buffer size could not be determined or if the
 session is linked (and therefore does not have socket connection to its client).

 Note: on Linux systems only, the value returned by getsockopt() is larger
 by a factor of 2 than the amount usable memory in the buffer due to overhead. 
 This method accounts for this overhead and divides by 2 the result returned 
 by getsockopt().  See the man page (socket(7)) for details."

 | result |
result := GsFile classUserAction: #GsfGciGetSocketBufSize 
                 onClient: false with: 'SNDBUF' .
result _isSmallInteger
  ifFalse:[ SocketError signal: result ] .
^ result
%
category: 'RPC GCI Socket - Accessing'
classmethod:
receiveBufferSizeForClient

"Returns the current size of the socket receive buffer in bytes used by 
 the RPC GCI client of the current session.  This is the size of the socket 
 buffer used by the client to receive data from the gem.

 Raises an exception if the buffer size could not be determined or if the
 session is linked (and therefore does not have socket connection to its client).

 Note: on Linux systems only, the value returned by getsockopt() is larger
 by a factor of 2 than the amount usable memory in the buffer due to overhead. 
 This method accounts for this overhead and divides by 2 the result returned 
 by getsockopt().  See the man page (socket(7)) for details."

 | result |
result := GsFile classUserAction: #GsfGciGetSocketBufSize 
                 onClient: true with: 'RCVBUF'.
result _isSmallInteger
  ifFalse:[ SocketError signal: result ] .
^ result
%
category: 'RPC GCI Socket - Accessing'
classmethod:
receiveBufferSizeForServer

"Returns the current size of the socket receive buffer in bytes used by 
 the current session to receive data from its client.  This is the size
 of the socket buffer used by the gem to receive data from its client.

 Raises an exception if the buffer size could not be determined or if the
 session is linked (and therefore does not have socket connection to its client).

 Note: on Linux systems only, the value returned by getsockopt() is larger
 by a factor of 2 than the amount usable memory in the buffer due to overhead. 
 This method accounts for this overhead and divides by 2 the result returned 
 by getsockopt().  See the man page (socket(7)) for details."

 | result |
result := GsFile classUserAction: #GsfGciGetSocketBufSize 
                 onClient: false with: 'RCVBUF' .
result _isSmallInteger
  ifFalse:[ SocketError signal: result ] .
^ result
%

category: 'RPC GCI Socket - Updating'
classmethod:
sendBufferSizeForClient: newSize

"Set the size of the socket send buffer used by the RPC GCI client of 
 the current session to a new size.  This is the size of the socket buffer 
 used by the client to send data to its gem.

 newSize must be an even, positive SmallInteger, larger than the current size.

 Note: on Linux systems only, the implementation of setsockopt() doubles the
 requested buffer size to account for overhead.  See the man page (socket(7))
 for details. It is not recommended to manually set socket buffer sizes on
 Linux, since this disables auto-tuning.

 Returns true if the change in size was successful. Raises an exception if 
 the size change could not be perfomed or if the session is linked (and 
 therefore does not have socket connection to its client)."

| result |
result := GsFile classUserAction: #GsfGciSetSocketBufSize 
                 onClient: true with: 'SNDBUF' with: newSize .
result == true
  ifFalse:[ SocketError signal: result ].
^  true
%

category: 'RPC GCI Socket - Updating'
classmethod:
receiveBufferSizeForClient: newSize

"Set the size of the socket receive buffer used by the RPC GCI
 client of the current session to a new size.  This is the size of the 
 socket buffer used by the client to receive data from its gem.

 newSize must be an even, positive SmallInteger, larger than the current size.

 Note: on Linux systems only, the implementation of setsockopt() doubles the
 requested buffer size to account for overhead.  See the man page (socket(7))
 for details. It is not recommended to manually set socket buffer sizes on
 Linux, since this disables auto-tuning.

 Returns true if the change in size was successful. Raises an exception if  
 the size change could not be perfomed or if the session is linked (and  
 therefore does not have socket connection to its client)."

| result |
result := GsFile classUserAction: #GsfGciSetSocketBufSize 
                 onClient: true with: 'RCVBUF' with: newSize . 
result == true
  ifFalse:[ SocketError signal: result ].
^  true
%
category: 'RPC GCI Socket - Updating'
classmethod:
sendBufferSizeForServer: newSize

"Set the size of the socket send buffer used by the current session
 gem to send data to its client.

 newSize must be an even, positive SmallInteger, larger than the current size.

 Note: on Linux systems only, the implementation of setsockopt() doubles the
 requested buffer size to account for overhead.  See the man page (socket(7)) 
 for details. It is not recommended to manually set socket buffer sizes on 
 Linux, since this disables auto-tuning.

 Returns true if the change in size was successful. Raises an exception if  
 the size change could not be perfomed or if the session is linked (and  
 therefore does not have socket connection to its client)."

| result |
result := GsFile classUserAction: #GsfGciSetSocketBufSize 
                 onClient: false with: 'SNDBUF' with: newSize .
result == true
  ifFalse:[ SocketError signal: result ].
^  true
%

category: 'RPC GCI Socket - Updating'
classmethod:
receiveBufferSizeForServer: newSize

"Set the size of the socket receive buffer used by the current session
 gem to receive data from its client.

 newSize must be an even, positive SmallInteger, larger than the current size.

 Note: on Linux systems only, the implementation of setsockopt() doubles the
 requested buffer size to account for overhead.  See the man page (socket(7)) 
 for details. It is not recommended to manually set socket buffer sizes on 
 Linux, since this disables auto-tuning.

 Returns true if the change in size was successful. Raises an exception if  
 the size change could not be perfomed or if the session is linked (and  
 therefore does not have socket connection to its client)."

| result |
result := GsFile classUserAction: #GsfGciSetSocketBufSize 
                 onClient: false with: 'RCVBUF' with: newSize .
result == true
  ifFalse:[ SocketError signal: result ].
^  true
%

category: 'Low Level Access'
method:
setCloseOnGc: aBoolean

  "If aBoolean is true the receiver's underlying socket will be 
   closed when the in-memory state of the receiver is garbage collected.
   aBoolean must be a Boolean ."

^ self _twoArgPrim: 29 with: aBoolean with: nil 
%
category: 'Queries'
classmethod:
hostIsLocalhost: aString 

"aString must be a hostname or ip address. Returns true
if that host is equivalent to localhost."

^ self _twoArgClassPrim: 30 with: aString with: nil 
%

