!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id: hiddenset.gs 40520 2016-10-20 17:20:32Z bretlb $
! Superclass Hierarchy:
!   System
!
! Moved all of the hidden set methods in system to this file for easier maintenance
!=========================================================================

set class System

category: 'Hidden Set Support'
classmethod:
HiddenSetSpecifiers

"See GsBitmap hiddenSetSpecifiers for new public implementation.

Returns a list of the hidden set specifiers ordered by index.

Hidden sets with index 41 through 45 are for customer use and may be
freely modified using methods in this category.

Other hidden sets are used internally by the system.  Many of them may
only be modifed by SystemUser; others may be accessed using specific
public protocol. Modifying internal hidden sets improperly may result
in gem and/or stone crashes and could potentially corrupt the database."

^#(
" 1"   'ListInstancesResult'
" 2"   'Reserved for GemStone' "Indexing maintenance"
" 3"   'RemovedDepMapEntries'
" 4"   'SaveNewPomObjs' "See method #_enableTraceNewPomObjs"
" 5"   'ObjectsRead'    "See method #_enableTraceObjectsRead"
" 6"   'Reserved for GemStone'
" 7"   'RcReadSet'
" 8"   'DepMapWriteSet'
" 9"   'PomWriteSet'  "Empty except after flush for commit, so only useful
                       after you get a transaction conflict."
"10"   'SaveDepMapChangedObjs'
"11"   'SaveWriteSetUnion'
"12"   'SaveWrittenObjs'
"13"   'ReadWriteConflicts'  "StrongRead-Write conflicts "
"14"   'SaveDepMapChangedUnion'
"15"   'WriteWriteConflicts'
"16"   'WriteDependencyConflicts'
"17"   'WriteReadLockConflicts'
"18"   'WriteWriteLockConflicts'
"19"   'Reserved for GemStone'
"20"   'AllocatedGciOops'
"21"   'Reserved for GemStone'
"22"   'ExportedDirtyObjs'
"23"   'TrackedDirtyObjs'
"24"   'enumeration of ReferencedSet'
"25"   'NotifySet'
"26"   'Reserved1' "used in Conversion"
"27"   'Reserved2' "used in managing depList weak references"
"28"   'Reserved3' "used in object inventory"
"29"   'CommitReleaseLocksSet'
"30"   'CommitOrAbortReleaseLocksSet'
"31"   'Reserved for GemStone'
"32"   'Reserved for GemStone'
"33"   'Reserved for GemStone'
"34"   'GcCandidates'
"35"   'Unused'
"36"   'WriteLockWriteSubset'
"37"   'NewDataPages'
"38"   'StrongReadSet'
"39"   'PureExportSet'
"40"   'GciTrackedObjs'
"41"   'Customer1'
"42"   'Customer2'
"43"   'Customer3'
"44"   'Customer4'
"45"   'Customer5' 
)
%
category: 'Legacy Hidden Set Support'
classmethod:
add: anObject toHiddenSet: hiddenSetSpecifier

"Will be deprecated, new code should use GsBitmaps"

"Adds anObject to the hiddenSet."

(GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: hiddenSetSpecifier)) add: anObject.
^self
%

category: 'Legacy Hidden Set Support'
classmethod:
addAll: anArray toHiddenSet: hiddenSetSpecifier

"Will be deprecated, new code should use GsBitmaps"

"Adds contents of anArray to the hiddenSet (anArray itself is not added to
 hiddenSet). Should only be used with public hidden sets 41-45."

^ (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: hiddenSetSpecifier)) addAll: anArray
%

category: 'Legacy Hidden Set Support'
classmethod:
removeAll: anArray fromHiddenSet: hiddenSetSpecifier

"Will be deprecated, new code should use GsBitmaps"

"Removes the contents of anArray from the hiddenSet (anArray itself is not
 removed from  the hiddenSet).  Objects in the array which are not in the
 hidden set are ignored. Returns a SmallInteger or LargeInteger indicating
 the number of elements successfully removed from the hidden set.

 Should only be used with public hidden sets 41-45."

^ (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: hiddenSetSpecifier)) removeAll: anArray
%

category: 'Private - Hidden Set Support'
classmethod: System
_removeAll: anArray from: hiddenSetSpecifier

"Will be deprecated, new code should use GsBitmaps"

^ self removeAll: anArray fromHiddenSet: hiddenSetSpecifier
%

category: 'Legacy Hidden Set Support'
classmethod:
remove: anObject fromHiddenSet: hiddenSetSpecifier

"Will be deprecated, new code should use GsBitmaps"

"Removes anObject from the hiddenSet.  Do nothing if anObject is not in
 the hidden set.

 Should only be used with public hidden sets 41-45."

| bm |
  bm := (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: hiddenSetSpecifier)).
  (bm includes: anObject) ifTrue: [ bm remove: anObject ].
^ self
%

category: 'Private - Hidden Set Support'
classmethod: System
_remove: anObject from: hiddenSetSpecifier

"Will be deprecated, new code should use GsBitmaps"

^ self remove: anObject fromHiddenSet: hiddenSetSpecifier
%

category: 'Legacy Hidden Set Support'
classmethod:
testIf: anObject isInHiddenSet: hiddenSetSpecifier

"Will be deprecated, new code should use GsBitmaps"

"Returns true if anObject is in the hiddenSet, false otherwise."

^ (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: hiddenSetSpecifier)) includes: anObject
%

category: 'Legacy Hidden Set Support'
classmethod:
hiddenSetAsArray: hiddenSetSpecifier

"Will be deprecated, new code should use GsBitmaps"

"Returns an Array containing the contents of the hiddenSet."

^ (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: hiddenSetSpecifier)) asArray
%

category: 'Legacy Hidden Set Support'
classmethod:
hiddenSetEnumerate: hiddenSetSpecifier limit: maxResultSize

"Will be deprecated, new code should use GsBitmaps"

"Returns an Array containing the first maxResultSize objects
 in the hiddenSet.  If the hidden set contains fewer than
 maxResultSize elements, returns an array containing
 the contents of the hiddenSet. If maxResultSize = 0, the result
 will contain all elements of the hidden set.

 All of the returned elements are removed from the hidden set.
 Should only be used with public hidden sets 41-45."

  | bm arr |
  bm := (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: hiddenSetSpecifier)).
  arr := bm enumerateWithLimit: maxResultSize startingAfter: 0.
  bm removeAll: arr.
^arr
%

category: 'Legacy Hidden Set Support'
classmethod:
hiddenSetEnumerateAsInts: hiddenSetSpecifier limit: maxResultSize

"Will be deprecated, new code should use GsBitmaps"

"Returns an Array containing the numeric object identifiers of the
 first maxResultSize objects in the hiddenSet.  If the hidden set
 contains fewer than maxResultSize elements, returns an array containing
 the contents of the hiddenSet.  If maxResultSize = 0, the result will
 contain all elements of the hidden set.

 All of the returned elements are removed from the hidden set.
 Should only be used with public hidden sets 41-45."

  | bm arr arr1|
  bm := (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: hiddenSetSpecifier)).
  arr := bm enumerateAsOopsWithLimit: maxResultSize startingAfter: 0.
  arr1 := bm enumerateWithLimit: maxResultSize startingAfter: 0.
  bm removeAll: arr1.
^arr
%

category: 'Legacy Hidden Set Support'
classmethod:
hiddenSetSize: hiddenSetSpecifier

"Will be deprecated, new code should use GsBitmaps" 

" Returns the number of elments in the hiddenSet."

^ (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: hiddenSetSpecifier)) size
%

category: 'Legacy Hidden Set Support'
classmethod:
hiddenSet: hiddenSetSpecifier do: aBlock

"Will be deprecated, new code should use GsBitmaps" 

"Executes the one argument block aBlock for each argument
 in the specified hidden set.  Does a destructive enumerate
 of the hidden set. 
 Should only be used with public hidden sets 41-45.
 Returns the number of objects enumerated."

 | bm count |
 bm := (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: hiddenSetSpecifier)).
 count := bm do: aBlock.
 bm removeAll. 
 ^ count
%

category: 'Legacy Hidden Set Support'
classmethod:
hiddenSetReinit: hiddenSetSpecifier

"Will be deprecated, new code should use GsBitmaps" 

"Reinitializes the hiddenSet to empty.  
 Should only be used with public hidden sets 41-45."

^ (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: hiddenSetSpecifier)) removeAll
%


category: 'Legacy Hidden Set Support'
classmethod:
writeHiddenSet: hiddenSetSpecifier toFile: aString

"Will be deprecated, new code should use GsBitmaps" 

"Format of the file has changed between Gs64 v3.4 and v3.5 .

 Write the designated hidden set to the specified bitmap file.  The file 
 must not already exist and the path to the file must be in a directory 
 which is writable by the process.  Any valid hidden set specifier may be 
 used.

 Returns true if successful, false if an error occurred.  Additional error 
 information may be written stderr for the process (either the terminal
 or the gem log file)."

 (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: hiddenSetSpecifier)) writeToFile: aString.
^ true
%

category: 'Legacy Hidden Set Support'
classmethod:
readHiddenSet: hiddenSetSpecifier fromFile: aString

"Will be deprecated, new code should use GsBitmaps" 

"Format of the file has changed between Gs64 v3.4 and v3.5.

 Read the contents of the specified bitmap file into the designated hidden 
 set.  The file must exist and  be readable by the gem process.   
 Should only be used with public hidden sets 41-45. 
 Call System class >> hiddenSetReinit: first to ensure the hidden set is empty.

 Returns true if successful, false if an error occurred.  Additional error 
 information may be written to stderr for the process (either the terminal
 or the gem log file)."

 | bm |
 bm := (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: hiddenSetSpecifier)).
 (bm isEmpty) ifFalse: [ ^ self _error: #rtErrInvalidArgument args: { 'hidden set is not empty' } ].
 bm readFromFile: aString.
 ^true
%

category: 'Legacy Hidden Set Support'
classmethod:
readHiddenSet: hiddenSetSpecifier fromSortedFile: aString

"hiddenSetSpecifier must be a Symbol per GsBitmap class >> hiddenSetSpecifiers .
 Format of the file has changed between Gs64 v3.4 and v3.5 .

 Read the contents of the specified sorted bitmap file into the designated
 hidden set. The file must exist and  be readable by the gem process.
 Should only be used with public hidden sets 41-45.
 Call System class >> hiddenSetReinit: first to ensure the hidden set is empty.

 The file is assumed to be a sorted bitmap file of objects produced by one
 of the Repository>>listInstancesInPageOrder: methods.  The objects are
 loaded into the given hidden set, but the ordering  by page ID is lost
 because hidden sets are always ordered by object ID.  No validation of the
 object IDs is done by this load.

 Returns true if successful, false if an error occurred.  Additional error
 information may be written stderr for the process (either the terminal
 or the gem log file)."

hiddenSetSpecifier _isSymbol ifFalse:[
  Error signal:'Argument must conform to GsBitmap class >> hiddenSetSpecifiers'.
 ].
^self _loadOrStoreHiddenSet: hiddenSetSpecifier toOrFromFile: aString opCode: 2
%
      

category: 'Legacy Hidden Set Support'
classmethod:
addHiddenSet: first to: second

"Will be deprecated, new code should use GsBitmaps" 

"Add all the objects in hidden set first to hidden set second.  Returns the 
 number of objects added to second hidden set that were not already present.
 Should only be used with a second set that is public, sets 41-45.
 User must have permission to modify the second hidden set or be SystemUser."
 
^(GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: first)) addAll:
        (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: second))
%

category: 'Legacy Hidden Set Support'
classmethod:
removeContentsOfHiddenSet: first from: second

"Will be deprecated, new code should use GsBitmaps" 

"Remove all objects in the first hidden set from the second hidden set second. 
 Returns the number of objects removed from second hidden set.  
 Should only be used with a second set that is public, sets 41-45.
 User must have permission to modify the second hidden set or be SystemUser."
 
 | bm1 bm2 |
 bm1 := (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: first)).
 bm2 := (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: second)).
^ bm2 removeAll: bm1
%

category: 'Legacy Hidden Set Support'
classmethod:
computeUnionOfHiddenSet: first and: second into: third

"Will be deprecated, new code should use GsBitmaps" 

"Add every object to the third hidden set that is in both the first and second hidden
 sets.  Returns the number of elements added to the third hidden set.  
 Should only be used with a third set that is public, sets 41-45.
 User must have permission to modify the third hidden set or be SystemUser."
 
 | bm1 bm2 bm3 |
 bm1 := (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: first)).
 bm2 := (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: second)).
 bm3 := (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: third)).
^ bm3 addAll: (bm1 intersect: bm2).
%

category: 'Legacy Hidden Set Support'
classmethod:
computeDifferenceOfHiddenSet: first and: second into: third

"Will be deprecated, new code should use GsBitmaps" 

"Add every object to the third hidden set that is in the first and but not the second hidden
 set.  Returns the number of elements added to the third hidden set.
 Should only be used with a third set that is public, sets 41-45.
 User must have permission to modify the third hidden set or be SystemUser."

| bm1 bm2 bm3 |
 bm1 := (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: first)).
 bm2 := (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: second)).
 bm3 := (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: third)).
^ bm3 addAll: (bm1 difference: bm2).
%

category: 'Legacy Hidden Set Support'
classmethod:
removeFirst: count objectsFromHiddenSet: hiddenSetSpecifier

"Will be deprecated, new code should use GsBitmaps" 

"Remove the first count objects from the given hidden set.  Objects are removed from
 the beginning, going from lowest to highest object ID.   
 
 Returns the number of objects removed from the hidden set.
 Should only be used with a third set that is public, sets 41-45.
 User must have permission to modify the third hidden set or be SystemUser."
 
^ ((GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: hiddenSetSpecifier)) removeFirst: count) size
%

category: 'Legacy Hidden Set Support'
classmethod:
truncateHiddenSet: hiddenSetSpecifier toSize: newSize

"Will be deprecated, new code should use GsBitmaps" 

"Truncate the given hidden set by removing objects from the end of the set until it reaches a
 size of newSize.  Objects are removed in order, going from highest to lowest object ID.   
 
 Returns the number of objects removed from the hidden set.
 Should only be used with a third set that is public, sets 41-45.
 User must have permission to modify the third hidden set or be SystemUser."
 
 | bm bmSize count |
 bm := (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: hiddenSetSpecifier)).
 bmSize := bm size.
 count := bmSize - newSize.
 (count <= 0) ifTrue: [ count := 0].
 bm removeLast: count.
^count
%

category: 'Legacy Hidden Set Support'
classmethod:
writeHiddenSet: hsId toPageOrderFile: aString maxBufferSizeMb: mb
 
"Will be deprecated, new code should use GsBitmaps" 

"Writes the object IDs for objects contained in the given hidden set to a 
 file in page ID order.

 If this session contains uncommitted changes to the repository, the method 
 signals a error: #rtErrAbortWouldLoseData, to indicate that data could be lost.
 Otherwise it puts the session into auto-begin transaction mode and aborts."

self needsCommit
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .

^self _writeHiddenSet: hsId toPageOrderFile: aString maxBufferSizeMb: mb
%

category: 'Legacy Hidden Set Support'
classmethod:
compareHiddenSet: first to: second

"Will be deprecated, new code should use GsBitmaps" 

"Compares two hidden sets to determine if they contain the same objects.
 Returns true if the hidden set contents are identical.  Otherwise returns 
 false.  Neither hidden set is modified."
 
 | bm1 bm2 |
 bm1 := (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: first)).
 bm2 := (GsBitmap newForHiddenSet: (GsBitmap hiddenSetIdAsSymbol: second)).
^ bm1 equals: bm2
%

!=======================================================================================
! Methods to support special gci set 
!=======================================================================================

category: 'Gci Set Support'
classmethod:
add: anObject toGciSet: hiddenSetSpecifier 

"Add the specified object to a GCI hidden set, where
   hiddenSetSpecifier = 39 specifies PureExportSet
   hiddenSetSpecifier = 40 specifies GciTrackedObjs .
 Other values of hiddenSetSpecifier are not allowed.
 Has no effect if anObject is a special object.

 If hiddenSetSpecifier = 39 and a userAction is 
 active an error is generated, since the PureExportSet
 cannot be modified from within a user action.

 If hiddenSetSpecifier = 40 and you have not
 executed System class >> gciTrackedObjsInit once since login,
 generates an error."

^ self _add: anObject toGciSet: hiddenSetSpecifier 
%

category: 'Gci Set Support'
classmethod:
addAll: anArray toGciSet: hiddenSetSpecifier 

"Add contents of anArray to a GCI hidden set, where
   hiddenSetSpecifier = 39 specifies PureExportSet
   hiddenSetSpecifier = 40 specifies GciTrackedObjs .
 Other values of hiddenSetSpecifier are not allowed.

 If hiddenSetSpecifier = 39 and a userAction is 
 active an error is generated, since the PureExportSet
 cannot be modified from within a user action.

 If hiddenSetSpecifier = 40 and you have not
 executed System class >> gciTrackedObjsInit once since login,
 generates an error."

^ self _addAll: anArray toGciSet: hiddenSetSpecifier 
%

category: 'Gci Set Support'
classmethod:
remove: anObject fromGciSet: hiddenSetSpecifier 

"Remove the specified object from a GCI hidden set, where
   hiddenSetSpecifier = 39 specifies PureExportSet
   hiddenSetSpecifier = 40 specifies GciTrackedObjs .
 Other values of hiddenSetSpecifier are not allowed.
 Has no effect if anObject is a special object.

 If hiddenSetSpecifier = 39 and a userAction is 
 active an error is generated, since the PureExportSet
 cannot be modified from within a user action.

 If hiddenSetSpecifier = 40 and you have not
 executed System class >> gciTrackedObjsInit once since 
 login, generates an error."

^ self _remove: anObject fromGciSet: hiddenSetSpecifier 
%

category: 'Gci Set Support'
classmethod:
removeAll: anArray fromGciSet: hiddenSetSpecifier 

"Remove contents of anArray from a GCI hidden set, where
   hiddenSetSpecifier = 39  specifies PureExportSet
   hiddenSetSpecifier = 40  specifies GciTrackedObjs .
 Other values of hiddenSetSpecifier are not allowed.

 If the argument anArray == true, all objects in
 the specified set are removed from that set .

 If hiddenSetSpecifier = 39 and a userAction is 
 active an error is generated, since the PureExportSet
 cannot be modified from within a user action.

 If hiddenSetSpecifier = 40  and you have not
 executed System class >> gciTrackedObjsInit once since 
 login, generates an error."

^ self _removeAll: anArray fromGciSet: hiddenSetSpecifier 
%

category: 'Gci Set Support'
classmethod:
getAndClearGciDirtySet: hiddenSetSpecifier into: anArray

"Destructively enumerate a GCI hidden set, where
   anInt = 22  specifies ExportedDirtyObjs
   anInt = 23  specifies TrackedDirtyObjs .
 Other values of hiddenSetSpecifier are not allowed .

 Corresponding gciTrackedObjsInit or gciDirtyObjsInit must have been 
 sent once during the session before this method can be used.  It is 
 intended that either this method or Gci calls be used to enumerate 
 these hidden sets, not both within one session.  

 If the specified set is empty the result is nil.
 Otherwise the result is an Array containing up to the first 2034 elements 
 of the hidden set, and those element are cleared from the hidden 
 set prior to return from the primitive.   If the set contains
 more than 2034 elements, repeated invocation of this method
 is needed to enumerate the set completely.

 anArray may be nil in which case the result is a newly created Array.
 If anArray is non-nil it is used as the result and is grown as needed.

 For most efficient enumeration use this style:
   [ | arr |
     arr := self _getAndClearGciDirtySet:22 into: arr .
     1 to: arr size do:[:j | |aDirtyObj|
       aDirtyObj := arr at: j .
       aDirtyObj applicationHandleDirtyObj  .
     ].
     arr size == 0 .
   ] untilTrue .
 "
^ self _getAndClearGciDirtySet: hiddenSetSpecifier into: anArray
%

category: 'Private - Gci Set Support'
classmethod:
_addAll: anArray toGciSet: hiddenSetSpecifier 

<primitive: 156>
hiddenSetSpecifier _validateClass: SmallInteger .
(hiddenSetSpecifier < 39 or:[ hiddenSetSpecifier > 40]) ifTrue:[
  hiddenSetSpecifier _error: #rtErrArgOutOfRange args:{ 39 . 40 }
].
anArray _validateClass: Array .
self _primitiveFailed: #_addAll:toGciSet: 
     args: { anArray . hiddenSetSpecifier } .
self _uncontinuableError
%

category: 'Private - Gci Set Support'
classmethod:
_remove: anObject fromGciSet: hiddenSetSpecifier 

<primitive: 157>
hiddenSetSpecifier _validateClass: SmallInteger .
(hiddenSetSpecifier < 39 or:[ hiddenSetSpecifier > 40]) ifTrue:[
  hiddenSetSpecifier _error: #rtErrArgOutOfRange args:{ 39 . 40 }
].
self _primitiveFailed: #_remove:fromGciSet: 
     args: { anObject . hiddenSetSpecifier } .
self _uncontinuableError
%

category: 'Private - Gci Set Support'
classmethod:
_removeAll: anArray fromGciSet: hiddenSetSpecifier 

<primitive: 158>
hiddenSetSpecifier _validateClass: SmallInteger .
(hiddenSetSpecifier < 39 or:[ hiddenSetSpecifier > 40]) ifTrue:[
  hiddenSetSpecifier _error: #rtErrArgOutOfRange args:{ 39 . 40 }
].
anArray _validateClass: Array .
self _primitiveFailed: #_removeAll:fromGciSet:
     args: { anArray . hiddenSetSpecifier } .
self _uncontinuableError
%

category: 'Private - Gci Set Support'
classmethod:
_gciDirtyInit: hiddenSetSpecifier

"Enable GCI tracking of dirty objects.
   hiddenSetSpecifier = 22, equivalent to GciDirtyObjsInit()
   hiddenSetSpecifier = 23, equivalent GciTrackedObjsInit()
 Other values of hiddenSetSpecifier are not allowed."

<primitive: 164>
hiddenSetSpecifier _validateClass: SmallInteger .
(hiddenSetSpecifier < 22 or:[ hiddenSetSpecifier > 23]) ifTrue:[
  hiddenSetSpecifier _error: #rtErrArgOutOfRange args:{ 22 . 23 }
].
self _primitiveFailed: #_gciDirtyInit: args: { hiddenSetSpecifier } .
self _uncontinuableError
%

category: 'Private - Gci Set Support'
classmethod:
_getAndClearGciDirtySet: hiddenSetSpecifier into: anArray

<primitive: 162>
hiddenSetSpecifier _validateClass: SmallInteger .
(hiddenSetSpecifier < 22 or:[ hiddenSetSpecifier > 23]) ifTrue:[
  hiddenSetSpecifier _error: #rtErrArgOutOfRange args:{ 22 . 23 }
].
anArray ~~ nil ifTrue:[ anArray _validateClass: Array ].
self _primitiveFailed: #_getAndClearGciDirtySet:into:
     args: { hiddenSetSpecifier . anArray } .
self _uncontinuableError
%

category: 'Private - Hidden Set Support'
classmethod:
_add: anObject toGciSet: hiddenSetSpecifier 

<primitive: 155>
hiddenSetSpecifier _validateClass: SmallInteger .
(hiddenSetSpecifier < 39 or:[ hiddenSetSpecifier > 40]) ifTrue:[
  hiddenSetSpecifier _error: #rtErrArgOutOfRange args:{ 39 . 40 }
].
self _primitiveFailed: #_add:toGciSet: 
     args: { anObject . hiddenSetSpecifier } .
self _uncontinuableError
%

!=======================================================================================
! Private methods used in the image
!=======================================================================================

category: 'Private - Hidden Set Support'
classmethod:
_conflictsReport: aBoolean withDetails: detailsBool

"If aBoolean true, 
 include conflict sets even if commit result read-only or success.
 Returns an Array describing conflicts or success of the last attempted commit.

 First element of the Array is a symbol describing the comit status.
 If detailsBool == false
   Second element is nil .
   Remaining elements of the Array are one or more triples
      conflictKindSymbol .  Array of objects in conflict .  nil

 If  detailsBool == true   and
 (self gemConfigurationAt: #GemCommitConflictDetails) == true 
 then
   Second element is a DateAndTime of the attempted commit.
   Remaining elements of the Array are one or more triples
     conflictKindSymbol .  Array of objects in conflict .  SessionsArray

   A SessionsArray is composed of one or more pairs
     String describing the session 
     Array of objects written by that session .
"
| status res detailedConflicts |

status := self _commitResult + 2  .
res :=  { }  .
res add: ( #( #readOnly  		"must agree with _commitResult method"
		 #success 
		 #rcFailure 
	         #dependencyFailure 
                 #failure 
		 #retryFailure
                 #commitDisallowed
                 #retryLimitExceeded ) at:  status )  .

( detailsBool and:[ self _gemCommitConflictDetails and:[ status >= 1 ]]) ifTrue:[
  detailedConflicts := self _detailedConflicts . 
].
detailedConflicts ifNotNil:[ | timeStr |
  (detailedConflicts at: 1) ifNotNil:[:tm |
     timeStr := (DateAndTime posixSeconds: tm asFloat / 1000 offset: nil) asStringMs
  ].
  res add: 'Attempt to commit at: ', (timeStr ifNil:[ '(Time not available)' ]) .
] ifNil:[ res add: nil ].

(status > 2 or:[ aBoolean]) ifTrue:[
  | blk blk3 wwConflicts rcReadSet rcObject sess rcRetryFailureReason rcRetryFailureDescription |

  blk := [ :key :setNum | | anArr |
     (self hiddenSetSize: setNum) > 0 ifTrue:[
        res add: key ; add:( anArr:= self hiddenSetAsArray: setNum) .
        res add: nil . "never any session details for these keys"
     ].
     anArr .
  ].
  blk3 := [ :key :setNum :details | | anArr |
     (self hiddenSetSize: setNum) > 0 ifTrue:[ | sesArray |
        res add: key ; add:( anArr:= self hiddenSetAsArray: setNum) .
        sesArray := nil . "Pairs of commitDescrString  { objs }" 
        details ifNotNil:[  | dOfs |
          dOfs :=  setNum == 15 ifTrue:[ 4 "w-w"] ifFalse:[ 5 "w-dep" ].
          2 to: details size do:[:dx | | dElem ww str tim | 
            dElem := details at: dx .
            (ww := dElem atOrNil: dOfs ) ifNotNil:[
              (str := 'session ' copy) add: (dElem at: 1) asString .
              tim := DateAndTime posixSeconds:(dElem at: 2) asFloat / 1000 offset: nil .  
              str add: ' at '; add: tim asStringMs ;
                  add: ' userId '; 
                  add: ((dElem at: 3) ifNil:[ 'unknown'] ifNotNil:[:up | up userId ]).
              sesArray ifNil:[ sesArray := { } ].
              sesArray add: str ; add: ww .
            ]
          ].
        ]. 
        res add: sesArray .
     ].
     anArr .
  ].

  blk value:#'Read-Write' value: 13 . "StrongRead - Write conflicts"
  wwConflicts := blk3 value:#'Write-Write' value: 15 value: detailedConflicts  .

  rcReadSet := blk value:#'RcReadSet' value: 7 .
  (wwConflicts size ~~ 0 and:[rcReadSet size ~~ 0]) ifTrue:[
    res add:#'WriteWrite_minusRcReadSet' ; 
      add:(Array withAll:(IdentitySet withAll: wwConflicts) - (IdentitySet withAll: rcReadSet));
      add: nil. "RcReadSet is for this session only, never other sessions info"
  ].
  blk3 value: #'Write-Dependency' value: 16 value: detailedConflicts .
  blk value: #'Write-ReadLock' value: 17 .
  blk value: #'Write-WriteLock' value: 18 .
  rcObject := self rcValueCacheAt: #'Rc-Write-Write' for: self otherwise: nil.
  rcObject ifNotNil:[ 
      res add: #'Rc-Write-Write' ; add: { rcObject }; add: nil .

  rcRetryFailureReason := self rcValueCacheAt: #'Rc-Retry-Failure-Reason' for: self otherwise: nil.
  rcRetryFailureReason ifNotNil:[ 
      res add: #'Rc-Retry-Failure-Reason' ; add: { rcRetryFailureReason } ; add: nil ].
  rcRetryFailureDescription := self rcValueCacheAt: #'Rc-Retry-Failure-Description' for: self otherwise: nil.
  rcRetryFailureDescription ifNotNil:[ 
      res add: #'Rc-Retry-Failure-Description' ; add: { rcRetryFailureDescription } ; add: nil ]
  ].
  sess := self rcValueCacheAt: #'Synchronized-Commit' for: self otherwise: nil.
  sess ifNotNil: [
      res add: #'Synchronized-Commit' ; add: { sess }; add: nil .
  ].
].
^ res
%


category: 'Reduced Conflict Support'
classmethod:
_getRedoAndConflictObjects

"Builds a list of objects that need to be scanned because they need to
 have conflicts resolved.  If the result array is empty, then the transaction
 has conflicts that cannot be resolved because we should only get here if
 some Rc conflicts have been detected."
          
| wwConflicts redoLog result |
     
	" Gs64 v2.0 optimization, access the hidden sets directly 
  	instead of computing complete transaction conflicts info."

" if there are read-write and write-read conflicts, we cannot resolve them "
"check for #'Read-Write' , i.e. strongRead-write conflicts "
((GsBitmap newForHiddenSet: #ReadWriteConflicts) size) > 0 ifTrue:[
  "cannot resolve strongRead-write conflicts. The Read-Write conflict
   has already had RcReadSet subtracted from it by the commit primitive."
  ^ nil  
].
" check for #'Write-Dependency' conflicts"
((GsBitmap newForHiddenSet: #WriteDependencyConflicts) size) > 0 ifTrue:[
  "cannot resolve write-dependency conflicts. "
  ^ nil  
].

wwConflicts := (GsBitmap newForHiddenSet: #WriteWriteConflicts) asArray.
wwConflicts size == 0 ifTrue: [ 
  "expected to find write-write conflicts to resolve, so fail"
  ^ nil 
].
result := { } .
redoLog := self _redoLog.
redoLog ifNil:[
  " no redo log, treat objects themselves as the one on which conflicts
        must be resolved "
  1 to: wwConflicts size do: [ :i | | conflictObject |
    conflictObject := wwConflicts at: i.
    result add: conflictObject ; add: { conflictObject }.
  ].
] ifNotNil:[ | conflictObject |
  1 to: wwConflicts size do: [ :i | | redoObject |
    conflictObject := wwConflicts at: i.
  
    " get the object that has to be replayed due to this conflict "
    redoObject := redoLog getRedoObjectForConflictingObject: conflictObject.
    redoObject ifNil:[
      conflictObject isInternalObject ifTrue:[
        "a leaf node that will be taken care of by replay of a parent"
      ] ifFalse:[ "RcPipe component or similar"
        (result includesIdentical: redoObject) ifFalse:[
          result add: conflictObject ; add: { conflictObject }.
        ] 
      ].
    ] ifNotNil:[  | idx |
      (idx := result indexOfIdentical: redoObject) ~~ 0 ifTrue: [ 
	(result at: idx + 1) add: conflictObject.
      ] ifFalse:[
	result add: redoObject ; add: { conflictObject }.
      ].
    ].
  ].
].
^ result
%

category: 'Reduced Conflict Support'
classmethod:
_resolveRcConflicts
        
  "Checks for selected conflicts.  If some are found, attempt to resolve those
   conflicts.  If any conflicts could not be resolved, returns false."
 [    
  | scanArray redoObject replayFailed conflicts idx limit |
  scanArray := self _getRedoAndConflictObjects .
  scanArray ifNil: [ 
    "no redo objects were found, cannot resolve conflicts"
    self _breakSerialization; _disallowSubsequentCommits: 3 .
    ^ false 
  ].
  replayFailed := false.
  limit := scanArray size .
  idx := 1 . 
  [ idx <= limit ] whileTrue:[ 
		redoObject := scanArray at: idx .
		conflicts := scanArray at: idx + 1 .
		idx := idx + 2 .
		(redoObject _resolveRcConflictsWith: conflicts) ifFalse: [
			self rcValueCacheAt: #'Rc-Write-Write' put: redoObject for: self.
			replayFailed := true.
			limit := 0 "exit loop" .
		].
	].
  replayFailed ifTrue: [
    " force subsequent attempts to commit to fail with retry limit error"
    self _breakSerialization; _disallowSubsequentCommits: 4 .
    ^ false
  ].      
  ^ true
 ] onException: Error do:[:ex |
   self _breakSerialization; _disallowSubsequentCommits: 5 .
   ex pass .
   ^ false
 ]
%

!=======================================================================================
!  hiddenSet private methods (only included because they are not yet deprecated)
!=======================================================================================

category: 'Private - Hidden Set Support'
classmethod:
_add: anObject to: hiddenSetSpecifier

"Will be deprecated, new code should use GsBitmaps" 

^ self add: anObject to: hiddenSetSpecifier
%

category: 'Private - Hidden Set Support'
classmethod:
_addAll: anArray to: hiddenSetSpecifier

"Will be deprecated, new code should use GsBitmaps" 

^ self addAll: anArray toHiddenSet: hiddenSetSpecifier
%

category: 'Private - Hidden Set Support'
classmethod:
_hiddenSetAsArray: hiddenSetSpecifier

"Will be deprecated, new code should use GsBitmaps" 

^ self hiddenSetAsArray: hiddenSetSpecifier
%

category: 'Private - Hidden Set Support'
classmethod:
_hiddenSetEnumerate: hiddenSetSpecifier limit: maxResultSize

"Will be deprecated, new code should use GsBitmaps" 

^ self hiddenSetEnumerate: hiddenSetSpecifier limit: maxResultSize
%

category: 'Private - Hidden Set Support'
classmethod:
_hiddenSetEnumerateAsInts: hiddenSetSpecifier limit: maxResultSize

"Will be deprecated, new code should use GsBitmaps"

^ self hiddenSetEnumerateAsInts: hiddenSetSpecifier limit: maxResultSize
%

category: 'Private - Hidden Set Support'
classmethod:
_hiddenSetReinit: hiddenSetSpecifier

"Will be deprecated, new code should use GsBitmaps"

^ self hiddenSetReinit: hiddenSetSpecifier
%

category: 'Private - Hidden Set Support'
classmethod:
_hiddenSetSize: hiddenSetSpecifier

"Will be deprecated, new code should use GsBitmaps"

^ self hiddenSetSize: hiddenSetSpecifier
%

category: 'Private - Hidden Set Support'
classmethod:
_performSetArithmeticOnHiddenSets: first and: second storeResultsInto: third opCode: anOpCode

"Will be deprecated, new code should use GsBitmaps"

"Perform a set arithmetic operation on the hidden sets with the specifiers
 given by first and second and possiblly store the results into third.
 The behavior of this primitive is as follows:

 opCode 	Operation	 Result stored in third
 =================================================================
 0		Add		 Not used, third expected to be 0.
 1		Subtract	 Not used, third expected to be 0.
 2		Union 		 result of the union
 3		Difference	 result of difference
 4		Clear from start Not used
 5		Clear from end   Not used
 6		Compare          Not used
 =================================================================
"

(anOpCode == 0) ifTrue: [ ^ self addHiddenSet: first to: second ].
(anOpCode == 1) ifTrue: [ ^ self removeContentsOfHiddenSet: first from: second ].
(anOpCode == 2) ifTrue: [ ^ self computeUnionOfHiddenSet: first and: second into: third ].
(anOpCode == 3) ifTrue: [ ^ self computeDifferenceOfHiddenSet: first and: second into: third ].
(anOpCode == 4) ifTrue: [ ^ self removeFirst: second objectsFromHiddenSet: first ].
(anOpCode == 5) ifTrue: [ ^ self truncateHiddenSet: first toSize: second ].
(anOpCode == 6) ifTrue: [ ^ self compareHiddenSet: first to: second ].
%

category: 'Private - Hidden Set Support'
classmethod:
_primHiddenSetEnumerate: hiddenSetSpecifier limit: maxResultSize storeAsInts: aBool

"Will be deprecated, new code should use GsBitmaps"

aBool ifTrue: [ ^ self hiddenSetEnumerateAsInts: hiddenSetSpecifier limit: maxResultSize ]
      ifFalse: [ ^ self hiddenSetEnumerate: hiddenSetSpecifier limit: maxResultSize ]
%

category: 'Private - Hidden Set Support'
classmethod:
_testIf: anObject isIn: hiddenSetSpecifier

"Will be deprecated, new code should use GsBitmaps"

^ self testIf: anObject isInHiddenSet: hiddenSetSpecifier
%

category: 'Private - Hidden Set Support'
classmethod:
_writeHiddenSet: hsId toPageOrderFile: aString maxBufferSizeMb: mb

"Will be deprecated, new code should use GsBitmaps"

"Writes the object IDs for objects contained in the given hidden set to a 
 file in page ID order.
 
 The hsid argument must be a SmallInteger indicating a hidden set id.  
 See the method System>>HiddenSetSpecifiers for a list of hidden set
 identifiers. The hidden set is not modified by this method.  It is an
 error if the hidden set is empty.

 aString must be the full path to a new file on the gem's host system 
 which will be created by this method.  It is an error if the file
 already exists.

 This method creates a temporary memory buffer used to sort objects by page ID.
 The mb argument specifies the maximum number of megabytes of memory to request
 from the operating system.  A value of 0 means the amount of memory to request
 is not limited.  Negative values are not allowed.

 The contents of the result file are guaranteed to be in page-order if, 
 and only if the method was able to acquire enough memory to sort the entire
 hidden set by page ID.  Less than the optimial amount of memory may be 
 available because of limits imposed by the mb argument, or the operating
 system.  In such cases where the optimal amount of memory is not available,
 the method will still succeed, however only parts of the result file
 will be in page order.   Methods that read page-ordered files (see list
 below) will still operate correctly on files with multiple page-ordered 
 sections.  The optimal memory buffer size is approximately 10 bytes per
 object ID contained in the hidden set.

 The result file size will be approximately 5 bytes for each object in the 
 hidden set, plus 24 bytes.  For example: for a hidden set which contains 
 10 million objects, this method will generate a file approximately
 50 megabytes in size.

 The result file may be accessed using the following methods:
   System>>readHiddenSet: hiddenSetSpecifier fromSortedFile: aString
   Repository>>openPageOrderOopFile: aString
   Repository>>readObjectsFromFileWithId: aSmallInt startingAt: startIndex 
               upTo: endIndex into: anArray
   Repository>>numberOfObjectsInPageOrderOopFileWithId: aSmallInt
   Repository>>closePageOrderOopFileWithId: aSmallInt
   Repository>>auditPageOrderOopFileWithId: aSmallInt

 Returns an Array with elements as follows:
  1 - the number of objects written to the file.
  2 - the number of sections in the file ordered by page ID.  This element 
      will be 1 if the optimal amount of memory was available, or greater 
      than 1 if not."
      
 
<primitive: 1022>
hsId _validateClass: SmallInteger .
mb _validateClass: SmallInteger .
aString _validateKindOfClass: String .
^ self _primitiveFailed: #writeHiddenSet:toPageOrderFile:maxBufferSizeMb:
       args: { hsId . aString . mb }
%

