!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   KeySoftValueDictionary, KeyValueDictionary, AbstractDictionary, Collection, Object.
!
!=========================================================================

expectvalue %String
run
| res cls oldCls clsName |
clsName := #KeySoftValueDictionary .
oldCls := Globals at:clsName otherwise: nil .
oldCls ifNil:[
  res := KeyValueDictionary _newKernelSubclass: clsName"KeySoftValueDictionary"
    instVarNames: #( )
    classVars: #()
    classInstVars: #()
    poolDictionaries: { }
    inDictionary: Globals
    options: #( instancesNonPersistent  disallowGciStore )
    reservedOop: 941 .
] ifNotNil: [
  res := clsName asString , 'already exists'.
  oldCls instancesNonPersistent ifFalse:[ nil error:'should be NP' ].
].
^ res
%

removeallmethods KeySoftValueDictionary
removeallclassmethods 
set class KeySoftValueDictionary

category: 'For Documentation Installation only'
classmethod
installDocumentation

self comment:
'KeySoftValueDictionary is a subclass of KeyValueDictionary.
 Instances of KeySoftValueDictionary are non-persistent and may not be
 committed ot the repository.

 All values in KeySoftValueDictionary are instances of SoftReference .

 Various instance methods in KeySoftValueDictionary will automatically
 remove from a KeySoftValueDictionary any SoftReferences whose value
 instVar has been cleared by the in-memory garbage collector.

 See comments on GC behavior in $GEMSTONE/bin/initial.config under
 GEM_SOFTREF_CLEANUP_PERCENT_MEM and GEM_KEEP_MIN_SOFTREFS. 

Constraints:
	numElements: SmallInteger
	numCollisions: SmallInteger
	collisionLimit: SmallInteger
	tableSize: SmallInteger' .
%

category: 'Private'
method
collisionBucketClass

"Returns the class of object to create when keys collide."

^ SoftCollisionBucket
%

category: 'Private
method
_referenceAt: tableIdx 

" for the key stored in the hash table at tableIdx ,
  return the softRef if softRef.value non-nil, otherwise 
  remove the soft reference and associated key .
"
| aSoftRef val |
aSoftRef := self _at: tableIdx + 1 .
aSoftRef ifNotNil:[
  val := aSoftRef _value .
  val ifNil:[ 
    self _basicAt: tableIdx     put: nil . "remove key"
    self _basicAt: tableIdx + 1 put: nil . "remove SoftRef"
    numElements := numElements - 1 .
    ^ nil .
  ].
  ^ aSoftRef
].
^ nil
%

category: 'Accessing'
method
referenceAt: aKey otherwise: aValue

" Return the SoftReference corresponding to aKey ,
  or return aValue if aKey not found."

| hash hashKey collisionBkt ref |

aKey ifNil:[ ^ aValue ].
hash := self hashFunction: aKey.
hashKey := self keyAt: hash.
hashKey ifNil: [
  (collisionBkt := self valueAt: hash) ifNotNil: [
    ref := collisionBkt referenceAt: aKey  otherwise: nil .
  ]
] ifNotNil: [ 
  (self compareKey: aKey with: hashKey) ifTrue: [ 
     ref := self _referenceAt: ( hash + hash - 1) .
   ] 
].
ref ifNotNil:[ ^ ref ].
^ aValue 
%

category: 'Accessing'
method
referenceAt: aKey

"Returns the SoftReference corresponding to aKey. 
 Generates an error if no such key exists."

| ref |
ref := self referenceAt: aKey otherwise: nil .
ref ifNotNil:[ ^ ref ].
^ self _errorKeyNotFound: aKey
%


category: 'Private
method:
_numValues
  "Return the number of SoftReferences in receiver which have non-nil values,
   without removing any SoftReferences  from the receiver."
  | count |
  count := 0 .
  self keysAndReferencesDo:[:k :ref |
    ref _value ifNotNil:[ count := count + 1 ].
  ].
  ^ count
%
method
_bucketSizeChangeFrom: oldNumElem to: newNum key: aKey bucket: aBucket
    remove: okToRemoveBucket

  "a bucket has removed some key/softRef pairs , now update
   dictionary size information."

| elemDelta collDelta |
elemDelta := oldNumElem - newNum .
elemDelta < 1 ifTrue:[ self error:'elemDelta underflow in a KeySoftValueDictionary' ].
collDelta := elemDelta .
newNum <= 0 ifTrue:[  
  newNum < 0 ifTrue:[ self error:'numElem underflow in a KeySoftValueDictionary' ].
  collDelta := elemDelta - 1 .
  collDelta < 0 ifTrue:[ self error:'collDelta underflow in a KeySoftValueDictionary' ].
].
numElements := numElements - elemDelta .
numCollisions := numCollisions - collDelta .
(okToRemoveBucket and:[ newNum <= 1 ]) ifTrue:[ | hash |
  "remove the bucket."
  aKey ifNil:[ self error:'nil key in _bucketSizeChangeFrom:'].
  hash := self hashFunction: aKey.
  (self valueAtHash: hash ) ~~ aBucket ifTrue:[ 
    self error:'inconsistent bucket in a KeySoftValueDictionary'
  ].
  newNum == 1 ifTrue:[ | pair pkey |
    pair := aBucket firstPair .
    (self hashFunction: (pkey := pair at:1)) == hash ifFalse:[
      self error:'inconsistent hash in _bucketSizeChangeFrom:'
    ].
    self atHash: hash putKey: pkey value: (pair at: 2) .
  ] ifFalse:[
    self atHash: hash putKey: nil value: nil 
  ]. 
]
%

! fixed 32863
category: 'Enumerating'
method
keysAndValuesDo: aBlock
 
"Evaluates aBlock with each of the receiver's key/value pairs as the
 arguments.  The argument aBlock must be a two-argument block,
 first argument is the key and the second argument is the value of
 the SoftReference for each non-cleared key/softRef pair in the dictionary.

 SoftReferences which have been cleared by the garbage collector
 are removed from the dictionary during enumeration of the dictionary.
 The useCount of each non-cleared SoftReference will be incremented.
"

| tableLimit |
tableLimit := tableSize * 2 .
1 to: tableLimit by: 2 do: [ :tableIndex | | aKey |
  aKey := self _at: tableIndex .
  aKey ifNil:[ | collisionBkt |
    collisionBkt := self _at: (tableIndex + 1).
    collisionBkt ifNotNil:[  | bktSize bktIdx |
      bktIdx := 1 .
      bktSize := collisionBkt _basicSize . 
      [ bktIdx <= bktSize] whileTrue:[  | bKey |
        bKey := collisionBkt _at: bktIdx .
        bKey ifNotNil:[  | oldSize bRef |
          oldSize := bktSize . 
          bRef := collisionBkt _referenceAt: bktIdx . "possible removal"
          bRef ifNotNil:[ | val |
            val := bRef value . 
            val ifNotNil:[
              aBlock value: bKey value: val
            ].
          ] .
          "advance bktIdx only if no removal done "
          bktSize := collisionBkt _basicSize . 
          bktSize = oldSize ifTrue:[ bktIdx := bktIdx + 2 ].
        ] ifNil:[
          bktIdx := bktIdx + 2.
        ]
      ]
    ].  
  ] ifNotNil: [ | aRef |
    aRef := self _referenceAt: tableIndex .
    aRef ifNotNil:[ | val |
      val := aRef value .
      val ifNotNil:[
        aBlock value: aKey value: val
      ].
    ].
  ].
].
%

! added for 36675, renamed for 39898
category: 'Enumerating'
method
accompaniedBy: anObj keysAndValuesDo: aBlock

"Evaluates aBlock with each of the receiver's key/value pairs as the
 2nd and 3rd arguments.
 aBlock must be a 3 argument block, with arguments anObj, key value .

 SoftReferences which have been cleared by the garbage collector
 are removed from the dictionary during enumeration of the dictionary.
 The useCount of each non-cleared SoftReference will be incremented.
"

| tableLimit |
tableLimit := tableSize * 2 .
1 to: tableLimit by: 2 do: [ :tableIndex | | aKey |
  aKey := self _at: tableIndex .
  aKey ifNil: [ | collisionBkt |
    collisionBkt := self _at: (tableIndex + 1).
    collisionBkt ifNotNil:[  | bktSize bktIdx |
      bktIdx := 1 .
      bktSize := collisionBkt _basicSize . 
      [ bktIdx <= bktSize] whileTrue:[  | bKey |
        bKey := collisionBkt _at: bktIdx .
        bKey ifNotNil:[  | oldSize bRef |
          oldSize := bktSize . 
          bRef := collisionBkt _referenceAt: bktIdx . "possible removal"
          bRef ifNotNil:[ | val |
            val := bRef value . 
            val ifNotNil:[
              aBlock value: anObj value: bKey value: val
            ].
          ] .
          "advance bktIdx only if no removal done "
          bktSize := collisionBkt _basicSize . 
          bktSize = oldSize ifTrue:[ bktIdx := bktIdx + 2 ].
        ] ifNil:[
          bktIdx := bktIdx + 2.
        ]
      ]
    ].  
  ] ifNotNil: [ | aRef |
    aRef := self _referenceAt: tableIndex .
    aRef ifNotNil:[ | val |
      val := aRef value .
      val ifNotNil:[
        aBlock value: anObj value: aKey value: val
      ].
    ].
  ].
].

%

category: 'Enumerating'
method
keysDo: aBlock
 
"Evaluates aBlock with each of the receiver's keys as the
 argument.  The argument aBlock must be a one-argument block .

 SoftReferences which have been cleared by the garbage collector
 are removed from the dictionary during enumeration of the dictionary,
 and their keys will not be passed to aBlock .

 The useCount of each non-cleared SoftReference is NOT incremented.
"

| tableLimit |
tableLimit := tableSize * 2 .
1 to: tableLimit by: 2 do: [ :tableIndex | | aKey |
  aKey := self _at: tableIndex .
  aKey ifNil:[ | collisionBkt |
    collisionBkt := self _at: (tableIndex + 1).
    collisionBkt ifNotNil:[  | bktSize bktIdx |
      bktIdx := 1 .
      bktSize := collisionBkt _basicSize . 
      [ bktIdx <= bktSize] whileTrue:[  | bKey |
        bKey := collisionBkt _at: bktIdx .
        bKey ifNotNil:[  | oldSize bRef |
          oldSize := bktSize . 
          bRef := collisionBkt _referenceAt: bktIdx . "possible removal"
          bRef ifNotNil:[ | val |
            val := bRef _value . 
            val ifNotNil:[
              aBlock value: bKey .
            ].
          ] .
          "advance bktIdx only if no removal done "
          bktSize := collisionBkt _basicSize . 
          bktSize = oldSize ifTrue:[ bktIdx := bktIdx + 2 ].
        ] ifNil:[
          bktIdx := bktIdx + 2.
        ]
      ]
    ].  
  ] ifNotNil: [ | aRef |
    aRef := self _referenceAt: tableIndex .
    aRef ifNotNil:[ | val |
      val := aRef _value .
      val ifNotNil:[
        aBlock value: aKey 
      ].
    ].
  ].
].
%

! fixed 32863
category: 'Enumerating'
method
keysAndReferencesDo: aBlock

"Evaluates aBlock with each of the receiver's key/softRef pairs as the
 arguments.  The argument aBlock must be a two-argument block,  
 first argument is the key and the second argument is 
 the SoftReference for each non-cleared key/softRef pair in the dictionary.

 SoftReferences which have been cleared by the garbage collector 
 are removed from the dictionary during enumeration of the dictionary.
 The useCount of each non-cleared SoftReference will NOT be incremented.
"

| tableLimit |
tableLimit := tableSize * 2 .
1 to: tableLimit by: 2 do: [ :tableIndex | | aKey |
  aKey := self _at: tableIndex .
  aKey ifNil:[ | collisionBkt |
    collisionBkt := self _at: (tableIndex + 1) .
    collisionBkt ifNotNil:[  | bktSize bktIdx |
      bktIdx := 1 .
      bktSize := collisionBkt _basicSize .
      [ bktIdx <= bktSize] whileTrue:[  | bKey |
        bKey := collisionBkt _at: bktIdx .
        bKey ifNotNil:[  | oldSize bRef |
          oldSize := bktSize . 
          bRef := collisionBkt _referenceAt: bktIdx  . "possible removal"
          bRef ifNotNil:[ 
            aBlock value: bKey value: bRef  
	  ].
          "advance bktIdx only if no removal done "
          bktSize := collisionBkt _basicSize .
          bktSize = oldSize ifTrue:[ bktIdx := bktIdx + 2 ].
        ] ifNil:[
          bktIdx := bktIdx + 2.
        ]
      ].
    ].
  ] ifNotNil: [ | aRef |
    aRef := self _referenceAt: tableIndex .
    aRef ifNotNil:[ 
      aBlock value: aKey value: aRef 
    ].
  ].
].
%

!  fixed  32785 , 32863
category: 'Accessing'
method
keyAtValue: anObject ifAbsent: aBlock

"Returns the key of the first SoftRef equal to the given object, anObject.
 If no match is found, evaluates and returns the result of the block aBlock."

| tableLimit |
tableLimit := tableSize * 2 .
1 to: tableLimit by: 2 do: [ :tableIndex | | aKey |
  aKey := self _at: tableIndex .
  aKey ifNil: [ | collisionBkt |
    collisionBkt := self _at: tableIndex + 1.
    collisionBkt ifNotNil:[  | bktSize bktIdx |
      bktIdx := 1 .
      bktSize := collisionBkt _basicSize .
      [ bktIdx <= bktSize] whileTrue:[  | bKey |
        bKey := collisionBkt _at: bktIdx .
        bKey ifNotNil:[ | oldSize bRef |
          oldSize := bktSize .
          bRef := collisionBkt _referenceAt: bktIdx .  "possible removal"
          bRef ifNotNil:[
            anObject = bRef _value ifTrue: [
              ^bKey
            ].
          ].
          "advance bktIdx only if no removal done "
          bktSize := collisionBkt _basicSize .
          bktSize = oldSize ifTrue:[ bktIdx := bktIdx + 2 ].
        ] ifNil:[
          bktIdx := bktIdx + 2.
        ].
      ].
    ].
  ] ifNotNil: [ | aSoftRef |
    aSoftRef := self _referenceAt: tableIndex .
    aSoftRef ifNotNil:[
      anObject = aSoftRef _value ifTrue:[
        ^aKey
      ].
    ].
  ].
].
^aBlock value.
%

category: 'Accessing'
method
at: aKey ifAbsent: aBlock

"Returns the value of the SoftReference that corresponds to aKey.  
 If no such key/SoftRef pair exists, returns the result of 
 evaluating the zero-argument block aBlock.

 Does not send setInUse to matching SoftReference.
 The useCount of the matching SoftReference is incremented, if the
 returned value is not special ."

| ref |

ref := self referenceAt: aKey otherwise: nil .
ref ifNil:[ 
  ^self _reportKeyNotFound: aKey with: aBlock
].
^ ref value
%

category: 'Accessing'
method
at: aKey otherwise: aValue

"Returns the value of the SoftReference that corresponds to aKey.
 If no such key/softRef pair exists, returns the given alternate value.  
 Does not send setInUse to the matching SoftReference. 
 The useCount of the matching SoftReference is incremented, if the
 returned value is not special ."

| ref |
ref := self referenceAt: aKey otherwise: nil .
ref ifNotNil:[ ^ ref value ].
^ aValue
%


category: 'Updating'
method
at: aKey putReference: aSoftReference

"Stores the aKey/SoftRef pair in the hash dictionary.  Rebuilds the hash table
 if the addition caused the number of collisions to exceed the limit allowed.
 
 aSoftReference must be a instance of SoftReference , else an error is 
 generated .
 The useCount of aSoftReference is incremented.

 Returns aSoftReference, or nil if the addition could not be completed.
 Returns nil if aSoftReference was cleared by the garbage collector
 before the addition could be completed."

| val |
aSoftReference class == SoftReference ifFalse:[
  aSoftReference _validateInstanceOf: SoftReference .
].
val := aSoftReference value  .  "increment useCount and keep val alive on stack"
val ifNil:[ ^ nil ].

super at: aKey put: aSoftReference .
^ aSoftReference
%

category: 'Updating'
method
at: aKey put: aValue

"Creates a SoftReference referencing aValue and
 inserts the key/softRef pair into the receiver .

 Rebuilds the hash table
 if the addition caused the number of collisions to exceed the limit allowed.

 Returns aValue. 
"
| ref validRef |
ref := SoftReference new .
ref setValue: aValue .
validRef := self at: aKey putReference: ref .
" should always get the ref back because we have a strong ref to value
  on the stack"
validRef == ref ifFalse:[ self error:'inconsistent softref state' ].
^ aValue
%

category: 'Clustering'
method
clusterDepthFirst

"Instances are non-persistent so this has no effect"

^ true
%


category: 'Accessing'
method
at: aKey ifAbsentPut: aBlock

"disallowed"
^ self shouldNotImplement: #at:ifAbsentPut:
%

! change to not increment use count
category: 'Hashing'
method
rebuildTable: newSize

"Rebuilds the hash table by saving the current state, initializing and
 changing the size of the table, and adding the key value pairs saved
 back to the hash dictionary.

 During the 'saving the current state' phase, cleared SoftReferences
 will be removed from the dictionary.  After 'saving the current state',
 if numCollisions is below the collisionLimit, the rebuild will be skipped. 

 If explicit rebuildTable is desired, you need to set the collisionLimit
 lower than the number of collisions before sending rebuildTable: .
 rebuildTable: will then install a new collision limit based on the
 newSize .  "

| startNc nc saveTable saveIdx saveCollLimit |

collisionLimit == 536870911 ifTrue:[
  ^ self  "prevent recursive rebuild"
  ].
saveTable := Array new: (self size * 3).
saveIdx := 0.
startNc := numCollisions .
self keysAndReferencesDo: [ :aKey :aSoftRef |
  saveTable at: saveIdx + 1 put: aKey.
  saveTable at: saveIdx + 2 put: aSoftRef.

  "keep values mostly alive"
  saveTable at: (saveIdx := saveIdx + 3) put: (aSoftRef _value)   "avoid incrementing useCount"
  ].
"enumeration of soft dictionaries may return less than total size"
saveTable size: saveIdx  .
((nc := numCollisions) < collisionLimit and:[ nc >= startNc]) ifTrue:[
  "rebuild not needed after removals from soft dictionary during enumeration"
  "GsFile gciLogServer:'--- rebuild not needed for newSize ' , newSize asString
		, '--------------- ' . ===debugging code not used"
  ^ self
  ].
self tableSize: newSize.  "installs new collision limit"
saveCollLimit := collisionLimit .
collisionLimit := 536870911 . "prevent recursive rebuild"
1 to: saveIdx by: 3 do: [ :j | | aSoftRef |
  aSoftRef := saveTable at: j + 1 "value" .
  aSoftRef _value ifNotNil:[     "avoid incrementing useCount"
    super at: (saveTable at: j "key") put: aSoftRef
    ].
  ].
collisionLimit := saveCollLimit .
"GsFile gciLogServer:'--- rebuild for newSize ' , newSize asString
                , '--------------- ' .   ===debugging code not used"
^self
%
 
