!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   LargeInteger, Integer, Number, Magnitude, Object.
!
!=========================================================================


set class LargeInteger

category: 'For Documentation Installation only'
classmethod:
installDocumentation

self comment:
'Instances of LargeInteger represent Integers not representable as a SmallInteger.
 Each instance of LargeInteger is stored as an array of bytes, where every
 4 bytes represents a base 4294967296 digit.  The first 4 bytes are the sign
 digit (0 or 1) , the next 4 bytes in that
 array constitute the least significant base 4294967296 digit, and the last 4 
 bytes are the most significant base 4294967296 digit.  
 Instance have a maximum size of 4067 digits plus the sign .
 
 For a LargeInteger loaded into memory, the bytes in each digit are in 
 machine native byte order for a C unsigned int.

 Coercion between LargeInteger and SmallInteger occurs automatically.' .
%

category: 'Private'
classmethod: 
_initializeMaximumValues
"Returns the maximum allowable LargeInteger value."
| nd v f df |
nd := MaximumDigits"classVar installed by bom.c" .
v := 1 bitShift: (nd * 32) - 1 .
v := (v - 1) bitOr: v .
v _digitLength = nd  ifFalse:[ self error:'incorrect size'].
(v bitXor: v) = 0 ifFalse:[ self error:'incorrect value'].
f := Float fmax asInteger. "largest Float < PlusInfinity"
df := 9.9999999999999999999f15000 asInteger. "largest DecimalFloat < Infinity"
LargeInteger _addInvariantClassVar: #MaximumValue value: v immediateInvariant .
LargeInteger _addInvariantClassVar: #MaximumFloat value: f asInteger immediateInvariant .
LargeInteger _addInvariantClassVar: #MaximumDecimalFloat value: df asInteger immediateInvariant .

Float _addInvariantClassVar: #MaximumFloat value: f immediateInvariant .
DecimalFloat _addInvariantClassVar: #MaximumFloat value: df immediateInvariant .
^ v
%

category: 'Queries'
classmethod:
maximumValue
"Returns the maximum allowable LargeInteger value."

^ MaximumValue "computed during filein"
%

classmethod:
maximumFloat
 ^ MaximumFloat .
%
classmethod:
maximumDecimalFloat
 ^ MaximumDecimalFloat .
%


category: 'Accessing'
method:
at: anIndex

"Disallowed."

self shouldNotImplement: #at:
%

category: 'Accessing'
method:
at: anIndex put: aNumber

"Disallowed.  You may not change the value of a LargeInteger."

self shouldNotImplement: #at:put:
%


category: 'Testing'
method:
negative

^ (self _digitAt:0) ~~ 0
%

category: 'Testing'
method:
positive

^ (self _digitAt:0) == 0
%

category: 'Accessing'
method:
sign
  self = 0 ifTrue:[ ^ 0 ].
  ^ (self _digitAt: 0) ~~ 0 ifTrue:[ -1 ] ifFalse:[ 1]
% 

category: 'Truncation and Rounding'
method:
truncated

"Returns the receiver, truncated to a SmallInteger if possible."

<primitive: 599>
self _primitiveFailed: #truncated .
self _uncontinuableError
%


category: 'Arithmetic'
method:
abs

"Returns a LargeInteger that is the absolute value of the receiver."

| sign |
sign := self _digitAt:0 .
sign == 0 ifTrue:[ ^ self ].
^ 0 - self 
%

! Gs64 v2.0, delete method  negated  , now inherited from Number

category: 'Bit Manipulation'
method:
highBit

"Returns the index of the high-order bit that is set
 in the binary representation of the receiver.  (If the receiver is a negative
 integer, takes its absolute value first.)  If the receiver is zero, this
 returns nil.  
 As of Gs64 v3.2  the least significant bit has index 1 ,
 in previous versions the least significant bit had index 0 .

 Example:   4 highBit == 3  "

| len |
len := self _digitLength .
^ (self _digitAt: len) highBit + (32 * (len - 1))
%

! following calls to prim 599 added to handle sub-normal large integers
category: 'Converting'
method:
asInteger

"Returns the receiver, truncated to a SmallInteger if possible."

<primitive: 599>
self _primitiveFailed: #asInteger .
self _uncontinuableError
%

category: 'Truncation and Rounding'
method:
ceiling

"Returns the receiver, truncated to a SmallInteger if possible."

<primitive: 599>
self _primitiveFailed: #ceiling .
self _uncontinuableError
%
category: 'Truncation and Rounding'
method:
floor

"Returns the receiver, truncated to a SmallInteger if possible."

<primitive: 599>
self _primitiveFailed: #floor .
self _uncontinuableError
%

category: 'Truncation and Rounding'
method:
rounded

"Returns the receiver, truncated to a SmallInteger if possible."

<primitive: 599>
self _primitiveFailed: #rounded .
self _uncontinuableError
%

category: 'Updating'
method: 
_digitAt: anIndex put: aValue

"Store aValue into the receiver's base 4294967296 digit specified by anIndex .
 indicated by index.  Fail if the value is negative or is larger than 4294967295.
 Fail if the index is not an Integer or is out of bounds.  Returns the value
 that was stored."

<primitive: 257>

anIndex _validateClass: SmallInteger.
aValue _validateClass: SmallInteger.
self _primitiveFailed: #_digitAt:Put: args: { anIndex . aValue } .
self _uncontinuableError
%

! fix 40233
category: 'Storing and Loading'
classmethod: 
loadFrom: passiveObject

"Reads from passiveObject and convert to active. This method creates
 subnormal (LargeIntegers in the SmallInteger range) if necessary."  

| value instance |
value := passiveObject readInt.
instance := value _isSmallInteger
   ifTrue: [self _newWithValue: value]
   ifFalse: [value].
passiveObject hasRead: instance.
^instance
%

! fix 44607
category: 'Instance Creation'
classmethod:
basicNew
  "Returns a instance of LargeInteger with specified size and value = 0"
  ^ self basicNew: 0
%
classmethod:
basicNew: aSize
  "Returns a instance of LargeInteger with specified size and value = 0"

^ self _new: aSize
%

! fix 46790
category: 'Copying'
method:
postCopy
  ^ self immediateInvariant
%

