! ========================================================================
! Copyright (C) by GemTalk Systems 1991-2020.  All Rights Reserved
!
! $Id: multibytestring.gs,v 1.14.6.1 2008-04-16 18:43:00 dhenrich Exp $
!
! ========================================================================

! Gs64 v3.0 MultiByteString always created by bom.c

removeallmethods MultiByteString
removeallclassmethods MultiByteString
set class MultiByteString

category: 'For Documentation Installation only'
classmethod:
installDocumentation

self comment: 
'MultiByteString is an abstract class for representing strings
 for which each Character occupies more than one byte.

 For historical reasons dealing with the now-deprecated constraints, 
 MultiByteString is in the ClassHistory of String.

 Unless noted otherwise, comparisons between MultiByteStrings are 
 case-sensitive.

 Some of these methods determine whether one strings collates before
 another.  In collation, the Characters of the receiver and aString are
 compared Character-by-Character, from left to right, in case-sensitive
 fashion.  If two strings are of different length, and all Characters in the
 shorter string are equal to their counterparts in the longer strings, the
 shorter string collates before the longer.

 These methods require that the argument be a kind of CharacterCollection, not
 necessarily a String.'.
%

! ------------------- Class methods for MultiByteString

category 'Formatting'
classmethod:
charSize

"Returns number of bytes that make up a character for this string class."

^ self subclassResponsibility: #charSize
%

category 'Instance Creation'
classmethod:
new: aSize

^ self subclassResponsibility: #new:
%

! fix 40867
category 'Instance Creation'
classmethod:
new

^ self subclassResponsibility: #new
%

! ------------------- Instance methods for MultiByteString
category 'Formatting'
method:
charSize

"Returns number of bytes that make up a character for this string class."

^ self subclassResponsibility: #charSize
%

category: 'Formatting'
method:
printOn: aStream

"Puts a displayable representation of the receiver on the given stream."

"For classes whose instances can be literals, the result should contain
 formatting information.  For example, the following expression should
 evaluate to true:

 #abc asString = String withAll: '#abc'"

| idx lastIdx sz |

aStream _nextPut: $' .

idx := self indexOf: $' startingAt: 1 .
(idx == 0 or:[ idx > aStream maxSize ]) ifTrue:[
  aStream nextPutAll: self .
  ]
ifFalse:[
  lastIdx := 1.
  [ (idx == 0) or: [aStream isFull]] whileFalse: [
    aStream nextPutAll: ( self copyFrom: lastIdx to: idx) .
    aStream nextPut: $' .
    lastIdx := idx + 1 .
    idx := self indexOf: $' startingAt: lastIdx .
    ].
  sz := self size .
  ((lastIdx <= sz) and: [aStream isFull not]) ifTrue: [
    aStream nextPutAll: (self copyFrom: lastIdx to: sz) .
    ].
  ].
aStream _nextPut: $' 
%

! bugfix 39431: check when to return original string


category: 'Converting'
method:
asByteArray

	^ByteArray fromString: self
%

category: 'Converting'
method:
_asString

"Returns a (single byte) String representation of the receiver if all of the
 Characters in the receiver can be represented as single-byte Characters.
 Returns nil if any of the Characters in the receiver cannot be represented as
 single-byte Characters."

<primitive: 229> 

^ self _primitiveFailed: #_asString .
%

method:
asString

"Returns a (single byte) String representation of the receiver if all of the
 Characters in the receiver can be represented as single-byte Characters.
 Otherwise, returns the receiver."

| result |
(result := self _asString) ifNil:[ ^ self ].
^ result
%

category: 'Converting'
method:
asSymbol

"Returns a canonical Symbol representation of the receiver."

^ Symbol withAll: self .
%

category: 'Converting'
method:
asSymbolKind

"Returns a (single byte) Symbol representation of the receiver."

^ Symbol withAll: self 
%

! Fix 36125
category: 'Accessing'
method:
wordAt: anIndex

"Returns the integer value of the Character at anIndex in the receiver."

^ (self at: anIndex) codePoint
%


category: 'Updating'
method:
wordAt: anIndex put: aValue

"Stores the integer value aValue in the character cell of the receiver
 specified by anIndex. Return aValue."

self at: anIndex put: (Character withValue: aValue) .
^ aValue
%

category: 'Accessing'
method:
at: anIndex

"Returns the Character at anIndex."

^ self subclassResponsibility: #at:
%

category: 'Updating'
method:
at: anIndex put: aChar

") Stores aChar at the specified location and returns aChar."

^ self subclassResponsibility: #at:put:
%

category: 'Case-Sensitive Comparisons'
method:
= aCharCollection

"Returns true if corresponding Characters in the receiver and argument are equal
 and aCharCollection is comparable with the receiver, and aCharCollection is
 not a kind of Symbol.  Returns false otherwise.

 The comparison for 'equal' is case-sensitive .
 Signals an Error if argument is a kind of Unicode7, Unicode16 or Unicode32 .

 Note that 'kind of Symbol' means either an instance of Symbol or instance of
 MultiByteSymbol."

<primitive: 449>
| argInfo |
argInfo := aCharCollection _stringCharSize .
(argInfo bitAnd: 16r8) ~~ 0 ifTrue:[  "arg is Unicode"
  ^ (self compareTo: aCharCollection collator: nil useMinSize: false) == 0
].
(aCharCollection isKindOf: Utf8) ifTrue:[ 
  ^ (self compareTo: aCharCollection collator: nil useMinSize: false) == 0
].
(aCharCollection isKindOf: CharacterCollection) ifTrue:[ ^ aCharCollection = self].
^ false
%

category: 'Accessing'
method:
numArgs

"Returns the number of arguments the receiver would take, if the receiver were
 a message selector."

self isInfix ifTrue: [ ^1 ].
self isKeyword ifTrue: [ ^self identicalOccurrencesOf: $: ].
^0
%

category: 'Accessing'
method:
size

"Returns the number of Characters in the receiver"

^ self subclassResponsibility: #size
%

category: 'Updating'
method:
size: anInteger


"Changes the size of the receiver to anInteger Characters."

^ self subclassResponsibility: #size:
%

category: 'Accessing'
method:
valueAt: anIndex

"Returns the value of the (multi-byte) Character at anIndex."

^ (self at: anIndex) codePoint
%

category: 'Copying'
method:
withLFs

"Returns a copy of the receiver with all back-slashes replaced by line-feeds."

^self copyReplaceAll: '\' with: (self class with: Character lf)
%

category: 'Compatibility'
method:
_at: anIndex equals: aCharCollection ignoreCase: aBoolean

"Returns true if aCharCollection is contained in the receiver, starting at
 anIndex.  Returns false otherwise.  The comparison is case-insensitive."

^ aBoolean ifTrue:[ self at: anIndex equalsNoCase: aCharCollection ]
	  ifFalse:[ self at: anIndex equals: aCharCollection ]
%

category: 'Case-Insensitive Comparisons'
method:
at: anIndex equalsNoCase: aCharCollection

"Returns true if aCharCollection is contained in the receiver, starting at
 anIndex.  Returns false otherwise.  The comparison is case-insensitive."

<primitive: 438>
anIndex _validateClass: SmallInteger.
((anIndex <= 0) or: [(anIndex > self size)])
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex ].
aCharCollection _validateClass: CharacterCollection.
^ self _primitiveFailed: #at:equalsNoCase: args: { anIndex . aCharCollection }
%

category: 'Case-Sensitive Comparisons'
method:
at: anIndex equals: aCharCollection

"Returns true if aCharCollection is contained in the receiver, starting at
 anIndex.  Returns false otherwise.  The comparison is case-sensitive."

<primitive: 668>
anIndex _validateClass: SmallInteger.
((anIndex <= 0) or: [(anIndex > self size)])
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex ].
aCharCollection _validateClass: CharacterCollection.
^ self _primitiveFailed: #at:equals: args: { anIndex . aCharCollection }
%

category: 'Case-Insensitive Comparisons'
method:
< aCharCollection

"Returns true if the receiver collates before the argument.
 Returns false otherwise.

 The comparison is case-insensitive.
 Signals an Error if argument is a kind of Unicode7, Unicode16 or Unicode32 .

 The ASCII letters A and a collate the same.
   'A' < 'a' returns false.
   'A' < 'b' returns true. "

<primitive: 440>
(aCharCollection _stringCharSize bitAnd: 16r7) ~~ 0 ifTrue:[
  ^ aCharCollection > self  
].
aCharCollection _validateClass: CharacterCollection .
^ aCharCollection > self
%

category: 'Case-Insensitive Comparisons'
method:
<= aCharCollection

"Returns true if the receiver collates before the argument or if all of the
 corresponding Characters in the receiver and argument are equal.
 Returns false otherwise.

 The comparison is consistent with that defined for the < method."

^(self > aCharCollection) not
%

category: 'Deprecated'
method:
asciiLessThan: aString

^ self subclassResponsibility: #asciiLessThan:
%

category: 'Case-Insensitive Comparisons'
method:
> aCharCollection

"Returns true if the receiver collates after the argument.  Returns false
 otherwise.

 The comparison is case-insensitive. 
 Signals an Error if argument is a kind of Unicode7, Unicode16 or Unicode32."

<primitive: 442>

aCharCollection _validateClass: CharacterCollection.
^ aCharCollection < self
%

category: 'Case-Insensitive Comparisons'
method:
>= aCharCollection

"Returns true if the receiver collates after the argument or if all of the
 corresponding Characters in the receiver and argument are equal.
 Returns false otherwise.

 The comparison is consistent with that defined for the < method."

^ (self < aCharCollection) not
%

category: 'Deprecated'
method:
lessThan: aString collatingTable: aTable

"Deprecated. Returns true if the receiver collates before the argument.
 The collating sequence is defined by aTable which must be
 a string with the same character size as the receiver."

self deprecated: 'MultiByteString>>lessThan:collatingTable: deprecated in 3.2'.
^self _primLessThan: aString collatingTable: aTable
%

category: 'Deprecated'
method:
_primLessThan: aString collatingTable: aTable

"Deprecated. Returns true if the receiver collates before the argument.
 The collating sequence is defined by aTable which must be
 a string with the same character size as the receiver." 

<primitive: 359 >
aTable _validateSameCharSizeAs: self .
aString _validateClasses:{ String }.
self _primitiveFailed: #lessThan:collatingTable: args: { aString . aTable }
%

category: 'Deprecated'
method:
greaterThan: aString collatingTable: aTable

"Deprecated. Returns true if the receiver collates after the argument.
 See lessThan:collatingTable: for description of aTable."

self deprecated: 'MultiByteString>>greaterThan:collatingTable: deprecated in 3.2'.
^self _primGreaterThan: aString collatingTable: aTable
%

category: 'Deprecated'
method:
_primGreaterThan: aString collatingTable: aTable

"Deprecated. Returns true if the receiver collates after the argument.
 See lessThan:collatingTable: for description of aTable."

<primitive: 433 >
aTable _validateSameCharSizeAs: self .
aString _validateClasses:{ String }.
self _primitiveFailed: #greaterThan:collatingTable: args: { aString . aTable }
%

category: 'Deprecated'
method:
greaterThanOrEqual: aString collatingTable: aTable

"Deprecated. Returns true if the receiver collates after or the same as the argument.
 The collating sequence is defined by aTable."

self deprecated: 'MultiByteString>>greaterThanOrEqual:collatingTable: deprecated in 3.2'.
^ (self lessThan: aString collatingTable: aTable) not
%

category: 'Deprecated'
method:
lessThanOrEqual: aString collatingTable: aTable

"Deprecated. Returns true if the receiver collates before or the same as the argument.
 The collating sequence is defined by aTable."

self deprecated: 'MultiByteString>>lessThanOrEqual:collatingTable: deprecated in 3.2'.
^ (self greaterThan: aString collatingTable: aTable) not
%

category: 'Deprecated'
method:
equals: aString collatingTable: aTable

"Deprecated. Returns true if the receiver collates the same as the argument.
 See lessThan:collatingTable: for description of aTable.
 The collating sequence is defined by aTable."

self deprecated: 'MultiByteString>>equals:collatingTable: deprecated in 3.2'.
^ self _primEquals: aString collatingTable: aTable
%

category: 'Deprecated'
method:
_primEquals: aString collatingTable: aTable

"Deprecated. Returns true if the receiver collates the same as the argument.
 See lessThan:collatingTable: for description of aTable.
 The collating sequence is defined by aTable."

<primitive: 446 >
aTable _validateSameCharSizeAs: self .
aString _validateClasses:{ String }.
self _primitiveFailed: #equals:collatingTable: args: { aString . aTable }
%

category: 'Private'
method:
_validateSameCharSizeAs: aString

"Returns nil if self has same character size as aString,
 otherwise generates an error."

| cSize cls |
self stringCharSize ~~ (cSize := aString stringCharSize) ifTrue:[
  cSize == 1 ifTrue:[ cls := String ].
  cSize == 2 ifTrue:[ cls := DoubleByteString ].
  cSize == 4 ifTrue:[ cls := QuadByteString ].
  self _errorExpectedClass: cls 
].
^ nil
%

! deleted _idxCompareLessThan: v2.0

!deleted _idxCompareGreaterThan: v2.0

category: 'Case-Insensitive Comparisons'
method:
equalsNoCase: aCharCollection

"Returns true if corresponding Characters in the receiver and argument are equal
 and aCharCollection is comparable with the receiver, and aCharCollection is
 not a kind of Symbol.  Returns false otherwise.

 The comparison for equal is case-insensitive.

 Note that 'kind of Symbol' means either an instance of Symbol or instance of
 DoubleByteSymbol."

<primitive: 450>
aCharCollection _stringCharSize ~~ 0 ifTrue:[
  self _primitiveFailed: #equalsNoCase:  args: { aCharCollection } .
  ^ self _uncontinuableError
].
(aCharCollection isKindOf: CharacterCollection) ifTrue: [
  ^ self equalsNoCase: (String withAll: aCharCollection)
] ifFalse:[
  ^ false
].
%

category: 'Case-Insensitive Comparisons'
method:
isEquivalent: aCharCollection
    "Returns true if the receiver is equivalent to aCharCollection.
    The receiver is equivalent to aCharCollection if the receiver
    contains the same Characters as aCharCollection regardless of case
    or internal representation.  For example, if $a is in
    aCharCollection, it is equivalent to any representation of an 'a'
    in the receiver's character set."

<primitive: 450>
aCharCollection _stringCharSize ~~ 0 ifTrue:[
  self _primitiveFailed: #isEquivalent:  args: { aCharCollection } .
  ^ self _uncontinuableError
].
(aCharCollection isKindOf: CharacterCollection) ifTrue: [
  ^ self isEquivalent: (String withAll: aCharCollection)
] ifFalse:[
  ^ false
].
%

category: 'Private'
method:
_findString: subString startingAt: startIndex ignoreCase: aBoolean

"If a receiver contains subString beginning at some point at or after
 startIndex, this returns the index at which subString begins.  If the
 receiver does not contain subString, this returns 0."

<primitive: 30>
startIndex _validateClass: SmallInteger.
(startIndex < 1) 
  ifTrue: [ ^ self _error: #objErrBadOffsetIncomplete args: { startIndex } ].
subString _validateClasses:{ String }.
aBoolean _validateClass: Boolean .
^ self _primitiveFailed: #_findString:startingAt:ignoreCase:
       args: { subString . startIndex . aBoolean }
%

category: 'Private'
method:
_findLastString: subString startingAt: startIndex ignoreCase: aBoolean

"Searches backwards through receiver    starting at startIndex  .
 Returns the index at which subString begins.  If the
 receiver does not contain subString, this returns 0."

<primitive: 451>

startIndex _validateClass: SmallInteger.
(startIndex < 1) | (startIndex > self size)
  ifTrue: [ ^ self _error: #objErrBadOffsetIncomplete args: { startIndex } ].
subString _validateClasses:{ String }. 
aBoolean _validateClass: Boolean .
^ self _primitiveFailed: #_findLastString:startingAt:ignoreCase:
       args: { subString . startIndex . aBoolean }
%

category: 'Case-Sensitive Searching'
method: 
includes: aCharacter

"Returns true if the receiver contains aCharacter, false otherwise.
 The search is case-sensitive."

^(aCharacter isKindOf: AbstractCharacter) and:
	[self includesValue: aCharacter].
%

category: 'Case-Sensitive Searching'
method:
includesValue: aCharacter

"Returns true if the receiver contains aCharacter.  The search is
 case-sensitive."

<primitive: 452>

"the following code is executed if aCharacter is a JISCharacter "

aCharacter _validateClass: AbstractCharacter .
^ self includesValue: aCharacter asCharacter .
%

category: 'Case-Sensitive Searching'
method:
indexOf: aCharacter startingAt: startIndex

"Returns the index of the first occurrence of aCharacter in the receiver,
 not preceding startIndex.  If the receiver does not contain aCharacter,
 returns zero.

 The search is case sensitive."

<primitive: 453>
startIndex _validateClass: Integer .
startIndex < 1 ifTrue:[ ^ self _errorIndexOutOfRange: startIndex].
^ self indexOf: aCharacter asCharacter startingAt: startIndex
%

category: 'Converting'
method:
asUppercase

"Returns a new instance of the receiver's class, with all lower-case Characters
 in the receiver changed to upper-case.  

 Upper-case Characters are defined by the Unicode standard and 
 u_toupper from libicu is called for each character in the receiver.

 If receiver is a DoubleByteSymbol, returns a DoubleByteString.
 If receiver is a QuadByteSymbol, returns a QuadByteString."

<primitive: 455>
self _primitiveFailed: #asUppercase .
self _uncontinuableError
%

method:
asUppercaseOld

"Returns a new instance of the receiver's class, with all lower-case Characters
 in the receiver changed to upper-case.  

 Upper-case Characters are the A to Z, 
 including letters with diacriticals, up to codePoint 255.
 Conversion of codepoints above 255 , requires that
 (deprecated)character data tables be installed.

 If receiver is a DoubleByteSymbol, returns a DoubleByteString.
 If receiver is a QuadByteSymbol, returns a QuadByteString."

<primitive: 983>
self _primitiveFailed: #asUppercaseOld .
self _uncontinuableError
%

category: 'Converting'
method:
asLowercase

"Returns a new instance of the receiver's class, with all upper-case Characters
 in the receiver changed to lower-case.  

 Lower-case Characters are defined by the Unicode standard and
 u_tolower from libicu is called for each character in the receiver.

 If receiver is a DoubleByteSymbol, returns a DoubleByteString.
 If receiver is a QuadByteSymbol, returns a QuadByteString."

<primitive: 821>
self _primitiveFailed: #asLowercase .
self _uncontinuableError
%

category: 'Converting'
method:
asLowercaseOld

"Returns a new instance of the receiver's class, with all upper-case Characters
 in the receiver changed to lower-case.

 Lower-case Characters are the a to z, 
 including letters with diacriticals, up to codePoint 255.
 Conversion of codepoints above 255 , requires that
 (deprecated)character data tables be installed.

 If receiver is a DoubleByteSymbol, returns a DoubleByteString.
 If receiver is a QuadByteSymbol, returns a QuadByteString."

<primitive: 984>
self _primitiveFailed: #asLowercaseOld .
self _uncontinuableError
%


method:
asUnicodeString

"If receiver is an Unicode16 or Unicode32, return the receiver. 
 Otherwise return an instance of Unicode7 , Unicode16 or Unicode32 ,
 using the class with smallest character size needed to
 represent a copy of the receiver. 
 Raises an ArgumentError if receiver contains a codePoint > 16r10FFFF,
 or a codePoint in the range 16rD800 to 16rDFFF "

<primitive: 927>

self _primitiveFailed:#asUnicodeString
%

method:
_asUnicode7

<primitive: 925>
"Return an instance of Unicode7 if receiver can be
represented as an Unicode7 string, else return nil."

self _primitiveFailed:#_asUnicode7
%
method: 
_asUnicode16

"Return an instance of Unicode7 or Unicode16 using the 
minimum bytes per character required to represent the receiver.
Return nil if the receiver is not representable as Unicode7 nor Unicode16."

<primitive: 926>

self _primitiveFailed:#_asUnicode16
%


!  _compileInContext:symbolList: not supported yet

category: 'Execution'
method:
evaluate

"Compiles the receiver as an unbound method and executes it using the current
 default symbol list."

^ (self _compileInContext: nil 
       symbolList: GsSession currentSession symbolList)
  _executeInContext: nil
%

category: 'Execution'
method:
evaluateInContext: anObject symbolList: aSymbolList

"Compiles the receiver as an instance method for the class of anObject, using
 aSymbolList as the symbol list.  Executes the resulting GsMethod using anObject
 as self and returns the result of the execution.  Generates an error if
 compilation errors occur."

^ (self _compileInContext: anObject symbolList: aSymbolList)
    _executeInContext: anObject
%

category: 'Hashing'
method:
hash

"Returns a positive Integer based on a case-sensitive hash of the contents 
 of the receiver."

"The algorithm implemented is described in:

 [Pearson 90]
 Pearson, Peter K., Fast Hashing of Variable-Length Text Strings,
 Communications of the ACM 33, 6, (June 1990), 677-680."

<primitive: 282>
self _primitiveFailed: #hash .
self _uncontinuableError
%

category: 'Storing and Loading'
classmethod:
loadFrom: passiveObj

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| inst |

inst := String new: passiveObj readSize . "yes, make a String !!"
inst loadFrom: passiveObj.
inst _changeClassToMultiByte: self .  "Now change to double/quad byte"
^inst
%

category: 'Storing and Loading'
method:
writeTo: passiveObj

"Converts the receiver to its passive form and writes that information on
 passiveObj."

passiveObj writeClass: self class;
  writeSize: self _basicSize;
    nextPutAllBytes: self; space
%
category: 'Removing'
method:
removeFrom: startIndex to: stopIndex

"Deletes the elements of the receiver from startIndex to stopIndex.  The
 size of the receiver is decreased by stopIndex - startIndex + 1.

 Both startIndex and stopIndex must be positive integers not larger than the
 size of the receiver, with startIndex <= stopIndex."

<primitive: 271>
(stopIndex < startIndex)
ifTrue:
   [ ^ startIndex _error: #rtErrBadCopyFromTo args: { stopIndex }].
((stopIndex > self size) or: [(stopIndex < 1)])
   ifTrue: [ ^ self _errorIndexOutOfRange: stopIndex].
(startIndex < 1)
   ifTrue: [ ^ self _errorIndexOutOfRange: startIndex].
^ self _primitiveFailed: #removeFrom:to: args: { startIndex . stopIndex } 
%

category: 'Indexing Support'
method:
_changingSizeTo: newSize
 
"Notifies any modification tracking objects that the receiver (an indexable
 object) is having its size changed."
 
"the object manager reports size changes in number of bytes; convert this
 to number of characters."

^ super _changingSizeTo: newSize // (self charSize)
%
category: 'Testing'
method:
containsSeparator

"Returns true if the receiver contains a separator Character."

1 to: self size do: [:i |
  (self at: i) isSeparator ifTrue: [^true].
].
^false
%

category: 'New Indexing Comparison - prims'
method:
_idxPrimCompareEqualTo: aCharCollection
  "This comparison operation is used for the indexing subsystem to
 determine an ordering for insertion into indexing objects.

 This method collates letters AaBb..Zz."

  "The comparison should be compatible with the case-insensitive semantics
 of the String method with selector #= .
 Same primitive as String>>lessThan: "

  <primitive: 981>
  aCharCollection isUnicodeString
    ifTrue: [
      "Unicode"
      ArgumentError signal:'Unicode argument disallowed in String comparison' ]
    ifFalse: [
      aCharCollection ifNil:[ ^ false ].
      aCharCollection _validateClass: CharacterCollection.
      "JapaneseString or MixedString"
      ^ self = aCharCollection ]
%

category: 'New Indexing Comparison - prims'
method:
_idxPrimCompareLessThan: aCharCollection

"This comparison operation is used for the indexing subsystem to
 determine an ordering for insertion into indexing objects.

 This method collates letters AaBb..Zz."

"The comparison should be compatible with the case-insensitive semantics
 of the String method with selector #< .
 Same primitive as String>>lessThan: "

<primitive: 441>

aCharCollection _validateClass: CharacterCollection.
^ aCharCollection isUnicodeString
    ifTrue: [ ArgumentError signal:'Unicode argument disallowed in String comparison' ]
    ifFalse: [ 
      "JapaneseString or MixedString"
      self < aCharCollection ]
%

category: 'New Indexing Comparison - prims'
method:
_idxPrimCompareGreaterThan: aCharCollection

"This comparison operation is used for the indexing subsystem to
 determine an ordering for insertion into indexing objects such as B-trees.

 This method collates letters AaBb..Zz."

"The comparison should be compatible with the case-insensitive semantics
 of the String method with selector #> .
 Same primitive as String>>greaterThan: ."

<primitive: 443>
aCharCollection _validateClass: CharacterCollection.
^ aCharCollection isUnicodeString
    ifTrue: [ ArgumentError signal:'Unicode argument disallowed in String comparison' ]
    ifFalse: [
      "JapaneseString or MixedString"
      self > aCharCollection ]
%

category: 'Encoding'
method:
encodeUsing: encodingArray

  "A new instance of the class of the receiver will be returned where 
   each character in the receiver is replaced by the character or string 
   from encodingArray that is located at the position in encodingArray for  
   the character.  The codePoint of the character is used as an index
   into the encodingArray,  value 16r0 accesses   encodingArray at:1 .
   An Error is signaled if any of the following are true
    the codePoint is >= size of the encodingArray

    the value from the encodingArray is a Character with codePoint too large 
    to fit in an instance of the receiver's class.

    the value from the encodingArray is a CharacterCollection with 
    bytes-per-character bigger than bytes-per-character of the receiver
     
    the value from the encodingArray is neither a String, MultiByteString,
    nor Character . "
  <primitive: 654>
  encodingArray _validateInstanceOf: Array  .
  self _primitiveFailed: #encodeUsing: args: { encodingArray }
%

category: 'Copying'
method:
_reverseFrom: aString

"copies elements of aString into receiver in reverse order.
 Size of aString must be same as size of receiver.
 Returns receiver. 
 aString==self is allowed. "

<primitive: 793>
aString stringCharSize == self stringCharSize ifFalse:[
  aString _error: #rtErrBadArgKind args:{ self class }.
].
aString size == self size ifFalse:[
  aString _error: #rtErrBadSize args:{ self size . aString size }.
].
self _primitiveFailed: #_reverseFrom args: { aString }
%

category: 'Copying'
method:
reverse

"Returns a copy of the receiver with its elements in reverse order."
| copy |
copy := self species new: self size .
^ copy _reverseFrom: self
%

category: 'Copying'
method:  
copyFrom: startIndex to: stopIndex

"Returns a new Array containing the elements of the receiver
 between startIndex and stopIndex, inclusive.  The result is of the same class
 as the receiver.

 If startIndex > stopIndex then an empty collection is returned.
 Otherwise both startIndex and stopIndex must be positive integers not larger than the
 size of the receiver, with startIndex <= stopIndex.
"

<primitive: 818>
(startIndex < 1) ifTrue: [ ^ self _errorIndexOutOfRange: startIndex].

((stopIndex > self size) or: [(stopIndex < 0)])
   ifTrue: [ ^ self _errorIndexOutOfRange: stopIndex].

self _primitiveFailed: #copyFrom:to: args: { startIndex . stopIndex } 
%

!
! 39484: next two methods moved up from DoubleByteString
!
category: 'Execution'
method:
_compileInContext: anObject symbolList: aSymbolList 
oldLitVars: litVarArray  environmentId: anEnvironmentId flags: flagsInt

"Compiles the receiver as an instance method for the class of
 anObject, using aSymbolList as the symbol list.
 The receiver is parsed as a doit, i.e. it does not have a selector.

 If litVarArray is not nil, it must be an Array of Symbol,SymbolAssociation pairs
 and this Array will be searched prior to searching aSymbolList to
 resolve literal variables within the method.

 anEnvironmentId must be a SmallInteger >= 0 and <= 255
 specifying a compilation environment.

 flagsInt must be a SmallInteger >= 0 and <= 255 and contains the bits
 specified by GCI_COMPILE_* in $GEMSTONE/include/gcicmn.ht .

 Returns a GsMethod that is not in the method dictionary of any class, or
 else generates an error if there are compilation errors."

<primitive: 150>
aSymbolList _validateClass: SymbolList .
^ self _primitiveFailed: #_compileInContext:symbolList:oldLitVars:environmentId:flags:
       args: { anObject . aSymbolList . litVarArray . anEnvironmentId . flagsInt}
%

category: 'Execution'
method:
_compileInContext: anObject symbolList: aSymbolList

"Compiles the receiver as an instance method for the class of
 anObject, using aSymbolList as the symbol list, and using the
 default compilation environment.
 The receiver is parsed as a doit, i.e. it does not have a selector.

 Returns a GsMethod that is not in the method dictionary of any class, or
 else generates an error if there are compilation errors."

^ self _compileInContext: anObject symbolList: aSymbolList oldLitVars: nil
      environmentId: 0 flags: 0
%
category: 'Deprecated'
method:
asDoubleByteSymbol

"Returns a Symbol, DoubleByteSymbol or QuadByteSymbol
    representation of the receiver.
 Deprecated."

self deprecated:'MultiByteString >> asDoubleByteSymbol is deprecated, use asSymbol' .
^ self asSymbol .
%
category: 'Decompiling without Sources'
method: 
_asSource

^ self quoted 
%

category: 'Encoding'
method: 
javaScriptEncode
  "Returns a String.
   The primitive encapulates all of the encoding logic 
   from Seaside JSStream>>encodeString:on: .
   Assumes Character>>greaseInteger always equivalent to Character>>codePoint
  "
<primitive: 1046>
^ self _primitiveFailed: #javaScriptEncode 
%
category: 'Comparing'
method:
codePointCompareTo: aString

"aString must be a CharacterCollection or ByteArray,
 and must not be a Utf8 or Utf16 .
 Return -1 , 0, or 1  based on how self compares to aString 
 using codePoint comparison of each character. "

<primitive: 844>
^ self _primitiveFailed: #codePointCompareTo: args: { aString }
%

