!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id: metacla.gs 23817 2010-07-14 16:59:48Z otisa $
!
! Superclass Hierarchy:
!   ObsoleteMetaclass, Behavior, Object.
!
!=========================================================================

removeallmethods ObsoleteMetaclass
removeallclassmethods ObsoleteMetaclass
set class ObsoleteMetaclass

category: 'For Documentation Installation only'
classmethod:
installDocumentation

self comment:
'An instance of ObsoleteMetaclass is the meta class on disk
 of a Class created in a prior version (Gemstone 2.x) repository.
 Instances of ObsoleteMetaclass are mutated to instances of Metaclass3
 when they are faulted into the VM.  
 Behavior in ObsoleteMetaclass is present for reference purposes and
 is not expected to be executed.

Constraints:
	superClass: Behavior
	format: SmallInteger
	instVarsInfo: SmallInteger
	instVarNames: Array
	constraints: Object
	classVars: SymbolDictionary
	methDicts: Object
	poolDictionaries: Array
	categorys: Object
	primaryCopy: Object
	thisClass: Class' .
%

category: 'Accessing'
method:
name

"Returns the name of the receiver.  For example, SmallInteger class
 name returns SmallInteger class (the receiver, a Metaclass)."

(thisClass == nil)
  ifTrue:[^'aMetaclass']
  ifFalse:[^thisClass name , ' class']
%

category: 'Accessing'
method:
thisClass

"Returns the Class of which the receiver is a Metaclass.  For example,
 SmallInteger class thisClass returns SmallInteger (the Class)."

^ thisClass
%

category: 'Accessing'
method:
classHistory

"Returns the classHistory for the Class of which the receiver is a Metaclass."

^ thisClass classHistory
%

category: 'Instance Creation'
method:
new

"Disallowed.  To create a new Class or Metaclass, use
 Class | subclass:instVarNames:.. instead."

self shouldNotImplement: #new
%

category: 'Instance Creation'
method:
new: anInteger

"Disallowed.  To create a new Class or Metaclass, use
 Class | subclass:instVarNames:.. instead."

self shouldNotImplement: #new:
%

category: 'Class Instance Variables'
method:
addInstVarNames: instVarNamesArray

"Adds the instance variables specified in the argument to the receiver and any
 of its subclasses.  Generates an error upon encountering a name that is not a
 valid instance variable name or that is already an instance variable of the
 receiver.

 Instance variables that are added to a Metaclass are called
 Class Instance Variables."

| ivnames oldEnd |

(thisClass isInvariant not or: [thisClass subclassesDisallowed]) ifFalse: [
  ^ self _error: #rtErrClassNotModifiable
  ].
instVarNamesArray size == 0 ifTrue: [ ^self ].
"allow an error handler to proceed from the errors signaled by the validation
 method and thus skip offending instance variable names"
ivnames := { } .
instVarNamesArray do:[ :aStr | ivnames add: aStr asSymbol ].
ivnames := ivnames select: [:e |
  self _validateNewClassInstVar: e
  ].
oldEnd := self instSize + 1.
self _inheritCIVs: ivnames at: oldEnd.
%

category: 'Class Instance Variables'
method:
_inheritCIVs: ivnames at: inheritedIVIndex

"Adds the instance variables specified in the argument to the receiver at the
 given location.

 Instance variables that are added to a Metaclass are called
 Class Instance Variables."

| constr names vname subClss numNewIvs |

(numNewIvs := ivnames size) == 0 ifTrue: [ ^self ].
(numNewIvs + self instSize) > GEN_MAX_INSTANCE_VARS ifTrue:[
  self error: 'new instVars would exceed max number of instVars'
].
constr := { Object }.
names := Array withAll: instVarNames.
1 to: numNewIvs do: [:i |
  vname := ( ivnames at: i) asSymbol .
  names insertObject: vname at: (inheritedIVIndex + i - 1).
  self _incrementInstVars: 1  .
  self _unsafeAt: 4 put: (InvariantArray withAll: names).
  self _unsafeAt: 5 put: constraints + constr.
  "force the class to increase in size by storing one position off the end"
  self _refreshClassCache: false .
  thisClass _insertCivAt: (inheritedIVIndex + i - 1).
  ].
thisClass _refreshClassCache: false .
thisClass subclassesDisallowed ifFalse: [
  subClss := self _subclasses.
  subClss ~~ nil ifTrue:[
    subClss do: [:e |
      e _inheritCIVs: ivnames at: inheritedIVIndex.
      ].
    ].
  ].
"finally, recompile my methods to ensure that the instance variable indexes are
 correct"
self _recompileMethodsAfterNewIvOffset: inheritedIVIndex.
%

category: 'Class Instance Variables'
method:
_validateNewClassInstVar: ivname

"Returns true if the name passes all tests.  Generates errors if the name fails
 due to a) being an invalid identifier or b) being a duplicate of an existing
 instance variable, either in this Metaclass or in one of this Metaclass's
 instance's subclasses Metaclasses."

| subs |
((ivname size == 0 
  or: [ivname isValidIdentifier not]) 
  or:[ ivname _isSymbol not]) ifTrue: [
  self _error: #classErrBadIdentifier args: { ivname }.
  ^false
  ].
(instVarNames includesIdentical: ivname) ifTrue: [
  self _error: #rtErrAddDupInstvar args: { ivname }.
  ^false
  ].
thisClass isModifiable ifTrue: [
  subs := self _subclasses .
  subs ~~ nil ifTrue: [
    subs do: [:sub |
      (sub _validateNewClassInstVar: ivname) ifFalse: [ ^false ]
      ].
    ].
  ].
^true
%

category: 'Updating the Method Dictionary'
method:
compileAccessingMethodsFor: anArrayOfSymbols

"Reimplemented to treat class instance variables specially.
 The new methods have environmentId == 0 .  "

| allVarNames varName symlst |

symlst := SymbolList new.
varName := 'newValue'.
allVarNames := self allInstVarNames.
[allVarNames includesValue: varName] whileTrue: [
  varName := 'z' , varName.
].
anArrayOfSymbols do: [ :var | | lf methodtext |
  lf := Character lf .
  (methodtext := String new) add: var ; add: lf ; add: lf ;
    add: '   "Return the value of the instance variable ''' ; add: var ;
    add: '''."' ; add: lf  ; add: '   ^' ; add: var ; add: lf .
  [  self compileMethod: methodtext dictionaries: symlst
       category: #Accessing environmentId: 0
  ] onException: CompileError do:[:ex |
     self _error: #classErrNotAVar args: { var }
  ].
  (allVarNames indexOf: var) > ObsoleteMetaclass instSize ifTrue: [ 
    "compile a method that lets the variable be modified if the user has
     the proper authority"
    (methodtext := String new )
      add: var; add: ': '; add: varName; add: lf;
      add: '  "changes the value of the receiver''s class instance variable ''';
      add: var; add: '''"'; add: lf;
      add: lf;
      add: '  self atClassInstVar: #'; add: var;
      add: ' put: '; add: varName; add: lf .
    [ self compileMethod: methodtext
       dictionaries: symlst
       category: #Updating environmentId: 0 
    ] onException: CompileError do:[:ex |
         self _error: #classErrNotAVar args: { var }
    ].
  ] ifFalse: [
    (methodtext := String new ) add: var ; add: ': ' ; add: varName ; add: lf ; add: lf ;
      add: '   "Modify the value of the instance variable ''' ; add: var ;
      add: '''."' ; add: lf ; add:'   ' ; add: var ; add:' := ' ; add: varName ; add: lf .
    [ self compileMethod: methodtext
      dictionaries: symlst
      category: #Updating environmentId: 0
    ] onException: CompileError do:[:ex |
      self _error: #classErrNotAVar args: { var }
    ].
  ].
].
%

category: 'Displaying'
method:
instanceString

"Returns a string that can be used to name an instance of the receiver.  Since
 the receiver has one instance, returns the name of that instance."

^thisClass name
%

category: 'Displaying'
method:
instanceSymbol

"Returns a symbol that can be used to name an instance of the receiver.  Since
 a Metaclass has only one instance, returns the name of that instance."

^thisClass name
%

category: 'Queries'
method:
isMeta

"Returns whether the receiver is a kind of Metaclass."

^true
%

category: 'Accessing'
method:
extraDict

"Returns the extraDict of the receiver's sole instance.  See Class | extraDict."

^thisClass extraDict
%

method:
transientMethodDictForEnv: envId
  "instances of ObsoleteMetaclass have no transient method dicts"
  ^ nil
%

category: 'Category'
method:
_classCategory 

"Returns the classCategory of the receiver."

^ thisClass _classCategory
%

category: 'Accessing'
method:
_subclasses

"If thisClass is tracking subclasses, returns the corresponding list
 of Metaclasses."

| subclss result |

subclss := thisClass _subclasses .
subclss == nil ifTrue:[ ^ nil ].
result := IdentitySet new .
subclss do:[ :aClass | result add: aClass class ].
^ result
%

category: 'Private'
method:
_gbsTraversalCallback

"Private.  When GemBuilder Smalltalk traverses a Metaclass, this method
 is called to return a description of the Metaclass."

^self printString
%

category: 'Modifying Classes'
method:
_setClassVars: aDict old: previousDict
  classVars ~~ aDict ifTrue:[ 
    previousDict ~~ classVars ifTrue:[ self error:'invalid store to classVars'].
    classVars := aDict .
    thisClass _setClassVars: aDict old: previousDict
  ].
%

!------------- session methods support for Metaclass ------------------
category: 'Method Timestamps'
method:
methodStampDictName
  ^ #GSMetaMethodStampDict
%

category: 'Pragmas'
method:
pragmaDictName

  ^ #GSMetaMethodPragmaDict
%

category: 'Class Timestamps'
method:
theNonMetaClass
  "Sent to a class or metaclass, always return the class.
   Used by Monticello."

  ^ thisClass
%

