!=========================================================================
! Copyright (C) GemTalk Systems 1991-2020.  All Rights Reserved
!
! $Id$
!
!=========================================================================

! Create the class PrintStream, if it does not already exist

expectvalue %String
run
^  Stream _newKernelSubclass: 'PrintStream'
    instVarNames: #( 'itsCollection' 'maxSize' 'approxPosition' )
    classVars: #()
    classInstVars: #()
    poolDictionaries: { }
    inDictionary: Globals
    options: #() 
    reservedOop: 845 .  
%

!========================================================================
! remove all methods

removeallmethods PrintStream
removeallclassmethods PrintStream

category: 'For Documentation Installation only'
classmethod: PrintStream
installDocumentation

self comment:
'PrintStream is designed for use in place of WriteStream to handle printing
of mixed string types. The default size of the stream is limited
to 100000 bytes.

Constraints:
	itsCollection: SequenceableCollection
	maxSize: Object
	approxPosition: Object'.
%

category: 'Adding'
method: PrintStream
_nextPut: anObject

"Add anObject to the stream, regardless of maxSize."

approxPosition := approxPosition + 1.
itsCollection addLast: anObject.
^ anObject
%

category: 'Adding'
method: PrintStream
nextPut: anObject

"If there's room, add anObject to the stream.
 If adding anObject would exceed maxSize, trigger overflow behavior.
 
 Returns anObject.
"

(self isFull) ifFalse: [
  approxPosition := approxPosition + 1.
  (approxPosition < maxSize) 
  ifTrue: [  itsCollection addLast: anObject ]
  ifFalse: [ self overflow ]].
^ anObject
%

category: 'Adding'
method: PrintStream
nextPutAll: aCollection

"If there's room, add the elements of aCollection to the stream.
 If adding aCollection would exceed maxSize, add part of aCollection
 to stream up to maxSize and then trigger overflow behavior.  
 Use #_basicSize to estimate size of aCollection, since #size can be 
 expensive for certain classes.

 Returns aCollection.  "
| ap |
ap := approxPosition + aCollection size .
ap < maxSize ifTrue:[
  itsCollection addAll: aCollection  .
  approxPosition := ap .
] ifFalse:[ 
  self isFull ifFalse:[
     itsCollection replaceFrom: itsCollection size + 1 to: maxSize 
			with: aCollection startingAt: 1 .
     approxPosition := maxSize .
     self overflow 
  ]
].
^ aCollection
%

category: 'Adding'
method: PrintStream
print: anObject

	anObject printOn: self.
%

category: 'Positioning'
method: PrintStream
_initStreamWith: aCollection maxSize: aMaxSize

"Initializes the receiver's 'itsCollection' instance variable to be
 aCollection."

itsCollection := aCollection.
maxSize := aMaxSize .
approxPosition := 0 .
%

category: 'Instance Creation'
classmethod: PrintStream
on: aCollection

"Returns an instance of the receiver that can stream over the elements of
 aCollection.  The maximum size of the receiver is set to 536870911."

| newStream |

newStream := self _basicNew.
newStream _initStreamWith: aCollection maxSize: 536870911 .
^ newStream
%

category: 'Instance Creation'
classmethod: PrintStream
printingOn: aCollection

"Returns an instance of the receiver that can stream over the elements of
 aCollection. The maximum size of the receiver is set to 100000."

^ self printingOn: aCollection maxSize: 100000
%

category: 'Instance Creation'
classmethod: PrintStream
printingOn: aCollection maxSize: n


| newStream |

newStream := self _basicNew.
newStream _initStreamWith: aCollection maxSize: n .
^ newStream
%

category: 'Accessing'
method: PrintStream
contents

"Returns the Collection associated with the receiver (that is,
 the sequence of objects that the receiver may access)."

^ itsCollection
%

category: 'Accessing'
method: PrintStream
_collection

"Returns the collection of the receiver."

^itsCollection
%

category: 'Accessing'
method: PrintStream
maxSize

"Returns the maxSize of the receiver."

^ maxSize
%

category: 'Testing'
method: PrintStream
isEmpty

"Returns true if the collection that the receiver accesses contains
 no elements; otherwise returns false."

^ itsCollection size == 0
%

category: 'Testing'
method: PrintStream
isFull

"Returns true if the size of the collection that the receiver accesses
 is equal or greater than maxSize."

^ approxPosition >= maxSize
%

category: 'Converting'
method: PrintStream
_convertToISOLatin

"Converts the receiver's collection from String to ISOLatin.
 Has no effect if the receiver's collection is already an ISOLatin, or
 if the receiver's collection is something other than a String, such as
 a GsFile or a JISString."

| oldStr |

itsCollection class == String ifTrue:[
  oldStr := itsCollection .
  itsCollection := ISOLatin withAll: oldStr  .
  ]
%


category: 'Positioning'
method: PrintStream
position

"Returns the receiver's approximate position reference for accessing the 
 sequence of objects. "

^ itsCollection _basicSize
%

category: 'Positioning'
method: PrintStream
overflow

"Do whatever is appropriate when we try to exceed maxSize.
 For a CharacterCollection, add ellipses.  
 For anything else, trigger #rtErrBadStreamPosition.
"

(itsCollection isKindOf: CharacterCollection)
  ifTrue: [ itsCollection addAll: '. . .' ]
  ifFalse: [self _error: #rtErrBadStreamPosition args: { approxPosition }]
%
