!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id: readstrPortable.gs 25521 2011-03-22 20:53:42Z jfoster $
!
! Based on Pharo-1.2.1 implementation
!
! Superclass Hierarchy:
!   ReadStream, PositionableStream, Stream, Object.
!
! See installStreamHierarchy.topaz for filein information
!
!=========================================================================

! class created by installStreamHierarchy.topaz

removeallmethods ReadStream
removeallclassmethods ReadStream

doit
ReadStream comment: 'ReadStream / ReadStreamPortable is an ANSI compliant implementation of 
 ReadStream, a PositionableStream that allows its objects to be read but 
 not written.'.
true
%

category: 'Instance Creation'
classmethod: ReadStream
on: aCollection from: firstIndex to: lastIndex 
"Answer with a new instance streaming over a copy of aCollection from
firstIndex to lastIndex."

^self basicNew
	on: aCollection
	from: firstIndex
	to: lastIndex
%

category: 'Instance Creation'
classmethod: ReadStream
new

"Disallowed.  To create a new ReadStream, use the class method on: instead."

self shouldNotImplement: #new
%

category: 'Accessing'
method: ReadStream
next
 
"Returns the next object that the receiver can access for reading. 
 Returns nil if an attempt is made to read beyond the end of the stream."
| pos |
(pos := position) >= readLimit
       ifTrue: [^nil]
       ifFalse: [^collection at: (position := pos + 1)]
%

category: 'Accessing'
method: ReadStream
nextOrNil

"Returns the next object that the receiver can access for reading.  
 Returns nil if an attempt is made to read beyond the end of the stream."
| pos |
(pos := position) >= readLimit
	ifTrue: [^nil]
	ifFalse: [^collection at: (position := pos + 1)]
%

category: 'Accessing'
method: ReadStream
next: n into: aCollection startingAt: startIndex
"Read n objects into the given collection. 
Return aCollection or a partial copy if less than
n elements have been read."
| max pos |
max := (readLimit - (pos := position)) min: n.
aCollection 
	replaceFrom: startIndex 
	to: startIndex+max-1
	with: collection
	startingAt: pos +1.
position := pos + max.
max = n
	ifTrue:[^aCollection]
	ifFalse:[^aCollection copyFrom: 1 to: startIndex+max-1]
%

category: 'Adding'
method: ReadStream
nextPut: anObject

"Disallowed.  You cannot write to a ReadStream."

self shouldNotImplement: #nextPut:
%

category: 'Accessing'
method: ReadStream
size
"Compatibility with other streams"
^readLimit
%

category: 'Accessing'
method: ReadStream
next: amount 
"Answer the next amount elements of my collection.  overriden for efficiency"

| ans endPosition pos |
amount < 0 ifTrue: [ ^self error: 'amount may not be less than 0' ].

endPosition := (pos := position) + amount  min:  readLimit.
ans := collection copyFrom: pos+1 to: endPosition.
position := endPosition.
^ans
%

category: 'Accessing'
method: ReadStream
upTo: anObject
"fast version using indexOf:"
| start end |

start := position+1.
end := collection indexOf: anObject startingAt: start ifAbsent: [ 0 ].

"not present--return rest of the collection"	
(end = 0 or: [end > readLimit]) ifTrue: [ ^self upToEnd ].

"skip to the end and return the data passed over"
position := end.
^collection copyFrom: start to: (end-1)
%

category: 'Accessing'
method: ReadStream
upToEnd
| start pos |
start := 1 + position .
position := (pos := readLimit) .
^collection copyFrom: start to: pos
%

category: 'Private'
method: ReadStream
on: aCollection from: firstIndex to: lastIndex

| len |
collection := aCollection .
readLimit :=  lastIndex > (len := aCollection size)
	ifTrue: [len]
	ifFalse: [lastIndex].
position := firstIndex <= 1
	ifTrue: [0]
	ifFalse: [firstIndex - 1]
%

category: 'Testing'
method: ReadStream
nextMatchFor: anObject

	"The first object is removed from the receiver's future sequence value and appended to the end of
	the receiver's past sequence values. The value that would result from sending #= to the object with
	anObject as the argument is returned.
	The results are undefined if there are no future sequence values in the receiver.

	ANSI 5.9.2.6"

	^self next = anObject.
%

category: 'Accessing'
method: ReadStream
skip: anInteger 
"Set the receiver's position to be the current position+anInteger. Do not
 throw error if skipAmount would exceed collection bounds - ANSI compliance. "

self position: ((position + anInteger max: 0) min: readLimit)
%

category: 'Passivation - Support'
set compile_env: 0
method: ReadStream
_fastNext
  "Returns the next item on the input stream without end of stream checks.
 This may result in a bad access error instead of an end of stream error."

  ^ collection at: (position := position + 1)
%
category: 'Passivation - Support'
set compile_env: 0
method: ReadStream
nextElements: count into: aCollection
  "Stores the next count elements that the receiver can access for reading
 into aCollection.  The receiver's collection and aCollection must be compatible
 SequenceableCollections.  Returns the count of elements read.

 Generates an error if an attempt is made to read beyond the end of the stream."

  "Used by PassiveObject."

  ^ self next: count into: aCollection startingAt: 1
%
category: 'Passivation - Support'
set compile_env: 0
method: ReadStream
_fastPosition
  "Returns the receiver's current position reference for accessing the sequence
 of objects.  The position is actually the next element of the collection to be
 read or written; the position is incremented by each read or write.  In
 general, to start reading or writing at the nth element of a collection, the
 position must be set to n."

  "See bug 42596: Compatible with Legacy Stream positioning"

  ^ position + 1
%
category: 'Passivation - Support'
set compile_env: 0
method: ReadStream
_fastPosition: pos
  "Sets the position of the receiver without checking the validity of the
 argument 'pos'.  This may result in bad index errors instead of end
 of stream errors in subsequent access of the stream."

  "See bug 42596: Compatible with Legacy Stream positioning"

  position := pos - 1
%

category: 'Accessing'
method:
peekN: anInteger
  "Peek returning the next N characters of collection."
  | end start |
  position >= readLimit ifTrue:[ ^ '' ].
  start := position + 1 .
  end := start + anInteger - 1.
  end >= readLimit ifTrue:[ end := readLimit - 1 ].
  ^ collection copyFrom: start to: end
%
  
