!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id: scaleddecimal2.gs 24839 2010-12-10 20:39:17Z otisa $
!
! Superclass Hierarchy:
!   ScaledDecimal, Number, Magnitude, Object.
!
!=========================================================================


set class ScaledDecimal

category: 'For Documentation Installation only'
classmethod: 
installDocumentation

self comment:
'ScaledDecimal stores numerical values as a mantissa and scale.
 The scale defines the position of the decimal point.  
 The mantissa is an Integer .
 The scale is a SmallInteger >= 0 and <= 30000 .
 The represented value is  
     mantissa / (10 raisedTo: scale)  .
 Arithmetic operations between ScaledDecimals result in a ScaledDecimal
 with a scale equal to either the receiver or the argument, whichever has the
 greater scale. If the answer cannot be exactly represented within the result
 scale, the result is rounded to the ScaledDecimal of that scale that is closest
 to the precise result. If the precise result is exactly halfway between two 
 adjacent representable values, the value with an even mantissa is answered.

Numbers with a very large number of decimal digits cannot be represented as 
a ScaledDecimal. The total digits (scale plus the number of digits to the left of 
the decimal point) may not be more than can be represented by a 
LargeInteger. This limit is set by the current implementation to be slightly over 
39177 decimal digits. Operations on ScaledDecimals that attempt to create 
numbers with more than this many digits will fail with an error.  This includes 
intermediate calculations as well as results. Multiplication of ScaledDecimals, 
for instance, internally computes an intermediate result with a number of 
digits roughly the sum of the receiver''s digits and the argument''s digits.

Constraints:
	mantissa: Integer
	scale: SmallInteger
' .
%
! ------------------- Class methods for ScaledDecimal
category: 'Instance Creation'
set compile_env: 0
classmethod: ScaledDecimal
for: aNumber scale: s
	"Returns an instance of the receiver, having the specified scale 
 and value as close to aNumber as that scale allows."

	| m |
	s _isSmallInteger ifFalse: [s _validateClass: SmallInteger].
	s > MaxScale
		ifTrue: 
			[(OutOfRange new)
				name: '' max: MaxScale actual: s;
				details: 'invalid scale';
				signal].
	m := (aNumber * (10 raisedToInteger: s)) roundedHalfToEven.
	^(self _basicNew _mantissa: m scale: s) immediateInvariant
%
category: 'Instance Creation'
set compile_env: 0
classmethod: ScaledDecimal
_fromString: aString decimalPoint: dp

"Given aString such as '34.23', returns an instance of the receiver with
 appropriate numerator and denominator, and with scale equal to the number 
 of digits to the right of the decimal point.  Characters in aString after 
 the first character which is neither a digit or decimal point are ignored.
 Signals an OutOfRange if the scale of the result would exceed 30000.  
 The specified specified decimal point character is used ,
 if dp == nil, the session's locale state is used."

<primitive: 465>
aString _validateClasses:{ String }.
dp ifNotNil:[ dp _validateClass: Character ].
self _errIncorrectFormat: aString .
self _primitiveFailed: #_fromString:decimalPoint: args: { aString . dp } .
%

classmethod: ScaledDecimal
fromString: aString 

"Given aString such as '34.23', returns an instance of the receiver with
 appropriate numerator and denominator, and with scale equal to the number 
 of digits to the right of the decimal point.  Characters in aString after 
 the first character which is neither a digit or decimal point are ignored.
 Signals an OutOfRange if the scale of the result would exceed 30000.  
 The session's locale state is used for the expected decimal point."

^ self _fromString: aString decimalPoint: nil
%

classmethod:
fromStringLocaleC: aString
"Given aString such as '34.23', returns an instance of the receiver with
 appropriate numerator and denominator, and with scale equal to the number
 of digits to the right of the decimal point.  Characters in aString after
 the first character which is neither a digit or decimal point are ignored.
 Signals an OutOfRange if the scale of the result would exceed 30000.
 The expected decimal point character is $.  "

^ self _fromString: aString decimalPoint: $. 
%

category: 'Storing and Loading'
set compile_env: 0
classmethod: ScaledDecimal
loadFrom: passiveObj

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| inst m sc |

passiveObj readNamedIV.
m := passiveObj ivValue.
passiveObj readNamedIV.
sc := passiveObj ivValue.
  
passiveObj skipNamedInstVars.

inst := self _mantissa: m scale: sc .
passiveObj hasRead: inst .
^ inst
%
category: 'Private'
set compile_env: 0
classmethod: ScaledDecimal
new
  
self shouldNotImplement: #new .
^ nil
%
category: 'Compatibility'
set compile_env: 0
classmethod: ScaledDecimal
numerator: numer denominator: denom scale: scale

"Returns an instance of ScaledDecimal.
 Signals an OutOfRange if the argument scale > 30000.  "
 | n d m | 
 denom = 0 ifTrue: [ ^ numer   _errorDivideByZero ].
 scale _isSmallInteger ifFalse: [ scale _validateClass: SmallInteger ].
(scale < 0 or: [scale > MaxScale])
	ifTrue: 
		[(OutOfRange new)
			name: 'scale' min: 0 max: MaxScale actual: scale;
      details: 'invalid scale';
			signal].
 n := numer .
 d := denom .
 d < 0 ifTrue:[  d := 0 - d . n := 0 - n ] . 
 m := ( (n * (10 raisedToInteger: scale )) / d ) roundedHalfToEven .

 ^ (self _basicNew _mantissa: m scale: scale) immediateInvariant
%
category: 'Private'
set compile_env: 0
classmethod: ScaledDecimal
_mantissa: m scale: s
 ^ (self _basicNew _mantissa: m scale: s) immediateInvariant
%
! ------------------- Instance methods for ScaledDecimal
category: 'Arithmetic'
set compile_env: 0
method: ScaledDecimal
* aNumber
  "Returns the result of multiplying the receiver by aNumber.
   If aNumber is an Integer or a ScaledDecimal, the result will be 
   an instance of the receiver's class with a scale equal to 
   the greater of the receiver's scale and aNumber's scale.
   The scale of an Integer is considered to be zero."

| osc om |
aNumber _isScaledDecimal
	ifTrue: 
		[om := aNumber mantissa.
		osc := aNumber scale]
	ifFalse: 
		[aNumber _isInteger
			ifTrue: 
				[om := aNumber.
				osc := 0]].
om
	ifNotNil: 
		[| m pten resultScale excessScale |
		m := mantissa * om.
		resultScale := scale max: osc.
		excessScale := scale + osc - resultScale.
		excessScale > 0
			ifTrue: 
				["reduce mantissa to match result scale"
				pten := 10 raisedTo: excessScale.
				m := (m / pten) roundedHalfToEven].
		^self class _mantissa: m scale: resultScale].
^self _retry: #* coercing: aNumber
%
category: 'Arithmetic'
set compile_env: 0
method: ScaledDecimal
+ aNumber
  "Returns the sum of the receiver and aNumber.  
   If aNumber is an Integer or a ScaledDecimal, the result will be 
   an instance of the receiver's class with a scale equal to 
   the greater of the receiver's scale and aNumber's scale.
   The scale of an Integer is considered to be zero."

| m sc |
aNumber _isScaledDecimal
	ifTrue: 
		[| osc pten |
		osc := aNumber scale.
		scale == osc
			ifTrue: 
				[sc := scale.
				m := mantissa + aNumber mantissa]
			ifFalse: 
				[sc := scale max: osc.
				sc > scale
					ifTrue: 
						[pten := 10 raisedToInteger: sc - scale.
						m := mantissa * pten + aNumber mantissa]
					ifFalse: 
						[pten := 10 raisedToInteger: sc - osc.
						m := mantissa + (pten * aNumber mantissa)]].
		^self class _mantissa: m scale: sc].
aNumber _isInteger
	ifTrue: 
		[m := mantissa + (aNumber * (10 raisedToInteger: scale)).
		^self class _mantissa: m scale: scale].
^self _retry: #+ coercing: aNumber
%
category: 'Arithmetic'
set compile_env: 0
method: ScaledDecimal
- aNumber
  "Returns the difference between the receiver and aNumber.
   If aNumber is an Integer or a ScaledDecimal, the result will be 
   an instance of the receiver's class with a scale equal to 
   the greater of the receiver's scale and aNumber's scale.
   The scale of an Integer is considered to be zero."

aNumber _isScaledDecimal
	ifTrue: 
		[^self
			+ (ScaledDecimal _mantissa: 0 - aNumber mantissa scale: aNumber scale)].
^self + (0 - aNumber)
%
category: 'Arithmetic'
set compile_env: 0
method: ScaledDecimal
/ aNumber
  "Returns the result of dividing the receiver by aNumber.
   If aNumber is an Integer or a ScaledDecimal, the result will be 
   an instance of the receiver's class with a scale equal to 
   the greater of the receiver's scale and aNumber's scale.
   The scale of an Integer is considered to be zero."

| divisorScale om |
aNumber _isScaledDecimal
	ifTrue: 
		[om := aNumber mantissa.
		divisorScale := aNumber scale]
	ifFalse: 
		[aNumber _isInteger
			ifTrue: 
				[om := aNumber.
				divisorScale := 0]].
om
	ifNotNil: 
		[| resultScale adjustFactor m |
		om == 0 ifTrue: [^self _errorDivideByZero].
		resultScale := scale max: divisorScale.
		adjustFactor := 10 raisedToInteger: divisorScale - scale + resultScale.
		m := (mantissa / om * adjustFactor) roundedHalfToEven.
		^self class _mantissa: m scale: resultScale].
^self _retry: #/ coercing: aNumber
%
category: 'Arithmetic'
method: ScaledDecimal
// aNumber
  "Returns an integer that is the result of dividing the receiver by aNumber, 
   then rounding towards negative infinity."

| divisorScale om |
aNumber _isScaledDecimal
	ifTrue: 
		[om := aNumber mantissa.
		divisorScale := aNumber scale]
	ifFalse: 
		[aNumber _isInteger
			ifTrue: 
				[om := aNumber.
				divisorScale := 0]].
om
	ifNotNil: 
		[| exactResult |
		om == 0 ifTrue: [^self _errorDivideByZero].
		exactResult := mantissa / om * (10 raisedToInteger: divisorScale - scale).
		^ exactResult floor].
^self _retry: #// coercing: aNumber
%
category: 'Comparing'
set compile_env: 0
method: ScaledDecimal
< aNumber
| osc om m |
(m := mantissa) ifNil:[ ^ false "self is a NaN"].
aNumber _isScaledDecimal ifTrue:[
  osc := aNumber scale .  om := aNumber mantissa .
  om ifNil:[ ^ false ].
] ifFalse:[
  aNumber _isInteger ifTrue:[ osc := 0 .   om := aNumber ].
].
om ifNotNil:[ | sc |
  sc := scale .
  sc == osc ifFalse:[
    sc > osc ifTrue:[ om := om * (10 raisedTo: sc - osc ) ]
            ifFalse:[ m := m * (10 raisedTo: osc - sc ) ].
  ].
  ^ m < om
].
^ self _retry: #< coercing: aNumber
%
category: 'Comparing'
set compile_env: 0
method: ScaledDecimal
<= aNumber
| osc om m |
(m := mantissa) ifNil:[ ^ false "self is a NaN"].
aNumber _isScaledDecimal ifTrue:[
  osc := aNumber scale .  om := aNumber mantissa.
  om ifNil:[ ^ false ].
] ifFalse:[
  aNumber _isInteger ifTrue:[ osc := 0 .   om := aNumber ].
].
om ifNotNil:[ | sc |
  sc := scale .
  sc == osc ifFalse:[
    sc > osc ifTrue:[ om := om * (10 raisedTo: sc - osc ) ]
            ifFalse:[ m := m * (10 raisedTo: osc - sc ) ].
  ].
  ^ m <= om
].
^ self _retry: #<= coercing: aNumber
%
category: 'Comparing'
set compile_env: 0
method: ScaledDecimal
= aNumber
| osc om m |
aNumber _isNumber ifFalse:[ ^ false ].
(m := mantissa) ifNil:[ ^ false "self is a NaN"].
self == aNumber ifTrue:[ ^ true ].
aNumber _isScaledDecimal ifTrue:[
  osc := aNumber scale .  om := aNumber mantissa.
  om ifNil:[ ^ false ].
] ifFalse:[
  aNumber _isInteger ifTrue:[ osc := 0 .   om := aNumber ].
].
om ifNotNil:[ | sc |
  sc := scale .
  sc == osc ifFalse:[
    sc > osc ifTrue:[ om := om * (10 raisedTo: sc - osc ) ]
            ifFalse:[ m := m * (10 raisedTo: osc - sc ) ].
  ].
  ^ m = om
].
^ self _retry: #= coercing: aNumber
%
category: 'Comparing'
set compile_env: 0
method: ScaledDecimal
> aNumber
| osc om m |
(m := mantissa) ifNil:[ ^ false "self is a NaN"].
aNumber _isScaledDecimal ifTrue:[
  osc := aNumber scale .  om := aNumber mantissa.
  om ifNil:[ ^ false ].
] ifFalse:[
  aNumber _isInteger ifTrue:[ osc := 0 .   om := aNumber ].
].
om ifNotNil:[ | sc |
  sc := scale .
  sc == osc ifFalse:[
    sc > osc ifTrue:[ om := om * (10 raisedTo: sc - osc ) ]
            ifFalse:[ m := m * (10 raisedTo: osc - sc ) ].
  ].
  ^ m > om
].
^ self _retry: #> coercing: aNumber
%
category: 'Comparing'
set compile_env: 0
method: ScaledDecimal
>= aNumber
| osc om m |
(m := mantissa) ifNil:[ ^ false "self is a NaN"].
aNumber _isScaledDecimal ifTrue:[
  osc := aNumber scale .  om := aNumber mantissa.
  om ifNil:[ ^ false ].
] ifFalse:[
  aNumber _isInteger ifTrue:[ osc := 0 .   om := aNumber ].
].
om ifNotNil:[ | sc |
  sc := scale .
  sc == osc ifFalse:[
    sc > osc ifTrue:[ om := om * (10 raisedTo: sc - osc ) ]
            ifFalse:[ m := m * (10 raisedTo: osc - sc ) ].
  ].
  ^ m >= om
].
^ self _retry: #>= coercing: aNumber
%
category: 'Converting'
set compile_env: 0
method: ScaledDecimal
asDecimalFloat
	"Answer an instance of DecimalFloat that has a numeric value as close 
	as possible to the value of the receiver, rounding if necessary"
	"DecimalFloats have a 20-decimal-digit precision, so the basic approach is to round the mantissa to 20 digits and then scale. In cases where the result should be a subnormal float, however, the 20-digit precision includes the leading zeroes, so we must round the mantissa to fewer than 20 digits. "

	| mantissaDigits roundedMantissaDigits excessMantissaDigits roundedMantissa adjustmentPower |
	mantissaDigits := mantissa _decimalDigitsLength: false.
	roundedMantissaDigits := mantissaDigits - scale + 15019 min: 20.
	excessMantissaDigits := mantissaDigits - roundedMantissaDigits max: 0.
	roundedMantissa := (mantissa / (10 raisedToInteger: excessMantissaDigits)) roundedHalfToEven.
	adjustmentPower := excessMantissaDigits - scale.
	"DecimalFloats with very small magnitude must be computed a bit more indirectly to avoid underflow to 0."
	^adjustmentPower < -15000
		ifFalse: [roundedMantissa asDecimalFloat * (10.0f0 raisedToInteger: adjustmentPower)]
		ifTrue: [roundedMantissa asDecimalFloat / 1.0f20 * (10.0f0 raisedToInteger: adjustmentPower + 20)]
%
category: 'Converting'
set compile_env: 0
method: ScaledDecimal
asFloat
  "Returns an instance of SmallDouble or Float that has the value of the receiver."
| f p |
f := mantissa asFloat .
f _getKind == 3 ifTrue:[  "an infinity" | nd m pten |
  nd := (m := mantissa) _decimalDigitsLength: true .
  p := nd - 307 .
  pten := 10 raisedTo: p .
  f := (m + (pten bitShift: -1) // pten ) asFloat .
  p := scale - p .
] ifFalse:[
  p := scale .
].
p > 308 ifTrue:[
  f := f / (10.0 raisedTo: p - 308)  .
  p := p - (p - 308) .
].
^ f  / (10.0 raisedTo: p)
%
category: 'Converting'
set compile_env: 0
method: ScaledDecimal
asFraction

"Returns a Fraction that represents the receiver."

^ Fraction numerator: mantissa denominator: (10 raisedTo: scale)
%
category: 'Converting'
set compile_env: 0
method: ScaledDecimal
asScaledDecimal: newScale
"Returns a new instance with the new scale, and a numeric value 
 as close to that of the receiver as possible."

| pten m |
scale == newScale ifTrue: [^self].
newScale _isSmallInteger ifFalse: [newScale _validateClass: SmallInteger].
(newScale < 0 or: [newScale > MaxScale])
	ifTrue: 
		[(OutOfRange new)
			name: 'scale' min: 0 max: MaxScale actual: newScale;
      details: 'invalid scale';
			signal].
pten := 10 raisedToInteger: (scale - newScale) abs.
scale < newScale
	ifTrue: [m := mantissa * pten]
	ifFalse: [m := (mantissa / pten) roundedHalfToEven].
^self class _mantissa: m scale: newScale
%

category: 'Private'
method: ScaledDecimal
_asString: dpChar

| str mstr sc sz len isNeg m nDigits |
(m := mantissa) ifNil:[ ^ 'ScaledDecimalNaN' ].
mstr := m asString .
len := mstr size .
sz := len .
sc := scale.
nDigits := sz .
(isNeg := (mstr at: 1) == $-) ifTrue:[ nDigits := nDigits - 1 ].
str := String new .
nDigits <= sc  ifTrue:[ | zeros zerosLen zcount idx srcIdx |
  isNeg ifTrue:[ str add: $- ] .
  str add: $0 ; add: dpChar .
  zeros := '00000000000000000000000000000000000000000000000000000000000000000' .
  zerosLen := zeros size .
  zcount := sc - nDigits .
  [ zcount >= zerosLen ] whileTrue:[
    str add: zeros .
    zcount := zcount - zerosLen .
  ].
  zcount > 0 ifTrue:[ 
    "zeros copyFrom: 1 to: zcount into: str startingAt: str size + 1"
    idx := str size .
    str replaceFrom: idx + 1 to: idx + zcount with: zeros startingAt: 1 .
  ].
  "mstr copyFrom: len - nDigits + 1 to: len into: str startingAt: str size + 1"
  idx := str size + 1 .
  srcIdx := len - nDigits + 1 .
  str replaceFrom: idx  to: idx + nDigits - 1  with: mstr startingAt: srcIdx 
] ifFalse:[  
  str := mstr .
  str insertAll: dpChar at: (sz + 1 - sc ) .
].
^ str 
%

category: 'Formatting'
method: ScaledDecimal
asString
  "Returns a String of the form '123.56 for a number with scale = 2  ,
   where the decimal point character in the result is per the current Locale."

  ^ self _asString: Locale decimalPoint
% 

category: 'Formatting'
method: ScaledDecimal
asStringLocaleC

  "Returns a String of the form '123.56 for a number with scale = 2.  
   Does not use Locale , decimal point character is always $.  " 

  ^ self _asString: $. 
% 



category: 'Truncation and Rounding'
set compile_env: 0
method: ScaledDecimal
ceiling

"Returns the integer that is closest to the receiver, on the same side
 of the receiver as positive infinity."

 | f |
 f := self floor.
^ self = f
	ifTrue: [f]
	ifFalse: [f + 1]
%
category: 'Compatibility'
set compile_env: 0
method: ScaledDecimal
denominator
  ^ self asFraction denominator
%
category: 'Testing'
set compile_env: 0
method: ScaledDecimal
even

"Returns true if the receiver is an even integer, false otherwise."
| frac intPart m sc | 
m := mantissa .
(sc := scale) == 0 ifTrue:[ ^ m even ].
sc := 10 raisedTo: sc .
intPart := m // sc .
frac := m - (intPart * sc) .
frac ~~ 0 ifTrue:[ ^ false ] .
^ intPart even 
%
category: 'Truncation and Rounding'
set compile_env: 0
method: ScaledDecimal
floor
"Returns the integer that is closest to the receiver, on the same side
 of the receiver as negative infinity."

^ mantissa // (10 raisedTo: scale)
%
category: 'Comparing'
set compile_env: 0
method: ScaledDecimal
hash
 
^ self asFloat hash
%
category: 'Truncation and Rounding'
set compile_env: 0
method: ScaledDecimal
integerPart
  ^ self class for: self truncated scale:  scale
%
category: 'Testing'
set compile_env: 0
method: ScaledDecimal
isZero

"Returns true if the receiver is zero."

^ mantissa == 0 .
%
category: 'Accessing'
set compile_env: 0
method: ScaledDecimal
mantissa

^ mantissa
%
category: 'Arithmetic'
set compile_env: 0
method: ScaledDecimal
negated
  "Returns 0 minus self. Result has same scale as receiver. "
^ self class _mantissa: 0 - mantissa scale: scale
%
category: 'Compatibility'
set compile_env: 0
method: ScaledDecimal
numerator
  ^ self asFraction numerator
%
category: 'Testing'
set compile_env: 0
method: ScaledDecimal
odd

"Returns true if the receiver is an odd integer, false otherwise."

| frac intPart m sc |
m := mantissa .
(sc := scale) == 0 ifTrue:[ ^ m odd ].
sc := 10 raisedTo: sc .
intPart := m // sc .
frac := m - (intPart * sc) .
frac ~~ 0 ifTrue:[ ^ false ] .
^ intPart odd
%
category: 'Compatibility'
set compile_env: 0
method: ScaledDecimal
reduced

 ^ self
%
category: 'Truncation and Rounding'
set compile_env: 0
method: ScaledDecimal
rounded
 "Returns the integer nearest in value to the receiver."
  | r |
  r := self + (self class numerator: self sign denominator: 2 scale: scale) .
  ^ r mantissa quo: (10 raisedTo: scale) .
%
category: 'Truncation and Rounding'
set compile_env: 0
method: ScaledDecimal
roundTo: aNumber

"Returns the multiple of aNumber that is nearest in value to the receiver.
  If the receiver is exactly between two multiples of aNumber, 
  answer the one that is even number times aNumber. Example:
    4.125s roundTo: 0.75s --> 4.500s
    4.875s roundTo: 0.75s --> 4.500s
  because 4.5 = 6 * 0.75 "

| r |
aNumber = 0 ifTrue:[ ^ 0 ].
r := (self / aNumber) roundedHalfToEven * aNumber  .
r _isInteger ifTrue:[ ^ self class numerator: r denominator: 1 scale: scale].
^ r
%
category: 'Accessing'
set compile_env: 0
method: ScaledDecimal
scale

^ scale
%
category: 'Accessing'
set compile_env: 0
method: ScaledDecimal
sign
  | m |
  ((m := mantissa) > 0) ifTrue:[ ^ 1 ].
  (m < 0) ifTrue:[ ^ -1 ].
  ^0
%
category: 'Truncation and Rounding'
set compile_env: 0
method: ScaledDecimal
truncated

"Returns the integer that is closest to the receiver, on the same side
 of the receiver as zero is located."

  | f |
  f := self floor.
  (mantissa < 0 and: [f ~= self]) 
      ifTrue:[ f := f + 1 ].
  ^ f 
%
category: 'Converting'
set compile_env: 0
method: ScaledDecimal
truncateTo: aNumber
  | arg |
  (arg := aNumber) _isInteger ifTrue:[
    arg := self class for: aNumber scale: self scale.
  ].
  ^super truncateTo: arg
%
category: 'Compatibility'
set compile_env: 0
method: ScaledDecimal
withScale: newScale
"Returns a new instance with the new scale, and a numeric value 
 as close to that of the receiver as possible."

	^self asScaledDecimal: newScale
%
category: 'Storing and Loading'
set compile_env: 0
method: ScaledDecimal
writeTo: passiveObj

"Converts the receiver to its passive form and writes that information on
 passiveObj."

"Reimplemented from Number since the receiver has a non-literal representation."

^super basicWriteTo: passiveObj
%
category: 'Converting'
set compile_env: 0
method: ScaledDecimal
_coerce: aNumber

"Reimplemented from Number."

^ aNumber asScaledDecimal: scale
%
category: 'Private'
set compile_env: 0
method: ScaledDecimal
_generality

"Returns an Integer representing the ordering of the receiver in
 the generality hierarchy."

^ 60
%
category: 'Testing'
set compile_env: 0
method: ScaledDecimal
_getKind
 | m |
 (m := mantissa) ifNil:[ ^ 5 ]. "nan, should never occur"
 m == 0 ifTrue:[ ^ 4 ]. "zero"
 ^ 0 "normal"
%
category: 'Private'
set compile_env: 0
method: ScaledDecimal
_mantissa: m scale: s
  m _isInteger ifFalse:[ m _validateClass: Integer ].
  s _isSmallInteger ifFalse:[ s _validateClass: SmallInteger ].
  s > MaxScale ifTrue:[ 
    OutOfRange new name: '' max: MaxScale actual: s ; 
	details: 'invalid scale' ; signal
  ].
  s < 0 ifTrue:[ OutOfRange new name: 'scale' min: 0 actual: s ; 
                    details: 'scale must be >= 0' ; signal ].
  mantissa := m .
  scale := s
%
! fix 46790
category: 'Copying'
method:
postCopy
  ^ self immediateInvariant
%

