!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
! Object
!   Collection
!     SequenceableCollection
!       Array
!         AbstractCollisionBucket( numElements)
!           CollisionBucket( keyValueDictionary)
!             SoftCollisionBucket(lastCleanupCount) 
!
!=========================================================================

expectvalue %String
run
| res oldCls clsName |
clsName := #SoftCollisionBucket .
oldCls := Globals at:clsName otherwise: nil .
oldCls ifNil:[
  res := CollisionBucket _newKernelSubclass: clsName"SoftCollisionBucket"
    instVarNames: #( #lastCleanupCount )
    classVars: #()
    classInstVars: #()
    poolDictionaries: { }
    inDictionary: Globals
    options: #( instancesNonPersistent disallowGciStore )
    reservedOop: 945 .
] ifNotNil: [
  res := clsName asString , 'already exists'.
  oldCls instancesNonPersistent ifFalse:[ nil error:'should be NP' ].
].
^ res
%

! Remove existing behavior 
removeallmethods SoftCollisionBucket
removeallclassmethods 
set class SoftCollisionBucket

category: 'For Documentation Installation only'
classmethod
installDocumentation

self comment:
'A SoftCollisionBucket is a CollisionBucket intended to
 store key/SoftReference pairs .
 key/SoftReference pairs whose SoftReferences have been cleared by 
 the in-memory garbage collector will be removed from a SoftCollisionBucket 
 during the next at:put: operation following a garbage collection which cleared
 the references.  
 
 Cleared SoftReferences will also be removed when their key matches
 the argument key for a lookup operation .  The lookup code will
 not remove cleared SoftReferences for non-matching keys.

 Instances of SoftCollisionBucket are non-persistent and may
 not be committed to the repository.

Constraints:
	numElements: SmallInteger
	keyValueDictionary: Object
	lastCleanupCount: SmallInteger

lastCleanupCount is private to the implementation of SoftCollisionBucket, 
 used to manage removal of SoftReferences.'.
%


category: 'Instance creation'
classmethod
new

^ super new _initializeCleanup
%

category: 'Private'
method
_initializeCleanup

  lastCleanupCount := 0 
%


category: 'Private'
method
_referenceAt: idx 

" for the key stored at offset idx ,
  return the SofReference if softRef.value non-nil, otherwise
  remove the soft reference and associated key .
" 
| aSoftRef val | 
aSoftRef := self _at: idx + 1 .
aSoftRef ifNotNil:[ 
  val := aSoftRef _value .
  val ifNil:[ | oldN n aKey |
    aKey := self _at: idx .
    self _at: idx     put: nil . "remove key"
    self _at: idx + 1 put: nil . "remove SoftRef"
    oldN := numElements. 
    n := oldN - 1 .
    n < 0 ifTrue:[ nil error:'numElements underflow in a SoftCollisionBucket'].
    numElements := n .
    keyValueDictionary _bucketSizeChangeFrom: oldN to: n key: aKey 
		bucket: self remove: false .
    ^ nil .
  ]. 
  ^ aSoftRef
].
^ nil
%

category: 'Accessing'
method
referenceAt: aKey ifAbsent: aBlock

"Returns the non-cleared SoftReference that corresponds to aKey.  
 If no such key/SoftRef pair exists,
 returns the result of evaluating the zero-argument block aBlock."

  | keyIndex index aSoftRef |
  keyIndex := self searchForKey: aKey.
  keyIndex ifNotNil: [ 
    index := (keyIndex + keyIndex) - 1 . 
    aSoftRef:=  self _referenceAt: index .
    aSoftRef ifNil:[ ^ self _reportKeyNotFound: aKey with: aBlock ] .
    ^ aSoftRef 
  ].
  ^ self _reportKeyNotFound: aKey with: aBlock 
%
 
category: 'Accessing'
method
referenceAt: aKey otherwise: aValue

"Returns the non-cleared SoftReference that corresponds to aKey.
 If no such key/SoftRef pair exists, returns aValue ."

  | index keyIndex aSoftRef |
  keyIndex := self searchForKey: aKey.
  keyIndex ifNotNil:[
    index := (keyIndex + keyIndex) - 1 .
    aSoftRef :=  self _referenceAt: index .
    aSoftRef ifNil:[ ^ aValue ].
    ^ aSoftRef 
  ].
  ^ aValue   
%

! inherited from AbstractCollisionBucket,  at: anIndex putKey: aKey
! inherited from AbstractCollisionBucket,  at: anIndex putValue: aValue

category: 'Updating'
method
at: aKey put: aValue  

"disallowed, should use  at:put:keyValDict_coll:  "

self shouldNotImplement: #at:put:
%

category: 'Updating'
method
at: aKey put: aValue keyValDict_coll: aKeyValDict

"Stores the aKey/aValue pair in the receiver.  
 Returns self size if this at:put: added a new key, 0 if this at:put:
 replaced a SoftReference for an existing  key .

 aValue is expected to be a SoftReference.

 Also removes key/SoftReference pairs whose SoftReference has been
 cleared by the garbage collector."

| startElem nElem |
self _cleanupReferences: false .
startElem := numElements .
super at: aKey put: aValue keyValDict_coll: aKeyValDict .
startElem < (nElem := numElements) ifTrue:[ ^ nElem ] ifFalse:[ ^ 0 ].
%

category: 'Removing'
method
removeKey: aKey ifAbsent: aBlock

"Removes the key/value pair having the key aKey.  If aKey is not found,
 returns the result of evaluating the zero-argument block aBlock.

 Also removes key/SoftReference pairs whose SoftReference has been
 cleared by the garbage collector."

| res |
res := super removeKey: aKey ifAbsent: aBlock . 
self _cleanupReferences: true .
^ res
%

! added for 36675
category: 'Removing'
method
removeKey: aKey otherwise: notFoundValue

"Removes the key/value pair having the key aKey.  If aKey is not found,
 returns the notFoundValue .

 Also removes key/SoftReference pairs whose SoftReference has been
 cleared by the garbage collector."

| res |
res := super removeKey: aKey otherwise: notFoundValue  .
self _cleanupReferences: true .
^ res
%

category: 'Private'
method
_markSweepsThatClearedSoftRefsCount

"Returns OM.markSweepsThatClearedSoftRefsCount as a positive SmallInteger"

<primitive: 553>

self _primitiveFailed: #_markSweepsThatClearedSoftRefsCount
%


category: 'Cleanup'
method
cleanupReferences
  ^ self _cleanupReferences: true
%

category: 'Cleanup'
method
_cleanupReferences: okToRemoveBucket

"remove key/SoftReference pairs whose SoftReferences have
 been cleared by the garbage collector."
| currCount |
currCount := self _markSweepsThatClearedSoftRefsCount .
currCount == lastCleanupCount ifFalse:[ | nElem firstKey startNumElem |
  startNumElem := (nElem := numElements) .
  1 to: self tableSize do: [:keyIdx | | aRef idx aKey |
    idx := (keyIdx + keyIdx) - 1 . 
    aKey := self _at: idx .
    aKey ifNotNil:[
      aRef := self _at: idx + 1 .
      aRef _value ifNil:[
        firstKey ifNil:[ firstKey := aKey ].
        self _at: idx     put: nil . "remove key"
        self _at: idx + 1 put: nil . "remove SoftRef"
        nElem := nElem - 1 .
        nElem < 0 ifTrue:[ nil error:'underflow1' ].
        numElements := nElem .
      ]. 
    ].
  ]. 
  lastCleanupCount := currCount .
  nElem == startNumElem ifFalse:[
    firstKey ifNil:[ self error:'nil firstKey in cleanupReferences' ].
    keyValueDictionary _bucketSizeChangeFrom: startNumElem to: nElem 
			key: firstKey bucket: self remove: okToRemoveBucket
  ].
].
%
