!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id$
!
! Superclass Hierarchy:
!   SortedCollection, OrderedCollection, SequenceableCollection,
!   Collection, Object.
!
!=========================================================================

! v3.0,  SortedCollection created in bom.c , or already exists from v2.x dbf

! recompile default sort block from v3.2.x or before if possible.
expectvalue %String
run
"recompile default block for SortedCollection>>new"
(SortedCollection class compiledMethodAt: #new otherwise: nil) ifNotNil:[:mth |
  | blk |
  blk := mth blockLiterals atOrNil: 1 .
  (blk isKindOfClass: ExecBlock) ifTrue:[ 
    blk needsRecompile ifTrue:[ blk recompile . ^ 'recompiled']
  ] ifFalse:[ ^ 'not an ExecBlock' ].
] ifNil:[ ^ 'method not found' ].
^ 'no action'
%

! remove existing behavior from SortedCollection
removeallmethods SortedCollection
removeallclassmethods SortedCollection

category: 'For Documentation Installation only'
classmethod: SortedCollection
installDocumentation

self comment:
'A SortedCollection is an OrderedCollection that maintains the order of its
 elements based on a sort block.  In GemStone, SortedCollections are not fixed
 in length as in other Smalltalk systems.

 instvar sortBlock is an ExecBlock that defines the sorting criterion.  The 
 block must take two arguments, and it should return true if the first
 argument should precede the second argument, and false if not.
'.
%

! ------------------- Class methods for SortedCollection
category: 'Instance Creation'
classmethod: SortedCollection
new

"Returns a new instance of the receiver with the sort block
 [ :a :b | a <= b ]."

| inst |
inst := self _basicNew .
inst sortBlock: inst _defaultBlock .
^inst
%

method: SortedCollection
_defaultBlock
  "a single ExecBlock used for the default sort block for both #new and #loadFrom"
^ [ :a :b | a <= b ]
%

! initialize class variables.
category: 'Class initialization'
classmethod: SortedCollection
initDefaultSortblock

"Initialize a reference to the default sort block."

  SortedCollection _classVars at: #DefaultSortBlock put: 
    SortedCollection new sortBlock
%

! fixed bug 7435

category: 'Instance Creation'
classmethod: SortedCollection
new: size

"Returns a new instance of the receiver with size 0 and the sort block
 [ :a :b | a <= b ].  This method is synonymous with new and is provided for
 compatibility with other Smalltalk dialects that do not have objects that are
 truly variable in size."

^ self new 
%

category: 'Updating'
method: SortedCollection
size: anInteger

"If anInteger is less than the current size of the receiver, shrinks
 the receiver, otherwise has no effect."

"This implementation choice allows removal methods to be safely inherited 
 from SequenceableCollection."

anInteger < self size ifTrue:[ super size: anInteger ].
%

category: 'Instance Creation'
classmethod: SortedCollection
sortBlock: aBlock

"Returns a new instance of the receiver with the given sort block."

| inst |
inst := super new.
inst sortBlock: aBlock.
^inst
%

category: 'Instance Creation'
classmethod: SortedCollection
sortBlock: aBlock fromSortResult: sortArray

"Returns a new instance of the receiver with the given sort block and contents.
 The argument sortArray is assumed to be in the proper sort order and is
 installed as the presorted contents of the new instance."

| inst |
inst := super new.
inst sortBlock: aBlock.
1 to: sortArray size do: [:i |
  inst _basicAt: i put: (sortArray at: i)
].
^inst
%

category: 'Instance Creation'
classmethod: SortedCollection
withAll: collection sortBlock: block

"Returns a new instance of the receiver with the given sort block and
 contents."

| instance |
instance := self sortBlock: block.
instance addAll: collection.
^instance
%

category: 'Instance Creation'
classmethod: SortedCollection
withAll: aCollection

"Returns an instance of the receiver containing the elements of the argument."

| inst |
inst := self new .
inst addAll: aCollection.
^ inst
%

category: 'Instance Creation'
classmethod: SortedCollection
with: aValue

"Returns an instance of the receiver containing the argument."

| inst |
inst := self new .
inst add: aValue .
^ inst
%

category: 'Instance Creation'
classmethod: SortedCollection
with: aValue with: val2

"Returns an instance of the receiver containing the arguments."

| inst |
inst := self new .
inst add: aValue ; add: val2 .
^ inst
%

category: 'Instance Creation'
classmethod: SortedCollection
with: aValue with: val2 with: val3

"Returns an instance of the receiver containing the arguments."

| inst |
inst := self new .
inst add: aValue ; add: val2 ; add: val3 .
^ inst
%

set compile_env: 0
category: 'Instance Creation'
classmethod: SortedCollection
with: aValue with: val2 with: val3 with: val4
	"Returns an instance of the receiver containing the arguments."

	| inst |
	inst := self new .
	inst add: aValue ; add: val2 ; add: val3; add: val4 .
	^inst.
%

! ------------------- Instance methods for SortedCollection

! deleted _addAll: which was unused

category: 'Private'
method: SortedCollection
_at: anIndex put: aValue

"Puts the given element into the receiver at the given index, without
 sorting."

^super at: anIndex put: aValue
%

category: 'Private'
method: SortedCollection
_findIndex: anObject

"Finds and returns the index for placing the given object into the receiver.
 A simple binary probe is used."

| obj lower upper half probe |

self size == 0 ifTrue: [ ^1 ].

lower := 1.
upper := self size.
[ half := upper - lower + 1 // 2.
  probe := lower + half.
  -1 < half and: [ lower <= upper ] ] whileTrue: [
  obj := self at: probe.
  (sortBlock value: obj value: anObject) ifTrue: [
    "after this index - go to the next higher cell"
    lower := probe + 1.
  ]
  ifFalse: [
    "before this index - go to the next lower cell"
    upper := probe - 1
  ]
].
^probe
%

category: 'Adding'
method: SortedCollection
add: anObject

"Adds anObject to the receiver.  Increases the size of the receiver by one.
 Enforces the sorting order.  Returns anObject."

| idx |
idx := self _findIndex: anObject.
self _insertObject: anObject at: idx .
^ anObject
%
method: 
_insertObject: anObject at: anIndex

"Inserts anObject into the receiver at index anIndex and returns anObject." 

<primitive: 398>
"primitive fails if receiver is byte format"
(anIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
(anIndex < 1 or: [ anIndex > (self size + 1)]) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].

^ self _primitiveFailed: #_insertAll:at: args: { anObject . anIndex }
%

category: 'Adding'
method: SortedCollection
addAll: aCollection

"Adds the elements of aCollection to the receiver.  Increases the size of the
 receiver by the number of elements in aCollection.  Enforces the sorting order.
 Returns aCollection."

| collCopy |
aCollection == self ifTrue:[
  collCopy := Array withAll: self . 
  1 to: collCopy size do:[:j | self add: (collCopy at: j) ].
]
ifFalse:[
  (self size == 0 and:[ aCollection size > 2000]) ifTrue:[
    self _mergeSortAddAll: aCollection .
  ] ifFalse:[
    (aCollection isKindOf: SequenceableCollection) ifTrue:[
      1 to: aCollection size do:[:j |
        self add:( aCollection at: j )
      ].
    ] ifFalse:[
      aCollection accompaniedBy: self do: [:me :each | me add: each] .
    ].
  ].
].
^ aCollection
%

category: 'Adding'
method: SortedCollection
addLast: anObject

"Disallowed.  Reports an error since SortedCollections have a sorting order
 that prohibits outside interference."

self shouldNotImplement: #addLast:
%

category: 'Updating'
method: SortedCollection
at: index put: anObject

"Disallowed.  Reports an error since SortedCollections have a sortBlock that
 determines the order of their contents."

^self shouldNotImplement: #at:put:
%

category: 'Updating'
method: SortedCollection
atAllPut: anObject

"Assigns anObject to each of the receiver's elements."

1 to: self size do: [:index |
   super at: index put: anObject
].
^anObject
%

set compile_env: 0
category: 'Concatenating'
method: SortedCollection
, aSequenceableCollection
	"Returns a new instance of the receiver's class that contains the elements of
	 the receiver followed by the elements of aSequenceableCollection.

	 Warning: Creating a new instance and copying the receiver take time.  If you
	 can safely modify the receiver, it can be much faster to use the addAll:
	 method.  See the documentation of the Concatenating category for more details."

	^(self asArray, aSequenceableCollection) asSortedCollection: self sortBlock.
%

category: 'Copying'
method: SortedCollection
_mergeSortAddAll: aCollection 
 
"The receiver must be empty. Adds aCollection to the receiver
 using the merge-sort implementation provided by BlockSorter.
 Returns the receiver."

| arr |
self size == 0 ifFalse:[ self error:'not empty' ].
arr := aCollection sortWithBlock: sortBlock . "merge sort"
self _insertAll: arr at: 1 .
^ self
%   

category: 'Copying'
method: SortedCollection
copyFrom: startIndex to: stopIndex

"Installs the receiver's sortBlock into the copy."

^(super copyFrom: startIndex to: stopIndex) sortBlock: sortBlock
%

set compile_env: 0
category: 'Copying'
method: SortedCollection
copyReplaceAll: oldSubCollection with: newSubCollection

	^(self asArray 
		copyReplaceAll: oldSubCollection with: newSubCollection)
			asSortedCollection: self sortBlock
%

set compile_env: 0
category: 'Copying'
method: SortedCollection
copyReplaceFrom: startIndex to: stopIndex with: aSequenceableCollection
	"Returns a copy of the receiver in which all elements in the
	 receiver between indexes startIndex and stopIndex inclusive have
	 been replaced by those contained in aSequenceableCollection.

	 Note that a SortedCollection will sort the replacement elements
	 into the resulting collection.

	 Rules, abbreviated from the ANSI specification:
		- start < 1 or start > receiver's size + 1 (error)
		- stop < start - 1 or stop > receiver's size (error)
		- stop = start - 1 and start <= receiver's size (insert between stop and start)
		- stop = start - 1 and start = receiver's size + 1 (append at the end of the new collection)
		- otherwise, replace from start through stop with the elements from replacementElements.
	"

	^(self asArray
		copyReplaceFrom: startIndex
		to: stopIndex
		with: aSequenceableCollection) 
			asSortedCollection: self sortBlock.
%

set compile_env: 0
category: 'Copying'
method: SortedCollection
copyReplacing: oldObject withObject: newObject
	"Returns a copy of the receiver in which all occurrences of objects equal to
	 oldObject have been replaced by newObject."

	^(self asArray copyReplacing: oldObject withObject: newObject)
		asSortedCollection: self sortBlock
%

set compile_env: 0
category: 'Copying'
method: SortedCollection
copyWith: anObject
	"Returns a copy of the receiver with the given object included in the correct sort order."
	| res |
	(res := self copy) add: anObject .
	^ res
%

category: 'Copying'
method: SortedCollection
copyWithout: anObject

"Returns a copy of the receiver that does not contain the given object.
 Comparisons are by equality (not identity)."

| copy element sz |

copy := self species sortBlock: sortBlock.
sz := 0.
1 to: self size do: [:i |
  element := self at: i.
  element = anObject ifFalse: [
    sz := sz + 1.
    copy _at: sz put: element
  ]
].
		"no need to resort "
^ copy
%

method: SortedCollection
replaceFrom: startIndex to: stopIndex with: aSeqCollection startingAt: repIndex

"Disallowed."
^ self shouldNotImplement: #replaceFrom:to:with:startingAt:
%

method: SortedCollection
replaceFrom: startIndex to: stopIndex with: aSeqCollection

"Disallowed."
^ self shouldNotImplement: #replaceFrom:to:with:
%

method: SortedCollection
replaceFrom: startIndex to: stopIndex withObject: anObject

"Disallowed."
^ self shouldNotImplement: #replaceFrom:to:withObject:
%



category: 'Copying'
method: SortedCollection
insertAll: aCollection at: anIndex

"Disallowed.  Reports an error since SortedCollections have a sorting order
 that prohibits outside interference."

self shouldNotImplement: #insertAll:at:
%
category: 'Private'
method: SortedCollection
_insertAll: aCollection at: anIndex

"Inserts all the elements of aCollection into the receiver beginning at index
 anIndex.  Returns aCollection."

"The argument anIndex must be greater than or equal to one.  If anIndex is one
 greater than the size of the receiver, appends aCollection to the receiver.  If
 anIndex is more than one greater than the size of the receiver, generates an
 error."

<primitive: 606 >
( self _hasModificationTracking and:
[ aCollection isKindOf: CharacterCollection ] )
  ifTrue: [ " primitive failed because aCollection is not an
             Array or OrderedCollection"
    ^ self insertAll: (Array withAll: aCollection) at: anIndex
  ].
self _primitiveFailed: #_insertAll:at: args: { aCollection . anIndex }
%

category: 'Copying'
method: SortedCollection
insertObject: anObject at: anIndex

"Disallowed.  Reports an error since SortedCollections have a sorting order
 that prohibits outside interference."

self shouldNotImplement: #insertObject:at:
%

category: 'Copying'
method: SortedCollection
reverse
  "Returns an OrderedCollection which is a copy of the receiver 
   with element order reversed."

  | n result src |
  n := self size.
  "GemStone disallows at: put: on a SortedCollection   
    ... reversing a SortedCollection without changing the sortBlock seems odd"
  result := OrderedCollection new: n.
  src := n + 1.
  1 to: n do: [:i | result at: i put: (self at: (src := src - 1))].
  ^ result
%

category: 'Searching'
method: SortedCollection
reject: aBlock

"Pass on the sort block."

|result mySize each |

result := self species sortBlock: sortBlock.
mySize := self size .
1 to: mySize do:[:j|
  each := self at: j .
  (aBlock value: each) ifFalse: [result _addLast: each]
  ].
^result
%

category: 'Private'
method: SortedCollection
_addLast: anObject

"Used when caller knows that appends to the receiver are being done in 
 sorted order."

^ super add: anObject
%

category: 'Searching'
method: SortedCollection
select: aBlock

"Returns an instance of the receiver's species that has the receiver's sort
 block."

| result mySize each |

result:= self species sortBlock: sortBlock.
mySize := self size .
1 to: mySize do:[:j|
  each := self at: j .
  (aBlock value: each) ifTrue: [result _addLast: each]
  ].
^result
%

category: 'Accessing'
method: SortedCollection
sortBlock

"Returns the value of the instance variable 'sortBlock'."

^sortBlock
%

category: 'Updating'
method: SortedCollection
sortBlock: newBlock

"Installs a new sort block in the receiver and forces a resort."

sortBlock := newBlock.
self size > 1 ifTrue: [
  self resort
]
%

category: 'Storing and Loading'
method: SortedCollection
loadFrom: passiveObj

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into the receiver."

super loadFrom: passiveObj.
sortBlock ifNil:[ sortBlock := self _defaultBlock ].
%

category: 'Searching'
method: SortedCollection
indexOfValue: anObject

self deprecated: 'SortedCollection>>indexOfValue: deprecated v3.2. Use indexOf: instead.'.
^ self indexOf: anObject
%

category: 'Searching'
method: SortedCollection
indexOf: anObject

"Returns the index of the first occurrence of an object equal to anObject 
 in the receiver.  If the receiver does not contain such an object, this 
 returns zero."

| idx |
" _findIndex: returns where anObject would be inserted, so look at
surrounding slots to see if the object is present "
idx := self _findIndex: anObject.

(sortBlock value: anObject value: anObject) ifTrue: [
  "Need to look at the slots less than idx"
  | foundIdx | "We need to find the first one that is equal"
  foundIdx := 0.
  (idx-1) _downTo: 1 do: [ :i | 
    (anObject = (self at: i)) ifTrue: [foundIdx := i].
    (i > 1) ifTrue: [
      "Check to see if the next element might be equal to the current element
       according to the sort block."
      (sortBlock value: (self at: i) value: (self at: i-1)) ifFalse: [
        "We don't need to look at any more because i-1 can't be equal
         to i."
        ^foundIdx
      ].
    ].
  ].
  ^foundIdx
] ifFalse: [
  "Need to look at the slots greater than idx"
  | lastIdx |
  lastIdx := self size.
  idx to: lastIdx do: [ :i |
    (anObject = (self at: i)) ifTrue: [^i].
    (i < lastIdx) ifTrue: [
      "Check to see if the next element might be equal to the current element
       according to the sort block."
      (sortBlock value: (self at: i) value: (self at: i+1)) ifTrue: [
        "We don't need to look at any more because i+1 can't be equal
         to i."
        ^0
      ].
    ].
  ].
  ^0.
].
%

category: 'Searching'
method: SortedCollection
indexOfIdentical: anObject

"Private.  Returns the index of the first element in the receiver that is
 identical to the argument.  If the receiver does not have any elements that are
 identical to the argument, returns zero."

| idx |

(self size <= 2000) ifTrue: [
  "OrderedCollection's indexOfIdentical: uses a primitive and is
   faster for smaller SortedCollections"
  ^super indexOfIdentical: anObject.
].

" _findIndex: returns where anObject would be inserted, so look at
surrounding slots to see if the object is present "
idx := self _findIndex: anObject.

(sortBlock value: anObject value: anObject) ifTrue: [
  "Need to look at the slots less than idx"
  | foundIdx | "We need to find the first one that is equal"
  foundIdx := 0.
  (idx-1) _downTo: 1 do: [ :i | 
    (anObject == (self at: i)) ifTrue: [foundIdx := i].
    (i > 1) ifTrue: [
      "Check to see if the next element might be equal to the current element
       according to the sort block."
      (sortBlock value: (self at: i) value: (self at: i-1)) ifFalse: [
        "We don't need to look at any more because i-1 can't be equal
         to i."
        ^foundIdx
      ].
    ].
  ].
  ^foundIdx
] ifFalse: [
  "Need to look at the slots greater than or equal to idx"
  | lastIdx |
  lastIdx := self size.
  idx to: lastIdx do: [ :i |
    (anObject == (self at: i)) ifTrue: [
      ^i
    ].
    (i < lastIdx) ifTrue: [
      "Check to see if the next element might be equal to the current element
       according to the sort block."
      (sortBlock value: (self at: i) value: (self at: i+1)) ifTrue: [
        "We don't need to look at any more because i+1 can't be equal
         to i."
        ^0
      ].
    ].
  ].
  ^0.
].
%

method: SortedCollection
binarySearchIncludes: anObject

"Returns true if the argument anObject is equal to an element of the receiver.
 Returns false otherwise.
 
 Note that this search is optimized based on the assumption that the sort criteria
 is consistent with the equality comparison. That is, if (x <= y) & (y <= x) then (x = y).
 This assumption will not hold if sorting is done on one instance variable and 
 equality comparison is done on another instance variable (for example).
 SortedCollection>>#'includes:' is no longer implemented and the linear search
 is inherited from Collection.  See bug 40575."

^ (self indexOf: anObject) ~~ 0
%

!  initDefaultSortblock executed later in filein1.topaz

category: 'Enumerating'
method: SortedCollection
speciesForCollect

"Returns a class, an instance of which should be used as the result of
 collect: or other projections applied to the receiver."

^ OrderedCollection
%

! deletions

! added for 36675
category: 'Removing'
method: SortedCollection
removeIdentical: anObject otherwise: notFoundValue

"Removes from the receiver an object that is identical to anObject and
 returns anObject.  If several elements of the receiver are identical to
 oldObject, only one instance is removed.  If anObject is not present in
 the receiver, returns notFoundValue."

| i |
i := self indexOfIdentical: anObject.
i ~~ 0 ifTrue: [
   self removeFrom: i to: i.
   ^ anObject 
] ifFalse: [ 
  ^ notFoundValue 
]
%


! _unscheduleProcess: aGsProcess  # deleted, should not be needed


category: 'Repository Conversion'
method: SortedCollection
_sortBlock: newBlock

"Installs a new sort block in the receiver"

sortBlock := newBlock.
%


! fixed 48050
category: 'Repository Conversion'
classmethod: SortedCollection
resortAll: anArray
  "Resort an Array of SortedCollections by sending resort to the collecion. anArray must contain 
   only SortedCollections.

   Caller must handle and catch AlmostOutOfMemory errors and handle commits
 and aborts."

  | persistentRoot sys ug commitBlock |
  persistentRoot := SymbolAssociation new
    key: #'SortedCollection_Resort_Persistent_Root'
    value: nil.
  sys := System.
  ug := GsSession currentSession symbolList objectNamed: #'UserGlobals'.
  ug addAssociation: persistentRoot.
  commitBlock := [
    sys commitTransaction
      ifFalse: [ 
        sys abortTransaction.
        TransactionError
          signal: 'Unexpected commit failure in post conversion'  ]].
  [ 
  1 to: anArray size do: [ :n | 
    | sc |
    sc := anArray at: n.
    sc size > 1
      ifTrue: [
        | saveArray |
        saveArray := Array new.
        persistentRoot value: saveArray.	"root saveArray in UserGlobals to allow larger than TOC collections"
        commitBlock value.
        saveArray addAll: sc. 
        commitBlock value. 
        sc _basicSize: 0.
        sc addAll: saveArray.
        saveArray _basicSize: 0.
        commitBlock value ] ] ]
    ensure: [ 
      ug removeAssociation: persistentRoot.
      commitBlock value ]
%

category: 'Repository Conversion'
classmethod: SortedCollection
buildFileNameForGem: anInt

| result |
result := String withAll: 'failedSortedCollections_' .
result addAll: anInt asString;
      addAll: '.bm' .
^ result
%

category: 'Repository Conversion'
classmethod: SortedCollection
getUpgradeLogDir

| uld |
uld := System gemEnvironmentVariable: 'upgradeLogDir' .
uld ifNil:[ LookupError signal: 'Could not resolve the $upgradLogDir environment variable' ] .
((uld at: uld size) == $/)
  ifFalse:[ uld add: $/ ].
^ uld
%

! 44610
category: 'Repository Conversion'
classmethod: SortedCollection
getSortedCollDir
| dir |
dir := self getUpgradeLogDir.
dir addAll: '/scbm'.
^ dir
%

category: 'Repository Conversion'
classmethod: SortedCollection
writeFailedConversionObjsToFileForGem: gemNum bmForErrors: errBm

| fileName numOops |

 numOops := errBm size.
 numOops == 0
  ifTrue:[ ^ true ]. "Nothing to do"

 fileName := self getUpgradeLogDir .
 fileName addAll: (self buildFileNameForGem: gemNum) .

"Delete an old version, if any"
(GsFile existsOnServer: fileName)
  ifTrue:[ GsFile removeServerFile: fileName ].

errBm writeToFile: fileName.
^ true
%

category: 'Repository Conversion'
classmethod: SortedCollection
findAllSortedCollectionsForUser: aUserId

| co allClasses dir |
co := ClassOrganizer newWithRoot: self forUserId: aUserId .
allClasses := co allSubclassesOf: self .
allClasses add: self .
dir := self getSortedCollDir. 
(GsFile exists: dir) 
  ifTrue:[ GsFile removeServerDirectory: dir ].
GsFile createServerDirectory: dir.
^ SystemRepository fastListInstances: allClasses toDirectory: dir
%


category: 'Repository Conversion'
classmethod: SortedCollection
writeTotalsFile: aString counterIndex: idx

| fn count gsf |
fn := self getUpgradeLogDir.
fn addAll: aString .
gsf := GsFile openWriteOnServer: fn .
gsf ifNil:[ IOError signal: ('Error opening file ', fn) ].
count := System globalSessionStatAt: idx .
gsf nextPutAll: count asString .
gsf close .
^ count
%

! Called from postconv upgrade script.
category: 'Repository Conversion'
classmethod: SortedCollection
writeTotalsFiles

self writeTotalsFile: 'sc_ok.txt' counterIndex: 0 .
self writeTotalsFile: 'sc_failed.txt' counterIndex: 1 .
self writeTotalsFile: 'sc_done.txt' counterIndex: 2 .
^ true
%

! Called from postconv upgrade script.
category: 'Repository Conversion'
classmethod: SortedCollection
initializeForConversion

System globalSessionStatAt: 0 put: 0 .
System globalSessionStatAt: 1 put: 0 .
System globalSessionStatAt: 2 put: 0 .
^ true
%

! Called from postconv upgrade script.
! Run as SystemUser
category: 'Repository Conversion'
classmethod: SortedCollection
updateDbfHistoryForConversion

| curHist verString message ok failed |

ok := (System globalSessionStatAt: 0) asString .
failed := (System globalSessionStatAt: 1) asString .

verString := System gemVersionAt: #gsRelease .
curHist := (Globals at: #DbfHistory ifAbsent: ['']).
message := curHist, 'GemStone ', verString, 
                    ' post conversion upgrade completed at ' ,
                     DateTime now asString , 
                    '.  SortedCollection conversion results: ', ok,
                    ' OK, ', failed, ' failed.'. Character lf .
Globals at: #DbfHistory put: message .
System addAllToStoneLog: message .
^ message
%

category: 'Subclass Creation'
classmethod: SortedCollection
byteSubclass: aString
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
newVersionOf: oldClass
description: aDescription
options: options

"Disallowed for SortedCollection and its subclasses."
^ self _error: #classErrByteSubclass
%

category: 'Disallowed'
classmethod: SortedCollection
byteSubclass: aString
classVars: anArrayOfClassVars
classInstVars: anArrayOfClassInstVars
poolDictionaries: anArrayOfPoolDicts
inDictionary: aDictionary
inClassHistory: aClassHistory
description: aDescription
isInvariant: invarBoolean

"Disallowed for SortedCollection and its subclasses."
^ self _error: #classErrByteSubclass
%

category: 'Filein Support'
classmethod: SortedCollection
_newKernelByteSubclass: clsName
classVars: classVarArg
poolDictionaries: poolDicts
inDictionary: aDict
options: options
reservedOop: aReservedOop

"Disallowed for SortedCollection and its subclasses."

^ self _error: #classErrByteSubclass
%

category: 'Accessing'
method: SortedCollection
at: anIndex

"Returns the value of an indexed variable in the receiver.
 The argument anIndex must not be larger than the size of the
 receiver, and must not be less than 1.

 Generates an error if anIndex is not a SmallInteger or is out of
 bounds, or if the receiver is not indexable."

<primitive: 699>
(anIndex _isInteger)
  ifTrue: [^ self _errorIndexOutOfRange: anIndex]
  ifFalse: [^ self _errorNonIntegerIndex: anIndex].
^ self _primitiveFailed: #at: args: { anIndex }
%

category: 'Accessing'
method: SortedCollection
atOrNil: anIndex

"Return the value   self at: anIndex ,
 or return nil if anIndex is out of range . "

<primitive: 699>
anIndex _isInteger ifTrue:[
  ^ nil 
].
self _errorNonIntegerIndex: anIndex .
^ self _primitiveFailed: #atOrNil: args: { anIndex }
%

category: 'Indexing Support'
classmethod: SortedCollection
_canCreateQueryOnInstances
  "Cannot create a GsQuery on the receiver"

  ^ false
%
