!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id: string.gs 47587 2020-01-07 18:05:31Z lalmarod $
!
! Superclass Hierarchy:
!   String, CharacterCollection, SequenceableCollection, Collection, Object.
!
!=========================================================================

removeallmethods String
removeallclassmethods String
set class String

category: 'For Documentation Installation only'
classmethod:
installDocumentation

self comment:

'Instances of class String are indexed collections of Characters.

 Comparing strings:
 Unless noted otherwise, comparisons between Strings are case-sensitive.

 Some of these methods determine whether one String collates before
 another.  In collation, the ASCII values of the receiver and aString are
 compared Character-by-Character, from left to right, in case-sensitive
 fashion.  If two Strings are of different length, and all Characters in the
 shorter String are equal to their counterparts in the longer String, the
 shorter String collates before the longer.

 Converting Strings:
 Separator Characters in a String are used to define white space that separates
 words or substrings within the String from each other.  These separators are
 used in conversion to break the String into its logical substrings.  Separator
 Characters are defined as those Characters for which the
 Character>>isSeparator method returns true.' .
%

category: 'Instance Creation'
classmethod:
withAll: aString

"Returns an instance of the receiver, DoubleByteString, or QuadByteString
 using the minimum bytes per character required to represent the argument."

<primitive: 294>
^ super withAll: aString
%

classmethod:
withBytes: aCollection

"Returns an instance of the receiver. aCollection must be a kind of 
 String or ByteArray"

<primitive: 1021>
ArgumentTypeError new name: 'aCollection' expectedClass: { String . ByteArray . Utf8 } 
		         actualArg: aCollection ; signal
%

category: 'Storing and Loading'
classmethod:
loadFrom: passiveObj

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| inst |

inst := self new: passiveObj readSize.
inst loadFrom: passiveObj.
^inst
%

category: 'Deprecated'
classmethod:
fromClientTextFile: fileName

| file instance |
self deprecated: 'String class>>fromClientTextFile: deprecated long before v3.0. Use an instance of GsFile to 
 access the file system.'.
file := GsFile openRead: fileName.
file == nil ifTrue: [
  self _halt: 'Unable to open file: ' , fileName.
].
instance := self new.
file next: file fileSize into: instance.
file close.
^instance
%
category: 'Instance Creation'
classmethod:
new: anInteger

"Returns an instance of the receiver with the specified size,
 with all codepoints set to zero.
 Generates an error if anInteger is not a positive SmallInteger."

<primitive: 53>
(anInteger _isInteger)
  ifFalse:[ anInteger _validateClass: Integer . ^ self new ]
  ifTrue:[
    (anInteger < 0) ifTrue:[ anInteger _error: #rtErrArgNotPositive .
                            ^ self new].
    anInteger _error: #rtErrArgOutOfRange .
    ^ self new
  ].
self _primitiveFailed: #new: args: { anInteger }.
self _uncontinuableError
%

classmethod:
new

"Returns an instance of the receiver of size zero"
<primitive: 51>
self _primitiveFailed: #new .
self _uncontinuableError
%

category: 'Concatenating'
method:
, aCharOrCharCollection

"Returns a new instance of the receiver's class that contains the elements of
 the receiver followed by the elements of aCharOrCharCollection.  The argument
 must be a String or an AbstractCharacter.

 The result may not be an instance of the class of the receiver if one of the
 following rules applies:

 1) If the receiver is a Symbol or InvariantString, the result is a String.

 2) If the class of the argument is not String, and is ISOLatin or some other
    user-defined subclass of String, and is not a subclass of
    ObsoleteInvariantString, then the result is an instance of the class of the
    argument.

 3) If the argument is a kind of DoubleByteString, the result is a
    DoubleByteString.

 4) If the argument is a kind of QuadByteString, the result is a
    QuadByteString.

 Warning: Creating a new instance and copying the receiver take time.  If you
 can safely modify the receiver, it can be much faster to use the addAll:
 method.  See the documentation of the Concatenating category of class
 SequenceableCollection for more details."

<primitive: 615 >

"If the primitive failed, the argument is bad, or the argument is
 a JapaneseString or JISCharacter, or Character with value > 255."

| cSize result |
(cSize := aCharOrCharCollection stringCharSize) >= 2 ifTrue:[
  (cSize == 2) ifTrue:[
    result := DoubleByteString withAll: self.
    result addAll: aCharOrCharCollection.
    ^ result .
  ] ifFalse:[
    result:= QuadByteString withAll: self.
    result addAll: aCharOrCharCollection.
    ^ result .
  ].
].
(aCharOrCharCollection class == Character) ifTrue:[ 
  "single byte char would have been handled by prim"
  aCharOrCharCollection codePoint <= 16rFFFF ifTrue:[
    result := DoubleByteString withAll: self.
  ] ifFalse:[
    result:= QuadByteString withAll: self.
  ].
  result at: (result size + 1) put: aCharOrCharCollection .
  ^ result
].   
aCharOrCharCollection _validateClasses:{ AbstractCharacter . CharacterCollection }.
^ self asEUCString , aCharOrCharCollection .
%

category: 'Accessing'
method:
at: anIndex

"Returns the Character at anIndex."

<primitive: 69>
(anIndex _isSmallInteger)
  ifTrue: [^ self _errorIndexOutOfRange: anIndex]
  ifFalse: [^ self _errorNonIntegerIndex: anIndex].
self _primitiveFailed: #at: args: { anIndex } .
self _uncontinuableError
%

method:
codePointAt: anIndex

"Returns the ascii value of the Character at anIndex."

<primitive: 756>

(anIndex _isSmallInteger)
  ifTrue: [^ self _errorIndexOutOfRange: anIndex]
  ifFalse: [^ self _errorNonIntegerIndex: anIndex].
self _primitiveFailed: #codePointAt: args: { anIndex } .
self _uncontinuableError
%


category: 'Accessing'
method:
size

"Returns the size of the receiver, in Characters."

<primitive: 0>
self _primitiveFailed: #size .
self _uncontinuableError
%

category: 'Updating'
method:
size: anInteger

"Changes the size of the receiver to anInteger.
 Returns the receiver.

 If anInteger is less than the current size of the receiver, the receiver is
 shrunk accordingly.  If anInteger is greater than the current size of the
 receiver, the receiver is extended and new elements are initialized to nil."

<primitive: 603 >
(anInteger _isSmallInteger)
   ifTrue: [^ self _errorIndexOutOfRange: anInteger]
   ifFalse: [^ self _errorNonIntegerIndex: anInteger].
self _primitiveFailed: #size: args: { anInteger } .
self _uncontinuableError
%

category: 'Private'
method:
_changeClassToMultiByte: aClass

"Private.   Used when loading strings from a PassiveObject.
 Change the class of the receiver to aClass, which must be
 a subclass of MultiByteString, and convert bytes of receiver
 from  big-endian to native byte order.  
 structure of the receiver.  Returns the receiver."

<primitive: 466>
self _primitiveFailed: #_changeClassToMultiByte: args: { aClass }
%


category: 'Updating'
method:
at: anIndex put: aChar

"Stores aChar at the specified location."

<primitive: 293>
(aChar class == Character) ifTrue:[  | av |
  (av := aChar codePoint) > 16rFFFF ifTrue:[
    ^ self _convertToQuadByte at:anIndex put:aChar
  ].
  av > 16rFF ifTrue:[
    ^ self _convertToDoubleByte at:anIndex put:aChar
  ].
] ifFalse:[
 aChar _validateClass: AbstractCharacter . 
 ^ self at: anIndex put: aChar asCharacter 
].
(anIndex _isSmallInteger) ifTrue: [
  ((anIndex > (self size + 1)) or: [anIndex <= 0]) ifTrue: [
    ^ self _errorIndexOutOfRange: anIndex
  ]
] ifFalse: [
  ^ self _errorNonIntegerIndex: anIndex
] .
self _primitiveFailed: #at:put: args: { anIndex . aChar }
%

category: 'Updating'
method:
codePointAt: anIndex put: aValue

"Stores aValue at the specified location, with autogrow, but
 without auto-conversion of receiver to larger character size.
 Returns aValue.
"
<primitive: 784>
aValue _validateClass: SmallInteger .
(aValue < 0 or:[ aValue > 255]) ifTrue:[
  aValue _error: #rtErrArgOutOfRange args:{ 0 . 255 }  
].
(anIndex _isSmallInteger) ifTrue: [ 
  ((anIndex > (self size + 1)) or: [anIndex < 1]) ifTrue: [
    ^ self _errorIndexOutOfRange: anIndex 
  ]
] ifFalse: [
  ^ self _errorNonIntegerIndex: anIndex
] .
self _primitiveFailed: #codePointAt:put: args: { anIndex . aValue }
%

category: 'Case-Insensitive Comparisons'
method:
isEquivalent: aCharCollection

"Returns true if the receiver is equivalent to aCharCollection.  The receiver
 and aCharCollection are compared without regard to case or internal
 representation of Characters."

<primitive: 95>
aCharCollection _stringCharSize ~~ 0 ifTrue:[
  self _primitiveFailed: #isEquivalent:  args: { aCharCollection } .
  ^ self _uncontinuableError
].
(aCharCollection isKindOf: CharacterCollection) ifTrue: [ 
  ^ self isEquivalent: (String withAll: aCharCollection)
] ifFalse:[
  ^ false 
].
%

category: 'Formatting'
method:
asString

"Returns the receiver."

^self
%
category: 'Converting'
method:
asByteArray

	^ByteArray fromString: self
%

! category: 'Formatting'
! method:
! describe
! 
! "If the receiver is less than half a megabyte, quotes it.
!  Otherwise, returns the receiver."
! 
! self size > 500000
!   ifTrue: [ ^self ]
!   ifFalse: [ ^self quoted ]
! %

category: 'Formatting'
method:
printOn: aStream

"Puts a displayable representation of the receiver on the given stream."

"For classes whose instances can be literals, the result should contain
 formatting information.  For example, the following expression should
 evaluate to true:

 #abc asString = String withAll: '#abc'

 The formatting information is may not be included if the String exceeds
 1000000 (1 MByte) in size."

| idx lastIdx sz maxSz|

aStream _nextPut: $' .

idx := 0.
self size < (maxSz := aStream maxSize) ifTrue:[
  idx := self indexOf: $' startingAt: 1 .
].
idx == 0 ifTrue:[
  aStream nextPutAll: self .
  ]
ifFalse:[
  lastIdx := 1.
  [ (idx == 0) or: [aStream isFull]] whileFalse: [
    aStream nextPutAll: (self copyFrom: lastIdx to: idx) .
    aStream nextPut: $' .
    lastIdx := idx + 1 .
    idx := self indexOf: $' startingAt: lastIdx .
    ].
  sz := self size .
  ((lastIdx <= sz) and: [aStream isFull not]) ifTrue: [
    aStream nextPutAll: (self copyFrom: lastIdx to: sz)  .
    ].
  ].
aStream _nextPut: $' 
%

category: 'Converting'
method:
asSymbol

"Returns a Symbol that corresponds to the receiver."

<primitive: 313>
self _primitiveFailed: #asSymbol
%

! asObsoleteSymbol moved to string2.gs

category: 'Converting'
method:
asSymbolKind

"Equivalent to asSymbol."

<primitive: 313>
self _primitiveFailed: #asSymbolKind
%

category: 'Converting'
method:
asUppercase

"Returns a new instance of the receiver's class, with all lower-case Characters
 in the receiver changed to upper-case.  

 Upper-case Characters are defined by the Unicode standard and
 u_toupper from libicu is called for each character in the receiver.

 If the receiver is a Symbol, returns an instance of String."
<primitive: 298>
self _primitiveFailed: #asUppercase .
self _uncontinuableError
%
category: 'Converting'
method:
asLowercase

"Returns a new instance of the receiver's class, with all lower-case Characters
 in the receiver changed to upper-case.  

 Lower-case Characters are defined by the Unicode standard and
 u_tolower from libicu is called for each character in the receiver.

 If the receiver is a Symbol, returns an instance of String."
<primitive: 868>
self _primitiveFailed: #asLowercase .
self _uncontinuableError
%
category: 'Converting'
method:
asUppercaseOld

"Returns a new instance of the receiver's class, with all lower-case Characters
 in the receiver changed to upper-case.  

 Upper-case Characters are the A to Z,
 including letters with diacriticals, up to codePoint 255.
 (deprecated)character data tables are installed are used if installed.
 
 If the receiver is a Symbol, returns an instance of String."
<primitive: 985>
self _primitiveFailed: #asUppercaseOld .
self _uncontinuableError
%
category: 'Converting'
method:
asLowercaseOld

"Returns a new instance of the receiver's class, with all lower-case Characters
 in the receiver changed to upper-case.  

 Lower-case Characters are the a to z,
 including letters with diacriticals, up to codePoint 255.
 (deprecated)character data tables are installed are used if installed.

 If the receiver is a Symbol, returns an instance of String."
<primitive: 986>
self _primitiveFailed: #asLowercaseOld .
self _uncontinuableError
%

category: 'Case-Sensitive Comparisons'
method:
= aCharCollection

"Returns true if corresponding Characters in the receiver and argument are equal
 and aCharCollection is comparable with the receiver, and aCharCollection is
 not a kind of Symbol.  Returns false otherwise.  

 The comparison for equal is case-sensitive.
 Signals an Error if argument is a kind of Unicode7, Unicode16 or Unicode32 .

 Note that 'kind of Symbol' means either an instance of Symbol or instance of
 DoubleByteSymbol."

<primitive: 27>
| argInfo |
argInfo := aCharCollection _stringCharSize .
(argInfo bitAnd: 16r8) ~~ 0 ifTrue:[  "arg is Unicode"
  ^ (self compareTo: aCharCollection collator: nil useMinSize: false) == 0
].
(aCharCollection isKindOf: Utf8) ifTrue:[
  ^ (self compareTo: aCharCollection collator: nil useMinSize: false) == 0
].
(aCharCollection isKindOf: CharacterCollection) ifTrue:[ ^ aCharCollection = self].
^ false .
%

category: 'Case-Insensitive Comparisons'
method:
equalsNoCase: aCharCollection

"Returns true if corresponding Characters in the receiver and argument are equal
 and aCharCollection is comparable with the receiver.  Returns false otherwise.

 The comparison for equal is case-insensitive.

 If aString is a Symbol, there is no difference in behavior 
 (contrast with String | =)."

<primitive: 95>
aCharCollection _stringCharSize ~~ 0 ifTrue:[
  self _primitiveFailed: #equalsNoCase:  args: { aCharCollection } .
  ^ self _uncontinuableError
].
(aCharCollection isKindOf: CharacterCollection) ifTrue: [ 
  ^ self equalsNoCase: (String withAll: aCharCollection)
] ifFalse:[
  ^ false 
].
%

category: 'Case-Insensitive Comparisons'
method:
< aCharCollection

"Returns true if the receiver collates before the argument.  
 Returns false otherwise.

 The comparison is case-insensitive unless the receiver and argument
 are equal ignoring case, in which case upper-case letters collate before
 lower-case letters.  The default behavior for SortedCollections and for
 the sortAscending method in UnorderedCollection is consistent with this
 method, and collates as follows:

     #( 'c' 'MM' 'Mm' 'mb' 'mM' 'mm' 'x' ) asSortedCollection 

   yields the following sort order:

        'c' 'mb' 'MM' 'Mm' 'mM' 'mm' 'x'

 Signals an Error if argument is a kind of Unicode7, Unicode16 or Unicode32."

<primitive: 28>
(aCharCollection _stringCharSize bitAnd: 16r7) ~~ 0 ifTrue:[
  ^ (DoubleByteString withAll: self) < aCharCollection .
].
aCharCollection _validateClass: CharacterCollection .
^ aCharCollection > self
%

category: 'Deprecated'
method:
asciiLessThan: aString

"Returns true if the receiver collates before the argument using the
 ASCII collating table, which collates AB...Z...ab..z."

self deprecated: 'String>>asciiLessThan: deprecated v3.2.'.
^ self lessThan: aString collatingTable: AsciiCollatingTable
%
 
! lessThan: prim 434 removed
! lessThanOrEqual: removed
! greaterThan: prim 184 removed
! greaterThanOrEqual: removed

category: 'Deprecated'
method:
lessThan: aString collatingTable: aTable

"Returns true if the receiver collates before the argument.
 The collating sequence is defined by aTable which must be
 a string with the same character size as the receiver."

self deprecated: 'String>>lessThan:collatingTable: deprecated v3.2'.
^ self _primLessThan: aString collatingTable: aTable
%

category: 'Deprecated'
method:
_primLessThan: aString collatingTable: aTable

"Returns true if the receiver collates before the argument.
 The collating sequence is defined by aTable which must be
 a string with the same character size as the receiver."

<primitive: 359>
aTable _validateClass: String .
aTable size == 256 ifFalse:[ 
  aTable _error: #rtErrBadSize args:{ 256 . aTable size }.
  ].
aString _validateClass: String .
self _primitiveFailed: #lessThan:collatingTable: args: { aString . aTable }
%

category: 'Deprecated'
method:
lessThanOrEqual: aString collatingTable: anArray

"Returns true if the receiver collates before or the same as the argument 
 as defined by the collating table anArray. "

self deprecated: 'String>>lessThanOrEqual:collatingTable: deprecated v3.2'.
^ ( self greaterThan: aString collatingTable: anArray ) not
%

category: 'Deprecated'
method:
greaterThan: aString collatingTable: aTable

"Returns true if the receiver collates after the argument. 
 See also lessThan:collatingTable:  for description of aTable."

self deprecated: 'String>>greaterThan:collatingTable: deprecated v3.2'.
^self _primGreaterThan: aString collatingTable: aTable
%

category: 'Deprecated'
method:
_primGreaterThan: aString collatingTable: aTable

"Returns true if the receiver collates after the argument.
 See also lessThan:collatingTable:  for description of aTable."

<primitive: 433>
aTable _validateInstanceOf: String .
aTable size == 256 ifFalse:[
  aTable _error: #rtErrBadSize args:{ 256 . aTable size }.
  ].
aString _validateClass: String .
self _primitiveFailed: #greaterThan:collatingTable: args: { aString . aTable }
%

category: 'Deprecated'
method:
greaterThanOrEqual: aString collatingTable: aTable

"Returns true if the receiver collates after or the same as the argument.
 The collating sequence is defined by aTable."

self deprecated: 'String>>greaterThanOrEqual:collatingTable: deprecated v3.2'.
^ (self lessThan: aString collatingTable: aTable) not
%

category: 'Deprecated'
method:
equals: aString collatingTable: aTable

"Returns true if the receiver collates the same as the argument.
 The collating sequence is defined by aTable, 
 See also lessThan:collatingTable:  for description of aTable.

 If aString is a Symbol, there is no difference in behavior 
 ( contrast with String>>= )."

self deprecated: 'String>>equals:collatingTable: deprecated v3.2'.
^ self _primEquals: aString collatingTable: aTable
%

category: 'Deprecated'
method:
_primEquals: aString collatingTable: aTable

"Returns true if the receiver collates the same as the argument.
 The collating sequence is defined by aTable,
 See also lessThan:collatingTable:  for description of aTable.

 If aString is a Symbol, there is no difference in behavior
 ( contrast with String>>= )."

<primitive: 446>
aTable _validateInstanceOf: String .
aTable size == 256 ifFalse:[
  aTable _error: #rtErrBadSize args:{ 256 . aTable size }.
  ].
aString _validateClass: String .
self _primitiveFailed: #equals:collatingTable: args: { aString . aTable }
%

category: 'Case-Insensitive Comparisons'
method:
<= aCharCollection

"Returns true if the receiver collates before the argument or if all of the
 corresponding Characters in the receiver and argument are equal.
 Returns false otherwise.

 The comparison is consistent with that defined for the < method."

^(self > aCharCollection) not
%

category: 'Case-Insensitive Comparisons'
method:
> aCharCollection

"Returns true if the receiver collates after the argument.  Returns false
 otherwise.

 The comparison is consistent with that defined for the < method.

 Signals an Error if argument is a kind of Unicode7, Unicode16 or Unicode32."

<primitive: 26>
(aCharCollection _stringCharSize bitAnd: 16r7) ~~ 0 ifTrue:[
  ^ (DoubleByteString withAll: self) > aCharCollection 
].
aCharCollection _validateClass: CharacterCollection.
^ aCharCollection < self
%

category: 'Case-Insensitive Comparisons'
method:
>= aCharCollection

"Returns true if the receiver collates after the argument or if all of the
 corresponding Characters in the receiver and argument are equal.
 Returns false otherwise.

 The comparison is consistent with that defined for the < method."

^ (self < aCharCollection) not
%

category: 'Private'
method:
_at: anIndex equals: aCharCollection ignoreCase: aBoolean

"Returns true if aCharCollection is contained in the receiver, starting at
 anIndex.  Returns false otherwise.  The comparison is case-insensitive."

^ aBoolean ifTrue:[ self at: anIndex equalsNoCase: aCharCollection]
	ifFalse:[ self at: anIndex equals: aCharCollection]
%

category: 'Case-Insensitive Comparisons'
method:
at: anIndex equalsNoCase: aCharCollection

"Returns true if aCharCollection is contained in the receiver, starting at
 anIndex.  Returns false otherwise.  The comparison is case-insensitive."

<primitive: 29>
anIndex _validateClass: SmallInteger.
((anIndex <= 0) or: [(anIndex > self size)])
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex ].
aCharCollection _validateClass: CharacterCollection.
^ self _primitiveFailed: #at:equalsNoCase: args: { anIndex . aCharCollection }
%

category: 'Case-Sensitive Comparisons'
method:
at: anIndex equals: aCharCollection

"Returns true if aCharCollection is contained in the receiver, starting at
 anIndex.  Returns false otherwise.  The comparison is case-sensitive."

<primitive: 778>
anIndex _validateClass: SmallInteger .
((anIndex <= 0) or: [(anIndex > self size)])
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex ].
aCharCollection _validateClass: CharacterCollection.
^ self _primitiveFailed: #at:equals: args: { anIndex . aCharCollection }
%

! fix 46127
category: 'Copying'
method:
replaceFrom: startIndex to: stopIndex with: charCollection startingAt: repIndex

"Replaces the elements of the receiver between the indexes startIndex and
 stopIndex inclusive with the elements of aSeqCollection starting at repIndex.
 If aSeqCollection is identical to the receiver, the source and
 destination blocks may overlap.

 The primitive supports directly the case where 
   (charCollection isKindOfClass: CByteArray) == true
 with repIndex being one-based .

 Returns the receiver."

<primitive: 297>
| cSize argInfo recInfo src7 |
startIndex _isSmallInteger ifFalse:[ startIndex _validateClass: SmallInteger ].
stopIndex _isSmallInteger ifFalse:[ stopIndex _validateClass: SmallInteger ].
repIndex _isSmallInteger ifFalse:[ repIndex _validateClass: SmallInteger ].
argInfo := charCollection _stringCharSize .
recInfo := self _stringCharSize .
(recInfo bitAnd: 16r8) ~~ 0 ifTrue:[  "receivier is a Utf string"
  (cSize := argInfo bitAnd: 16r7) ~~ 0 ifTrue:[
    src7 := charCollection _asUnicode7 .
    (src7 ~~ nil and:[ src7 ~~ charCollection]) ifTrue:[
      ^ self replaceFrom: startIndex to: stopIndex with: src7 startingAt: repIndex
    ].
  ]. 
  cSize == 2 ifTrue:[
    ^ self _convertToDoubleByte 
      replaceFrom: startIndex to: stopIndex with: charCollection startingAt: repIndex
  ].
  ^ self _convertToQuadByte
      replaceFrom: startIndex to: stopIndex with: charCollection startingAt: repIndex
].
(cSize := argInfo bitAnd: 16r7) >= 2 ifTrue:[
   src7 := charCollection _asString .
   (src7 ~~ nil and:[ src7 ~~ charCollection]) ifTrue:[
      ^ self replaceFrom: startIndex to: stopIndex with: src7 startingAt: repIndex
   ].
   cSize == 2 ifTrue:[ ^ self _convertToDoubleByte replaceFrom: startIndex to: stopIndex 
			     with: charCollection startingAt: repIndex 
   ].
   ^ self _convertToQuadByte replaceFrom: startIndex to: stopIndex 
                        with: charCollection startingAt: repIndex
]. 
^ super replaceFrom: startIndex to: stopIndex with: charCollection startingAt: repIndex
%

category: 'Private'
method:
_findString: subString startingAt: startIndex ignoreCase: aBoolean

"If a receiver contains subString beginning at some point at or after
 startIndex, this returns the index at which subString begins.  If the
 receiver does not contain subString, this returns 0."

<primitive: 30>

startIndex _validateClass: SmallInteger.  
"startIndex > self size returns 0"
(startIndex < 1) 
  ifTrue: [ ^ self _error: #objErrBadOffsetIncomplete args: { startIndex } ].
subString _validateClasses:{ String }. 
aBoolean _validateClass: Boolean .
^ self _primitiveFailed: #_findString:startingAt:ignoreCase:
       args: { subString . startIndex . aBoolean }
%

category: 'Private'
method:
_findLastString: subString startingAt: startIndex ignoreCase: aBoolean

"Searches backwards through receiver    starting at startIndex  .
 Returns the index at which subString begins.  If the
 receiver does not contain subString, this returns 0."

<primitive: 451>

startIndex _validateClass: SmallInteger.
(startIndex < 1) | (startIndex > self size)
  ifTrue: [ ^ self _error: #objErrBadOffsetIncomplete args: { startIndex } ].
subString _validateClasses:{ String }. 
aBoolean _validateClass: Boolean .
^ self _primitiveFailed: #_findLastString:startingAt:ignoreCase:
       args: { subString . startIndex . aBoolean }
%

category: 'Case-Sensitive Searching'
method: 
includes: aCharacter

"Returns true if the receiver contains aCharacter, false otherwise.
 The search is case-sensitive."

^(aCharacter isKindOf: AbstractCharacter) and:
	[self includesValue: aCharacter].
%

category: 'Case-Sensitive Searching'
method:
includesValue: aCharacter

"Returns true if the receiver contains aCharacter, false otherwise.
 The search is case-sensitive."

<primitive: 94>
aCharacter _validateClass: AbstractCharacter .
^ self includesValue: aCharacter asCharacter .
%

category: 'Case-Sensitive Searching'
method:
indexOf: aCharacter startingAt: startIndex

"Returns the index of the first occurrence of aCharacter in the receiver,
 not preceding startIndex.  If the receiver does not contain aCharacter,
 returns zero.   

 The search is case-sensitive."

<primitive: 93>
startIndex _validateClass: Integer .
startIndex < 1 ifTrue:[ ^ self _errorIndexOutOfRange: startIndex].

^ self indexOf: aCharacter asCharacter startingAt: startIndex
%

category: 'Case-Sensitive Searching'
method:
indexOfByte: anInt startingAt: startIndex

"Returns the index of the first occurrence of (Character withValue: anInt) 
 in the receiver, not preceding startIndex.    
 If the receiver does not contain the specified Character, returns zero.   

 The search is case-sensitive."

<primitive: 704>
startIndex _validateClass: SmallInteger .
startIndex < 1 ifTrue:[ ^ self _errorIndexOutOfRange: startIndex].
anInt _validateClass: SmallInteger  .
(anInt < 0 or:[ anInt > 16rFFFFFFFF]) ifTrue:[ ^ self _errorIndexOutOfRange: anInt ] .
^ self _primitiveFailed: #indexOfByte:startingAt: args: { anInt . startIndex }
%

method:
indexOfLastByte: anInt startingAt: startIndex

"Returns the index of the last occurrence of (Character withValue: anInt) 
 in the receiver, starting from startIndex and searching backwards.
 If the receiver does not contain the specified Character, returns zero.   

 The search is case-sensitive."

<primitive: 780>
startIndex _validateClass: SmallInteger .
startIndex < 1 ifTrue:[ ^ self _errorIndexOutOfRange: startIndex].
anInt _validateClass: SmallInteger  .
(anInt < 0 or:[ anInt > 16rFFFFFFFF]) ifTrue:[ ^ self _errorIndexOutOfRange: anInt ] .
^ self _primitiveFailed: #indexOfLastByte:startingAt:
       args: { anInt . startIndex }
%

category: 'Adding'
method:
add: aCharOrCharColl

"Appends all of the elements of aCharOrCharColl to the receiver and returns
 aCharOrCharColl."

<primitive: 296>
| cSize aString |
aCharOrCharColl class == Character ifTrue:[ | av |
  (av := aCharOrCharColl codePoint) > 16rFFFF ifTrue:[
    ^ self _convertToQuadByte add: aCharOrCharColl
  ].
  av > 16rFF ifTrue:[
    ^ self _convertToDoubleByte add: aCharOrCharColl
  ].
  ^ self _primitiveFailed: #add: args: { aCharOrCharColl }
].
((cSize := aCharOrCharColl stringCharSize) >= 2) ifTrue:[
  cSize == 2 ifTrue:[
    (aString := aCharOrCharColl _asString)
      ifNil:[ ^ self _convertToDoubleByte addAll: aCharOrCharColl ]
      ifNotNil:[ ^ self addAll: aString ].
  ].
  (aString := aCharOrCharColl _asString) 
    ifNil:[ ^ self _convertToQuadByte addAll: aCharOrCharColl ]
    ifNotNil:[ ^ self addAll: aString ].
].
(aCharOrCharColl isKindOf: AbstractCharacter) ifTrue:[ 
  ^ self add: aCharOrCharColl asCharacter .
].
aCharOrCharColl _validateClass: CharacterCollection .
^ self add: aCharOrCharColl asString.
%

category: 'Adding'
method:
addAll: aCharOrCharColl

"Equivalent to add: aCharOrCharColl."

<primitive: 296>
| cSize aString |
aCharOrCharColl class == Character ifTrue:[ | av |
  (av := aCharOrCharColl codePoint) > 16rFFFF ifTrue:[
    ^ self _convertToQuadByte add: aCharOrCharColl
  ].
  av > 16rFF ifTrue:[
    ^ self _convertToDoubleByte add: aCharOrCharColl
  ].
  ^ self _primitiveFailed: #add: args: { aCharOrCharColl }
].
((cSize := aCharOrCharColl stringCharSize) >= 2) ifTrue:[
  cSize == 2 ifTrue:[
    (aString := aCharOrCharColl _asString)
      ifNil:[ ^ self _convertToDoubleByte addAll: aCharOrCharColl ]
      ifNotNil:[ ^ self addAll: aString ].
  ].
  (aString := aCharOrCharColl _asString) 
    ifNil:[ ^ self _convertToQuadByte addAll: aCharOrCharColl ]
    ifNotNil:[ ^ self addAll: aString ].
].
(aCharOrCharColl isKindOf: AbstractCharacter) ifTrue:[ 
  ^ self add: aCharOrCharColl asCharacter .
].
(aCharOrCharColl isKindOf: CharacterCollection) ifTrue:[ 
  ^ self add: aCharOrCharColl asString.
].
aCharOrCharColl do: [:each | self add: each].
^aCharOrCharColl.
%

category: 'Adding'
method:
addLast: aCharOrCharColl

"Equivalent to add: aCharOrCharColl."

<primitive: 296>
| cSize aString |
aCharOrCharColl class == Character ifTrue:[ | av |
  (av := aCharOrCharColl codePoint) > 16rFFFF ifTrue:[
    ^ self _convertToQuadByte add: aCharOrCharColl
  ].
  av > 16rFF ifTrue:[
    ^ self _convertToDoubleByte add: aCharOrCharColl
  ].
  ^ self _primitiveFailed: #add: args: { aCharOrCharColl }
].
((cSize := aCharOrCharColl stringCharSize) >= 2) ifTrue:[
  cSize == 2 ifTrue:[
    (aString := aCharOrCharColl _asString)
      ifNil:[ ^ self _convertToDoubleByte addAll: aCharOrCharColl ]
      ifNotNil:[ ^ self addAll: aString ].
  ].
  (aString := aCharOrCharColl _asString) 
    ifNil:[ ^ self _convertToQuadByte addAll: aCharOrCharColl ]
    ifNotNil:[ ^ self addAll: aString ].
].
(aCharOrCharColl isKindOf: AbstractCharacter) ifTrue:[ 
  ^ self add: aCharOrCharColl asCharacter .
].
aCharOrCharColl _validateClass: CharacterCollection .
^ self add: aCharOrCharColl asString.
%

category: 'Adding'
method:
addAllBytes: aCharacterCollection

"Adds the byte contents of aCharacterCollection to the receiver ,
 using big-endian byte ordering of aCharacterCollection . 
 Returns aCharacterCollection.  
 
 The aCharacterCollection argument must be a kind of String or
 DoubleByteString."

"Used in implementation of methods to support PassiveObject."

<primitive: 470>
aCharacterCollection _validateClasses:{ String }.
^ self _primitiveFailed: #addAllBytes: args: { aCharacterCollection }
%

category: 'Adding'
method:
insertAll: aCharOrCharColl at: anIndex

"Inserts aCharOrCharColl at the specified index. Returns aCharOrCharColl"

<primitive: 299>
| aString cSize |
anIndex _isSmallInteger ifFalse:[ anIndex _validateClass: SmallInteger ].
((anIndex <= 0) or: [anIndex > (self size + 1)])
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].

(cSize := aCharOrCharColl stringCharSize) == 2 ifTrue:[
  (aString := aCharOrCharColl _asString) 
    ifNil:[ ^ self _convertToDoubleByte insertAll: aCharOrCharColl at: anIndex]
    ifNotNil:[ ^ self insertAll: aString at: anIndex ].
].
cSize == 4 ifTrue:[
  (aString := aCharOrCharColl _asString) 
    ifNil:[ ^ self _convertToQuadByte insertAll: aCharOrCharColl at: anIndex]
    ifNotNil:[ ^ self insertAll: aString at: anIndex ].
].
aCharOrCharColl _validateClasses: { AbstractCharacter . CharacterCollection }.
^self insertAll: aCharOrCharColl asString at: anIndex.
%

category: 'Converting'
method:
asArrayOfKeywords

"Returns an Array of keyword substrings held by the receiver.	 The receiver
 is assumed to be a colon-separated list of substrings.  These substrings
 are extracted and collected in an Array.  If the receiver contains no
 colons, the Array will hold a copy of the receiver."

| c nextName result |

result := { }.
nextName := self class _newString .
1 to: self size do: [ :i |
  c := self at: i.
  nextName add: c.
  c == $: ifTrue: [
    result add: nextName.
    nextName := self class _newString .
  ].
].
nextName size ~~ 0 ifTrue: [
  result add: nextName
]
ifFalse: [
  result size == 0 ifTrue: [result add: nextName]
].
^result
%

!asArrayOfSubstrings moved to CharacterCollection 
! inherit asNumber from CharacterCollection to fix 36647

category: 'Testing'
method:
containsSeparator

"Returns true if the receiver contains a separator Character."

1 to: self size do: [:i |
  (self at: i) isSeparator ifTrue: [^true].
].
^false
%

! lf now inherited

! String >> replaceFrom:to:with:startingAt: deleted for bug36177 (superclass implementation is correct

! space now inherited
! tab now inherited

! deletion

category: 'Copying'
method:
withLFs

"Returns a copy of the receiver with all back-slashes replaced by line-feeds."

^self copyReplaceAll: '\' with: Character lf
%

category: 'Execution'
method:
_compileInContext: anObject symbolList: aSymbolList 
oldLitVars: litVarArray  environmentId: anEnvironmentId flags: flagsInt

"Compiles the receiver as an instance method for the class of
 anObject, using aSymbolList as the symbol list.
 The receiver is parsed as a doit, i.e. it does not have a selector.

 If litVarArray is not nil, it must be an Array of Symbol,SymbolAssociation pairs
 and this Array will be searched prior to searching aSymbolList to
 resolve literal variables within the method.

 anEnvironmentId must be a SmallInteger >= 0 and <= 255
 specifying a compilation environment.

 flagsInt must be a SmallInteger >= 0 and <= 255 and contains the bits
 specified by GCI_COMPILE_* in $GEMSTONE/include/gcicmn.ht .

 Returns a GsMethod that is not in the method dictionary of any class, or
 else generates an error if there are compilation errors."

<primitive: 150>
aSymbolList _validateClass: SymbolList .
^ self _primitiveFailed: 
       #_compileInContext:symbolList:oldLitVars:environmentId:flags:
       args: { anObject . aSymbolList . litVarArray . anEnvironmentId . flagsInt}
%

category: 'Execution'
method:
_compileInContext: anObject symbolList: aSymbolList

"Compiles the receiver as an instance method for the class of
 anObject, using aSymbolList as the symbol list, and using the
 default compilation environment.
 The receiver is parsed as a doit, i.e. it does not have a selector.

 Returns a GsMethod that is not in the method dictionary of any class, or
 else generates an error if there are compilation errors."

^ self _compileInContext: anObject symbolList: aSymbolList oldLitVars: nil
      environmentId: 0 flags: 0 
%

! category: 'Repository Conversion'
! method:
! _compileForConversionWith: aSymbolList 
! oldNamesDict: oldNames
! oldLitVarsArray: oldLitVars
!
! "Private.  Compiles the receiver as an instance method for the class of
! anObject, using aSymbolList as the symbol list.
!
! Returns a GsMethod that is not in the method dictionary of any class, or
! else generates an error if there are compilation errors."
!
! <primitive: 475>   ++++ NOT SUPPORTED
!
!^ self _primitiveFailed: 
!       #_compileForConversionWith:oldNamesDict:oldLitVarsArray
!       args: { aSymbolList . oldNames . oldLitVars }
!%

category: 'Execution'
method:
evaluate

"Compiles the receiver as an unbound method and executes it using the current
 default symbol list."

^ (self _compileInContext: nil 
       symbolList: GsSession currentSession symbolList
       oldLitVars: nil environmentId: 0 flags: 0)
  _executeInContext: nil
%

category: 'Execution'
method:
evaluateInContext: anObject symbolList: aSymbolList 

"Compiles the receiver as an instance method for the class of anObject, using
 aSymbolList as the symbol list.  Executes the resulting GsMethod using anObject
 as self and returns the result of the execution.  Generates an error if
 compilation errors occur."

^ (self _compileInContext: anObject symbolList: aSymbolList oldLitVars: nil 
	environmentId: 0 flags: 0)
    _executeInContext: anObject
%

category: 'Deprecated'
method:
_executeAsSource
	"Compiles the receiver as an unbound method and executes it using the current
	default symbolList."

self deprecated: 'String>>_executeAsSource: deprecated long before v3.0. Use #evaluate'.
^ self evaluate
%

! deleted _idxCompareLessThan: v2.0

! deleted _idxCompareGreaterThan: v2.0

category: 'Removing'
method:
removeFrom: startIndex to: stopIndex

"Deletes the elements of the receiver from startIndex to stopIndex.  The
 size of the receiver is decreased by stopIndex - startIndex + 1.

 Both startIndex and stopIndex must be positive integers not larger than the
 size of the receiver, with startIndex <= stopIndex."

<primitive: 386>
(stopIndex < startIndex)
ifTrue:
   [ ^ startIndex _error: #rtErrBadCopyFromTo args: { stopIndex }].
((stopIndex > self size) or: [(stopIndex < 1)])
   ifTrue: [ ^ self _errorIndexOutOfRange: stopIndex].
(startIndex < 1)
   ifTrue: [ ^ self _errorIndexOutOfRange: startIndex].
^ self _primitiveFailed: #removeFrom:to: args: { startIndex . stopIndex }
%

category: 'Storing and Loading'
method:
loadFrom: passiveObj

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into the receiver."

passiveObj hasRead: self.
passiveObj next: self _basicSize bytesTo: self.
passiveObj next.  "consume the space put after all strings"
^self
%

category: 'Storing and Loading'
method:
writeTo: passiveObj
"Converts the receiver to its passive form and writes that information on
 passiveObj."

passiveObj writeClass: self class;
  writeSize: self size;
    nextPutAll: self; space
%

category: 'Deprecated'
method:
toClientTextFile: fileName

| file |
self deprecated: 'String>>toClientTextFile: deprecated long before v3.0.  Use an instance of GsFile to access the file system.'.

file := GsFile openWrite: fileName.
[file == nil] whileTrue: [
  self _halt: 'Unable to open file: ' , fileName.
  file := GsFile openWrite: fileName
].
file nextPutAll: self.
file close
%

category: 'Decompiling without Sources'
method:
_asSource

^ self quoted 
%

category: 'Repository Conversion'
method:
transformIntoString

"Private. Transforms the receiver to an instance of String."

^ self.
%

! deleted _oldMatchPattern: 

category: 'Comparing'
method:
matchPattern: patternArray

"Returns true if the receiver matches the pattern, false if it doesn't.
 An exact match is required.  For partial matching, use the 'Searching' method
 findPattern:startingAt: instead.

 The pattern is a kind of Array containing zero or more
 CharacterCollections, plus zero or more occurrences of the special Characters
 $* or $?.  If either $* or $? occurs in the pattern, it acts as a wild card.
 The Character $? matches any single Character in the receiver, and $* matches
 any sequence of zero or more Characters in the receiver.  For example,

 'weimaraner' matchPattern: #('w' $* 'r')

 returns true, because the Character $* is interpreted as a wild card.

 If either of these special Characters occurs in the receiver, it is
 interpreted literally.  For example,

 'w*r' matchPattern: #('weimaraner')

 returns false - because the Character $* occurs in the receiver, it is
 interpreted as a literal asterisk (not as a wild card)."
 | res |
 res := self
    _primMatchPattern: patternArray
    ignoreCase: false
    findSubString: false
    startingAt: 1 .
 res == -1 ifTrue:[ ^ super matchPattern: patternArray "handle Unicode in arg" ].
 ^ res
%

category: 'Comparing'
method:
_matchPatternNoCase: patternArray

"Just like matchPattern: except the match is case-insensitive."

| res |
res := self
    _primMatchPattern: patternArray
    ignoreCase: true
    findSubString: false
    startingAt: 1 .
res == -1 ifTrue:[ ArgumentError signal:'invalid element of patternArray'].
^ res
%

! deleted _oldFindPattern: aPattern startingAt: anIndex 
! deleted _oldFindPatternNoCase: aPattern startingAt: anIndex 
! deleted _oldIndexOf: pattern matchCase: flag startingAt: startIndex

category: 'Searching'
method:
findPattern: aPattern startingAt:  anIndex 

"This method searches the receiver, beginning at anIndex, for a substring that
 matches aPattern.  If a matching substring is found, this method returns the
 index of the first Character of the substring.  Otherwise, this returns 0.

 The argument aPattern is an Array containing zero or more CharacterCollections
 plus zero or more occurrences of the special Characters asterisk or
 question-mark.  See the description of the matchPattern: method for more
 information about this argument.

 Performs a case-sensitive search."

^self
    indexOf: aPattern
    matchCase: true
    startingAt: anIndex 
%

category: 'Searching'
method:
findPatternNoCase: aPattern startingAt:  anIndex 

"This method searches the receiver, beginning at anIndex, for a substring that
 matches aPattern.  If a matching substring is found, this method returns the
 index of the first Character of the substring.  Otherwise, this returns 0.

 The argument aPattern is an Array containing zero or more CharacterCollections
 plus zero or more occurrences of the special Characters asterisk or
 question-mark.  See the description of the matchPattern: method for more
 information about this argument.

 Performs a case-insensitive search."

^self
    indexOf: aPattern
    matchCase: false
    startingAt: anIndex 
%

category: 'Searching'
method:
indexOf: patternArray matchCase: caseBoolean startingAt:  anInteger

"Searches the receiver, starting at the specified index,
 for a substring that matches the pattern.  If a matching substring
 is found, returns the index of the first Character of the substring.
 Otherwise, returns 0.

 The pattern is an Array containing zero or more CharacterCollections
 plus zero or more occurrences of the special Characters asterisk or
 question-mark.  See the description of the matchPattern: method for more
 information about this argument.

 If matchCase is true, a case-sensitive search is performed.  Otherwise,
 a case-insensitive search is performed."

  | result |
  result := self
      _primMatchPattern: patternArray
      ignoreCase: (caseBoolean not)
      findSubString: true 
      startingAt: anInteger.
  result == -1 ifTrue:[ ^ super indexOf: patternArray matchCase: caseBoolean startingAt:  anInteger].
  ^ result ifNil:[ 0 ] ifNotNil:[ result at: 1 ]
%

category: 'Comparing'
method:
_primMatchPattern: anArray ignoreCase: caseBoolean findSubString: subStringBoolean startingAt:  anInteger

"If findSubString is false then the pattern array must match the entire
 receiver starting at the specified index and the result is true or false.
 If findSubString is true then the pattern array must match a substring
 starting at the specified index and the result is nil if no substring is
 found and a two element Array containing the start and end index if a match
 is found. 
 The Strings in the pattern array must be Strings or ByteArrays .
 multi-byte strings are not supported.

 Returns -1 if an element of anArray is a kind of MultiByteString .
"
 
<primitive: 514>
^ self _primitiveFailed: 
       #_primMatchPattern:ignoreCase:findSubString:startingAt:
       args: { anArray . caseBoolean . subStringBoolean . anInteger }
%

category: 'New Indexing Comparison - prims'
method:
_idxPrimCompareEqualTo: aCharCollection
  "This comparison operation is used for the indexing subsystem to
 determine an ordering for insertion into indexing objects.

 This method collates letters AaBb..Zz."

  "The comparison should be compatible with the case-insensitive semantics
 of the String method with selector #= .
 Same primitive as String>>lessThan: "

  <primitive: 981>
  aCharCollection isUnicodeString
    ifTrue: [ ArgumentError signal:'Unicode argument disallowed in String comparison' ]
    ifFalse: [
      aCharCollection ifNil:[ ^ false ].
      aCharCollection _validateClass: CharacterCollection.
      "JapaneseString or MixedString"
      ^ self = aCharCollection ]
%

category: 'New Indexing Comparison - prims'
method:
_idxPrimCompareLessThan: aCharCollection

"This comparison operation is used for the indexing subsystem to
 determine an ordering for insertion into indexing objects and for
 doing indexing subsystem comparisons.

 This method collates the same as the < method, except that
 it returns false if the argument is nil."

  <primitive: 434>
  aCharCollection isUnicodeString
    ifTrue: [ ArgumentError signal:'Unicode argument disallowed in String comparison' ]
    ifFalse: [ 
      aCharCollection _validateClass: CharacterCollection.
      "JapaneseString or MixedString"
      ^ self < aCharCollection ]
%

category: 'New Indexing Comparison - prims'
method:
_idxPrimCompareGreaterThan: aCharCollection

"This comparison operation is used for the indexing subsystem to
 determine an ordering for insertion into indexing objects and for
 doing indexing subsystem comparisons.

 This method collates the same as the > method, except that
 it returns true if the argument is nil."

  <primitive: 184>
  (aCharCollection _stringCharSize bitAnd: 16r8) ~~ 0
    ifTrue: [ 
      "Unicode"
      ArgumentError signal:'Unicode argument disallowed in String comparison' ]
    ifFalse: [
      aCharCollection _validateClass: CharacterCollection.
      "JapaneseString or MixedString"
      ^ self > aCharCollection ]
%

category: 'Encoding'
method:
encodeUsing: encodingArray

  "A new instance of the class of the receiver will be returned where
   each character in the receiver is replaced by the character or string
   from encodingArray that is located at the position in encodingArray for
   the character.  The codePoint of the character is used as an index 
   into the encodingArray,  value 16r0 accesses   encodingArray at:1 .
   An Error is signaled if any of the following are true
    the codePoint is >= size of the encodingArray

    the value from the encodingArray is a Character with codePoint too large 
    to fit in an instance of the receiver's class.

    the value from the encodingArray is a CharacterCollection with
    bytes-per-character bigger than bytes-per-character of the receiver

    the value from the encodingArray is neither a String, MultiByteString,
    nor Character . "
  <primitive: 654>
  encodingArray _validateInstanceOf: Array  .
  self _primitiveFailed: #encodeUsing: args: { encodingArray }
%

category: 'Encoding'
method: 
encodeAsUTF8

"Encode receiver in UTF8 format.  Returns a Utf8 ."

<primitive: 468>
self _primitiveFailed: #encodeAsUTF8
%

category: 'Encoding'
method: 
encodeAsUTF16

"Encode receiver in UTF16 format.  Returns a Utf16 ."

<primitive: 1084>
self _primitiveFailed: #encodeAsUTF16
%

method:
encodeAsUTF8IntoString
"Encode receiver in UTF8 format.  Returns a String. For Seaside"

<primitive: 994>
self _primitiveFailed: #encodeAsUTF8IntoString
%

! fixed 45510
category: 'Encoding'
method:
decodeFromUTF8

"Decode receiver from UTF8 format.
 Returns either a Unicode7 , Unicode16 or Unicode32 ,
 using the minimum character size needed to represent decoded result."

^ self _decodeUtf8StartingAt: 1 unicodeResult: true maxSize: nil bytesConsumed: nil
%

category: 'Encoding'
method:
decodeFromUTF8ToString

"Decode UTF8 contents of the receiver returning a String, DoubleByteString or
  QuadByteString."

^ self _decodeUtf8StartingAt: 1 unicodeResult: false maxSize: nil bytesConsumed: nil
%
method:
decodeFromUTF8ToUnicode

"Decode UTF8 contents of the receiver returning a Unicode7 , Unicode16 or Unicode32"

^ self _decodeUtf8StartingAt: 1 unicodeResult: true maxSize: nil bytesConsumed: nil
%

!  _decodeFromUTF8: is used by topaz, do not deprecate
category: 'Private'
method:
_decodeFromUtf8: unicodeResultBool maxSize: aSize

"Decode UTF8 contents of the receiver.
 If unicodeResultBool == true, result is a Unicode7 , Unicode16 or Unicode32.
 If unicodeResultBool == false, result is a String, DoubleByteString or 
  QuadByteString."

^ self _decodeUtf8StartingAt: 1 unicodeResult: unicodeResultBool maxSize: aSize bytesConsumed: nil
%

method: 
_decodeFromUtf8: unicodeResultBool

"Decode UTF8 contents of the receiver.
 If unicodeResultBool == true, result is a Unicode7 , Unicode16 or Unicode32.
 If unicodeResultBool == false, result is a String, DoubleByteString or
  QuadByteString."

^ self _decodeUtf8StartingAt: 1 unicodeResult: unicodeResultBool maxSize: nil bytesConsumed: nil
%

method:
_decodeUtf8StartingAt: anOffset unicodeResult: unicodeResultBool 
                 maxSize: aSize bytesConsumed: anArray

"Attempt to decode Utf8 data starting at anOffset in the receiver.
 If unicodeResultBool == true, result is a Unicode7 , Unicode16 or Unicode32.
 If unicodeResultBool == false, result is a String, DoubleByteString or
  QuadByteString.
 If there are insufficient bytes in the receiver and anArray ~~ nil, 
 returns nil .
  If there are insufficient bytes in the receiver and anArray == nil, signals an Error.
 Signals an Error if the bytes starting at anOffset are not legal UTF8 data .
 If anArray ~~ nil, returns in (anArray at: 1) the number of bytes consumed by the decode."
 
<primitive: 493>
unicodeResultBool _validateClass: Boolean .
aSize _validateClass: SmallInteger .
anArray ifNotNil:[ anArray _validateClass: Array  ].
anOffset _validateClass: SmallInteger .

^ self _primitiveFailed: #_decodeUtf8StartingAt:unicodeResult:maxSize:bytesConsumed:
          args: { anOffset . unicodeResultBool . aSize . anArray }
%

category: 'Converting'
method:
_convertToDoubleByte

"Change the receiver from a String to a DoubleByteString,
 or from an Unicode7 to Unicode16.  Changes the
 class of the receiver and the receiver's storage structure.
 Returns the receiver."

<primitive: 196>
%

method: 
_asUnicode7

<primitive: 925>
"Return an instance of Unicode7 if receiver can be
 represented as an Unicode7 string, else return nil."

self _primitiveFailed:#_asUnicode7
%


category: 'Converting'
method:
_convertToQuadByte

"Change the receiver from a DoubleByteString to a QuadByteString,
 or from a Unicode16 to Unicode32.  Changes the
 class of the receiver and the receiver's storage structure.
 Returns the receiver."

<primitive: 445>
%
method: 
asUnicodeString

"If receiver is an Unicode7 , return the receiver,
 else return an instance of Unicode7 or Unicode16
 using the class with smallest character size needed to
 represent a copy of the receiver."

<primitive: 927>

self _primitiveFailed:#asUnicodeString
%

category: 'Hashing'
method: 
hash

"Returns a positive Integer based on a case-sensitive hash of the contents 
 of the receiver.  The algorithm implemented is described in:

 [Pearson 90]
 Pearson, Peter K., Fast Hashing of Variable-Length Text Strings,
 Communications of the ACM 33, 6, (June 1990), 677-680."

<primitive: 31>
self _primitiveFailed: #hash .
self _uncontinuableError
%

category: 'Copying'
method:
_reverseFrom: aString

"copies elements of aString into receiver in reverse order. 
 Size of aString must be same as size of receiver.
 Returns receiver. 
 aString==self is allowed. "

<primitive: 793>
aString stringCharSize == 1 ifFalse:[
  aString _error: #rtErrBadArgKind args:{ String }.
].
aString size == self size ifFalse:[
  aString _error: #rtErrBadSize args:{ self size . aString size }.
].
self _primitiveFailed: #_reverseFrom: args: { aString } .
%

category: 'Copying'
method:
reverse

"Returns a copy of the receiver with its elements in reverse order."
| copy |
copy := self species new: self size .
^ copy _reverseFrom: self .
%
category: 'Copying'
method:  
copyFrom: startIndex to: stopIndex

"Returns a new Array containing the elements of the receiver
 between startIndex and stopIndex, inclusive.  The result is of the same class
 as the receiver.

 If startIndex > stopIndex then an empty collection is returned.
 Otherwise both startIndex and stopIndex must be positive integers not larger than the
 size of the receiver, with startIndex <= stopIndex.
"

<primitive: 818>
(startIndex < 1) ifTrue: [ ^ self _errorIndexOutOfRange: startIndex].

((stopIndex > self size) or: [(stopIndex < 0)])
   ifTrue: [ ^ self _errorIndexOutOfRange: stopIndex].

self _primitiveFailed: #copyFrom:to: args: { startIndex . stopIndex }
%
category: 'Formatting'
method:
charSize
"Returns number of bytes that make up a character for this string class."

^ 1
%
classmethod:
charSize
"Returns number of bytes that make up a character for this string class."

^ 1
%

category: 'Class Membership'
classmethod:
_classHistoryIncludesIdentical: aClass
  aClass == MultiByteString ifTrue:[ ^ true ] .
  ^ super _classHistoryIncludesIdentical: aClass
%

category: 'Encoding'
method: 
addAllUtf8: aCharacterOrString
  "appends the UTF8 encoding of the argument to the receiver."
  <primitive: 1045>
  aCharacterOrString _validateKindOfClasses: { Character . String } .
  ^ self _primitiveFailed: #nextPutAllUtf8: args: { aCharacterOrString }
%

! fixed 47011
method: 
javaScriptEncode
  "Returns a String.
   The primitive encapulates all of the encoding logic 
   from Seaside JSStream>>encodeString:on: .
   See   String class>>_javaScriptEncode: for the algorithm in Smalltalk.
   Assumes Character>>greaseInteger always equivalent to Character>>codePoint
  "
<primitive: 1046>
^ self _primitiveFailed: #javaScriptEncode 
%

category: 'Documentation'
classmethod: 
_javascriptCharacters
  "Documentation/testing for String>>javaScriptEncode primitive.
   Returns a Translation Array."
  | jsc escapes |
  jsc := Array new: 128.
  0 to: 31 do: [ :index |
    jsc
      at: index + 1
      put: '\x' , (index asHexStringWithLength: 2)
  ].
  escapes := #( 0 '\0' 7 '\a' 8 '\b' 9 '\t' 10 '\n' 12 '\f' 13 '\r' 34 '\"' 92 '\\' ).
  1 to: escapes size - 1 by: 2 do: [ :index |
    jsc at: (escapes at: index) + 1  put: (escapes at: index + 1)
  ].
  ^ jsc
%

category: 'Documentation'
classmethod:
_javaScriptEncode: aString
 "Documentation/testing for String>>javaScriptEncode primitive.
 The smalltalk algorithm which is the specification for javaScriptEncode primitive."
 | estr jscChars jscCharsSize |
  estr := aString class new .
  estr add: $" .
  jscChars := self _javascriptCharacters . "method temp access faster than classVar"
  jscCharsSize := jscChars size . "avoid multiple sends of size "
  1 to: aString size do: [ :index |
    | char value encoded |
    char := aString at: index.
    value := char codePoint"was greaseInteger" .
    value < jscCharsSize
      ifFalse: [
        "U+2028 and U+2029 have to be treated as new lines"
        value == 16r2028 "Line separator"
          ifTrue: [ estr addAll: '\u2028' ]
          ifFalse: [
            value == 16r2029 "Paragraph separator"
              ifTrue: [ estr addAll: '\u2029' ]
              ifFalse: [ estr add: char ] ] ]
      ifTrue: [
        encoded := jscChars at: value + 1.
        "we use nil markers because #ifNil is faster than #isString because it's not
        actually sent"
        encoded
          ifNil: [ estr add: char ]
          ifNotNil: [ estr addAll: encoded ] ] ].
  estr add: $" .
  ^ estr
%



category: 'Adding'
method:
addCodePoint: aSmallInteger

<primitive: 1047>
^ self add: (Character codePoint: aSmallInteger).
%
category: 'Comparing'
method:
codePointCompareTo: aString

"aString must be a CharacterCollection or ByteArray,
 and must not be a Utf8 or Utf16 .
 Return -1 , 0, or 1  based on how self compares to aString 
 using codePoint comparison of each character. "

<primitive: 844>
^ self _primitiveFailed: #codePointCompareTo: args: { aString }
%
