!=========================================================================
! Copyright (C) GemTalk Systems 1986-2020.  All Rights Reserved.
!
! $Id: writestPortable.gs 25471 2011-03-16 19:39:56Z lalmarod $
!
! Based on Pharo-1.2.1 implementation
!
! Superclass Hierarchy:
!   WriteStream, PositionableStream, Stream, Object.
!
! See installStreamHierarchy.topaz for filein information
!
!=========================================================================

! class created by installStreamHierarchy.topaz

! to filein this file interactively for debugging compile errors,
!   UserGlobals at: #WriteStream put: WriteStreamPortable

expectvalue /String
run
Stream _initializeWriteStreamClassVars: WriteStream
%    

set class WriteStream
removeallmethods 
removeallclassmethods 

doit
WriteStream comment: 'WriteStream / WriteStreamPortable is an ANSI compliant implementation of 
 WriteStream, a PositionableStream that allows its objects to be written, 
 but not read.

--- instVar writeLimit
A SmallInteger; the last position in the collection available for writing.'.
true
%

category: 'Instance Creation'
classmethod:
with: aCollection from: firstIndex to: lastIndex 
"Answer an instance of me on the subcollection of the argument, 
aCollection, determined by the indices firstIndex and lastIndex. Position 
the instance to store at the end of the subcollection."

^self basicNew with: (aCollection copyFrom: firstIndex to: lastIndex)
%

category: 'Instance Creation'
classmethod:
on: aCollection from: firstIndex to: lastIndex 
"Answer an instance of me on a copy of the argument, aCollection, 
determined by the indices firstIndex and lastIndex. Position the instance 
at the beginning of the collection."

^self basicNew
	on: aCollection
	from: firstIndex
	to: lastIndex
%

category: 'Instance Creation'
classmethod:
new

"Disallowed.  To create a new WriteStream, use the class method on: instead."

self shouldNotImplement: #new
%

category: 'Instance Creation'
classmethod:
with: aCollection 
"Answer an instance of me on the argument, aCollection, positioned to 
store objects at the end of aCollection."

^self basicNew with: aCollection
%

category: 'Accessing'
method:
next

"Disallowed.  You cannot read a WriteStream."

self shouldNotImplement: #next
%

category: 'Adding'
method:
nextPut: anObject

"Inserts anObject as the next element that the receiver can access for writing.
 Returns anObject."

position >= writeLimit
	ifTrue: [^ self pastEndPut: anObject]
	ifFalse:[ | res |
		res := collection at: position + 1 put: anObject.
		position := position + 1.
		^res]
%

category: 'Private'
method:
pastEndPut: anObject
"Grow the collection by adding anObject one past end.
Then put <anObject> at the current write position."

collection at: position + 1 put: anObject.
position := position + 1.
writeLimit := collection size.
^ anObject
%

category: 'Adding'
method:
nextPutAll: aCollection

"Inserts the elements of aCollection as the next elements that the receiver can
 access.  Returns aCollection."

 
 | newEnd |
 collection class == aCollection class ifFalse:
 	[^ super nextPutAll: aCollection ].
 
 newEnd := position + aCollection size.
 collection replaceFrom: position+1 to: newEnd  with: aCollection startingAt: 1.

 newEnd > writeLimit 
 	ifTrue: [
  	 	collection size: newEnd.
 	 	writeLimit := newEnd ]. 
 position := newEnd.
	
^ aCollection
%

category: 'Adding'
method:
print: anObject

	anObject printOn: self.
%

category: 'Accessing'
method:
contents
"WriteStreams return the portion of their collection that has been written:
 the collection up to the next write-position."

readLimit := readLimit max: position.
^collection copyFrom: 1 to: position
%

category: 'ANSI'
method:
flush
	"Update a stream's backing store.
	Upon return, if the receiver is a write-back stream, the state of the
	stream backing store must be consistent with the current state of the 
	receiver.
	If the receiver is not a write-back stream, the effect of this 
	message is unspecified."

	"We do nothing; this method is provided for ANSI compatibility"
%

category: 'Adding'
method:
next: anInteger putAll: aCollection startingAt: startIndex
"Store the next anInteger elements from the given collection."

| newEnd numPut |
collection class == aCollection class ifFalse:
	[^ super next: anInteger putAll: aCollection startingAt: startIndex ].

numPut := anInteger min: (aCollection size - startIndex + 1).
newEnd := position + numPut.
newEnd > writeLimit ifTrue:
	[^ super next: anInteger putAll: aCollection startingAt: startIndex "Trigger normal pastEndPut: logic"].

collection replaceFrom: position+1 to: newEnd with: aCollection startingAt: startIndex.
position := newEnd.
%

category: 'Positioning'
method:
position: anInteger 
"Refer to the comment in PositionableStream|position:."

readLimit := readLimit max: position.
super position: anInteger
%

category: 'Positioning'
method:
reset 
"Refer to the comment in PositionableStream|reset."

readLimit := readLimit max: position.
position := 0
%

category: 'Positioning'
method:
setToEnd 
"Refer to the comment in PositionableStream|setToEnd."

readLimit := readLimit max: position.
super setToEnd.
%

category: 'Positioning'
method:
size

^readLimit := readLimit max: position
%

category: 'Private'
method:
on: aCollection from: firstIndex to: lastIndex

| len |
collection := aCollection.
readLimit := writeLimit := lastIndex > (len := collection size)
	ifTrue: [len]
	ifFalse: [lastIndex].
position := firstIndex <= 1
	ifTrue: [0]
	ifFalse: [firstIndex - 1]
%

category: 'Private'
method:
on: aCollection

super on: aCollection.
readLimit := 0.
writeLimit := aCollection size
%

category: 'Private'
method:
with: aCollection

super on: aCollection.
position := readLimit := writeLimit := aCollection size
%

category: 'Character writing'
method:
cr
"Append a return character to the receiver."

self nextPut:  Cr   .
%

category: 'Character writing'
method:
crlf
"Append a carriage return character followed by a line feed character to the receiver."

self nextPutAll: CrLf .
%

category: 'Character writing'
method:
crtab
"Append a return character, followed by a single tab character, to the 
receiver."

self nextPutAll: CrTab .
%

category: 'Character writing'
method:
crtab: anInteger 
"Append a return character, followed by anInteger tab characters, to the 
receiver."

self nextPut: Cr .
anInteger timesRepeat: [self nextPut: Tab  ]
%

category: 'Character writing'
method:
space: anInteger 
"Append anInteger space characters to the receiver."

anInteger timesRepeat: [self nextPut: $    ]
%

category: 'Character writing'
method:
tab: anInteger 
"Append anInteger tab characters to the receiver."

anInteger timesRepeat: [ self nextPut: Tab ]
%

category: 'Accessing'
method:
skip: anInteger 
"Set the receiver's position to be the current position+anInteger. Do not
 throw error if skipAmount would exceed collection bounds - ANSI compliance. "

self position: ((position + anInteger max: 0) min: writeLimit)
%

! fix 37918/36521: add _nextPut:

category: 'Adding'
method:
_nextPut: anObject
"see PrintStream>>_nextPut:"

self nextPut: anObject
%
category: 'Adding'
method:
nextPutAllBytes: aCharacterCollection
  "Inserts the byte contents of aCharacterCollection as the next elements 
 that the receiver can access.  Returns aCollection.  The receiver's collection
 must be a type of String."

  "Used in the implementation of PassiveObject."

  position == collection size
    ifFalse: [ ^ super nextPutAllBytes: aCharacterCollection ].
  collection addAllBytes: aCharacterCollection.
  position := position + aCharacterCollection _basicSize.
  ^ aCharacterCollection
%

! test optimized Character literals
run
  | strm ary exp |
  strm := WriteStream on: String new .
  strm cr ; crlf ; crtab ; crtab: 2 ; lf ; space ; tab ; space: 2 ; tab: 2 .
  ary := { } . 
  strm contents do:[:c | ary add: c  codePoint ] . 
  ary = #( 13 13 10 13 9 13 9 9 10 32 9 32 32 9 9 ) ifFalse:[ ^ ary ].
  true
%
